#define KUBUNTU

#include <qvbox.h>
#include <qlabel.h>
#include <qwidgetstack.h>
#include <qsplitter.h>
#include <qtimer.h>

#include <klocale.h>
#include <kmessagebox.h>
#include <kaction.h>
#include <kactionclasses.h>
#include <kapplication.h>
#include <kdebug.h>
#include <kparts/part.h>
#include <kstatusbar.h>

// bleeeh
#include <apt-pkg/init.h>
#include <cassert>

#include <apt-front/manager.h>
#include <apt-front/init.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/cache/component/state.h>
#include <apt-front/cache/component/history.h>
#include <apt-front/predicate/factory.h>

#include <adept/acqprogresswidget.h>
#include <adept/progress.h>
#include <adept/utils.h>

#ifdef KUBUNTU
#include <kubuntu_upgrader/upgradewizard.h>
#endif

#include "app.h"

using namespace aptFront;
using namespace aptFront::cache;
using namespace adept;

TestApp::TestApp() {
    kdDebug() << "ctor running" << endl;
    setAcceptReadOnly( true );

    setupActions();
    setupGUI();

    m_rebuilds = 0;
    m_stack = new QWidgetStack( this );

    m_stack->addWidget( m_loading = new QLabel( i18n( "Loading, please wait..." ), m_stack ) );
    m_loading->setAlignment( Qt::AlignHCenter | Qt::AlignVCenter );

    Application::setStatusBar( statusBar() );

    setCentralWidget( m_stack );

    QTimer::singleShot(
        0, this,
        SLOT( delayed() ) );

    kdDebug() << "ctor done" << endl;

#ifdef KUBUNTU
    m_upgradeButtonAdded = false;
#endif
}

void TestApp::guardLister( Lister *l )
{
    connect( l, SIGNAL( rebuildStarted() ),
             this, SLOT( rebuildStarted() ) );
    connect( l, SIGNAL( rebuildFinished() ),
             this, SLOT( rebuildFinished() ) );
}

void TestApp::delayed() {
    initialize();

    observeComponent< component::State >();

    m_stack->addWidget( m_list = new adept::Browser( m_stack ) );
    guardLister( m_list->searchView()->lister() );

    m_stack->addWidget( m_sources = new adept::SourcesEditor(
                            _config->FindFile(
                                "Dir::Etc::sourcelist").c_str(),
                            this ) );

    connect ( m_sources, SIGNAL( close() ),
              this, SLOT( closeSources() ) );

    // set up preview widget
    m_stack->addWidget( m_preview = new adept::Browser( m_stack ) );

    m_preview->searchView()->setPreviewMode();
    guardLister( m_preview->searchView()->lister() );

    m_stack->addWidget( m_progress = new adept::AcqProgressWidget( m_stack ) );
    m_stack->addWidget( m_commitProgress = new adept::CommitProgress( m_stack ) );

    m_stack->raiseWidget( m_list );

    addMode( Sources, m_sourcesAction, m_sources );
    addMode( Preview, m_previewAction, m_preview );
    addMode( Download, m_progressAction, m_progress );
    addMode( Commit, m_commitProgressAction, m_commitProgress );

    m_list->searchView()->lister()->cleanRebuild();
    m_preview->searchView()->lister()->cleanRebuild();
    setActionsEnabled( true );
    statusBar()->clear();
    notifyPostChange( 0 );
}

void TestApp::setupActions()
{
    (new KAction(
        i18n( "Fetch Updates" ), u8( "adept_update" ),
        0, this, SLOT( update() ), actionCollection(),
        "update" ))->setEnabled( false );

    (new KAction(
        i18n( "Reload Cache" ), u8( "adept_reload" ),
        0, this, SLOT( reload() ), actionCollection(),
        "reload" ))->setEnabled( false );

    m_upgrade = new KAction(
        i18n( "Safe Upgrade" ), u8( "adept_upgrade" ),
        0, this, SLOT( upgrade() ), actionCollection(),
        "upgrade" );
    m_distUpgrade = new KAction(
        i18n( "Full Upgrade" ), u8( "adept_distupgrade" ),
        0, this, SLOT( distUpgrade() ), actionCollection(),
        "dist-upgrade" );
    m_commit = new KAction(
        i18n( "Apply Changes" ), u8( "adept_commit" ),
        0, this, SLOT( commit() ), actionCollection(),
        "commit" );

    m_sourcesAction = new KToggleAction(
        i18n( "Manage Repositories" ), u8( "adept_sourceseditor" ),
        0, this, SLOT( toggleSources() ), actionCollection(),
        "sourceseditor" );
    m_previewAction = new KToggleAction(
        i18n( "Preview Changes" ), u8( "adept_preview" ),
        0, this, SLOT( togglePreview() ), actionCollection(),
        "preview" );

    m_progressAction = new KToggleAction(
        i18n( "Show Last Download" ), u8( "adept_download_out" ),
        0, this, SLOT( toggleDownload() ), actionCollection(),
        "download_out" );

    m_commitProgressAction = new KToggleAction(
        i18n( "Show Last DPkg Run" ), u8( "adept_commit_out" ),
        0, this, SLOT( toggleCommit() ), actionCollection(),
        "commit_out" );

    m_listAction = new KToggleAction(
        i18n( "Show Package List" ), u8( "adept_packagelist" ),
        0, this, SLOT( closeModes() ), actionCollection(),
        "packagelist" );

    m_sourcesAction->setEnabled( false );
    m_progressAction->setEnabled( false );
    m_previewAction->setEnabled( false );
    m_commitProgressAction->setEnabled( false );
    m_listAction->setEnabled( false );

    m_modesClosed = m_listAction;
    m_listAction->setChecked( true );

    m_undo = KStdAction::undo( this, SLOT( undo() ), actionCollection() );
    m_redo = KStdAction::redo( this, SLOT( redo() ), actionCollection() );

    m_distUpgrade->setEnabled( false );
    m_upgrade->setEnabled( false );

    m_commit->setEnabled( false );
    KStdAction::quit( this, SLOT( close() ), actionCollection() );
    setHistoryEnabled( false );
    createStandardStatusBarAction();
}

void TestApp::setHistoryEnabled( bool e ) {
    if ( e && history() ) {
        m_undo->setEnabled( history()->canUndo() );
        m_redo->setEnabled( history()->canRedo() );
    } else {
        m_undo->setEnabled( false );
        m_redo->setEnabled( false );
    }
}

void TestApp::setActionsEnabled( bool e )
{
    m_actionsEnabled = e;
    updateActionState();
}

void TestApp::updateActionState()
{
    bool e = m_actionsEnabled;
    bool w = cache::Global::get().writeable();
    component::State &s = cache::Global::get().state();
    KActionPtrList a = actionCollection()->actions();
    for (KActionPtrList::iterator i = a.begin(); i != a.end(); ++i)
        if ( u8( (*i)->name() ) == u8( "update" ) )
            (*i)->setEnabled( e && w );
        else if ( u8( (*i)->name() ) == u8( "preview" ) )
            (*i)->setEnabled( e );
    setHistoryEnabled( e );
    m_commit->setEnabled( e && w && s.changed() );
    m_upgrade->setEnabled( e && w && s.upgradableCount() );
    m_distUpgrade->setEnabled( e && w && s.upgradableCount() );
}

template< typename T, typename In >
void TestApp::aptAction( In b, In e ) {
    const KAction *a = dynamic_cast<const KAction *>( sender() ); // HACK
    typename T::Vector v = T::list();
    for (typename T::Vector::iterator i = v.begin(); i != v.end(); ++i) {
        if( a->name() == i->name() ) {
            (*i)( b, e );
            return;
        }
    }
}

void TestApp::notifyPostRebuild( component::Base *b )
{
    Application::notifyPostRebuild( b );
    notifyPostChange( b );
}

void TestApp::notifyPreChange( component::Base *b )
{
    Application::notifyPreChange( b );
    checkpoint();
}

void TestApp::notifyPostChange( component::Base *b )
{
    Application::notifyPostChange( b );
    updateActionState();
}

void TestApp::closeEvent( QCloseEvent *e ) {
    cache::component::State &s = cache::Global::get().state();
    if (s.changed()) {
        if (KMessageBox::warningYesNo(
                this, i18n( "You have done changes that were left uncommited. "
                            "Are you sure you want to exit? " ),
                i18n( "Uncommited changes, really quit?" ) ) == KMessageBox::Yes)
            e->accept();
    } else
        e->accept();
}

void TestApp::foregroundClosed()
{
    m_stack->raiseWidget( m_list );
}

void TestApp::update() {
    closeModes();
    setActionsEnabled( false );
    aptFront::Manager m;
    m.setProgressCallback( m_progress->callback() );
    m.setUpdateInterval( 100000 );
    try {
        m_stack->raiseWidget( m_progress );
        m.update();
    } catch ( exception::OperationCancelled ) { // ignore
    } catch (...) {
        KMessageBox::sorry( this,
                            i18n( "There was an error downloading updates. " ),
                            i18n( "Could not fetch updates" ) );
    }
    kdDebug() << "closing progress widget" << endl;
    m_stack->raiseWidget( m_list );
    setActionsEnabled( true );
    notifyPostChange( 0 );
#ifdef KUBUNTU
	wizard = new UpgradeWizard(this, 0, 1);
	if( wizard->checkForDistUpgrade(false) && !m_upgradeButtonAdded ) {
	  KToolBar* tool = toolBar();
          tool->insertButton(QString("2uparrow"), 12345, SIGNAL(released()), this, SLOT(releaseUpgrade()), true, i18n("Version Upgrade"));
	  m_upgradeButtonAdded = true;
	} else {
		wizard->close();
	}
#endif
}

#ifdef KUBUNTU
void TestApp::releaseUpgrade() {
    wizard->show();
}
#endif

#ifdef KUBUNTU
void TestApp::addVersionUpgrade() {
  wizard = new UpgradeWizard(this, 0, 1);
  wizard->checkForDistUpgrade(true);
  KToolBar* tool = toolBar();
  tool->insertButton(QString("2uparrow"), 12345, SIGNAL(released()), this, SLOT(releaseUpgrade()), true, i18n("Version Upgrade"));
  m_upgradeButtonAdded = true;
}
#endif

void TestApp::reload() {
    cache().reopen();
}

void TestApp::commit() {
    closeModes();
    setActionsEnabled( false );

    aptFront::Manager m;
    m.setProgressCallback( m_progress->callback() );
    m.setUpdateInterval( 100000 );
    try {
        m_stack->raiseWidget( m_progress );
        m.download();
        m_stack->raiseWidget( m_commitProgress );
        m.commit();
    } catch ( exception::OperationCancelled ) {
    } catch (...) {
        KMessageBox::sorry(
            this, i18n( "There was an error commiting changes. "
                        "Possibly there was a problem downloading some "
                        "packages or the commit would break packages. " ),
            i18n( "Could not commit changes" ) );
        // FIXME: this should be handled by libapt-front
        cache::Global::get().reopen();
    }
    m_stack->raiseWidget( m_list );
    setActionsEnabled( true );
    notifyPostChange( 0 );
}

void TestApp::upgrade() {
    closeModes();
    cache::Global::get().state().upgrade();
}

void TestApp::distUpgrade() {
    closeModes();
    cache::Global::get().state().distUpgrade();
}

void TestApp::togglePreview() {
    if ( modeActive( Preview ) )
        closeModes();
    else
        openPreview();
}

void TestApp::toggleSources() {
    if ( modeActive( Sources ) )
        closeModes();
    else
        openSources();
}

void TestApp::toggleDownload() {
    toggleMode( Download );
}

void TestApp::toggleCommit() {
    toggleMode( Commit );
}

void TestApp::toggleMode( Mode m ) {
    if ( modeActive( m ) )
        closeModes();
    else
        openMode( m );
}

void TestApp::openPreview() {
    if ( openMode( Preview ) )
        m_preview->searchView()->lister()->scheduleRebuild();
}

void TestApp::closePreview() {
    if ( closeMode( Preview ) )
        m_list->searchView()->lister()->scheduleRebuild();
}

void TestApp::openSources() {
    if ( openMode( Sources ) )
        m_sources->reset();
}

void TestApp::closeSources() {
    closeMode( Sources );
}

bool TestApp::closeMode ( Mode m ) {
    if ( modeActive( m ) ) {
        m_stack->raiseWidget( m_list );
        modeAction( m )->setChecked( false );
        return true;
    }
    return false;
}

bool TestApp::openMode( Mode m ) {
    if ( !modeActive( m ) ) {
        closeModes();
        m_modesClosed->setChecked( false );
        kdDebug() << "openMode widget: " << modeWidget( m ) << endl;
        m_stack->raiseWidget( modeWidget( m ) );
        return true;
    }
    return false;
}

#include "app.moc"
