# Storage device related problems
# Author: Martin Pitt <martin.pitt@ubuntu.com>
# (C) 2009 Canonical Ltd.
# License: GPL v2 or later.

from glob import glob
import subprocess
import apport.hookutils
import time
import os

description = 'External or internal storage devices (e. g. USB sticks)'

def run(report, ui):
    problem = ui.choice('What particular problem do you observe?',
        ['Removable storage device is not mounted automatically',
         'Internal hard disk partition cannot be mounted manually',
         'Internal hard disk partition is not displayed in Places menu',
         'No permission to access files on storage device',
         'Documents do not work on storage device',
         'Other problem',
        ])

    if problem is None:
        raise StopIteration
    problem = problem[0]

    if problem == 0:
        return problem_removable(report, ui)
    if problem == 3:
        return problem_permission(report, ui)
    if problem == 4:
        return get_desktop_vfs(ui)
    if problem == 5:
        ui.information('Please use "ubuntu-bug <packagename>" to report a bug against the particular package')
        return None

    return 'devicekit-disks'

def problem_removable(report, ui):
    ui.information('Please disconnect the problematic device now if it is still plugged in.')

    dk_mon = subprocess.Popen(['devkit-disks', '--monitor-detail'],
            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    udev_mon = subprocess.Popen(['udevadm', 'monitor', '--udev', '-e'],
            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    old_mounts = set(open('/proc/mounts').readlines())
    old_devices = set(glob('/dev/sd*'))

    ui.information('Please connect the problematic device now.')
    time.sleep(10)

    new_mounts = set(open('/proc/mounts').readlines())
    new_devices = set(glob('/dev/sd*'))

    dk_mon.terminate()
    out, err = dk_mon.communicate()
    report['DKDisksMonitorLog'] = out
    if err:
        report['DKDisksMonitorError'] = err

    udev_mon.terminate()
    out, err = udev_mon.communicate()
    report['UdevMonitorLog'] = out
    if err:
        report['UdevMonitorError'] = err

    new_devices = new_devices - old_devices
    new_mounts = new_mounts - old_mounts

    report['HotplugNewDevices'] = ' '.join(new_devices)
    report['HotplugNewMounts'] = '\n'.join(new_mounts)

    apport.hookutils.attach_dmesg(report)

    if not new_devices:
        return apport.packaging.get_kernel_package()

    if new_mounts:
        if ui.yesno('The plugged in device was automounted:\n\n%s\n'
                'Do you still need to report a problem about this?' % report['HotplugNewMounts']):
            return 'devicekit-disks'
        else:
            raise StopIteration

    if 'SUBSYSTEM=block' not in report['UdevMonitorLog']:
        return 'udev'

    for d in new_devices:
        if 'DEVNAME=' + d not in report['UdevMonitorLog']:
            return 'udev'
        if ' %s\n' % d not in report['DKDisksMonitorLog']:
            return 'devicekit-disks'
    return get_desktop_vfs(ui)

def problem_permission(report, ui):
    '''No permission to access files on storage device'''

def get_desktop_vfs(ui):
    if subprocess.call(['pgrep', '-u', str(os.getuid()), '-x', 'gnome-session']) == 0:
        return 'gvfs'
    if subprocess.call(['pgrep', '-u', str(os.getuid()), '-x', 'ksmserver']) == 0:
        return 'kdelibs5'

    ui.information('Sorry, you are not running GNOME or KDE. Automounting needs to be provided by your desktop environment.')
    raise StopIteration
