#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Provides unit tests for the APT worker."""
# Copyright (C) 2011 Sebastian Heinlein <devel@glatzor.de>
#
# Licensed under the GNU General Public License Version 2
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# Licensed under the GNU General Public License Version 2

__author__  = "Sebastian Heinlein <devel@glatzor.de>"

import glob
import os
import shutil
import unittest2

import dbus
import gobject


import aptdaemon.test
from aptdaemon.worker import AptWorker
from aptdaemon.core import Transaction
from aptdaemon import enums, errors

REPO_PATH = os.path.join(aptdaemon.test.get_tests_dir(), "repo")


class MockQueue(object):

    """A fake TransactionQueue which only provides a limbo attribute."""

    def __init__(self):
        self.limbo = {}


class WorkerTestCase(aptdaemon.test.AptDaemonTestCase):

    """Test suite for the worker which performs the actual package
    installation and removal."""

    def setUp(self):
        self.chroot = aptdaemon.test.Chroot()
        self.chroot.setup()
        self.addCleanup(self.chroot.remove)
        self.start_dbus_daemon()
        self.dbus = dbus.bus.BusConnection(self.dbus_address)
        self.loop = gobject.MainLoop()
        self.queue = MockQueue()
        self.worker = AptWorker(self.chroot.path)
        self.worker.connect("transaction-done", lambda w,t: self.loop.quit())

    def test_update_cache(self):
        """Test updating the cache using a local repository."""
        # Add a working and a non-working repository
        self.chroot.add_trusted_key()
        path = os.path.join(self.chroot.path,
                            "etc/apt/sources.list.d/test.list")
        with open(path, "w") as part_file:
            part_file.write("deb file://%s ./" % REPO_PATH)
        self.chroot.add_repository("/does/not/exist", copy_list=False)
        # Only update the repository from the working snippet
        trans = Transaction(enums.ROLE_UPDATE_CACHE, self.queue, os.getuid(),
                            "org.debian.apt.test", bus=self.dbus,
                            kwargs={"sources_list": "test.list"})
        self.worker.simulate(trans)
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.worker._cache.open()
        self.assertEqual(len(self.worker._cache), 9)
        pkg = self.worker._cache["silly-base"]
        self.assertTrue(pkg.candidate.origins[0].trusted)

    def test_upgrade_system(self):
        """Test upgrading the system."""
        self.chroot.add_test_repository()
        self.chroot.install_debfile(os.path.join(REPO_PATH,
                                                 "silly-base_0.1-0_all.deb"))
        # Install the package
        trans = Transaction(enums.ROLE_UPGRADE_SYSTEM, self.queue, os.getuid(),
                            "org.debian.apt.test", bus=self.dbus,
                            kwargs={"safe_mode": False})
        self.worker.simulate(trans)
        self.assertEqual(trans.depends[enums.PKGS_UPGRADE],
                         ["silly-base=0.1-0update1"])
        self.assertTrue(trans.space > 0)
        self.assertTrue(trans.download == 0)
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.worker._cache.open()
        self.assertEqual(self.worker._cache["silly-base"].installed.version,
                         "0.1-0update1")

    def test_check_unauth(self):
        """Test if packages from an unauthenticated repo are detected."""
        self.chroot.add_test_repository(copy_sig=False)
        # Install the package
        trans = Transaction(enums.ROLE_INSTALL_PACKAGES, self.queue,
                            os.getuid(), "org.debian.apt.test", bus=self.dbus,
                            packages=[["silly-base"],[],[],[],[], []])
        self.worker.simulate(trans)
        trans.allow_unauthenticated = False
        self.assertEqual(trans.unauthenticated, ["silly-base"])
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_FAILED)
        self.assertEqual(trans.error.code, enums.ERROR_PACKAGE_UNAUTHENTICATED)

        # Allow installation of unauthenticated packages
        trans = Transaction(enums.ROLE_INSTALL_PACKAGES, self.queue,
                            os.getuid(), "org.debian.apt.test", bus=self.dbus,
                            packages=[["silly-base"],[],[],[],[], []])
        trans.allow_unauthenticated = True
        self.worker.simulate(trans)
        self.assertEqual(trans.unauthenticated, ["silly-base"])
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.worker._cache.open()
        self.assertTrue(self.worker._cache["silly-base"].is_installed)

    def test_install(self):
        """Test installation of a package from a repository."""
        self.chroot.add_test_repository()
        # Install the package
        trans = Transaction(enums.ROLE_INSTALL_PACKAGES, self.queue,
                            os.getuid(), "org.debian.apt.test", bus=self.dbus,
                            packages=[["silly-depend-base"],[],[],[],[], []])
        self.worker.simulate(trans)
        self.assertEqual(trans.depends[enums.PKGS_INSTALL],
                         ["silly-base=0.1-0update1"])
        self.assertTrue(trans.space > 0)
        self.assertTrue(trans.download == 0)
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.worker._cache.open()
        self.assertTrue(self.worker._cache["silly-depend-base"].is_installed)

    def test_remove(self):
        """Test the removal of packages."""
        for pkg in ["silly-base_0.1-0_all.deb", "silly-essential_0.1-0_all.deb",
                    "silly-depend-base_0.1-0_all.deb"]:
            self.chroot.install_debfile(os.path.join(REPO_PATH, pkg))
        trans = Transaction(enums.ROLE_REMOVE_PACKAGES, self.queue, os.getuid(),
                            "org.debian.apt.test", bus=self.dbus,
                            packages=[[],[],["silly-base"],[],[],[]])
        self.worker.simulate(trans)
        self.assertEqual(trans.depends[enums.PKGS_REMOVE],
                         ["silly-depend-base=0.1-0"])
        self.assertTrue(trans.space < 0)
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.worker._cache.open()
        try:
            installed = self.worker._cache["silly-depend-base"].is_installed
            self.assertFalse(installed)
        except KeyError:
            pass
        # Don't allow to remove essential packages
        trans = Transaction(enums.ROLE_REMOVE_PACKAGES, self.queue, os.getuid(),
                            "org.debian.apt.test", bus=self.dbus,
                            packages=[[],[],["silly-essential"],[],[],[]])
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_FAILED,
                         "Allowed to remove an essential package")
        self.assertEqual(trans.error.code,
                         enums.ERROR_NOT_REMOVE_ESSENTIAL_PACKAGE,
                         "Allowed to remove an essential package")

    def test_downgrade(self):
        """Test downgrading of packages."""
        self.chroot.add_test_repository()
        pkg = os.path.join(REPO_PATH, "silly-base_0.1-0update1_all.deb")
        self.chroot.install_debfile(pkg)
        trans = Transaction(enums.ROLE_COMMIT_PACKAGES, self.queue,
                            os.getuid(), "org.debian.apt.test", bus=self.dbus,
                            packages=[[],[],[],[],[],["silly-base=0.1-0"]])
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.worker._cache.open()
        self.assertEqual(self.worker._cache["silly-base"].installed.version,
                         "0.1-0", "Failed to downgrade.")

    @unittest2.skip("KNOWN TO NOT WORK")
    def test_purge(self):
        """Test the purging of packages."""
        for pkg in ["silly-base_0.1-0_all.deb", "silly-config_0.1-0_all.deb"]:
            self.chroot.install_debfile(os.path.join(REPO_PATH, pkg))
        trans = Transaction(enums.ROLE_REMOVE_PACKAGES, self.queue, os.getuid(),
                            "org.debian.apt.test", bus=self.dbus,
                            packages=[[],[],[],["silly-config"],[],[]])
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.assertFalse(os.path.exists(os.path.join(self.chroot.path,
                                                     "etc/silly-packages.cfg")),
                         "Configuration file wasn't removed.")

    def test_install_file(self):
        """Test the installation of a local package file."""
        # add custom lintian file
        target = os.path.join(self.chroot.path, "usr", "share", "aptdaemon")
        os.makedirs(target)
        for tags_file in glob.glob(os.path.join(aptdaemon.test.get_tests_dir(),
                                                "../data/lintian*tags*")):
            shutil.copy(tags_file, target)
        # test
        self.chroot.add_test_repository()
        pkg = os.path.join(
            REPO_PATH, "silly-depend-base-lintian-broken_0.1-0_all.deb")
        trans = Transaction(enums.ROLE_INSTALL_FILE, self.queue, os.getuid(),
                            "org.debian.apt.test", bus=self.dbus,
                            kwargs={"path": os.path.join(REPO_PATH, pkg),
                                    "force": False})
        try:
            self.worker.simulate(trans)
        except errors.TransactionFailed, error:
            self.assertEqual(error.code, enums.ERROR_INVALID_PACKAGE_FILE)
        else:
            raise Exception("Lintian failed to detect a broken package")
        # Now allow to install invalid packages
        trans.kwargs["force"] = True
        self.worker.simulate(trans)
        self.assertEqual(trans.depends[enums.PKGS_INSTALL],
                         ["silly-base=0.1-0update1"])
        self.assertTrue(trans.space > 0)
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.worker._cache.open()
        self.assertTrue(self.worker._cache["silly-depend-base-lintian-broken"].is_installed)

    def test_install_unknown_file(self):
        """Test the installation of a local package file which is not known
        to the cache.

        Regression test for LP #702217
        """
        pkg = os.path.join(REPO_PATH, "silly-base_0.1-0_all.deb")
        trans = Transaction(enums.ROLE_INSTALL_FILE, self.queue, os.getuid(),
                            "org.debian.apt.test", bus=self.dbus,
                            kwargs={"path": os.path.join(REPO_PATH, pkg),
                                    "force": True})
        self.worker.simulate(trans)
        self.assertEqual(trans.packages, (["silly-base"], [], [], [], [], []))
        self.assertTrue(trans.space > 0)
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.worker._cache.open()
        self.assertTrue(self.worker._cache["silly-base"].is_installed)

    def test_fix_broken_depends(self):
        """Test the fixing of broken dependencies."""
        for pkg in ["silly-base_0.1-0_all.deb", "silly-broken_0.1-0_all.deb"]:
            self.chroot.install_debfile(os.path.join(REPO_PATH, pkg), True)
        trans = Transaction(enums.ROLE_FIX_BROKEN_DEPENDS, self.queue,
                            os.getuid(), "org.debian.apt.test",
                            bus=self.dbus)
        self.worker.simulate(trans)
        self.assertEqual(trans.depends[enums.PKGS_REMOVE],
                         ["silly-broken=0.1-0"])
        self.worker.run(trans)
        self.loop.run()
        self.assertEqual(trans.exit, enums.EXIT_SUCCESS,
                         "%s: %s" % (trans._error_property[0],
                                     trans._error_property[1]))
        self.worker._cache.open()
        self.assertEqual(self.worker._cache.broken_count, 0)


if __name__ == "__main__":
    unittest2.main()

# vim: ts=4 et sts=4
