// vs_table.cc        -*-c++-*-
//
//  Implementation of the vs_table class

#include "vs_table.h"

#include "vscreen.h"

#include "config/keybindings.h"

#include <sigc++/bind.h>
#include <sigc++/object_slot.h>

using namespace std;

keybindings *vs_table::bindings=NULL;

vs_table::vs_table()
  :rowsep(0), colsep(0)
{
  do_layout.connect(slot(*this, &vs_table::layout_me));
  focus=children.end();

  focussed.connect(slot(*this, &vs_table::got_focus));
  unfocussed.connect(slot(*this, &vs_table::lost_focus));
}

void vs_table::set_rowsep(int n)
{
  if(n!=rowsep)
    {
      rowsep=n;
      if(get_visible())
	vscreen_update();
    }
}

void vs_table::set_colsep(int n)
{
  if(n!=colsep)
    {
      colsep=n;
      if(get_visible())
	vscreen_update();
    }
}

// We need to call get_focus() here to update the "focus" pointer.
void vs_table::got_focus()
{
  vscreen_widget *w=get_focus();

  if(w)
    w->focussed();
}

void vs_table::lost_focus()
{
  vscreen_widget *w=get_focus();

  if(w)
    w->unfocussed();
}

int vs_table::num_rows()
{
  int max_rows=0;

  for(childlist::iterator i=children.begin(); i!=children.end(); i++)
    if(i->row_start+i->row_span>max_rows)
      max_rows=i->row_start+i->row_span;

  return max_rows;
}

int vs_table::num_cols()
{
  int max_cols=0;

  for(childlist::iterator i=children.begin(); i!=children.end(); i++)
    if(i->col_start+i->col_span>max_cols)
      max_cols=i->col_start+i->col_span;

  return max_cols;
}

void vs_table::add_widget(vscreen_widget *w, int row_start, int col_start, int row_span, int col_span, bool expand, bool shrink)
{
  int opts=ALIGN_CENTER;
  if(expand)
    opts|=EXPAND;
  if(shrink)
    opts|=SHRINK;

  add_widget_opts(w, row_start, col_start, row_span, col_span, opts, opts);
}

void vs_table::add_widget_opts(vscreen_widget *w, int row_start, int col_start, int row_span, int col_span, int xopts, int yopts)
{
  // sanity check
  for(childlist::iterator i=children.begin(); i!=children.end(); i++)
    if(i->w==w)
      // FIXME: throw something/print a nasty error message?
      abort();

  SigC::Connection shown_conn=w->shown_sig.connect(bind(slot(*this, &vs_table::show_widget), w));
  SigC::Connection hidden_conn=w->hidden_sig.connect(bind(slot(*this, &vs_table::hide_widget), w));

  children.push_back(child_info(w, row_start, col_start, row_span, col_span, xopts, yopts, shown_conn, hidden_conn));
  w->set_owner(this);

  if(focus==children.end() && w->focus_me() && w->get_visible())
    {
      focus=children.end();
      focus--;

      if(get_isfocussed())
	focus->w->focussed();
    }

  vscreen_queuelayout();
}

void vs_table::hide_widget(vscreen_widget *w)
{
  if(w==focus->w)
    {
      if(get_isfocussed())
	focus->w->unfocussed();

      focus++;

      while(focus!=children.end() && focus->w->focus_me())
	focus++;

      if(focus==children.end())
	{
	  focus=children.begin();

	  while(focus!=children.end() && focus->w->focus_me())
	    focus++;
	}

      if(focus!=children.end() && get_isfocussed())
	focus->w->focussed();
    }

  vscreen_queuelayout();
}

void vs_table::show_widget(vscreen_widget *w)
{
  if(w->focus_me() && focus==children.end())
    {
      for(childlist::iterator i=children.begin(); i!=children.end(); i++)
	if(i->w==w)
	  {
	    focus=i;
	    if(get_isfocussed())
	      focus->w->focussed();
	    break;
	  }
    }

  vscreen_queuelayout();
}

void vs_table::add_widget(vscreen_widget *w)
{
  add_widget(w, num_rows(), 0, 1);
}

void vs_table::rem_widget(vscreen_widget *w)
{
  for(childlist::iterator i=children.begin(); i!=children.end(); i++)
    if(i->w==w)
      {
	if(i==focus)
	  {
	    if(get_isfocussed())
	      focus->w->unfocussed();

	    focus++;

	    while(focus!=children.end() && focus->w->focus_me())
	      focus++;

	    if(focus==children.end())
	      {
		focus=children.begin();

		while(focus!=children.end() && focus->w->focus_me())
		  focus++;
	      }

	    if(focus!=children.end() && get_isfocussed())
	      focus->w->focussed();
	  }

	i->shown_conn.disconnect();
	i->hidden_conn.disconnect();

	children.erase(i);
	vscreen_queuelayout();
	w->set_owner(NULL);
	return;
      }
}

void vs_table::focus_widget(vscreen_widget *w)
{
  for(childlist::iterator i=children.begin(); i!=children.end(); i++)
    if(i->w==w)
      {
	if(i!=focus)
	  {
	    assert(i->w->get_visible() && i->w->focus_me());

	    if(focus!=children.end() && get_isfocussed())
	      focus->w->unfocussed();

	    focus=i;

	    if(get_isfocussed())
	      focus->w->focussed();

	    vscreen_update();
	  }
	return;
      }
}

vscreen_widget *vs_table::get_focus()
{
  if(focus!=children.end() && focus->w->focus_me())
    return focus->w;
  else
    {
      if(focus!=children.end() && get_isfocussed())
	focus->w->unfocussed();

      for(focus=children.begin(); focus!=children.end(); focus++)
	{
	  if(focus->w->focus_me())
	    {
	      focus->w->focussed();
	      return focus->w;
	    }
	}

      return NULL;
    }
}

void vs_table::show_all()
{
  for(childlist::iterator i=children.begin(); i!=children.end(); i++)
    i->w->show_all();

  show();
}

class vs_table::better_fit
{
  const child_info &base;

  int dx;
  int dy;

  // the table dimensions
  int width;
  int height;
public:
  better_fit(const child_info &c,
	     int _dx, int _dy, int _width, int _height)
    :base(c), dx(_dx), dy(_dy), width(_width), height(_height)
  {
  }

  // Basically operate by shifting the world so that the base object lies in
  // the lower-right corner, then working from there.
  inline bool operator()(const childlist::iterator &a,
			 const childlist::iterator &b)
  {
    int aminx=a->col_start-base.col_start-base.col_span;
    if(aminx<0)
      aminx+=width;

    int aminy=a->row_start-base.row_start-base.row_span;
    if(aminy<0)
      aminy+=height;

    int bminx=b->col_start-base.col_start-base.col_span;
    if(bminx<0)
      bminx+=width;

    int bminy=b->row_start-base.row_start-base.row_span;
    if(bminy<0)
      bminy+=height;

    int amaxy=a->row_start+a->row_span-base.row_start-base.row_span-1;
    if(amaxy<0)
      amaxy+=height;

    int bmaxy=b->row_start+b->row_span-base.row_start-base.row_span-1;
    if(bmaxy<0)
      bmaxy+=height;

    int amaxx=a->col_start+a->col_span-base.col_start-base.col_span-1;
    if(amaxx<0)
      amaxx+=width;

    int bmaxx=b->col_start+b->col_span-base.col_start-base.col_span-1;
    if(bmaxx<0)
      bmaxx+=width;

    if(dy==0)
      {
	if(dx>0)
	  {
	    if(aminx<bminx)
	      return true;
	    else if(aminx>bminx)
	      return false;
	  }
	else
	  {
	    if(bmaxx<amaxx)
	      return true;
	    else if(bmaxx>amaxx)
	      return false;
	  }

	int besty=(height-base.row_span)/2;

	int adiff=abs((aminy+amaxy)/2-besty);
	int bdiff=abs((bminy+bmaxy)/2-besty);

	if(adiff<bdiff)
	  return true;
	else if(adiff>bdiff)
	  return false;

	// ERRRR, they're exactly the same?

	return false;
      }
    else
      {
	if(dy>0)
	  {
	    if(aminy<bminy)
	      return true;
	    else if(aminy>bminy)
	      return false;
	  }
	else
	  {
	    if(bmaxy<amaxy)
	      return true;
	    else if(bmaxy>amaxy)
	      return false;
	  }

	int bestx=(width-base.col_span)/2;

	int adiff=abs((aminx+amaxx)/2-bestx);
	int bdiff=abs((bminx+bmaxx)/2-bestx);

	if(adiff<bdiff)
	  return true;
	else if(adiff>bdiff)
	  return false;

	return false;
      }

  }
};

// a simple predicate--separated from the function below to keep
// find_best_focus's high-level logic as clean as possible.
// Checks whether the given child lies within the "shadow" of the given
// base widget in the given direction.

// There are two cases to worry about: either the "beginning" of the child is
// within the base, or it isn't.
// If it is, we can determine that the child does overlap the base in that
// dimension.
// If it is not, and it lies before the "beginning" of the base, we have to
// check its ending.  If its ending lies after the "beginning" of the base,
// there is (again) clearly overlap.  Otherwise, there is not.
// If it is not, and it lies after the "ending" of the base, there is no
// overlap.

inline bool vs_table::lies_on_axis(const child_info &base,
				   bool horizontal,
				   const child_info &c)
{
  if(horizontal)
    return (c.row_start<=base.row_start && c.row_start+c.row_span-1>=base.row_start) ||
      (c.row_start>=base.row_start && c.row_start<=base.row_start+base.row_span-1);
  else
    return (c.col_start<=base.col_start && c.col_start+c.col_span-1>=base.col_start) ||
      (c.col_start>=base.col_start && c.col_start<=base.col_start+base.col_span-1);
}

// FIXME: either dx or dy must be 0; only their signs are checked..
vs_table::childlist::iterator vs_table::find_best_focus(childlist::iterator start,
							int dx,
							int dy)
{
  assert(start!=children.end());
  assert(dx==0 || dy==0);
  assert(!(dx==dy));

  list<childlist::iterator> sorted_children;

  for(childlist::iterator i=children.begin();
      i!=children.end();
      i++)
    if(i!=start && i->w->get_visible() &&
       i->w->focus_me() && lies_on_axis(*start, (dy==0), *i))
       sorted_children.push_back(i);

  if(sorted_children.size()==0)
    return start;

  sorted_children.sort(better_fit(*start, dx, dy, num_cols(), num_rows()));

  return sorted_children.front();
}

bool vs_table::handle_char(chtype ch)
{
  if(focus!=children.end())
    {
      if(focus->w->dispatch_char(ch))
	return true;
      else if(bindings->key_matches(ch, "Cycle"))
	{
	  childlist::iterator oldfocus=focus;

	  focus++;

	  while(focus!=children.end() && !focus->w->focus_me())
	    focus++;

	  if(focus==children.end())
	    {
	      focus=children.begin();
	      while(focus!=children.end() && !focus->w->focus_me())
		focus++;
	    }

	  if(focus!=children.end() && focus!=oldfocus)
	    {
	      if(get_isfocussed())
		{
		  oldfocus->w->unfocussed();
		  focus->w->focussed();
		}
	      vscreen_updatecursor();
	    }

	  return focus!=oldfocus;
	}
      else if(bindings->key_matches(ch, "Left"))
	{
	  childlist::iterator oldfocus=focus;

	  focus=find_best_focus(focus, -1, 0);

	  if(focus!=children.end() && focus!=oldfocus)
	    {
	      if(get_isfocussed())
		{
		  oldfocus->w->unfocussed();
		  focus->w->focussed();
		}
	      vscreen_updatecursor();
	    }

	  return focus!=oldfocus;
	}
      else if(bindings->key_matches(ch, "Right"))
	{
	  childlist::iterator oldfocus=focus;

	  focus=find_best_focus(focus, 1, 0);

	  if(focus!=children.end() && focus!=oldfocus)
	    {
	      if(get_isfocussed())
		{
		  oldfocus->w->unfocussed();
		  focus->w->focussed();
		}
	      vscreen_updatecursor();
	    }

	  return focus!=oldfocus;
	}
      else if(bindings->key_matches(ch, "Up"))
	{
	  childlist::iterator oldfocus=focus;

	  focus=find_best_focus(focus, 0, -1);

	  if(focus!=children.end() && focus!=oldfocus)
	    {
	      if(get_isfocussed())
		{
		  oldfocus->w->unfocussed();
		  focus->w->focussed();
		}
	      vscreen_updatecursor();
	    }

	  return focus!=oldfocus;
	}
      else if(bindings->key_matches(ch, "Down"))
	{
	  childlist::iterator oldfocus=focus;

	  focus=find_best_focus(focus, 0, 1);

	  if(focus!=children.end() && focus!=oldfocus)
	    {
	      if(get_isfocussed())
		{
		  oldfocus->w->unfocussed();
		  focus->w->focussed();
		}
	      vscreen_updatecursor();
	    }

	  return focus!=oldfocus;
	}
      else
	return vs_passthrough::handle_char(ch);
    }
  else
    return vs_passthrough::handle_char(ch);
}

class vs_table::nrow_lt
{
public:
  inline bool operator()(const child_info &a,
			 const child_info &b)
  {
    return a.row_span<b.row_span;
  }
};

class vs_table::ncol_lt
{
public:
  inline bool operator()(const child_info &a,
			 const child_info &b)
  {
    return a.col_span<b.col_span;
  }
};

void vs_table::calc_initial_sizes(int *row_sizes, int *col_sizes, int nrows, int ncols, size *final_size)
{
  // When we're trying to allocate space for multi-row/column widgets, only
  // allocate it in rows/columns that contain widgets of a lower order
  // already, if possible.
  bool *row_expandable=new bool[nrows];
  bool *col_expandable=new bool[ncols];

  for(int i=0; i<nrows; i++)
    {
      row_sizes[i]=0;
      row_expandable[i]=false;
    }
  for(int i=0; i<ncols; i++)
    {
      col_sizes[i]=0;
      col_expandable[i]=false;
    }

  childlist sorted_children;

  // Calculate requested sizes for all the visible children and add them to
  // the list
  for(childlist::iterator i=children.begin();
      i!=children.end();
      i++)
    if(i->w->get_visible())
      {
	i->request_size=i->w->size_request();
	sorted_children.push_back(*i);
      }

  // Sort the children according to how many rows they span
  sorted_children.sort(nrow_lt());

  // Find out which rows should be expanded.
  for(childlist::iterator i=sorted_children.begin();
      i!=sorted_children.end();
      i++)
    if(i->y_options&EXPAND)
      {
	bool not_expanded=true;

	for(int j=i->row_start; j<i->row_start+i->row_span; j++)
	  if(row_expandable[j])
	    {
	      not_expanded=false;
	      break;
	    }

	if(not_expanded)
	  for(int j=i->row_start; j<i->row_start+i->row_span; j++)
	    row_expandable[j]=true;
      }

  // Now we iterate over that list in increasing order, expanding rows as
  // necessary.
  for(childlist::iterator i=sorted_children.begin();
      i!=sorted_children.end();
      i++)
    {
      // First, calculate the available space (so we know whether we *need* to
      // expand a row)
      int curspace=0;
      int n=0;
      for(int j=i->row_start; j<i->row_start+i->row_span; j++)
	{
	  curspace+=row_sizes[j];
	  if(row_expandable[j])
	    ++n;
	}

      if(n==0)
	// If it's completely un-expandable, give up and expand equally
	// at all points.  (we don't have a constrained size; figuring out
	// whether to expand a given row/column is just done so we give
	// priority to expandable items)
	n=i->row_span;

      // If necessary, do the expansion
      if(curspace<i->request_size.h)
	{
	  int shortfall=i->request_size.h-curspace;
	  for(int j=i->row_start; n>0; n--, j++)
	    {
	      int amt=shortfall/n;

	      row_sizes[j]+=amt;
	      shortfall-=amt;
	    }
	}
    }

  // Similarly for columns
  // Sort the children according to how many columns they span
  sorted_children.sort(ncol_lt());

  // Find out which columns should be expanded.
  for(childlist::iterator i=sorted_children.begin();
      i!=sorted_children.end();
      i++)
    if(i->x_options & EXPAND)
      {
	bool not_expanded=true;

	for(int j=i->col_start; j<i->col_start+i->col_span; j++)
	  if(col_expandable[j])
	    {
	      not_expanded=false;
	      break;
	    }

	if(not_expanded)
	  for(int j=i->col_start; j<i->col_start+i->col_span; j++)
	    col_expandable[j]=true;
      }

  // Now we iterate over that list in increasing order, expanding columns as
  // necessary.
  for(childlist::iterator i=sorted_children.begin();
      i!=sorted_children.end();
      i++)
    {
      // First, calculate the available space (so we know whether we *need* to
      // expand a column)
      int curspace=0;
      int n=0;
      for(int j=i->col_start; j<i->col_start+i->col_span; j++)
	{
	  curspace+=col_sizes[j];
	  if(col_expandable[j])
	    ++n;
	}

      if(n==0)
	// If it's completely un-expandable, give up and expand equally
	// at all points.  (we don't have a constrained size; figuring out
	// whether to expand a given row/column is just done so we give
	// priority to expandable items)
	n=i->col_span;

      // If necessary, do the expansion
      if(curspace<i->request_size.w)
	{
	  int shortfall=i->request_size.w-curspace;
	  for(int j=i->col_start; n>0; n--, j++)
	    {
	      int amt=shortfall/n;

	      col_sizes[j]+=amt;
	      shortfall-=amt;
	    }
	}
    }

  final_size->h=final_size->w=0;

  for(int i=0; i<nrows; i++)
    final_size->h+=row_sizes[i];
  for(int j=0; j<ncols; j++)
    final_size->w+=col_sizes[j];

  delete[] row_expandable;
  delete[] col_expandable;
}

size vs_table::size_request()
{
  int nrows=num_rows(), ncols=num_cols();
  int *row_sizes=new int[nrows], *col_sizes=new int[ncols];

  size final_size(0,0);

  calc_initial_sizes(row_sizes, col_sizes, nrows, ncols, &final_size);

  delete[] row_sizes;
  delete[] col_sizes;

  return final_size;
}

// FIXME: the shrinking code incorrectly handles the case where a widget that
// cannot be shrunk spans several rows which in total require more space than
// it.  The rows should and can be shrunk (the unshrinkable rows will just have
// less excess) but aren't right now because they are marked as unshrinkable.
//
// Solution -- don't mark rows as unshrinkable simply because something
// unshrinkable is in them; make sure that the unshrinkable thing is at its
// minimum size before doing that.  Also requires some trickery that I don't
// have an elegant solution to yet to make sure that the unshrinkable stuff isn't
// shrunk by accident.
void vs_table::layout_me()
{
  // Check whether we need to update the focus state.
  //
  // Layout changes can be triggered when a widget becomes focussable or
  // non-focussable.
  get_focus();

  if(get_win())
    {
      int nrows=num_rows(), ncols=num_cols();
      int *row_sizes=new int[nrows], *col_sizes=new int[ncols];
      size final_size(0,0);
      int width=getmaxx()-colsep*max(ncols-1, 0);
      int height=getmaxy()-rowsep*max(nrows-1, 0);

      calc_initial_sizes(row_sizes, col_sizes, nrows, ncols, &final_size);

      if(final_size.h<height)
	{
	  // We were allocated more space than we requested.
	  // Find out how many rows can be expanded
	  bool *can_expand=new bool[nrows];
	  int n_expandable=0;
	  int overflow=height-final_size.h;

	  for(int i=0; i<nrows; i++)
	    can_expand[i]=false;

	  childlist sorted_children;

	  for(childlist::iterator i=children.begin();
	      i!=children.end();
	      ++i)
	    if(i->w->get_visible() && (i->y_options & EXPAND))
	      sorted_children.push_back(*i);

	  sorted_children.sort(nrow_lt());

	  for(childlist::iterator i=sorted_children.begin();
	      i!=sorted_children.end();
	      ++i)
	    {
	      // Only try to expand for this widget if no expandable
	      // row exists yet.
	      bool not_expanded=true;

	      for(int j=i->row_start; j<i->row_start+i->row_span; j++)
		if(can_expand[j])
		  {
		    not_expanded=false;
		    break;
		  }

	      if(not_expanded)
		for(int j=i->row_start; j<i->row_start+i->row_span; j++)
		  can_expand[j]=true;
	    }

	  for(int i=0; i<nrows; i++)
	    if(can_expand[i])
	      n_expandable++;

	  for(int i=0; i<nrows && n_expandable>0; i++)
	    if(can_expand[i])
	      {
		int amt=overflow/n_expandable;
		row_sizes[i]+=amt;
		overflow-=amt;
		n_expandable--;
	      }

	  delete[] can_expand;
	}
      else if(final_size.h>height)
	{
	  // We weren't allocated enough space :(
	  // Try to shrink as much as possible.  (never shrink anything below
	  // a single character cell, though) (if no children occupy a
	  // row/column, it'll already be size 0 anyway)
	  bool *can_shrink=new bool[nrows];
	  int n_shrinkable=0;
	  int shortfall=final_size.h-height;

	  // To save time, don't even bother trying to shrink unshrinkable
	  // rows..
	  for(int i=0; i<nrows; i++)
	    can_shrink[i]=(row_sizes[i]>1);

	  for(childlist::iterator i=children.begin();
	      i!=children.end();
	      i++)
	    if(i->w->get_visible())
	      {
		if(!(i->y_options & SHRINK))
		  {
		    for(int j=i->row_start; j<i->row_start+i->row_span; j++)
		      can_shrink[j]=false;
		  }
	      }

	  for(int i=0; i<nrows; i++)
	    if(can_shrink[i])
	      ++n_shrinkable;

	  // The GTK+ algorithm for shrinking rows is quite clever; I've
	  // shamelessly stolen^H^H^H^H^H^Hborrowed it.
	  //
	  // Basically, we make repeated attempts to shrink stuff until
	  // we can no longer shrink anything.  Usually, the first pass should
	  // be enough, but in extreme cases we might iterate many
	  // times without reaching a conclusion..
	  while(n_shrinkable>0 && shortfall>0)
	    {
	      int toshrink=n_shrinkable;

	      for(int i=0; i<nrows && toshrink>0; i++)
		if(can_shrink[i])
		  {
		    int amt=shortfall/toshrink;
		    if(row_sizes[i]-amt<1)
		      amt=row_sizes[i]-1;
		    row_sizes[i]-=amt;
		    shortfall-=amt;
		    toshrink--;

		    if(row_sizes[i]<2)
		      {
			can_shrink[i]=false;
			n_shrinkable--;
		      }
		  }
	    }

	  // eek!  Just clip the end of the table, it's all we can do.. :(
	  if(shortfall>0)
	    {
	      int cursize=0;

	      int i=0;
	      while(i<nrows && cursize+row_sizes[i]<height)
		cursize+=row_sizes[i++];

	      assert(i<nrows);

	      row_sizes[i++]=height-cursize;

	      while(i<nrows)
		row_sizes[i++]=0;
	    }

	  delete[] can_shrink;
	}

      // Perform exactly the same calculations on columns
      if(final_size.w<width)
	{
	  // We were allocated more space than we requested.
	  // Find out how many columns can be expanded
	  bool *can_expand=new bool[ncols];
	  int n_expandable=0;
	  int overflow=width-final_size.w;

	  for(int i=0; i<ncols; i++)
	    can_expand[i]=false;

	  childlist sorted_children;

	  for(childlist::iterator i=children.begin();
	      i!=children.end();
	      ++i)
	    if(i->w->get_visible() && (i->x_options & EXPAND))
	      sorted_children.push_back(*i);

	  sorted_children.sort(ncol_lt());

	  for(childlist::iterator i=sorted_children.begin();
	      i!=sorted_children.end();
	      ++i)
	    {
	      bool not_expanded=true;

	      for(int j=i->col_start; j<i->col_start+i->col_span; j++)
		if(can_expand[j])
		  {
		    not_expanded=false;
		    break;
		  }

	      if(not_expanded)
		for(int j=i->col_start; j<i->col_start+i->col_span; j++)
		  can_expand[j]=true;
	    }

	  for(int i=0; i<ncols; i++)
	    if(can_expand[i])
	      n_expandable++;

	  for(int i=0; i<ncols && n_expandable>0; i++)
	    if(can_expand[i])
	      {
		int amt=overflow/n_expandable;
		col_sizes[i]+=amt;
		overflow-=amt;
		n_expandable--;
	      }

	  delete[] can_expand;
	}
      else if(final_size.w>width)
	{
	  // We weren't allocated enough space :(
	  // Try to shrink as much as possible.  (never shrink anything below
	  // a single character cell, though) (if no children occupy a
	  // row/column, it'll already be size 0 anyway)
	  bool *can_shrink=new bool[ncols];
	  int n_shrinkable=0;
	  int shortfall=final_size.w-width;

	  // To save time, don't even bother trying to shrink unshrinkable
	  // rows..
	  for(int i=0; i<ncols; i++)
	    can_shrink[i]=(col_sizes[i]>1);

	  for(childlist::iterator i=children.begin();
	      i!=children.end();
	      i++)
	    if(i->w->get_visible())
	      {
		if(!(i->x_options & SHRINK))
		  {
		    for(int j=i->col_start; j<i->col_start+i->col_span; j++)
		      can_shrink[j]=false;
		  }
	      }

	  for(int i=0; i<ncols; i++)
	    if(can_shrink[i])
	      ++n_shrinkable;

	  // The GTK+ algorithm for shrinking rows is quite clever; I've
	  // shamelessly stolen^H^H^H^H^H^Hborrowed it.
	  //
	  // Basically, we make repeated attempts to shrink stuff until
	  // we can no longer shrink anything.  Usually, the first pass should
	  // be enough, but in extreme cases we might iterate many
	  // times without reaching a conclusion..
	  while(n_shrinkable>0 && shortfall>0)
	    {
	      int toshrink=n_shrinkable;

	      for(int i=0; i<ncols && toshrink>0; i++)
		if(can_shrink[i])
		  {
		    int amt=shortfall/toshrink;
		    if(col_sizes[i]-amt<1)
		      amt=col_sizes[i]-1;
		    col_sizes[i]-=amt;
		    shortfall-=amt;
		    toshrink--;

		    if(col_sizes[i]<2)
		      {
			can_shrink[i]=false;
			n_shrinkable--;
		      }
		  }
	    }

	  // eek!  Just clip the end of the table, it's all we can do.. :(
	  if(shortfall>0)
	    {
	      int cursize=0;

	      int i=0;
	      while(i<ncols && cursize+col_sizes[i]<width)
		cursize+=col_sizes[i++];

	      assert(i<ncols);

	      col_sizes[i++]=width-cursize;

	      while(i<ncols)
		col_sizes[i++]=0;
	    }

	  delete[] can_shrink;
	}

      // Now actually allocate the sizes. (whew!)

      for(childlist::iterator i=children.begin(); i!=children.end(); i++)
	if(i->w->get_visible())
	  {
	    int x=0, y=0;
	    int w=0, h=0;

	    for(int j=0; j<i->row_start; j++)
	      y+=row_sizes[j]+rowsep;

	    for(int j=i->row_start; j<i->row_start+i->row_span; j++)
	      h+=row_sizes[j];

	    h+=rowsep*max(i->row_span-1, 0);

	    for(int j=0; j<i->col_start; j++)
	      x+=col_sizes[j]+colsep;

	    for(int j=i->col_start; j<i->col_start+i->col_span; j++)
	      w+=col_sizes[j];

	    w+=colsep*max(i->col_span-1, 0);

	    int curx=x, cury=y, curw=w, curh=h;
	    if(curw>i->request_size.w && !(i->x_options & EXPAND))
	      {
		if((i->x_options & ALIGN_CENTER) == ALIGN_CENTER)
		  curx+=(curw-i->request_size.w)/2;
		else if(i->x_options & ALIGN_RIGHT)
		  curx+=curw-i->request_size.w;

		curw=i->request_size.w;
	      }

	    if(curh>i->request_size.h && !(i->y_options & EXPAND))
	      {
		if((i->y_options & ALIGN_CENTER) == ALIGN_CENTER)
		  cury+=(curh-i->request_size.h)/2;
		else if(i->y_options & ALIGN_RIGHT)
		  cury+=curh-i->request_size.h;

		curh=i->request_size.h;
	      }

	    i->w->alloc_size(curx, cury, curw, curh);
	  }
	else
	  i->w->alloc_size(0, 0, 0, 0);

      delete[] row_sizes;
      delete[] col_sizes;
    }
  else
    for(childlist::iterator i=children.begin(); i!=children.end(); i++)
      i->w->alloc_size(0, 0, 0, 0);
}

void vs_table::paint()
{
  for(childlist::iterator i=children.begin(); i!=children.end(); i++)
    if(i->w->get_visible())
      i->w->display();
}

void vs_table::dispatch_mouse(short id, int x, int y, int z, mmask_t bstate)
{
  for(childlist::iterator i=children.begin(); i!=children.end(); i++)
    {
      if(i->w->get_visible() && i->w->enclose(y, x))
	{
	  if(i->w->focus_me())
	    focus_widget(i->w);

	  i->w->dispatch_mouse(id, x-i->w->get_startx(), y-i->w->get_starty(),
			       z, bstate);
	  return;
	}
    }
}

void vs_table::init_bindings()
{
  bindings=new keybindings(global_bindings);
}
