// download_list.cc
//
//  Copyright 2001 Daniel Burrows

#include "aptitude.h"
#include "download_list.h"
#include "download_manager.h"
#include "ui.h"

#include <generic/apt.h>
#include <generic/config_signal.h>

#include <apt-pkg/acquire-worker.h>
#include <apt-pkg/acquire-item.h>
#include <apt-pkg/strutl.h>

#include <vscreen/vscreen.h>
#include <vscreen/vs_util.h>
#include <vscreen/config/colors.h>
#include <vscreen/config/keybindings.h>

#include <sigc++/bind.h>
#include <sigc++/object_slot.h>

#include <algorithm>

using namespace std;

// FIXME: HACK
static void download_summary_nasty_hack(SigC::Slot0<void> cancel_slot)
{
  if(cancel_slot)
    cancel_slot();

  vscreen_exitmain();
}

// Unfortunately the cancel_slot is necessary so we know we've cancelled..
static vscreen_widget *download_summary(download_list *l,
					bool already_cancelled,
					Slot0Arg cancel_slot,
					double FetchedBytes,
					unsigned long ElapsedTime,
					double CurrentCPS)
{
  char buf[256];

  snprintf(buf, 256, _("Downloaded %sB in %s (%sB/s)."), SizeToStr(FetchedBytes).c_str(), TimeToStr(ElapsedTime).c_str(), SizeToStr(CurrentCPS).c_str());

  vscreen_widget *rval;
  if(already_cancelled || !cancel_slot)
    rval=vs_dialog_ok(buf, SigC::slot(vscreen_exitmain));
  else
    rval=vs_dialog_yesno(buf,
			 SigC::slot(vscreen_exitmain), _("Continue"),
			 SigC::bind(SigC::slot(download_summary_nasty_hack),
				    cancel_slot), _("Cancel"),
			 get_color("DefaultWidgetBackground")|A_REVERSE);

  rval->connect_key("PrevPage",
		    &global_bindings,
		    slot(*l, &download_list::pageup));
  rval->connect_key("Up",
		    &global_bindings,
		    slot(*l, &download_list::lineup));
  rval->connect_key("NextPage",
		    &global_bindings,
		    slot(*l, &download_list::pagedown));
  rval->connect_key("Down",
		    &global_bindings,
		    slot(*l, &download_list::linedown));
  rval->connect_key("Begin",
		    &global_bindings,
		    slot(*l, &download_list::skip_to_top));
  rval->connect_key("End",
		    &global_bindings,
		    slot(*l, &download_list::skip_to_bottom));
  rval->connect_key("Left",
		    &global_bindings,
		    slot(*l, &download_list::shift_left));
  rval->connect_key("Right",
		    &global_bindings,
		    slot(*l, &download_list::shift_right));

  rval->show_all();

  return rval;
}
				 

download_list::download_list(Slot0Arg _abortslot, bool _display_messages)
  :start(0), sticky_end(true), cancelled(false), abortslot(_abortslot),
   display_messages(_display_messages),

   startx(0),

   TotalItems(0), CurrentItems(0),
   CurrentCPS(0), TotalBytes(0), CurrentBytes(0)
{
  do_layout.connect(slot(*this, &download_list::layout_me));
}

// FIXME: just discarding this is not only bad -- it's also dangerous: it's a
// *huge* potential memory leak.
void download_list::destroy()
{
  cancel();
}

// all that CRITICAL_ENTER/CRITICAL_EXIT stuff is left out until I'm sure I
// really need it.
void download_list::paint()
{
  int y=0;
  unsigned int where=start;
  int width,height;
  getmaxyx(height, width);

  // Display the completed items
  while(y<height-1 && where<msgs.size())
    {
      std::string disp(msgs[where].first,
		       min<string::size_type>(startx, msgs[where].first.size()));

      show_string_as_progbar(0, y, disp,
			     msgs[where].second, msgs[where].second,
			     width, width);

      y++;
      where++;
    }

  // Display the currently active workers
  where=0;
  while(y<height-1 && where<workers.size())
    {
      int width1;

      if(workers[where].total!=0)
	width1=(int) ((((double) width)*(double) workers[where].current)/(double) workers[where].total);
      else
	width1=0;

      if(width1>width)
	width1=width;

      string disp(workers[where].msg,
		  min<string::size_type>(startx, workers[where].msg.size()));

      show_string_as_progbar(0, y, disp,
			     get_color("DownloadProgress"), get_bg(),
			     width1, width);

      y++;
      where++;
    }

  // status line.
  string output="";

  output+=_("Total Progress: ");

  int barsize=0;

  if((TotalBytes+TotalItems)>0)
    {
      char progress_string[50]; // More'n enough chars.
      unsigned long ETA=0;
      if(CurrentCPS>0)
	// Be careful!  CurrentCPS might be 0.
	ETA=(unsigned long)((TotalBytes-CurrentBytes)/CurrentCPS);
      // Straight from acqprogress.cc, that is..

      barsize=int((double(width*(CurrentBytes+CurrentItems)))/(TotalBytes+TotalItems));
      if(barsize>width)
	barsize=width;

      attrset(get_color("ScreenStatusColor"));
      if(CurrentCPS>0)
	snprintf(progress_string, 50, _(" [ %i%% ] (%sB/s, %s remaining)"), int(double((100.0*(CurrentBytes+CurrentItems)))/(TotalBytes+TotalItems)), SizeToStr(CurrentCPS).c_str(), TimeToStr(ETA).c_str());
      else if(CurrentBytes>0 || CurrentItems>0)
	snprintf(progress_string, 50, _(" [ %i%% ] (stalled)"), int(double((100.0*(CurrentBytes+CurrentItems)))/(TotalBytes+TotalItems)));
      else
	snprintf(progress_string, 50, _(" [ %i%% ]"), int(double((100.0*(CurrentBytes+CurrentItems)))/(TotalBytes+TotalItems)));


      output+=progress_string;
    }

  show_string_as_progbar(0,
			 height-1,
			 output,
			 get_color("Progress"),
			 get_color("ScreenStatusColor"),
			 barsize,
			 width);
}

void download_list::update_workers(pkgAcquire *Owner)
{
  int width,height;
  getmaxyx(height, width);

  workers.erase(workers.begin(), workers.end());

  // FIXME: do "something" if there are no active workers
  pkgAcquire::Worker *serf=Owner->WorkersBegin();
  while(serf)
    {
      if(!serf->CurrentItem)
	{
	  if(!serf->Status.empty())
	    {
	      workers.push_back(workerinf(serf->Status, 0, 1));
	      vscreen_queuelayout();
	    }
	}
      else
	{
	  pkgAcquire::ItemDesc *item=serf->CurrentItem;
	  string output=(item->Owner->Status==pkgAcquire::Item::StatFetching)?item->ShortDesc:item->Description+": ";

	  char intbuf[50]; // Waay more than enough.

	  sprintf(intbuf,
		  " [ %sB/%sB ]",
		  SizeToStr(serf->CurrentSize).c_str(),
		  SizeToStr(serf->TotalSize).c_str());

	  output+=intbuf;

	  workers.push_back(workerinf(output,
				      serf->CurrentSize,
				      serf->TotalSize));
	  vscreen_queuelayout();
	}

      serf=Owner->WorkerStep(serf);
    }

  sync_top();
}

// I'm not sure this works!
bool download_list::MediaChange(string media, string drive,
				download_manager &manager)
{
  char buf[512];

  snprintf(buf, 512,
	   _("Please insert the following disc into the drive \"%s\":\n%s"),
	   drive.c_str(), media.c_str());

  vscreen_widget *w=vs_dialog_ok(buf, SigC::slot(vscreen_exitmain),
				 get_color("MediaChange"));
  w->show_all();
  popup_widget(w);

  vscreen_mainloop();

  return true;
}

void download_list::IMSHit(pkgAcquire::ItemDesc &itmdesc,
			   download_manager &manager)
{
  if(display_messages)
    {
      msgs.push_back(pair<string, int>(itmdesc.Description+" "+_("[Hit]"),
					    get_color("DownloadHit")));

      sync_top();

      vscreen_queuelayout();
      vscreen_tryupdate();
    }
}

void download_list::Fetch(pkgAcquire::ItemDesc &itmdesc,
			  download_manager &manager)
{
}

void download_list::Done(pkgAcquire::ItemDesc &itmdesc,
			 download_manager &manager)
{
  if(display_messages)
    {
      msgs.push_back(pair<string, int>(itmdesc.Description+" "+_("[Downloaded]"),
				       get_color("DownloadProgress")));

      sync_top();

      vscreen_queuelayout();
      vscreen_tryupdate();
    }
}

void download_list::Fail(pkgAcquire::ItemDesc &itmdesc,
			 download_manager &manager)
{
  if(display_messages)
    {
      if(itmdesc.Owner->Status==pkgAcquire::Item::StatIdle)
	return;

      // ???
      if(itmdesc.Owner->Status==pkgAcquire::Item::StatDone)
	msgs.push_back(pair<string, int>(itmdesc.Description+" "+_("[IGNORED]"),
					      get_color("DownloadHit")));
      else
	{
	  msgs.push_back(pair<string, int>(itmdesc.Description+" "+_("[ERROR]"),
						get_color("Error")));
	  msgs.push_back(pair<string, int>(" "+itmdesc.Owner->ErrorText,
						get_color("Error")));
	}

      sync_top();

      vscreen_queuelayout();
      vscreen_tryupdate();
    }
}

void download_list::Start(download_manager &manager)
{
  // Delete stuff from previous runs (eg, for multiple CDs)
  workers.erase(workers.begin(), workers.end());
  msgs.erase(msgs.begin(), msgs.end());
}

void download_list::Stop(download_manager &manager)
{
  if(aptcfg->FindB(PACKAGE "::UI::Pause-After-Download", true))
    {
      popup_widget(download_summary(this,
				    cancelled,
				    abortslot,
				    manager.get_fetched_bytes(),
				    manager.get_elapsed_time(),
				    manager.get_currentCPS()));
      vscreen_mainloop();
    }
}

void download_list::Complete(download_manager &manager)
{
  // Really destroy ourselves.
  vscreen_widget::destroy();
}

bool download_list::Pulse(pkgAcquire *Owner, download_manager &manager)
{
  TotalBytes=manager.get_total_bytes();
  TotalItems=manager.get_total_items();
  CurrentBytes=manager.get_current_bytes();
  CurrentItems=manager.get_current_items();
  CurrentCPS=manager.get_currentCPS();

  update_workers(Owner);

  vscreen_queuelayout(); // Force an update

  while(vscreen_poll()) // Eat all pending input.
    ;

  vscreen_tryupdate(); // Update if we haven't yet.

  if(cancelled)
    return false;
  else
    return true;
}

size download_list::size_request()
{
  return size(40, msgs.size()+workers.size()+1);
}

bool download_list::handle_char(chtype ch)
{
  if(global_bindings.key_matches(ch, "NextPage"))
    pagedown();
  if(global_bindings.key_matches(ch, "Down"))
    linedown();
  else if(global_bindings.key_matches(ch, "PrevPage"))
    pageup();
  else if(global_bindings.key_matches(ch, "Up"))
    lineup();
  else if(global_bindings.key_matches(ch, "Begin"))
    skip_to_top();
  else if(global_bindings.key_matches(ch, "End"))
    skip_to_bottom();
  else if(global_bindings.key_matches(ch, "Left"))
    shift_left();
  else if(global_bindings.key_matches(ch, "Right"))
    shift_right();
  else
    return vscreen_widget::handle_char(ch);

  return true;
}

void download_list::pageup()
{
  if(start>(unsigned) getmaxy())
    {
      start-=getmaxy()-1;

      vscreen_update();
      vscreen_tryupdate();
    }
  else
    {
      start=0;

      vscreen_update();
      vscreen_tryupdate();
    }

  sticky_end=false;
}

void download_list::lineup()
{
  if(start>0)
    {
      --start;

      vscreen_update();
      vscreen_tryupdate();
    }

  sticky_end=false;
}

void download_list::pagedown()
{
  if(start<msgs.size()+workers.size()-getmaxy())
    {
      start+=getmaxy();
      vscreen_update();
      vscreen_tryupdate();
    }
  else
    sticky_end=true;
}

void download_list::linedown()
{
  if(start<msgs.size()+workers.size()-getmaxy())
    {
      ++start;
      vscreen_update();
      vscreen_tryupdate();
    }
  else
    sticky_end=true;
}

void download_list::skip_to_bottom()
{
  if(start+getmaxy()<msgs.size()+workers.size())
    start=msgs.size()+workers.size()-getmaxy();

  sticky_end=true;

  vscreen_update();
  vscreen_tryupdate();
}

void download_list::skip_to_top()
{
  start=0;
  sticky_end=false;

  vscreen_update();
  vscreen_tryupdate();
}

void download_list::shift_left()
{
  if(startx>0)
    {
      startx-=8;
      vscreen_update();
      vscreen_tryupdate();
    }
}

void download_list::shift_right()
{
  string::size_type maxx=0;

  if(display_messages)
    for(vector<string>::size_type n=0; n<msgs.size(); ++n)
      maxx=max<int>(maxx, msgs[n].first.size());

  for(vector<string>::size_type n=0; n<workers.size(); ++n)
    maxx=max<int>(maxx, workers[n].msg.size());

  if(startx+8<maxx)
    {
      startx+=8;
      vscreen_update();
      vscreen_tryupdate();
    }
}

void download_list::layout_me()
{
  sync_top();
}

void download_list::sync_top()
{
  if(getmaxy()==0)
    return;

  if(sticky_end)
    {
      unsigned int optimal_start=(msgs.size()+workers.size()+1);

      if(optimal_start>=(unsigned) getmaxy())
	{
	  optimal_start-=getmaxy();

	  if(start<optimal_start)
	    start=optimal_start;
	}
    }
}
