// pkg_view.cc
//
//  Copyright 2000-2002 Daniel Burrows

#include "pkg_view.h"

#include "aptitude.h"

#include "desc_parse.h"
#include "edit_pkg_hier.h"
#include "menu_redirect.h"
#include "pkg_columnizer.h"
#include "reason_fragment.h"
#include "ui.h"

#include <vscreen/fragment.h>
#include <vscreen/vs_label.h>
#include <vscreen/vs_multiplex.h>
#include <vscreen/vs_scrollbar.h>
#include <vscreen/vs_table.h>
#include <vscreen/vs_text_layout.h>
#include <vscreen/vscreen_widget.h>
#include <vscreen/config/keybindings.h>

#include <generic/apt.h>
#include <generic/config_signal.h>

#include <apt-pkg/error.h>

#include <sigc++/bind.h>
#include <sigc++/object_slot.h>

#include <ctype.h>

#include <string>

using namespace std;

class pkg_handling_label:public vs_label
{
  column_definition_list *columns;

  bool have_pkg;
  pkgCache::PkgIterator pkg;
  pkgCache::VerIterator ver;

  void zap_package()
  {
    have_pkg=false;
  }

public:
  pkg_handling_label(column_definition_list *_columns)
    :vs_label(" "), columns(_columns), have_pkg(false)
  {
    cache_closed.connect(SigC::slot(*this, &pkg_handling_label::zap_package));
  }

  ~pkg_handling_label() {delete columns;}

  size size_request() {return size(1,1);}

  void set_columns(column_definition_list *_columns)
  {
    delete columns;
    columns=_columns;
    vscreen_update();
  }

  void do_columnify(const pkgCache::PkgIterator &_pkg,
		    const pkgCache::VerIterator &_ver)
  {
    pkg=_pkg;
    ver=_ver;

    have_pkg=!pkg.end();

    vscreen_update();
  }

  void paint()
  {
    if(apt_cache_file)
      {
	// Needed to initialize translated widths and stuff.
	pkg_item::pkg_columnizer::setup_columns();

	if(!have_pkg)
	  {
	    pkg=pkgCache::PkgIterator();
	    // Reinitialize it all the time to avoid the "magic autochanging
	    // pointers" bug.
	    ver=pkgCache::VerIterator(*apt_cache_file);
	  }

	set_text(pkg_item::pkg_columnizer(pkg, ver, *columns, 0).layout_columns(getmaxx()));
      }
    else
      set_text("");

    vs_label::paint();
  }
};

static void do_set_column_format(string key, string the_default,
				 pkg_handling_label *l)
{
  string format=aptcfg->Find(key, the_default.c_str());

  column_definition_list *columns=parse_columns(format,
						pkg_item::pkg_columnizer::parse_column_type,
						pkg_item::pkg_columnizer::defaults);

  if(!columns)
    _error->Error("Couldn't parse column definition");
  else
    l->set_columns(columns);
}

class pkg_description_widget:public vs_text_layout
{
  int start_line;
public:
  void set_package(const pkgCache::PkgIterator &pkg,
		   const pkgCache::VerIterator &ver)
  {
    string newdesc;

    if(!pkg.end() && !ver.end())
      newdesc=apt_package_records->Lookup(ver.FileList()).LongDesc();
    else
      newdesc="";

    set_fragment(make_desc_fragment(newdesc));
  }
};

// In order to properly dispatch line-up/down events to the sub-widgets,
// we need a meta-widget that knows about them.
//
// Note: this is not the most efficient way of doing things, but
// (a) it would be very error-prone to reproduce the multiplex's
//     behavior for this special case, and
// (b) only one of these is created per package view, so the overhead
//     of a few redundant pointers is acceptable.  In my opinion.
//
// This is still rather gross, and a better way would be nice.
class info_area_multiplex:public vs_multiplex
{
  vs_hier_editor *editor;
  pkg_description_widget *description;
  vs_table *description_table;
  vs_text_layout *reasons;
  vs_table *reasons_table;

  pkgCache::PkgIterator lastPkg;
  pkgCache::VerIterator lastVer;
  string lastDesc;

  /** True if the package had breakage the last time we checked. */
  bool hadBreakage;
  /** If the view was autoswitched to breakage reasons, this is set
   *  to the widget we switched away from; otherwise, it is \b NULL.
   */
  vscreen_widget *autoswitch;
public:
  info_area_multiplex(vs_hier_editor *_editor,
		      pkg_description_widget *_description,
		      vs_table *_description_table,
		      vs_text_layout *_reasons,
		      vs_table *_reasons_table)
    :vs_multiplex(),
     editor(_editor),
     description(_description), description_table(_description_table),
     reasons(_reasons), reasons_table(_reasons_table),
     hadBreakage(false), autoswitch(NULL)
  {
    package_states_changed.connect(slot(*this,
					&info_area_multiplex::reset_package));
  }

  void line_up()
  {
    vscreen_widget *w=visible_widget();

    if(w==description_table)
      description->line_up();
    else if(w==reasons_table)
      reasons->line_up();
  }

  void line_down()
  {
    vscreen_widget *w=visible_widget();

    if(w==description_table)
      description->line_down();
    else if(w==reasons_table)
      reasons->line_down();
  }

  void set_package(const pkgCache::PkgIterator &pkg,
		   const pkgCache::VerIterator &ver)
  {
    bool hasBreakage;

    description->set_package(pkg, ver);
    reasons->set_fragment(reason_fragment(pkg, hasBreakage));
    editor->set_package(pkg, ver);

    // autoswitch if a package is newly broken, or if we have just
    // moved to a broken package.
    if(hasBreakage &&
       (!hadBreakage ||
	!(pkg==lastPkg && ver==lastVer)) &&
       aptcfg->FindB(PACKAGE "::UI::Auto-Show-Reasons", true))
      {
	// Don't clobber the autoswitch bread crumb if we were
	// autoswitched and are still autoswitched.
	if(!autoswitch)
	  autoswitch=visible_widget();
	reasons_table->show();
      }

    // We always set the package anyway in case something changed,
    // but only scroll to the top in this case:
    if(pkg!=lastPkg || ver!=lastVer)
      {
	lastPkg=pkg;
	lastVer=ver;

	description->move_to_top();
	reasons->move_to_top();
      }
    // If we are autoswitched and unbroken, switch back.
    else if(!hasBreakage && autoswitch)
      {
	autoswitch->show();
	autoswitch=NULL;
      }

    hadBreakage=hasBreakage;
  }

  /** Cycles the multiplex, taking autoswitch behavior into account. */
  void cycle()
  {
    if(autoswitch && autoswitch!=visible_widget())
      {
	autoswitch->show();
	autoswitch=NULL;
      }
    else
      {
	autoswitch=NULL;
	cycle_forward();
      }
  }

  /** Re-updates the package views, given that the package state may
   *  have changed.
   */
  void reset_package()
  {
    set_package(lastPkg, lastVer);
  }

  void set_description(string s)
  {
    if(s!=lastDesc)
      {
	lastDesc=s;

	description->set_fragment(make_desc_fragment(s));
	reasons->set_fragment(sequence_fragment(make_desc_fragment(s),
						newline_fragment(),
						nopackage(),
						NULL));

	description->move_to_top();
	reasons->move_to_top();
      }
  }
};

static bool do_menu_test(vscreen_widget *view,
			 menu_redirect *redirect,
			 bool (menu_redirect::* test)())
{
  return active_main_widget()==view &&
    (redirect->*test)();
}

static bool do_menu_action(vscreen_widget *view,
			   menu_redirect *redirect,
			   bool (menu_redirect::* action)())
{
  if(active_main_widget()==view)
    return (redirect->*action)();
  else
    return false;
}

vscreen_widget *make_package_view(list<package_view_item> &format,
				  vscreen_widget *mainwidget,
				  menu_redirect *menu_handler,
				  pkg_signal *sig, desc_signal *desc_sig,
				  bool show_reason_first)
{
  bool found_mainwidget=false;

  vs_table *rval=new vs_table;

  assert(mainwidget);

  for(list<package_view_item>::iterator i=format.begin();
      i!=format.end();
      i++)
    {
      switch(i->type)
	{
	case PACKAGE_VIEW_MAINWIDGET:
	  if(found_mainwidget)
	    _error->Error(_("make_package_view: error in arguments -- two main widgets??"));
	  else
	    i->widget=mainwidget;
	  break;
	case PACKAGE_VIEW_STATIC:
	  if(!i->columns)
	    _error->Error(_("make_package_view: error in arguments -- bad column list for static item"));
	  else
	    {
	      pkg_handling_label *l=new pkg_handling_label(i->columns);
	      i->widget=l;

	      if(sig)
		sig->connect(slot(*l, &pkg_handling_label::do_columnify));

	      if(!i->columns_cfg.empty())
		aptcfg->connect(i->columns_cfg,
				SigC::bind(bind(SigC::slot(do_set_column_format),
						l),
					   i->columns_cfg,
					   i->columns_cfg_default));
	    }
	  break;
	case PACKAGE_VIEW_DESCRIPTION:
	  {
	    vs_hier_editor *e=new vs_hier_editor;
	    pkg_description_widget *w=new pkg_description_widget;
	    vs_text_layout *l=new vs_text_layout;

	    vs_table *wt=new vs_table;
	    vs_table *lt=new vs_table;
	    info_area_multiplex *m=new info_area_multiplex(e,
							   w, wt,
							   l, lt);
	    vs_scrollbar *ws=new vs_scrollbar(vs_scrollbar::VERTICAL);
	    vs_scrollbar *ls=new vs_scrollbar(vs_scrollbar::VERTICAL);

	    w->location_changed.connect(slot(*ws, &vs_scrollbar::set_slider));
	    l->location_changed.connect(slot(*ls, &vs_scrollbar::set_slider));

	    wt->add_widget_opts(w, 0, 0, 1, 1, vs_table::EXPAND, vs_table::EXPAND);
	    wt->add_widget_opts(ws, 0, 1, 1, 1, 0, vs_table::EXPAND);

	    lt->add_widget_opts(l, 0, 0, 1, 1, vs_table::EXPAND, vs_table::EXPAND);
	    lt->add_widget_opts(ls, 0, 1, 1, 1, 0, vs_table::EXPAND);

	    // HACK: speaks for itself
	    vs_tree *thetree=dynamic_cast<vs_tree *>(mainwidget);

	    i->widget=m;

	    // Set up a null reason to start with.  (the package
	    // signal is only called if an actual package is
	    // highlighted)
	    l->set_fragment(nopackage());

	    if(sig)
	      sig->connect(slot(*m, &info_area_multiplex::set_package));

	    if(desc_sig)
	      desc_sig->connect(slot(*m, &info_area_multiplex::set_description));

	    mainwidget->connect_key("DescriptionDown", &global_bindings,
				    slot(*m,
					 &info_area_multiplex::line_down));
	    mainwidget->connect_key("DescriptionUp", &global_bindings,
				    slot(*m,
					 &info_area_multiplex::line_up));
	    mainwidget->connect_key("DescriptionCycle", &global_bindings,
				    slot(*m, &info_area_multiplex::cycle));
	    mainwidget->connect_key("EditHier", &global_bindings,
				    slot(*e, &vscreen_widget::show));
	    mainwidget->connect_key("EditHier", &global_bindings,
				    slot(*m, &vscreen_widget::show));
	    mainwidget->connect_key("EditHier", &global_bindings,
				    bind(slot(*rval, &vs_table::focus_widget),
					 m));

	    e->hidden_sig.connect(bind(slot(*rval, &vs_table::focus_widget),
				       mainwidget));

	    if(thetree)
	      e->commit_changes.connect(slot(*thetree, &vs_tree::line_down));

	    m->add_visible_widget(e, false);
	    m->add_visible_widget(wt, true);
	    m->add_visible_widget(lt, true);

	    if(show_reason_first)
	      lt->show();
	    else
	      wt->show();

	    // FIXME: this is a grotesque hack.
	    if(mainwidget->get_visible())
	      rval->focus_widget(mainwidget);
	  }
	  break;
	default:
	  _error->Error(_("make_package_view: bad argument!"));
	  break;
	}

      if(i->widget)
	{
	  rval->add_widget_opts(i->widget, i->row, i->col, i->h, i->w,
				i->xopts, i->yopts);

	  i->widget->set_bg(i->bg);

	  if(i->popupdownkey.size()>0)
	    rval->connect_key(i->popupdownkey,
			      &global_bindings,
			      slot(*i->widget,
				   &vscreen_widget::toggle_visible));

	  if(i->visible)
	    i->widget->show();
	}
    }

  // Slow, but the list is (hopefully) << 10 elements or so.
  for(list<package_view_item>::iterator i=format.begin();
      i!=format.end();
      i++)
    if(i->popupdownlinked.size()>0)
      for(list<package_view_item>::iterator j=format.begin();
	  j!=format.end();
	  j++)
	{
	  if(!strcasecmp(j->name.c_str(), i->popupdownlinked.c_str()))
	    {
	      // Having to make two connections is annoying.
	      j->widget->shown_sig.connect(slot(*i->widget, &vscreen_widget::show));
	      j->widget->hidden_sig.connect(slot(*i->widget, &vscreen_widget::hide));
	      break;
	    }
	}

  if(!mainwidget)
    _error->Error(_("make_package_view: no main widget found"));

  if(menu_handler)
    {
      // Connect the appropriate signals..

      // TODO: need to change the condition for enabling the menu to
      // check whether the overall widget is visible...
      package_menu_enabled.connect(bind(bind(SigC::slot(do_menu_test),
					     &menu_redirect::package_enabled),
					rval,
					menu_handler));

      package_install.connect(bind(bind(SigC::slot(do_menu_action),
					&menu_redirect::package_install),
				   rval,
				   menu_handler));

      package_remove.connect(bind(bind(SigC::slot(do_menu_action),
				       &menu_redirect::package_remove),
				  rval,
				  menu_handler));

      package_purge.connect(bind(bind(SigC::slot(do_menu_action),
				      &menu_redirect::package_purge),
				 rval,
				 menu_handler));

      package_hold.connect(bind(bind(SigC::slot(do_menu_action),
				     &menu_redirect::package_hold),
				rval,
				menu_handler));

      package_keep.connect(bind(bind(SigC::slot(do_menu_action),
				     &menu_redirect::package_keep),
				rval,
				menu_handler));

      package_mark_auto.connect(bind(bind(SigC::slot(do_menu_action),
				     &menu_redirect::package_mark_auto),
				rval,
				menu_handler));

      package_unmark_auto.connect(bind(bind(SigC::slot(do_menu_action),
					    &menu_redirect::package_unmark_auto),
				       rval,
				       menu_handler));

      package_forbid_enabled.connect(bind(bind(SigC::slot(do_menu_test),
					       &menu_redirect::package_forbid_enabled),
					  rval,
					  menu_handler));

      package_forbid.connect(bind(bind(SigC::slot(do_menu_action),
				     &menu_redirect::package_forbid),
				rval,
				menu_handler));

      package_information_enabled.connect(bind(bind(SigC::slot(do_menu_test),
						    &menu_redirect::package_information_enabled),
					       rval,
					       menu_handler));

      package_information.connect(bind(bind(SigC::slot(do_menu_action),
					    &menu_redirect::package_information),
				       rval,
				       menu_handler));

      package_changelog_enabled.connect(bind(bind(SigC::slot(do_menu_test),
						  &menu_redirect::package_changelog_enabled),
					     rval,
					     menu_handler));

      package_changelog.connect(bind(bind(SigC::slot(do_menu_action),
					  &menu_redirect::package_changelog),
				     rval, menu_handler));

      find_search_enabled.connect(bind(bind(SigC::slot(do_menu_test),
					    &menu_redirect::find_search_enabled),
				       rval,
				       menu_handler));

      find_search.connect(bind(bind(SigC::slot(do_menu_action),
				    &menu_redirect::find_search),
			       rval,
			       menu_handler));

      find_research_enabled.connect(bind(bind(SigC::slot(do_menu_test),
					      &menu_redirect::find_research_enabled),
					 rval,
					 menu_handler));

      find_research.connect(bind(bind(SigC::slot(do_menu_action),
				      &menu_redirect::find_research),
				 rval,
				 menu_handler));

      find_limit_enabled.connect(bind(bind(SigC::slot(do_menu_test),
					   &menu_redirect::find_limit_enabled),
				      rval,
				      menu_handler));

      find_limit.connect(bind(bind(SigC::slot(do_menu_action),
				   &menu_redirect::find_limit),
			      rval,
			      menu_handler));

      find_cancel_limit_enabled.connect(bind(bind(SigC::slot(do_menu_test),
						  &menu_redirect::find_reset_limit_enabled),
					     rval,
					     menu_handler));

      find_cancel_limit.connect(bind(bind(SigC::slot(do_menu_action),
					  &menu_redirect::find_reset_limit),
				     rval,
				     menu_handler));

      find_broken_enabled.connect(bind(bind(SigC::slot(do_menu_test),
					    &menu_redirect::find_broken_enabled),
				       rval,
				       menu_handler));

      find_broken.connect(bind(bind(SigC::slot(do_menu_action),
				    &menu_redirect::find_broken),
			       rval,
			       menu_handler));
    }

  return rval;
}
