// ui.cc
//
//  Copyright 2000-2002 Daniel Burrows <Daniel_Burrows@brown.edu>
//
//  Various UI glue-code and routines.

#include "ui.h"

#include <sigc++/bind.h>
#include <sigc++/object_slot.h>

#include <apt-pkg/acquire.h>
#include <apt-pkg/clean.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/error.h>

#include <errno.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include <utility>

#include "../config.h"
#include "aptitude.h"

#include "apt_options.h"
#include "defaults.h"
#include "load_config.h"
#include "load_grouppolicy.h"
#include "load_pkgview.h"

#include <vscreen/curses++.h>
#include <vscreen/fragment.h>
#include <vscreen/vscreen.h>
#include <vscreen/vs_button.h>
#include <vscreen/vs_center.h>
#include <vscreen/vs_editline.h>
#include <vscreen/vs_frame.h>
#include <vscreen/vs_label.h>
#include <vscreen/vs_menubar.h>
#include <vscreen/vs_multiplex.h>
#include <vscreen/vs_pager.h>
#include <vscreen/vs_scrollbar.h>
#include <vscreen/vs_stacked.h>
#include <vscreen/vs_statuschoice.h>
#include <vscreen/vs_table.h>
#include <vscreen/vs_text_layout.h>
#include <vscreen/vs_tree.h>
#include <vscreen/vs_util.h>

#include <mine/cmine.h>

#include <generic/apt.h>
#include <generic/apt_undo_group.h>
#include <generic/config_signal.h>

#include "dep_item.h"
#include "download.h"
#include "download_screen.h"
#include "download_bar.h"
#include "download_list.h"
#include "download_manager.h"
#include "pkg_columnizer.h"
#include "pkg_info_screen.h"
#include "pkg_grouppolicy.h"
#include "pkg_tree.h"
#include "pkg_ver_item.h"
#include "pkg_view.h"
#include "vs_progress.h"

using namespace std;

static vs_menubar *main_menu;
static vs_menu *views_menu;

static vs_stacked *main_stacked;

// Hmm, is this table the best idea?..
static vs_table *main_table;

static vs_multiplex *main_multiplex;
static vs_multiplex *main_status_multiplex;

// I think it's better to only have a single preview screen at once.  (note to
// self: data-segment stuff initializes to 0 already..)
static pkg_tree *active_preview_tree;
static vscreen_widget *active_preview;

// True if a download or package-list update is proceeding.  This hopefully will
// avoid the nasty possibility of collisions between them.
// FIXME: uses implicit locking -- if multithreading happens, should use a mutex
//       instead.
static bool active_download;

// While a status-widget download progress thingy is active, this will be
// set to it.
vscreen_widget *active_status_download;

SigC::Signal0<void> file_quit;
SigC::Signal0<void> package_states_changed;

SigC::Signal0<bool, MarshalOr> package_menu_enabled;
SigC::Signal0<bool, MarshalOr> package_forbid_enabled;
SigC::Signal0<bool, MarshalOr> package_information_enabled;
SigC::Signal0<bool, MarshalOr> package_changelog_enabled;

SigC::Signal0<bool, MarshalOr> find_search_enabled;
SigC::Signal0<bool, MarshalOr> find_research_enabled;
SigC::Signal0<bool, MarshalOr> find_limit_enabled;
SigC::Signal0<bool, MarshalOr> find_cancel_limit_enabled;
SigC::Signal0<bool, MarshalOr> find_broken_enabled;

SigC::Signal0<bool, MarshalOr> package_install;
SigC::Signal0<bool, MarshalOr> package_remove;
SigC::Signal0<bool, MarshalOr> package_purge;
SigC::Signal0<bool, MarshalOr> package_hold;
SigC::Signal0<bool, MarshalOr> package_keep;
SigC::Signal0<bool, MarshalOr> package_mark_auto;
SigC::Signal0<bool, MarshalOr> package_unmark_auto;
SigC::Signal0<bool, MarshalOr> package_forbid;
SigC::Signal0<bool, MarshalOr> package_information;
SigC::Signal0<bool, MarshalOr> package_changelog;

SigC::Signal0<bool, MarshalOr> find_search;
SigC::Signal0<bool, MarshalOr> find_research;
SigC::Signal0<bool, MarshalOr> find_limit;
SigC::Signal0<bool, MarshalOr> find_cancel_limit;
SigC::Signal0<bool, MarshalOr> find_broken;

const char *default_pkgstatusdisplay="%d";
const char *default_pkgheaderdisplay="%N %n #%B %u %o";
const char *default_grpstr="filter(missing),task,status,section(subdir,passthrough),section(topdir)";

class apt_error_pager:public vs_text_layout
{
  string::size_type maxlen;
public:
  apt_error_pager():vs_text_layout(), maxlen(0)
  {
    vector<fragment *> frags;

    if(_error->empty())
      frags.push_back(text_fragment(_("Er, there aren't any errors, this shouldn't have happened..")));
    else while(!_error->empty())
      {
	string currerr, tag;
	bool iserr=_error->PopMessage(currerr);
	if(iserr)
	  tag="E:";
	else
	  tag="W:";

	maxlen=max(maxlen, tag.size()+1+currerr.size());

	frags.push_back(indentbox(0, 3,
				  wrapbox(fragf("%s %s", tag.c_str(), currerr.c_str()))));
      }

    set_fragment(attr_fragment(sequence_fragment(frags),
			       get_color("Error")));
  }

  // Try to get a large enough size to display all errors with or
  // without wrapping.
  size size_request()
  {
    // HACK: I know the frame takes up a certain X space, so...
    //
    // The long-term solution is to decouple X and Y layout (ie, allocate
    // space in the X dimension in one step, then allocate space in the Y
    // direction once X has been finalized).  However, this takes a lot of
    // effort and testing and is not suitable for the current aptitude
    // release (which hopefully will be what goes into sarge)
    string::size_type top_w=rootwin.getmaxx();

    // Allow 2 characters for the frame and 1 for the scrollbar.
    if(top_w<3)
      top_w=0;
    else
      top_w-=3;

    // See how many lines we need to display things at the size of
    // the largest widget we know about.  Add extra padding to the length
    // because of the indentbox's behavior.
    return size(min(top_w, maxlen+3), height_for_width(top_w));
  }
};

// Search callback
static void pager_do_search(std::string s, vs_pager *p)
{
  p->search_for(s);
}

// Handles "search" dialogs for pagers
static void pager_search(vs_pager *p)
{
  prompt_string(_("Search for:"),
		p->get_last_search(),
		bind(slot(pager_do_search), p),
		NULL,
		NULL,
		NULL);
}

void check_apt_errors()
{
  if(!_error->empty())
    {
      vs_label *l=new vs_label("Apt errors");
      l->set_bg(get_color("Error"));
      apt_error_pager *p=new apt_error_pager;
      p->set_bg(get_color("Error"));
      vs_button *b=new vs_button("Ok");
      vs_table *t=new vs_table;
      t->set_bg(get_color("Error"));
      vs_center *c=new vs_center(new vs_frame(t));

      vs_scrollbar *scrl=new vs_scrollbar(vs_scrollbar::VERTICAL);
      p->location_changed.connect(slot(*scrl, &vs_scrollbar::set_slider));
      scrl->scrollbar_interaction.connect(slot(*p, &vs_text_layout::scroll));

      b->pressed.connect(slot(*c, &vs_center::destroy));
      t->add_widget(l, 0, 0, 1, 2, false);
      t->add_widget_opts(p, 1, 0, 1, 1,
			 vs_table::EXPAND,
			 vs_table::EXPAND);
      t->add_widget(scrl, 1, 1, 1, 1,
		    vs_table::ALIGN_CENTER,
		    vs_table::ALIGN_CENTER | vs_table::EXPAND);
      t->add_widget(new vs_center(b), 2, 0, 1, 2, false);
      t->connect_key("Confirm", &global_bindings, b->pressed.slot());

      main_stacked->add_visible_widget(c, true);
    }
}

// Runs a sub-aptitude with the same selections that the user
// has currently made, but as root.
//
// Because tagfiles don't work on FIFOs, and su closes all open fds,
// this is a lot hairier than it should be.
//
// This writes the current status to a file in ~/.aptitude/.tmp, then
// loads it in a su'd instance.  A FIFO is used to make the reader
// block until the writer is done writing.  Not foolproof but the user
// would have to go to a lot of pointless trouble to screw it up.

static void do_su_to_root(string args)
{
  if(getuid()==0)
    {
      show_message(_("You already are root!"));
      return;
    }

  // Copied from pkg_changelog.cc.  This should be its own routine?
  string fifoname="aptitudeXXXXXX", statusfile="aptitudeXXXXXX";
  if(getenv("HOME"))
    {
      string dotdir(getenv("HOME"));
      dotdir+="/.aptitude";
      if(mkdir(dotdir.c_str(), 0700)<0 && errno!=EEXIST)
	{
	  _error->Errno("mkdir", "%s", dotdir.c_str());
	  return;
	}
      string tmpdotdir(dotdir+"/.tmp");
      if(mkdir(tmpdotdir.c_str(), 0700)<0 && errno!=EEXIST)
	{
	  _error->Errno("mkdir", "%s", tmpdotdir.c_str());
	  return;
	}
      fifoname=tmpdotdir+"/"+fifoname;
      statusfile=tmpdotdir+"/"+statusfile;
    }
  else
    {
      _error->Warning("Can't get value of $HOME, using TMPDIR (insecure)");
      if(getenv("TMPDIR"))
	{
	  fifoname=getenv("TMPDIR")+("/"+fifoname);
	  statusfile=getenv("TMPDIR")+("/"+statusfile);
	}
      else
	{
	  fifoname="/tmp/"+fifoname;
	  statusfile="/tmp/"+statusfile;
	}
    }
  fifoname=mktemp((char *) fifoname.c_str());
  statusfile=mktemp((char *) statusfile.c_str());

  if(mkfifo(fifoname.c_str(), 0600)==-1)
    {
      _error->Error("Couldn't create temporary FIFO");
      return;
    }

  int pid=fork();

  if(pid==-1)
    _error->Error("Unable to fork: %s", strerror(errno));
  else if(pid==0) // I'm a child!
    {
      // Read one byte from the FIFO for synchronization
      char tmp;
      int fd=open(fifoname.c_str(), O_RDONLY);
      read(fd, &tmp, 1); // Will block until the other process writes.
      close(fd);

      // It's ok to use argv0 to generate the command,
      // since the user has to authenticate themselves as root (ie, if
      // they're going to do something evil that way they'd have su'd
      // directly)
      char cmdbuf[512];
      snprintf(cmdbuf, 512, "%s -S %s %s",
	       argv0, statusfile.c_str(), args.c_str());

      // I think I shall assume that su is in a sane place and behaves
      // sanely.
      execl("/bin/su", "/bin/su", "-c", cmdbuf, NULL);
    }
  else
    {
      int status;
      OpProgress foo; // Need a generic non-outputting progress bar

      // Save the selection list.
      (*apt_cache_file)->save_selection_list(foo, statusfile.c_str());

      // Shut curses down.
      vscreen_suspend();

      // Ok, wake the other process up.
      char tmp=0;
      int fd=open(fifoname.c_str(), O_WRONLY);
      write(fd, &tmp, 1);
      close(fd);

      // Wait for a while so we don't accidentally daemonize ourselves.
      while(waitpid(pid, &status, 0)!=pid)
	;

      unlink(fifoname.c_str());
      unlink(statusfile.c_str());

      if(!WIFEXITED(status) || WEXITSTATUS(status))
	{
	  _error->Error("%s",
			_("Subprocess exited with an error -- did you type your password correctly?"));
	  vscreen_resume();
	  check_apt_errors();
	  // We have to clear these out or the cache won't reload properly (?)
	  apt_reload_cache(gen_progress_bar(), true);
	}
      else
	exit(0);
    }
}

static bool su_to_root_enabled()
{
  return getuid()!=0;
}

static void update_menubar_autohide()
{
  main_menu->set_always_visible(main_multiplex->num_children()==0 ||
				!aptcfg->FindB(PACKAGE "::UI::Menubar-Autohide",
					       false));
}

vscreen_widget *reload_message;
static void do_show_reload_message()
{
  if(!reload_message)
    {
      vscreen_widget *w=new vs_frame(new vs_label(_("Loading cache")));
      reload_message=new vs_center(w);
      reload_message->show_all();
      popup_widget(reload_message);

      vscreen_tryupdate();
    }
}

static void do_hide_reload_message()
{
  if(reload_message)
    {
      reload_message->destroy();
      reload_message=NULL;
      vscreen_tryupdate();
    }
}

static void do_quit()
{
  if(aptcfg->FindB(PACKAGE "::UI::Prompt-On-Exit", true))
    prompt_yesno(_("Really quit Aptitude?"), false, file_quit.slot(), NULL);
  else
    file_quit();
}

static void do_destroy_visible()
{
  if(active_status_download)
    {
      active_status_download->destroy();
      active_status_download=NULL;
    }
  else if(aptcfg->FindB(PACKAGE "::UI::Exit-On-Last-Close", true) &&
     main_multiplex->num_children()<=1)
    do_quit();
  else
    {
      vscreen_widget *w=main_multiplex->visible_widget();
      if(w)
	w->destroy();

      // If all the screens are destroyed, we make the menu visible (so the
      // user knows that something is still alive :) )
      update_menubar_autohide();
    }
}

static bool view_next_prev_enabled()
{
  return main_multiplex->num_visible()>1;
}

static bool any_view_visible()
{
  return main_multiplex->visible_widget()!=NULL;
}

// These are necessary because main_multiplex isn't created until after
// the slot in the static initializer..
// (creating the menu info at a later point would solve this problem)
static void do_view_next()
{
  main_multiplex->cycle_forward();
}

static void do_view_prev()
{
  main_multiplex->cycle_backward();
}

static void do_show_ui_options_dlg()
{
  vscreen_widget *w=make_ui_options_dialog();
  main_stacked->add_visible_widget(w, true);
  w->show();
}

static void do_show_misc_options_dlg()
{
  vscreen_widget *w=make_misc_options_dialog();
  main_stacked->add_visible_widget(w, true);
  w->show();
}

static void do_show_dependency_options_dlg()
{
  vscreen_widget *w=make_dependency_options_dialog();
  main_stacked->add_visible_widget(w, true);
  w->show();
}

static void really_do_revert_options()
{
  apt_revertoptions();
  apt_dumpcfg(PACKAGE);
}

static void do_revert_options()
{
  prompt_yesno(_("Really discard your personal settings and reload the defaults?"),
	       false,
	       SigC::slot(really_do_revert_options),
	       NULL);
}

static vscreen_widget *make_default_view(pkg_menu_tree *mainwidget,
					 pkg_signal *sig,
					 desc_signal *desc_sig,
					 bool allow_visible_desc=true,
					 bool show_reason_first=false)
{
  if(aptcfg->Exists(PACKAGE "::UI::Default-Package-View"))
    {
      list<package_view_item> *format=load_pkgview(PACKAGE "::UI::Default-Package-View");

      if(format)
	{
	  vscreen_widget *rval=make_package_view(*format, mainwidget,
						 mainwidget, sig,
						 desc_sig, show_reason_first);
	  delete format;

	  if(rval)
	    return rval;
	}
    }

  list<package_view_item> basic_format;

  // FIXME: do the config lookup inside the package-view code?
  basic_format.push_back(package_view_item("static1",
					   parse_columns(aptcfg->Find(PACKAGE "::UI::Package-Header-Format", default_pkgheaderdisplay),
							 pkg_item::pkg_columnizer::parse_column_type,
							 pkg_item::pkg_columnizer::defaults),
					   PACKAGE "::UI::Package-Header-Format",
					   0, 0, 1, 1,
					   vs_table::ALIGN_CENTER | vs_table::EXPAND,
					   vs_table::ALIGN_CENTER,
					   get_color("ScreenHeaderColor"),
					   "",
					   "",
					   true));

  basic_format.push_back(package_view_item("main", 1, 0, 1, 1,
					   vs_table::EXPAND | vs_table::SHRINK,
					   vs_table::EXPAND | vs_table::SHRINK,
					   get_color("DefaultWidgetBackground"),
					   true));

  basic_format.push_back(package_view_item("static2",
					   parse_columns(aptcfg->Find(PACKAGE "::UI::Package-Status-Format", default_pkgstatusdisplay),
							 pkg_item::pkg_columnizer::parse_column_type,
							 pkg_item::pkg_columnizer::defaults),
					   PACKAGE "::UI::Package-Status-Format",
					   2, 0, 1, 1,
					   vs_table::ALIGN_CENTER | vs_table::EXPAND,
					   vs_table::ALIGN_CENTER,
					   get_color("ScreenStatusColor"),
					   "", "",
					   true));

  basic_format.push_back(package_view_item("desc", 3, 0, 1, 1,
					   vs_table::EXPAND | vs_table::SHRINK,
					   vs_table::EXPAND | vs_table::SHRINK,
					   get_color("DefaultWidgetBackground"),
					   "ShowHideDescription", "",
					   allow_visible_desc && aptcfg->FindB(PACKAGE "::UI::Description-Visible-By-Default", true)));

  return make_package_view(basic_format, mainwidget, mainwidget, sig, desc_sig,
			   show_reason_first);
}

void do_new_package_view(OpProgress &progress)
{
  pkg_grouppolicy_factory *grp=NULL;
  std::string grpstr="";

  if(aptcfg->Exists(PACKAGE "::UI::Default-Grouping"))
    {
      grpstr=aptcfg->Find(PACKAGE "::UI::Default-Grouping");
      grp=parse_grouppolicy(grpstr);
    }

  if(!grp) // Fallback
    {
      grpstr=default_grpstr;
      grp=parse_grouppolicy(grpstr);

      if(!grp)
	// Eek!  The default grouping failed to parse.  Fall all the
	// way back.
	grp=new pkg_grouppolicy_filter_factory(pkg_missing_filter,new pkg_grouppolicy_task_factory(new pkg_grouppolicy_status_factory(new pkg_grouppolicy_section_factory(pkg_grouppolicy_section_factory::split_subdir,true,new pkg_grouppolicy_section_factory(pkg_grouppolicy_section_factory::split_topdir,false,new pkg_grouppolicy_end_factory())))));
    }

  pkg_tree *tree=new pkg_tree(grpstr.c_str(), grp, NULL, progress);

  add_main_widget(make_default_view(tree,
				    &tree->selected_signal,
				    &tree->selected_desc_signal),
		  _("Packages"),
		  _("View available packages and choose actions to perform"));
}

// For signal connections.
static void do_new_package_view_with_new_bar()
{
  vs_progress *p=gen_progress_bar();
  do_new_package_view(*p);
  p->destroy();
}

void do_new_hier_view(OpProgress &progress)
{
  pkg_grouppolicy_factory *grp=NULL;
  std::string grpstr="";

  grpstr="filter(missing),hier";
  grp=parse_grouppolicy(grpstr);

  pkg_tree *tree=new pkg_tree(grpstr.c_str(), grp, NULL, progress);
  tree->set_limit("!~v");
  //tree->set_hierarchical(false);

  add_main_widget(make_default_view(tree,
				    &tree->selected_signal,
				    &tree->selected_desc_signal),
		  "Packages",
		  _("View available packages and choose actions to perform"));
}

// For signal connections.
static void do_new_hier_view_with_new_bar()
{
  vs_progress *p=gen_progress_bar();
  do_new_hier_view(*p);
  p->destroy();
}

vscreen_widget *make_info_screen(const pkgCache::PkgIterator &pkg,
				 const pkgCache::VerIterator &ver)
{
  pkg_info_screen *w=new pkg_info_screen(pkg, ver);
  return make_default_view(w, w->get_sig(), w->get_desc_sig(), false);
}

vscreen_widget *make_dep_screen(const pkgCache::PkgIterator &pkg,
				const pkgCache::VerIterator &ver,
				bool reverse)
{
  pkg_dep_screen *w=new pkg_dep_screen(pkg, ver, reverse);
  return make_default_view(w, w->get_sig(), w->get_desc_sig(), true);
}

vscreen_widget *make_ver_screen(const pkgCache::PkgIterator &pkg)
{
  pkg_ver_screen *w=new pkg_ver_screen(pkg);
  return make_default_view(w, w->get_sig(), w->get_desc_sig(), true);
}

//static void silly_test_error(string s)
//{
//  _error->Error("%s", s.c_str());
//}

static void do_help_about()
{
  char buf[512];

  snprintf(buf, 512, _("Aptitude %s\n\nCopyright 2000 Daniel Burrows.\n\naptitude comes with ABSOLUTELY NO WARRANTY; for details see 'license' in the\nHelp menu.  This is free software, and you are welcome to redistribute it\nunder certain conditions; see 'license' for details."), VERSION);

  vscreen_widget *w=vs_dialog_ok(buf);
  w->show_all();

  popup_widget(w);
}

static void do_help_license()
{
  vscreen_widget *w=vs_dialog_fileview(HELPDIR "/COPYING",
				       NULL, slot(pager_search));
  w->show_all();

  popup_widget(w);
}

static void do_help_help()
{
  char buf[512];

  snprintf(buf, 512, HELPDIR "/%s", _("help.txt"));

  // Deal with missing localized docs.
  if(access(buf, R_OK)!=0)
    strncpy(buf, HELPDIR "/help.txt", 512);

  vscreen_widget *w=vs_dialog_fileview(buf, NULL, slot(pager_search));
  w->show_all();

  popup_widget(w);
}

static void do_help_readme()
{
  char buf[512];

  snprintf(buf, 512, HELPDIR "/%s", _("README")); // README can be translated..

  // Deal with missing localized docs.
  if(access(buf, R_OK)!=0)
    strncpy(buf, HELPDIR "/README", 512);

  vscreen_widget *w=vs_dialog_fileview(buf, NULL, slot(pager_search));
  w->show_all();
  popup_widget(w);
}

static void do_help_faq()
{
  vscreen_widget *w=vs_dialog_fileview(HELPDIR "/FAQ", NULL, slot(pager_search));
  w->show_all();

  popup_widget(w);
}

// news isn't translated since it's just a changelog.
static void do_help_news()
{
  vscreen_widget *w=vs_dialog_fileview(HELPDIR "/NEWS", NULL, slot(pager_search));
  w->show_all();

  popup_widget(w);
}

// There are some circular references because of the code to test
// for consistency before doing a package run; the last routine called before
// the program starts downloading will verify that the selections are
// consistent and call its predecessors if they are not.
//
// (er, can I disentangle this by rearranging the routines?  I think maybe I
//  can to some degree)

void install_or_remove_packages()
{
  active_download=true;

  if(active_preview_tree)
    active_preview_tree->destroy();

  vs_progress *p=gen_progress_bar();

  do_install_run(p, false, false);

  p->destroy();

  active_download=false;
}

static void actually_do_package_run();
static void fix_pkgs_and_do_package_run();

static void reset_preview()
{
  active_preview=NULL;
  active_preview_tree=NULL;
}

// One word: ewwwwwwww.  I REALLY need to get my act together on the
// view-customization stuff.
static void do_show_preview()
{
  if(!active_preview_tree)
    {
      pkg_grouppolicy_factory *grp=NULL;
      std::string grpstr;

      if(aptcfg->Exists(PACKAGE "::UI::Default-Preview-Grouping"))
	{
	  grpstr=aptcfg->Find(PACKAGE "::UI::Default-Preview-Grouping");
	  grp=parse_grouppolicy(grpstr);
	}

      //if(!grp && aptcfg->Exists(PACKAGE "::UI::Default-Grouping"))
	//{
	//  grpstr=aptcfg->Find(PACKAGE "::UI::Default-Grouping");
	//  grp=parse_grouppolicy(grpstr);
	//}

      if(!grp)
	{
	  grpstr="action";
	  grp=new pkg_grouppolicy_mode_factory(new pkg_grouppolicy_end_factory);
	}

      if(aptcfg->Exists(PACKAGE "::UI::Preview-Limit"))
	active_preview_tree=new pkg_tree(grpstr.c_str(), grp, aptcfg->Find(PACKAGE "::UI::Preview-Limit").c_str());
      else
	active_preview_tree=new pkg_tree(grpstr.c_str(), grp, NULL);

      active_preview=make_default_view(active_preview_tree,
				       &active_preview_tree->selected_signal,
				       &active_preview_tree->selected_desc_signal,
				       true,
				       true);

      active_preview->destroyed.connect(SigC::slot(reset_preview));
      active_preview_tree->destroyed.connect(SigC::slot(*active_preview, &vscreen_widget::destroy));
      active_preview_tree->connect_key("DoInstallRun",
				       &global_bindings,
				       SigC::slot(actually_do_package_run));
      add_main_widget(active_preview, _("Preview of package installation"),
		      _("View and/or adjust the actions that will be performed"));
    }
  else
    active_preview_tree->show();
}

// Note that there's currently not a nice way to revert changes made by the
// fixer (or indeed even see them)  This should be fixed at the same time that
// the issue with alerting the user to dependencies is fixed (same problem)
//
//  Huge FIXME: the preview interacts badly with the menu.  This can be solved
// in a couple ways, including having the preview be a popup dialog -- the best
// thing IMO, though, would be to somehow allow particular widgets to override
// the meaning of global commands.  This needs a little thought, though.  (fake
// keys?  BLEACH)
static void actually_do_package_run()
{
  if(apt_cache_file)
    {
      if(!active_download)
	{
	  if((*apt_cache_file)->BrokenCount()>0)
	    {
	      if(_config->FindB(PACKAGE "::Auto-Fix-Broken", true))
		{
		  undo_group *undo=new apt_undo_group;
		  (*apt_cache_file)->try_fix_broken(undo);
		  if(!undo->empty())
		    apt_undos->add_item(undo);
		  else
		    delete undo;

		  // Ok, we (hopefully) modified stuff; be nice and tell
		  // the user.

		  if(active_preview_tree)
		    active_preview_tree->build_tree();
		  do_show_preview();

		  if((*apt_cache_file)->BrokenCount()==0)
		    show_message(_("Some packages were broken and have been fixed"));
		  else
		    show_message(_("Some broken packages could not be fixed!"),
						NULL,
				 get_color("Error"));

		  return;
		}
	      else
		{
		  prompt_yesno(_("Some packages are broken.  Try to fix them?"),
			       true,
			       SigC::slot(fix_pkgs_and_do_package_run),
			       NULL);
		  return;
		  // Wait to see what the user says
		}
	    }

	  if(getuid()==0  || !aptcfg->FindB(PACKAGE "::Warn-Not-Root", true))
	    install_or_remove_packages();
	  else
	{
	  popup_widget(vs_dialog_yesno(_("Installing/removing packages requires administrative privileges, which\nyou currently do not have.  Would you like to change to the root account?\n"),
				       bind(SigC::slot(&do_su_to_root),
					    "-i"),
				       _("Become root"),
				       SigC::slot(&install_or_remove_packages),
				       _("Don't become root"),
				       get_color("Error")));
	}
	}
      else
	show_message(_("A package-list update or install run is already taking place."), NULL, get_color("Error"));
    }
}

void do_package_run_or_show_preview()
{
  // UI: check that something will actually be done first.
  bool some_action_happening=false;
  bool some_non_simple_keep_happening=false;
  for(pkgCache::PkgIterator i=(*apt_cache_file)->PkgBegin(); !i.end(); ++i)
    {
      pkg_action_state state=find_pkg_state(i);

      if(state!=pkg_unchanged)
	{
	  some_action_happening=true;

	  if(!(state==pkg_hold && !(*apt_cache_file)->is_held(i)))
	    some_non_simple_keep_happening=true;
	}

      if(some_action_happening && some_non_simple_keep_happening)
	break;
    }

  if(!some_action_happening)
    {
      show_message(_("No packages are scheduled to be installed, removed, or upgraded."));
      return;
    }
  else if(!some_non_simple_keep_happening &&
	  !aptcfg->FindB(PACKAGE "::Allow-Null-Upgrade", false))
    {
      show_message(_("No packages will be installed, removed or upgraded.\n\n"
		     "Some packages could be upgraded, but you have not chosen to upgrade them.\n"
		     "Type \"U\" to prepare an upgrade."));
      return;
    }

  if(apt_cache_file)
    {
      if(!active_preview_tree)
	{
	  if(aptcfg->FindB(PACKAGE "::Display-Planned-Action", true))
	    do_show_preview();
	  else
	    actually_do_package_run();
	}
      else
	{
	  active_preview_tree->build_tree();
	  // We need to rebuild the tree since this is called after a
	  // broken-fixing operation.  This feels like a hack, though..
	  active_preview_tree->show();
	}
    }
}

static void fix_pkgs_and_do_package_run()
{
  if(apt_cache_file)
    {
      undo_group *undo=new apt_undo_group;
      (*apt_cache_file)->try_fix_broken(undo);
      if(!undo->empty())
	apt_undos->add_item(undo);
      else
	delete undo;

      do_package_run_or_show_preview();
    }
}

static bool can_start_download()
{
  return !active_download;
}

void do_package_run()
{
  if(apt_cache_file)
    {
      if(active_preview_tree && active_preview_tree->get_visible())
	actually_do_package_run();
      else if((*apt_cache_file)->BrokenCount()>0)
	{
	  if(aptcfg->FindB(PACKAGE "::Auto-Fix-Broken", true))
	    fix_pkgs_and_do_package_run();
	  else
	    prompt_yesno(_("Some packages are broken.  Try to fix them?"),
			 true,
			 SigC::slot(fix_pkgs_and_do_package_run),
			 SigC::slot(do_package_run_or_show_preview));
	}
      else
	do_package_run_or_show_preview();
    }
}

static void really_do_update_lists()
{
  active_download=true;

  vs_progress *p=gen_progress_bar();

  do_pkglist_update(p, false);

  p->destroy();

  active_download=false;
}

void do_update_lists()
{
  if(!active_download)
    {
      if(getuid()==0 || !aptcfg->FindB(PACKAGE "::Warn-Not-Root", true))
	really_do_update_lists();
      else
	{
	  popup_widget(vs_dialog_yesno(_("Updating the package lists requires administrative privileges, which\nyou currently do not have.  Would you like to change to the root account?\n"),
				       bind(SigC::slot(&do_su_to_root),
					    "-u"),
				       _("Become root"),
				       SigC::slot(&really_do_update_lists),
				       _("Don't become root"),
				       get_color("Error")));
	}
    }
  else
    show_message(_("A package-list update or install run is already taking place."), NULL, get_color("Error"));
}

static void do_sweep()
{
  add_main_widget(new cmine, _("Minesweeper"), _("Waste time trying to find mines"));
}

static void do_clean()
{
  if(active_download)
    // Erk!  That's weird!
    _error->Error(_("Cleaning while a download is in progress is not allowed"));
  else
    {
      vscreen_widget *msg=new vs_center(new vs_frame(new vs_label(_("Deleting downloaded files"))));
      msg->show_all();
      popup_widget(msg);
      vscreen_tryupdate();

      if(aptcfg)
	{
	  pkgAcquire fetcher;
	  fetcher.Clean(aptcfg->FindDir("Dir::Cache::archives"));
	  fetcher.Clean(aptcfg->FindDir("Dir::Cache::archives")+"partial/");
	}

      msg->destroy();

      show_message(_("Downloaded package files have been deleted"));
    }
}

// Ok, this is, uh, weird.  Erase has to be overridden to at least
// call unlink()
//
// Should I list what I'm doing like apt-get does?
class my_cleaner:public pkgArchiveCleaner
{
protected:
  virtual void Erase(const char *file,
		     string pkg,
		     string ver,
		     struct stat &stat)
  {
    unlink(file);
  }
};

static void do_autoclean()
{
  if(active_download)
    // Erk!  That's weird!
    _error->Error(_("Cleaning while a download is in progress is not allowed"));
  else
    {
      vscreen_widget *msg=new vs_center(new vs_frame(new vs_label(_("Deleting obsolete downloaded files"))));
      msg->show_all();
      popup_widget(msg);
      vscreen_tryupdate();

      if(aptcfg)
	{
	  my_cleaner cleaner;
	  cleaner.Go(aptcfg->FindDir("Dir::Cache::archives"), *apt_cache_file);
	  cleaner.Go(aptcfg->FindDir("Dir::Cache::archives")+"partial/",
		     *apt_cache_file);
	}

      msg->destroy();
      show_message(_("Obsolete downloaded package files have been deleted"));
    }
}

static void do_mark_upgradable()
{
  if(apt_cache_file)
    {
      undo_group *undo=new apt_undo_group;

      (*apt_cache_file)->mark_all_upgradable(true, undo);

      if(!undo->empty())
	apt_undos->add_item(undo);
      else
	delete undo;

      package_states_changed();
    }
}

static bool forget_new_enabled()
{
  if(!apt_cache_file)
    return false;
  else
    return (*apt_cache_file)->get_new_package_count()>0;
}

void do_forget_new()
{
  if(apt_cache_file)
    {
      undoable *undo=NULL;
      (*apt_cache_file)->forget_new(&undo);
      if(undo)
	apt_undos->add_item(undo);

      package_states_changed();
    }
}

static bool undo_enabled()
{
  return apt_undos->size()>0;
}

static void do_undo()
{
  apt_undos->undo();
  package_states_changed();
}

#ifdef WITH_RELOAD_CACHE
static void do_reload_cache()
{
  apt_reload_cache(gen_progress_bar(), true);
}
#endif

// NOTE ON TRANSLATIONS!
//
//   Implicitly translating stuff in the widget set would be ugly.  Everything
// would need to make sure its input to the widget set was able to survive
// translation.
//
//   Using N_ here and translating when we need to display the description
// would be ugly.  Everything would need to make sure its input to the UI would
// be able to handle translation.
//
//   What I do is ugly, but it doesn't force other bits of the program to handle
// input to us with velvet gloves, or otherwise break stuff.  So these structures
// just contain a char *.  I can modify the char *.  In particular, I can mark
// these for translation, then walk through them and munge the pointers to point
// to the translated version.  "EWWWWW!" I hear you say, and you're right, but
// the alternatives are worse.

vs_menu_info actions_menu[]={
  //{vs_menu_info::VS_MENU_ITEM, N_("Test ^Errors"), NULL,
  //N_("Generate an APT error for testing purposes"),
  //SigC::bind(SigC::slot(&silly_test_error), "This is a test error item.")},

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Install/remove packages"), "DoInstallRun",
	       N_("Perform all pending installs and removals"), SigC::slot(do_package_run), SigC::slot(can_start_download)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Update package list"), "UpdatePackageList",
	       N_("Check for new versions of packages"), SigC::slot(do_update_lists), SigC::slot(can_start_download)),

  // FIXME: this is a bad name for the menu item.
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Forget new packages"), "ForgetNewPackages",
	       N_("Forget which packages are \"new\""),
	       SigC::slot(do_forget_new), SigC::slot(forget_new_enabled)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Clean package cache"), NULL,
	       N_("Delete package files which were previously downloaded"),
	       SigC::slot(do_clean)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("Clean ^obsolete files"), NULL,
	       N_("Delete package files which can no longer be downloaded"),
	       SigC::slot(do_autoclean)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("Mark ^Upgradable"), "MarkUpgradable",
	       N_("Mark all upgradable packages which are not held for upgrade"),
	       SigC::slot(do_mark_upgradable)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Play Minesweeper"), NULL,
	       N_("Waste time trying to find mines"), SigC::slot(do_sweep)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Become root"), NULL,
	       N_("Run 'su' to become root; this will restart the program, but your\nsettings will be preserved"), bind(SigC::slot(do_su_to_root), ""), SigC::slot(su_to_root_enabled)),

#ifdef WITH_RELOAD_CACHE
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Reload package cache"), NULL,
	       N_("Reload the package cache"),
	       SigC::slot(do_reload_cache)),
#endif

  VS_MENU_SEPARATOR,

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Quit"), "QuitProgram",
	       N_("Exit the program"), SigC::slot(do_quit)),

  VS_MENU_END
};

vs_menu_info undo_menu[]={
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("Undo"), "Undo",
	       N_("Undo the last package operation or group of operations"),
	       SigC::slot(do_undo), SigC::slot(undo_enabled)),

  VS_MENU_END
};

vs_menu_info package_menu[]={
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Install"), "Install",
	       N_("Flag the currently selected package for installation or upgrade"),
	       package_install.slot(),
	       package_menu_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Remove"), "Remove",
	       N_("Flag the currently selected package for removal"),
	       package_remove.slot(),
	       package_menu_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Purge"), "Purge",
	       N_("Flag the currently selected package and its configuration files for removal"),
	       package_purge.slot(),
	       package_menu_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Keep"), "Keep",
	       N_("Cancel any action on the selected package"),
	       package_keep.slot(),
	       package_menu_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Hold"), "Hold",
	       N_("Cancel any action on the selected package, and protect it from future upgrades"),
	       package_hold.slot(),
	       package_menu_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("Mark ^Auto"), "SetAuto",
	       N_("Mark the selected package as having been automatically installed;\nit will automatically be removed if no other packages depend on it"),
	       package_mark_auto.slot(),
	       package_menu_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("Mark ^Manual"), "ClearAuto",
	       N_("Mark the selected package as having been manually installed;\nit will not be removed unless you manually remove it"),
	       package_unmark_auto.slot(),
	       package_menu_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Forbid Version"), "ForbidUpgrade",
	       N_("Forbid the candidate version of the selected package from being installed;\nnewer versions of the package will be installed as usual"),
	       package_forbid.slot(),
	       package_forbid_enabled.slot()),
  VS_MENU_SEPARATOR,
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("I^nformation"), "InfoScreen",
	       N_("Display more information about the selected package"),
	       package_information.slot(),
	       package_information_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Changelog"), "Changelog",
	       N_("Display the Debian changelog of the selected package"),
	       package_changelog.slot(),
	       package_changelog_enabled.slot()),
  VS_MENU_END
};

vs_menu_info search_menu[]={
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Find"), "Search",
	       N_("Search for a package"),
	       find_search.slot(), find_search_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("Find ^Again"), "ReSearch",
	       N_("Repeat the last search"),
	       find_research.slot(), find_research_enabled.slot()),
  VS_MENU_SEPARATOR,
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Limit Display"),
	       "ChangePkgTreeLimit", N_("Apply a filter to the package list"),
	       find_limit.slot(), find_limit_enabled.slot()),
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Un-Limit Display"),
	       NULL, N_("Remove the filter from the package list"),
	       find_cancel_limit.slot(), find_cancel_limit_enabled.slot()),
  VS_MENU_SEPARATOR,
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("Find ^Broken"),
	       "SearchBroken", N_("Find the next package with unsatisfied dependencies"),
	       find_broken.slot(), find_broken_enabled.slot()),
  VS_MENU_END
};

vs_menu_info options_menu[]={
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^UI options"), NULL,
	       N_("Change the settings which affect the user interface"),
	       SigC::slot(do_show_ui_options_dlg)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Dependency handling"), NULL,
	       N_("Change the settings which affect how package dependencies are handled"),
	       SigC::slot(do_show_dependency_options_dlg)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Miscellaneous"), NULL,
	       N_("Change miscellaneous program settings"),
	       SigC::slot(do_show_misc_options_dlg)),

  VS_MENU_SEPARATOR,

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Revert options"), NULL,
	       N_("Reset all settings to the system defaults"),
	       SigC::slot(do_revert_options)),

  //{vs_menu_info::VS_MENU_ITEM, N_("^Save options"), NULL,
  // N_("Save current settings for future sessions"), bind(SigC::slot(apt_dumpcfg),
  //							 PACKAGE)},

  VS_MENU_END
};

vs_menu_info views_menu_info[]={
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Next"), "CycleNext",
	       N_("View next display"), SigC::slot(do_view_next),
	       SigC::slot(view_next_prev_enabled)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Prev"), "CyclePrev",
	       N_("View previous display"), SigC::slot(do_view_prev),
	       SigC::slot(view_next_prev_enabled)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Close"), "Quit",
	       N_("Close this display"), SigC::slot(do_destroy_visible),
	       SigC::slot(any_view_visible)),

  VS_MENU_SEPARATOR,

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("New Package ^View"), NULL,
	       N_("Create a new default package view"),
	       SigC::slot(do_new_package_view_with_new_bar)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("New Categorical ^Browser"),
	       NULL,
	       N_("Browse packages by category"),
	       SigC::slot(do_new_hier_view_with_new_bar)),

  VS_MENU_SEPARATOR,
  VS_MENU_END
};

vs_menu_info help_menu_info[]={
  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^About"), NULL,
	       N_("View information about this program"),
	       SigC::slot(do_help_about)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^Help"), "Help",
	       N_("View the on-line help"), SigC::slot(do_help_help)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("User's ^Manual"), NULL,
	       N_("View the detailed program manual"),
	       SigC::slot(do_help_readme)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^FAQ"), NULL,
	       N_("View a list of frequently asked questions"),
	       SigC::slot(do_help_faq)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^ChangeLog"), NULL,
	       N_("View the important changes made in each version of the program"),
	       SigC::slot(do_help_news)),

  vs_menu_info(vs_menu_info::VS_MENU_ITEM, N_("^License"), NULL,
	       N_("View the terms under which you may copy the program"),
	       SigC::slot(do_help_license)),

  VS_MENU_END
};

// This is responsible for converting a particular menu-info thingy.
static void munge_menu(vs_menu_info *menu)
{
  while(menu->item_type!=vs_menu_info::VS_MENU_END)
    {
      if(menu->item_description)
	menu->item_description=_(menu->item_description);

      menu->item_name=_(menu->item_name);

      ++menu;
    }
}

static void do_show_menu_description(vs_menu_item *item, vs_label *label)
{
  if(item && item->get_description().size()>0)
    {
      label->show();
      label->set_text(item->get_description());
    }
  else
    {
      label->hide();
      label->set_text("");
    }
}

// I want discardable signal arguments!
static void do_setup_columns()
{
  pkg_item::pkg_columnizer::setup_columns(true);
}

static void load_options(string base, bool usetheme)
{
  load_colors(base+"::Colors", usetheme);
  load_bindings(base+"::Keybindings", &global_bindings, usetheme);
  load_bindings(base+"::Keybindings::EditLine", vs_editline::bindings, usetheme);
  load_bindings(base+"::Keybindings::Menu", vs_menu::bindings, usetheme);
  load_bindings(base+"::Keybindings::Menubar", vs_menubar::bindings, usetheme);
  load_bindings(base+"::Keybindings::Minesweeper", cmine::bindings, usetheme);
  load_bindings(base+"::Keybindings::MinibufChoice", vs_statuschoice::bindings, usetheme);
  load_bindings(base+"::Keybindings::Pager", vs_pager::bindings, usetheme);
  load_bindings(base+"::KeyBindings::PkgNode", pkg_tree_node::bindings, usetheme);
  load_bindings(base+"::Keybindings::PkgTree", pkg_tree::bindings, usetheme);
  load_bindings(base+"::Keybindings::Table", vs_table::bindings, usetheme);
  load_bindings(base+"::Keybindings::TextLayout", vs_text_layout::bindings, usetheme);
  load_bindings(base+"::Keybindings::Tree", vs_tree::bindings, usetheme);
}

static vs_menu *add_menu(vs_menu_info *info, const std::string &name,
			 vs_label *menu_description)
{
  munge_menu(info);

  vs_menu *menu=new vs_menu(0, 0, 0, info);

  main_menu->append_item(name, menu);

  menu->item_highlighted.connect(bind(slot(do_show_menu_description),
				      menu_description));

  return menu;
}

// argh
class help_bar:public vs_label
{
public:
  help_bar(string txt, int attr):vs_label(txt, attr)
  {
    set_visibility();
  }

  inline void set_visibility()
  {
    set_visible(aptcfg->FindB(PACKAGE "::UI::HelpBar", true));
  }
};

void ui_init()
{
  vscreen_init();
  init_defaults();

  // The basic behavior of the package state signal is to update the
  // display.
  package_states_changed.connect(SigC::slot(vscreen_update));

  if(aptcfg->Exists(PACKAGE "::Theme"))
    load_options(PACKAGE "::Themes::"+string(aptcfg->Find(PACKAGE "::Theme"))+"::"+PACKAGE "::UI", true);

  load_options(PACKAGE "::UI", false);

  vs_label *menu_description=new vs_label("");
  menu_description->set_bg(get_color("ScreenStatusColor"));

  main_menu=new vs_menubar(!aptcfg->FindB(PACKAGE "::UI::Menubar-Autohide", false));

  aptcfg->connect(string(PACKAGE "::UI::Menubar-Autohide"),
		  SigC::slot(update_menubar_autohide));
  aptcfg->connect(string(PACKAGE "::UI::Package-Display-Format"),
		  SigC::slot(do_setup_columns));

  cache_closed.connect(SigC::slot(do_show_reload_message));
  cache_reloaded.connect(SigC::slot(do_hide_reload_message));
  cache_reload_failed.connect(SigC::slot(do_hide_reload_message));

  add_menu(actions_menu, _("Actions"), menu_description);
  add_menu(undo_menu, _("Undo"), menu_description);
  add_menu(package_menu, _("Package"), menu_description);
  add_menu(search_menu, _("Search"), menu_description);
  add_menu(options_menu, _("Options"), menu_description);
  views_menu=add_menu(views_menu_info, _("Views"), menu_description);
  add_menu(help_menu_info, _("Help"), menu_description);

  main_stacked=new vs_stacked;
  main_menu->set_subwidget(main_stacked);
  main_stacked->show();

  main_stacked->connect_key_post("QuitProgram", &global_bindings, SigC::slot(do_quit));
  main_stacked->connect_key_post("Quit", &global_bindings, SigC::slot(do_destroy_visible));
  main_stacked->connect_key_post("CycleNext",
				 &global_bindings,
				 SigC::slot(do_view_next));
  main_stacked->connect_key_post("CyclePrev",
				 &global_bindings,
				 SigC::slot(do_view_prev));
  main_stacked->connect_key_post("DoInstallRun",
				 &global_bindings,
				 SigC::slot(do_package_run));
  main_stacked->connect_key_post("UpdatePackageList",
				 &global_bindings,
				 SigC::slot(do_update_lists));
  main_stacked->connect_key_post("MarkUpgradable",
				 &global_bindings,
				 SigC::slot(do_mark_upgradable));
  main_stacked->connect_key_post("ForgetNewPackages",
				 &global_bindings,
				 SigC::slot(do_forget_new));
  main_stacked->connect_key_post("Help",
				 &global_bindings,
				 SigC::slot(do_help_help));
  main_stacked->connect_key_post("Undo",
				 &global_bindings,
				 SigC::slot(do_undo));

  main_table=new vs_table;
  main_stacked->add_widget(main_table);
  main_table->show();

  // FIXME: highlight the keys.
  string menu_key=global_bindings.keyname("ToggleMenuActive"),
    help_key=global_bindings.keyname("Help"),
    quit_key=global_bindings.keyname("Quit"),
    update_key=global_bindings.keyname("UpdatePackageList"),
    install_key=global_bindings.keyname("DoInstallRun");

  char buf[512];
  snprintf(buf, 512, _("%s: Menu  %s: Help  %s: Quit  %s: Update  %s: Download/Install/Remove Pkgs"),
	   menu_key.c_str(),
	   help_key.c_str(),
	   quit_key.c_str(),
	   update_key.c_str(),
	   install_key.c_str());

  help_bar *help_label=new help_bar(buf, get_color("ScreenStatusColor"));
  main_table->add_widget_opts(help_label, 0, 0, 1, 1,
			      vs_table::EXPAND | vs_table::SHRINK,
			      vs_table::ALIGN_CENTER);
  aptcfg->connect(string(PACKAGE "::UI::HelpBar"),
		  SigC::slot(*help_label, &help_bar::set_visibility));

  main_multiplex=new vs_multiplex;
  main_table->add_widget_opts(main_multiplex, 1, 0, 1, 1,
			      vs_table::EXPAND | vs_table::SHRINK,
			      vs_table::EXPAND | vs_table::SHRINK);
  main_multiplex->show();

  main_status_multiplex=new vs_multiplex;
  main_table->add_widget_opts(main_status_multiplex, 2, 0, 1, 1,
			      vs_table::EXPAND | vs_table::SHRINK,
			      vs_table::ALIGN_CENTER);
  main_status_multiplex->show();

  main_status_multiplex->add_widget(menu_description);

  main_menu->show();

  vscreen_settoplevel(main_menu);

  check_apt_errors();
  main_hook.connect(SigC::slot(check_apt_errors));

  help_label->set_visibility();

  update_menubar_autohide();

  // Force parsing of the column stuff.
  // FIXME: put this in load_options() and kill all other references
  //       to setup_columns?
  pkg_item::pkg_columnizer::setup_columns();
}

void ui_main()
{
  vscreen_mainloop();

  if(apt_cache_file &&
     (aptitudeDepCache *) (*apt_cache_file) &&
     apt_cache_file->is_locked())
    {
      vs_progress *p=gen_progress_bar();
      (*apt_cache_file)->save_selection_list(*p);
      p->destroy();
    }

  rootwin.bkgdset(' ');
  rootwin.clear();
  rootwin.refresh();

  endwin();
}

void popup_widget(vscreen_widget *w, bool do_show_all)
{
  main_stacked->add_widget(w);

  if(do_show_all)
    w->show_all();
  else
    w->show();
}

static void setup_main_widget(vscreen_widget *w, const std::string &menuref,
			      const std::string &menudesc)
{
  vs_menu_item *menuentry=new vs_menu_item(menuref, "", menudesc);

  // FIXME: if w is removed from the multiplexer but not destroyed, this may
  //       break.  Fix for now: Don't Do That Then!
  w->destroyed.connect(bind(slot(*views_menu, &vs_menu::remove_item), menuentry));
  menuentry->selected.connect(slot(*w, &vscreen_widget::show));

  views_menu->append_item(menuentry);
}

// Handles the case where the last view is destroyed directly (other than
// through do_destroy_visible); for instance, when a download completes.
static void main_widget_destroyed()
{
  if(aptcfg->FindB(PACKAGE "::UI::Exit-On-Last-Close", true) &&
     main_multiplex->num_children()==0)
    // Don't prompt -- if the last view is destroyed, assume it was by
    // the user's request.
    file_quit();
}

void add_main_widget(vscreen_widget *w, const std::string &menuref,
		     const std::string &menudesc)
{
  setup_main_widget(w, menuref, menudesc);
  main_multiplex->add_widget(w);
  w->show();
  w->destroyed.connect(SigC::slot(main_widget_destroyed));

  update_menubar_autohide();
}

void insert_main_widget(vscreen_widget *w, const std::string &menuref,
			const std::string &menudesc)
{
  setup_main_widget(w, menuref, menudesc);
  main_multiplex->add_widget_after(w, main_multiplex->visible_widget());
  w->show();

  update_menubar_autohide();
}

vscreen_widget *active_main_widget()
{
  return main_multiplex->visible_widget();
}

vs_progress *gen_progress_bar()
{
  vs_progress *rval=new vs_progress;

  main_status_multiplex->add_visible_widget(rval, true);

  return rval;
}

fragment *wrapbox(fragment *contents)
{
  if(aptcfg->FindB(PACKAGE "::UI::Fill-Text", false))
    return fillbox(contents);
  else
    return flowbox(contents);
}

static void reset_status_download()
{
  active_status_download=NULL;
}

download_manager *gen_download_progress(bool force_noninvasive,
					const string &title,
					const string &longtitle,
					Slot0Arg abortslot)
{
  download_manager *m=new download_manager;
  download_list *w=NULL;

  if(force_noninvasive ||
     aptcfg->FindB(PACKAGE "::UI::Minibuf-Download-Bar", false))
    {
      w=new download_list(abortslot, false);
      main_status_multiplex->add_visible_widget(w, true);
      active_status_download=w;
      w->destroyed.connect(SigC::slot(&reset_status_download));
    }
  else
    {
      w=new download_list(abortslot);
      add_main_widget(w, title, longtitle);
    }

  m->MediaChange_sig.connect(slot(*w, &download_list::MediaChange));
  m->IMSHit_sig.connect(slot(*w, &download_list::IMSHit));
  m->Fetch_sig.connect(slot(*w, &download_list::Fetch));
  m->Done_sig.connect(slot(*w, &download_list::Done));
  m->Fail_sig.connect(slot(*w, &download_list::Fail));
  m->Pulse_sig.connect(slot(*w, &download_list::Pulse));
  m->Start_sig.connect(slot(*w, &download_list::Start));
  m->Stop_sig.connect(slot(*w, &download_list::Stop));
  m->Complete_sig.connect(slot(*w, &download_list::Complete));

  return m;
}

static void do_prompt_string(string s,
			     vs_editline *e,
			     SigC::Slot0<void> realslot)
{
  e->add_to_history(s);
  realslot();
}

void prompt_string(const std::string &prompt,
		   const std::string &text,
		   SlotArg<SigC::Slot1<void, string> > slot,
		   SlotArg<SigC::Slot0<void> > cancel_slot,
		   SlotArg<SigC::Slot1<void, string> > changed_slot,
		   vs_editline::history_list *history)
{
  if(aptcfg->FindB(PACKAGE "::UI::Minibuf-Prompts"))
    {
      vs_editline *e=new vs_editline(prompt, text, history);
      if(slot)
	e->entered.connect(slot);

      e->entered.connect(bind(SigC::slot(do_prompt_string),
			      e,
			      SigC::slot(*e, &vscreen_widget::destroy)));
      if(changed_slot)
	e->text_changed.connect(changed_slot);

      e->connect_key("Cancel",
		     &global_bindings,
		     SigC::slot(*e, &vscreen_widget::destroy));

      if(cancel_slot)
	e->connect_key("Cancel",
		       &global_bindings,
		       cancel_slot);

      main_status_multiplex->add_visible_widget(e, true);
      main_table->focus_widget(main_status_multiplex);
    }
  else
    main_stacked->add_visible_widget(vs_dialog_string(prompt, text,
						      slot, cancel_slot,
						      changed_slot,
						      history),
				     true);
}

static void do_prompt_yesno(int cval,
			    bool deflt,
			    Slot0Arg yesslot,
			    Slot0Arg noslot)
{
  bool rval;

  if(deflt)
    rval=!cval;
  else
    rval=cval;

  if(rval)
    {
      if(yesslot)
	((SigC::Slot0<void>) yesslot)();
    }
  else
    {
      if(noslot)
	((SigC::Slot0<void>) noslot)();
    }
}

void prompt_yesno(const std::string &prompt,
		  bool deflt,
		  Slot0Arg yesslot,
		  Slot0Arg noslot)
{
  if(aptcfg->FindB(PACKAGE "::UI::Minibuf-Prompts"))
    {
      string yesstring, nostring;

      yesstring+=_("yes_key")[0];
      nostring+=_("no_key")[0];
      string yesnostring=deflt?yesstring+nostring:nostring+yesstring;

      vs_statuschoice *c=new vs_statuschoice(prompt, yesnostring.c_str());
      c->set_bg(get_color("ScreenStatusColor"));
      c->chosen.connect(bind(bind(SigC::slot(&do_prompt_yesno),
				  yesslot,
				  noslot),
			     deflt));

      main_status_multiplex->add_visible_widget(c, true);
      main_table->focus_widget(main_status_multiplex);
    }
  else
    main_stacked->add_visible_widget(vs_dialog_yesno(prompt,
						     yesslot,
						     noslot,
						     deflt),
				     true);
}

void show_message(const std::string &msg,
		  Slot0Arg okslot,
		  int attr)
{
  if(aptcfg->FindB(PACKAGE "::UI::Minibuf-Prompts"))
    {
      vs_transientlabel *l=new vs_transientlabel(msg,
						 attr);
      if(okslot)
	l->destroyed.connect(okslot);

      main_status_multiplex->add_visible_widget(l, true);
      main_table->focus_widget(main_status_multiplex);
    }
  else
    main_stacked->add_visible_widget(vs_dialog_ok(msg, okslot), true);
}

void show_message(const std::string &msg,
		  Slot0Arg okslot)
{
  show_message(msg, okslot, get_color("ScreenStatusColor"));
}
