#define _GNU_SOURCE
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>

#define LENOF(a) (sizeof(a) / sizeof(*(a)))

const char *force_base_prefix = "linux-image-";

/* These aren't really in Desktop, but are installed by base-config so
 * archive-copier needs to consider them as such.
 */
const char *force_desktop[] = { "laptop-detect", "mdetect", "xresprobe" };

void die(const char *format, ...)
{
    va_list ap;

    va_start(ap, format);
    vfprintf(stderr, format, ap);
    va_end(ap);

    exit(1);
}

void die_errno(const char *format, ...)
{
    va_list ap;

    va_start(ap, format);
    vfprintf(stderr, format, ap);
    va_end(ap);
    fprintf(stderr, ": %s", strerror(errno));

    exit(1);
}

void *xmalloc(size_t size)
{
    void *ret = malloc(size);
    if (!ret)
	die("malloc failed");
    return ret;
}

void *xrealloc(void *ptr, size_t size)
{
    void *ret = realloc(ptr, size);
    if (!ret)
	die("realloc failed");
    return ret;
}

char *get_paragraph(const char **p, size_t *left)
{
    static char *space = NULL;
    static size_t space_alloc = 0;
    const char *sep;
    size_t len;

    /* get next stanza */
    sep = memmem(*p, *left, "\n\n", 2);
    if (sep)
	len = sep - *p;
    else
	len = *left;

    /* make sure we have enough space */
    if (!space || len + 1 > space_alloc) {
	space = xrealloc(space, len + 1);
	space_alloc = len + 1;
    }

    /* copy, terminate, advance */
    memcpy(space, *p, len);
    space[len] = '\0';
    *p = sep;
    *left -= len;
    if (*p) {
	*p += 2; /* skip \n\n */
	*left -= 2;
    }

    return space;
}

char *encode_colons(const char *version)
{
    static char *space = NULL;
    static size_t space_alloc = 0;
    size_t len;
    const char *p;
    char *spacep;

    /* make sure we have enough space */
    len = strlen(version) * 3 + 1;
    if (!space || len > space_alloc) {
	space = xrealloc(space, len);
	space_alloc = len + 1;
    }

    /* encode colons; copy everything else */
    for (p = version, spacep = space; *p; ++p) {
	if (*p == ':') {
	    strcpy(spacep, "%3a");
	    spacep += 3;
	} else
	    *spacep++ = *p;
    }
    *spacep = '\0';

    return space;
}

int packages_compare(const void *a, const void *b)
{
    const char *key = *(const char **) a;
    const char *memb = *(const char **) b;
    return strcmp(key, memb);
}

int main(int argc, char **argv)
{
    /* base package names */
    size_t n_basepackages = 0;
    const char **basepackages = NULL;

    /* desktop task name */
    const char *desktop_task;

    /* ship task name */
    const char *ship_task;

    const char *packages_filename;
    int packages_fd;
    struct stat packages_st;
    char *packages;
    const char *packages_p;
    size_t packages_left;
    char *para;

    if (argc < 4) {
	fprintf(stderr, "Usage: %s <Packages> <desktop task> <ship task> <base package names>\n", argv[0]);
	exit(1);
    }

    packages_filename = argv[1];
    desktop_task = argv[2];
    ship_task = argv[3];

    if (argc >= 5) {
	int i;
	n_basepackages = argc - 4;
	basepackages = xmalloc(n_basepackages * sizeof *basepackages);
	for (i = 0; i < n_basepackages; ++i)
	    basepackages[i] = argv[i + 4];
	qsort(basepackages, n_basepackages, sizeof *basepackages,
	      &packages_compare);
    }

    packages_fd = open(packages_filename, O_RDONLY);
    if (packages_fd < 0)
	die_errno("open %s failed", packages_filename);
    if (fstat(packages_fd, &packages_st) < 0)
	die_errno("stat %s failed", packages_filename);
    packages = mmap((void *) 0, packages_st.st_size, PROT_READ, MAP_SHARED,
		    packages_fd, 0);
    if (packages == MAP_FAILED)
	die_errno("mmap %s failed", packages_filename);

    packages_p = packages;
    packages_left = (size_t) packages_st.st_size;
    for (para = get_paragraph(&packages_p, &packages_left); *para;
	 para = get_paragraph(&packages_p, &packages_left)) {
	const char *line;
	const char *package = NULL, *version = NULL, *filename = NULL;
	const char *task = NULL;

	for (line = strsep(&para, "\n"); line; line = strsep(&para, "\n")) {
	    if (!strncasecmp(line, "Package: ", 9))
		package = line + 9;

	    if (!strncasecmp(line, "Version: ", 9))
		version = encode_colons(line + 9);
	    else if (!strncasecmp(line, "Filename: ", 10))
		filename = line + 10;
	    else if (!strncasecmp(line, "Task: ", 6))
		task = line + 6;
	}

	if (version && filename) {
	    const char *status;
	    if (package && bsearch(&package, basepackages, n_basepackages,
				   sizeof *basepackages,
				   &packages_compare))
		status = "base";
	    else if (!strncmp(package, force_base_prefix,
			      strlen(force_base_prefix)))
		status = "base";
	    else if (task && !strcmp(task, desktop_task))
		status = "desktop";
	    else if (package && bsearch(&package, force_desktop,
					LENOF(force_desktop),
					sizeof *force_desktop,
					&packages_compare))
		status = "desktop";
	    else if (task && !strcmp(task, ship_task))
		status = "ship";
	    else
		status = "supported";
	    printf("%s %s %s\n", filename, version, status);
	}
    }

    munmap(packages, packages_st.st_size);
    close(packages_fd);

    return 0;
}
