/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *  Copyright (C) 2006-2007  Bastien Nocera <hadess@hadess.net>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "client.h"

#include "general.h"
#include "adapter.h"
#include "dialog.h"
#include "input.h"

static DBusGConnection *connection = NULL;
static DBusGProxy *manager = NULL;

static GtkListStore *store = NULL;
static GtkWidget *tree = NULL;
static GtkWidget *button_remove;

static void proxy_callback(DBusGProxy *proxy,
				DBusGProxyCall *call, void *user_data)
{
	GtkWidget *notebook = user_data;
	GtkWidget *button;
	GtkWidget *label;
	DBusGProxy *client;
	const char *path, *busname;
	const char *adapter = NULL, *address = NULL, *name = NULL;
	gchar *text;

	g_object_set_data(G_OBJECT(notebook), "call", NULL);

	dbus_g_proxy_end_call(proxy, call, NULL,
				G_TYPE_STRING, &path, G_TYPE_INVALID);

	button = g_object_get_data(G_OBJECT(notebook), "cancel");
	gtk_widget_hide(button);

	button = g_object_get_data(G_OBJECT(notebook), "close");
	gtk_widget_set_sensitive(button, TRUE);

	busname = dbus_g_proxy_get_bus_name(proxy);

	client = dbus_g_proxy_new_for_name(connection, busname,
					path, "org.bluez.input.Device");

	dbus_g_proxy_call(client, "GetAdapter", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(client, "GetAddress", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &address, G_TYPE_INVALID);

	dbus_g_proxy_call(client, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	label = g_object_get_data(G_OBJECT(notebook), "label");
	text = g_strdup_printf("<b>%s %s</b>",
				name ? name : address, _("connected"));
	gtk_label_set_markup(GTK_LABEL(label), text);
	g_free(text);

	set_trusted(adapter, address);

	g_object_unref(client);
}

static void connect_callback(const gchar *address, gpointer user_data)
{
	DBusGProxyCall *call;

	call = dbus_g_proxy_begin_call(manager, "CreateDevice",
				proxy_callback, user_data, NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID);

	g_object_set_data(G_OBJECT(user_data), "call", call);
}

static void cleanup_callback(gpointer user_data)
{
	DBusGProxyCall *call;

	call = g_object_get_data(G_OBJECT(user_data), "call");
	if (call != NULL)
		dbus_g_proxy_cancel_call(manager, call);

	g_object_set_data(G_OBJECT(user_data), "call", NULL);
}

static gboolean device_filter(GtkTreeModel *model,
				GtkTreeIter *iter, gpointer user_data)
{
	gboolean active;
	guint type;

	gtk_tree_model_get(model, iter, COLUMN_ACTIVE, &active,
						COLUMN_TYPE, &type, -1);

	if (active == FALSE)
		return FALSE;

	if (type == BLUETOOTH_TYPE_KEYBOARD || type == BLUETOOTH_TYPE_MOUSE)
		return TRUE;

	return FALSE;
}

static void create_callback(GtkWidget *button, gpointer user_data)
{
	show_device_dialog(connect_callback, cleanup_callback, device_filter);
}

static void remove_callback(GtkWidget *button, gpointer user_data)
{
	GtkTreeSelection *selection = user_data;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *path;

	if (show_remove_dialog() == FALSE)
		return;

	gtk_tree_selection_get_selected(selection, &model, &iter);

	gtk_tree_model_get(model, &iter, 0, &path, -1);

	dbus_g_proxy_call(manager, "RemoveDevice", NULL,
			G_TYPE_STRING, path, G_TYPE_INVALID, G_TYPE_INVALID);

	g_free(path);
}

static void select_callback(GtkTreeSelection *selection, gpointer user_data)
{
	gboolean selected;

	selected = gtk_tree_selection_get_selected(selection, NULL, NULL);

	gtk_widget_set_sensitive(button_remove, selected);
}

GtkWidget *create_input(void)
{
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *scrolled;
	GtkTreeViewColumn *column;
	GtkTreeSelection *selection;
	GtkWidget *buttonbox;
	GtkWidget *button;

	vbox = gtk_vbox_new(FALSE, 6);

	label = create_label(_("Input devices"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled),
							GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(vbox), scrolled);

	tree = gtk_tree_view_new();
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tree), FALSE);
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(tree), TRUE);
	gtk_tree_view_set_model(GTK_TREE_VIEW(tree), GTK_TREE_MODEL(store));

	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(tree), 0,
					"Name", gtk_cell_renderer_text_new(),
							"text", 3, NULL);
	column = gtk_tree_view_get_column(GTK_TREE_VIEW(tree), 0);
	gtk_tree_view_column_set_expand(GTK_TREE_VIEW_COLUMN(column), TRUE);

	gtk_container_add(GTK_CONTAINER(scrolled), tree);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
	g_signal_connect(G_OBJECT(selection), "changed",
				G_CALLBACK(select_callback), NULL);

	buttonbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox),
						GTK_BUTTONBOX_START);
	gtk_box_set_spacing(GTK_BOX(buttonbox), 6);
	gtk_box_set_homogeneous(GTK_BOX(buttonbox), FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), buttonbox, FALSE, FALSE, 0);

	button = gtk_button_new_from_stock(GTK_STOCK_ADD);
	gtk_container_add(GTK_CONTAINER(buttonbox), button);

	g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(create_callback), NULL);

	button = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
	gtk_container_add(GTK_CONTAINER(buttonbox), button);

	button_remove = button;

	g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(remove_callback), selection);

	gtk_widget_show_all(vbox);

	return vbox;
}

static void add_device(const char *busname, const char *path)
{
	DBusGProxy *proxy;
	const char *adapter = NULL, *address = NULL, *name = NULL;
	GtkTreeIter iter;

	proxy = dbus_g_proxy_new_for_name(connection, busname,
					path, "org.bluez.input.Device");

	dbus_g_proxy_call(proxy, "GetAdapter", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(proxy, "GetAddress", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &address, G_TYPE_INVALID);

	dbus_g_proxy_call(proxy, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	gtk_list_store_insert_with_values(store, &iter, -1,
					0, path, 1, adapter, 2, address,
					3, name ? name : address, -1);

	g_object_unref(proxy);
}

static void device_created(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	const char *busname;

	busname = dbus_g_proxy_get_bus_name(object);

	add_device(busname, path);
}

static void device_removed(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	GtkTreeIter iter;
	gboolean cont;

	cont = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);

	while (cont == TRUE) {
		gchar *temp;

		gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, 0, &temp, -1);

		if (g_ascii_strcasecmp(path, temp) == 0) {
			gtk_list_store_remove(store, &iter);
			g_free(temp);
			break;
		}

		g_free(temp);

		cont = gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter);
	}
}

static void list_devices(DBusGProxy *object,
				DBusGProxyCall *call, void *user_data)
{
	GError *error = NULL;
	const gchar **array = NULL;
	const char *busname;

	dbus_g_proxy_end_call(object, call, &error,
					G_TYPE_STRV, &array, G_TYPE_INVALID);

	if (error == NULL) {
		busname = dbus_g_proxy_get_bus_name(object);

		while (*array) {
			add_device(busname, *array);
			array++;
		}
	} else
		g_error_free(error);
}

void enable_input(DBusGConnection *conn, const char *busname)
{
	store = gtk_list_store_new(4, G_TYPE_STRING, G_TYPE_STRING,
					G_TYPE_STRING, G_TYPE_STRING);
	if (store == NULL)
		return;

	if (tree != NULL)
		gtk_tree_view_set_model(GTK_TREE_VIEW(tree),
						GTK_TREE_MODEL(store));

	manager = dbus_g_proxy_new_for_name(conn, busname,
				"/org/bluez/input", "org.bluez.input.Manager");
	if (manager == NULL)
		return;

	connection = dbus_g_connection_ref(conn);

	dbus_g_proxy_add_signal(manager, "DeviceCreated",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(manager, "DeviceCreated",
				G_CALLBACK(device_created), NULL, NULL);

	dbus_g_proxy_add_signal(manager, "DeviceRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(manager, "DeviceRemoved",
				G_CALLBACK(device_removed), NULL, NULL);

	dbus_g_proxy_begin_call(manager, "ListDevices", list_devices,
						NULL, NULL, G_TYPE_INVALID);
}

void disable_input(void)
{
	if (tree != NULL)
		gtk_tree_view_set_model(GTK_TREE_VIEW(tree), NULL);

	if (store != NULL) {
		g_object_unref(store);
		store = NULL;
	}

	if (manager != NULL) {
		g_object_unref(manager);
		manager = NULL;
	}

	if (connection != NULL) {
		dbus_g_connection_unref(connection);
		connection = NULL;
	}
}
