;;; eval.scm - Interpreter for CHICKEN
;
; Copyright (c) 2000-2003, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare
  (unit eval)
  (usual-integrations)
  (hide ##sys#unregister-macro ##sys#macroexpand-0 ##sys#split-at-separator ##sys#lookup-required-file
	##sys#r4rs-environment ##sys#r5rs-environment
	##sys#interaction-environment pds pdss)
  (foreign-declare #<<EOF
#ifndef C_INSTALL_LIB_HOME
# define C_INSTALL_LIB_HOME    "."
#endif
EOF
) )

(cond-expand
 [paranoia]
 [else
  (declare
    (no-bound-checks)
    (bound-to-procedure 
     ##sys#check-char ##sys#check-exact ##sys#check-port ##sys#check-string ##sys#load-library
     ##sys#for-each ##sys#map ##sys#setslot ##sys#allocate-vector ##sys#check-pair ##sys#not-a-proper-list-error
     ##sys#check-symbol ##sys#check-vector ##sys#floor ##sys#ceiling ##sys#truncate ##sys#round 
     ##sys#check-number ##sys#cons-flonum ##sys#copy-env-table
     ##sys#flonum-fraction ##sys#make-port ##sys#fetch-and-check-port-arg ##sys#print ##sys#check-structure 
     ##sys#make-structure ##sys#test-feature
     ##sys#error-handler ##sys#hash-symbol ##sys#register-macro ##sys#check-syntax
     ##sys#hash-table-ref ##sys#hash-table-set! ##sys#canonicalize-body ##sys#decompose-lambda-list
     ##sys#make-c-string ##sys#resolve-include-filename ##sys#register-macro-2 
     ##sys#load ##sys#error ##sys#warn ##sys#hash-table-location
     ##sys#make-flonum ##sys#make-pointer ##sys#null-pointer ##sys#address->pointer 
     ##sys#pointer->address ##sys#compile-to-closure
     ##sys#ensure-heap-reserve ##sys#syntax-error-hook ##sys#read-prompt-hook
     ##sys#repl-eval-hook ##sys#append ##sys#secondary-macroexpand ##sys#lookup-id
     ##sys#macroexpand-hook ##sys#macroexpand-0) ) ] )

(cond-expand
 [unsafe
  (eval-when (compile)
    (define-macro (##sys#check-structure . _) '(##core#undefined))
    (define-macro (##sys#check-range . _) '(##core#undefined))
    (define-macro (##sys#check-pair . _) '(##core#undefined))
    (define-macro (##sys#check-list . _) '(##core#undefined))
    (define-macro (##sys#check-symbol . _) '(##core#undefined))
    (define-macro (##sys#check-string . _) '(##core#undefined))
    (define-macro (##sys#check-char . _) '(##core#undefined))
    (define-macro (##sys#check-exact . _) '(##core#undefined))
    (define-macro (##sys#check-port . _) '(##core#undefined))
    (define-macro (##sys#check-number . _) '(##core#undefined))
    (define-macro (##sys#check-id . _) '(##core#undefined))
    (define-macro (##sys#check-byte-vector . _) '(##core#undefined)) ) ]
 [else] )


(include "parameters")

(define-foreign-variable install-lib-home c-string "C_INSTALL_LIB_HOME")

(define pds pathname-directory-separator)
(define pdss (string pathname-directory-separator))


;;; Macro handling:

(define ##sys#macro-environment (make-vector macro-table-size '()))

(define (##sys#register-macro-2 name handler)
  (##sys#hash-table-set! 
   ##sys#macro-environment name
   (lambda (form) (handler (##sys#slot form 1))) ) )

(define ##sys#register-macro
  (lambda (name handler)
    (##sys#hash-table-set! 
     ##sys#macro-environment name
     (lambda (form) (apply handler (##sys#slot form 1))) ) ) )

(define (macro? sym)
  (##sys#check-symbol sym 'macro?)
  (and (##sys#hash-table-ref ##sys#macro-environment sym) #t) )

(define (##sys#unregister-macro name)
  (##sys#hash-table-set! ##sys#macro-environment name #f) )

(define (undefine-macro! name)
  (##sys#check-symbol name 'undefine-macro!)
  (##sys#unregister-macro name) )

(define ##sys#macroexpand-0
  (let ([string-append string-append])
    (lambda (exp me)

      (define (call-handler name handler exp)
	(handle-exceptions ex
	    (##sys#abort
	     (if (and (##sys#structure? ex 'condition)
		      (memv 'exn (##sys#slot ex 1)) )
		 (##sys#make-structure
		  'condition
		  (##sys#slot ex 1)
		  (let copy ([ps (##sys#slot ex 2)])
		    (if (null? ps)
			'()
			(let ([p (car ps)]
			      [r (cdr ps)])
			  (if (and (eq? 'message p)
				   (pair? r)
				   (string? (car r)) )
			      (cons 
				'message
				(cons (string-append
				       "during expansion of (" (##sys#slot name 1) " ...) - "
				       (car r) )
				      (cdr r) ) )
			      (copy r) ) ) ) ) )
		 ex) )
	  (handler exp) ) )
				   
      (define (expand exp head)
	(cond [(assq head me) => (lambda (mdef) (values ((##sys#slot mdef 1) exp) #t))]
	      [(##sys#hash-table-ref ##sys#macro-environment head) 
	       => (lambda (handler)
		    (cond-expand
		     [unsafe (values (call-handler head handler exp) #t)]
		     [else
		      (let scan ([x exp])
			(cond [(null? x) (values (call-handler head handler exp) #t)]
			      [(pair? x) (scan (##sys#slot x 1))]
			      [else (##sys#syntax-error-hook "invalid syntax in macro form" exp)] ) ) ] ) ) ]
	      [else (values exp #f)] ) )

      (cond [(symbol? exp) (expand exp exp)]
	    [(pair? exp)
	     (let ([head (##sys#slot exp 0)]
		   [body (##sys#slot exp 1)] )
	       (if (symbol? head)
		   (cond [(eq? head 'let)
			  (##sys#check-syntax 'let body '#(_ 2))
			  (let ([bindings (car body)])
			    (cond [(symbol? bindings)
				   (##sys#check-syntax 'let body '(_ #((variable _) 0) . #(_ 1)))
				   (let ([bs (cadr body)])
				     (values
				      `(##core#app
					(letrec ([,bindings (##core#loop-lambda ,(map (lambda (b) (car b)) bs) ,@(cddr body))])
					  ,bindings)
					,@(##sys#map cadr bs) )
				      #t) ) ]
				  [else (values exp #f)] ) ) ]
			 [(and (memq head '(set! ##core#set!))
			       (pair? body)
			       (pair? (##sys#slot body 0))
			       (memq #:srfi-17 ##sys#features) )
			  (let ([dest (##sys#slot body 0)])
			    (##sys#check-syntax 'set! body '(#(_ 1) _))
			    (values
			     (append (list (list '##sys#setter (##sys#slot dest 0)))
				     (##sys#slot dest 1)
				     (##sys#slot body 1) ) 
			     #t) ) ]
			 [else (expand exp head)] )
		   (values exp #f) ) ) ]
	    [else (values exp #f)] ) ) ) )

(define (##sys#macroexpand-hook exp me)
  (let loop ([exp exp])
    (let-values ([(exp2 m) (##sys#macroexpand-0 exp me)])
      (if m
	  (loop exp2)
	  exp2) ) ) )

(define (##sys#macroexpand-1-hook exp me)
  (nth-value 0 (##sys#macroexpand-0 exp me)) )

(define (##sys#compiler-toplevel-macroexpand-hook exp) exp)

(define (macroexpand exp . me)
  (##sys#macroexpand-hook exp (if (pair? me) (car me) '())) )

(define (macroexpand-1 exp . me)
  (##sys#macroexpand-1-hook exp (if (pair? me) (car me) '())) )

(define ##sys#strict-mode #f)
(define ##sys#enable-runtime-macros #t)

(define (##sys#undefine-non-standard-macros leave)
  (let ([leave (##sys#append leave '(define and or cond case let* letrec do quasiquote delay))])
    (do ([i 0 (fx+ i 1)])
	((fx>= i macro-table-size))
      (##sys#setslot 
       ##sys#macro-environment i
       (let loop ([bs (##sys#slot ##sys#macro-environment i)])
	 (if (null? bs)
	     '()
	     (let ([b (##sys#slot bs 0)]
		   [r (##sys#slot bs 1)] )
	       (if (memq (##sys#slot b 0) leave)
		   (cons b (loop r))
		   (loop r) ) ) ) ) ) ) ) )


;;; "Compiler" macros:

(define ##sys#secondary-macro-table '())

(define (##sys#secondary-macroexpand x)
  (if (and (pair? x) (symbol? (##sys#slot x 0)))
      (let ([a (assq (##sys#slot x 0) ##sys#secondary-macro-table)])
	(if a 
	    (##sys#secondary-macroexpand ((##sys#slot a 1) x))
	    x) )
      x) )


;;; Expansion of bodies:

(define ##sys#canonicalize-body
  (let ([reverse reverse]
	[map map] )
    (lambda (body)
      (define (fini vars vals mvars mvals body)
	(if (and (null? vars) (null? mvars))
	    (if ##sys#strict-mode
		`(begin ,@body)
		(let loop ([body2 body] [exps '()])
		  (if (not (pair? body2)) 
		      `(begin ,@body)	; no more defines, otherwise we would have called `expand'
		      (let ([x (##sys#slot body2 0)])
			(if (and (pair? x) (memq (##sys#slot x 0) `(define define-values)))
			    `(begin . ,(##sys#append (reverse exps) (list (expand body2))))
			    (loop (##sys#slot body2 1) (cons x exps)) ) ) ) ) )
	    (let ([vars (reverse vars)])
	      `(let ,(##sys#map (lambda (v) (##sys#list v (##sys#list '##core#undefined))) 
				(apply ##sys#append vars mvars) )
		 ,@(map (lambda (v x) `(##core#set! ,v ,x)) vars (reverse vals))
		 ,@(map (lambda (vs x)
			  (let ([tmps (##sys#map gensym vs)])
			    `(##sys#call-with-values
			      (lambda () ,x)
			      (lambda ,tmps 
				,@(map (lambda (v t) `(##core#set! ,v ,t)) vs tmps) ) ) ) ) 
			(reverse mvars)
			(reverse mvals) )
		 ,@body) ) ) )
      (define (expand body)
	(let loop ([body body] [vars '()] [vals '()] [mvars '()] [mvals '()])
	  (if (not (pair? body))
	      `(begin ,@body)
	      (let* ([x (##sys#slot body 0)]
		     [rest (##sys#slot body 1)] 
		     [head (and (pair? x) (##sys#slot x 0))] )
		(cond [(not head) (fini vars vals mvars mvals body)]
		      [(eq? 'define head)
		       (##sys#check-syntax 'define x '(define _ . #(_ 1)) #f)
		       (let ([head (cadr x)])
			 (cond [(not (pair? head))
				(##sys#check-syntax 'define x '(define variable _) #f)
				(loop rest (cons head vars) (cons (caddr x) vals) mvars mvals) ]
			       [else
				(##sys#check-syntax 'define x '(define (variable . lambda-list) . #(_ 1)) #f)
				(loop rest
				      (cons (##sys#slot head 0) vars)
				      (cons `(lambda ,(##sys#slot head 1) ,@(cddr x)) vals)
				      mvars mvals) ] ) ) ]
		      [(eq? 'define-values head)
		       (##sys#check-syntax 'define-values x '(define-values #(_ 0) _) #f)
		       (loop rest vars vals (cons (cadr x) mvars) (cons (caddr x) mvals)) ]
		      [(eq? 'begin head)
		       (##sys#check-syntax 'begin x '(begin . #(_ 1)) #f)
		       (loop (##sys#append (##sys#slot x 1) rest) vars vals mvars mvals) ]
		      [else (fini vars vals mvars mvals body)] ) ) ) ) )
      (expand body) ) ) )

(define ##sys#match-expression
  (lambda (exp pat vars)
    (let ((env '()))
      (define (mwalk x p)
	(cond ((or (not (##core#inline "C_blockp" p)) (not (##core#inline "C_pairp" p)))
	       (cond ((assq p env) => (lambda (a) (equal? x (##sys#slot a 1))))
		     ((memq p vars)
		      (set! env (cons (cons p x) env))
		      #t)
		     (else (eq? x p)) ) )
	      ((or (not (##core#inline "C_blockp" x)) (not (##core#inline "C_pairp" x))) #f)
	      ((mwalk (##sys#slot x 0) (##sys#slot p 0))
	       (mwalk (##sys#slot x 1) (##sys#slot p 1)) )
	      (else #f) ) )
      (and (mwalk exp pat) env) ) ) )


;;; Lo-level hashtable support:

(define ##sys#hash-symbol
  (let ([cache-s #f]
	[cache-h #f] )
    (lambda (s n)
      (if (eq? s cache-s)
	  (##core#inline "C_fixnum_modulo" cache-h n)
	  (let ([h (##core#inline "C_hash_string" (##sys#slot s 1))])
	    (set! cache-s s)
	    (set! cache-h h)
	    (##core#inline "C_fixnum_modulo" h n) ) ) ) ) )

(define (##sys#hash-table-ref ht key)
  (let ((k (##sys#hash-symbol key (##core#inline "C_block_size" ht))))
    (let loop ((bucket (##sys#slot ht k)))
      (if (eq? bucket '())
	  #f
	  (let ((b (##sys#slot bucket 0)))
	    (if (eq? key (##sys#slot b 0))
		(##sys#slot b 1)
		(loop (##sys#slot bucket 1)) ) ) ) ) ) )

(define ##sys#hash-table-set! 
  (lambda (ht key val)
    (let* ((k (##sys#hash-symbol key (##core#inline "C_block_size" ht)))
	   (bucket0 (##sys#slot ht k)) )
      (let loop ((bucket bucket0))
	(if (eq? bucket '())
	    (##sys#setslot ht k (cons (cons key val) bucket0))
	    (let ((b (##sys#slot bucket 0)))
	      (if (eq? key (##sys#slot b 0))
		  (##sys#setslot b 1 val)
		  (loop (##sys#slot bucket 1)) ) ) ) ) ) ) )

(define (##sys#hash-table-for-each p ht)
  (let ((len (##core#inline "C_block_size" ht)))
    (do ((i 0 (fx+ i 1)))
	((fx>= i len))
      (##sys#for-each (lambda (bucket) 
		   (p (##sys#slot bucket 0)
		      (##sys#slot bucket 1) ) )
		 (##sys#slot ht i) ) ) ) )

(define ##sys#hash-table-location
  (let ([unbound (##sys#slot '##sys#arbitrary-unbound-symbol 0)])
    (lambda (ht key addp)
      (let* ([k (##sys#hash-symbol key (##sys#size ht))]
	     [bucket0 (##sys#slot ht k)] )
	(let loop ([bucket bucket0])
	  (if (null? bucket)
	      (and addp
		   (let ([p (vector key unbound #t)])
		     (##sys#setslot ht k (cons p bucket0))
		     p) )
	      (let ([b (##sys#slot bucket 0)])
		(if (eq? key (##sys#slot b 0))
		    b
		    (loop (##sys#slot bucket 1)) ) ) ) ) ) ) ) )


;;; Compile lambda to closure:

(define ##sys#eval-environment #f)
(define ##sys#unqualified-quoted-symbols #f)
(define ##sys#environment-is-mutable #f)

(define ##sys#compile-to-closure
  (let ([macroexpand-1 macroexpand-1]
	[macro? macro?]
	[write write]
	[cadadr cadadr]
	[reverse reverse]
	[with-input-from-file with-input-from-file]
	[unbound (##sys#slot '##sys#arbitrary-unbound-symbol 0)]
	[display display] )
    (lambda (exp env me)

      (define (lookup var e)
	(let loop ((envs e) (ei 0))
	  (cond ((null? envs) (values #f var))
		((posq var (##sys#slot envs 0)) => (lambda (p) (values ei p)))
		(else (loop (##sys#slot envs 1) (fx+ ei 1))) ) ) )

      (define (defined? var e)
	(receive (i j) (lookup var e) i) )

      (define (undefine vars e)
	(let loop ([envs e])
	  (if (null? envs)
	      '()
	      (let ([envi (##sys#slot envs 0)])
		(cons
		 (let delq ([ee envi])
		   (if (null? ee)
		       '()
		       (let ([h (##sys#slot ee 0)]
			     [r (##sys#slot ee 1)] )
			 (if (memq h vars)
			     r
			     (cons h (delq r)) ) ) ) )
		 (loop (##sys#slot envs 1)) ) ) ) ) )

      (define (posq x lst)
	(let loop ((lst lst) (i 0))
	  (cond ((null? lst) #f)
		((eq? x (##sys#slot lst 0)) i)
		(else (loop (##sys#slot lst 1) (fx+ i 1))) ) ) )

      (define (macroexpand-1-checked x e me)
	(let ([x2 (##sys#secondary-macroexpand (macroexpand-1 x me))])
	  (if (pair? x2)
	      (let ([h (##sys#slot x2 0)])
		(if (and (eq? h 'let) (not (defined? 'let e)))
		    (let ([next (##sys#slot x2 1)])
		      (if (and (pair? next) (symbol? (##sys#slot next 0)))
			  (macroexpand-1-checked x2 e me)
			  x2) )
		    x2) )
	      x2) ) )

      (define (compile x e h me)
	(cond [(symbol? x)
	       (receive (i j) (lookup x e)
		 (cond [(not i)
			(let ([y (macroexpand-1-checked x e me)])
			  (if (eq? x y)
			      (if ##sys#eval-environment
				  (let ([loc (##sys#hash-table-location ##sys#eval-environment x #t)])
				    (unless loc (##sys#error "reference to undefined identifier" x))
				    (cond-expand 
				     [unsafe (lambda v (##sys#slot loc 1))]
				     [else
				      (lambda v 
					(let ([val (##sys#slot loc 1)])
					  (if (eq? unbound val)
					      (##sys#error "unbound variable" x)
					      val) ) ) ] ) )
				  (cond-expand
				   [unsafe (lambda v (##core#inline "C_slot" x 0))]
				   [else (lambda v (##core#inline "C_retrieve" x))] ) )
			      (compile y e h me) ) ) ]
		       [(zero? i) (lambda (v) (##sys#slot (##sys#slot v 0) j))]
		       [else (lambda (v) (##sys#slot (##core#inline "C_u_i_list_ref" v i) j))] ) ) ]
	      [(number? x)
	       (case x
		 [(-1) (lambda v -1)]
		 [(0) (lambda v 0)]
		 [(1) (lambda v 1)]
		 [(2) (lambda v 2)]
		 [else (lambda v x)] ) ]
	      [(boolean? x)
	       (if x
		   (lambda v #t)
		   (lambda v #f) ) ]
	      [(or (char? x)
		   (string? x) )
	       (lambda v x) ]
	      [(not (pair? x)) (##sys#error "syntax error - illegal non-atomic object" x)]
	      [(symbol? (##sys#slot x 0))
	       (let ([head (##sys#slot x 0)])
		 (if (defined? head e)
		     (compile-call x e me)
		     (let ([x2 (macroexpand-1-checked x e me)])
		       (if (eq? x2 x)
			   (case head

			     [(quote)
			      (##sys#check-syntax 'quote x '(quote _) #f)
			      (let* ([c (cadr x)])
				(case c
				  [(-1) (lambda v -1)]
				  [(0) (lambda v 0)]
				  [(1) (lambda v 1)]
				  [(2) (lambda v 2)]
				  [(#t) (lambda v #t)]
				  [(#f) (lambda v #f)]
				  [(()) (lambda v '())]
				  [else (lambda v c)] ) ) ]

			     [(##core#qualified)
			      (compile (cadr x) e h me) ]

			     [(##core#check)
			      (compile (cadr x) e h me) ]

			     [(##core#immutable)
			      (compile (cadr x) e #f me) ]
		   
			     [(##core#undefined) (lambda (v) (##core#undefined))]

			     [(if)
			      (##sys#check-syntax 'if x '(if _ _ . #(_)) #f)
			      (let* ([test (compile (cadr x) e #f me)]
				     [cns (compile (caddr x) e #f me)]
				     [alt (if (pair? (cdddr x))
					      (compile (cadddr x) e #f me)
					      (compile '(##core#undefined) e #f me) ) ] )
				(lambda (v) (if (##core#app test v) (##core#app cns v) (##core#app alt v))) ) ]

			     [(begin)
			      (##sys#check-syntax 'begin x '(begin . #(_ 0)) #f)
			      (let* ([body (##sys#slot x 1)]
				     [len (length body)] )
				(case len
				  [(0) (compile '(##core#undefined) e #f me)]
				  [(1) (compile (##sys#slot body 0) e #f me)]
				  [(2) (let* ([x1 (compile (##sys#slot body 0) e #f me)]
					      [x2 (compile (cadr body) e #f me)] )
					 (lambda (v) (##core#app x1 v) (##core#app x2 v)) ) ]
				  [else
				   (let* ([x1 (compile (##sys#slot body 0) e #f me)]
					  [x2 (compile (cadr body) e #f me)] 
					  [x3 (compile `(begin ,@(##sys#slot (##sys#slot body 1) 1)) e #f me)] )
				     (lambda (v) (##core#app x1 v) (##core#app x2 v) (##core#app x3 v)) ) ] ) ) ]

			     [(set! ##core#set!)
			      (##sys#check-syntax 'set! x '(_ variable _) #f)
			      (let ([var (cadr x)])
				(receive (i j) (lookup var e)
				  (let ([val (compile (caddr x) e var me)])
				    (cond [(not i)
					   (if ##sys#eval-environment
					       (let ([loc (##sys#hash-table-location
							   ##sys#eval-environment 
							   var
							   ##sys#environment-is-mutable) ] )
						 (unless loc (##sys#error "assignment of undefined identifier" var))
						 (if (##sys#slot loc 2)
						     (lambda (v) (##sys#setslot loc 1 (##core#app val v)))
						     (lambda v (##sys#error "assignment to immutable variable" var)) ) )
					       (lambda (v) (##sys#setslot j 0 (##core#app val v))) ) ]
					  [(zero? i) (lambda (v) (##sys#setslot (##sys#slot v 0) j (##core#app val v)))]
					  [else
					   (lambda (v)
					     (##sys#setslot (##core#inline "C_u_i_list_ref" v i) j (##core#app val v)) ) ] ) ) ) ) ]

			     [(let)
			      (##sys#check-syntax 'let x '(let #((variable _) 0) . #(_ 1)) #f)
			      (let* ([bindings (cadr x)]
				     [n (length bindings)] 
				     [vars (map (lambda (x) (car x)) bindings)] 
				     [body (##sys#compile-to-closure
					    (##sys#canonicalize-body (cddr x))
					    (cons vars e) 
					    me) ] )
				(case n
				  [(1) (let ([val (compile (cadar bindings) e #f me)])
					 (lambda (v)
					   (##core#app body (cons (vector (##core#app val v)) v)) ) ) ]
				  [(2) (let ([val1 (compile (cadar bindings) e #f me)]
					     [val2 (compile (cadadr bindings) e #f me)] )
					 (lambda (v)
					   (##core#app body (cons (vector (##core#app val1 v) (##core#app val2 v)) v)) ) ) ]
				  [(3) (let* ([val1 (compile (cadar bindings) e #f me)]
					      [val2 (compile (cadadr bindings) e #f me)] 
					      [t (cddr bindings)]
					      [val3 (compile (cadar t) e #f me)] )
					 (lambda (v)
					   (##core#app body (cons (vector (##core#app val1 v) (##core#app val2 v) (##core#app val3 v)) v)) ) ) ]
				  [(4) (let* ([val1 (compile (cadar bindings) e #f me)]
					      [val2 (compile (cadadr bindings) e #f me)] 
					      [t (cddr bindings)]
					      [val3 (compile (cadar t) e #f me)] 
					      [val4 (compile (cadadr t) e #f me)] )
					 (lambda (v)
					   (##core#app 
					    body
					    (cons (vector (##core#app val1 v) (##core#app val2 v) (##core#app val3 v) (##core#app val4 v)) v)) ) ) ]
				  [else
				   (let ([vals (map (lambda (x) (compile (cadr x) e #f me)) bindings)])
				     (lambda (v)
				       (let ([v2 (##sys#make-vector n)])
					 (do ([i 0 (fx+ i 1)]
					      [vlist vals (##sys#slot vlist 1)] )
					     ((fx>= i n))
					   (##sys#setslot v2 i (##core#app (##sys#slot vlist 0) v)) )
					 (##core#app body (cons v2 v)) ) ) ) ] ) ) ]
			     ;; (compile
			     ;; `((lambda ,(##sys#map (lambda (x) (car x)) bindings) 
			     ;; ,@(##sys#slot (##sys#slot x 1) 1) )
			     ;; ,@(##sys#map (lambda (x) (cadr x)) bindings) )
			     ;; e #f me) ) ]

			     [(lambda)
			      (##sys#check-syntax 'lambda x '(lambda lambda-list . #(_ 1)) #f)
			      (##sys#decompose-lambda-list
			       (cadr x)
			       (lambda (vars argc rest)
				 (let ([body (##sys#compile-to-closure
					      (##sys#canonicalize-body (cddr x))
					      (cons vars e)
					      me) ] )
				   (case argc
				     [(0) (if rest
					      (lambda (v) (lambda r (##core#app body (cons (vector r) v))))
					      (lambda (v) (lambda () (##core#app body (cons #f v))))) ]
				     [(1) (if rest
						  (lambda (v) (lambda (a1 . r) (##core#app body (cons (vector a1 r) v))))
						  (lambda (v) (lambda (a1) (##core#app body (cons (vector a1) v))))) ]
				     [(2) (if rest
					      (lambda (v) (lambda (a1 a2 . r) (##core#app body (cons (vector a1 a2 r) v))))
					      (lambda (v) (lambda (a1 a2) (##core#app body (cons (vector a1 a2) v))))) ]
				     [(3) (if rest
					      (lambda (v) (lambda (a1 a2 a3 . r) (##core#app body (cons (vector a1 a2 a3 r) v))))
					      (lambda (v) (lambda (a1 a2 a3) (##core#app body (cons (vector a1 a2 a3) v))))) ]
				     [(4) (if rest
					      (lambda (v)
						(lambda (a1 a2 a3 a4 . r) (##core#app body (cons (vector a1 a2 a3 a4 r) v))) )
					      (lambda (v)
						(lambda (a1 a2 a3 a4) (##core#app body (##sys#cons (##sys#vector a1 a2 a3 a4) v))))) ]
				     [else (if rest
					       (lambda (v)
						 (lambda as
						   (##core#app body (##sys#cons (apply ##sys#vector (fudge-argument-list argc as)) v)) ) )
					       (lambda (v)
						 (lambda as 
						   (let ([len (length as)])
						     (if (not (fx= len argc))
							 (##sys#error "bad argument count" argc len)
							 (##core#app body (##sys#cons (apply ##sys#vector as) v))) ) ) ) ) ] ) ) ) ) ]

			     [(##core#loop-lambda)
			      (compile `(lambda ,@(cdr x)) e #f me) ]

			     [(##core#named-lambda)
			      (compile `(lambda ,@(cddr x)) e (cadr x) me) ]

			     [(##core#require-for-syntax)
			      (let ([ids (map (lambda (x) ((##sys#compile-to-closure x '() '()) '())) (cdr x))])
				(apply ##sys#require ids)
				(let ([rs (##sys#lookup-runtime-requirements ids)])
				  (compile
				   (if (null? rs)
				       '(##core#undefined)
				       `(##sys#require ,@(map (lambda (x) `',x) rs)) )
				   e #f me) ) ) ]

			     [(##core#elaborationtimeonly ##core#elaborationtimetoo) ; <- Note this!
			      (##core#app (##sys#compile-to-closure (cadr x) '() '()) '())
			      (compile '(##core#undefined) e #f me) ]

			     [(##core#compiletimetoo)
			      (compile (cadr x) e #f me) ]

			     [(##core#compiletimeonly ##core#declare ##core#callunit) 
			      (compile '(##core#undefined) e #f me) ]

                             [(##core#define-inline ##core#define-constant)
                              (compile `(set! ,(cadadr x) ,@(cddr x)) e #f me) ]
                   
			     [(##core#include)
			      (compile
			       (##sys#compiler-toplevel-macroexpand-hook
				(with-input-from-file (##sys#resolve-include-filename (cadadr x))
				  (lambda ()
				    (do ([x (read) (read)]
					 [xs '() (cons x xs)] )
					((eof-object? x) 
					 `(begin ,@(reverse xs))) ) ) ) )
			       e #f me) ]

			     [(##core#primitive ##core#inline ##core#inline_allocate ##core#foreign-lambda 
			       ##core#define-foreign-variable ##core#define-external-variable ##core#let-location
			       ##core#foreign-lambda* ##core#define-foreign-type)
			      (##sys#error "syntax error - can not evaluate compiler-special-form" x) ]

			     [(##core#app)
			      (compile-call (cdr x) e me) ]

			     [else
			      (cond [##sys#strict-mode (compile-call x e me)]

				    [(eq? head 'let-macro)
				     (##sys#check-syntax 'let-macro x '(let-macro #(list 0) . #(_ 1)) #f)
				     (set! ##sys#syntax-error-culprit #f)
				     (let ([me2 (##sys#expand-local-macrodefs (cadr x))])
				       (compile
					(##sys#canonicalize-body (cddr x))
					(undefine (map (lambda (x) (car x)) me2) e)
					#f
					(##sys#append me2 me) ) ) ]

				    [(eq? head 'location)
				     (##sys#error "syntax error - can not evaluate compiler-special-form" x) ]

				    [else (compile-call x e me)] ) ] )

			   (compile x2 e h me) ) ) ) ) ]

	      [else (compile-call x e me)] ) )

      (define (fudge-argument-list n alst)
	(if (null? alst) 
	    (list alst)
	    (do ([n n (fx- n 1)]
		 [args alst (##sys#slot args 1)]
		 [last #f args] )
		((fx= n 0)
		 (##sys#setslot last 1 (list args))
		 alst) ) ) )

      (define (checked-length lst)
	(let loop ([lst lst] [n 0])
	  (cond [(null? lst) n]
		[(pair? lst) (loop (##sys#slot lst 1) (fx+ n 1))]
		[else #f] ) ) )

      (define (compile-call x e me)
	(let* ([fn (compile (##sys#slot x 0) e #f me)]
	       [args (##sys#slot x 1)]
	       [argc (checked-length args)] )
	  (case argc
	    [(#f) (##sys#error "syntax error - malformed expression" x)]
	    [(0) (lambda (v) ((fn v)))]
	    [(1) (let ([a1 (compile (##sys#slot args 0) e #f me)])
		   (lambda (v) ((##core#app fn v) (##core#app a1 v))) ) ]
	    [(2) (let* ([a1 (compile (##sys#slot args 0) e #f me)]
			[a2 (compile (##core#inline "C_u_i_list_ref" args 1) e #f me)] )
		   (lambda (v) ((##core#app fn v) (##core#app a1 v) (##core#app a2 v))) ) ]
	    [(3) (let* ([a1 (compile (##sys#slot args 0) e #f me)]
			[a2 (compile (##core#inline "C_u_i_list_ref" args 1) e #f me)]
			[a3 (compile (##core#inline "C_u_i_list_ref" args 2) e #f me)] )
		   (lambda (v) ((##core#app fn v) (##core#app a1 v) (##core#app a2 v) (##core#app a3 v))) ) ]
	    [(4) (let* ([a1 (compile (##sys#slot args 0) e #f me)]
			[a2 (compile (##core#inline "C_u_i_list_ref" args 1) e #f me)]
			[a3 (compile (##core#inline "C_u_i_list_ref" args 2) e #f me)] 
			[a4 (compile (##core#inline "C_u_i_list_ref" args 3) e #f me)] )
		   (lambda (v) ((##core#app fn v) (##core#app a1 v) (##core#app a2 v) (##core#app a3 v) (##core#app a4 v))) ) ]
	    [else (let ([as (##sys#map (lambda (a) (compile a e #f me)) args)])
		    (lambda (v) (apply (##core#app fn v) (##sys#map (lambda (a) (##core#app a v)) as))) ) ] ) ) )

      (compile exp env #f me) ) ) )

(define ##sys#eval-handler 
  (make-parameter
   (lambda (x . env)
     (let ([sm ##sys#strict-mode]
	   [mut ##sys#environment-is-mutable]
	   [e #f] )
       (when (pair? env)
	 (let ([env (car env)])
	   (when env
	     (##sys#check-structure env 'environment)
	     (set! e (##sys#slot env 1)) 
	     (set! mut (##sys#slot env 2)) )
	   (when e (set! sm #t)) ) )
       ((fluid-let ([##sys#strict-mode sm]
		    [##sys#environment-is-mutable mut]
		    [##sys#eval-environment e] )
	  (##sys#compile-to-closure x '() '()) )
	'() ) ) ) ) )

(define eval-handler ##sys#eval-handler)
(define (eval . args) (apply (##sys#eval-handler) args))


;;; Expand local macro definitions:

(define (##sys#expand-local-macrodefs defs)
  (let loop ([defs defs])
    (if (null? defs) 
	'()
	(let ([def (##sys#slot defs 0)])
	  (cond [(and (pair? def) (symbol? (car def)))
		 (##sys#check-syntax 'let-macro def '(variable (lambda lambda-list . #(_ 1))))
		 (let ([expander ((##sys#eval-handler) (cadr def))])
		   (cons (cons (car def) (lambda (form) (apply expander (cdr form))))
			 (loop (##sys#slot defs 1))) ) ]
		[else
		 (##sys#check-syntax 'let-macro def '((variable . lambda-list) . #(_ 1)))
		 (let ([expander ((##sys#eval-handler) `(lambda ,(cdar def) ,@(cdr def)))])
		   (cons (cons (caar def) (lambda (form) (apply expander (cdr form))))
			 (loop (##sys#slot defs 1)) ) ) ] ) ) ) ) )


;;; Split lambda-list into its parts:

(define ##sys#decompose-lambda-list
  (let ([reverse reverse])
    (lambda (llist0 k)

      (define (err)
	(set! ##sys#syntax-error-culprit #f)
	(##sys#syntax-error-hook "illegal lambda-list syntax" llist0) )

      (let loop ([llist llist0] [vars '()] [argc 0])
	(cond [(eq? llist '()) (k (reverse vars) argc #f)]
	      [(not (##core#inline "C_blockp" llist)) (err)]
	      [(##core#inline "C_symbolp" llist) (k (reverse (cons llist vars)) argc llist)]
	      [(not (##core#inline "C_pairp" llist)) (err)]
	      [else (loop (##sys#slot llist 1)
			  (cons (##sys#slot llist 0) vars)
			  (fx+ argc 1) ) ] ) ) ) ) )


;;; Loading source/object files:

(define load-verbose (make-parameter #f))

(define (##sys#abort-load) #f)

(define-foreign-variable _dlerror c-string "C_dlerror")

(let ([read read]
      [write write]
      [display display]
      [newline newline]
      [open-input-file open-input-file]
      [close-input-port close-input-port]
      [file-exists? file-exists?]
      [string-append string-append] 
      [load-verbose load-verbose]
      [topentry (##sys#make-c-string "C_toplevel")] )
  (define (has-sep? str)
    (let loop ([i (fx- (##sys#size str) 1)])
      (and (not (zero? i))
	   (or (char=? pds (##core#inline "C_subchar" str i))
	       (loop (fx- i 1)) ) ) ) )
  (define (badfile x)
    (##sys#signal-hook #:type-error 'load "bad argument type - not a port or string" x) )
  (set! ##sys#load 
    (lambda (input evaluator pf)
      (let ([fname 
	     (cond [(port? input) #f]
		   [(not (string? input)) (badfile input)]
		   [(file-exists? input) input]
		   [else
		    (let ([fname2 (string-append input ##sys#load-library-extension)])
		      (if (file-exists? fname2)
			  fname2
			  (let ([fname3 (string-append input source-file-extension)])
			    (if (file-exists? fname3)
				fname3
				input) ) ) ) ] ) ]
	    [evproc (or evaluator (##sys#eval-handler))] ) 
	(when (and (load-verbose) fname)
	  (display "; loading ")
	  (display fname)
	  (display " ...\n") )
	(or (and (string? input)
		 (or (##sys#dload (##sys#make-c-string fname) topentry) 
		     (and (not (has-sep? fname))
			  (##sys#dload (##sys#make-c-string (string-append "." pdss fname)) topentry) ) ) )
	    (call-with-current-continuation
	     (lambda (abrt)
	       (fluid-let ([##sys#read-line-counter 1]
			   [##sys#read-error-with-line-number #t] 
			   [##sys#abort-load (lambda () (abrt #f))] )
		 (let ([in (if fname (open-input-file fname) input)])
		   (##sys#dynamic-wind
		    (lambda () #f)
		    (lambda ()
		      (let ([c1 (peek-char in)])
			(when (char=? c1 (integer->char 127))
			  (##sys#error 'load "unable to load compiled module" fname _dlerror) ) )
		      (do ((x (read in) (read in)))
			  ((eof-object? x))
			(##sys#call-with-values
			 (lambda () (evproc x)) 
			 (lambda results
			   (when pf
			     (for-each
			      (lambda (r) 
				(write r)
				(newline) )
			      results) ) ) ) ) )
		    (lambda () (close-input-port in)) ) ) ) ) ) )
	(##core#undefined) ) ) )
  (set! load
    (lambda (filename . evaluator)
      (fluid-let ([##sys#current-namespace ##sys#current-namespace])
	(##sys#load filename (:optional evaluator #f) #f) ) ) )
  (set! load-noisily
    (lambda (filename . evaluator)
      (fluid-let ([##sys#current-namespace ##sys#current-namespace])
	(##sys#load filename (:optional evaluator #f) #t) ) ) )
  (set! load-srfi-7-program
    (lambda (input . evaluator)
      (fluid-let ([##sys#current-namespace ##sys#current-namespace])
	((:optional evaluator (##sys#eval-handler))
	 (##sys#expand-srfi-7-program
	  ""
	  (let ([in (cond [(port? input) input]
			  [(not (string? input)) (badfile input)]
			  [else (open-input-file input)] ) ] )
	    (##sys#dynamic-wind
	     (lambda () #f)
	     read
	     (lambda () (close-input-port in)) ) )
	  ##sys#syntax-error-hook) ) ) ) ) )

(define ##sys#load-library-extension 
  (case (software-type)
    [(windows) windows-load-library-extension]
    [else unix-load-library-extension] ) )

(define ##sys#default-dynamic-load-libraries
  (case (software-type)
    [(windows)
     (case (build-platform)
       [(msvc) msvc-default-dynamic-load-libraries]
       [else cygwin-default-dynamic-load-libraries] ) ]
    [else unix-default-dynamic-load-libraries] ) )

(define dynamic-load-libraries 
  (make-parameter
   (map (cut string-append <> ##sys#load-library-extension) ##sys#default-dynamic-load-libraries)
   (lambda (x)
     (##sys#check-list x)
     x) ) )

(define ##sys#load-library
  (let ([load-verbose load-verbose]
	[string-append string-append]
	[dynamic-load-libraries dynamic-load-libraries]
	[display display] )
    (lambda (uname lib)
      (when (load-verbose)
	(display "; loading library ")
	(display uname)
	(display " ...\n") )
      (let ([id (##sys#->feature-id uname)])
	(or (memq id ##sys#features)
	    (let ([libs
		   (if lib
		       (##sys#list lib)
		       (cons (string-append (##sys#slot uname 1) ##sys#load-library-extension)
			     (dynamic-load-libraries) ) ) ]
		  [top 
		   (##sys#make-c-string
		    (string-append 
		     "C_"
		     (##sys#string->c-identifier (##sys#slot uname 1)) 
		     "_toplevel") ) ] )
	      (let loop ([libs libs])
		(cond [(null? libs) #f]
		      [(##sys#dload (##sys#make-c-string (##sys#slot libs 0)) top)
		       (unless (memq id ##sys#features) (set! ##sys#features (cons id ##sys#features)))
		       #t]
		      [else (loop (##sys#slot libs 1))] ) ) ) ) ) ) ) )

(define load-library
  (lambda (uname . lib)
    (##sys#check-symbol uname 'load-library)
    (or (##sys#load-library uname (and (pair? lib) (car lib)))
	(##sys#error 'load-library "unable to load library" uname _dlerror) ) ) )

(define ##sys#split-at-separator
  (let ([reverse reverse] )
    (lambda (str sep)
      (let ([len (##sys#size str)])
	(let loop ([items '()] [i 0] [j 0])
	  (cond [(fx>= i len)
		 (reverse (cons (##sys#substring str j len) items)) ]
		[(char=? (##core#inline "C_subchar" str i) sep)
		 (let ([i2 (fx+ i 1)])
		   (loop (cons (##sys#substring str j i) items) i2 i2) ) ]
		[else (loop items (fx+ i 1) j)] ) ) ) ) ) )


;;; Extensions:

(define ##sys#current-package '())
(define ##sys#registry #f)

(define (##sys#check-id id . loc)
  (or (symbol? id)
      (and (list? id)
	   (for-each ##sys#check-id id) )
      (##sys#signal-hook #:type-error (and (pair? loc) (car loc)) "bad argument type - not a valid extension-id" id) ) )

(define (package id)
  (##sys#check-id id 'package)
  (set! ##sys#current-package (if (symbol? id) (list id) id) ) )

(define ##sys#find-registry-path
  (let ([getenv getenv]
	[string-append string-append]
	[file-exists? file-exists?] )
    (lambda ()
      (or (getenv registry-environment-variable)
	  (and-let* ([hp (getenv "HOME")]
		     [p1 (string-append hp pdss registry-local-filename)] )
	    (and (file-exists? p1) p1) )
	  install-lib-home) ) ) )

(define ##sys#load-registry
  (let ([with-input-from-file with-input-from-file]
	[string-append string-append]
	[file-exists? file-exists?]
	[read read] )
    (lambda ()
      (unless ##sys#registry
	(let ([fname (string-append (##sys#find-registry-path) pdss registry-filename)])
	  (if (file-exists? fname)
	      (with-input-from-file fname
		(lambda () 
		  (let loop ([rs '()])
		    (let ([r (read)])
		      (if (eof-object? r)
			  (set! ##sys#registry `(chicken () ,@rs))
			  (loop (cons (with-input-from-file r read) rs)) ) ) ) ) )
	      (set! ##sys#registry '(chicken ())) ) ) )
      ##sys#registry) ) )

(define ##sys#lookup-id
  (let ([reverse reverse])
    (lambda (id prefix)
      (define (lookup pref)
	(let search ([id (append (if pref ##sys#current-package '()) (if (symbol? id) (list id) id))]
		     [n (##sys#load-registry)] 
		     [path '()] )
	  (if (null? id)
	      (values #t n '() (reverse path))
	      (let ([a (assq (##sys#slot id 0) (##sys#slot (##sys#slot n 1) 1))])
		(if a 
		    (let ([f (or (and-let* ([fa (assq 'file (##sys#slot (##sys#slot a 1) 0))]) 
				   (##sys#slot (##sys#slot fa 1) 0) )
				 (symbol->string (##sys#slot a 0)) ) ] )
		      (search (##sys#slot id 1) a (cons f path)) )
		    (values #f n id (reverse path)) ) ) ) ) )
      (receive (f n rid p) (lookup prefix)
	(if f
	    (values f n rid p)
	    (if prefix
		(lookup #f)
		(values #f ##sys#registry id '()) ) ) ) ) ) )

(define (extension-path ext)
  (##sys#check-id ext 'extension-path)
  (let-values ([(f n rid p) (##sys#lookup-id ext #f)])
    (and f 
	 (apply 
	  string-append
	  (##sys#find-registry-path)
	  pdss
	  (let loop ([p p])
	    (let ([a (##sys#slot p 0)]
		  [r (##sys#slot p 1)] )
	      (if (null? r)
		  (list a)
		  (cons a (cons pdss (loop r)) ) ) ) ) ) ) ) )

(define (##sys#provide . ids)
  (for-each
   (lambda (id)
     (##sys#check-id id 'provide)
     (receive (f n rid p) (##sys#lookup-id id #t)
       (let ([cd (##sys#slot n 1)])
	 (if f
	     (##sys#setslot cd 0 (cons '(loaded) (##sys#slot cd 0)))
	     (##sys#setslot 
	      cd 1
	      (append
	       (let build ([id rid])
		 (if (null? id)
		     '()
		     (list (append (list (##sys#slot id 0) '((loaded))) (build (##sys#slot id 1)))) ) )
	       (##sys#slot cd 1) ) ) ) ) ) )
   ids) )

(define provide ##sys#provide)

(define ##sys#provided?
  (let ([andmap andmap])
    (lambda ids
      (andmap
       (lambda (id)
	 (##sys#check-id id 'provided?)
	 (receive (f n rid p) (##sys#lookup-id id #t)
	   (or (and f (null? rid) (assq 'loaded (##sys#slot (##sys#slot n 1) 0)))
	       (memq (##sys#->feature-id id) ##sys#features) ) ) )
       ids) ) ) )

(define provided? ##sys#provided?)

(define ##sys#require
  (let ([file-exists? file-exists?]
	[string-append string-append] )
    (define (strip-ext fname)
      (let loop ([i (fx- (##sys#size fname) 1)])
	(cond [(fx= i 0) fname]
	      [(char=? pathname-extension-separator (##core#inline "C_subchar" fname i))
	       (##sys#substring fname 0 i) ]
	      [else (loop (fx- i 1))] ) ) )
    (lambda ids
      (for-each
       (lambda (id)
	 (##sys#check-id id 'require)
	 (receive (f n rid p) (##sys#lookup-id id #t)
	   (if (and f (null? rid))
	       (let* ([cdn (##sys#slot n 1)]
		      [cadn (##sys#slot cdn 0)]
		      [a1 (assq 'library cadn)]
		      [a2 (assq 'loaded cadn)] )
		 (unless a2
		   (##sys#setslot cdn 0 (cons '(loaded) cadn))
		   (if a1
		       (let* ([ll (##sys#slot a1 1)]
			      [unit (if (null? ll) (##sys#slot n 0) (##sys#slot ll 0))]
			      [lib (and (pair? ll)
					(pair? (##sys#slot ll 1))
					(##sys#slot (##sys#slot ll 1) 0) ) ] )
			 (##sys#load-library unit lib) )
		       (let ([pathname
			      (let conc ([p p])
				(let ([pi (##sys#slot p 0)]
				      [r (##sys#slot p 1)] )
				  (if (null? r)
				      (let ([sf (or (assq 'source cadn) (assq 'syntax cadn))]
					    [ff (assq 'file cadn)] )
					(string-append
					 (if ff (strip-ext pi) pi)
					 (if sf
					     source-file-extension
					     ##sys#load-library-extension) ) )
				      (string-append
				       pi
				       pdss
				       (conc r) ) ) ) ) ] )
			 (##sys#load
			  (if (file-exists? pathname)
			      pathname
			      (string-append (##sys#find-registry-path) pdss pathname) )
			  #f #f) ) ) ) )
	       (let ([f2 (and (symbol? id)
			      (##sys#lookup-required-file (##sys#slot id 1)) ) ] )
		 (if f2
		     (##sys#load f2 #f #f)
		     (##sys#error 'require "can not find required extension" id) ) ) ) ) )
       ids) ) ) )

(define require ##sys#require)

(define ##sys#lookup-required-file
  (let ([file-exists? file-exists?]
	[append append]
	[getenv getenv]
	[string-append string-append] 
	[flag #f] )
    (lambda (fname)
      (define (test fname)
	(let ([fname2 (string-append fname ##sys#load-library-extension)])
	  (if (file-exists? fname2)
	      fname2
	      (let ([fname3 (string-append fname source-file-extension)])
		(and (file-exists? fname3)
		     fname3) ) ) ) )
      (or (test fname)
	  (let loop ([paths
		      (if (or (pair? ##sys#include-pathnames) flag)
			  ##sys#include-pathnames
			  (let ([cip (getenv "CHICKEN_INCLUDE_PATH")]
				[ch (getenv "CHICKEN_HOME")] )
			    (set! flag #t)
			    (set! ##sys#include-pathnames
			      (append 
			       (if cip (##sys#split-at-separator cip #\;) '())
			       (if ch (list ch) '())
			       ".") ) ) ) ] )
	    (and (pair? paths)
		 (or (test (string-append (##sys#slot paths 0) pdss fname))
		     (loop (##sys#slot paths 1)) ) ) ) ) ) ) )

(define (##sys#lookup-runtime-requirements ids)
  (let loop1 ([ids ids])
    (if (null? ids)
	'()
	(append
	 (let-values ([(f n rid _) (##sys#lookup-id (car ids) #t)])
	   (if (and f (null? rid))
	       (let loop2 ([props (cadr n)])
		 (if (null? props)
		     '()
		     (let ([a (##sys#slot props 0)]
			   [r (##sys#slot props 1)] )
		       (if (eq? 'require-at-runtime (##sys#slot a 0))
			   (append (##sys#slot a 1) (loop2 r))
			   (loop2 r) ) ) ) )
	       '() ) )
	 (loop1 (##sys#slot ids 1)) ) ) ) )


;;; Convert string into valid C-identifer:

(define ##sys#string->c-identifier
  (let ([string-copy string-copy])
    (lambda (str)
      (let* ([s2 (string-copy str)]
	     [n (##sys#size s2)] )
	(do ([i 0 (fx+ i 1)])
	    ((fx>= i n) s2)
	  (let ([c (##core#inline "C_subchar" s2 i)])
	    (when (and (not (char-alphabetic? c)) (or (not (char-numeric? c)) (fx= i 0)))
	      (##core#inline "C_setsubchar" s2 i #\_) ) ) ) ) ) ) )


;;; Environments:

(define ##sys#r4rs-environment (make-vector environment-table-size '()))
(define ##sys#r5rs-environment #f)
(define ##sys#interaction-environment (##sys#make-structure 'environment #f #t))

(define ##sys#copy-env-table
  (lambda (e mff mf)
    (let* ([s (##sys#size e)]
	   [e2 (##sys#make-vector s '())] )
      (do ([i 0 (fx+ i 1)])
	  ((fx>= i s) e2)
	(##sys#setslot 
	 e2 i
	 (let copy ([b (##sys#slot e i)])
	   (if (null? b)
	       '()
	       (let ([bi (##sys#slot b 0)])
		 (cons (vector 
			(##sys#slot bi 0)
			(##sys#slot bi 1)
			(if mff mf (##sys#slot bi 2)) )
		       (copy (##sys#slot b 1)) ) ) ) ) ) ) ) ) )

(define (interaction-environment) ##sys#interaction-environment)

(define scheme-report-environment
  (lambda (n . mutable)
    (##sys#check-exact n 'scheme-report-environment)
    (let ([mf (and (pair? mutable) (car mutable))])
      (case n
	[(4) (##sys#make-structure 'environment (##sys#copy-env-table ##sys#r4rs-environment #t mf) mf)]
	[(5) (##sys#make-structure 'environment (##sys#copy-env-table ##sys#r5rs-environment #t mf) mf)]
	[else (##sys#error 'scheme-report-environment "no support for version" n)] ) ) ) )

(define null-environment
  (let ([make-vector make-vector])
    (lambda (n . mutable)
      (##sys#check-exact n 'null-environment)
      (when (or (fx< n 4) (fx> n 5))
	(##sys#error 'null-environment "no support for version" n) )
      (##sys#make-structure
       'environment
       (make-vector environment-table-size '())
       (and (pair? mutable) (car mutable)) ) ) ) )

(let ()
  (define (initb ht) 
    (lambda (b)
      (let ([loc (##sys#hash-table-location ht b #t)])
	(##sys#setslot loc 1 (##sys#slot b 0)) ) ) )
  (for-each 
   (initb ##sys#r4rs-environment)
   '(not boolean? eq? eqv? equal? pair? cons car cdr caar cadr cdar cddr caaar caadr cadar caddr cdaar
     cdadr cddar cdddr caaaar caaadr caadar caaddr cadaar cadadr cadddr cdaaar cdaadr cdadar cdaddr
     cddaar cddadr cdddar cddddr set-car! set-cdr! null? list? list length list-tail list-ref
     append reverse memq memv member assq assv assoc symbol? symbol->string string->symbol
     number? integer? exact? real? complex? inexact? rational? zero? odd? even? positive? negative?
     max min + - * / = > < >= <= quotient remainder modulo gcd lcm abs floor ceiling truncate round
     exact->inexact inexact->exact exp log expt sqrt sin cos tan asin acos atan number->string
     string->number char? char=? char>? char<? char>=? char<=? char-ci=? char-ci<? char-ci>?
     char-ci>=? char-ci<=? char-alphabetic? char-whitespace? char-numeric? char-upper-case?
     char-lower-case? char-upcase char-downcase char->integer integer->char string? string=?
     string>? string<? string>=? string<=? string-ci=? string-ci<? string-ci>? string-ci>=? string-ci<=?
     make-string string-length string-ref string-set! string-append string-copy string->list 
     list->string substring string-fill! vector? make-vector vector-ref vector-set! string vector
     vector-length vector->list list->vector vector-fill! procedure? map for-each apply force 
     call-with-current-continuation input-port? output-port? current-input-port current-output-port
     call-with-input-file call-with-output-file open-input-file open-output-file close-input-port
     close-output-port load transcript-on transcript-off read eof-object? read-char peek-char
     write display write-char newline with-input-from-file with-output-to-file ##sys#call-with-values
     ##sys#values ##sys#dynamic-wind
     ##sys#list->vector ##sys#list ##sys#append ##sys#cons ##sys#make-promise) )
  (set! ##sys#r5rs-environment (##sys#copy-env-table ##sys#r4rs-environment #t #t))
  (for-each
   (initb ##sys#r5rs-environment)
   '(dynamic-wind values call-with-values eval scheme-report-environment null-environment interaction-environment) ) )


;;; Find included file:

(define ##sys#include-pathnames '())

(define ##sys#resolve-include-filename
  (let ((file-exists? file-exists?)
	(string string)
	(string-append string-append) )
    (lambda (fname . flag)

      (define (test fname)
	(if (file-exists? fname) 
	    fname
	    (let ((fname2 (string-append fname source-file-extension)))
	      (and (file-exists? fname2) fname2) ) ) )
		
      (or (test fname)
	  (let loop ((paths ##sys#include-pathnames))
	    (cond ((eq? paths '()) (if (pair? flag) (car flag) fname))
		  ((test (string-append (##sys#slot paths 0)
					(string pathname-directory-separator)
					fname) ) )
		  (else (loop (##sys#slot paths 1))) ) ) ) ) ) )


;;; Print timing information (support for "time" macro):

(define ##sys#display-times
  (let* ((display display)
	 (spaces 
	  (lambda (n)
	    (do ((i n (fx- i 1)))
		((fx<= i 0))
	      (display #\space) ) ) )
	 (display-rj 
	  (lambda (x w)
	    (let* ((xs (if (zero? x) "0" (number->string x)))
		   (xslen (##core#inline "C_block_size" xs)) )
	      (spaces (fx- w xslen))
	      (display xs) ) ) ) )
    (lambda (info)
      (display-rj (##sys#slot info 0) 8)
      (display " seconds elapsed\n") 
      (display-rj (##sys#slot info 1) 8)
      (display " seconds in (major) GC\n")
      (display-rj (##sys#slot info 2) 8)
      (display " mutations\n")
      (display-rj (##sys#slot info 3) 8)
      (display " minor GCs\n")
      (display-rj (##sys#slot info 4) 8)
      (display " major GCs\n") ) ) )


;;; General syntax checking routine:

(define ##sys#line-number-database #f)
(define ##sys#syntax-error-hook ##sys#error)
(define ##sys#syntax-error-culprit #f)

(define (get-line-number sexp)
  (and ##sys#line-number-database
       (##core#inline "C_blockp" sexp)
       (##core#inline "C_pairp" sexp)
       (let ([head (##sys#slot sexp 0)])
	 (and (symbol? head)
	      (cond [(##sys#hash-table-ref ##sys#line-number-database head)
		     => (lambda (pl)
			  (let ([a (assq sexp pl)])
			    (and a (##sys#slot a 1)) ) ) ]
		    [else #f] ) ) ) ) )

(define ##sys#check-syntax
  (let ([string-append string-append]
	[keyword? keyword?]
	[get-line-number get-line-number]
	[symbol->string symbol->string] )
    (lambda (id exp pat . culprit)

      (define (test x pred msg)
	(unless (pred x) (err msg)) )

      (define (err msg)
	(let* ([sexp ##sys#syntax-error-culprit]
	       [ln (get-line-number sexp)] )
	  (##sys#syntax-error-hook
	   (if ln 
	       (string-append "(" (symbol->string id) ") in line " (number->string ln) " - " msg)
	       (string-append "(" (symbol->string id) ") " msg) )
	   exp) ) )

      (define (lambda-list? x)
	(let loop ((x x))
	  (cond ((eq? x '()))
		((not (##core#inline "C_blockp" x)) #f)
		((##core#inline "C_symbolp" x) (not (keyword? x)))
		((##core#inline "C_pairp" x)
		 (let ((s (##sys#slot x 0)))
		   (if (or (not (##core#inline "C_blockp" s)) (not (##core#inline "C_symbolp" s)))
		       #f
		       (loop (##sys#slot x 1)) ) ) ) 
		(else #f) ) ) )

      (define (proper-list? x)
	(let loop ((x x))
	  (cond ((eq? x '()))
		((and (##core#inline "C_blockp" x) (##core#inline "C_pairp" x)) (loop (##sys#slot x 1)))
		(else #f) ) ) )

      (when (pair? culprit) (set! ##sys#syntax-error-culprit (car culprit)))
      (let walk ((x exp) (p pat))
	(cond ((and (##core#inline "C_blockp" p) (##core#inline "C_vectorp" p))
	       (let* ((p2 (##sys#slot p 0))
		      (vlen (##core#inline "C_block_size" p))
		      (min (if (fx> vlen 1) 
			       (##sys#slot p 1)
			       0) )
		      (max (cond ((eq? vlen 1) 1)
				 ((fx> vlen 2) (##sys#slot p 2))
				 (else 99999) ) ) )
		 (do ((x x (##sys#slot x 1))
		      (n 0 (fx+ n 1)) )
		     ((eq? x '())
		      (if (fx< n min)
			  (err "not enough arguments") ) )
		   (cond ((fx>= n max) 
			  (err "too many arguments") )
			 ((or (not (##core#inline "C_blockp" x)) (not (##core#inline "C_pairp" x)))
			  (err "not a proper list") )
			 (else (walk (##sys#slot x 0) p2) ) ) ) ) )
	      ((not (##core#inline "C_blockp" p))
	       (if (not (eq? p x)) (err "unexpected object")) )
	      ((##core#inline "C_symbolp" p)
	       (case p
		 ((_) #t)
		 ((pair) (test x pair? "pair expected"))
		 ((variable) (test x (lambda (x) (and (symbol? x) (not (keyword? x)))) "identifer expected"))
		 ((symbol) (test x symbol? "symbol expected"))
		 ((list) (test x proper-list? "proper list expected"))
		 ((number) (test x number? "number expected"))
		 ((string) (test x string? "string expected"))
		 ((lambda-list) (test x lambda-list? "lambda-list expected"))
		 (else (test x (lambda (y) (eq? y p)) "missing keyword")) ) )
	      ((or (not (##core#inline "C_blockp" x)) (not (##core#inline "C_pairp" x)))
	       (err "incomplete form") )
	      (else
	       (walk (##sys#slot x 0) (##sys#slot p 0))
	       (walk (##sys#slot x 1) (##sys#slot p 1)) ) ) ) ) ) )


;;; Macro definitions:

(##sys#register-macro-2
 'define
 (lambda (form)
   (let ((head (car form))
	 (body (cdr form)) )
     (cond ((not (and (##core#inline "C_blockp" head) (##core#inline "C_pairp" head)))
	    (##sys#check-syntax 'define head 'symbol)
	    (##sys#check-syntax 'define body '#(_ 1))
	    `(##core#set! ,head ,(car body)) )
	   (else
	    (##sys#check-syntax 'define head '(symbol . lambda-list))
	    (##sys#check-syntax 'define body '#(_ 1))
	    `(##core#set! ,(car head) (lambda ,(cdr head) ,@body)) ) ) ) ) )

(##sys#register-macro-2
 'and
 (lambda (body)
   (if (eq? body '())
       #t
       (let ((rbody (##sys#slot body 1))
	     (hbody (##sys#slot body 0)) )
	 (if (eq? rbody '())
	     hbody
	     `(if ,hbody (and ,@rbody) #f) ) ) ) ) )

(##sys#register-macro-2
 'or 
 (let ((gensym gensym))
   (lambda (body)
     (if (eq? body '())
	 #f
	 (let ((rbody (##sys#slot body 1))
	       (hbody (##sys#slot body 0)) )
	   (if (eq? rbody '())
	       hbody
	       (let ((tmp (gensym)))
		 `(let ((,tmp ,hbody))
		    (if ,tmp ,tmp (or ,@rbody)) ) ) ) ) ) ) ) )

(##sys#register-macro-2
 'cond
 (let ((gensym gensym))
   (lambda (body)
     (let expand ((clauses body))
       (if (not (pair? clauses))
	   '(##core#undefined)
	   (let ((clause (##sys#slot clauses 0))
		 (rclauses (##sys#slot clauses 1)) )
	     (##sys#check-syntax 'cond clause '#(_ 1))
	     (cond ((eq? 'else (car clause)) `(begin ,@(cdr clause)))
		   ((eq? (cdr clause) '()) `(or ,(car clause) ,(expand rclauses)))
		   ((eq? '=> (car (cdr clause)))
		    (let ((tmp (gensym)))
		      `(let ((,tmp ,(car clause)))
			 (if ,tmp
			     (,(car (cdr (cdr clause))) ,tmp)
			     ,(expand rclauses) ) ) ) )
		   (else `(if ,(car clause) 
			      (begin ,@(cdr clause))
			      ,(expand rclauses) ) ) ) ) ) ) ) ) )

(##sys#register-macro-2
 'case
 (let ((gensym gensym))
   (lambda (form)
     (let ((exp (car form))
	   (body (cdr form)) )
       (let ((tmp (gensym)))
	 `(let ((,tmp ,exp))
	    ,(let expand ((clauses body))
	       (if (not (pair? clauses))
		   '(##core#undefined)
		   (let ((clause (##sys#slot clauses 0))
			 (rclauses (##sys#slot clauses 1)) )
		     (##sys#check-syntax 'case clause '#(_ 1))
		     (if (eq? 'else (car clause))
			 `(begin ,@(cdr clause))
			 `(if (or ,@(##sys#map (lambda (x) `(eqv? ,tmp ',x)) (car clause)))
			      (begin ,@(cdr clause)) 
			      ,(expand rclauses) ) ) ) ) ) ) ) ) ) ) )

(##sys#register-macro-2
 'let*
 (lambda (form)
   (let ((bindings (car form))
	 (body (cdr form)) )
     (##sys#check-syntax 'let* bindings '#((symbol _) 0))
     (##sys#check-syntax 'let* body '#(_ 1))
     (let expand ((bs bindings))
       (if (eq? bs '())
	   (##sys#canonicalize-body body)
	   `(let (,(car bs)) ,(expand (cdr bs))) ) ) ) ) )

(##sys#register-macro-2
 'letrec
 (lambda (form)
   (let ((bindings (car form))
	 (body (cdr form)) )
     (##sys#check-syntax 'letrec bindings '#((symbol _) 0))
     (##sys#check-syntax 'letrec body '#(_ 1))
     `(let ,(##sys#map (lambda (b) (list (car b) '(##core#undefined))) bindings)
	(begin ,@(##sys#append (##sys#map (lambda (b) `(##core#set! ,(car b) ,(cadr b))) bindings)
			   (list (##sys#canonicalize-body body)) ) ) ) ) ) )

(define (##sys#enable-strict-letrec)
  (##sys#register-macro-2
   'letrec
   (lambda (form)
     (let ((bindings (car form))
	   (body (cdr form)) )
       (##sys#check-syntax 'letrec bindings '#((symbol _) 0))
       (##sys#check-syntax 'letrec body '#(_ 1))
       (let* ([vars (map car bindings)]
	      [tmps (map gensym vars)] )
	 `(let ,(map (lambda (v) (list v #f)) vars)
	    (let ,(map (lambda (t b) (list t (cadr b))) tmps bindings)
	      (begin 
		,@(append 
		   (map (lambda (v t) `(##core#set! ,v ,t)) vars tmps)
		   (list (##sys#canonicalize-body body)) ) ) ) ) ) ) ) ) )

(##sys#register-macro
 'do
 (let ((gensym gensym))
   (lambda (bindings test . body)
     (##sys#check-syntax 'do bindings '#((symbol _ . #(_)) 0))
     (##sys#check-syntax 'do test '#(_ 1))
     (let ((dovar (gensym "do")))
       `(let ,dovar ,(##sys#map (lambda (b) (list (car b) (car (cdr b)))) bindings)
	     (if ,(car test)
		 ,(let ((tbody (cdr test)))
		    (if (eq? tbody '())
			'(##core#undefined)
			`(begin ,@tbody) ) )
		 (begin
		   ,(if (eq? body '())
			'(##core#undefined)
			(##sys#canonicalize-body body) )
		   (##core#app
		    ,dovar ,@(##sys#map (lambda (b) 
					  (if (eq? (cdr (cdr b)) '())
					      (car b)
					      (car (cdr (cdr b))) ) )
					bindings) ) ) ) ) ) ) ) )

(##sys#register-macro
 'quasiquote
 (let ((vector->list vector->list))
   (lambda (form)
     
     (define (walk x n) (simplify (walk1 x n)))

     (define (walk1 x n)
       (if (##core#inline "C_blockp" x)
	   (cond ((##core#inline "C_vectorp" x)
		  `(##sys#list->vector ,(walk (vector->list x) n)) )
		 ((not (##core#inline "C_pairp" x)) `(quote ,x))
		 (else
		  (let ((head (##sys#slot x 0))
			(tail (##sys#slot x 1)) )
		    (case head
		      ((unquote)
		       (if (and (##core#inline "C_blockp" tail) (##core#inline "C_pairp" tail))
			   (let ((hx (##sys#slot tail 0)))
			     (if (eq? n 0)
				 hx
				 (list '##sys#list '(quote unquote)
				       (walk hx (fx- n 1)) ) ) )
			   '(quote unquote) ) )
		      ((quasiquote)
		       (if (and (##core#inline "C_blockp" tail) (##core#inline "C_pairp" tail))
			   `(##sys#list (quote quasiquote) 
				   ,(walk (##sys#slot tail 0) (fx+ n 1)) ) 
			   (list '##sys#cons (list 'quote 'quasiquote) (walk tail n)) ) )
		      (else
		       (if (and (##core#inline "C_blockp" head) (##core#inline "C_pairp" head))
			   (let ((hx (##sys#slot head 0))
				 (tx (##sys#slot head 1)) )
			     (if (and (eq? hx 'unquote-splicing)
				      (##core#inline "C_blockp" tx)
				      (##core#inline "C_pairp" tx) )
				 (let ((htx (##sys#slot tx 0)))
				   (if (eq? n 0)
				       `(##sys#append ,htx
						 ,(walk tail n) )
				       `(##sys#cons (##sys#list 'unquote-splicing
							,(walk htx (fx- n 1)) )
					       ,(walk tail n) ) ) )
				 `(##sys#cons ,(walk head n) ,(walk tail n)) ) )
			   `(##sys#cons ,(walk head n) ,(walk tail n)) ) ) ) ) ) )
	   `(quote ,x) ) )

     (define (simplify x)
       (cond ((##sys#match-expression x '(##sys#cons a '()) '(a))
	      => (lambda (env) (simplify `(##sys#list ,(##sys#slot (assq 'a env) 1)))) )
	     ((##sys#match-expression x '(##sys#cons a (##sys#list . b)) '(a b))
	      => (lambda (env)
		   (let ([bxs (assq 'b env)])
		     (if (fx< (length bxs) 32)
			 (simplify `(##sys#list ,(##sys#slot (assq 'a env) 1)
					    ,@(##sys#slot bxs 1) ) ) 
			 x) ) ) )
	     ((##sys#match-expression x '(##sys#append a '()) '(a))
	      => (lambda (env) (##sys#slot (assq 'a env) 1)) )
	     (else x) ) )
     
     (walk form 0) ) ) )

(##sys#register-macro
 'delay
 (lambda (x) `(##sys#make-promise (lambda () ,x))) )


;;; SRFI-0 support code:

(set! ##sys#features (append '(#:srfi-8 #:srfi-6 #:srfi-2 #:srfi-0 #:srfi-10 #:srfi-7 #:srfi-9) ##sys#features))

(define (##sys#test-feature f)
  (let ([f (##sys#->feature-id f)])
    (memq f ##sys#features) ) )


;;;; Read-Eval-Print loop:

(define ##sys#repl-eval-hook #f)
(define ##sys#repl-print-length-limit #f)
(define ##sys#repl-read-hook #f)

(define ##sys#read-prompt-hook
  (let ((display display))
    (lambda () (display ">>> ")) ) )

(define read-eval-print-loop
  (let ([eval eval]
	[read read]
	[write write]
	[call-with-current-continuation call-with-current-continuation]
	[display display]
	[reset reset]
	[unregister-feature! unregister-feature!]
	[newline newline] )
    (lambda ()

      (define (writeargs xs)
	(if (or (null? xs) (pair? (cdr xs)) (not (eq? (##core#undefined) (car xs))))
	    (##sys#for-each 
	     (lambda (x)
	       (##sys#with-print-length-limit ##sys#repl-print-length-limit (lambda () (write x)))
	       (newline) )
	     xs) ) )

      (let ([stdin ##sys#standard-input]
	    [stdout ##sys#standard-output]
	    [stderr ##sys#standard-error] 
	    [ehandler (##sys#error-handler)] 
	    [rhandler (##sys#reset-handler)] )

	(define (saveports)
	  (set! stdin ##sys#standard-input)
	  (set! stdout ##sys#standard-output)
	  (set! stderr ##sys#standard-error) )

	(define (resetports)
	  (set! ##sys#standard-input stdin)
	  (set! ##sys#standard-output stdout)
	  (set! ##sys#standard-error stderr) )

	(saveports)
	(##sys#dynamic-wind
	 (lambda ()
	   (##sys#error-handler
	    (lambda (msg . args)
	      (resetports)
	      (display "Error: ")
	      (display msg)
	      (if (fx= 1 (length args))
		  (begin
		    (display ": ")
		    (writeargs args) )
		  (begin
		    (newline)
		    (writeargs args) ) ) ) ) )
	 (lambda ()
	   (let loop ()
	     (call-with-current-continuation
	      (lambda (c)
		(##sys#reset-handler
		 (lambda ()
		   (set! ##sys#read-error-with-line-number #f)
		   (set! ##sys#default-namespace-prefix #f)
		   (set! ##sys#enable-qualifiers #t)
		   (set! ##sys#current-namespace #f)
		   (resetports)
		   (c #f) ) ) ) )
	     (##sys#read-prompt-hook)
	     (let ([exp ((or ##sys#repl-read-hook read))])
	       (unless (eof-object? exp)
		 (receive result ((or ##sys#repl-eval-hook eval) exp)
		   (writeargs result) 
		   (loop) ) ) ) ) )
	 (lambda ()
	   (##sys#error-handler ehandler)
	   (##sys#reset-handler rhandler) ) ) ) ) ) )


;;; SRFI-10:

(define ##sys#sharp-comma-reader-ctors (make-vector 301 '()))

(define (define-reader-ctor spec proc)
  (##sys#check-symbol spec 'define-reader-ctor)
  (##sys#hash-table-set! ##sys#sharp-comma-reader-ctors spec proc) )

(set! ##sys#user-read-hook
  (let ([old ##sys#user-read-hook]
	[read-char read-char]
	[read read] )
    (lambda (char port)
      (cond [(char=? char #\,)
	     (read-char port)
	     (let* ([exp (read port)]
		    [err (lambda () (##sys#read-error "invalid sharp-comma external form" exp))] )
	       (if (or (null? exp) (not (list? exp)))
		   (err)
		   (let ([spec (##sys#slot exp 0)])
		     (if (not (symbol? spec))
			 (err) 
			 (let ([ctor (##sys#hash-table-ref ##sys#sharp-comma-reader-ctors spec)])
			   (if ctor
			       (apply ctor (##sys#slot exp 1))
			       (##sys#read-error "undefined sharp-comma constructor" spec) ) ) ) ) ) ) ]
	    [else (old char port)] ) ) ) )


;;; SRFI-7:

(define ##sys#expand-srfi-7-program
  (let ([string-append string-append]
	[string string] )
    (lambda (path prg failed)

      (define (check fr)
	(if (pair? fr)
	    (case (##sys#slot fr 0)
	      [(not)
	       (##sys#check-syntax 'program-feature-cond fr '(not _))
	       (not (check (##sys#slot (##sys#slot 1) 0))) ]
	      [(and)
	       (##sys#check-syntax 'program-feature-cond fr '(and . #(_ 0)))
	       (let fold ([frs (##sys#slot fr 1)])
		 (or (null? frs)
		     (and (check (##sys#slot frs 0))
			  (fold (##sys#slot frs 1)) ) ) ) ]
	      [(or)
	       (##sys#check-syntax 'program-feature-cond fr '(or . #(_ 0)))
	       (let fold ([frs (##sys#slot fr 1)])
		 (and (pair? frs)
		      (or (check (##sys#slot frs 0))
			  (fold (##sys#slot frs 1)) ) ) ) ]
	      [else (##sys#syntax-error-hook "invalid feature requirement" fr)] )
	    (##sys#test-feature fr) ) )

      (define (check-file f)
	(if (file-exists? f)
	    f
	    (string-append path (string pathname-directory-separator) f) ) )

      (##sys#check-syntax 'program prg '(program . #(_ 0)))
      (let loop ([clauses (##sys#slot prg 1)])
	(if (null? clauses) 
	    '(begin)
	    (let ([clause (##sys#slot clauses 0)]
		  [rest (##sys#slot clauses 1)] )
	      (##sys#check-syntax 'program clause '(symbol . #(_ 0)))
	      (let ([r (##sys#slot clause 1)])
		(case (##sys#slot clause 0)
		  [(requires)
		   (##sys#check-syntax 'program-requires r '#(symbol 1))
		   (for-each
		    (lambda (f)
		      (unless (##sys#test-feature f)
			(failed "missing required feature" f) ) )
		    r)
		   (loop rest) ]
		  [(files)
		   (##sys#check-syntax 'program-files r '#(string 0))
		   `(begin 
		      ,@(map (lambda (f) `(include ,(check-file f))) r)
		      ,(loop rest) ) ]
		  [(code)
		   (##sys#check-syntax 'program-code r '#(_ 0))
		   `(begin ,@r ,(loop rest)) ]
		  [(feature-cond)
		   (##sys#check-syntax 'program-feature-cond r '#(_ 0))
		   (let fold ([fcs r])
		     (if (null? fcs)
			 (failed "no matching `feature-cond' clause")
			 (let ([fc (##sys#slot fcs 0)]
			       [r (##sys#slot fcs 1)] )
			   (##sys#check-syntax 'program-feature-cond fc '#(_ 2))
			   (let ([fr (##sys#slot fc 0)]
				 [frr (##sys#slot fc 1)] )
			     (if (or (eq? 'else fr) (check fr))
				 (loop frr)
				 (fold r) ) ) ) ) ) ]
		  [else (##sys#syntax-error-hook "invalid program clause" clause)] ) ) ) ) ) ) ) )


;;; To catch missing `-hygienic' option:

(##sys#register-macro-2
 'define-syntax
 (lambda (form)
   (##sys#error 'define-syntax "highlevel macros not available - try `-hygienic' option") ) )


;;; Visit:

(define visit
  (let ([read read]
	[write write]
	[display display]
	[newline newline]
	[open-input-file open-input-file]
	[close-input-port close-input-port]
	[file-exists? file-exists?]
	[string-append string-append] 
	[load-verbose load-verbose] )
    (lambda (filename)
      (##sys#check-string filename 'visit)
      (when (load-verbose)
	(display "; visiting ")
	(display filename)
	(display " ...\n") )
      (call-with-current-continuation
       (lambda (abrt)
	 (fluid-let ([##sys#read-line-counter 1]
		     [##sys#read-error-with-line-number #t] )
	   (let ([in (open-input-file filename)])
	     (##sys#dynamic-wind
	      (lambda () #f)
	      (lambda ()
		(do ((x (read in) (read in)))
		    ((eof-object? x))
		  (##sys#visit-toplevel-expression x) ) )
	      (lambda () (close-input-port in)) ) ) ) ) ) ) ) )

(define (##sys#visit-define-hook . _) #f)

(define ##sys#visit-toplevel-expression
  (let ([macroexpand-1 macroexpand-1]
	[cadadr cadadr]
	[reverse reverse]
	[eval eval]
	[with-input-from-file with-input-from-file]
	[display display] )
    (lambda (exp)

      (define (defined? var e)
	(let loop ((envs e) (ei 0))
	  (cond ((null? envs) #f)
		((posq var (##sys#slot envs 0)))
		(else (loop (##sys#slot envs 1) (fx+ ei 1))) ) ) )

      (define (undefine vars e)
	(let loop ([envs e])
	  (if (null? envs)
	      '()
	      (let ([envi (##sys#slot envs 0)])
		(cons
		 (let delq ([ee envi])
		   (if (null? ee)
		       '()
		       (let ([h (##sys#slot ee 0)]
			     [r (##sys#slot ee 1)] )
			 (if (memq h vars)
			     r
			     (cons h (delq r)) ) ) ) )
		 (loop (##sys#slot envs 1)) ) ) ) ) )

      (define (posq x lst)
	(let loop ((lst lst) (i 0))
	  (cond ((null? lst) #f)
		((eq? x (##sys#slot lst 0)) i)
		(else (loop (##sys#slot lst 1) (fx+ i 1))) ) ) )

      (define (macroexpand-1-checked x e me)
	(let ([x2 (##sys#secondary-macroexpand (macroexpand-1 x me))])
	  (if (pair? x2)
	      (let ([h (##sys#slot x2 0)])
		(if (and (eq? h 'let) (not (defined? 'let e)))
		    (let ([next (##sys#slot x2 1)])
		      (if (and (pair? next) (symbol? (##sys#slot next 0)))
			  (macroexpand-1-checked x2 e me)
			  x2) )
		    x2) )
	      x2) ) )

      (define (walk-rest xs e me)
	(for-each (cut walk <> e me) xs) )

      (define (walk x e me)
	(cond [(not (pair? x))]
	      [(symbol? (##sys#slot x 0))
	       (let ([head (##sys#slot x 0)])
		 (if (defined? head e)
		     (walk-rest x e me)
		     (let ([x2 (macroexpand-1-checked x e me)])
		       (if (eq? x2 x)
			   (case head

			     [(quote ##core#qualified ##core#immutable ##core#undefined ##core#compiletimeonly
				     ##core#declare ##core#callunit ##core#primitive ##core#foreign-lambda
				     ##core#define-external-variable ##core#define-foreign-variable
				     ##core#foreign-lambda* ##core#foreign-callback-lambda ##core#foreign-callback-lambda*)]

			     [(##core#check) 
			      (walk (cadr x) e me) ]
		   
			     [(if)
			      (##sys#check-syntax 'if x '(if _ _ . #(_)) #f)
			      (walk-rest (cdr x) e me) ]

			     [(begin)
			      (##sys#check-syntax 'begin x '(begin . #(_ 0)) #f)
			      (walk-rest (cdr x) e me) ]

			     [(set! ##core#set!)
			      (##sys#check-syntax 'set! x '(_ variable _) #f)
			      (walk (caddr x) e me) ]

			     [(let)
			      (##sys#check-syntax 'let x '(let #((variable _) 0) . #(_ 1)) #f)
			      (for-each (lambda (x) (walk (cadr x) e me)) (cadr x))
			      (walk (##sys#canonicalize-body (cddr x))
				    (cons (map (cut ##sys#slot <> 0) (cadr x)) e)
				    me) ]

			     [(lambda)
			      (##sys#check-syntax 'lambda x '(lambda lambda-list . #(_ 1)) #f)
			      (##sys#decompose-lambda-list
			       (cadr x)
			       (lambda (vars argc rest)
				 (walk (##sys#canonicalize-body (cddr x)) (cons vars e) me) ) ) ]

			     [(##core#loop-lambda)
			      (walk `(lambda ,@(cdr x)) e me) ]

			     [(##core#named-lambda)
			      (walk `(lambda ,@(cddr x)) e me) ]

			     [(##core#require-for-syntax)
			      (let ([ids (map eval (cdr x))])
				(apply ##sys#require ids) ) ]

			     [(##core#elaborationtimeonly ##core#elaborationtimetoo) ; <- Note this!
			      (eval (cadr x)) ]

			     [(##core#compiletimetoo)
			      (walk-rest (cdr x) e me) ]

			     [(##core#define-inline ##core#define-constant)
			      (##sys#visit-define-hook x) ]
                   
			     [(##core#include)
			      (walk
			       (##sys#compiler-toplevel-macroexpand-hook
				(with-input-from-file (##sys#resolve-include-filename (cadadr x))
				  (lambda ()
				    (do ([x (read) (read)]
					 [xs '() (cons x xs)] )
					((eof-object? x) 
					 `(begin ,@(reverse xs))) ) ) ) )
			       e me) ]

			     [(##core#inline ##core#inline_allocate)
			      (walk-rest (cddr x) e me) ]

			     [(##core#let-location)
			      (walk-rest (cdddr x) e me) ]

			     [(##core#define-foreign-type)
			      (##sys#visit-define-hook x) ]

			     [(##core#app)
			      (walk (cdr x) e me) ]

			     [else
			      (cond [##sys#strict-mode (walk-rest x e me)]

				    [(eq? head 'let-macro)
				     (##sys#check-syntax 'let-macro x '(let-macro #(list 0) . #(_ 1)) #f)
				     (set! ##sys#syntax-error-culprit #f)
				     (let ([me2 (##sys#expand-local-macrodefs (cadr x))])
				       (walk 
					(##sys#canonicalize-body (cddr x))
					(undefine (map (lambda (x) (car x)) me2) e)
					(##sys#append me2 me) ) ) ]

				    [(eq? head 'location)]

				    [else (walk-rest x e me)] ) ] )

			   (walk x2 e me) ) ) ) ) ]

	      [else (walk-rest x e me)] ) )

      (walk (##sys#compiler-toplevel-macroexpand-hook exp) '() '()) ) ) )
