;;;; library.scm - R5RS library for the CHICKEN compiler
;
; Copyright (c) 2000-2003, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare
  (unit library)
  (disable-interrupts)
  (usual-integrations)
  (hide ##sys#dynamic-unwind 
	##sys#grow-vector ##sys#default-parameter-vector 
	##sys#print-length-limit
	##sys#fetch-and-check-port-arg ##sys#print-exit)
  (foreign-declare #<<EOF
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <errno.h>

#ifdef HAVE_SYSEXITS_H
# include <sysexits.h>
#endif

#if !defined(_MSC_VER) && !defined(__DJGPP__) && !defined(__MWERKS__)
# include <unistd.h>
#endif

#ifndef EX_SOFTWARE
# define EX_SOFTWARE    70
#endif

#define C_m_op_exp(n)         (C_temporary_flonum = exp(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_log(n)         (C_temporary_flonum = log(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_sin(n)         (C_temporary_flonum = sin(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_cos(n)         (C_temporary_flonum = cos(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_tan(n)         (C_temporary_flonum = tan(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_asin(n)        (C_temporary_flonum = asin(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_acos(n)        (C_temporary_flonum = acos(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_atan(n)        (C_temporary_flonum = atan(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_atan2(n1, n2)  (C_temporary_flonum = atan2(C_c_double(n1), C_c_double(n2)), C_SCHEME_UNDEFINED)
#define C_m_op_sqrt(n)        (C_temporary_flonum = sqrt(C_c_double(n)), C_SCHEME_UNDEFINED)

#define C_close_file(p)       (fclose((FILE *)(C_port_file(p))), C_SCHEME_UNDEFINED)
#define C_f64peek(b, i)       (C_temporary_flonum = ((double *)C_data_pointer(b))[ C_unfix(i) ], C_SCHEME_UNDEFINED)
#define C_fetch_c_strlen(b, i) C_fix(strlen((char *)C_block_item(b, C_unfix(i))))
#define C_peek_c_string(b, i, to, len) (C_memcpy(C_data_pointer(to), (char *)C_block_item(b, C_unfix(i)), C_unfix(len)), C_SCHEME_UNDEFINED)
#define C_free_mptr(p, i)     (C_free((void *)C_block_item(p, C_unfix(i))), C_SCHEME_UNDEFINED)

#define C_direct_continuation(dummy)  t1
EOF
) )

(cond-expand
 [paranoia]
 [else
  (declare
    (no-bound-checks)
    (bound-to-procedure
     ##sys#check-char ##sys#check-exact ##sys#check-port ##sys#check-string ##sys#substring ##sys#check-port-mode
     ##sys#for-each ##sys#map ##sys#setslot ##sys#allocate-vector ##sys#check-pair 
     ##sys#not-a-proper-list-error ##sys#error ##sys#warn ##sys#signal-hook
     ##sys#check-symbol ##sys#check-vector ##sys#floor ##sys#ceiling ##sys#truncate ##sys#round 
     ##sys#check-number ##sys#cons-flonum ##sys#check-integer
     ##sys#flonum-fraction ##sys#make-port ##sys#fetch-and-check-port-arg ##sys#print 
     ##sys#check-structure ##sys#make-structure
     ##sys#macroexpand ##sys#gcd ##sys#lcm ##sys#ensure-heap-reserve ##sys#check-list 
     ##sys#enable-interrupts ##sys#disable-interrupts ##sys#->feature-id
     ##sys#fudge ##sys#user-read-hook ##sys#check-range ##sys#read
     ##sys#string->symbol ##sys#symbol->string ##sys#dynamic-unwind
     ##sys#error-handler ##sys#signal ##sys#abort ##sys#port-data
     ##sys#reset-handler ##sys#exit-handler ##sys#dynamic-wind
     ##sys#grow-vector ##sys#run-pending-finalizers
     ##sys#schedule ##sys#make-thread ##sys#print-to-string
     ##sys#update-thread-state-buffer ##sys#restore-thread-state-buffer ##sys#user-print-hook 
     ##sys#current-exception-handler ##sys#default-exception-handler ##sys#abandon-mutexes ##sys#make-mutex
     ##sys#register-entry-point ##sys#port-has-file-pointer? ##sys#record-field-index
     ##sys#dispatch-to-entry-point ##sys#intern-symbol ##sys#make-string
     ##sys#append ##sys#list ##sys#cons ##sys#list->vector ##sys#list ##sys#apply ##sys#make-vector
     ##sys#write-char ##sys#force-finalizers ##sys#cleanup-before-exit
     ##sys#default-read-info-hook ##sys#read-error) ) ] )


(include "build")
(include "parameters")


;;; System routines:

(define (exit . code) (apply (##sys#exit-handler) code))
(define (reset) ((##sys#reset-handler)))

(define (##sys#error msg . args)
  (apply ##sys#signal-hook #:error msg args) )

(define ##sys#warnings-enabled #t)

(define (##sys#warn msg . args)
  (when ##sys#warnings-enabled
    (apply ##sys#signal-hook #:warning msg args) ) )

(define error ##sys#error)

(define-foreign-variable main_argc int "C_main_argc")
(define-foreign-variable main_argv c-pointer "C_main_argv")

(define (set-gc-report! flag) (##core#inline "C_set_gc_report" flag))
(define ##sys#gc (##core#primitive "C_gc"))
(define (##sys#setslot x i y) (##core#inline "C_i_setslot" x i y))
(define (##sys#setislot x i y) (##core#inline "C_i_set_i_slot" x i y))
(define ##sys#allocate-vector (##core#primitive "C_allocate_vector"))
(define argv (##core#primitive "C_get_argv"))
(define (argc+argv) (##sys#values main_argc main_argv))
(define ##sys#make-structure (##core#primitive "C_make_structure"))
(define ##sys#ensure-heap-reserve (##core#primitive "C_ensure_heap_reserve"))
(define (##sys#fudge fudge-factor) (##core#inline "C_fudge" fudge-factor))
(define ##sys#call-host (##core#primitive "C_call_host"))
(define ##sys#host-data (##core#primitive "C_host_data"))
(define ##sys#set-host-data! (##core#primitive "C_set_host_data"))
(define ##sys#file-info (##core#primitive "C_file_info"))
(define ##sys#symbol-table-info (##core#primitive "C_get_symbol_table_info"))
(define ##sys#memory-info (##core#primitive "C_get_memory_info"))
(define (current-seconds) (##sys#fudge 2))
(define (current-milliseconds) (##sys#fudge 16))
(define cpu-time (##core#primitive "C_cpu_time"))
(define ##sys#decode-seconds (##core#primitive "C_decode_seconds"))
(define getenv (##core#primitive "C_get_environment_variable"))
(define (##sys#start-timer) (##core#inline "C_start_timer"))
(define ##sys#stop-timer (##core#primitive "C_stop_timer"))
(define (##sys#immediate? x) (not (##core#inline "C_blockp" x)))
(define (##sys#message str) (##core#inline "C_message" str))
(define (##sys#byte x i) (##core#inline "C_subbyte" x i))
(define (##sys#setbyte x i n) (##core#inline "C_setbyte" x i n))
(define (##sys#void) (##core#undefined))
(define void ##sys#void)
(define (end-of-file) (##sys#fudge 1))
(define (##sys#halt) (##core#inline "C_halt" #f))
(define ##sys#dload (##core#primitive "C_dload"))
(define (##sys#flo2fix n) (##core#inline "C_quickflonumtruncate" n))
(define ##sys#become! (##core#primitive "C_become"))

(define (##sys#check-structure x y . z) 
  (if (pair? z)
      (##core#inline "C_i_check_structure_2" x y (car z))
      (##core#inline "C_i_check_structure" x y) ) )

(define (##sys#check-byte-vector x . y) 
  (if (pair? y)
      (##core#inline "C_i_check_bytevector_2" x (car y))
      (##core#inline "C_i_check_bytevector" x) ) )

(define (##sys#check-pair x . y) 
  (if (pair? y)
      (##core#inline "C_i_check_pair_2" x (car y))
      (##core#inline "C_i_check_pair" x) ) )

(define (##sys#check-list x . y) 
  (if (pair? y)
      (##core#inline "C_i_check_list_2" x (car y))
      (##core#inline "C_i_check_list" x) ) )

(define (##sys#check-string x . y) 
  (if (pair? y)
      (##core#inline "C_i_check_string_2" x (car y))
      (##core#inline "C_i_check_string" x) ) )

(define (##sys#check-number x . y) 
  (if (pair? y)
      (##core#inline "C_i_check_number_2" x (car y))
      (##core#inline "C_i_check_number" x) ) )

(define (##sys#check-exact x . y) 
  (if (pair? y)
      (##core#inline "C_i_check_exact_2" x (car y))
      (##core#inline "C_i_check_exact" x) ) )

(define (##sys#check-symbol x . y) 
  (if (pair? y)
      (##core#inline "C_i_check_symbol_2" x (car y))
      (##core#inline "C_i_check_symbol" x) ) )

(define (##sys#check-vector x . y) 
  (if (pair? y)
      (##core#inline "C_i_check_vector_2" x (car y))
      (##core#inline "C_i_check_vector" x) ) )

(define (##sys#check-char x . y) 
  (if (pair? y)
      (##core#inline "C_i_check_char_2" x (car y))
      (##core#inline "C_i_check_char" x) ) )

(define (##sys#check-integer x . y)
  (unless (##core#inline "C_i_integerp" x) 
    (##sys#signal-hook #:type-error (if (pair? y) (car y) #f) "bad argument type - not an integer" x) ) )

(define ##sys#check-range 
  (lambda (i from to loc)
    (##sys#check-exact i loc)
    (if (or (not (fx>= i from))
	    (not (fx< i to)) ) 
	(##sys#error loc "index out of range" i from to) ) ) )

(cond-expand
 [unsafe
  (eval-when (compile)
    (define-macro (##sys#check-structure . _) '(##core#undefined))
    (define-macro (##sys#check-range . _) '(##core#undefined))
    (define-macro (##sys#check-pair . _) '(##core#undefined))
    (define-macro (##sys#check-list . _) '(##core#undefined))
    (define-macro (##sys#check-symbol . _) '(##core#undefined))
    (define-macro (##sys#check-string . _) '(##core#undefined))
    (define-macro (##sys#check-char . _) '(##core#undefined))
    (define-macro (##sys#check-exact . _) '(##core#undefined))
    (define-macro (##sys#check-port . _) '(##core#undefined))
    (define-macro (##sys#check-port-mode . _) '(##core#undefined))
    (define-macro (##sys#check-number . _) '(##core#undefined))
    (define-macro (##sys#check-byte-vector . _) '(##core#undefined)) ) ]
 [else] )

(define (force promise)
  (if (##sys#structure? promise 'promise)
      ((##sys#slot promise 1))
      promise) )

(define (system cmd)
  (##sys#check-string cmd 'system)
  (##core#inline "C_execute_shell_command" cmd) )


;;; Operations on booleans:

(define (not x) (##core#inline "C_i_not" x))
(define (boolean? x) (##core#inline "C_booleanp" x))


;;; Equivalence predicates:

(define (eq? x y) (##core#inline "C_eqp" x y))
(define (eqv? x y) (##core#inline "C_i_eqvp" x y))
(define (equal? x y) (##core#inline "C_i_equalp" x y))


;;; Pairs and lists:

(define (pair? x) (##core#inline "C_i_pairp" x))
(define (cons x y) (##core#inline_allocate ("C_a_i_cons" 3) x y))
(define (car x) (##core#inline "C_i_car" x))
(define (cdr x) (##core#inline "C_i_cdr" x))

(define (set-car! x y) (##core#inline "C_i_set_car" x y))
(define (set-cdr! x y) (##core#inline "C_i_set_cdr" x y))
(define (cadr x) (##core#inline "C_i_cadr" x))
(define (caddr x) (##core#inline "C_i_caddr" x))
(define (cadddr x) (##core#inline "C_i_cadddr" x))
(define (cddddr x) (##core#inline "C_i_cddddr" x))

(define (caar x) (car (car x)))
(define (cdar x) (cdr (car x)))
(define (cddr x) (cdr (cdr x)))
(define (caaar x) (car (car (car x))))
(define (caadr x) (car (##core#inline "C_i_cadr" x)))
(define (cadar x) (##core#inline "C_i_cadr" (car x)))
(define (cdaar x) (cdr (car (car x))))
(define (cdadr x) (cdr (##core#inline "C_i_cadr" x)))
(define (cddar x) (cdr (cdr (car x))))
(define (cdddr x) (cdr (cdr (cdr x))))
(define (caaaar x) (car (car (car (car x)))))
(define (caaadr x) (car (car (##core#inline "C_i_cadr" x))))
(define (caadar x) (car (##core#inline "C_i_cadr" (car x))))
(define (caaddr x) (car (##core#inline "C_i_caddr" x)))
(define (cadaar x) (##core#inline "C_i_cadr" (car (car x))))
(define (cadadr x) (##core#inline "C_i_cadr" (##core#inline "C_i_cadr" x)))
(define (caddar x) (##core#inline "C_i_caddr" (car x)))
(define (cdaaar x) (cdr (car (car (car x)))))
(define (cdaadr x) (cdr (car (##core#inline "C_i_cadr" x))))
(define (cdadar x) (cdr (##core#inline "C_i_cadr" (car x))))
(define (cdaddr x) (cdr (##core#inline "C_i_caddr" x)))
(define (cddaar x) (cdr (cdr (car (car x)))))
(define (cddadr x) (cdr (cdr (##core#inline "C_i_cadr" x))))
(define (cdddar x) (cdr (cdr (cdr (car x)))))

(define (null? x) (eq? x '()))
(define (list . lst) lst)
(define (length lst) (##core#inline "C_i_length" lst))
(define (list-tail lst i) (##core#inline "C_i_list_tail" lst i))
(define (list-ref lst i) (##core#inline "C_i_list_ref" lst i))

(define ##sys#not-a-proper-list-error
  (lambda (arg . loc)
    (##sys#signal-hook #:type-error (if (pair? loc) (car loc) #f) "argument is not a proper list" arg) ) )

(define append
  (lambda lsts
    (if (eq? lsts '())
	lsts
	(let loop ((lsts lsts))
	  (if (eq? (##sys#slot lsts 1) '())
	      (##sys#slot lsts 0)
	      (let copy ((node (##sys#slot lsts 0)))
		(cond-expand
		 [unsafe
		  (if (eq? node '()) 
		      (loop (##sys#slot lsts 1))
		      (cons (##sys#slot node 0) (copy (##sys#slot node 1))) ) ]
		 [else
		  (cond ((eq? node '()) (loop (##sys#slot lsts 1)))
			((pair? node)
			 (cons (##sys#slot node 0) (copy (##sys#slot node 1))) )
			(else (##sys#not-a-proper-list-error (##sys#slot lsts 0) 'append)) ) ] ) ) ) ) ) ) )

(define reverse 
  (lambda (lst0)
    (let loop ((lst lst0) (rest '()))
      (cond-expand
       [unsafe
	(if (eq? lst '()) 
	    rest
	    (loop (##sys#slot lst 1) (cons (##sys#slot lst 0) rest))  ) ]
       [else
	(cond ((eq? lst '()) rest)
	      ((pair? lst)
	       (loop (##sys#slot lst 1) (cons (##sys#slot lst 0) rest)) )
	      (else (##sys#not-a-proper-list-error lst0 'reverse)) ) ] ) ) ) )

(define (memq x lst) (##core#inline "C_i_memq" x lst))
(define (memv x lst) (##core#inline "C_i_memv" x lst))
(define (member x lst) (##core#inline "C_i_member" x lst))
(define (assq x lst) (##core#inline "C_i_assq" x lst))
(define (assv x lst) (##core#inline "C_i_assv" x lst))
(define (assoc x lst) (##core#inline "C_i_assoc" x lst))

(define (list? x) (##core#inline "C_i_listp" x))


;;; Strings:

(define (string? x) (##core#inline "C_i_stringp" x))
(define (string-length s) (##core#inline "C_i_string_length" s))
(define (string-ref s i) (##core#inline "C_i_string_ref" s i))
(define (string-set! s i c) (##core#inline "C_i_string_set" s i c))

(define (##sys#make-string size . fill)
  (##sys#check-exact size 'make-string)
  (cond-expand 
   [unsafe] 
   [else (when (fx< size 0) (##sys#signal-hook #:bounds-error 'make-string "size is negative" size))])
  (##sys#allocate-vector
   size #t
   (if (null? fill)
       #\space
       (let ((c (car fill)))
	 (begin (##sys#check-char c 'make-string) c) ) )
   #f) )

(define make-string ##sys#make-string)

(define string->list 
  (lambda (s)
    (##sys#check-string s 'string->list)
    (let ((len (##core#inline "C_block_size" s)))
      (let loop ((i 0))
	(if (fx>= i len)
	    '()
	    (cons (##core#inline "C_subchar" s i)
		  (loop (fx+ i 1)) ) ) ) ) ) )

(define (list->string lst0)
  (if (not (list? lst0))
      (##sys#not-a-proper-list-error lst0 'list->string)
      (let* ([len (length lst0)]
	     [s (##sys#make-string len)] )
	(do ([i 0 (fx+ i 1)]
	     [lst lst0 (##sys#slot lst 1)] )
	    ((fx>= i len) s)
	  (let ([c (##sys#slot lst 0)])
	    (##sys#check-char c 'list->string)
	    (##core#inline "C_setsubchar" s i c) ) ) ) ) )

(define (string-fill! s c)
  (##sys#check-string s 'string-fill!)
  (##sys#check-char c 'string-fill!)
  (##core#inline "C_set_memory" s c (##sys#size s))
  (##core#undefined) )

(define string-copy
    (lambda (s)
      (##sys#check-string s 'string-copy)
      (let* ([len (##sys#size s)]
	     [s2 (##sys#make-string len)] )
	(##core#inline "C_copy_memory" s2 s len)
	s2) ) )

(define substring
    (lambda (s start end)
      (##sys#check-string s 'substring)
      (##sys#check-exact start 'substring)
      (##sys#check-exact end 'substring)
      (cond-expand
       [unsafe (##sys#substring s start end)]
       [else
	(let ([len (##sys#size s)])
	  (if (or (fx< start 0)
		  (fx> start len)
		  (fx< end 0)
		  (fx> end len)
		  (fx> start end) )
	      (##sys#signal-hook #:bounds-error 'substring "index out of bounds" start end)
	      (##sys#substring s start end) ) ) ] ) ) )

(define ##sys#substring
    (lambda (s start end)
      (let* ([len (##sys#size s)]
	     [len2 (fx- end start)]
	     [s2 (##sys#make-string len2)] )
	(##core#inline "C_substring_copy" s s2 start end 0)
	s2) ) )

(define (string=? x y)
  (cond-expand [unsafe (##core#inline "C_u_i_string_equal_p" x y)]
	       [else (##core#inline "C_i_string_equal_p" x y)] ) )

(define (string-ci=? x y) (##core#inline "C_i_string_ci_equal_p" x y))

(letrec ((compare 
	  (lambda (s1 s2 loc k)
	    (##sys#check-string s1 loc)
	    (##sys#check-string s2 loc)
	    (let ((len1 (##core#inline "C_block_size" s1))
		  (len2 (##core#inline "C_block_size" s2)) )
	      (k len1 len2
		 (##core#inline "C_string_compare"
			    s1
			    s2
			    (if (fx< len1 len2)
				len1
				len2) ) ) ) ) ) )
  (set! string<? (lambda (s1 s2)
		   (compare 
		    s1 s2 'string<?
		    (lambda (len1 len2 cmp)
		      (or (fx< cmp 0)
			  (and (fx< len1 len2)
			       (eq? cmp 0) ) ) ) ) ) )
  (set! string>? (lambda (s1 s2)
		   (compare 
		    s1 s2 'string>?
		    (lambda (len1 len2 cmp)
		      (or (fx> cmp 0)
			  (and (fx< len2 len1)
			       (eq? cmp 0) ) ) ) ) ) )
  (set! string<=? (lambda (s1 s2)
		    (compare 
		     s1 s2 'string<=?
		     (lambda (len1 len2 cmp)
		       (if (eq? cmp 0)
			   (fx>= len1 len2)
			   (fx< cmp 0) ) ) ) ) )
  (set! string>=? (lambda (s1 s2)
		    (compare 
		     s1 s2 'string>=?
		     (lambda (len1 len2 cmp)
		       (if (eq? cmp 0)
			   (fx<= len1 len2)
			   (fx> cmp 0) ) ) ) ) ) )

(letrec ((compare 
	  (lambda (s1 s2 loc k)
	    (##sys#check-string s1 loc)
	    (##sys#check-string s2 loc)
	    (let ((len1 (##core#inline "C_block_size" s1))
		  (len2 (##core#inline "C_block_size" s2)) )
	      (k len1 len2
		 (##core#inline "C_string_compare_case_insensitive"
				s1
				s2
				(if (fx< len1 len2)
				    len1
				    len2) ) ) ) ) ) )
  (set! string-ci<? (lambda (s1 s2)
		      (compare 
		       s1 s2 'string-ci<?
		       (lambda (len1 len2 cmp)
			 (or (fx< cmp 0)
			     (and (fx< len1 len2)
				  (eq? cmp 0) ) ) ) ) ) )
  (set! string-ci>? (lambda (s1 s2)
		      (compare 
		       s1 s2 'string-ci>?
		       (lambda (len1 len2 cmp)
			 (or (fx> cmp 0)
			     (and (fx< len2 len1)
				  (eq? cmp 0) ) ) ) ) ) )
  (set! string-ci<=? (lambda (s1 s2)
		       (compare 
			s1 s2 'string-ci<=?
			(lambda (len1 len2 cmp)
			  (if (eq? cmp 0)
			      (fx>= len1 len2)
			      (fx< cmp 0) ) ) ) ) )
  (set! string-ci>=? (lambda (s1 s2)
		       (compare 
			s1 s2 'string-ci>=?
			(lambda (len1 len2 cmp)
			  (if (eq? cmp 0)
			      (fx<= len1 len2)
			      (fx> cmp 0) ) ) ) ) ) )

(define string-append
    (lambda all
      (let ([snew #f])
	(let loop ([strs all] [n 0])
	  (if (eq? strs '())
	      (set! snew (##sys#make-string n))
	      (let ([s (##sys#slot strs 0)])
		(##sys#check-string s 'string-append)
		(let ([len (##sys#size s)])
		  (loop (##sys#slot strs 1) (fx+ n len))
		  (##core#inline "C_substring_copy" s snew 0 len n) ) ) ) )
	snew) ) )

(define string
  (let ([list->string list->string])
    (lambda chars (list->string chars)) ) )

(define (##sys#fragments->string total fs)
  (let ([dest (##sys#make-string total)])
    (let loop ([fs fs] [pos 0])
      (if (null? fs)
	  dest
	  (let* ([f (##sys#slot fs 0)]
		 [flen (##sys#size f)] )
	    (##core#inline "C_substring_copy" f dest 0 flen pos)
	    (loop (##sys#slot fs 1) (fx+ pos flen)) ) ) ) ) )


;;; Numeric routines:

(define (fixnum? x) (##core#inline "C_fixnump" x))
(define (fx+ x y) (##core#inline "C_fixnum_plus" x y))
(define (fx- x y) (##core#inline "C_fixnum_difference" x y))
(define (fx* x y) (##core#inline "C_fixnum_times" x y))
(define (fx= x y) (eq? x y))
(define (fx> x y) (##core#inline "C_fixnum_greaterp" x y))
(define (fx< x y) (##core#inline "C_fixnum_lessp" x y))
(define (fx>= x y) (##core#inline "C_fixnum_greater_or_equal_p" x y))
(define (fx<= x y) (##core#inline "C_fixnum_less_or_equal_p" x y))
(define (fxmin x y) (##core#inline "C_i_fixnum_min" x y))
(define (fxmax x y) (##core#inline "C_i_fixnum_max" x y))
(define (fxneg x) (##core#inline "C_fixnum_negate" x))

(define fx/
  (lambda (x y)
    (cond-expand
     [unsafe (##core#inline "C_fixnum_divide" x y)]
     [else
      (if (eq? y 0)
	  (##sys#signal-hook #:arithmetic-error 'fx/ "division by zero" x y)
	  (##core#inline "C_fixnum_divide" x y) ) ] ) ) )

(define fxmod
  (lambda (x y)
    (cond-expand
     [unsafe (##core#inline "C_fixnum_modulo" x y)]
     [else
      (if (eq? y 0)
	  (##sys#signal-hook #:arithmetic-error 'fxmod "division by zero" x y)
	  (##core#inline "C_fixnum_modulo" x y) ) ] ) ) )

(define * (##core#primitive "C_times"))
(define - (##core#primitive "C_minus"))
(define + (##core#primitive "C_plus"))
(define / (##core#primitive "C_divide"))
(define = (##core#primitive "C_nequalp"))
(define > (##core#primitive "C_greaterp"))
(define < (##core#primitive "C_lessp"))
(define >= (##core#primitive "C_greater_or_equal_p"))
(define <= (##core#primitive "C_less_or_equal_p"))

(define add1 (lambda (n) (+ n 1)))
(define sub1 (lambda (n) (- n 1)))

(define ##sys#floor (##core#primitive "C_flonum_floor"))
(define ##sys#ceiling (##core#primitive "C_flonum_ceiling"))
(define ##sys#truncate (##core#primitive "C_flonum_truncate"))
(define ##sys#round (##core#primitive "C_flonum_round"))
(define quotient (##core#primitive "C_quotient"))
(define ##sys#cons-flonum (##core#primitive "C_cons_flonum"))
(define (number? x) (##core#inline "C_i_numberp" x))
(define complex? number?)
(define real? number?)
(define rational? number?)
(define ##sys#flonum-fraction (##core#primitive "C_flonum_fraction"))
(define (integer? x) (##core#inline "C_i_integerp" x))
(define (exact? x) (##core#inline "C_i_exactp" x))
(define (inexact? x) (##core#inline "C_i_inexactp" x))
(define expt (##core#primitive "C_expt"))
(define (##sys#fits-in-int? n) (##core#inline "C_fits_in_int_p" n))
(define (##sys#fits-in-unsigned-int? n) (##core#inline "C_fits_in_unsigned_int_p" n))
(define (##sys#flonum-in-fixnum-range? n) (##core#inline "C_flonum_in_fixnum_range_p" n))
(define (zero? n) (##core#inline "C_i_zerop" n))
(define (positive? n) (##core#inline "C_i_positivep" n))
(define (negative? n) (##core#inline "C_i_negativep" n))
(define (abs n) (##core#inline_allocate ("C_a_i_abs" 4) n))	; 4 => words-per-flonum

(define signum
  (lambda (n)
    (cond ((> n 0) 1)
	  ((< n 0) -1)
	  (else 0) ) ) )

(define exact->inexact (##core#primitive "C_exact_to_inexact"))
(define (inexact->exact n) (##core#inline "C_i_inexact_to_exact" n))

(define (floor x)
  (##sys#check-number x 'floor)
  (if (##core#inline "C_fixnump" x) 
      x
      (##sys#floor x) ) )

(define (ceiling x)
  (##sys#check-number x 'ceiling)
  (if (##core#inline "C_fixnump" x) 
      x
      (##sys#ceiling x) ) )

(define (truncate x)
  (##sys#check-number x 'truncate)
  (if (##core#inline "C_fixnump" x) 
      x
      (##sys#truncate x) ) )

(define (round x)
  (##sys#check-number x 'round)
  (if (##core#inline "C_fixnump" x) 
      x
      (##sys#round x) ) )

(define remainder 
  (lambda (x y) (- x (* (quotient x y) y))) )

(define modulo
  (let ([floor floor])
    (lambda (x y)
      (let ((div (/ x y)))
	(- x (* (if (integer? div)
		    div
		    (let* ([fd (floor div)]
			   [fdx (##core#inline "C_quickflonumtruncate" fd)] )
		      (if (= fd fdx)
			  fdx
			  fd) ) )
		y) ) ) ) ) )

(define (even? n) (##core#inline "C_i_evenp" n))
(define (odd? n) (##core#inline "C_i_oddp" n))

(let ([> >]
      [< <] )
  (letrec ([maxmin
	    (lambda (n1 ns pred)
	      (let loop ((nbest n1) (ns ns))
		(if (eq? ns '())
		    nbest
		    (let ([ni (##sys#slot ns 0)])
		      (loop (if (pred ni nbest)
				(if (and (##core#inline "C_blockp" nbest) 
					 (##core#inline "C_flonump" nbest) 
					 (not (##core#inline "C_blockp" ni)) )
				    (exact->inexact ni)
				    ni)
				nbest)
			    (##sys#slot ns 1) ) ) ) ) ) ] )
    (set! max (lambda (n1 . ns) (maxmin n1 ns >)))
    (set! min (lambda (n1 . ns) (maxmin n1 ns <))) ) )

(define (exp n)
  (##sys#check-number n 'exp)
  (##core#inline "C_m_op_exp" n)
  (##sys#cons-flonum) )

(define (log n)
  (##sys#check-number n 'log)
  (##core#inline "C_m_op_log" n)
  (##sys#cons-flonum) )

(define (sin n)
  (##sys#check-number n 'sin)
  (##core#inline "C_m_op_sin" n)
  (##sys#cons-flonum) )

(define (cos n)
  (##sys#check-number n 'cos)
  (##core#inline "C_m_op_cos" n)
  (##sys#cons-flonum) )

(define (tan n)
  (##sys#check-number n 'tan)
  (##core#inline "C_m_op_tan" n)
  (##sys#cons-flonum) )

(define (asin n)
  (##sys#check-number n 'asin)
  (##core#inline "C_m_op_asin" n)
  (##sys#cons-flonum) )

(define (acos n)
  (##sys#check-number n 'acos)
  (##core#inline "C_m_op_acos" n)
  (##sys#cons-flonum) )

(define (atan n1 . n2)
  (##sys#check-number n1 'atan)
  (cond ((null? n2) (##core#inline "C_m_op_atan" n1))
	(else
	 (let ((n2 (car n2)))
	   (##sys#check-number n2 'atan)
	   (##core#inline "C_m_op_atan2" n1 n2) ) ) )
  (##sys#cons-flonum) )

(define (sqrt n)
  (##sys#check-number n 'sqrt)
  (##core#inline "C_m_op_sqrt" n)
  (##sys#cons-flonum) )

(define ##sys#gcd
  (let ((remainder remainder))
    (lambda (x y)
      (let loop ((x x) (y y))
	(if (zero? y)
	    (abs x)
	    (loop y (remainder x y)) ) ) ) ) )

(define (gcd . ns)
  (if (eq? ns '())
      0
      (let loop ([ns ns] [f #t])
	(let ([head (##sys#slot ns 0)]
	      [next (##sys#slot ns 1)] )
	  (cond-expand [unsafe] [else (when f (##sys#check-integer head 'gcd))])
	  (if (null? next)
	      (abs head)
	      (let ([n2 (##sys#slot next 0)])
		(cond-expand [unsafe] [else (##sys#check-integer n2 'gcd)])
		(loop (cons (##sys#gcd head n2) (##sys#slot next 1)) #f) ) ) ) ) ) )

(define (##sys#lcm x y)
  (quotient (* x y) (##sys#gcd x y)) )

(define (lcm . ns)
  (if (null? ns)
      1
      (let loop ([ns ns] [f #t])
	(let ([head (##sys#slot ns 0)]
	      [next (##sys#slot ns 1)] )
	  (cond-expand [unsafe] [else (when f (##sys#check-integer head 'lcm))])
	  (if (null? next)
	      (abs head)
	      (let ([n2 (##sys#slot next 0)])
		(cond-expand [unsafe] [else (##sys#check-integer n2 'lcm)])
		(loop (cons (##sys#lcm head (##sys#slot next 0)) (##sys#slot next 1)) #f) ) ) ) ) ) )

(define string->number (##core#primitive "C_string_to_number"))
(define number->string (##core#primitive "C_number_to_string"))


;;; Symbols:

(define ##sys#make-symbol (##core#primitive "C_make_symbol"))
(define (symbol? x) (##core#inline "C_i_symbolp" x))
(define ##sys#snafu '##sys#fnord)
(define ##sys#intern-symbol (##core#primitive "C_string_to_symbol"))

(define (##sys#string->symbol str)
  (##sys#check-string str)
  (##sys#intern-symbol str) )

(let ([string-append string-append]
      [string-copy string-copy] )
  (define (split str len)
    (let ([b0 (##sys#byte str 0)])	; we fetch the byte, wether len is 0 or not
      (if (and (fx> len 0) (fx< b0 len) (fx<= b0 namespace-max-id-len))
	  (fx+ b0 1)
	  #f) ) )
  (set! ##sys#symbol->string
    (lambda (s)
      (let* ([str (##sys#slot s 1)]
	     [len (##sys#size str)]
	     [i (split str len)] )
	(if i (##sys#substring str i len) str) ) ) )
  (set! ##sys#symbol->qualified-string 
    (lambda (s)
      (let* ([str (##sys#slot s 1)]
	     [len (##sys#size str)] 
	     [i (split str len)] )
	(if i
	    (string-append "##" (##sys#substring str 1 i) "#" (##sys#substring str i len))
	    str) ) ) )
  (set! ##sys#qualified-symbol-prefix 
    (lambda (s)
      (let* ([str (##sys#slot s 1)]
	     [len (##sys#size str)]
	     [i (split str len)] )
	(and i (##sys#substring str 0 i)) ) ) ) )

(define ##sys#string->qualified-symbol
  (let ([string-append string-append])
    (lambda (prefix str)
      (##sys#string->symbol
       (if prefix
	   (string-append prefix str)
	   str) ) ) ) )

(define (symbol->string s)
  (##sys#check-symbol s 'symbol->string)
  (##sys#symbol->string s) )

(define string->symbol
  (let ([string-copy string-copy])
    (lambda (str)
      (##sys#check-string str 'string->symbol)
      (##sys#intern-symbol (string-copy str)) ) ) )

(define string->uninterned-symbol
  (let ([string-copy string-copy])
    (lambda (str)
      (##sys#check-string str 'string->uninterned-symbol)
      (##sys#make-symbol (string-copy str)) ) ) )

(define gensym
  (let ([counter -1]
	[string-append string-append] )
    (lambda str-or-sym
      (let ([err (lambda (prefix) (##sys#signal-hook #:type-error 'gensym "argument is not a string or symbol" prefix))])
	(set! counter (fx+ counter 1))
	(##sys#make-symbol
	 (string-append
	  (if (eq? str-or-sym '())
	      "g"
	      (let ([prefix (car str-or-sym)])
		(or (and (##core#inline "C_blockp" prefix)
			 (cond [(##core#inline "C_stringp" prefix) prefix]
			       [(##core#inline "C_symbolp" prefix) (##sys#symbol->string prefix)]
			       [else (err prefix)] ) )
		    (err prefix) ) ) )
	  (number->string counter) ) ) ) ) ) )


;;; Keywords:

(define (keyword? x)
  (and (symbol? x) (fx= 0 (##sys#byte (##sys#slot x 1) 0))) )

(define string->keyword
  (let ([string-append string-append]
	[string string] )
    (lambda (s)
      (##sys#check-string s 'string-keyword)
      (##sys#intern-symbol (string-append (string (integer->char 0)) s)) ) ) )

(define keyword->string
  (let ([keyword? keyword?])
    (lambda (kw)
      (if (keyword? kw)
	  (##sys#symbol->string kw)
	  (##sys#signal-hook #:type-error 'keyword->string "bad argument type - not a keyword" kw) ) ) ) )

(define get-keyword 
  (lambda (key args0 . default)
    (##sys#check-list args0 'get-keyword)
    (let loop ([args args0])
      (if (null? args)
	  (and (pair? default) ((car default)))
	  (cond-expand
	   [unsafe
	    (let ([x (##sys#slot args 0)]
		  [r (##sys#slot args 1)] )
	      (if (eq? x key)
		  (##sys#slot r 0)
		  (loop (##sys#slot r 1)) ) ) ]
	   [else
	    (if (and (pair? args) (pair? (##sys#slot args 1)))
		(let ([x (##sys#slot args 0)]
		      [r (##sys#slot args 1)] )
		  (if (eq? key x)
		      (##sys#slot r 0)
		      (loop (##sys#slot r 1)) ) )
		(##sys#signal-hook #:type-error 'get-keyword "invalid keyword argument list" args0) ) ] ) ) ) ) )


;;; Vectors:

(define (vector? x) (##core#inline "C_i_vectorp" x))
(define (vector-length v) (##core#inline "C_i_vector_length" v))
(define (vector-ref v i) (##core#inline "C_i_vector_ref" v i))
(define (vector-set! v i x) (##core#inline "C_i_vector_set" v i x))

(define (##sys#make-vector size . fill)
  (##sys#check-exact size 'make-vector)
  (cond-expand [unsafe] [else (when (fx< size 0) (##sys#error 'make-vector "size is negative" size))])
  (##sys#allocate-vector
   size #f
   (if (null? fill)
       (##core#undefined)
       (car fill) )
   #f) )

(define make-vector ##sys#make-vector)

(define list->vector
  (lambda (lst0)
    (let* ([len (length lst0)]
	   [v (##sys#make-vector len)] )
      (let loop ([lst lst0] [i 0])
	(if (null? lst)
	    v
	    (begin
	      (##sys#setslot v i (##sys#slot lst 0))
	      (loop (##sys#slot lst 1) (fx+ i 1)) ) ) ) ) ) )

(define vector->list
  (lambda (v)
    (##sys#check-vector v 'vector->list)
    (let ((len (##core#inline "C_block_size" v)))
      (let loop ((i 0))
	(if (fx>= i len)
	    '()
	    (cons (##sys#slot v i)
		  (loop (fx+ i 1)) ) ) ) ) ) )

(define vector
  (lambda xs (##sys#list->vector xs)) )

(define (vector-fill! v x)
  (##sys#check-vector v 'vector-fill!)
  (let ((len (##core#inline "C_block_size" v)))
    (do ((i 0 (fx+ i 1)))
	((fx>= i len))
      (##sys#setslot v i x) ) ) )

(define vector-copy!
  (lambda (from to . n)
    (##sys#check-vector from 'vector-copy!)
    (##sys#check-vector to 'vector-copy!)
    (let* ([len-from (##sys#size from)]
	   [len-to (##sys#size to)] 
	   [n (if (pair? n) (car n) (fxmin len-to len-from))] )
      (##sys#check-exact n 'vector-copy!)
      (cond-expand
       [(not unsafe)
	(when (or (fx> n len-to) (fx> n len-from))
	  (##sys#signal-hook 
	   #:bounds-error 'vector-copy!
	   "can not copy vector - count exceeds length" from to n) ) ]
       [else] )
      (do ([i 0 (fx+ i 1)])
	  ((fx>= i n))
	(##sys#setslot to i (##sys#slot from i)) ) ) ) )

(define (vector-resize v n . init)
  (##sys#check-vector v 'vector-resize)
  (##sys#check-exact n 'vector-resize)
  (let ([v2 (##sys#apply ##sys#make-vector n init)]
	[len (##sys#size v)] )
    (do ([i 0 (fx+ i 1)])
	((fx>= i len) v2)
      (##sys#setslot v2 i (##sys#slot v i)) ) ) )

(define ##sys#grow-vector 
  (lambda (v n init)
    (let ([v2 (##sys#make-vector n init)]
	  [len (##sys#size v)] )
      (do ([i 0 (fx+ i 1)])
	  ((fx>= i len) v2)
	(##sys#setslot v2 i (##sys#slot v i)) ) ) ) )
	

;;; Characters:

(define (char? x) (##core#inline "C_charp" x))

(define (char->integer c)
  (##sys#check-char c 'char->integer)
  (##core#inline "C_fix" (##core#inline "C_character_code" c)) )

(define (integer->char n)
  (##sys#check-exact n 'integer->char)
  (##core#inline "C_make_character" (##core#inline "C_unfix" n)) )

(define (char=? c1 c2)
  (##sys#check-char c1 'char=?)
  (##sys#check-char c2 'char=?)
  (eq? c1 c2) )

(define (char>? c1 c2)
  (##sys#check-char c1 'char>?)
  (##sys#check-char c2 'char>?)
  (fx> c1 c2) )

(define (char<? c1 c2)
  (##sys#check-char c1 'char<?)
  (##sys#check-char c2 'char<?)
  (fx< c1 c2) )

(define (char>=? c1 c2)
  (##sys#check-char c1 'char>=?)
  (##sys#check-char c2 'char>=?)
  (fx>= c1 c2) )

(define (char<=? c1 c2)
  (##sys#check-char c1 'char<=?)
  (##sys#check-char c2 'char<=?)
  (fx<= c1 c2) )

(define (char-upcase c)
  (##sys#check-char c 'char-upcase)
  (##core#inline "C_make_character"
	     (##core#inline toupper (##core#inline "C_character_code" c)) ) )

(define (char-downcase c)
  (##sys#check-char c 'char-downcase)
  (##core#inline "C_make_character"
	     (##core#inline tolower (##core#inline "C_character_code" c)) ) )

(let ((char-downcase char-downcase))
  (set! char-ci=? (lambda (x y) (eq? (char-downcase x) (char-downcase y))))
  (set! char-ci>? (lambda (x y) (fx> (char-downcase x) (char-downcase y))))
  (set! char-ci<? (lambda (x y) (fx< (char-downcase x) (char-downcase y))))
  (set! char-ci>=? (lambda (x y) (fx>= (char-downcase x) (char-downcase y))))
  (set! char-ci<=? (lambda (x y) (fx<= (char-downcase x) (char-downcase y)))) )

(define (char-upper-case? c)
  (##sys#check-char c 'char-upper-case?)
  (##core#inline "C_u_i_char_upper_casep" c) )

(define (char-lower-case? c)
  (##sys#check-char c 'char-lower-case?)
  (##core#inline "C_u_i_char_lower_casep" c) )

(define (char-numeric? c)
  (##sys#check-char c 'char-numeric?)
  (##core#inline "C_u_i_char_numericp" c) )

(define (char-whitespace? c)
  (##sys#check-char c 'char-whitespace?)
  (##core#inline "C_u_i_char_whitespacep" c) )

(define (char-alphabetic? c)
  (##sys#check-char c 'char-alphabetic?)
  (##core#inline "C_u_i_char_alphabeticp" c) )

(define char-name
  (let ([chars-to-names (make-vector char-name-table-size '())]
	[names-to-chars '()] )

    (define (lookup-char c)
      (let* ([code (char->integer c)]
	     [key (##core#inline "C_fixnum_modulo" code char-name-table-size)] )
	(let loop ([b (##sys#slot chars-to-names key)])
	  (and (pair? b)
	       (let ([a (##sys#slot b 0)])
		 (if (eq? (##sys#slot a 0) c)
		     a
		     (loop (##sys#slot b 1)) ) ) ) ) ) )

    (lambda (x . y)
      (let ([chr (if (pair? y) (car y) #f)])
	(cond [(char? x)
	       (and-let* ([a (lookup-char x)])
		 (##sys#slot a 1) ) ]
	      [chr
	       (##sys#check-symbol x 'char-name)
	       (##sys#check-char chr 'char-name)
	       (let ([a (lookup-char chr)])
		 (if a 
		     (let ([b (assq x names-to-chars)])
		       (##sys#setslot a 1 x)
		       (if b
			   (##sys#setislot b 1 chr)
			   (set! names-to-chars (cons (cons x chr) names-to-chars)) ) )
		     (let ([key (##core#inline "C_fixnum_modulo" (char->integer chr) char-name-table-size)])
		       (set! names-to-chars (cons (cons x chr) names-to-chars))
		       (##sys#setslot chars-to-names key (cons (cons chr x) (##sys#slot chars-to-names key))) ) ) ) ]
	      [else
	       (##sys#check-symbol x 'char-name)
	       (and-let* ([a (assq x names-to-chars)])
		 (##sys#slot a 1) ) ] ) ) ) ) )

(char-name 'space #\space)
(char-name 'tab #\tab)
(char-name 'linefeed #\linefeed)
(char-name 'newline #\newline)
(char-name 'return #\return)
(char-name 'page (integer->char 12))
(char-name 'backspace (integer->char 8))


;;; Procedures:

(define (procedure? x) (##core#inline "C_i_closurep" x))
(define apply (##core#primitive "C_apply"))
(define ##sys#call-with-current-continuation (##core#primitive "C_call_cc"))
(define (##sys#call-with-direct-continuation k) (##core#app k (##core#inline "C_direct_continuation" #f)))
(define (##sys#direct-return dk x) (##core#inline "C_direct_return" dk x))
(define values (##core#primitive "C_values"))
(define ##sys#call-with-values (##core#primitive "C_call_with_values"))
(define call-with-values ##sys#call-with-values)

(define (##sys#for-each p lst0)
  (let loop ((lst lst0))
    (cond-expand
     [unsafe
      (if (eq? lst '()) 
	  (##core#undefined)
	  (begin
	    (p (##sys#slot lst 0))
	    (loop (##sys#slot lst 1)) ) ) ]
     [else
      (cond ((eq? lst '()) (##core#undefined))
	    ((pair? lst)
	     (p (##sys#slot lst 0))
	     (loop (##sys#slot lst 1)) )
	    (else (##sys#not-a-proper-list-error lst0 'for-each)) ) ] ) ) )

(define (##sys#map p lst0)
  (let loop ((lst lst0))
    (cond-expand
     [unsafe
      (if (eq? lst '()) 
	  lst
	  (cons (p (##sys#slot lst 0)) (loop (##sys#slot lst 1))) ) ]
     [else
      (cond ((eq? lst '()) lst)
	    ((pair? lst)
	     (cons (p (##sys#slot lst 0)) (loop (##sys#slot lst 1))) )
	    (else (##sys#not-a-proper-list-error lst0 'map)) ) ] ) ) )

(let ([car car]
      [cdr cdr] )
  (letrec ((mapsafe
	    (lambda (p lsts start loc)
	      (if (eq? lsts '())
		  lsts
		  (let ((item (##sys#slot lsts 0)))
		    (cond ((eq? item '())
			   (cond-expand [unsafe (##core#undefined)]
					[else (check lsts start loc)] ) )
			  ((pair? item)
			   (cons (p item) (mapsafe p (##sys#slot lsts 1) #f loc)) )
			  (else (##sys#not-a-proper-list-error item loc)) ) ) ) ) )
	   (check 
	    (lambda (lsts start loc)
	      (if (or (not start)
		      (let loop ((lsts lsts))
			(and (not (eq? lsts '()))
			     (not (eq? (##sys#slot lsts 0) '()))
			     (loop (##sys#slot lsts 1)) ) ) )
		  (##sys#error loc "lists are not of same length" lsts) ) ) ) )
    (set! for-each
	  (lambda (fn lst1 . lsts)
	    (if (null? lsts)
		(##sys#for-each fn lst1)
		(let loop ((all (cons lst1 lsts)))
		  (let ((first (##sys#slot all 0)))
		    (cond ((pair? first)
			   (apply fn (mapsafe car all #t 'for-each))
			   (loop (mapsafe cdr all #t 'for-each)) )
			  (else (check all #t 'for-each)) ) ) ) ) ) )
    (set! map
	  (lambda (fn lst1 . lsts)
	    (if (null? lsts)
		(##sys#map fn lst1)
		(let loop ((all (cons lst1 lsts)))
		  (let ((first (##sys#slot all 0)))
		    (cond ((pair? first)
			   (cons (apply fn (mapsafe car all #t 'map))
				 (loop (mapsafe cdr all #t 'map)) ) )
			  (else (check (##core#inline "C_i_cdr" all) #t 'map)
				'() ) ) ) ) ) ) ) ) )


;;; dynamic-wind:
;
; (taken more or less directly from SLIB)
;
; This implementation is relatively costly: we have to shadow call/cc
; with a new version that unwinds suspended thunks, but for this to
; happen the return-values of the escaping procedure have to be saved
; temporarily in a list. Since call/cc is very efficient under this
; implementation, and because allocation of memory that is to be
; garbage soon has also quite low overhead, the performance-penalty
; might be acceptable (ctak needs about 4 times longer).

(define ##sys#dynamic-winds '())

(define dynamic-wind
    (lambda (before thunk after)
      (before)
      (set! ##sys#dynamic-winds (cons (cons before after) ##sys#dynamic-winds))
      (##sys#call-with-values thunk
	(lambda results
	  (set! ##sys#dynamic-winds (##sys#slot ##sys#dynamic-winds 1))
	  (after)
	  (apply ##sys#values results) ) ) ) )

(define ##sys#dynamic-wind dynamic-wind)

(define call-with-current-continuation
  (lambda (proc)
    (let ((winds ##sys#dynamic-winds))
      (##sys#call-with-current-continuation
       (lambda (cont)
	 (proc
	  (lambda results
	    (unless (eq? ##sys#dynamic-winds winds)
	      (##sys#dynamic-unwind winds (fx- (length ##sys#dynamic-winds) (length winds))) )
	    (apply cont results) ) ) ) ) ) ) )

(define call/cc call-with-current-continuation)

(define (##sys#dynamic-unwind winds n)
  (cond [(eq? ##sys#dynamic-winds winds)]
	[(fx< n 0)
	 (##sys#dynamic-unwind (##sys#slot winds 1) (fx+ n 1))
	 ((##sys#slot (##sys#slot winds 0) 0))
	 (set! ##sys#dynamic-winds winds) ]
	[else
	 (let ([after (##sys#slot (##sys#slot ##sys#dynamic-winds 0) 1)])
	   (set! ##sys#dynamic-winds (##sys#slot ##sys#dynamic-winds 1))
	   (after)
	   (##sys#dynamic-unwind winds (fx- n 1)) ) ] ) )


;;; Ports:

(define (port? x) (##core#inline "C_i_portp" x))

(define (input-port? x)
  (and (##core#inline "C_blockp" x)
       (##core#inline "C_portp" x)
       (##sys#slot x 1) ) )

(define (output-port? x)
  (and (##core#inline "C_blockp" x)
       (##core#inline "C_portp" x)
       (not (##sys#slot x 1)) ) )

;;; Port layout:
;
; 0:  FP (special)
; 1:  input/output (bool)
; 2:  class (vector of procedures)
; 3:  name (string)
; 4:  row (fixnum)
; 5:  col (fixnum)
; 6:  EOF (bool)
; 7:  type ('stream | 'custom | 'string)
; 8:  closed (bool)
; 9:  data
; 10-15: reserved
;
; Port-class:
;
; 0:  (read-char PORT) -> CHAR | EOF
; 1:  (peek-char PORT) -> CHAR | EOF
; 2:  (write-char PORT CHAR)
; 3:  (write-string PORT STRING)
; 4:  (close PORT)
; 5:  (flush-output PORT)
; 6:  (char-ready? PORT) -> BOOL

(define (##sys#make-port i/o class name type)
  (let ([port (##core#inline_allocate ("C_a_i_port" 17))])
    (##sys#setislot port 1 i/o)
    (##sys#setslot port 2 class)
    (##sys#setslot port 3 name)
    (##sys#setislot port 4 0)
    (##sys#setislot port 5 0)
    (##sys#setslot port 7 type)
    port) )

(define ##sys#stream-port-class
  (vector (lambda (p)			; read-char
	    (##core#inline "C_read_char" p) )
	  (lambda (p)			; peek-char
	    (##core#inline "C_peek_char" p) )
	  (lambda (p c)			; write-char
	    (##core#inline "C_display_char" p c) )
	  (lambda (p s)			; write-string
	    (##core#inline "C_display_string" p s) )
	  (lambda (p)	    		; close
	    (##core#inline "C_close_file" p)
	    (##sys#update-errno) )
	  (lambda (p)			; flush-output
	    (##core#inline "C_flush_output" p) )
	  (lambda (p)			; char-ready?
	    (##core#inline "C_char_ready_p" p) ) ) )

(define ##sys#open-file-port (##core#primitive "C_open_file_port"))

(define ##sys#standard-input (##sys#make-port #t ##sys#stream-port-class "(stdin)" 'stream))
(define ##sys#standard-output (##sys#make-port #f ##sys#stream-port-class "(stdout)" 'stream))
(define ##sys#standard-error (##sys#make-port #f ##sys#stream-port-class "(stderr)" 'stream))

(##sys#open-file-port ##sys#standard-input 0 #f)
(##sys#open-file-port ##sys#standard-output 1 #f)
(##sys#open-file-port ##sys#standard-error 2 #f)

(define ##sys#check-port
  (lambda (x . loc)
    (if (or (not (##core#inline "C_blockp" x))
	    (not (##core#inline "C_portp" x)) )
	(##sys#signal-hook #:type-error (if (pair? loc) (car loc) #f) "argument is not a port" x) ) ) )

(define ##sys#check-port-mode
  (lambda (port mode . loc)
    (unless (eq? mode (##sys#slot port 1))
      (##sys#signal-hook 
       #:type-error (if (pair? loc) (car loc) #f)
       (if mode "port is not an input port" "port is not an output-port") ) ) ) )

(define (##sys#fetch-and-check-port-arg parg default)
  (let ((p (if (eq? parg '())
	       default
	       (##sys#slot parg 0) ) ) )
    (##sys#check-port p)
    p) )

(define (current-input-port . arg)
  (if (pair? arg)
      (let ([p (car arg)])
	(##sys#check-port p 'current-input-port)
	(set! ##sys#standard-input p) )
      ##sys#standard-input) )

(define (current-output-port . arg)
  (if (pair? arg)
      (let ([p (car arg)])
	(##sys#check-port p 'current-output-port)
	(set! ##sys#standard-output p) )
      ##sys#standard-output) )

(define (current-error-port . arg)
  (if (pair? arg)
      (let ([p (car arg)])
	(##sys#check-port p 'current-error-port)
	(set! ##sys#standard-error p) )
      ##sys#standard-error) )

(define (##sys#tty-port? port)
  (and (not (zero? (##sys#peek-unsigned-integer port 0)))
       (##core#inline "C_tty_portp" port) ) )

(define (##sys#port-data port) (##sys#slot port 9))
(define (##sys#pathname-resolution name thunk . _) (thunk name))

(let ([string-append string-append])
  (define (open name inp modes loc)
    (##sys#check-string name loc)
    (##sys#pathname-resolution
     name
     (lambda (name)
       (let ([fmode (if inp "r" "w")]
	     [bmode ""] 
	     [amode ""] )
	 (do ([modes modes (##sys#slot modes 1)])
	     ((null? modes))
	   (let ([o (##sys#slot modes 0)])
	     (case o
	       [(#:binary) (set! bmode "b")]
	       [(#:text) (set! bmode "")]
	       [(#:append) (set! amode "a")]
	       [else (##sys#error "invalid file option" o)] ) ) )
	 (let ([port (##sys#make-port inp ##sys#stream-port-class name 'stream)])
	   (unless (##sys#open-file-port port name (string-append fmode bmode amode))
	     (##sys#update-errno)
	     (##sys#signal-hook #:file-error loc "can not open file" name) )
	   port) ) )
     #:open (not inp) modes) )
  (define (close port loc)
    (##sys#check-port port loc)
    (unless (##sys#slot port 8)		; closed?
      ((##sys#slot (##sys#slot port 2) 4) port) ; close
      (##sys#setislot port 8 #t) )
    (##core#undefined) )
  (set! open-input-file (lambda (name . mode) (open name #t mode 'open-input-file)))
  (set! open-output-file (lambda (name . mode) (open name #f mode 'open-output-file)))
  (set! close-input-port (lambda (port) (close port 'close-input-port)))
  (set! close-output-port (lambda (port) (close port 'close-output-port))) )

(define call-with-input-file
  (let ([open-input-file open-input-file]
	[close-input-port close-input-port] )
    (lambda (name p . mode)
      (let ([f (apply open-input-file name mode)])
	(##sys#call-with-values
	 (lambda () (p f))
	 (lambda results
	   (close-input-port f)
	   (apply ##sys#values results) ) ) ) ) ) )

(define call-with-output-file
  (let ([open-output-file open-output-file]
	[close-output-port close-output-port] )
    (lambda (name p . mode)
      (let ([f (apply open-output-file name mode)])
	(##sys#call-with-values
	 (lambda () (p f))
	 (lambda results
	   (close-output-port f)
	   (apply ##sys#values results) ) ) ) ) ) )

(define with-input-from-file 
  (let ((open-input-file open-input-file)
	(close-input-port close-input-port) )
    (lambda (str thunk . mode)
      (let ((old ##sys#standard-input)
	    (file (apply open-input-file str mode)) )
	(set! ##sys#standard-input file)
	(##sys#call-with-values thunk
	  (lambda results
	    (close-input-port file)
	    (set! ##sys#standard-input old)
	    (apply ##sys#values results) ) ) ) ) ) )

(define with-output-to-file 
  (let ((open-output-file open-output-file)
	(close-output-port close-output-port) ) 
    (lambda (str thunk . mode)
      (let ((old ##sys#standard-output)
	    (file (apply open-output-file str mode)) )
	(set! ##sys#standard-output file)
	(##sys#call-with-values thunk
	  (lambda results
	    (close-output-port file)
	    (set! ##sys#standard-output old)
	    (apply ##sys#values results) ) ) ) ) ) )

(define (file-exists? name)
  (##sys#check-string name 'file-exists?)
  (##sys#pathname-resolution
   name
   (lambda (name) (not (eq? #f (##sys#file-info name))))
   #:exists?) )

(define (flush-output . port)
  (let ([port (##sys#fetch-and-check-port-arg port ##sys#standard-output)])
    (##sys#check-port-mode port #f 'flush-output)
    ((##sys#slot (##sys#slot port 2) 5) port) ; flush-output
    (##core#undefined) ) )

(define port-name
  (lambda (port)
    (##sys#check-port port 'port-name)
    (##sys#slot port 3) ) )

(define (set-port-name! port name)
  (##sys#check-port port 'set-port-name!)
  (##sys#check-string name 'set-port-name!)
  (##sys#setslot port 3 name) )

(define port-position
  (lambda (port)
    (##sys#check-port port 'port-position)
    (if (##sys#slot port 1) 
	(##sys#values (##sys#slot port 4) (##sys#slot port 5))
	(##sys#error 'port-position "can not compute position of port" port) ) ) )

(define delete-file
  (lambda (filename)
    (##sys#check-string filename 'delete-file)
    (##sys#pathname-resolution
     filename
     (lambda (name)
       (unless (eq? 0 (##core#inline "C_delete_file" (##sys#make-c-string filename)))
	 (##sys#update-errno)
	 (##sys#signal-hook #:file-error 'delete-file "can not delete file" filename) ) )
     #:delete) ) )

(define rename-file
  (lambda (old new)
    (##sys#check-string old 'rename-file)
    (##sys#check-string new 'rename-file)
    (##sys#pathname-resolution
     old
     (lambda (name)
       (unless (eq? 0 (##core#inline "C_rename_file" (##sys#make-c-string old) (##sys#make-c-string new)))
	 (##sys#update-errno)
	 (##sys#signal-hook #:file-error 'rename-file "can not rename file" old new) ) )
     #:rename new) ) )


;;; Parameters:

(define ##sys#default-parameter-vector (##sys#make-vector default-parameter-vector-size))
(define ##sys#current-parameter-vector '#())

(define make-parameter
  (let ([count 0])
    (lambda (init . guard)
      (let* ([guard (if (pair? guard) (car guard) (lambda (x) x))]
	     [val (guard init)] 
	     [i count] )
	(set! count (fx+ count 1))
	(when (fx>= i (##sys#size ##sys#default-parameter-vector))
	  (set! ##sys#default-parameter-vector 
	    (##sys#grow-vector ##sys#default-parameter-vector (fx+ i 1) (##core#undefined)) ) )
	(##sys#setslot ##sys#default-parameter-vector i val)
	(lambda arg
	  (let ([n (##sys#size ##sys#current-parameter-vector)])
	    (cond [(pair? arg)
		   (when (fx>= i n)
		     (set! ##sys#current-parameter-vector
		       (##sys#grow-vector ##sys#current-parameter-vector (fx+ i 1) ##sys#snafu) ) )
		   (##sys#setslot ##sys#current-parameter-vector i (guard (##sys#slot arg 0)))
		   (##core#undefined) ]
		  [(fx>= i n)
		   (##sys#slot ##sys#default-parameter-vector i) ]
		  [else
		   (let ([val (##sys#slot ##sys#current-parameter-vector i)])
		     (if (eq? val ##sys#snafu)
			 (##sys#slot ##sys#default-parameter-vector i) 
			 val) ) ] ) ) ) ) ) ) )


;;; Input:

(define (eof-object? x) (##core#inline "C_eofp" x))

(define (char-ready? . port)
  (let ([port (##sys#fetch-and-check-port-arg port ##sys#standard-input)])
    (##sys#check-port-mode port #t 'char-ready?)
    ((##sys#slot (##sys#slot port 2) 6) port) ) ) ; char-ready?

(define (read-char . port)
  (let ([p (##sys#fetch-and-check-port-arg port ##sys#standard-input)])
    (##sys#check-port-mode p #t 'read-char)
    (let ([c (if (##sys#slot p 6)
		 (begin
		   (##sys#setislot p 6 #f)
		   (##sys#fudge 1) ) ; eof
		 ((##sys#slot (##sys#slot p 2) 0) p) ) ] ) ; read-char
      (cond [(eq? c #\newline)
	     (##sys#setslot p 4 (fx+ (##sys#slot p 4) 1))
	     (##sys#setslot p 5 0) ]
	    [(not (##core#inline "C_eofp" c))
	     (##sys#setslot p 5 (fx+ (##sys#slot p 5) 1)) ] )
      c) ) )

(define (peek-char . port)
  (let ([p (##sys#fetch-and-check-port-arg port ##sys#standard-input)])
    (##sys#check-port-mode p #t 'peek-char)
    (if (##sys#slot p 6)
	(##sys#fudge 1)
	(let ([c ((##sys#slot (##sys#slot p 2) 1) p)]) ; peek-char
	  (when (##core#inline "C_eofp" c)
	    (##sys#setislot p 6 #t) )
	  c) ) ) )

(define (read . port)
  (let ([port (##sys#fetch-and-check-port-arg port ##sys#standard-input)])
    (##sys#check-port-mode port #t 'read)
    (##sys#read port ##sys#default-read-info-hook) ) )

(define ##sys#read-line-counter 0)
(define ##sys#default-read-info-hook #f)
(define ##sys#read-error-with-line-number #f)
(define ##sys#current-namespace #f)
(define ##sys#default-namespace-prefix #f)
(define ##sys#enable-qualifiers #t)

(define strict-reader (make-parameter #f))
(define case-sensitive (make-parameter #t))
(define keyword-style (make-parameter #:normal))

(define ##sys#read-warning
  (let ([string-append string-append])
    (lambda (msg . args)
      (apply
       ##sys#warn
       (if ##sys#read-error-with-line-number
	   (string-append msg " in line " (number->string ##sys#read-line-counter))
	   msg)
       args) ) ) )

(define ##sys#read-error
  (let ([string-append string-append] )
    (lambda (msg . args)
      (apply
       ##sys#error 
       (if ##sys#read-error-with-line-number
	   (string-append msg " in line " (number->string ##sys#read-line-counter))
	   msg)
       args) ) ) )

(define ##sys#read
  (let ([list->string list->string]
	[read-char read-char]
	[peek-char peek-char]
	[reverse reverse]
	[list? list?]
	[string-append string-append]
	[string string]
	[container (lambda (c) (##sys#read-error "unexpected list terminator" c))] 
	[char-name char-name]
	[srp strict-reader]		; keep original def of parameter...
	[csp case-sensitive]
	[ksp keyword-style]
	[kwprefix (string (integer->char 0))] )
    (lambda (port infohandler)
      (let ([terminating-characters '(#\, #\; #\( #\) #\[ #\] #\{ #\} #\' #\")]
	    [inexact-flag #f] 
	    [srp (srp)]			; ...and get it's value
	    [csp (csp)]
	    [ksp (ksp)]
	    [rat-flag #f] )

	(define (info class data val)
	  (if infohandler
	      (infohandler class data val)
	      data) )

	(define (advance)
	  (set! ##sys#read-line-counter (fx+ ##sys#read-line-counter 1)) )

	(define (badsyntax msg)
	  (##sys#read-error (string-append msg " not supported in strict reader mode")) )

        (define (readrec)

          (define (r-spaces)
            (let loop ([c (peek-char port)])
	      (cond ((##core#inline "C_eofp" c))
		    ((eq? #\; c)
		     (let skip ((c (read-char port)))
		       (if (and (not (##core#inline "C_eofp" c)) (not (eq? #\newline c)))
			   (skip (read-char port))
			   (begin
			     (advance)
			     (loop (peek-char port)) ) ) ) )
		    ((eq? c #\newline)
		     (advance)
		     (read-char port)
		     (loop (peek-char port)) )
		    ((char-whitespace? c)
		     (read-char port)
		     (loop (peek-char port)) ) ) ) )
          
          (define (r-string term)
            (if (eq? (read-char port) term)
		(let loop ((c (read-char port)) (lst '()))
		  (cond ((##core#inline "C_eofp" c) 
			 (##sys#read-error "unterminated string") )
			((eq? #\\ c)
			 (set! c (read-char port))
			 (case c
			   ((#\t) (loop (read-char port) (cons #\tab lst)))
			   ((#\r) (loop (read-char port) (cons #\return lst)))
			   ((#\b) (loop (read-char port) (cons #\backspace lst)))
			   ((#\n) (loop (read-char port) (cons #\newline lst)))
			   ((#\x) 
			    (let* ([x1 (read-char port)]
				   [x2 (read-char port)] )
			      (if (or (eof-object? x1) (eof-object? x2))
				  (##sys#read-error "unterminated string constant") 
				  (let ([n (string->number (string x1 x2) 16)])
				    (if (and n (exact? n))
					(loop (read-char port) (cons (integer->char n) lst))
					(##sys#read-error (string-append "invalid escape-sequence '\\x" (string x1 x2) "\'")) ) ) ) ) )
			   (else (loop (read-char port) (cons c lst))) ) )
			((eq? term c) (list->string (reverse lst)))
			((eq? c #\newline) 
			 (advance)
			 (loop (read-char port) (cons c lst)) )
			(else (loop (read-char port) (cons c lst))) ) )
		(##sys#read-error (string-append "missing `" (string term) "'")) ) )
                    
	  (define (r-list start end)
	    (if (eq? (read-char port) start)
		(let ([first #f]
		      [ln0 #f]
		      [outer-container container] )
		  (##sys#call-with-current-continuation
		   (lambda (return)
		     (set! container
		       (lambda (c)
			 (if (eq? c end)
			     (return #f)
			     (##sys#read-error "list-terminator mismatch" c end) ) ) )
		     (let loop ([last '()])
		       (r-spaces)
		       (unless first (set! ln0 ##sys#read-line-counter))
		       (let ([c (peek-char port)])
			 (cond ((##core#inline "C_eofp" c)
				(##sys#read-error "unterminated list") )
			       ((eq? c end)
				(read-char port) )
			       ((eq? c #\.)
				(read-char port)
				(let ([c2 (peek-char port)])
				  (cond [(char-whitespace? c2)
					 (unless (pair? last)
					   (##sys#read-error "invalid use of '.'") )
					 (r-spaces)
					 (##sys#setslot last 1 (readrec))
					 (r-spaces)
					 (unless (eq? (read-char port) end)
					   (##sys#read-error "missing ')'") ) ]
					[else
					 (let* ((tok (string-append "." (r-token)))
						(n (and (char-numeric? c2) (string->number tok)))
						(val (or n (resolve-symbol tok))) 
						(node (cons val '())) )
					   (if first 
					       (##sys#setslot last 1 node)
					       (set! first node) )
					   (loop node) ) ] ) ) )
			       (else
				(let ([node (cons (readrec) '())])
				  (if first
				      (##sys#setslot last 1 node)
				      (set! first node) )
				  (loop node) ) ) ) ) ) ) )
		  (set! container outer-container)
		  (if first
		      (info 'list-info first ln0)
		      '() ) )
		(##sys#read-error "missing token" start) ) )
          
	    (define (r-vector)
	      (let ([lst (r-list #\( #\))])
		(if (list? lst)
		    (##sys#list->vector lst)
		    (##sys#read-error "invalid vector syntax" lst) ) ) )
          
	    (define (r-number radix)
	      (set! inexact-flag #f)
	      (set! rat-flag #f)
	      (let* ([tok (r-token)]
		     [val (string->number tok (or radix 10))] )
		(cond [val
		       (when (and (inexact? val) rat-flag)
			 (##sys#read-warning "can not represent exact fraction - coerced to flonum" tok) )
		       val]
		      [radix (##sys#read-error "illegal number syntax" tok)]
		      [else (resolve-symbol tok)] ) ) )

	    (define (r-number-with-exactness radix)
	      (cond [(char=? #\# (peek-char port))
		     (read-char port)
		     (let ([c2 (read-char port)])
		       (cond [(eof-object? c2) (##sys#read-error "unexpected end of numeric literal")]
			     [(char=? c2 #\i) (exact->inexact (r-number radix))]
			     [(char=? c2 #\e) (inexact->exact (r-number radix))]
			     [else (##sys#read-error "illegal number syntax - invalid exactness prefix" c2)] ) ) ]
		    [else (r-number radix)] ) )
          
	    (define (r-number-with-radix)
	      (cond [(char=? #\# (peek-char port))
		     (read-char port)
		     (let ([c2 (read-char port)])
		       (cond [(eof-object? c2) (##sys#read-error "unexpected end of numeric literal")]
			     [(char=? c2 #\x) (r-number 16)]
			     [(char=? c2 #\o) (r-number 8)]
			     [(char=? c2 #\b) (r-number 2)]
			     [else (##sys#read-error "illegal number syntax - invalid radix" c2)] ) ) ]
		    [else (r-number 10)] ) )
        
	    (define (r-token)
	      (let loop ([c (peek-char port)] [lst '()])
		(cond [(or (eof-object? c)
			   (char-whitespace? c)
			   (memq c terminating-characters) )
		       (list->string (reverse lst)) ]
		      [else
		       (cond [(or (char=? c #\.) (char=? c #\#) (char=? c #\e) (char=? c #\E)) (set! inexact-flag #t)]
			     [(char=? c #\/) (set! rat-flag #t)] )
		       (read-char port)
		       (loop (peek-char port) 
			     (cons (if csp
				       c
				       (char-downcase c) )
				   lst) ) ] ) ) )

	    (define (r-next-token)
	      (r-spaces)
	      (r-token) )
          
	    (define (r-symbol)
	      (let ((s (resolve-symbol
			(if (and (char=? (peek-char port) #\|) (not srp))
			    (r-xtoken)
			    (r-token) ) ) ) )
		(info 'symbol-info s ##sys#read-line-counter) ) )

	    (define (r-xtoken)
	      (if (char=? #\| (read-char port))
		  (let loop ((c (read-char port)) (lst '()))
		    (cond ((eof-object? c) (##sys#read-error "unexpected end of `| ... |' symbol"))
			  ((char=? c #\\)
			   (let ((c (read-char port)))
			     (loop (read-char port) (cons c lst)) ) )
			  ((char=? c #\|)
			   (list->string (reverse lst)) )
			  (else (loop (read-char port) (cons c lst))) ) )
		  (##sys#read-error "missing \'|\'") ) )
          
	    (define (r-char)
	      (let* ([c (peek-char port)]
		     [tk (r-token)] )
		(cond [(char-name (##sys#intern-symbol tk))]
		      [(fx> (string-length tk) 1) (##sys#read-error "unknown named character" tk)]
		      [(memq c terminating-characters) (read-char port)]
		      [else c] ) ) )

	    (define (r-comment)
	      (let loop ((i 0))
		(let ((c (read-char port)))
		  (case c
		    ((#\newline)
		     (advance)
		     (loop i) )
		    ((#\|) (if (eq? #\# (read-char port))
			       (if (not (eq? i 0))
				   (loop (fx- i 1)) )
			       (loop i) ) )
		    ((#\#) (loop (if (eq? #\| (read-char port))
				     (fx+ i 1)
				     i) ) )
		    (else (if (eof-object? c)
			      (##sys#read-error "unterminated block-comment")
			      (loop i) ) ) ) ) ) )

	    (define (r-namespace)
	      (set! ##sys#current-namespace (##sys#make-vector namespace-size '()))
	      (let* ([ns (r-next-token)]
		     [nslen (##sys#size ns)]
		     [p (##sys#make-string 1)] )
		(when (fx> nslen namespace-max-id-len)
		  (set! ns (##sys#substring ns 0 namespace-max-id-len))
		  (set! nslen namespace-max-id-len) )
		(##sys#setbyte p 0 (##sys#size ns))
		(let ([prefix (string-append p ns)])
		  (let loop ([toks '()])
		    (r-spaces)
		    (cond [(char=? #\} (peek-char port))
			   (read-char port)
			   (for-each
			    (lambda (tok)
			      (let ([i (##core#inline
					"C_fixnum_modulo"
					(##core#inline "C_hash_string" tok) namespace-size)])
				(##sys#setslot 
				 ##sys#current-namespace i
				 (cons (cons tok (##sys#intern-symbol (string-append prefix tok)))
				       (##sys#slot ##sys#current-namespace i) ) ) ) )
			    toks) ]
			  [else (loop (cons (r-next-token) toks))] ) ) ) ) )

	    (define (r-ext-symbol)
	      (let* ([p (##sys#make-string 1)]
		     [tok (r-token)] 
		     [toklen (##sys#size tok)] )
		(unless ##sys#enable-qualifiers 
		  (##sys#read-error "qualified symbol syntax is not allowed" tok) )
		(let loop ([i 0])
		  (cond [(fx>= i toklen) (##sys#read-error "invalid qualified symbol syntax" tok)]
			[(fx= (##sys#byte tok i) (char->integer #\#))
			 (when (fx> i namespace-max-id-len)
			   (set! tok (##sys#substring tok 0 namespace-max-id-len)) )
			 (##sys#setbyte p 0 i)
			 (##sys#intern-symbol
			  (string-append p (##sys#substring tok 0 i) (##sys#substring tok (fx+ i 1) toklen)) ) ]
			[else (loop (fx+ i 1))] ) ) ) )

	    (define (resolve-symbol tok)
	      (if (string=? tok ".")
		  (##sys#read-error "invalid use of '.'")
		  (let ([len (##sys#size tok)])
		    (cond [(and (fx> len 0)
				(or (and (eq? ksp #:prefix)
					 (char=? #\: (##core#inline "C_subchar" tok 0)) 
					 (##sys#substring tok 1 len) )
				    (and (eq? ksp #:suffix) 
					 (char=? #\: (##core#inline "C_subchar" tok (fx- len 1)))
					 (##sys#substring tok 0 (fx- len 1)) ) ) )
			   => build-keyword] ; ugh
			  [(not ##sys#current-namespace) (build-symbol tok)]
			  [else
			   (let ([i (##core#inline "C_fixnum_modulo" (##core#inline "C_hash_string" tok) namespace-size)])
			     (let loop ([bucket (##sys#slot ##sys#current-namespace i)])
			       (if (null? bucket)
				   (build-symbol tok)
				   (let ([e (##sys#slot bucket 0)])
				     (if (string=? tok (##sys#slot e 0))
					 (##sys#slot e 1)
					 (loop (##sys#slot bucket 1)) ) ) ) ) ) ] ) ) ) )

	    (define (build-symbol tok)
	      (##sys#intern-symbol
	       (if ##sys#default-namespace-prefix
		   (string-append ##sys#default-namespace-prefix tok)
		   tok) ) )
	  
	    (define (build-keyword tok)
	      (##sys#intern-symbol (string-append kwprefix tok)) )

	    (r-spaces)
	    (let* ([c (peek-char port)]
		   [h (and ##sys#special-read-syntax-table
			   (##sys#slot ##sys#special-read-syntax-table (char->integer c)) ) ] )
	      (if h
		  (h c port)
		  (case c
		    ((#\')
		     (read-char port)
		     (list 'quote (readrec)) )
		    ((#\`)
		     (read-char port)
		     (list 'quasiquote (readrec)) )
		    ((#\,)
		     (read-char port)
		     (cond ((eq? (peek-char port) #\@)
			    (read-char port)
			    (list 'unquote-splicing (readrec)) )
			   (else (list 'unquote (readrec))) ) )
		    ((#\#)
		     (read-char port)
		     (let ((dchar (peek-char port)))
		       (case (char-downcase dchar)
			 ((#\x) (read-char port) (r-number-with-exactness 16))
			 ((#\o) (read-char port) (r-number-with-exactness 8))
			 ((#\b) (read-char port) (r-number-with-exactness 2))
			 ((#\i) (read-char port) (exact->inexact (r-number-with-radix)))
			 ((#\e) (read-char port) (inexact->exact (r-number-with-radix)))
			 ((#\() (r-vector))
			 ((#\\) (read-char port) (r-char))
			 ((#\|)
			  (read-char port)
			  (r-comment) (readrec) )
			 ((#\{) 
			  (read-char port) 
			  (r-namespace) (readrec) )
			 ((#\#) 
			  (read-char port)
			  (r-ext-symbol) )
			 ((#\;) 
			  (read-char port)
			  (readrec) (readrec) )
			 ((#\') 
			  (read-char port)
			  (list 'syntax (readrec)) )
			 ((#\:) 
			  (read-char port)
			  (build-keyword (r-token)) )
			 (else (##sys#user-read-hook dchar port)) ) ) )
		    ((#\() (r-list #\( #\)))
		    ((#\[) 
		     (r-list #\[ #\]) )
		    ((#\) #\]) 
		     (read-char port)
		     (container c) )
		    ((#\{ #\})
		     (read-char port)
		     (##sys#read-error "illegal character" c))
		    ((#\") (r-string #\"))
		    ((#\.) (r-number #f))
		    ((#\-) (r-number #f))
		    (else (cond [(eof-object? c) c]
				[(char-numeric? c) (r-number #f)]
				[else (r-symbol)] ) ) ) ) ) )

	(readrec) ) ) ) )


;;; Hooks for user-defined read-syntax:
;
; - Redefine this to handle new read-syntaxes. If 'char' doesn't match
;   your character then call the previous handler.
; - Don't forget to read 'char', it's only peeked at this point.

(define ##sys#user-read-hook
  (let ([read-char read-char] )
    (lambda (char port)
      (case char
	;; I put it here, so the SRFI-4 unit can intercept '#f...'
	((#\f #\F) (read-char port) #f)
	((#\t #\T) (read-char port) #t)
	(else (##sys#read-error "invalid sharp-sign read syntax" char) ) ) ) ) )

;;; Table for specially handled read-syntax:
;
; - should be either #f or a 256-element vector containing procedures
; - the procedure is called with two arguments, a char (peeked) and a port and should return an expression

(define ##sys#special-read-syntax-table #f) 


;;; Output:

(define (##sys#write-char c . port)
  (##sys#check-char c 'write-char)
  (let ([p (##sys#fetch-and-check-port-arg port ##sys#standard-output)])
    (##sys#check-port-mode p #f 'write-char)
    ((##sys#slot (##sys#slot p 2) 2) p c) ) )

(define write-char ##sys#write-char)
(define (newline . port) (apply ##sys#write-char #\newline port))

(define (write x . port)
  (##sys#print x #t (##sys#fetch-and-check-port-arg port ##sys#standard-output)) )

(define (display x . port)
  (##sys#print x #f (##sys#fetch-and-check-port-arg port ##sys#standard-output)) )

(define print
  (lambda args
    (for-each (lambda (x) (##sys#print x #f ##sys#standard-output)) args)
    (##sys#write-char #\newline) ) )

(define (print* . args)
  (for-each (lambda (x) (##sys#print x #f ##sys#standard-output)) args) )

(define ##sys#current-print-length 0)
(define ##sys#print-length-limit #f)
(define ##sys#print-exit #f)

(define ##sys#print
  (let ([char-name char-name]
	[csp case-sensitive]
	[ksp keyword-style]
	[string-append string-append] )
    (lambda (x readable port)
      (##sys#check-port-mode port #f)
      (let ([csp (csp)]
	    [ksp (ksp)] )

	(define (outstr port str)
	  (set! ##sys#current-print-length (fx+ ##sys#current-print-length (##sys#size str)))
	  (when (and ##sys#print-length-limit (fx>= ##sys#current-print-length ##sys#print-length-limit))
	    (outstr0 port "...")
	    (##sys#print-exit #t) )
	  (outstr0 port str) )
	       
	(define (outstr0 port str)
	  ((##sys#slot (##sys#slot port 2) 3) port str) )

	(define (outchr port chr)
	  (set! ##sys#current-print-length (fx+ ##sys#current-print-length 1))
	  (when (and ##sys#print-length-limit (fx>= ##sys#current-print-length ##sys#print-length-limit))
	    (outstr0 port "...")
	    (##sys#print-exit #t) )
	  ((##sys#slot (##sys#slot port 2) 2) port chr) )

	(define (specialchar? chr)
	  (let ([c (char->integer chr)])
	    (or (fx<= c 32)
		(fx>= c 128)
		(memq chr '(#\( #\) #\| #\, #\[ #\] #\{ #\} #\' #\" #\; #\\)) ) ) )

	(define (outreadablesym port str)
	  (let ([len (##sys#size str)])
	    (outchr port #\|)
	    (let loop ([i 0])
	      (if (fx>= i len)
		  (outchr port #\|)
		  (let ([c (##core#inline "C_subchar" str i)])
		    (when (specialchar? c) (outchr port #\\))
		    (outchr port c)
		    (loop (fx+ i 1)) ) ) ) ) )

	(define (sym-is-readable? str)
	  (let ([len (##sys#size str)])
	    (and (fx> len 0)
		 (not (string=? "." str))
		 (not (string=? "#" str))
		 (not (string=? "##" str))
		 (let loop ([i (fx- len 1)])
		   (or (fx< i 0)
		       (let ([c (##core#inline "C_subchar" str i)])
			 (and (or csp (not (char-upper-case? c)))
			      (not (specialchar? c))
			      (loop (fx- i 1)) ) ) ) ) ) ) )

	(let out ([x x])
	  (cond ((eq? x '()) (outstr port "()"))
		((eq? x #t) (outstr port "#t"))
		((eq? x #f) (outstr port "#f"))
		((##core#inline "C_eofp" x) (outstr port "#<eof>"))
		((##core#inline "C_undefinedp" x) (outstr port "#<unspecified>"))
		((##core#inline "C_charp" x)
		 (cond (readable
			(outstr port "#\\")
			(let ([cn (char-name x)])
			  (if cn
			      (outstr port (##sys#slot cn 1))
			      (outchr port x) ) ) )
		       (else (outchr port x)) ) )
		((##core#inline "C_fixnump" x) (outstr port (number->string x)))
		((not (##core#inline "C_blockp" x)) (outstr port "#<unprintable object>"))
		((##core#inline "C_pointerp" x)
		 (outstr port "#<pointer ")
		 (outstr port (number->string (##sys#pointer->address x) 16))
		 (outchr port #\>) )
		((##core#inline "C_taggedpointerp" x)
		 (outstr port "#<tagged pointer ")
		 (out (##sys#slot x 1))
		 (outchr port #\>) )
		((##core#inline "C_flonump" x) (outstr port (number->string x)))
		((##core#inline "C_stringp" x)
		 (cond (readable
			(outchr port #\")
			(do ((i 0 (fx+ i 1))
			     (c (##core#inline "C_block_size" x) (fx- c 1)) )
			    ((eq? c 0)
			     (outchr port #\") )
			  (let ((chr (##core#inline "C_subbyte" x i)))
			    (case chr
			      ((34) (outstr port "\\\""))
			      ((92) (outstr port "\\\\"))
			      (else
			       (if (fx< chr 32)
				   (begin
				     (outchr port #\\)
				     (outchr
				      port
				      (case chr
					((9) #\t)
					((10) #\n)
					((13) #\r)
					((8) #\b)
					(else (##core#inline "C_fix_to_char" chr)) ) ) )
				   (outchr port (##core#inline "C_fix_to_char" chr)) ) ) ) ) ) )
		       (else (outstr port x)) ) )
		((##core#inline "C_symbolp" x)
		 (cond [(fx= 0 (##sys#byte (##sys#slot x 1) 0))
			(let ([str (##sys#symbol->string x)])
			  (case ksp
			    [(#:prefix) 
			     (outchr port #\:)
			     (outstr port str) ]
			    [(#:suffix) 
			     (outstr port str)
			     (outchr port #\:) ]
			    [else
			     (outstr port "#:")
			     (outstr port str) ] ) ) ]
		       [else
			(let ([str (##sys#symbol->qualified-string x)])
			  (if (or (not readable) (sym-is-readable? str))
			      (outstr port str)
			      (outreadablesym port str) ) ) ] ) )
		((##core#inline "C_pairp" x)
		 (outchr port #\()
		 (out (##sys#slot x 0))
		 (do ((x (##sys#slot x 1) (##sys#slot x 1)))
		     ((or (not (##core#inline "C_blockp" x)) (not (##core#inline "C_pairp" x)))
		      (if (not (eq? x '()))
			  (begin
			    (outstr port " . ")
			    (out x) ) )
		      (outchr port #\)) )
		   (outchr port #\space)
		   (out (##sys#slot x 0)) ) )
		((##core#inline "C_bytevectorp" x)
		 (if (##core#inline "C_permanentp" x)
		     (outstr port "#<static bytevector>")
		     (outstr port "#<bytevector>") ) )
		((##core#inline "C_structurep" x) (##sys#user-print-hook x readable port))
		((##core#inline "C_closurep" x) (outstr port "#<procedure>"))
 		((##core#inline "C_locativep" x) (outstr port "#<locative>"))
		((##core#inline "C_portp" x)
		 (if (##sys#slot x 1)
		     (outstr port "#<input port ")
		     (outstr port "#<output port ") )
		 (outstr port (##sys#slot x 3))
		 (outchr port #\>) )
		((##core#inline "C_vectorp" x)
		 (let ((n (##core#inline "C_block_size" x)))
		   (cond ((eq? 0 n)
			  (outstr port "#()") )
			 (else
			  (outstr port "#(")
			  (out (##sys#slot x 0))
			  (do ((i 1 (fx+ i 1))
			       (c (fx- n 1) (fx- c 1)) )
			      ((eq? c 0)
			       (outchr port #\)) )
			    (outchr port #\space)
			    (out (##sys#slot x i)) ) ) ) ) )
		(else (##sys#error "unprintable non-immediate object encountered")) ) ) ) ) ) )

(define ##sys#record-printers '())

(define (##sys#register-record-printer type proc)
  (let ([a (assq type ##sys#record-printers)])
    (if a 
	(##sys#setslot a 1 proc)
	(set! ##sys#record-printers (cons (cons type proc) ##sys#record-printers)) )
    (##core#undefined) ) )

(define (##sys#user-print-hook x readable port)
  (let* ([type (##sys#slot x 0)]
	 [typename (if (eq? 'record type) (##sys#slot (##sys#slot x 1) 1) type)]
	 [a (assq typename ##sys#record-printers)] )
    (cond [a ((##sys#slot a 1) x port)]
	  [else
	   (##sys#print "#<" #f port)
	   (##sys#print (##sys#symbol->string type) #f port)
	   (case type
	     [(condition)
	      (##sys#print ": " #f port)
	      (##sys#print (##sys#slot x 1) #f port) ]
	     [(record-type)
	      (##sys#print ": " #f port)
	      (##sys#print (##sys#slot x 1) #f port) ]
	     [(record)
	      (##sys#print ": " #f port)
	      (##sys#print (##sys#slot (##sys#slot x 1) 1) #f port) ]
	     [(thread)
	      (##sys#print ": " #f port)
	      (##sys#print (##sys#slot x 6) #f port) ] )
	   (##sys#print #\> #f port) ] ) ) )

(define ##sys#with-print-length-limit
  (let ([call-with-current-continuation call-with-current-continuation])
    (lambda (limit thunk)
      (call-with-current-continuation
       (lambda (return)
	 (fluid-let ((##sys#print-length-limit limit)
		     (##sys#print-exit return) 
		     (##sys#current-print-length 0) )
	   (thunk) ) ) ) ) ) )


;;; Bitwise fixnum operations:

(define (bitwise-and . xs)
  (let loop ([x -1] [xs xs])
    (if (null? xs)
	x
	(loop (##core#inline "C_fixnum_and" x (##sys#slot xs 0)) (##sys#slot xs 1)) ) ) )

(define (bitwise-ior . xs)
  (let loop ([x 0] [xs xs])
    (if (null? xs)
	x
	(loop (##core#inline "C_fixnum_or" x (##sys#slot xs 0)) (##sys#slot xs 1)) ) ) )

(define (bitwise-xor . xs)
  (let loop ([x 0] [xs xs])
    (if (null? xs)
	x
	(loop (##core#inline "C_fixnum_xor" x (##sys#slot xs 0)) (##sys#slot xs 1)) ) ) )

(define (bitwise-not x)
  (##sys#check-exact x 'bitwise-not)
  (##core#inline "C_fixnum_not" x) )

(define (arithmetic-shift x y)
  (##sys#check-exact x 'arithmetic-shift)
  (##sys#check-exact y 'arithmetic-shift)
  (if (fx< y 0)
      (##core#inline "C_fixnum_shift_right" x (##core#inline "C_fixnum_negate" y))
      (##core#inline "C_fixnum_shift_left" x y) ) )


;;; String ports:
;
; - Port-slots:
;
;   Input:
;
;   10: position
;   11: len
;   12: string
;
;   Output:
;
;   10: position
;   11: limit
;   12: output

(define ##sys#string-port-class
  (letrec ([check 
	    (lambda (p n)
	      (let* ([position (##sys#slot p 10)]
		     [limit (##sys#slot p 11)] 
		     [output (##sys#slot p 12)]
		     [limit2 (fx+ position n)] )
		(when (fx>= limit2 limit)
		  (when (fx>= limit2 maximal-string-length)
		    (##sys#error "string buffer full" p) )
		  (let* ([limit3 (fxmin maximal-string-length (fx+ limit limit))]
			 [buf (##sys#make-string limit3)] )
		    (##sys#copy-bytes output buf 0 0 position)
		    (##sys#setslot p 12 buf)
		    (##sys#setislot p 11 limit3)
		    (check p n) ) ) ) ) ] )
    (vector (lambda (p)			; read-char
	      (let ([position (##sys#slot p 10)]
		    [string (##sys#slot p 12)]
		    [len (##sys#slot p 11)] )
		(if (>= position len)
		    (##sys#fudge 1)
		    (let ((c (##core#inline "C_subchar" string position)))
		      (##sys#setislot p 10 (fx+ position 1))
		      c) ) ) )
	    (lambda (p)			; peek-char
	      (let ([position (##sys#slot p 10)]
		    [string (##sys#slot p 12)]
		    [len (##sys#slot p 11)] )
		(if (fx>= position len)
		    (##sys#fudge 1)
		    (##core#inline "C_subchar" string position) ) ) )
	    (lambda (p c)		; write-char
	      (check p 1)	
	      (let ([position (##sys#slot p 10)]
		    [output (##sys#slot p 12)] )
		(##core#inline "C_setsubchar" output position c)
		(##sys#setislot p 10 (fx+ position 1)) ) )
	    (lambda (p str)		; write-string
	      (let ([len (##core#inline "C_block_size" str)])
		(check p len)
		(let ([position (##sys#slot p 10)]
		      [output (##sys#slot p 12)] )
		  (do ((i 0 (fx+ i 1)))
		      ((fx>= i len) (##sys#setislot p 10 position))
		    (##core#inline "C_setsubchar" output position (##core#inline "C_subchar" str i))
		    (set! position (fx+ position 1)) ) ) ) )
	    (lambda (p)	    		; close
	      (##sys#setislot p 10 (##sys#slot p 11)) )
	    (lambda (p) #f)		; flush-output
	    (lambda (p)			; char-ready?
	      (fx< (##sys#slot p 10) (##sys#slot p 11)) ) ) ) )

(define open-input-string 
  (lambda (string)
    (##sys#check-string string 'open-input-string)
    (let ([port (##sys#make-port #t ##sys#string-port-class "(string)" 'string)])
      (##sys#setislot port 11 (##core#inline "C_block_size" string))
      (##sys#setislot port 10 0)
      (##sys#setslot port 12 string)
      port) ) )

(define open-output-string
  (lambda ()
    (let ([port (##sys#make-port #f ##sys#string-port-class "(string)" 'string)])
      (##sys#setislot port 10 0)
      (##sys#setislot port 11 output-string-initial-size)
      (##sys#setslot port 12 (##sys#make-string output-string-initial-size))
      port) ) )

(define get-output-string
  (lambda (port)
    (##sys#check-port port 'get-output-string)
    (##sys#check-port-mode port #f 'get-output-string)
    (if (not (eq? 'string (##sys#slot port 7)))
	(##sys#signal-hook #:type-error 'get-output-string "argument is not a string-output-port" port) 
	(##sys#substring (##sys#slot port 12) 0 (##sys#slot port 10)) ) ) )

(define ##sys#print-to-string
  (let ([get-output-string get-output-string]
	[open-output-string open-output-string] )
    (lambda xs
      (let ([out (open-output-string)])
	(for-each (lambda (x) (##sys#print x #f out)) xs)
	(get-output-string out) ) ) ) )


;;; Platform configuration inquiry:

(define software-type
  (let ([sym (string->symbol ((##core#primitive "C_software_type")))])
    (lambda () sym) ) )

(define machine-type
  (let ([sym (string->symbol ((##core#primitive "C_machine_type")))])
    (lambda () sym) ) )

(define software-version
  (let ([sym (string->symbol ((##core#primitive "C_software_version")))])
    (lambda () sym) ) )

(define build-platform
  (let ([sym (string->symbol ((##core#primitive "C_build_platform")))])
    (lambda () sym) ) )

(define chicken-version
  (let ([v (string-append (number->string build-version) "." (number->string build-number))])
    (lambda () v) ) )

(define pathname-directory-separator
  (let ([st (software-type)])
    (if (or (eq? 'msdos st)
	    (and (eq? 'windows st) 
		 (let ([bp (build-platform)])
		   (not (or (eq? 'cygwin bp) (eq? 'mingw32 bp))) ) ) )
	#\\
	#\/) ) )

(define pathname-extension-separator #\.)


;;; Feature identifiers:

(define ##sys#->feature-id
  (let ([string->keyword string->keyword]
	[string-append string-append]
	[keyword? keyword?] )
    (define (err . args)
      (apply ##sys#signal-hook #:type-error "bad argument type - not a valid feature identifer" args) )
    (define (prefix s)
      (if s 
	  (string-append s "-")
	  "") )
    (lambda (x)
      (cond [(string? x) (string->keyword x)]
	    [(keyword? x) x]
	    [(symbol? x) (string->keyword (##sys#symbol->string x))]
	    [else (err x)] ) ) ) )

(define ##sys#features '(#:chicken #:srfi-22 #:srfi-23 #:srfi-30 #:srfi-39))

(define (register-feature! . fs)
  (for-each
   (lambda (f)
     (let ([id (##sys#->feature-id f)])
       (unless (memq id ##sys#features) (set! ##sys#features (cons id ##sys#features))) ) )
   fs)
  (##core#undefined) )

(define (unregister-feature! . fs)
  (let ([fs (map ##sys#->feature-id fs)])
    (set! ##sys#features
      (let loop ([ffs ##sys#features])
	(if (null? ffs)
	    '()
	    (let ([f (##sys#slot ffs 0)]
		  [r (##sys#slot ffs 1)] )
	      (if (memq f fs)
		  (loop r)
		  (cons f (loop r)) ) ) ) ) )
    (##core#undefined) ) )

(define (features) ##sys#features)


;;; Interrupt handling:

(let ([count 0])
  (set! ##sys#enable-interrupts
    (lambda val
      (set! count (fx+ count (if (pair? val) (car val) 1)))
      (when (eq? count 0) (##core#inline "C_enable_interrupts")) ) )
  (set! ##sys#disable-interrupts
    (lambda ()
      (when (eq? count 0) (##core#inline "C_disable_interrupts"))
      (set! count (fx- count 1)) ) ) )

(define enable-interrupts ##sys#enable-interrupts)
(define disable-interrupts ##sys#disable-interrupts)

(define (##sys#user-interrupt-hook)
  (define (break) (##sys#signal-hook #:user-interrupt #f))
  (if (eq? ##sys#current-thread ##sys#primordial-thread)
      (break)
      (##sys#setslot ##sys#primordial-thread 1 break) ) )


;;; Default handlers:

(define ##sys#error-handler
  (make-parameter
   (let ([string-append string-append]
	 [open-output-string open-output-string]
	 [get-output-string get-output-string] )
     (lambda (msg . args)
       (##sys#error-handler (lambda args (##core#inline "C_halt" "error in error")))
       (cond ((##sys#fudge 4)
	      (##core#inline "C_display_string" ##sys#standard-error "Error: ")
	      (##sys#print msg #f ##sys#standard-error)
	      (cond [(fx= 1 (length args))
		     (##core#inline "C_display_string" ##sys#standard-error ": ")
		     (##sys#print (##sys#slot args 0) #t ##sys#standard-error) ]
		    [else
		     (##sys#for-each
		      (lambda (x)
			(##core#inline "C_display_char" ##sys#standard-error #\newline)
			(##sys#print x #t ##sys#standard-error) )
		      args) ] )
	      (##core#inline "C_display_char" ##sys#standard-error #\newline)
	      (##core#inline "C_halt" #f) )
	     (else
	      (let ((out (open-output-string)))
		(##sys#print msg #f out)
		(##sys#print #\newline #f out)
		(##sys#for-each (lambda (x) (##sys#print x #t out) (##sys#print #\newline #f out)) args)
		(##core#inline "C_halt" (get-output-string out)) ) ) ) ) ) ) )

(define-foreign-variable _ex_software int "EX_SOFTWARE")

(define reset-handler 
  (make-parameter 
   (lambda ()
     ((##sys#exit-handler) _ex_software)) ) )

(define exit-handler
  (make-parameter
   (lambda code
     (##sys#cleanup-before-exit)
     (##core#inline
      "C_exit_runtime"
      (if (null? code)
	  0
	  (let ([code (car code)])
	    (##sys#check-exact code)
	    code) ) ) ) ) )

(define implicit-exit-handler
  (make-parameter
   (lambda ()
     (##sys#cleanup-before-exit)
     (when ##sys#entry-points-defined-flag
       (##sys#dispatch-to-entry-point) ) ) ) )

(define ##sys#exit-handler exit-handler)
(define ##sys#reset-handler reset-handler)
(define ##sys#implicit-exit-handler implicit-exit-handler)

(define force-finalizers (make-parameter #t))

(define ##sys#cleanup-before-exit
  (let ([ffp force-finalizers])
    (lambda ()
      (when (ffp) (##sys#force-finalizers)) ) ) )


;;; Condition handling:

(define ##sys#signal-hook 
  (lambda (mode msg . args)
    (case mode
      [(#:user-interrupt)
       (##sys#abort (##sys#make-structure 'condition '(user-interrupt) '())) ]
      [(#:warning)
       (##sys#print "Warning: " #f ##sys#standard-error)
       (##sys#print msg #f ##sys#standard-error)
       (##sys#write-char #\newline ##sys#standard-error)
       (for-each
	(lambda (x)
	  (##sys#print x #t ##sys#standard-error)
	  (##sys#write-char #\newline ##sys#standard-error) )
	args) ] 
      [else
       (when (and (symbol? msg) (null? args))
	 (set! msg (##sys#symbol->string msg)) )
       (let* ([hasloc (or (not msg) (symbol? msg))]
	      [loc (and hasloc msg)]
	      [msg (if hasloc (##sys#slot args 0) msg)]
	      [args (if hasloc (##sys#slot args 1) args)] )
	 (##sys#abort
	  (##sys#make-structure
	   'condition 
	   (case mode
	     [(#:type-error) '(exn type)]
	     [(#:bounds-error) '(exn bounds)]
	     [(#:arithmetic-error) '(exn arithmetic)]
	     [(#:file-error) '(exn i/o file)]
	     [(#:runtime-error) '(exn runtime)]
	     [(#:network-error) '(exn i/o net)]
	     [else '(exn)] )
	   (list 'message msg
		 'arguments args
		 'location loc) ) ) ) ] ) ) )

(define (##sys#abort x)
  (##sys#current-exception-handler x)
  (##sys#abort (make-property-condition 'exn 'message "exception handler returned")) )

(define (##sys#signal x)
  (##sys#current-exception-handler x) )

(define abort ##sys#abort)
(define signal ##sys#signal)

(define ##sys#current-exception-handler
  ;; Exception-handler for the primordial thread:
  (let ([string-append string-append])
    (lambda (c)
      (when (##sys#structure? c 'condition)
	(let ([kinds (##sys#slot c 1)])
	  (cond [(memq 'exn kinds)
		 (let* ([props (##sys#slot c 2)]
			[msga (memq 'message props)]
			[argsa (memq 'arguments props)]
			[loca (memq 'location props)] )
		   (apply
		    (##sys#error-handler)
		    (if msga
			(let ([msg (cadr msga)]
			      [loc (and loca (cadr loca))] )
			  (if loc
			      (string-append "(" (##sys#symbol->qualified-string loc) ") " msg)
			      msg) )
			"<exn: has no `message' property>")
		    (if argsa
			(cadr argsa)
			'() ) )
		   ((##sys#reset-handler)) ) ]
		[(eq? 'user-interrupt (##sys#slot kinds 0))
		 (##sys#print "*** User interrupt ***\n" #f ##sys#standard-error)
		 ((##sys#reset-handler)) ] 
		[(eq? 'uncaught-exception (##sys#slot kinds 0))
		 ((##sys#error-handler)
		  "uncaught exception"
		  (cadr (memq 'reason (##sys#slot c 2))) )
		 ((##sys#reset-handler)) ] ) ) )
      (##sys#abort
       (##sys#make-structure
	'condition 
	'(uncaught-exception) 
	(list 'reason c)) ) ) ) )

(define (with-exception-handler handler thunk)
  (let ([oldh ##sys#current-exception-handler])
    (##sys#dynamic-wind 
	(lambda () (set! ##sys#current-exception-handler handler))
	thunk
	(lambda () (set! ##sys#current-exception-handler oldh)) ) ) )

(define (current-exception-handler) ##sys#current-exception-handler)

(define (make-property-condition kind . props)
  (##sys#make-structure 'condition (list kind) props) )

(define make-composite-condition
  (lambda (c1 . conds)
    (let ([conds (cons c1 conds)])
      (for-each (lambda (c) (##sys#check-structure c 'condition 'make-composite-condition)) conds)
      (##sys#make-structure
       'condition
       (apply ##sys#append (map (lambda (c) (##sys#slot c 1)) conds))
       (apply ##sys#append (map (lambda (c) (##sys#slot c 2)) conds)) ) ) ) )

(define (condition? x) (##sys#structure? x 'condition))

(define (condition-predicate kind)
  (lambda (c) 
    (##sys#check-structure c 'condition)
    (if (memv kind (##sys#slot c 1)) #t #f) ) )

(define (condition-property-accessor kind prop)
  (lambda (c)
    (##sys#check-structure c 'condition)
    (and (memv kind (##sys#slot c 1))
	 (let ([a (memq prop (##sys#slot c 2))])
	   (if a
	       (cadr a)
	       (##sys#signal-hook 
		#:type-error 'condition-property-accessor
		"condition has no such property" prop) ) ) ) ) )


;;; Error hook (called by runtime-system):

(define ##sys#error-hook
  (lambda (code loc . args)
    (case code
      ((1) (apply ##sys#error loc "bad argument count" args))
      ((2) (apply ##sys#error loc "too few arguments" args))
      ((3) (apply ##sys#signal-hook #:type-error loc "bad argument type" args))
      ((4) (apply ##sys#error loc "unbound variable" args))
      ((5) (apply ##sys#error loc "parameter limit exceeded" args))
      ((6) (apply ##sys#signal-hook #:runtime-error loc "out of memory" args))
      ((7) (apply ##sys#signal-hook #:arithmetic-error loc "division by zero" args))
      ((8) (apply ##sys#signal-hook #:bounds-error loc "out of range" args))
      ((9) (apply ##sys#signal-hook #:type-error loc "call of non-procedure" args))
      ((10) (apply ##sys#error loc "continuation can not receive multiple values" args))
      ((12) (apply ##sys#signal-hook #:runtime-error loc "recursion too deep" args))
      ((13) (apply ##sys#signal-hook #:type-error loc "inexact number can not be represented as an exact number" args))
      ((14) (apply ##sys#signal-hook #:type-error loc "argument is not a proper list" args))
      ((15) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a fixnum" args))
      ((16) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a number" args))
      ((17) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a string" args))
      ((18) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a pair" args))
      ((19) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a list" args))
      ((20) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a character" args))
      ((21) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a vector" args))
      ((22) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a symbol" args))
      ((23) (apply ##sys#signal-hook #:runtime-error loc "stack overflow" args))
      ((24) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a structure of the required type" args))
      ((25) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a bytevector" args))
      ((26) (apply ##sys#signal-hook #:type-error loc "locative refers to reclaimed object" args))
      ((27) (apply ##sys#signal-hook #:type-error loc "bad argument type - non-immediate value expected" args))
      ((28) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a number-vector or not one of the right type" args))
      ((29) (apply ##sys#signal-hook #:type-error loc "bad argument type - not an integer" args))
      ((30) (apply ##sys#signal-hook #:type-error loc "bad argument type - not an unsigned integer" args))
      ((31) (apply ##sys#signal-hook #:type-error loc "bad argument type - not a pointer" args))
      (else (apply ##sys#signal-hook #:runtime-error loc "unknown internal error" args)) ) ) )


;;; Miscellaneous low-level routines:

(define (##sys#structure? x s) (##core#inline "C_i_structurep" x s))
(define (##sys#generic-structure? x) (##core#inline "C_structurep" x))
(define (##sys#slot x i) (##core#inline "C_slot" x i))
(define (##sys#size x) (##core#inline "C_block_size" x))
(define ##sys#make-pointer (##core#primitive "C_make_pointer"))
(define ##sys#make-tagged-pointer (##core#primitive "C_make_tagged_pointer"))
(define (##sys#pointer? x) (##core#inline "C_pointerp" x))
(define (##sys#set-pointer-address! ptr addr) (##core#inline "C_update_pointer" addr ptr))
(define (##sys#bytevector? x) (##core#inline "C_bytevectorp" x))
(define (##sys#string->pbytevector s) (##core#inline "C_string_to_pbytevector" s))
(define (##sys#permanent? x) (##core#inline "C_permanentp" x))
(define (##sys#block-address x) (##core#inline "C_block_address" x))
(define (##sys#locative? x) (##core#inline "C_locativep" x))

(define (##sys#null-pointer)
  (let ([ptr (##sys#make-pointer)])
    (##core#inline "C_update_pointer" 0 ptr)
    ptr) )

(define (##sys#null-pointer? x)
  (eq? 0 (##sys#pointer->address x)) )

(define (##sys#address->pointer addr)
  (let ([ptr (##sys#make-pointer)])
    (##core#inline "C_update_pointer" addr ptr)
    ptr) )

(define (##sys#pointer->address ptr)
  ;; *** '4' is platform dependent!
  (##core#inline_allocate ("C_a_int_to_num" 4) (##sys#slot ptr 0)) )

(define ##sys#make-c-string 
  (let ((string-append string-append)
	(string string) )
    (lambda (str)
      (string-append str (string (##core#inline "C_make_character" (##core#inline "C_unfix" 0)))) ) ) )

(define ##sys#peek-signed-integer (##core#primitive "C_peek_signed_integer"))
(define ##sys#peek-unsigned-integer (##core#primitive "C_peek_unsigned_integer"))
(define (##sys#peek-fixnum b i) (##core#inline "C_peek_fixnum" b i))
(define (##sys#peek-byte ptr i) (##core#inline "C_peek_byte" ptr i))

(define (##sys#vector->structure! vec) (##core#inline "C_vector_to_structure" vec))

(define (##sys#peek-double b i)
  (##core#inline "C_f64peek" b i)
  (##sys#cons-flonum) )

(define ##sys#peek-c-string
    (lambda (b i)
      (and (not (##sys#null-pointer? b))
	   (let* ([len (##core#inline "C_fetch_c_strlen" b i)]
		  [str2 (##sys#make-string len)] )
	     (##core#inline "C_peek_c_string" b i str2 len)
	     str2) ) ) )

(define ##sys#peek-nonnull-c-string
    (lambda (b i)
      (let* ([len (##core#inline "C_fetch_c_strlen" b i)]
	     [str2 (##sys#make-string len)] )
	(##core#inline "C_peek_c_string" b i str2 len)
	str2) ) )

(define ##sys#peek-and-free-c-string
    (lambda (b i)
      (and (not (##sys#null-pointer? b))
	   (let* ([len (##core#inline "C_fetch_c_strlen" b i)]
		  [str2 (##sys#make-string len)] )
	     (##core#inline "C_peek_c_string" b i str2 len)
	     (##core#inline "C_free_mptr" b i)
	     str2) ) ) )

(define ##sys#peek-and-free-nonnull-c-string
    (lambda (b i)
      (let* ([len (##core#inline "C_fetch_c_strlen" b i)]
	     [str2 (##sys#make-string len)] )
	(##core#inline "C_peek_c_string" b i str2 len)
        (##core#inline "C_free_mptr" b i)
	str2) ) )

(define (##sys#poke-c-string b i s) 
  (##core#inline "C_poke_c_string" b i (##sys#make-c-string s)) )

(define (##sys#poke-integer b i n) (##core#inline "C_poke_integer" b i n))

(define (##sys#vector->closure! vec addr)
  (##core#inline "C_vector_to_closure" vec)
  (##core#inline "C_update_pointer" addr vec) )

(define (##sys#symbol-has-toplevel-binding? s)
  (not (eq? (##sys#slot s 0) (##sys#slot '##sys#arbitrary-unbound-symbol 0))) )

(define (##sys#copy-bytes from to offset1 offset2 bytes)
  (##core#inline 
   "C_substring_copy"
   from to
   offset1 (fx+ offset1 bytes)
   offset2) )

(define (##sys#copy-words from to offset1 offset2 words)
  (##core#inline 
   "C_subvector_copy"
   from to
   offset1 (fx+ offset1 words)
   offset2) )

(define (##sys#compare-bytes from to offset1 offset2 bytes)
  (##core#inline 
   "C_substring_compare"
   from to
   offset1 offset2 bytes) )

(define ##sys#zap-strings (foreign-lambda void "C_zap_strings" scheme-object))

(define (##sys#block-pointer x)
  (let ([ptr (##sys#make-pointer)])
    (##core#inline "C_pointer_to_block" ptr x)
    ptr) )


;;; Support routines for foreign-function calling:

(define (##sys#foreign-char-argument x) (##core#inline "C_i_foreign_char_argumentp" x))
(define (##sys#foreign-fixnum-argument x) (##core#inline "C_i_foreign_fixnum_argumentp" x))
(define (##sys#foreign-flonum-argument x) (##core#inline "C_i_foreign_flonum_argumentp" x))
(define (##sys#foreign-block-argument x) (##core#inline "C_i_foreign_block_argumentp" x))
(define (##sys#foreign-number-vector-argument t x) (##core#inline "C_i_foreign_number_vector_argumentp" t x))
(define (##sys#foreign-string-argument x) (##core#inline "C_i_foreign_string_argumentp" x))
(define (##sys#foreign-pointer-argument x) (##core#inline "C_i_foreign_pointer_argumentp" x))
(define (##sys#foreign-integer-argument x) (##core#inline "C_i_foreign_integer_argumentp" x))
(define (##sys#foreign-unsigned-integer-argument x) (##core#inline "C_i_foreign_unsigned_integer_argumentp" x))

(define ##sys#host-data-string
    (lambda (hd)
      (let* ([len (##core#inline "C_block_size" hd)]
	     [s2 (##sys#make-string len)] )
	(##core#inline "C_copy_memory" s2 hd len)
	s2) ) )


;;; Low-level threading interface:

(define ##sys#default-thread-quantum 10000)

(define (##sys#default-exception-handler arg) 
  (##core#inline "C_halt" "internal error: default exception handler shouldn't be called!") )

(define (##sys#make-thread thunk state name q)
  (##sys#make-structure
   'thread
   thunk				; thunk
   #f					; result
   state				; state
   #f					; block-timeout
   (vector				; state buffer
    ##sys#dynamic-winds
    ##sys#standard-input
    ##sys#standard-output
    ##sys#standard-error
    ##sys#default-exception-handler
    (##sys#grow-vector ##sys#current-parameter-vector (##sys#size ##sys#current-parameter-vector) #f) )
   name					; name
   (##core#undefined)			; end-exception
   '()					; owned mutexes
   q					; quantum
   (##core#undefined)			; specific
   #f					; block-thread
   '() ) )				; recipients

(define ##sys#primordial-thread (##sys#make-thread #f 'running 'primordial ##sys#default-thread-quantum))
(define ##sys#current-thread ##sys#primordial-thread)

(define (##sys#make-mutex id owner)
  (##sys#make-structure
   'mutex
   id					; name
   owner				; thread or #f
   '()					; list of waiting threads
   #f					; abandoned
   #f					; locked
   (##core#undefined) ) )		; specific

(define (##sys#abandon-mutexes thread)
  (let ([ms (##sys#slot thread 8)])
    (unless (null? ms)
      (##sys#for-each
       (lambda (m)
	 (##sys#setislot m 2 #f)
	 (##sys#setislot m 4 #t) 
	 (##sys#setislot m 5 #f)
	 (##sys#setislot m 3 '()) )
       ms) ) ) )

(define (##sys#schedule) ((##sys#slot ##sys#current-thread 1)))


;;; Interrupt-handling:

(define ##sys#context-switch (##core#primitive "C_context_switch"))

(define (##sys#interrupt-hook reason state)
  (if (fx> (##sys#slot ##sys#pending-finalizers 0) 0)
      (##sys#run-pending-finalizers state)
      (##sys#context-switch state) ) )


;;; Entry point dispatching:

(define ##sys#entry-points-defined-flag #f)

(define ##sys#undefined-entry-point
    (lambda (id buffer)
      (##sys#error "undefined entry point" id) ) )

(define ##sys#entry-point-dispatch-table (##sys#make-vector 8 ##sys#undefined-entry-point))

(define ##sys#register-entry-point
  (let ([vector-copy! vector-copy!])
    (lambda (id handler)
      (when (fx> id (##sys#size ##sys#entry-point-dispatch-table))
	(let ([epdt2 (##sys#make-vector (fx+ id 8) ##sys#undefined-entry-point)])
	  (vector-copy! ##sys#entry-point-dispatch-table epdt2)
	  (set! ##sys#entry-point-dispatch-table epdt2) ) )
      (set! ##sys#entry-points-defined-flag #t)
      (##sys#setslot ##sys#entry-point-dispatch-table id handler) ) ) )

(define ##sys#dispatch-to-entry-point
  (lambda ()
    (let loop ()
      (let ([idbuf (##sys#host-data 0)])
	(when idbuf
	  (let ([id (##sys#peek-unsigned-integer idbuf 0)]
		[data (##sys#host-data 1)] )
	    ((vector-ref ##sys#entry-point-dispatch-table id) id data)
	    (##sys#set-host-data! 1 data) ) )
	(##sys#call-host) 
	(loop) ) ) ) )


;;; Accessing "errno":

(define-foreign-variable ##sys#errno int "errno")

(let ([rn 0])
  (set! ##sys#update-errno (lambda () (set! rn ##sys#errno)))
  (set! errno (lambda () rn)) )


;;; Special string quoting syntax:

(set! ##sys#user-read-hook
  (let ([old ##sys#user-read-hook]
	[read-char read-char]
	[peek-char peek-char] 
	[write-char write-char]
	[open-output-string open-output-string]
	[get-output-string get-output-string] 
	[reverse reverse]
	[read read]
	[display display] )
    (define (readln port)
      (let ([ln (open-output-string)])
	(do ([c (read-char port) (read-char port)])
	    ((or (eof-object? c) (char=? #\newline c))
	     (cond [(char? c)
		    (set! ##sys#read-line-counter (fx+ ##sys#read-line-counter 1))
		    (get-output-string ln) ]
		   [else c] ) )
	  (write-char c ln) ) ) )
    (define (err) (##sys#error 'read "unexpected end of file - unterminated string literal"))
    (define (fetch str lst)
      (let ([s (get-output-string str)])
	(if (fx= 0 (##sys#size s))
	    lst
	    (cons s lst) ) ) )
    (lambda (char port)
      (cond [(not (char=? #\< char)) (old char port)]
	    [else
	     (read-char port)
	     (case (peek-char port)
	       [(#\<)
		(read-char port)
		(let ([str (open-output-string)]
		      [end (readln port)] 
		      [f #f] )
		  (do ([ln (readln port) (readln port)])
		      ((string=? end ln) (get-output-string str))
		    (when (eof-object? ln) (err))
		    (if f 
			(write-char #\newline str)
			(set! f #t) )
		    (display ln str) ) ) ]
	       [(#\#)
		(read-char port)
		(let ([lst '()]
		      [end (readln port)] 
		      [str (open-output-string)] )
		  (let loop ()
		    (let ([c (read-char port)])
		      (when (eof-object? c) (err))
		      (case c
			[(#\newline)
			 (let ([s (get-output-string str)])
			   (set! str (open-output-string))
			   (cond [(string=? end s) `(##sys#print-to-string ,@(reverse (cdr lst)))] ; drop last newline
				 [else
				  (set! lst (cons #\newline (cons s lst)))
				  (loop) ] ) ) ]
			[(#\#)
			 (let ([c (peek-char port)])
			   (case c
			     [(#\#)
			      (write-char (read-char port) str)
			      (loop) ]
			     [(#\{)
			      (read-char port)
			      (set! lst (cons (read port) (fetch str lst)))
			      (set! str (open-output-string))
			      (let loop2 ()
				(let ([c (read-char port)])
				  (cond [(eof-object? c) (err)]
					[(char=? #\} c) (loop)]
					[else (loop2)] ) ) ) ]
			     [else
			      (set! lst (cons (read port) (fetch str lst)))
			      (set! str (open-output-string))
			      (loop) ] ) ) ]
			[else
			 (write-char c str)
			 (loop) ] ) ) ) ) ]
	       [else (##sys#error 'read "unreadable object")] ) ] ) ) ) )


;;; Script invocation:

(define ##sys#script-main
  (let ([argv argv]
	[list-tail list-tail] )
    (lambda (prgi n main)
      (let* ([av (argv)]
	     [args (list-tail av n)] )
	((##sys#exit-handler) 
	 (let ([r (main args)])
	   (if (fixnum? r)
	       r
	       _ex_software) ) ) ) ) ) )

(define command-line-arguments
  (make-parameter
   (let ([args (argv)])
     (if (pair? args)
	 (##sys#slot args 1) 
	 args) )
   (lambda (x)
     (##sys#check-list x)
     x) ) )


;;; Finalization:

(define-foreign-variable _max_pending_finalizers int "C_MAX_PENDING_FINALIZERS")

(define ##sys#pending-finalizers 
  (##sys#make-vector (fx+ (fx* 2 _max_pending_finalizers) 1) (##core#undefined)) )

(##sys#setislot ##sys#pending-finalizers 0 0)

(define set-finalizer! (##core#primitive "C_register_finalizer"))

(define ##sys#run-pending-finalizers
  (let ([vector-fill! vector-fill!]
	[working #f] )
    (lambda (state)
      (unless working
	(set! working #t)
	(let* ([n (##sys#size ##sys#pending-finalizers)]
	       [c (##sys#slot ##sys#pending-finalizers 0)] )
	  (do ([i 0 (fx+ i 1)])
	      ((fx>= i c))
	    (let ([i2 (fx+ 1 (fx* i 2))])
	      ((##sys#slot ##sys#pending-finalizers (fx+ i2 1)) (##sys#slot ##sys#pending-finalizers i2)) ) )
	  (vector-fill! ##sys#pending-finalizers (##core#undefined))
	  (##sys#setislot ##sys#pending-finalizers 0 0) 
	  (set! working #f) ) )
      (when state (##sys#context-switch state) ) ) ) )

(define (##sys#force-finalizers)
  (let loop ()
    (let ([n (##sys#gc)])
      (if (fx> (##sys#slot ##sys#pending-finalizers 0) 0)
	  (begin
	    (##sys#run-pending-finalizers #f)
	    (loop) )
	  n) ) ) )

(define (gc . arg)
  (let ([a (and (pair? arg) (car arg))])
    (if a
	(##sys#force-finalizers)
	(apply ##sys#gc arg) ) ) )


;;; Auxilliary definitions for safe use in quasiquoted forms and evaluated code:

(define ##sys#list->vector list->vector)
(define ##sys#list list)
(define ##sys#cons cons)
(define ##sys#append append)
(define ##sys#vector vector)
(define ##sys#apply apply)
(define ##sys#values values)


;;; Promises:

(define ##sys#make-promise
    (lambda (proc)
      (let ([result-ready #f]
	    [results #f] )
	(##sys#make-structure
	 'promise
	 (lambda ()
	   (if result-ready
	       (apply ##sys#values results)
	       (##sys#call-with-values 
		proc
		(lambda xs
		  (if result-ready
		      (apply ##sys#values results)
		      (begin
			(set! result-ready #t)
			(set! results xs)
			(apply ##sys#values results) ) ) ) ) ) ) ) ) ) )


;;; SRFI-9 support code:

(define (##sys#make-record-type name field-tags)
  (##sys#make-structure 'record-type name field-tags) )

(define (##sys#record-field-index type tag)
  (let loop ((i 2) (tags (##sys#slot type 2)))
    (cond ((null? tags)
           (##sys#error "record type has no such field" (##sys#slot type 1) tag))
          ((eq? tag (##sys#slot tags 0))
           i)
          (else
           (loop (fx+ i 1) (##sys#slot tags 1))))))

(define (##sys#record-constructor type tags)
  (let ((size (length (##sys#slot type 2)))
        (arg-count (length tags))
        (indexes (map (lambda (tag) (##sys#record-field-index type tag)) tags)))
    (lambda args
      (if (fx= (length args) arg-count)
          (let ((new (make-vector (fx+ size 2))))
	    (##sys#vector->structure! new)
	    (##sys#setslot new 0 'record)
            (##sys#setslot new 1 type)
            (for-each (lambda (arg i) (##sys#setslot new i arg))
                      args
                      indexes)
            new)
          (##sys#error "wrong number of arguments to constructor" type args)))))

(define (##sys#record-predicate type)
  (lambda (thing)
    (and (##sys#structure? thing 'record)
         (eq? (##sys#slot thing 1) type))))

(define (##sys#record-accessor type tag)
  (let ((index (##sys#record-field-index type tag)))
    (lambda (thing)
      (if (and (##sys#structure? thing 'record) (eq? (##sys#slot thing 1) type))
	  (##sys#slot thing index)
          (##sys#signal-hook #:type-error "bad argument type - accessor applied to bad value" type tag thing)))))

(define (##sys#record-modifier type tag)
  (let ((index (##sys#record-field-index type tag)))
    (lambda (thing value)
      (if (and (##sys#structure? thing 'record) (eq? (##sys#slot thing 1) type))
          (##sys#setslot thing index value)
          (##sys#signal-hook #:type-error "bad argument type - modifier applied to bad value" type tag thing)))))


;;; andmap + ormap:

(define andmap
  (lambda (f first . rest)
    (cond ((null? rest)
	   (let mapf ((l first))
	     (or (null? l)
		 (and (f (car l)) (mapf (cdr l))))))
	  ((null? (cdr rest))
	   (let mapf ((l1 first) (l2 (car rest)))
	     (or (null? l1)
		 (and (f (car l1) (car l2)) (mapf (cdr l1) (cdr l2))))))
	  (else
	   (let mapf ((first first) (rest rest))
	     (or (null? first)
		 (and (apply f (car first) (map (lambda (x) (car x)) rest))
		      (mapf (cdr first) (map (lambda (x) (cdr x)) rest)))))))))

(define ormap
  (lambda (f first . rest)
    (if (null? first)
        (or)
	(let ([lists (cons first rest)])
	  (or (apply f (map (lambda (x) (car x)) lists))
	      (apply ormap f (map (lambda (x) (cdr x)) lists)) ) ) ) ) )


;;; Support code for macro libraries (match):

(define ##sys#match-error
  (let ([write-char write-char])
    (lambda (val . args)
      (##sys#print "\nFailed match:\n" #f ##sys#standard-error)
      (for-each
       (lambda (x)
	 (##sys#print x #t ##sys#standard-error)
	 (write-char #\newline ##sys#standard-error) )
       args)
      (##sys#error "no matching clause for " val))) )


;;; `error' expands into this (with extended-bindings):

(define ##sys#error-at
  (let ([string-append string-append])
    (lambda (where msg . args)
      (apply
       ##sys#signal-hook #:error 
       (string-append 
	where " " 
	(cond [(string? msg) msg]
	      [(symbol? msg) (##sys#symbol->string msg)]
	      [else ""] ) )
       args) ) ) )


;;; Internal string-reader:

(define ##sys#read-from-string 
  (let ([open-input-string open-input-string])
    (lambda (s)
      (let ([i (open-input-string s)])
	(read i) ) ) ) )


;;; Convenient error printing:

(define print-error-message
  (let* ([display display]
	 [newline newline] 
	 [write write]
	 [errmsg (condition-property-accessor 'exn 'message)]
	 [errargs (condition-property-accessor 'exn 'arguments)] 
	 [writeargs
	  (lambda (args port)
	    (##sys#for-each 
	     (lambda (x)
	       (##sys#with-print-length-limit 80 (lambda () (write x port)))
	       (newline port) )
	     args) ) ] )
    (lambda (ex . port)
      (let ([port (:optional port ##sys#standard-error)])
	(display "Error: " port)
	(cond [(and (not (##sys#immediate? ex)) (eq? 'condition (##sys#slot ex 0)))
	       (display (errmsg ex) port)
	       (let ([args (errargs ex)])
		 (if (fx= 1 (length args))
		     (begin
		       (display ": " port)
		       (writeargs args port) )
		     (begin
		       (newline port)
		       (writeargs args port) ) ) ) ]
	      [(string? ex)
	       (display ex port)
	       (newline port) ]
	      [else
	       (display "uncaught exception: " port)
	       (writeargs (list ex) port) ] ) ) ) ) )
