/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */
/** 
 * SECTION:ctk-effect-glow
 * @short_description: A glow effect.
 * @include: ctk-effect-glow.h
 *
 * #CtkEffectGlow is an implementation of #CtkEffect that causes the #CtkActor
 * it's attached to to glow.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <GL/glew.h>
#include <GL/glxew.h>

#include "ctk-gfx-private.h"
#include "ctk-arb-asm-private.h"
#include "ctk-actor.h"
#include "ctk-effect-context.h"
#include "ctk-effect-glow.h"
#include "ctk-private.h"
#include "ctk-actor.h"

G_DEFINE_TYPE (CtkEffectGlow, ctk_effect_glow, CTK_TYPE_EFFECT);

enum
{
  PROP_0,
  GLOW_FACTOR,
  GLOW_COLOR,
};

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), CTK_TYPE_EFFECT_GLOW, CtkEffectGlowPrivate))

struct _CtkEffectGlowPrivate
{
  gfloat factor;
  ClutterColor color;
  guint background_texture_id;
  guint background_texture_width;
  guint background_texture_height;

  /* opengl texture id use to cache the effect. The texture is updated when the actor is damaged */
  guint cached_effect_texture;
  gboolean cached_init;
};


/* Globals */

/* Forwards */
static void ctk_effect_glow_paint (CtkEffect          *effect,
                                   CtkEffectPaintFunc func,
                                   gboolean           is_last_effect);

static void ctk_effect_glow_paint_dummy (CtkEffect          *effect,
                                         CtkEffectPaintFunc  func,
                                         gboolean            is_last_effect);

static void ctk_effect_glow_set_property (GObject* gobject,
                                            guint prop,
                                            const GValue* value,
                                            GParamSpec* spec);

static void ctk_effect_glow_get_property (GObject* gobject,
                                            guint prop,
                                            GValue* value,
                                            GParamSpec* spec);
/*
 * GObject stuff
 */

static void
ctk_effect_glow_finalize (GObject *object)
{
  CtkEffectGlowPrivate *priv = GET_PRIVATE (CTK_EFFECT_GLOW (object));
  g_return_if_fail (CTK_IS_EFFECT_GLOW (object));

  g_return_if_fail (priv);

  if (priv->background_texture_id)
    {
      CHECKGL (glDeleteTextures (1, &(priv->background_texture_id)));
      priv->background_texture_id = 0;
      priv->background_texture_width = 0; 
      priv->background_texture_height = 0;
    }

  if (priv && priv->cached_effect_texture)
    {
      CHECKGL (glDeleteTextures (1, &priv->cached_effect_texture));
    }
  
  G_OBJECT_CLASS (ctk_effect_glow_parent_class)->finalize (object);
}

static void
ctk_effect_glow_class_init (CtkEffectGlowClass *klass)
{
  CtkEffectClass *eff_class = CTK_EFFECT_CLASS (klass);

  GObjectClass*   obj_class = G_OBJECT_CLASS (klass);
  GParamSpec*     prop_factor;

  if (ctk_has_capability (CTK_CAPABILITY_VERTEX_PROGRAM)
      && ctk_has_capability (CTK_CAPABILITY_FRAGMENT_PROGRAM)
      && ctk_asm_shaders_compiled_and_ready ())
    eff_class->paint = ctk_effect_glow_paint;
  else
    {
      if (!ctk_has_capability (CTK_CAPABILITY_FBO) || !ctk_has_capability (CTK_CAPABILITY_VERTEX_PROGRAM) || !ctk_has_capability (CTK_CAPABILITY_FRAGMENT_PROGRAM))
        {
          g_message ("WARNING: CtkEffectGlow cannot work without FBO and assembly shader program capabiltities");
        }
      else
        {
          if (!ctk_asm_shaders_compiled_and_ready ())
            g_message ("WARNING: Shaders programs are not ready");
        }
      eff_class->paint = ctk_effect_glow_paint_dummy;
    }

  obj_class->finalize     = ctk_effect_glow_finalize;
  obj_class->set_property = ctk_effect_glow_set_property;
  obj_class->get_property = ctk_effect_glow_get_property;

  prop_factor = g_param_spec_float ("factor",
                                 "factor",
                                 "glow intensity",
                                 CTK_EFFECT_GLOW_MIN_FACTOR,
                                 CTK_EFFECT_GLOW_MAX_FACTOR,
                                 CTK_EFFECT_GLOW_DEFAULT_FACTOR,
                                 G_PARAM_READWRITE);
  g_object_class_install_property (obj_class, GLOW_FACTOR, prop_factor);

  prop_factor = g_param_spec_boxed ("color",
                                 "Color",
                                 "glow color",
                                 CLUTTER_TYPE_COLOR,
                                 G_PARAM_READWRITE);
  
  g_object_class_install_property (obj_class, GLOW_COLOR, prop_factor);
  
  g_type_class_add_private (obj_class, sizeof (CtkEffectGlowPrivate));
}

static void
ctk_effect_glow_init (CtkEffectGlow *self)
{
  CtkEffectGlowPrivate *priv;
  priv = GET_PRIVATE (self);

  priv->factor      = 3.0f;
  priv->color.red   = 255;
  priv->color.green = 255;
  priv->color.blue  = 255;
  priv->color.alpha = 255;
  priv->background_texture_id = 0;
  priv->background_texture_width = 0;
  priv->background_texture_height = 0;

  priv->cached_effect_texture = 0;
  CHECKGL (glGenTextures (1, &priv->cached_effect_texture));
  CHECKGL (glActiveTextureARB(GL_TEXTURE0) );
  CHECKGL (glBindTexture (GL_TEXTURE_2D, priv->cached_effect_texture));
  CHECKGL (glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA8, 1, 1, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));

  priv->cached_init = FALSE;
}

static void ctk_effect_glow_set_property (GObject* gobject,
                                            guint prop,
                                            const GValue* value,
                                            GParamSpec* spec)
{
  CtkEffectGlow*        glow = NULL;
  CtkEffectGlowPrivate* priv = NULL;
 
  glow = CTK_EFFECT_GLOW (gobject);
  priv = GET_PRIVATE (glow);
 
  switch (prop)
  {
    case GLOW_FACTOR:
      {
        gfloat factor = g_value_get_float (value);
        ctk_effect_glow_set_factor (glow, factor);
      }
    break;

    case GLOW_COLOR:
      {
        ClutterColor *color = NULL;
        if ((color = g_value_get_boxed (value)))
          ctk_effect_glow_set_color (glow, color);
      }
    break;
    
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
    break;
  }
}

static void ctk_effect_glow_get_property (GObject* gobject,
                                            guint prop,
                                            GValue* value,
                                            GParamSpec* spec)
{
  CtkEffectGlow*        glow = NULL;
  CtkEffectGlowPrivate* priv = NULL;
 
  glow = CTK_EFFECT_GLOW (gobject);
  priv = GET_PRIVATE (glow);
 
  switch (prop)
  {
    case GLOW_FACTOR:
      {
        gfloat factor = ctk_effect_glow_get_factor (glow);
        g_value_set_float (value, factor);
      }
    break;

    case GLOW_COLOR:
      {
        ClutterColor color;
        ctk_effect_glow_get_color (glow, &color);
        g_value_set_boxed (value, &color);
      }
    break;
    
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
    break;
  }
}

/*
 * Private Methods
 */
static void
ctk_effect_glow_paint_dummy (CtkEffect          *effect,
                             CtkEffectPaintFunc  paint_func,
                             gboolean            is_last_effect)
{
  ClutterActor *actor = ctk_effect_get_actor (effect);

  paint_func (actor);
}

static void
ctk_effect_glow_paint (CtkEffect          *effect,
                       CtkEffectPaintFunc paint_func,
                       gboolean           is_last_effect)
{
  ClutterActor*               actor        = NULL;
  CtkEffectContext*           fxctx        = NULL;
  ClutterActor*               stage        = NULL;
  CtkRenderTarget*            top_rt       = NULL;
  CtkRenderTarget*            rt0          = NULL;
  CtkRenderTarget*            rt1          = NULL;
  CtkRenderTarget*            final_rt     = NULL;
  CtkEffectGlowPrivate*       priv         = NULL;
  gfloat                      stage_width  = 0.0f;
  gfloat                      stage_height = 0.0f;
  gfloat actor_screen_width, actor_screen_height;
  gfloat actor_screen_x, actor_screen_y;
  /* Make the rt a little bit bigger than the actor size. */
  gint margin = 0;
  /* The number of additional passes for the blur algorithm */
  gint num_passes = 0;
  
  ClutterVertex vtx[4];
  float actor_opacity;
  float effect_opacity;
  gboolean UseCache = FALSE;
  gboolean dirty_effect_cache = TRUE;
  
  actor = ctk_effect_get_actor (effect);
  g_return_if_fail (CLUTTER_IS_ACTOR (actor));

  priv = GET_PRIVATE (CTK_EFFECT_GLOW (effect));
  margin = ctk_effect_get_margin (CTK_EFFECT (effect));
  num_passes = ctk_effect_get_strength (CTK_EFFECT (effect));
  dirty_effect_cache = ctk_effect_is_effect_cache_dirty (CTK_EFFECT (effect));
  
  /* before rendering an effect, render everything that has bee cahched so far */
  cogl_flush ();

  /* get effect-context to which this actor belongs */
  fxctx = ctk_effect_context_get_default_for_actor (actor);
  
  stage = clutter_actor_get_stage (actor);
  clutter_actor_get_size (CLUTTER_ACTOR (stage), &stage_width, &stage_height);

  ctk_get_actor_screen_position (CTK_ACTOR(actor),
      &actor_screen_x,
      &actor_screen_y,
      &actor_screen_width,
      &actor_screen_height, vtx);

  UseCache = TRUE; //ctk_actor_get_damaged (CTK_ACTOR (actor) );
  /* Get the size and position of the actor on screen */
  if ((UseCache == FALSE) || (dirty_effect_cache == TRUE))
    {
      /* reserve 2 render targets */
      rt0 = ctk_effect_context_grab_render_target (fxctx);
      rt1 = ctk_effect_context_grab_render_target (fxctx);

      // make sure the render target has a color and depth buffer
      ctk_render_target_resize (rt0, stage_width, stage_height);
      
      ctk_render_target_unbind ();
    
      ctk_effect_context_push_render_target (fxctx, rt0);

      /* clear render target and prepare the surface */
      {
        CHECKGL (glViewport (0, 0, stage_width, stage_height));
        CHECKGL (glScissor (0, 0, stage_width, stage_height));
        CHECKGL (glDisable (GL_SCISSOR_TEST));
        CHECKGL (glDisable (GL_ALPHA_TEST));
        CHECKGL (glDisable (GL_STENCIL_TEST));


        /* make sure we are not using a shader program */
        CHECKGL (glClearDepth (1.0f));
        CHECKGL (glClearStencil (0));
        CHECKGL (glClearColor (0.0f, 0.0f, 0.0f, 0.0f));
        CHECKGL (glColorMask (GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE));
    
        /* clear the FBO content */
        CHECKGL (glClear (GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT));
      }

      /* Paint the actor and its children. If ctk_actor_set_effects_painting has been set to true
        for this actor, it will draw itself without any effect (avoiding an infinite loop). */
      paint_func (actor);
      cogl_flush ();

      /* Deactivate blending state set by clutter */
      glDisable(GL_BLEND); 

      if (rt0 != ctk_effect_context_pop_render_target(fxctx))
        {
          /* This is to make sure that all the right push and pop have been while drawing
          the children of this actor.*/
          g_warning("%s() at line %s: Invalid push/pop detected", G_STRFUNC, G_STRLOC);
        }
      
      /* Copy from a stage sized render target to an actor sized render target */
      {
        float u0 = actor_screen_x/(float)ctk_render_target_get_width(rt0);
        float v1 = ((float)ctk_render_target_get_height(rt0) - actor_screen_y)/(float)ctk_render_target_get_height(rt0);
        float u1 = (actor_screen_x+actor_screen_width)/(float)ctk_render_target_get_width(rt0);
        float v0 = ((float)ctk_render_target_get_height(rt0) - (actor_screen_y + actor_screen_height))/(float)ctk_render_target_get_height(rt0);
      
        // Make the rt a little bit bigger than the actor size.
        ctk_render_target_resize(rt1, actor_screen_width+2*margin, actor_screen_height+2*margin);
        
        ctk_render_target_bind(rt1);
        CHECKGL( glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT) );

        ctk_copy_rendertarget_to_rendertarget_asm (
          rt0,
          u0,
          v0,
          u1,
          v1,
          rt1, margin, margin, actor_screen_width, actor_screen_height);
    
        actor_screen_x -= margin;
        actor_screen_y -= margin;
        actor_screen_width += 2*margin;
        actor_screen_height += 2*margin;
        ctk_render_target_resize(rt0, actor_screen_width, actor_screen_height);
      }

      {
        int iPass = 0;
        ctk_render_target_bind (rt0);
        CHECKGL( glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT) );
        ctk_render_quad_alpha_mask_asm (
              ctk_render_target_get_color_buffer_ogl_id(rt1),
              ctk_render_target_get_width(rt1),
              ctk_render_target_get_height(rt1),
              1.0f,
              1.0f,
              1.0f,
              1.0f,
              actor_screen_width, actor_screen_height, 0, 0, actor_screen_width, actor_screen_height);

        for (iPass = 0; iPass < num_passes; iPass++)
          {
            ctk_render_target_bind(rt1);
            CHECKGL( glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT) );
              opengl_blur_pass (ctk_render_target_get_color_buffer_ogl_id(rt0),
              ctk_render_target_get_width(rt0),
              ctk_render_target_get_height(rt0),
                      0.0f,
                      TRUE,
                      actor_screen_width,
                      actor_screen_height,
              FALSE);

            ctk_render_target_bind(rt0);
            CHECKGL( glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT) );
            opengl_blur_pass (ctk_render_target_get_color_buffer_ogl_id(rt1),
              ctk_render_target_get_width(rt1),
              ctk_render_target_get_height(rt1),
    				  0.0f,
    				  FALSE,
    				  actor_screen_width,
    				  actor_screen_height,
              FALSE);
    		  }

          /*ctk_render_target_bind(rt1);
          CHECKGL( glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT) );          
          ctk_render_texture_exponent_asm(
            ctk_render_target_get_color_buffer_ogl_id(rt0),
            ctk_render_target_get_width(rt0),
            ctk_render_target_get_height(rt0),
            ctk_render_target_get_width(rt1),
            ctk_render_target_get_height(rt1),
		  	    0,
			      0,
            ctk_render_target_get_width(rt0),
            ctk_render_target_get_height(rt0),
            0.4f);*/

        final_rt = rt0;
      }
      /* Save the result in the cache texture */
      {
        ctk_copy_render_target_to_cached_texture_asm(fxctx, final_rt, priv->cached_effect_texture);
        ctk_effect_set_invalidate_effect_cache (CTK_EFFECT (effect), FALSE);
      }

      glEnable(GL_BLEND);
    } /* end damaged */
    
  /* Now draw into the top render target on the stack */
  top_rt = ctk_effect_context_peek_render_target (fxctx);
  if(top_rt)
    {
      ctk_render_target_bind(top_rt);
      CHECKGL( glViewport(0, 0, ctk_render_target_get_width(top_rt), ctk_render_target_get_height(top_rt)) );
      CHECKGL( glScissor(0, 0, ctk_render_target_get_width(top_rt), ctk_render_target_get_height(top_rt)) );
    }
  else
    {
      /* There is no render target on the stack. Set back the regular frame buffer */
      ctk_render_target_unbind(); /* unbind whatever render target is binded */
      CHECKGL( glViewport(0, 0, stage_width, stage_height) );
      CHECKGL( glScissor(0, 0, stage_width, stage_height) );
    }

  actor_opacity = (float) clutter_actor_get_paint_opacity (CLUTTER_ACTOR (actor)) / 255.0f;
  effect_opacity = (float) ctk_effect_get_opacity (effect);
  {
    if (!UseCache)
      {
        ctk_render_quad_asm (
          priv->cached_effect_texture,
          actor_screen_width,
          actor_screen_height,
          g_shTexture_asm,
          stage_width,
          stage_height,
          actor_screen_x,
          actor_screen_y,
          actor_screen_width,
          actor_screen_height,
          (gfloat)priv->color.red/255.0f,
          (gfloat)priv->color.green/255.0f,
          (gfloat)priv->color.blue/255.0f,
          actor_opacity*effect_opacity);
      }
    else
      {
        /* this does not work if the actor is not screen aligned (ie rotated) */

        vtx[0].x += - margin;
        vtx[0].y += - margin;
        vtx[1].x += - margin;
        vtx[1].y += + margin;
        vtx[2].x += + margin;
        vtx[2].y += + margin;
        vtx[3].x += + margin;
        vtx[3].y += - margin;
        
        custom_render_quad_texture_mask_asm(
          priv->cached_effect_texture,
          priv->background_texture_id,
          priv->background_texture_width,
          priv->background_texture_height,
          g_shTextureMask_asm,
          stage_width,
          stage_height,
          vtx[0].x, vtx[0].y,
          vtx[3].x - vtx[0].x, vtx[1].y - vtx[0].y,
          actor_opacity*effect_opacity);
          
        ctk_render_custom_quad_asm (
          priv->cached_effect_texture,
          actor_screen_width,
          actor_screen_height,
          g_shTexture_asm,
          stage_width,
          stage_height,
          vtx,
          (gfloat)priv->color.red/255.0f,
          (gfloat)priv->color.green/255.0f,
          (gfloat)priv->color.blue/255.0f,
          actor_opacity*effect_opacity);
      }
  }
      
  /* render the actor on top of the glow */
  if (is_last_effect)
    {
      paint_func (actor);
      cogl_flush ();
    }

  /* leave and make sure the top render target on the stack is binded */
  if (top_rt)
    ctk_render_target_bind (top_rt);
  else
    ctk_render_target_unbind ();

  /* We cannot test ctk_actor_get_damaged anymore as it may have been changed during the last call to paint_func */
  if (rt0)
    {
      ctk_effect_context_release_render_target (fxctx, rt0);
    }
  if (rt1)
    {
      ctk_effect_context_release_render_target (fxctx, rt1);
    }
}

/*
 * Public Methods
 */

/**
 * ctk_effect_glow_new
 *
 * Returns: A new #CtkEffectGlow
 **/
CtkEffect *
ctk_effect_glow_new (void)
{
  return g_object_new (CTK_TYPE_EFFECT_GLOW,
                       NULL);
}

/**
 * ctk_effect_glow_set_factor
 *
 * @glow: A pointer to a CtkEffectGlow object
 * factor: A float representing the intensity of the glow effect.
 **/
void
ctk_effect_glow_set_factor (CtkEffectGlow* glow,
                            gfloat factor)
{
  ClutterActor *actor;

  g_return_if_fail (CTK_IS_EFFECT_GLOW(glow));

  if (factor > CTK_EFFECT_GLOW_MAX_FACTOR ||
      factor < CTK_EFFECT_GLOW_MIN_FACTOR)
    {
      g_warning ("%s() at line %s: Out of bound parameter", G_STRFUNC, G_STRLOC);
    }

  factor = CLAMP (factor, CTK_EFFECT_GLOW_MIN_FACTOR, CTK_EFFECT_GLOW_MAX_FACTOR);
  GET_PRIVATE (glow)->factor = factor;
  g_object_notify (G_OBJECT (glow), "factor");

  actor = ctk_effect_get_actor (CTK_EFFECT (glow));
  if (CLUTTER_IS_ACTOR (actor) && CLUTTER_ACTOR_IS_VISIBLE (actor))
    clutter_actor_queue_redraw (actor);
}
 
/**
 * ctk_effect_glow_get_factor:
 *
 * @glow: A pointer to a #CtkEffectGlow object
 * Returns: The glow intensity
 **/
gfloat
ctk_effect_glow_get_factor (CtkEffectGlow* glow)
{
  g_return_val_if_fail (CTK_IS_EFFECT_GLOW(glow), CTK_EFFECT_GLOW_DEFAULT_FACTOR);

  return GET_PRIVATE (glow)->factor;
}

/**
 * ctk_effect_glow_set_color
 *
 * @glow: A pointer to a #CtkEffectGlow object
 * @color: The glow color to set
 **/
void
ctk_effect_glow_set_color (CtkEffectGlow* glow,
                            ClutterColor* color)
{
  ClutterActor *actor;

  g_return_if_fail (CTK_IS_EFFECT_GLOW(glow));

  if(color)
    {
      GET_PRIVATE (glow)->color.red = color->red;
      GET_PRIVATE (glow)->color.green = color->green;
      GET_PRIVATE (glow)->color.blue = color->blue;
      GET_PRIVATE (glow)->color.alpha = color->alpha;
    }
  g_object_notify (G_OBJECT (glow), "color");

  actor = ctk_effect_get_actor (CTK_EFFECT (glow));
  if (CLUTTER_IS_ACTOR (actor) && CLUTTER_ACTOR_IS_VISIBLE (actor))
    clutter_actor_queue_redraw (actor);
}

/**
 * ctk_effect_glow_get_color
 *
 * @glow: A pointer to a #CtkEffectGlow object
 * @color: Returns the glow color
 **/
void
ctk_effect_glow_get_color (CtkEffectGlow* glow, ClutterColor* color)
{
  g_return_if_fail (CTK_IS_EFFECT_GLOW (glow));

  if(color)
    {
      color->red = GET_PRIVATE (glow)->color.red;
      color->green = GET_PRIVATE (glow)->color.green;
      color->blue = GET_PRIVATE (glow)->color.blue;
      color->alpha = GET_PRIVATE (glow)->color.alpha;
    }
}

/**
 * ctk_effect_glow_set_background_texture
 *
 * @glow: A pointer to a #CtkEffectGlow object
 * @pixbuf: Pixbux image containing the background texture
 *
 * This function takes a GdkPixbuf as parameter to create an opengl texture for the GlowEffect object.
 * The created opengl texture must remain private to the GlowEffect. 
 **/
void
ctk_effect_glow_set_background_texture (CtkEffectGlow *glow, GdkPixbuf *pixbuf) 
{
  g_return_if_fail (CTK_IS_EFFECT_GLOW (glow));

  if (GET_PRIVATE (glow)->background_texture_id)
    {
      CHECKGL (glDeleteTextures (1, &(GET_PRIVATE (glow)->background_texture_id)));
      GET_PRIVATE (glow)->background_texture_id = 0;
      GET_PRIVATE (glow)->background_texture_width = 0; 
      GET_PRIVATE (glow)->background_texture_height = 0;
    }
  
  if(pixbuf == 0)
    {
      return;
    }
  
  {
    guint opengl_texture_id = 0;
    guint texture_width = 0;
    guint texture_height = 0;
    int unpack = 0;
    int nChannel = 0;
  
    unsigned char *pixelbuffer = 0;
    pixelbuffer = gdk_pixbuf_get_pixels(pixbuf);

    /*guint rowstride = gdk_pixbuf_get_rowstride (pixbuf);*/
    /*g_debug("[ctk_effect_glow_set_background_texture] row stride is %d\n", rowstride);*/
  
    texture_width = gdk_pixbuf_get_width(pixbuf); 
    texture_height = gdk_pixbuf_get_height(pixbuf);
    nChannel = gdk_pixbuf_get_n_channels(pixbuf);

    if (pixelbuffer == 0)
      {
        g_warning("[ctk_effect_glow_set_background_texture] %s() at line %s: Invalid pixbuf", G_STRFUNC, G_STRLOC);
        return;
      }

    /* this line seems to be causing a crash... */
    /* CHECKGL( glActiveTextureARB(GL_TEXTURE0) ); */
    CHECKGL( glGenTextures(1, &opengl_texture_id) );
    CHECKGL( glBindTexture(GL_TEXTURE_2D, opengl_texture_id) );
    CHECKGL( glEnable(GL_TEXTURE_2D) );

    CHECKGL( glGetIntegerv(GL_UNPACK_ALIGNMENT, &unpack) );
    CHECKGL( glPixelStorei(GL_UNPACK_ALIGNMENT, 4) );
    CHECKGL( glPixelStorei(GL_UNPACK_ROW_LENGTH, 0) );
    CHECKGL( glPixelStorei(GL_UNPACK_IMAGE_HEIGHT, 0) );
    CHECKGL( glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0) );
    CHECKGL( glPixelStorei(GL_UNPACK_SKIP_ROWS, 0) );

    /* Read bits */
    if(nChannel == 3)
      {
        CHECKGL( glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB8, texture_width, texture_height, 0, GL_RGB, GL_UNSIGNED_BYTE, pixelbuffer) );
      }
    else if(nChannel == 4)
      {
        CHECKGL( glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, texture_width, texture_height, 0, GL_RGBA, GL_UNSIGNED_BYTE, pixelbuffer) );
      }
    else
      {
        /* We can support many texture format but for now, we support only RGB and RGB */
        g_warning ("[ctk_effect_glow_set_background_texture] %s() at line %s: invalid texture format", G_STRFUNC, G_STRLOC);
        if (opengl_texture_id)
          {
            CHECKGL (glDeleteTextures (1, &opengl_texture_id));
          }
      }

    CHECKGL( glPixelStorei(GL_UNPACK_ALIGNMENT, unpack) );
    
    CHECKGL( glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST) );
    CHECKGL( glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST) );
    CHECKGL( glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT) );
    CHECKGL( glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT) );

    CHECKGL( glBindTexture(GL_TEXTURE_2D, 0) );    
 
    GET_PRIVATE (glow)->background_texture_id = opengl_texture_id;
    GET_PRIVATE (glow)->background_texture_width = texture_width;
    GET_PRIVATE (glow)->background_texture_height = texture_height;
    
    /*g_debug("GL id: %d - Texture size %dx%d - %d Components\n",
    GET_PRIVATE (glow)->background_texture_id, 
    GET_PRIVATE (glow)->background_texture_width,
    GET_PRIVATE (glow)->background_texture_height, nChannel);*/
  }
}


