/*
    cobbler-enlist - Cobbler enlistment tool
    Copyright (C) 2011 Canonical Ltd.

    Authors: Dave Walker (Daviey) <DaveWalker@ubuntu.com>
             Carlos de-Avillez <carlos.de.avillez@ubuntu.com>
             Adam Gandelman <adamg@canonical.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, version 3 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Usage: ./cobbler-enlist --serverurl=http://cobbler.server/cobbler_api \
             --username=cobbler --password=cobbler --name=test \
             --profile=ubuntu-server-x86_64
            optionally, '--netif=<network I/F>' can also be passed, restricting
            registration to this I/F. If no --netif, then all I/Fs will be
            registered.

*/

#define NAME "Cobbler Enlistment Tool"
#define VERSION "1.0"

#define OPT_URL      0
#define OPT_USRNAME  1
#define OPT_PASWD    2
#define OPT_NAME     3
#define OPT_PROFILE  4
#define OPT_NETIF    5
#define OPT_HELP     6

static int debug =  0;

struct cobbler_client {
  xmlrpc_env env;
  xmlrpc_client *client;
  xmlrpc_value *resultP;
  const char *token;
  const char *system;
  const char *serverUrl;
  const char *username;
  const char *password;
  const char *name;
  const char *profile;
  const char *netif;
  const char *mgmt_class;
  const char *macaddr;
};

int init_client(struct cobbler_client *enlister) {
  enlister->client = NULL;
  enlister->token = NULL;
  enlister->system = NULL;
  enlister->serverUrl = NULL;
  enlister->username = NULL;
  enlister->password = NULL;
  enlister->name = NULL;
  enlister->profile = NULL;
  enlister->netif = NULL;
  enlister->mgmt_class = NULL;
  enlister->macaddr = NULL;
  return 0;
}

void display_config(struct cobbler_client *enlister) {
  printf("[DEBUG] Enlisting system using the following config:\n");
  printf("\tserverUrl:  %s\n", enlister->serverUrl);
  printf("\tusername:   %s\n", enlister->username);
  printf("\tpassword:   %s\n", enlister->password);
  printf("\tname:       %s\n", enlister->name);
  printf("\tprofile:    %s\n", enlister->profile);
  if (enlister->netif)
    printf("\tnetif:      %s\n", enlister->netif);
  if (enlister->mgmt_class)
    printf("\tmgmt class: %s\n", enlister->mgmt_class);
  printf("\tmac addr:   %s\n", enlister->macaddr);
}

// Overide ether_ntoa_r because stock decides the convention of leading zeros is silly.
char *
ether_ntoa_r (const struct ether_addr *addr, char *buf)
{
  snprintf (buf, 18, "%02x:%02x:%02x:%02x:%02x:%02x",
            addr->ether_addr_octet[0], addr->ether_addr_octet[1],
            addr->ether_addr_octet[2], addr->ether_addr_octet[3],
            addr->ether_addr_octet[4], addr->ether_addr_octet[5]);
  return buf;
}

int get_mac_address(struct cobbler_client *enlister) {
  if (debug)
    printf("[DEBUG] get_mac_address()\n");

  int fd;
  struct if_nameindex *curif, *ifs;
  char mac_str[25];
  char interface[255];
  char *macaddr = malloc(sizeof(mac_str));

  int ioctl_data = 0;
  struct ifreq ifr;

  if ((fd = socket (PF_INET, SOCK_DGRAM, 0)) != -1) {
    ifs = if_nameindex ();
    for (curif = ifs; curif && curif->if_name; curif++) {
      if (((enlister->netif == NULL) ||
          (strcasecmp(enlister->netif, curif->if_name) == 0))
           && (strcasecmp(curif->if_name, "lo") !=0)) {

          ioctl_data = socket (PF_INET, SOCK_STREAM, 0);
          memset (&ifr, 0, sizeof (ifr));
          strncpy (ifr.ifr_name, curif->if_name, sizeof (ifr.ifr_name));
          strncpy (interface, curif->if_name, sizeof (interface));
          ioctl (ioctl_data, SIOCGIFHWADDR, &ifr);
          ether_ntoa_r( (const struct ether_addr *)
                        &(ifr.ifr_hwaddr.sa_data), mac_str);
          if (debug)
            printf("[DEBUG] Using interface: %s Addr: %s\n", interface, mac_str);
          if (enlister->netif == NULL)
            enlister->netif = curif->if_name;
          strncpy(macaddr, mac_str, sizeof(mac_str));
          enlister->macaddr = macaddr;
          return 0;
      }
    }
  }

  return 1;
}
void help(char *progname)
{
  printf("Usage: %s --help  -- provides this help\n" , progname);
  printf("     : %s <parameters>\n", progname);
  printf("Parameters: -s, --serverurl   -- resolvable Cobbler server API URL\n");
  printf("            -u, --username    -- valid Cobbler user\n");
  printf("            -p, --password    -- password for above user\n");
  printf("            -n, --name        -- Cobbler name\n");
  printf("            -P, --profile     -- Cobbler profile\n");
  printf("            -i, --netif       -- single network I/F to be registered\n");
  printf("            -m, --manclass    -- Cobbler management class\n");
  printf("\n  --netif is optional; if not specified, all network I/Fs will be registered\n");
  printf("  all other parameters are required\n");
}
