# app_tests.py - unit tests for app.py
# Copyright (C) 2008, 2009  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import os
import shutil
import tempfile
import unittest

import computerjanitor
import computerjanitorapp


class MockUI(object):

    def __init__(self, testcase, app, pm):
        self.testcase = testcase
        
    def run(self, options, args):
        self.testcase.ui_ran = True


class MockPackage(object):

    def __init__(self, downloadable=False):
        self.downloadable = downloadable
        self.candidate = self


class MockApt(dict):

    def __init__(self):
        self._depcache = self
        
    def ReadPinFile(self, filename):
        pass

    def Cache(self):
        return self


class MockCruft(object):

    def __init__(self, name):
        self.name = name

    def get_name(self):
        return self.name


class ApplicationTests(unittest.TestCase):

    def setUp(self):
        self.mock_apt = MockApt()
        self.app = computerjanitorapp.Application(apt=self.mock_apt)
        self.app.apt_cache["dash"] = MockPackage(downloadable=True)
        self.app.apt_cache["gzip"] = MockPackage(downloadable=True)

    def testSetsUpAptAttributeCorrectly(self):
        self.assertEqual(self.app.apt, self.mock_apt)
        
    def testSetsUpAndReturnsState(self):
        self.assert_(self.app.state)

    def testSetsUpAptCacheWhenRequested(self):
        self.assertNotEqual(self.app.apt_cache, None)

    def testSetsOptionDefaultsCorrectly(self):
        options, args = self.app.parse_options(args=[])
        self.assertEqual(args, [])
        self.assertEqual(options.all, None)
        self.assertEqual(options.state_file, 
                         "/var/lib/computer-janitor/state.dat")
        self.assertEqual(options.no_act, None)

    def testAcceptsDashDashAllOption(self):
        options, args = self.app.parse_options(args=["--all"])
        self.assertEqual(options.all, True)

    def testAcceptsDashDashStateFileOption(self):
        options, args = self.app.parse_options(args=["--state-file=foo"])
        self.assertEqual(options.state_file, "foo")

    def testAcceptsDashDashNoActOption(self):
        options, args = self.app.parse_options(args=["--no-act"])
        self.assertEqual(options.no_act, True)

    def testRunsUserInterface(self):

        def pm_class(app, plugin_dirs):
            self.pm_ran = True

        def ui_class(app, pm):
            return MockUI(self, app, pm)

        self.pm_ran = False
        self.ui_ran = False
        self.app.run(ui_class=ui_class, plugin_manager_class=pm_class)
        self.assert_(self.ui_ran)

    def testAcceptsAptCacheWhenEssentialPackagesAreThere(self):
        self.assertEqual(self.app.verify_apt_cache(), None)

    def testRejectsAptCacheWhenDashIsMissing(self):
        del self.app.apt_cache["dash"]
        self.assertRaises(computerjanitor.Exception, 
                          self.app.verify_apt_cache)

    def testRejectsAptCacheWhenGzipIsMissing(self):
        del self.app.apt_cache["gzip"]
        self.assertRaises(computerjanitor.Exception, 
                          self.app.verify_apt_cache)

    def testRejectsAptCacheWhenDashIsNotDownloadable(self):
        self.app.apt_cache["dash"].candidate.downloadable = False
        self.assertRaises(computerjanitor.Exception, 
                          self.app.verify_apt_cache)

    def testRejectsAptCacheWhenGzipIsNotDownloadable(self):
        self.app.apt_cache["dash"].candidate.downloadable = False
        self.assertRaises(computerjanitor.Exception, 
                          self.app.verify_apt_cache)

    def testSetsDefaultListOfWhitelistDirectoriesCorrectly(self):
        self.assert_("/etc/computer-janitor.d" in self.app.whitelist_dirs)

    def testReturnsEmptyWhitelistByDefault(self):
        dirname = tempfile.mkdtemp()
        whitelist = self.app.whitelisted_cruft(dirnames=[dirname])
        shutil.rmtree(dirname)
        self.assertEqual(whitelist, [])

    def testDoesNotMindNonExistentWhitelistDirectory(self):
        dirname = tempfile.mkdtemp()
        subdir = os.path.join(dirname, "foo")
        whitelist = self.app.whitelisted_cruft(dirnames=[subdir])
        shutil.rmtree(dirname)
        self.assertEqual(whitelist, [])

    def testReadsWhitelistFilesCorrectly(self):
        dirname = tempfile.mkdtemp()
        temp1 = os.path.join(dirname, "foo.whitelist")
        temp2 = os.path.join(dirname, "foo.whitelist~")

        file(temp1, "w").write("deb:foo\n")
        file(temp2, "w").write("deb:bar\n")

        whitelist = self.app.whitelisted_cruft(dirnames=[dirname])
        shutil.rmtree(dirname)
        self.assertEqual(whitelist, ["deb:foo"])

    def testFindsCorrectWhitelistFilesInDotDDirectory(self):
        dirname = tempfile.mkdtemp()
        file(os.path.join(dirname, "foo.whitelist"), "w").write("deb:foo\n")
        file(os.path.join(dirname, "foo.whitelist~"), "w").write("deb:bar\n")
        list = self.app.whitelist_files([dirname])
        shutil.rmtree(dirname)
        self.assertEqual(list, [os.path.join(dirname, "foo.whitelist")])

    def testRemovesWhitelistedCruftCorrectly(self):
        crufts = [MockCruft("deb:foo"), MockCruft("deb:bar")]
        dirname = tempfile.mkdtemp()
        file(os.path.join(dirname, "foo.whitelist"), "w").write("deb:foo\n")
        crufts2 = self.app.remove_whitelisted(crufts, dirnames=[dirname])
        shutil.rmtree(dirname)
        self.assertEqual(crufts2, crufts[1:])
