# Copyright (C) 2008, 2009, 2010  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""dbus service for cleaning up crufty packages that are no longer needed."""


from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'Service',
    ]


import atexit
import logging

import dbus.service

from computerjanitor import PackageCruft
from computerjanitord.application import Application
from computerjanitord.authenticator import Authenticator
from computerjanitord.collector import Collector
from computerjanitord.errors import NoSuchCruftError, PermissionDeniedError
from computerjanitord.state import State, DEFAULT_STATE_FILE

import glib

log = logging.getLogger('computerjanitor')
MISSING = object()

DBUS_INTERFACE_NAME = 'com.ubuntu.ComputerJanitor'
PRIVILEGE = 'com.ubuntu.computerjanitor.updatesystem'


class Service(dbus.service.Object):
    """Backend dbus service that handles removing crufty packages."""

    def __init__(self, options):
        """Create the dbus service.

        :param options: The command line options class.
        :type options: `Options`
        """
        self.dry_run = options.arguments.dry_run
        self.state_file = (DEFAULT_STATE_FILE
                           if options.arguments.state_file is None
                           else options.arguments.state_file)
        self.application = Application()
        self.state = State()
        self.state.load(self.state_file)
        self.collector = Collector(self.application, service=self)
        self.authenticator = Authenticator()
        bus_name = dbus.service.BusName(
            DBUS_INTERFACE_NAME, bus=dbus.SystemBus())
        dbus.service.Object.__init__(self, bus_name, '/')
        # We can't use the decorator because that doesn't work with methods;
        # self doesn't get passed to the handler.
        atexit.register(self._exit_handler)

    def _exit_handler(self):
        """Ensure that the state file is saved at exit."""
        if not self.dry_run:
            self.state.save(self.state_file)

    def _authenticate(self, sender, connection):
        """Authenticate via PolicyKit.

        :param sender: The dbus client sender.
        :param connection: The dbus client connection.
        :raises PermissionDeniedError: when the authentication fails.
        """
        if not self.authenticator.authenticate(sender, connection, PRIVILEGE):
            log.error('Permission denied: {0} for {1} on {2}'.format(
                PRIVILEGE, sender, connection))
            raise PermissionDeniedError(PRIVILEGE)
        log.debug('Permission granted: {0} for {1} on {2}'.format(
            PRIVILEGE, sender, connection))

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         out_signature='as')
    def find(self):
        """Find all the non-whitelisted cruft on the system.

        Because this is a read-only interface it does not need authorization
        to be called.

        :return: A list of matching cruft names.
        """
        return list(cruft.get_name() for cruft in self.collector.cruft)

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         out_signature='b')
    def find_async(self):
        """Find all the non-whitelisted cruft on the system, asynchronously.

        Because this is a read-only interface it does not need authorization
        to be called.  Use this method when you can't wait for the cruft
        searching process to complete, since it might take some time.  To
        respond when the cruft has been found, set up a `find_finished` signal
        handler.

        :return: True
        """
        glib.timeout_add_seconds(1, self._find_async)
        return True

    def _find_async(self):
        """Find all cruft asynchronously and call the signal handler."""
        cruft = list(cruft.get_name() for cruft in self.collector.cruft)
        self.find_finished(cruft)
        # Only call the callback once.
        return False

    @dbus.service.signal(DBUS_INTERFACE_NAME, signature='as')
    def find_finished(self, cruft):
        """dbus signal called when `_find_async()` completes."""
        log.debug('find_finished: {0}'.format(cruft))

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         out_signature='as',
                         # Must wrap these in str() because Python < 2.6.5
                         # does not like unicode keyword arguments.
                         sender_keyword=str('sender'),
                         connection_keyword=str('connection'))
    def load(self, sender=None, connection=None):
        """Load the state file."""
        self._authenticate(sender, connection)
        self.state.load(self.state_file)
        return list(self.state.ignore)

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         # Must wrap these in str() because Python < 2.6.5
                         # does not like unicode keyword arguments.
                         sender_keyword=str('sender'),
                         connection_keyword=str('connection'))
    def save(self, sender=None, connection=None):
        """Save the state file."""
        self._authenticate(sender, connection)
        if not self.dry_run:
            self.state.save(self.state_file)

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         in_signature='s',
                         # Must wrap these in str() because Python < 2.6.5
                         # does not like unicode keyword arguments.
                         sender_keyword=str('sender'),
                         connection_keyword=str('connection'))
    def ignore(self, name, sender=None, connection=None):
        """Ignore the named cruft.

        :param name: The name of the cruft to ignore.
        :type filename: string
        """
        # Make sure this is known cruft first.
        cruft = self.collector.cruft_by_name.get(name, MISSING)
        if cruft is MISSING:
            log.error('ignore(): No such cruft: {0}'.format(name))
            raise NoSuchCruftError(name)
        self._authenticate(sender, connection)
        if not self.dry_run:
            self.state.ignore.add(name)

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         in_signature='s',
                         # Must wrap these in str() because Python < 2.6.5
                         # does not like unicode keyword arguments.
                         sender_keyword=str('sender'),
                         connection_keyword=str('connection'))
    def unignore(self, name, sender=None, connection=None):
        """Unignore the named cruft.

        :param name: The name of the cruft to unignore.
        :type filename: string
        """
        cruft = self.collector.cruft_by_name.get(name, MISSING)
        if cruft is MISSING:
            log.error('ignore(): No such cruft: {0}'.format(name))
            raise NoSuchCruftError(name)
        self._authenticate(sender, connection)
        if not self.dry_run:
            # Don't worry if we're already not ignoring the cruft (i.e. don't
            # raise a KeyError here if 'name' is not in the set).
            self.state.ignore.discard(name)

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         out_signature='as')
    def ignored(self):
        """Return the list of ignored cruft.

        :return: The names of the ignored cruft.
        :rtype: list of strings
        """
        return list(self.state.ignore)

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         in_signature='s',
                         out_signature='s')
    def get_description(self, name):
        """Return the description of the named cruft.

        :param name: The cruft name.
        :type name: string
        :return: The description of the cruft.
        :rtype: string
        """
        cruft = self.collector.cruft_by_name.get(name, MISSING)
        if cruft is MISSING:
            log.error('get_description(): No such cruft: {0}'.format(name))
            raise NoSuchCruftError(name)
        return cruft.get_description()

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         in_signature='s',
                         out_signature='s')
    def get_shortname(self, name):
        """Return the short name of the named cruft.

        :param name: The cruft name.
        :type name: string
        :return: The short nameof the cruft.
        :rtype: string
        """
        cruft = self.collector.cruft_by_name.get(name, MISSING)
        if cruft is MISSING:
            log.error('get_shortname(): No such cruft: {0}'.format(name))
            raise NoSuchCruftError(name)
        return cruft.get_shortname()

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         in_signature='s',
                         out_signature='st')
    def get_details(self, name):
        """Return some extra details about the named cruft.

        :param name: The cruft name.
        :type name: string
        :return: Some extra details about the named cruft, specifically its
            'type' and the amount of disk space it consumes.  The type is
            simply the name of the cruft instance's class.
        :rtype: string, uint64
        """
        cruft = self.collector.cruft_by_name.get(name, MISSING)
        if cruft is MISSING:
            log.error('get_shortname(): No such cruft: {0}'.format(name))
            raise NoSuchCruftError(name)
        return cruft.__class__.__name__, cruft.get_disk_usage()

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         in_signature='as', # array of strings
                         # Must wrap these in str() because Python < 2.6.5
                         # does not like unicode keyword arguments.
                         sender_keyword=str('sender'),
                         connection_keyword=str('connection'))
    def clean(self, names, sender=None, connection=None):
        """Clean the named crufts.

        :param names: The names of the cruft to clean.
        :type names: list of strings
        """
        self._authenticate(sender, connection)
        self.collector.clean(names, self.dry_run)

    @dbus.service.signal(DBUS_INTERFACE_NAME,
                         signature='s')
    def cleanup_status(self, cruft):
        """Signal cleanup status.

        This signal is used to incrementally inform clients that some cleanup
        work is being done.  It is called at the beginning of the cleanup
        process and after each plugin has completed its `post_cleanup()`
        method.

        :param done: The name of the next piece of cruft to be cleaned up, or
            the empty string when there's nothing left to do.
        :type done: string
        """
        log.debug('cleanup_status: {0}'.format(cruft))
