# Copyright (C) 2008, 2009, 2010  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Test the package state."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'test_suite',
    ]


import os
import difflib
import tempfile
import textwrap
import unittest

from computerjanitord.state import State

NL = '\n'


class TestState(unittest.TestCase):
    """Test the `State` class."""

    def setUp(self):
        self.state = State()
        # Create a temporary file with some enabled and disabled packages.
        fd, self._state_file = tempfile.mkstemp()
        os.close(fd)
        with open(self._state_file, 'w') as fp:
            print >> fp, textwrap.dedent("""\
            [deb:foo]
            ignore: false
            [deb:bar]
            ignore: true
            [deb:baz]
            ignore: true
            """)
        fd, self._state_file_old = tempfile.mkstemp()
        os.close(fd)
        with open(self._state_file_old, 'w') as fp:
            print >> fp, textwrap.dedent("""\
            [deb:qux]
            enabled: false
            [deb:fno]
            enabled: true
            [deb:bla]
            enabled: true
            """)
        fd, self._write_file = tempfile.mkstemp()
        os.close(fd)

    def tearDown(self):
        os.remove(self._state_file)
        os.remove(self._state_file_old)
        os.remove(self._write_file)

    def assertEqualNdiff(self, expected, got):
        expected_lines = expected.splitlines()
        got_lines = got.splitlines()
        self.assertEqual(
            expected, got,
            '\n' + NL.join(difflib.ndiff(expected_lines, got_lines)))

    def test_initially_no_previously_ignored(self):
        self.assertEqual(self.state.ignore, set())

    def test_load_state(self):
        self.state.load(self._state_file)
        self.assertEqual(self.state.ignore, set(('deb:bar', 'deb:baz')))

    def test_backward_compatibility_file_format(self):
        # Here, enabled:false means to ignore the package.
        self.state.load(self._state_file_old)
        self.assertEqual(self.state.ignore, set(('deb:qux',)))

    def test_ignore(self):
        self.state.ignore.add('deb:buz')
        self.state.ignore.add('deb:baz')
        self.assertEqual(self.state.ignore, set(('deb:buz', 'deb:baz')))

    def test_more_ignores(self):
        self.state.load(self._state_file)
        self.state.ignore.add('deb:buz')
        self.state.ignore.add('deb:baz')
        self.assertEqual(self.state.ignore,
                         set(('deb:bar', 'deb:buz', 'deb:baz')))

    def test_unignore(self):
        self.state.load(self._state_file)
        self.state.ignore.remove('deb:bar')
        self.assertEqual(self.state.ignore, set(('deb:baz',)))

    def test_write(self):
        self.state.load(self._state_file)
        self.state.ignore.add('deb:buz')
        self.state.ignore.remove('deb:baz')
        self.state.save(self._write_file)
        with open(self._write_file) as fp:
            got = fp.read()
        expected = textwrap.dedent("""\
        [deb:bar]
        ignore: true

        [deb:buz]
        ignore: true

        """)
        self.assertEqualNdiff(expected, got)

    def test_write_new_format(self):
        self.state.load(self._state_file_old)
        self.state.save(self._write_file)
        with open(self._write_file) as fp:
            got = fp.read()
        expected = textwrap.dedent("""\
        [deb:qux]
        ignore: true

        """)
        self.assertEqualNdiff(expected, got)


def test_suite():
    suite = unittest.TestSuite()
    suite.addTests(unittest.makeSuite(TestState))
    return suite
