# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Gtk user interface for computer janitor."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'UserInterface',
    ]


import os
import dbus
import glib
import gobject
import logging

from operator import mod

from gi.repository import Gtk, Pango

from computerjanitorapp import __version__, setup_gettext
from computerjanitorapp.gtk.dialogs import AreYouSure, CleanupProblem
from computerjanitorapp.gtk.store import (
    ListStoreColumns, Store, optimize, unused)
from computerjanitorapp.utilities import format_size
from computerjanitord.service import DBUS_INTERFACE_NAME


Gtk.require_version('2.0')
log = logging.getLogger('computerjanitor')
_ = setup_gettext()


GLADE = '/usr/share/computer-janitor/ComputerJanitor.ui'
ROOT_WIDTH = 900
ROOT_HEIGHT = 500
NL = '\n'

# Keys are lower-cased cruft types, i.e. class name of cruft instances.
ACTIONS = dict(
    packagecruft=_('Package will be <b>removed</b>.'),
    filecruft=_('Package will be <b>installed</b>.'),
    missingpackagecruft=_('File will be <b>removed</b>.'),
    )

SENSITIVE_WIDGETS = (
    'do_button',
    'optimize_treeview',
    'quit_menuitem',
    'show_previously_ignored',
    'sort_by_name',
    'sort_by_size',
    'unused_treeview',
    )


class UserInterface:
    """Implementation of the Gtk user interface."""

    def __init__(self):
        # Connect to the dbus service.
        system_bus = dbus.SystemBus()
        proxy = system_bus.get_object(DBUS_INTERFACE_NAME, '/')
        self.janitord = dbus.Interface(
            proxy, dbus_interface=DBUS_INTERFACE_NAME)
        # Create the model on which the TreeViews will be based.
        self.store = Store(self.janitord)
        self.popup_menus = {}
        self.cruft_name_columns = set()
        # Sort by name by default.
        self._sort_key = self._by_name
        # Work is happening asynchronously on the dbus service.
        self.working = False
        # Connect to the signal the server will emit when cleaning up.
        self.janitord.connect_to_signal('cleanup_status', self._clean_working)

    def run(self):
        """Set up the widgets and run the main loop."""
        builder = Gtk.Builder()
        builder.set_translation_domain('computerjanitor')
        # Load the glade ui file, which can be overridden from the environment
        # for testing purposes.
        glade_file = os.environ.get('COMPUTER_JANITOR_GLADE', GLADE)
        builder.add_from_file(glade_file)
        # Bind widgets to callbacks.  Initialize the Do... button to
        # insensitive, but possibly toggle it back to sensitive if we find
        # cruft.
        self.find_and_bind_widgets(builder)
        self.widgets['do_button'].set_sensitive(False)
        self.janitord.connect_to_signal('find_finished', self._find_finished)
        # Do the initial search for cruft and set up the TreeView model.
        self.store.find_cruft()
        self.sort_cruft()
        # Now hook the TreeViews up to there view of the model.
        self.create_column('unused_treeview', unused)
        self.create_column('optimize_treeview', optimize)
        # Set the dimensions of the root window.
        root = self.widgets['window']
        root.set_default_size(ROOT_WIDTH, ROOT_HEIGHT)
        # Map the root window and go!
        root.show()
        Gtk.main()

    def find_and_bind_widgets(self, builder):
        """Bind widgets and callbacks."""
        # Start by extracting all the bindable widgets from the ui builder,
        # keeping track of them as mapped to their name.
        self.widgets = {}
        for ui_object in builder.get_objects():
            if issubclass(type(ui_object), Gtk.Buildable):
                widget_name = Gtk.Buildable.get_name(ui_object)
                self.widgets[widget_name] = ui_object
                # Search through the attributes of this instance looking for
                # callbacks for this widget.  We use the naming convention
                # 'on_<widget>_<signal>' for such methods.  Both the widget
                # name and signal (or event) can contain underscores.  Connect
                # the widget to the callback.
                prefix = 'on_{0}_'.format(widget_name)
                for method_name in dir(self):
                    if method_name.startswith(prefix):
                        signal_name = method_name[len(prefix):]
                        ui_object.connect(
                            signal_name, getattr(self, method_name))

    def create_column(self, widget_name, filter_func):
        """Set up a column in the named TreeView."""
        treeview = self.widgets[widget_name]
        # XXX 2010-03-04 barry: This is a bit of an abuse because it's
        # supposed to specify whether users are to read across the rows.  As a
        # side effect it renders the columns with alternating row colors, but
        # that's not it's primary function.
        treeview.set_rules_hint(True)
        # Each TreeView contains two columns.  The leftmost one is a toggle
        # that when select tells c-j to act on that cruft.  Deselecting the
        # toggle ignores the package for next time.
        toggle_cr = Gtk.CellRendererToggle()
        toggle_cr.connect('toggled', self._toggled, treeview)
        toggle_cr.set_property('yalign', 0)
        toggle_col = Gtk.TreeViewColumn()
        toggle_col.pack_start(toggle_cr, True)
        toggle_col.add_attribute(toggle_cr, 'active', ListStoreColumns.active)
        treeview.append_column(toggle_col)
        # The rightmost column contains the details of the cruft.  It will
        # always contain the cruft name and can be expanded to display cruft
        # details.  Tell the column to get its toggle's active state from the
        # model.
        name_cr = Gtk.CellRendererText()
        name_cr.set_property('yalign', 0)
        name_cr.set_property('wrap-mode', Pango.WrapMode.WORD)
        name_col = Gtk.TreeViewColumn()
        name_col.pack_start(name_cr, True)
        name_col.add_attribute(name_cr, 'markup', ListStoreColumns.text)
        treeview.append_column(name_col)
        self.cruft_name_columns.add(name_col)
        # The individual crufts may or may not be visible in this TreeView.
        # It's the filter function that controls this, so set that now.
        filter_store = self.store.filter_new(None)
        filter_store.set_visible_func(filter_func, None)
        treeview.set_model(filter_store)
        # Each TreeView has a popup menu for select or deselecting all visible
        # cruft.
        self.create_popup_menu_for_treeview(treeview)

    def create_popup_menu_for_treeview(self, treeview):
        """The tree views have a popup menu to select/deselect everything.

        :param treeview: The `TreeView` to attach the menu to.
        """
        select_all = Gtk.MenuItem(label='Select all')
        select_all.connect('activate', self.popup_menu_select_all, treeview)
        unselect_all = Gtk.MenuItem(label='Unselect all')
        unselect_all.connect('activate',
                             self.popup_menu_unselect_all, treeview)
        menu = Gtk.Menu()
        menu.append(select_all)
        menu.append(unselect_all)
        menu.show_all()
        self.popup_menus[treeview] = menu

    def _by_name(self, cruft_name):
        """Sort by cruft name."""
        return cruft_name

    def _by_size(self, cruft_name):
        """Sort by cruft size, from largest to smallest."""
        # Return negative size to sort from largest to smallest.
        return -self.janitord.get_details(cruft_name)[1]

    def sort_cruft(self):
        """Sort the cruft displays, either by name or size."""
        # The way reordering (not technically 'sorting') works in gtk is that
        # you give a list of integer indexes to the the ListStore.  These
        # indexes are in sorted order, and refer to the pre-sort indexes of
        # the items in the store.  IOW, the ListStore knows that if the first
        # integer in the list is 7, it will move the 7th item to the top.
        #
        # Start by getting the indexes and names of the currenly sorted cruft.
        # We'll fill this list with 2-tuples of the format:
        # (sort-key, current-index).
        cruft_data = []
        def get(model, path, iter, crufts):
            cruft_name = model.get_value(iter, ListStoreColumns.name)
            crufts.append((self._sort_key(cruft_name), len(crufts)))
            # Continue iterating.
            return False
        self.store.foreach(get, cruft_data)
        cruft_data.sort()
        cruft_indexes = [index for key, index in cruft_data]
        # No need to do anything if there is no cruft.
        if len(cruft_indexes) > 0:
            self.store.reorder(cruft_indexes)

    def get_cleanable_cruft(self):
        """Return the list of cleanable cruft candidates.

        :return: List of cleanable cruft.
        :rtype: list of 2-tuples of (cruft_name, is_package_cruft)
        """
        cleanable_cruft = []
        def collect(model, path, iter, crufts):
            # Only clean up active cruft, i.e. those that are specifically
            # checked as ready for cleaning.
            cruft_active = model.get_value(iter, ListStoreColumns.active)
            if cruft_active:
                cruft_name = model.get_value(iter, ListStoreColumns.name)
                cruft_is_package_cruft = model.get_value(
                    iter, ListStoreColumns.is_package_cruft)
                crufts.append((cruft_name, cruft_is_package_cruft))
            # Continue iterating.
            return False
        self.store.foreach(collect, cleanable_cruft)
        return cleanable_cruft

    def toggle_long_description(self, treeview):
        """Toggle the currently selected cruft's long description.

        :param treeview: The TreeView
        """
        selection = treeview.get_selection()
        filter_model, selected = selection.get_selected()
        if not selected:
            return
        model = filter_model.get_model()
        iter = filter_model.convert_iter_to_child_iter(selected)
        cruft_name = model.get_value(iter, ListStoreColumns.name)
        expanded = model.get_value(iter, ListStoreColumns.expanded)
        shortname = model.get_value(iter, ListStoreColumns.short_name)
        if expanded:
            # Collapse it.
            value = gobject.markup_escape_text(shortname)
        else:
            cruft_type, size = self.janitord.get_details(cruft_name)
            lines = [gobject.markup_escape_text(shortname)]
            action = ACTIONS.get(cruft_type.lower())
            if action is not None:
                lines.append(action)
            # XXX barry 2010-08-23: LP: #622720
            lines.append(mod(_('Size: %(bytes)s'),
                             dict(bytes=format_size(size))))
            lines.append('')
            description = self.janitord.get_description(cruft_name)
            lines.append(gobject.markup_escape_text(description))
            value = NL.join(lines)
        model.set_value(iter, ListStoreColumns.text, value)
        model.set_value(iter, ListStoreColumns.expanded, not expanded)

    def desensitize(self):
        """Make certain ui elements insensitive during work."""
        for widget in SENSITIVE_WIDGETS:
            self.widgets[widget].set_sensitive(False)

    def sensitize(self):
        """Make certain ui elements sensitive after work."""
        for widget in SENSITIVE_WIDGETS:
            self.widgets[widget].set_sensitive(True)

    def _find_finished(self, all_cruft_names):
        """dbus signal handler."""
        self.widgets['do_button'].set_sensitive(len(all_cruft_names) > 0)

    # Popup menu support.

    def _treeview_foreach_set_set_state(self, treeview, enabled):
        """Set the state of the cruft 'active' flag for all cruft.

        :param treeview: The `TreeView` to set cruft state on.
        :param enabled: The new state flag for all cruft.  True means enabled.
        :type enabled: bool
        """
        def set_state(model, path, iter, user_data):
            # Set the state on an individual piece of cruft.  Start by
            # changing the state of the cruft on the dbus service.
            child_iter = model.convert_iter_to_child_iter(iter)
            cruft_name = self.store.get_value(
                child_iter, ListStoreColumns.name)
            if enabled:
                self.janitord.unignore(cruft_name)
            else:
                self.janitord.ignore(cruft_name)
            # Now set the active state in the model.
            self.store.set_value(child_iter, ListStoreColumns.active, enabled)
        treeview.get_model().foreach(set_state, None)
        # Save the updated state on the dbus service.
        self.janitord.save()

    def popup_menu_select_all(self, menuitem, treeview):
        self._treeview_foreach_set_set_state(treeview, True)

    def popup_menu_unselect_all(self, menuitem, treeview):
        self._treeview_foreach_set_set_state(treeview, False)

    # Progress bar

    def pulse(self):
        """Progress bar callback, showing that something is happening."""
        progress = self.widgets['progressbar_status']
        if self.working:
            progress.show()
            progress.pulse()
            return True
        else:
            # All done.  Hide the progress bar, make the ui elements sensitive
            # again, update the store, and kill the timer.
            progress.hide()
            self.store.clear()
            self.store.find_cruft()
            self.sensitize()
            return False

    def _clean_working(self, cruft):
        """dbus signal handler; the 'clean' operation is in progress.

        :param done: The cruft that is being cleaned up.
        :type done: string
        """
        # Just mark the status here.  The progress bar pulsar will handle
        # doing the actual work.
        self.working = (cruft != '')
        if self.working:
            self.widgets['progressbar_status'].set_text(
                # XXX barry 2010-08-23: LP: #622720
                mod(_('Processing %(cruft)s'), dict(cruft=cruft)))

    # Callbacks

    def _toggled(self, widget, path, treeview):
        """Handle the toggle button in a TreeView cell.

        :param widget: The CellRendererToggle
        :param path: The cell's path.
        :param treeview: The TreeView
        """
        # Find out which cruft's toggle was clicked.
        model = treeview.get_model()
        filter_iter = model.get_iter(path)
        child_iter = model.convert_iter_to_child_iter(filter_iter)
        cruft_name = self.store.get_value(child_iter, ListStoreColumns.name)
        state = self.store.get_value(child_iter, ListStoreColumns.active)
        # Toggle the current state.
        new_state = not state
        if new_state:
            self.janitord.unignore(cruft_name)
        else:
            self.janitord.ignore(cruft_name)
        self.store.set_value(child_iter, ListStoreColumns.active, new_state)
        self.store.set_value(
            child_iter, ListStoreColumns.server_ignored, not new_state)
        # Save the new ignored state on the dbus service.
        self.janitord.save()

    # Signal and event handlers.

    def on_quit_menuitem_activate(self, *args):
        """Signal and event handlers for quitting.

        Since we just want things to go away, we don't really care about the
        arguments.  Just tell the main loop to exit.
        """
        # Don't quit while we're working.
        if self.working:
            return True
        Gtk.main_quit()

    on_window_delete_event = on_quit_menuitem_activate

    def treeview_button_press_event(self, treeview, event):
        """Handle mouse button press events on the TreeView ourselves.

        We handle mouse button presses ourselves so that we can either
        toggle the long description (button 1, typically left) or
        pop up a menu (button 3, typically right).
        """
        # Original comment: This is slightly tricky and probably a source of
        # bugs.  Oh well.
        if event.button.button == 1:
            # Left button event.  Select the row being clicked on.  If the
            # click is on the cruft name, show or hide its long description.
            # If the click the click is elsewhere do not handle it.  This
            # allows the toggle button event to be handled separately.
            x = int(event.x)
            y = int(event.y)
            time = event.time
            pathinfo = treeview.get_path_at_pos(x, y)
            if pathinfo is None:
                # The click was not in a cell, but we've handled it anyway.
                return True
            path, column, cell_x, cell_y = pathinfo
            if column in self.cruft_name_columns:
                treeview.set_cursor(path, column, False)
                self.toggle_long_description(treeview)
                return True
            else:
                # We are not handling this event so that the toggle button
                # handling can occur.
                return False
        elif event.button.button == 3:
            # Right button event.  Pop up the select/deselect all menu.
            treeview.grab_focus()
            x = int(event.x)
            y = int(event.y)
            time = event.time
            pathinfo = treeview.get_path_at_pos(x, y)
            if pathinfo is not None:
                path, column, cell_x, cell_y = pathinfo
                treeview.set_cursor(path, column, False)
            menu = self.popup_menus[treeview]
            try:
                menu.popup_for_device(None, None, None, None, None,
                        event.button.button, time)
            except AttributeError:
                # popup_for_device() is introspection safe, but only exists in
                # GTK3. popup() isn't introspectable, so in GTK 2 we need to
                # disable the popup menu functionality
                log.warning('popup menu not supported when using GTK2')
            return True
        else:
            # No other events are handled by us.
            return False

    # The actual event handler is totally generic.  Alias it to names
    # recognized by the automatic event binding scheme.
    on_unused_treeview_button_press_event = treeview_button_press_event
    on_optimize_treeview_button_press_event = treeview_button_press_event

    def treeview_size_allocate(self, treeview, *args):
        """Allocate space for the tree view and set wrap width.

        :param treeview: The TreeView
        :param args: Additional ignored positional arguments
        """
        # Get the rightmost of the two columns in the TreeView, i.e. the one
        # containing the text.
        column = treeview.get_column(1)
        name_cr = column.get_cells()[0]
        # Wrap to the entire width of the column.
        width = column.get_width()
        name_cr.set_property('wrap-width', width)

    on_unused_treeview_size_allocate = treeview_size_allocate
    on_optimize_treeview_size_allocate = treeview_size_allocate

    def on_sort_by_name_toggled(self, menuitem):
        """Reorder the crufts to be sorted by name or size."""
        if menuitem.get_active():
            self._sort_key = self._by_name
        else:
            self._sort_key = self._by_size
        self.sort_cruft()

    def on_about_menuitem_activate(self, *args):
        dialog = self.widgets['about_dialog']
        dialog.set_name(_('Computer Janitor'))
        dialog.set_version(__version__)
        dialog.show()
        dialog.run()
        dialog.hide()

    def on_show_previously_ignored_toggled(self, menuitem):
        """Show all cruft, even those being ignored.

        Normally, we only show cruft that wasn't explicitly ignored.  By
        toggling this menu item, the janitor can also display cruft that is
        marked as ignored on the dbus service.
        """
        show_ignored_cruft = menuitem.get_active()
        iter = self.store.get_iter_first()
        while iter:
            server_ignored = self.store.get_value(
                iter, ListStoreColumns.server_ignored)
            show = (show_ignored_cruft or not server_ignored)
            self.store.set_value(iter, ListStoreColumns.show, show)
            iter = self.store.iter_next(iter)

    def _edit_menuitem_set_select_state(self, enabled, *treeviews):
        """Set the specified state on the selected treeviews.

        :param enabled: The new state flag for all cruft.  True means enabled.
        :type enabled: bool
        :param treeviews: Sequence of `TreeView` object to set cruft state on.
        """
        for treeview in treeviews:
            self._treeview_foreach_set_set_state(treeview, enabled)

    def on_select_all_cruft_activate(self, menuitem):
        """Select all cruft, both package and other."""
        self._edit_menuitem_set_select_state(
            True,
            self.widgets['unused_treeview'],
            self.widgets['optimize_treeview'])

    def on_select_all_packages_activate(self, menuitem):
        """Select all cruft, both package and other."""
        self._edit_menuitem_set_select_state(
            True, self.widgets['unused_treeview'])

    def on_select_all_other_activate(self, menuitem):
        """Select all cruft, both package and other."""
        self._edit_menuitem_set_select_state(
            True, self.widgets['optimize_treeview'])

    def on_deselect_all_cruft_activate(self, menuitem):
        """Select all cruft, both package and other."""
        self._edit_menuitem_set_select_state(
            False,
            self.widgets['unused_treeview'],
            self.widgets['optimize_treeview'])

    def on_deselect_all_packages_activate(self, menuitem):
        """Select all cruft, both package and other."""
        self._edit_menuitem_set_select_state(
            False, self.widgets['unused_treeview'])

    def on_deselect_all_other_activate(self, menuitem):
        """Select all cruft, both package and other."""
        self._edit_menuitem_set_select_state(
            False, self.widgets['optimize_treeview'])

    def on_do_button_clicked(self, *args):
        """JFDI, well almost."""
        self.count = 0
        response = AreYouSure(self).verify()
        if not response:
            return
        # This can take a long time.  Make an asynchronous call to the dbus
        # service and arrange for it to occasionally provide us with status.
        # This isn't great ui, but OTOH, the package cruft cleaners themselves
        # don't provide much granularity, so there's little we can do anyway
        # without a major rewrite of the plugin architecture.
        self.working = True
        glib.timeout_add(150, self.pulse)
        # Make various ui elements insensitive.
        self.desensitize()
        cleanable = [cruft for cruft, ispkg in self.get_cleanable_cruft()]
        def error(exception):
            # XXX 2010-05-19 barry: This will (almost?) always be caused by
            # some other package manager already running, so if we get that
            # exception, display a (hopefully) useful dialog.  I'm not sure
            # it's very helpful to display the low-level apt exception in a
            # dialog.  `exception` will always be a DBusException; we won't
            # get the more useful PackageCleanupError.
            #
            # We use log.error() instead of log.exception() because we're not
            # in an exception handler.
            log.error('%s', exception)
            CleanupProblem(self).run()
            self.working = False
        def reply():
            pass
        # Make the asynchronous call because this can take a long time.  We'll
        # get status updates periodically.  Note however that even though this
        # is asynchronous, dbus still expects a response within a certain
        # amount of time.  We have no idea how long it will take to clean up
        # the cruft though, so just crank the timeout up to some insanely huge
        # number (of seconds).
        self.widgets['progressbar_status'].set_text('Authenticating...')
        self.janitord.clean(cleanable,
                            reply_handler=reply,
                            error_handler=error,
                            # If it takes longer than an hour, we're screwed.
                            timeout=3600)
