/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *              Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <dee.h>

typedef struct
{
  DeeTermList *terms;
  DeeModel    *model;
} Fixture;

static void setup    (Fixture *fix, gconstpointer data);
static void teardown (Fixture *fix, gconstpointer data);

static void
setup (Fixture *fix, gconstpointer data)
{
  fix->terms = g_object_new (DEE_TYPE_TERM_LIST, NULL);
  fix->model = dee_sequence_model_new (4,
                                       G_TYPE_STRING,
                                       G_TYPE_INT,
                                       G_TYPE_STRING,
                                       G_TYPE_UINT);

  dee_model_append (fix->model,
                    0, "Hello world",
                    1, 27,
                    2, "Three Danish characters... æøå?",
                    3, 68,
                    -1);
}

static void
teardown (Fixture *fix, gconstpointer data)
{
  g_object_unref (fix->terms);
  g_object_unref (fix->model);
  fix->terms = NULL;
  fix->model = NULL;
}

static void
test_key0 (Fixture *fix, gconstpointer data)
{
  DeeAnalyzer *analyzer;

  analyzer = dee_analyzer_new_for_key_column (0);
  analyzer->analyze (fix->model, dee_model_get_first_iter (fix->model),
                     fix->terms, analyzer->userdata);

  g_assert_cmpint (dee_term_list_num_terms (fix->terms), ==, 1);
  g_assert_cmpstr ("Hello world", ==, dee_term_list_get_term (fix->terms, 0));
}

static void
test_key2 (Fixture *fix, gconstpointer data)
{
  DeeAnalyzer *analyzer;

  analyzer = dee_analyzer_new_for_key_column (2);
  analyzer->analyze (fix->model, dee_model_get_first_iter (fix->model),
                     fix->terms, analyzer->userdata);

  g_assert_cmpint (dee_term_list_num_terms (fix->terms), ==, 1);
  g_assert_cmpstr ("Three Danish characters... æøå?", ==, dee_term_list_get_term (fix->terms, 0));
}

static void
test_full_text0 (Fixture *fix, gconstpointer data)
{
  DeeAnalyzer *analyzer;

  analyzer = dee_analyzer_new_for_full_text_column (0);
  analyzer->analyze (fix->model, dee_model_get_first_iter (fix->model),
                     fix->terms, analyzer->userdata);

  g_assert_cmpint (dee_term_list_num_terms (fix->terms), ==, 2);
  g_assert_cmpstr ("hello", ==, dee_term_list_get_term (fix->terms, 0));
  g_assert_cmpstr ("world", ==, dee_term_list_get_term (fix->terms, 1));
}

static void
test_full_text2 (Fixture *fix, gconstpointer data)
{
  DeeAnalyzer *analyzer;

  analyzer = dee_analyzer_new_for_full_text_column (2);
  analyzer->analyze (fix->model, dee_model_get_first_iter (fix->model),
                     fix->terms, analyzer->userdata);

  g_assert_cmpint (dee_term_list_num_terms (fix->terms), ==, 4);
  g_assert_cmpstr ("three", ==, dee_term_list_get_term (fix->terms, 0));
  g_assert_cmpstr ("danish", ==, dee_term_list_get_term (fix->terms, 1));
  g_assert_cmpstr ("characters", ==, dee_term_list_get_term (fix->terms, 2));
  g_assert_cmpstr ("æøå", ==, dee_term_list_get_term (fix->terms, 3));
}

static void
test_int (Fixture *fix, gconstpointer data)
{
  DeeAnalyzer *analyzer;

  analyzer = dee_analyzer_new_for_int_column (1);
  analyzer->analyze (fix->model, dee_model_get_first_iter (fix->model),
                     fix->terms, analyzer->userdata);

  g_assert_cmpint (dee_term_list_num_terms (fix->terms), ==, 1);
  g_assert_cmpstr ("27", ==, dee_term_list_get_term (fix->terms, 0));
}

static void
test_uint (Fixture *fix, gconstpointer data)
{
  DeeAnalyzer *analyzer;

  analyzer = dee_analyzer_new_for_uint_column (3);
  analyzer->analyze (fix->model, dee_model_get_first_iter (fix->model),
                     fix->terms, analyzer->userdata);

  g_assert_cmpint (dee_term_list_num_terms (fix->terms), ==, 1);
  g_assert_cmpstr ("68", ==, dee_term_list_get_term (fix->terms, 0));
}

void
test_analyzers_create_suite (void)
{
#define DOMAIN "/Index/Analyzers"

  g_test_add (DOMAIN"/Key0", Fixture, 0,
              setup, test_key0, teardown);
  g_test_add (DOMAIN"/Key2", Fixture, 0,
                setup, test_key2, teardown);
  g_test_add (DOMAIN"/FullText0", Fixture, 0,
                setup, test_full_text0, teardown);
  g_test_add (DOMAIN"/FullText2", Fixture, 0,
                  setup, test_full_text2, teardown);
  g_test_add (DOMAIN"/Int", Fixture, 0,
                    setup, test_int, teardown);
  g_test_add (DOMAIN"/UInt", Fixture, 0,
                    setup, test_uint, teardown);
}
