/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *              Neil Jagdish Patel <neil.patel@canonical.com>
 *              Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <dee.h>

typedef struct
{
  DeeModel *model;

} RowsFixture;

static void seq_rows_setup    (RowsFixture *fix, gconstpointer data);
static void seq_rows_teardown (RowsFixture *fix, gconstpointer data);
static void proxy_rows_setup    (RowsFixture *fix, gconstpointer data);
static void proxy_rows_teardown (RowsFixture *fix, gconstpointer data);

static void test_rows_allocation (RowsFixture *fix, gconstpointer data);
static void test_rows_clear      (RowsFixture *fix, gconstpointer data);
static void test_insert_at_pos   (RowsFixture *fix, gconstpointer data);
static void test_insert_at_iter  (RowsFixture *fix, gconstpointer data);
static void test_prepend         (RowsFixture *fix, gconstpointer data);
static void test_no_transfer     (RowsFixture *fix, gconstpointer data);

void
test_model_rows_create_suite (void)
{
#define SEQ_DOMAIN "/Model/Sequence/Rows"
#define PROXY_DOMAIN "/Model/Proxy/Rows"

  g_test_add (SEQ_DOMAIN"/Allocation", RowsFixture, 0,
              seq_rows_setup, test_rows_allocation, seq_rows_teardown);
  g_test_add (PROXY_DOMAIN"/Allocation", RowsFixture, 0,
              proxy_rows_setup, test_rows_allocation, proxy_rows_teardown);
              
  g_test_add (SEQ_DOMAIN"/Clear", RowsFixture, 0,
              seq_rows_setup, test_rows_clear, seq_rows_teardown);
  g_test_add (PROXY_DOMAIN"/Clear", RowsFixture, 0,
              proxy_rows_setup, test_rows_clear, proxy_rows_teardown);
              
  g_test_add (SEQ_DOMAIN"/InsertAtPos", RowsFixture, 0,
              seq_rows_setup, test_insert_at_pos, seq_rows_teardown);
  g_test_add (PROXY_DOMAIN"/InsertAtPos", RowsFixture, 0,
              proxy_rows_setup, test_insert_at_pos, proxy_rows_teardown);
              
  g_test_add (SEQ_DOMAIN"/InsertAtIter", RowsFixture, 0,
              seq_rows_setup, test_insert_at_iter, seq_rows_teardown);
  g_test_add (PROXY_DOMAIN"/InsertAtIter", RowsFixture, 0,
              proxy_rows_setup, test_insert_at_iter, proxy_rows_teardown);
  
  g_test_add (SEQ_DOMAIN"/Prepend", RowsFixture, 0,
              seq_rows_setup, test_prepend, seq_rows_teardown);
  g_test_add (PROXY_DOMAIN"/Prepend", RowsFixture, 0,
              proxy_rows_setup, test_prepend, proxy_rows_teardown);
  
  g_test_add (SEQ_DOMAIN"/NoTransfer", RowsFixture, 0,
              seq_rows_setup, test_no_transfer, seq_rows_teardown);
  g_test_add (PROXY_DOMAIN"/NoTransfer", RowsFixture, 0,
              proxy_rows_setup, test_no_transfer, proxy_rows_teardown);
              
}

static void
seq_rows_setup (RowsFixture *fix, gconstpointer data)
{
  fix->model = dee_sequence_model_new (2,
                                       G_TYPE_INT,
                                       G_TYPE_STRING);

  g_assert (DEE_IS_SEQUENCE_MODEL (fix->model));
}

static void
seq_rows_teardown (RowsFixture *fix, gconstpointer data)
{
  g_object_unref (fix->model);
  fix->model = NULL;
}

static void
proxy_rows_setup (RowsFixture *fix, gconstpointer data)
{
  seq_rows_setup (fix, data);
  fix->model = g_object_new (DEE_TYPE_PROXY_MODEL,
                             "back-end", fix->model,
                             NULL);
  
  g_assert (DEE_IS_PROXY_MODEL (fix->model));
}

static void
proxy_rows_teardown (RowsFixture *fix, gconstpointer data)
{
  g_object_unref (fix->model);
  fix->model = NULL;
}

static void
test_rows_allocation (RowsFixture *fix, gconstpointer data)
{
  g_assert (DEE_IS_MODEL (fix->model));
}

static gint n_clear_sigs = 0;

static void
on_clear_row_removed (DeeModel *model, DeeModelIter *iter)
{
  n_clear_sigs++;
}

static void
test_rows_clear (RowsFixture *fix, gconstpointer data)
{
  gint i;

  for (i = 0; i < 1000; i++)
    {
      dee_model_append (fix->model, 0, 10, 1, "Rooney", -1);
    }

  g_assert_cmpint (1000, ==, dee_model_get_n_rows (fix->model));

  g_signal_connect (fix->model, "row-removed",
                    G_CALLBACK (on_clear_row_removed), NULL);

  n_clear_sigs = 0;
  dee_model_clear (fix->model);

  g_assert_cmpint (0, ==, dee_model_get_n_rows (fix->model));
  g_assert_cmpint (1000, ==, n_clear_sigs);
}

static void
test_insert_at_pos (RowsFixture *fix, gconstpointer data)
{
  DeeModelIter *iter;
  gint           i;
  gchar         *str;

  dee_model_append (fix->model, 0, 10, 1, "Rooney", -1);
  dee_model_append (fix->model, 0, 10, 1, "Rooney", -1);
  g_assert_cmpint (2, ==, dee_model_get_n_rows (fix->model));

  dee_model_insert (fix->model, 1, 0, 27, 1, "Not Rooney", -1);
  g_assert_cmpint (3, ==, dee_model_get_n_rows (fix->model));

  iter = dee_model_get_first_iter (fix->model);
  dee_model_get (fix->model, iter, 0, &i, 1, &str, -1);
  g_assert_cmpint (10, ==, i);
  g_assert_cmpstr (str, ==, "Rooney");
  g_free (str);

  iter = dee_model_next (fix->model, iter);
  dee_model_get (fix->model, iter, 0, &i, 1, &str, -1);
  g_assert_cmpint (27, ==, i);
  g_assert_cmpstr (str, ==, "Not Rooney");
  g_free (str);

  iter = dee_model_next (fix->model, iter);
  dee_model_get (fix->model, iter, 0, &i, 1, &str, -1);
  g_assert_cmpint (10, ==, i);
  g_assert_cmpstr (str, ==, "Rooney");
  g_free (str);
}

static void
test_insert_at_iter (RowsFixture *fix, gconstpointer data)
{
  DeeModelIter *iter;
  gint           i;
  gchar         *str;

  dee_model_append (fix->model, 0, 10, 1, "Rooney", -1);
  dee_model_append (fix->model, 0, 10, 1, "Rooney", -1);
  g_assert_cmpint (2, ==, dee_model_get_n_rows (fix->model));

  iter = dee_model_get_first_iter (fix->model);
  iter = dee_model_next (fix->model, iter);
  dee_model_insert_before (fix->model, iter, 0, 27, 1, "Not Rooney", -1);
  g_assert_cmpint (3, ==, dee_model_get_n_rows (fix->model));

  iter = dee_model_get_first_iter (fix->model);
  dee_model_get (fix->model, iter, 0, &i, 1, &str, -1);
  g_assert_cmpint (10, ==, i);
  g_assert_cmpstr (str, ==, "Rooney");
  g_free (str);

  iter = dee_model_next (fix->model, iter);
  dee_model_get (fix->model, iter, 0, &i, 1, &str, -1);
  g_assert_cmpint (27, ==, i);
  g_assert_cmpstr (str, ==, "Not Rooney");
  g_free (str);

  iter = dee_model_next (fix->model, iter);
  dee_model_get (fix->model, iter, 0, &i, 1, &str, -1);
  g_assert_cmpint (10, ==, i);
  g_assert_cmpstr (str, ==, "Rooney");
  g_free (str);
}

static void
test_prepend (RowsFixture *fix, gconstpointer data)
{
  DeeModelIter *iter;
  gint           i;
  gchar         *str;

  dee_model_prepend (fix->model, 0, 11, 1, "Mid", -1);
  dee_model_append (fix->model, 0, 12, 1, "Last", -1);
  dee_model_prepend (fix->model, 0, 10, 1, "First", -1);
  g_assert_cmpint (3, ==, dee_model_get_n_rows (fix->model));

  iter = dee_model_get_first_iter (fix->model);
  dee_model_get (fix->model, iter, 0, &i, 1, &str, -1);
  g_assert_cmpint (10, ==, i);
  g_assert_cmpstr (str, ==, "First");
  g_free (str);

  iter = dee_model_next (fix->model, iter);
  dee_model_get (fix->model, iter, 0, &i, 1, &str, -1);
  g_assert_cmpint (11, ==, i);
  g_assert_cmpstr (str, ==, "Mid");
  g_free (str);

  iter = dee_model_next (fix->model, iter);
  dee_model_get (fix->model, iter, 0, &i, 1, &str, -1);
  g_assert_cmpint (12, ==, i);
  g_assert_cmpstr (str, ==, "Last");
  g_free (str);
}

/* Make sure the the memory allocated for strings is not reffed inside
 * the model */
static void
test_no_transfer (RowsFixture *fix, gconstpointer data)
{
  DeeModelIter *iter;
  gchar         *orig, *str;

  orig = g_strdup ("test");

  dee_model_append (fix->model, 1, orig, -1);
  g_assert_cmpint (1, ==, dee_model_get_n_rows (fix->model));

  /* This should work, of course */
  iter = dee_model_get_first_iter (fix->model);
  dee_model_get (fix->model, iter, 1, &str, -1);
  g_assert_cmpstr (str, ==, "test");
  g_free (str);
  
  /* Modify orig in place and assert it doesn't affect the model */
  orig[0] = 'P';
  dee_model_get (fix->model, iter, 1, &str, -1);
  g_assert_cmpstr (str, ==, "test");
  g_free (str);
  
  /* Now free orig and make sure we can still read from the model */
  g_free (orig);
  dee_model_get (fix->model, iter, 1, &str, -1);
  g_assert_cmpstr (str, ==, "test");
  g_free (str);
}
