/* commandfuncs.cpp
 * functions invoked by user keypresses
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 1999-2005 Matthew Hiller, Adam Tee
 */

#include <string.h>
#include "commandfuncs.h"
#include "calculatepositions.h"
#include "chordops.h"
#include "contexts.h"
#include "draw.h"
#include "measureops.h"
#include "midi.h"
#include "objops.h"
#include "moveviewport.h"
#include "selectops.h"
#include "staffops.h"
#include "utils.h"
#include "tupletops.h"
#include "graceops.h"
#include "instrumentname.h"
#include "file.h"
#include "exportmudela.h"
#include "exportxml.h"

/**
 * Macro to get the current mudelaobject
 */
#define declarecurmudelaobj \
mudelaobject *curmudelaobj = \
       (mudelaobject *) ( (si->currentobject && si->currentobject->data) ? \
       ((((mudelaobject *)si->currentobject->data)->type == TUPCLOSE) ? \
       si->currentobject->prev->data : si->currentobject->data) : NULL)


/**
 * Helper function for calculating the 
 * beam and stem direction
 */
static void
beamandstemdirhelper (struct scoreinfo *si)
{
  calculatebeamsandstemdirs
    ((objnode *) si->currentmeasure->data, &(si->curmeasureclef),
     &(si->cursortime1), &(si->cursortime2),
     &(si->curmeasure_stem_directive));
}

/**
 * Set si->current* variables
 *
 */
void
setcurrents (struct scoreinfo *si)
{
  si->currentmeasure =
    g_list_nth (firstmeasurenode (si->currentstaff),
		si->currentmeasurenum - 1);
  si->cursor_x = 0;
  si->currentobject = (objnode *) si->currentmeasure->data;
  if (si->currentobject)
    si->cursor_appending = FALSE;
  else
    si->cursor_appending = TRUE;
  calcmarkboundaries (si);
}

/**
 * Push the score to the left off the 
 * displayed portion
 */
void
nudgerightward (struct scoreinfo *si)
{
  set_rightmeasurenum (si);
  while (si->currentmeasurenum > si->rightmeasurenum)
    {
      si->leftmeasurenum++;
      set_rightmeasurenum (si);
    }
  find_leftmost_allcontexts (si);
  update_hscrollbar (si);
}

/**
 * Push the score upwards off the displayed
 * portion
 */
void
nudge_downward (struct scoreinfo *si)
{
  set_bottom_staff (si);

  while (si->currentstaffnum > si->bottom_staff)
    {
      si->top_staff++;
      set_bottom_staff (si);
    }
  update_vscrollbar (si);
}

/**
 * Set the width of the working scorearea
 */
void
set_width_to_work_with (struct scoreinfo *si)
{
  si->widthtoworkwith
    = (si->scorearea->allocation.width
       - (RIGHT_MARGIN + KEY_MARGIN + si->maxkeywidth + SPACE_FOR_TIME));
}

/**
 * Change the basic width of each measure
 */
void
adjustmeasurewidth (struct scoreinfo *si, gint amount)
{
  si->measurewidth += amount;
  if (si->measurewidth < 10)
    si->measurewidth = 10;
  if (si->widthtoworkwith < si->measurewidth + SPACE_FOR_BARLINE)
    si->measurewidth = si->widthtoworkwith - SPACE_FOR_BARLINE;
  find_xes_in_all_measures (si);
  nudgerightward (si);
}

/**
 * Determines whether the closer jump will be up or down 
 */
gint
jumpcursor (gint cursor_y, gint fromnote, gint tonote)
{
  int distance;

  distance = (tonote - fromnote + 7) % 7;
  if (distance <= 3)		/* an upward jump is good */
    return cursor_y + distance;
  else				/* jump down */
    return cursor_y - 7 + distance;
}

/**
 *  General function for inserting a mudelaobject
 *  into the score
 */
void
object_insert (struct scoreinfo *si, mudelaobject * mudela_obj_new)
{
  struct undo_data *undo = (struct undo_data *)
    g_malloc (sizeof (struct undo_data));



  declarecurmudelaobj;

  /* First, check to see if the operation would add something before an
   * indicator of a time signature change. This would be bad, so don't
   * allow it to happen */
  if (curmudelaobj && curmudelaobj->type == TIMESIG && !si->cursor_appending)
    {
      si->cursor_x++;
      if (si->currentobject->next)
	si->currentobject = si->currentobject->next;
      else
	si->cursor_appending = TRUE;
    }

  si->currentmeasure->data = g_list_insert
    ((objnode *) si->currentmeasure->data, mudela_obj_new, si->cursor_x);

  /* update undo information */
  undo->position = si->cursor_x;
  undo->measurenum = si->currentmeasurenum;
  undo->staffnum = si->currentstaffnum;
  undo->object = mudela_obj_new;
  undo->action = ACTION_INSERT;

  g_print ("Action %d\n", undo->action);

  si->cursor_x++;
  if (si->cursor_appending)
    si->currentobject = g_list_last ((objnode *) si->currentmeasure->data);
  else
    si->currentobject = g_list_nth ((objnode *) si->currentmeasure->data,
				    si->cursor_x);

  update_undo_info (si, undo);
  displayhelper (si);

  si->markstaffnum = 0;
}

/**
 *  Change the y position of each staff
 */
void
adjuststaffheight (struct scoreinfo *si, gint amount)
{
  si->staffspace += amount;
  if (si->staffspace < 2 * STAFF_HEIGHT)
    si->staffspace = 2 * STAFF_HEIGHT;
  nudge_downward (si);
}

void
measureleft (struct scoreinfo *si)
{
  if (!si->cursor_x && si->currentmeasure->prev)
    {
      si->currentmeasurenum--;
      isoffleftside (si);
    }
  setcurrents (si);
}

void
measureright (struct scoreinfo *si)
{
  if (si->currentmeasure->next)
    {
      si->currentmeasurenum++;
      isoffrightside (si);
      setcurrents (si);
    }
}

void
staffup (struct scoreinfo *si)
{
  if (si->currentstaff->prev)
    {
      si->currentstaffnum--;
      si->currentstaff = si->currentstaff->prev;
      setcurrentprimarystaff (si);
      setcurrents (si);
      move_viewport_up (si);
    }
}

void
staffdown (struct scoreinfo *si)
{
  if (si->currentstaff->next)
    {
      si->currentstaffnum++;
      si->currentstaff = si->currentstaff->next;
      setcurrentprimarystaff (si);
      setcurrents (si);
      move_viewport_down (si);
    }
}

void
cursorleft (struct scoreinfo *si)
{
  if (!si->cursor_x)
    {
      /* also the only situation where si->currentobject == NULL */
      if (si->currentmeasure->prev)
	{
	  /* Go to end of preceding measure */
	  si->cursor_appending = TRUE;
	  si->currentmeasure = si->currentmeasure->prev;
	  si->currentmeasurenum--;
	  isoffleftside (si);
	  si->currentobject =
	    g_list_last ((objnode *) si->currentmeasure->data);
	  /* The preceding statement will set currentobject to
	   * NULL if appropriate */
	  si->cursor_x = g_list_length ((objnode *) si->currentmeasure->data);
	  /* Despite appearances, there is not an off-by-one error in the
	   * preceding command */
	}
    }
  else if (si->cursor_appending)
    {
      /* Can back off from appending */
      si->cursor_appending = FALSE;
      si->cursor_x--;
    }
  else
    {
      /* Can go back in the measure */
      si->currentobject = si->currentobject->prev;
      si->cursor_x--;
    }
  calcmarkboundaries (si);
}

void
cursorright (struct scoreinfo *si)
{
  if (si->cursor_appending && si->currentmeasure->next)
    {
      /* Go to the next measure */
      si->currentmeasure = si->currentmeasure->next;
      si->currentmeasurenum++;
      isoffrightside (si);
      si->currentobject = (objnode *) si->currentmeasure->data;
      si->cursor_x = 0;
      if (si->currentobject)
	si->cursor_appending = FALSE;
    }
  else if (si->currentobject)
    {
      /* See if we should go to appending position. If not, go to the
       * next note (if possible) */
      if (!si->cursor_appending && !si->currentobject->next)
	{
	  /* Go to appending position */
	  si->cursor_appending = TRUE;
	  si->cursor_x++;
	}
      else if (si->currentobject->next)
	{
	  si->currentobject = si->currentobject->next;
	  si->cursor_x++;
	}
    }
  calcmarkboundaries (si);
}

void
cursorup (struct scoreinfo *si)
{
  si->cursor_y++;
  si->staffletter_y = (si->staffletter_y + 1) % 7;
}

void
cursordown (struct scoreinfo *si)
{
  si->cursor_y--;
  si->staffletter_y = (si->staffletter_y + 6) % 7;
}

void
shiftcursor (struct scoreinfo *si, gint note_value)
{
  gint oldstaffletter_y = si->staffletter_y;

  si->staffletter_y = note_value;
  si->cursor_y = jumpcursor (si->cursor_y, oldstaffletter_y,
			     si->staffletter_y);
}


void
nextmeasure (struct scoreinfo *si)
{
  gboolean next_measure;

  /* First, check to see if the insertion'll cause the cursor to
   * jump to the next measure. (Denemo will implicitly create it
   * if it doesn't exist already.) */

  next_measure = si->cursoroffend && si->cursor_appending
    && (!si->currentmeasure->next || !si->currentmeasure->next->data);
  if (next_measure)
    {
      if (!si->currentmeasure->next)
	/* Add a measure and make it currentmeasure */
	si->currentmeasure = addmeasures (si, si->currentmeasurenum, 1);
      else
	si->currentmeasure = si->currentmeasure->next;
      /* Now the stuff that needs to be done for each case */
      si->currentmeasurenum++;
      si->currentobject = (objnode *) si->currentmeasure->data;
      si->cursor_x = 0;
      memcpy (si->cursoraccs, si->nextmeasureaccs, SEVENGINTS);
      memcpy (si->curmeasureaccs, si->nextmeasureaccs, SEVENGINTS);
      si->curmeasureclef = si->cursorclef;
    }

}


void
insertchord (struct scoreinfo *si, gint duration, gboolean rest)
{

  mudelaobject *mudela_obj_new;
  int prognum;
  staff *curstaffstruct;

  /* First, check to see if the insertion'll cause the cursor to
   * jump to the next measure. (Denemo will implicitly create it
   * if it doesn't exist already.) */

  nextmeasure (si);

  /* Now actually create the chord */
  mudela_obj_new = newchord (duration, 0);
  if (si->mode == INPUTBLANK && (rest != TRUE))
    {
      addtone (mudela_obj_new, si->cursor_y,
	       si->cursoraccs[si->staffletter_y], si->cursorclef);
      mudela_obj_new->isinvisible = TRUE;
    }
  else if (si->mode == INPUTBLANKREST)
    mudela_obj_new->isinvisible = TRUE;
  else if (si->mode == INPUTNORMAL && (rest != TRUE))
    addtone (mudela_obj_new, si->cursor_y, si->cursoraccs[si->staffletter_y],
	     si->cursorclef);
  else
    si->mode = INPUTREST;



  if (si->is_grace_mode)
    ((chord *) mudela_obj_new->object)->is_grace = TRUE;


  /* Insert the new note into the score.  Note that while we may have
     added a measure above, object_insert will invoke nudgerightward,
     which will in turn invoke update_hscrollbar, so we
     don't need to invoke that here.  */

  object_insert (si, mudela_obj_new);
  curstaffstruct = (staff *) si->currentstaff->data;
  prognum = select_program (curstaffstruct->midi_instrument->str);
  playnotes (si->prefs->immediateplayback, *(chord *) mudela_obj_new->object,
	     prognum);
}

void
inserttuplet (struct scoreinfo *si, gint duration)
{
  mudelaobject *mudela_obj_new;

  nextmeasure (si);
  switch (duration)
    {
    case 0:
    case 1:
    case 2:
      mudela_obj_new = newtupopen (2, 3);
      tupletchangedialog (mudela_obj_new, si->scorearea);
      break;
    case 3:
      mudela_obj_new = newtupopen (2, 3);
      break;
    case 5:
      mudela_obj_new = newtupopen (4, 5);
      break;
    case 6:
      mudela_obj_new = newtupopen (4, 6);
      break;
    case 7:
      if (si->cursortime2 == 8)
	{
	  mudela_obj_new = newtupopen (8, 7);
	}
      else
	mudela_obj_new = newtupopen (4, 7);
      break;
    case 9:
      mudela_obj_new = newtupopen (8, 9);
      break;
    default:
      mudela_obj_new = newtupopen (2, 3);
      break;
    }
  g_print ("Cursor pos %d (Before tup open)\n", si->cursor_x);
  object_insert (si, mudela_obj_new);
  g_print ("Cursor pos %d (After tup open, before tup close)\n",
	   si->cursor_x);
  /* Add the closing bracket */
  object_insert (si, newtupclose ());
  g_print ("Cursor pos %d (After tup close)\n", si->cursor_x);
  si->cursor_x--;
  g_print ("Cursor pos %d( After move back)\n", si->cursor_x);
  /* And clean up */
  /*si->currentobject = 
     g_list_nth ((objnode *)si->currentmeasure->data, si->cursor_x); */
  si->currentobject = si->currentobject->prev;
  si->cursor_appending = FALSE;
}

void
insertgrace (struct scoreinfo *si)
{
  mudelaobject *mudela_obj_new;
  nextmeasure (si);


  mudela_obj_new = newgracestart ();

  object_insert (si, mudela_obj_new);

  object_insert (si, newgraceend ());
  si->cursor_x--;
  si->currentobject =
    g_list_nth ((objnode *) si->currentmeasure->data, si->cursor_x);
  si->cursor_appending = FALSE;
  si->is_grace_mode = TRUE;
}

void
changeduration (struct scoreinfo *si, gint duration)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      changedur (curmudelaobj, duration, 0);
      displayhelper (si);
    }
}

void
tonechange (struct scoreinfo *si, gboolean remove)
{
  declarecurmudelaobj;
  int prognum;
  staff *curstaffstruct;
  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      if (remove == TRUE)
	removetone (curmudelaobj, si->cursor_y, si->cursorclef);
      else
	addtone (curmudelaobj, si->cursor_y,
		 si->cursoraccs[si->staffletter_y], si->cursorclef);

      //displayhelper (si);
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      curstaffstruct = (staff *) si->currentstaff->data;
      prognum = select_program (curstaffstruct->midi_instrument->str);
      playnotes (si->prefs->immediateplayback,
		 *(chord *) curmudelaobj->object, prognum);
    }
}

void
displayhelper (struct scoreinfo *si)
{

  beamandstemdirhelper (si);
  showwhichaccidentals ((objnode *) si->currentmeasure->data,
			si->curmeasurekey, si->curmeasureaccs);
  find_xes_in_measure (si, si->currentmeasurenum, si->cursortime1,
		       si->cursortime2);
  nudgerightward (si);
  si->haschanged = TRUE;
  gtk_widget_draw (si->scorearea, NULL);
}

/* Change the enharmonic shift of the tone closest to the cursor. The
 * function double-checks to be sure that it's passed a chord, though
 * at the moment (28 July 2000), it probably doesn't strictly need to */

void
changeenshift (struct scoreinfo *si, gint amount)
{
  declarecurmudelaobj;
  int prognum;
  staff *curstaffstruct;
  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      shiftpitch (curmudelaobj, si->cursor_y, amount == 1);
      showwhichaccidentals ((objnode *) si->currentmeasure->data,
			    si->curmeasurekey, si->curmeasureaccs);
      find_xes_in_measure (si, si->currentmeasurenum, si->cursortime1,
			   si->cursortime2);
      nudgerightward (si);
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      si->haschanged = TRUE;
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      curstaffstruct = (staff *) si->currentstaff->data;
      prognum = select_program (curstaffstruct->midi_instrument->str);
      playnotes (si->prefs->immediateplayback,
		 *(chord *) curmudelaobj->object, prognum);
    }
}

/**
 * Change the stemdirection of the current chord object
 * by a given amount
 */
void
change_stem_directive (struct scoreinfo *si, enum stemdirections amount)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == STEMDIRECTIVE)
    {
      switch (amount)
	{
	case DENEMO_STEMDOWN:
	  ((stemdirective *) curmudelaobj->object)->type = DENEMO_STEMDOWN;
	  break;
	case DENEMO_STEMUP:
	  ((stemdirective *) curmudelaobj->object)->type = DENEMO_STEMUP;
	  break;
	default:
	  ((stemdirective *) curmudelaobj->object)->type = DENEMO_STEMBOTH;
	  break;
	}

      beamsandstemdirswholestaff ((staff *) si->currentstaff->data);
      find_xes_in_all_measures (si);
      nudgerightward (si);
      si->haschanged = TRUE;
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      gtk_widget_draw (si->scorearea, NULL);
    }
}

/**
 * Change the number of dots on the current chord
 */
void
changedots (struct scoreinfo *si, gint amount)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      changenumdots (curmudelaobj, amount);
      displayhelper (si);
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
    }
}

/**
 *  Insert measure into the score at the current position
 *
 */
void
insertmeasures (struct scoreinfo *si, gint number)
{
  si->currentmeasure = addmeasures (si, si->currentmeasurenum - 1, number);
  si->cursor_x = 0;
  si->cursor_appending = TRUE;
  si->currentobject = NULL;
  set_rightmeasurenum (si);
  si->haschanged = TRUE;
  si->markstaffnum = 0;
  calcmarkboundaries (si);
  update_hscrollbar (si);
}

/**
 * Add measure to the end of the score
 *
 */
void
appendmeasures (struct scoreinfo *si, gint number)
{
  addmeasures (si, g_list_length (firstmeasurenode (si->currentstaff)),
	       number);
  /* Reset these two variables because si->currentmeasure and
   * si->currentobject may now be pointing to dead data */
  si->currentmeasure =
    g_list_nth (firstmeasurenode (si->currentstaff),
		si->currentmeasurenum - 1);
  si->currentobject =
    g_list_nth ((objnode *) si->currentmeasure->data,
		si->cursor_x - (si->cursor_appending == TRUE));
  set_rightmeasurenum (si);
  si->haschanged = TRUE;
  update_hscrollbar (si);
}

void
delete_staff_before (GtkAction * action, gpointer data)
{
  delete_staff (data, BEFORE);
}

void
delete_staff_after (GtkAction * action, gpointer data)
{
  delete_staff (data, AFTER);
}

void
delete_staff_current (GtkAction * action, gpointer data)
{
  delete_staff (data, 9);
}

void
delete_staff (gpointer callback_data, guint callback_action)
{
  struct scoreinfo *si = (struct scoreinfo *) callback_data;
  g_print ("Current staff num %d, callback_action %d", si->currentstaffnum,
	   callback_action);
  switch (callback_action)
    {
    case BEFORE:
      if (si->currentstaffnum != 1)
	deletestaff (si, si->currentstaffnum - 1);
      break;
    case AFTER:
      deletestaff (si, si->currentstaffnum + 1);
      break;
    default:
      deletestaff (si, si->currentstaffnum);
      break;
    }

}

/**
 * Delete staff from the score
 */
void
deletestaff (struct scoreinfo *si, gint pos)
{
  if (g_list_length (si->thescore) > 1)
    {
      removestaff (si, si->currentstaffnum - 1, 1);
      setcurrents (si);
      find_xes_in_all_measures (si);
      nudgerightward (si);
      si->haschanged = TRUE;
      si->markstaffnum = 0;
    }
}

/**
 * Delete measure from the score
 *
 * TODO remove measure from current staff 
 * rather than the entire score
 */
void
deletemeasure (struct scoreinfo *si)
{
  si->currentmeasure = removemeasures (si, si->currentmeasurenum - 1, 1);
  isoffleftside (si);
  /* In case that was the last measure we just deleted, which'd cause
   * the current measure to be the left of what's displayed */
  nudgerightward (si);
  setcurrents (si);
  si->haschanged = TRUE;
  si->markstaffnum = 0;
  update_hscrollbar (si);
}

/**
 * Remove current object from the score
 *
 */
static void
remove_object (measurenode * cur_measure, objnode * cur_objnode)
{
  cur_measure->data = g_list_remove_link ((objnode *) cur_measure->data,
					  cur_objnode);
  freeobject ((mudelaobject *) cur_objnode->data);
  g_list_free_1 (cur_objnode);
}


/**
 * Reset the cursor stats 
 */
static void
reset_cursor_stats (struct scoreinfo *si)
{
  si->currentobject = g_list_nth ((objnode *) si->currentmeasure->data,
				  si->cursor_x);
  if (!si->currentobject)
    {
      si->currentobject = g_list_last ((objnode *) si->currentmeasure->data);
      si->cursor_appending = TRUE;
    }
}

/**
 * Helper to remove the current object an reset cursor stats
 */
static void
delete_object_helper (struct scoreinfo *si)
{
  remove_object (si->currentmeasure, si->currentobject);
  reset_cursor_stats (si);
}

/**
 * Function to delete object from the score
 *
 */
void
deleteobject (struct scoreinfo *si)
{
  declarecurmudelaobj;
  staffnode *curstaff;
  measurenode *curmeasure;
  if (si->undo_redo_mode == 0)
    {
      struct undo_data undo;
      undo.action = ACTION_DELETE;
      undo.staffnum = si->currentstaffnum;
      undo.measurenum = si->currentmeasurenum;
      undo.position = si->cursor_x;
      undo.object = curmudelaobj;
      update_undo_info (si, &undo);
    }
  else if (si->undo_redo_mode == 1)
    {
      struct redo_data redo;
      redo.action = ACTION_INSERT;
      redo.staffnum = si->currentstaffnum;
      redo.measurenum = si->currentmeasurenum;
      redo.position = si->cursor_x;
      redo.object = curmudelaobj;
      update_redo_info (si, &redo);
    }
  si->undo_redo_mode = 0;	//Reset undo_redo_mode to it's default


  if (!si->cursor_appending)
    {
      switch (curmudelaobj->type)
	{
	case CHORD:
	  delete_object_helper (si);
	  displayhelper (si);
	  break;
	case TUPOPEN:
	case TUPCLOSE:
	  /* TODO - add code that will automatically delete a tupbracket's
	   * corresponding bracket */

	  delete_object_helper (si);
	  displayhelper (si);
	  break;
	case CLEF:
	  delete_object_helper (si);
	  fixnoteheights ((staff *) si->currentstaff->data);
	  beamsandstemdirswholestaff ((staff *) si->currentstaff->data);
	  find_xes_in_all_measures (si);
	  break;
	case KEYSIG:
	  /* Doesn't automatically delete sibling key signatures, though
	   * I probably will have it do so soon */
	  delete_object_helper (si);
	  beamsandstemdirswholestaff ((staff *) si->currentstaff->data);
	  showwhichaccidentalswholestaff ((staff *) si->currentstaff->data);
	  find_xes_in_all_measures (si);
	  break;
	case TIMESIG:
	  /* For time signatures, deletion is linked to all
	   * the staffs on the score */
	  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
	    {
	      curmeasure = g_list_nth (firstmeasurenode (curstaff),
				       si->currentmeasurenum - 1);
	      remove_object (curmeasure, (objnode *) curmeasure->data);
	      beamsandstemdirswholestaff ((staff *) curstaff->data);
	    }
	  reset_cursor_stats (si);
	  find_xes_in_all_measures (si);
	  break;
	case STEMDIRECTIVE:
	  delete_object_helper (si);
	  beamsandstemdirswholestaff ((staff *) si->currentstaff->data);
	  find_xes_in_all_measures (si);
	  break;
	case DYNAMIC:
	  delete_object_helper (si);
	  displayhelper (si);
	  break;
	case LILYDIRECTIVE:
	  delete_object_helper (si);
	  displayhelper (si);
	  break;
	case GRACE_START:
	case GRACE_END:
	  delete_object_helper (si);
	  displayhelper (si);
	  break;
	case LYRIC:
	case FIGURE:
	  delete_object_helper (si);
	  displayhelper (si);
	  break;
	case BARLINE:
	case COMMENT:
	case MEASUREBREAK:
	  break;
	}
      nudgerightward (si);
      si->haschanged = TRUE;
      si->markstaffnum = 0;
    }
}

/**
 * Insert cloned chordobject into the 
 * score
 */
void
insertclone (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    object_insert (si, clone_object (curmudelaobj));
}

/**
 * Go to end of score 
 */
void
toend (GtkAction * action, struct scoreinfo *si)
{
  si->currentmeasurenum = si->leftmeasurenum = si->rightmeasurenum =
    g_list_length (((staff *) si->currentstaff->data)->measures);
  setcurrents (si);
  find_leftmost_allcontexts (si);
  update_hscrollbar (si);
  gtk_widget_draw (si->scorearea, NULL);
}

/**
 * Go to beginning of score 
*/
void
tohome (GtkAction * action, struct scoreinfo *si)
{
  si->currentmeasurenum = si->leftmeasurenum = 1;
  set_rightmeasurenum (si);
  setcurrents (si);
  find_leftmost_allcontexts (si);
  update_hscrollbar (si);
  gtk_widget_draw (si->scorearea, NULL);
}

/**
 * Go to next score 
*/
void
tonextscore (GtkAction * action, struct scoreinfo *si)
{
  GList *g = si->scoreblocks;
  if (g == NULL)
    return;
  /* copy the si to the last in the list si->next,
     then copy the first to si, and then move the first to the last */
  memcpy (g_list_last (g)->data, si, sizeof (scoreinfo));
  memcpy (si, g->data, sizeof (scoreinfo));
  g = g_list_append (g, g->data);	/* put it on the end */
#if GTK_MAJOR_VERSION > 1
  g = g_list_delete_link (g, g);	/* and remove it from the beginning */
#else
  g = g->next;
#endif
  si->scoreblocks = g;
  updatescoreinfo (si);
}

/**
 * Insert stem directive,  absolute stemdirection for the 
 * entire staff or until a new stem directive in added
 */
void
stem_directive_insert (GtkAction * action, gpointer data)
{
  struct scoreinfo *si = (struct scoreinfo *) data;
  object_insert (si, stem_directive_new (DENEMO_STEMBOTH));
  /* This sets beams and stem directions in the measure, but that's
   * not sufficient */
  beamsandstemdirswholestaff ((staff *) si->currentstaff->data);
  find_xes_in_all_measures (si);
  nudgerightward (si);
  si->haschanged = TRUE;
  gtk_widget_draw (si->scorearea, NULL);
}

/**
 * Toggle start_slur flag for the current chord
 */
void
toggle_begin_slur (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      ((chord *) curmudelaobj->object)->slur_begin_p
	= !((chord *) curmudelaobj->object)->slur_begin_p;
      si->haschanged = TRUE;
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      gtk_widget_draw (si->scorearea, NULL);
    }
}


void
caution (struct scoreinfo *si)
{
  declarecurmudelaobj;

  forceaccidentals (curmudelaobj);
  find_xes_in_measure (si, si->currentmeasurenum, si->cursortime1,
		       si->cursortime2);
  nudgerightward (si);
  if (curmudelaobj->user_string)
    {
      g_free (curmudelaobj->user_string);
      curmudelaobj->user_string = NULL;
    }
  si->haschanged = TRUE;
}

/**
 * Toggle end_slur flag for the current chord
 */
void
toggle_end_slur (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      ((chord *) curmudelaobj->object)->slur_end_p
	= !((chord *) curmudelaobj->object)->slur_end_p;
      si->haschanged = TRUE;
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      gtk_widget_draw (si->scorearea, NULL);
    }
}

/**
 * Toggle start crescendo flag for current chord
 */
void
toggle_start_crescendo (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      ((chord *) curmudelaobj->object)->crescendo_begin_p
	= !((chord *) curmudelaobj->object)->crescendo_begin_p;
      si->haschanged = TRUE;
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      displayhelper(si);
    }
}

/**
 * Toggle end crescendo flag for current chord
 */
void
toggle_end_crescendo (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      ((chord *) curmudelaobj->object)->crescendo_end_p
	= !((chord *) curmudelaobj->object)->crescendo_end_p;
      si->haschanged = TRUE;
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      displayhelper(si);
    }
}

/**
 * Toggle start diminuendo flag for current chord
 */
void
toggle_start_diminuendo (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      ((chord *) curmudelaobj->object)->diminuendo_begin_p
	= !((chord *) curmudelaobj->object)->diminuendo_begin_p;
      si->haschanged = TRUE;
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      gtk_widget_draw (si->scorearea, NULL);
    }
}

/**
 * Toggle end diminuendo flag for current chord
 */
void
toggle_end_diminuendo (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      ((chord *) curmudelaobj->object)->diminuendo_end_p
	= !((chord *) curmudelaobj->object)->diminuendo_end_p;
      si->haschanged = TRUE;
      if (curmudelaobj->user_string)
	{
	  g_free (curmudelaobj->user_string);
	  curmudelaobj->user_string = NULL;
	}
      gtk_widget_draw (si->scorearea, NULL);
    }
}





gboolean
auto_save_document_timeout (struct scoreinfo *si)
{

  g_print ("Autosaving\n");
  if (si->filename->len != 0 && !si->autosavename)
    {
      si->autosavename = si->filename;
      g_string_append (si->autosavename, "#");
    }
  else if (si->filename->len == 0)
    {
      si->autosavename = g_string_new ("autosave.denemo");
    }
  g_print ("Auto save file name %s\n", si->autosavename->str);
  if (si->lily_file)
    {
      exportmudela (si->autosavename->str, si, 0, 0);
    }
  else
    {
      exportXML (si->autosavename->str, si, 0, 0);
    }

  return TRUE;
}
