# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>
#          Mark G. Saye <mark.saye@canonical.com>
#          Stuart Langridge <stuart.langridge@canonical.com>
#          Chad Miller <chad.miller@canonical.com>
#          Vincenzo Di Somma <vincenzo.di.somma@canonical.com>

"""The Desktop Couch Records API."""

import warnings

from couchdb import Server

from desktopcouch.application.local_files import DEFAULT_CONTEXT
from desktopcouch.records.database import Database
from desktopcouch.application.platform import find_port
from desktopcouch.records.http import OAuthSession
from desktopcouch.application import local_files

DCTRASH = 'dctrash'


class DesktopServer(Server):
    """Subclass Server to provide oauth magic"""

    def __init__(self, uri, oauth_tokens=None, ctx=None):
        """Subclass of couchdb.client.Server which understands OAuth"""
        if ctx is None:
            ctx = DEFAULT_CONTEXT
        if oauth_tokens is None:
            oauth_tokens = local_files.get_oauth_tokens(ctx)
        session = OAuthSession(credentials=oauth_tokens)
        super(DesktopServer, self).__init__(url=uri, session=session)


class OAuthCapableServer(DesktopServer):
    """Deprecated."""

    def __init__(self, uri, oauth_tokens=None, ctx=None):
        warnings.warn(
            "OAuthCapableServer is deprecated, use DesktopServer "
            "instead.", DeprecationWarning, stacklevel=2)
        super(OAuthCapableServer, self).__init__(
            uri, oauth_tokens=oauth_tokens, ctx=ctx)


class DesktopDatabase(Database):
    """A desktopcouch specific CouchDb database."""

    def __init__(self, database, uri=None, record_factory=None,
                 create=False, server_class=DesktopServer, oauth_tokens=None,
                 ctx=None, views_factory=None):
        if ctx is None:
            ctx = DEFAULT_CONTEXT
        self.ctx = ctx
        self.server_uri = uri
        super(DesktopDatabase, self).__init__(
            database, uri, record_factory=record_factory,
            create=create, server_class=server_class,
            oauth_tokens=oauth_tokens, ctx=ctx, views_factory=views_factory)

    # pylint: disable=W0221
    # Arguments number differs from overridden method
    def _reconnect(self):
        if not self.server_uri:
            port = find_port(ctx=self.ctx)
            uri = "http://localhost:%s" % port
        else:
            uri = self.server_uri
        super(DesktopDatabase, self)._reconnect(uri=uri)
    # pylint: enable=W0221


class CouchDatabase(DesktopDatabase):
    """Deprecated."""

    def __init__(self, database, uri=None, record_factory=None,
                 create=False, server_class=DesktopServer, oauth_tokens=None,
                 ctx=None):
        warnings.warn(
            "CouchDatabase is deprecated, use DesktopDatabase instead.",
            DeprecationWarning, stacklevel=2)
        super(CouchDatabase, self).__init__(
            database, uri=uri, record_factory=record_factory,
            create=create, server_class=server_class,
            oauth_tokens=oauth_tokens, ctx=ctx)
