# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

__maintainer__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.core import common
from elisa.core.input_event import *
from elisa.base_components.controller import Controller
 
from twisted.internet import reactor

plugin_registry = common.application.plugin_registry
ListController = plugin_registry.get_component_class('raval:list_controller')

class SlideshowController(ListController):
    """
    @ivar playing:       whether or not the slideshow is currently playing
    @type playing:       boolean
    @ivar duration:      delay between two pictures (in seconds)
    @type duration:      float
    """

    supported_models = ('base:list_model',)

    duration = 5.0
    _call_later = None
    _playing = False

    def _cancel_last_call_later(self):
        if self._call_later != None and self._call_later.active():
            self._call_later.cancel()

    def handle_input(self, input_event):
        if input_event.action == EventAction.GO_LEFT:
            if len(self.model) > 0:
                self.previous_image()
            return True

        elif input_event.action == EventAction.GO_RIGHT:
            if len(self.model) > 0:
                self.next_image()
            return True

        elif input_event.action == EventAction.OK:
            self.playing = not self.playing
            return True

        elif input_event.action == EventAction.PLAY:
            self.playing = True
            return True
         
        elif input_event.action == EventAction.PAUSE:
            self.playing = False
            return True
        
        elif input_event.action == EventAction.STOP:
            self.playing = False
            self.current_index = 0
            return True

        return False

    def playing__set(self, playing):
        self._playing = playing
        if playing:
            self._cancel_last_call_later()
            self._call_later = reactor.callLater(self.duration,
                                                 self.next_image)
        else:
            self._cancel_last_call_later()

    def playing__get(self):
        return self._playing

    def focused_changed(self, old_focused, new_focused):
        if new_focused == False and self.playing == True:
            self.playing = False

    def next_image(self):
        self._cancel_last_call_later()

        if self.model == None:
            return

        index = self.current_index + 1
        if index < len(self.model):
            self.info("Loading image at index %r", index)
            self.current_index = index
            if self.playing == True:
                self._call_later = reactor.callLater(self.duration,
                                                     self.next_image)
            return True
        else:
            self.info("Reached end of slideshow")
            return False

    def previous_image(self):
        self._cancel_last_call_later()

        if self.model == None:
            return

        index = self.current_index - 1
        if index < len(self.model) and index >= 0:
            self.info("Loading image at index %r", index)
            self.current_index = index
            if self.playing == True:
                self._call_later = reactor.callLater(self.duration,
                                                     self.previous_image)
            return True
        else:
            self.info("Reached beginning of slideshow")
            return False

