# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


import os
import tempfile

from twisted.trial.unittest import TestCase, SkipTest
from twisted.internet import defer, reactor
from elisa.core import bus
from elisa.core import common
from elisa.core.plugin_registry import PluginRegistry
from elisa.core.config import Config
from elisa.core.media_uri import MediaUri
from elisa.plugins.coherence.coherence_resource import CoherenceResource
from elisa.plugins.base.models.network import NetworkServiceModel

from elisa.core.application import ComponentsLoadedMessage

class FakeApplication(object):
    def __init__(self, config_file, plugin_registry, bus):
        self.config = Config(config_file)
        self.plugin_registry = plugin_registry
        self.bus = bus

class FakeService:

    def get_sid(self):
        return 'uuid:idontremember'

    def get_id(self):
        return 'iamanid'

    def get_device(self):
        class Device:
            def get_uuid(self):
                return 'goo'

        d = Device()
        d.friendly_name = 'iamfriendly'
        return d

class FakeClient:

    def browse(self, container_id):
        return defer.succeed(['woo', 'foo', 'bar'])

class TestCoherenceResourceProvider(TestCase):
    plugin_registry = PluginRegistry([tempfile.gettempdir(),])
    plugin_registry.load_plugins()
    old_application = None
    test_dir = os.path.abspath('test_plugin_registry')

    def setUp(self):
        def set_provider(provider):
            self.provider = provider
            return provider

        self.patch_application()
        config = {}
        dfr = CoherenceResource.create(config)
        dfr.addCallback(set_provider)
        return dfr

    def tearDown(self):
        self.unpatch_application()
        return self.provider.clean()

    
    def patch_application(self):
        """
        Setup common.application, saving the old application object.

        Make common.application a generic object so we can set
        common.application.config and common.application.plugin_registry
        """
        assert self.old_application is None

        self.old_application = common.application
        config_file = os.path.join(self.test_dir,
                'test_plugin_registry.conf')

        self.bus = bus.Bus()
        common.application = FakeApplication(config_file,
                self.plugin_registry, self.bus)

    def unpatch_application(self):
        """
        Restore the application object saved in patch_application().
        """
        common.application = self.old_application
        self.old_application = None


    def test_unsupported_uri(self):
        model, dfr = self.provider.get(MediaUri('coherence://idontexist'))
        self.failUnlessFailure(dfr, NotImplementedError)
        return dfr


    def test_list_media_servers(self):

        self.provider._known_media_servers = {u'upnp://iamfake': ('imanudn', FakeService())
                                             }

        def got_media_servers(model):
            self.failUnless(model.media_servers)
            first = model.media_servers[0]
            self.failUnless(isinstance(first, NetworkServiceModel))
            self.assertEquals(first.uid, 'uuid:goo')
            self.assertEquals(first.name, 'iamfriendly')
            self.assertEquals(first.type, 'iamanid')
            self.assertEquals(first.elisa_uri, MediaUri('upnp://goo?sid=idontremember'))

        uri = MediaUri('coherence://media_servers')
        model, dfr = self.provider.get(uri)
        dfr.addCallback(got_media_servers)
        return dfr
