# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>


from elisa.core.common import application

from elisa.core.media_uri import MediaUri
from elisa.plugins.poblesec.link import Link

from elisa.plugins.poblesec.widgets.menu_item import MenuItemWidget

from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.database.music_controller import GenericAlbumsDbController, \
                                                    GenericArtistsController, \
                                                    GenericTracksDbController

from elisa.plugins.database.music_controller import ArtistsDbViewMode, \
                                                    AlbumsDbViewMode, \
                                                    DBTracksViewMode

from elisa.plugins.database.photo_controller import GenericPhotoAlbumsController, \
                                                    GenericPhotosController

from elisa.plugins.database.photo_controller import PhotoViewMode, \
                                                    PhotoAlbumViewMode

from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController

from elisa.plugins.poblesec.section import SectionMenuController

from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.database.models import *
from elisa.plugins.favorites.models import *

from elisa.plugins.shoutcast.models import ShoutcastRadioStationModel

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.shoutcast.controller import GenericShoutcastController
from elisa.plugins.shoutcast.controller import ShoutcastViewMode

from twisted.internet import defer

from storm.locals import Select

from elisa.core.utils.i18n import install_translation

_ = install_translation('favorites')


def music_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/music/favorites'
    link.controller_args = {'section' : '/music'}
    link.label = _('Favorites')
    link.icon = 'elisa.plugins.favorites.icon'
    controller.model.append(link)

    return defer.succeed(None)

def music_internet_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/music/internet/favorites'
    link.label = _('Favorites')
    link.icon = 'elisa.plugins.favorites.icon'
    controller.model.append(link)

    return defer.succeed(None)

def pictures_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/pictures/favorites'
    link.controller_args = {'section' : '/pictures'}
    link.label = _('Favorites')
    link.icon = 'elisa.plugins.favorites.icon'
    controller.model.append(link)

    return defer.succeed(None)


class FavoritesController(SectionMenuController):

    def initialize(self, section=None):
        self.section = section

        dfr = super(FavoritesController, self).initialize()

        if section == '/music':
            link = Link()
            link.controller_path = '/poblesec/music/favorites/tracks'
            link.controller_args = {}
            link.label = _('Tracks')
            link.icon = 'elisa.plugins.poblesec.by_track'

            self.model.append(link)

            link = Link()
            link.controller_path = '/poblesec/music/favorites/artists'
            link.controller_args = {}
            link.label = _('Artists')
            link.icon = 'elisa.plugins.poblesec.by_artist'

            self.model.append(link)

            link = Link()
            link.controller_path = '/poblesec/music/favorites/albums'
            link.controller_args = {}
            link.label = _('Albums')
            link.icon = 'elisa.plugins.poblesec.by_album'

            self.model.append(link)
        elif section == '/pictures':
            link = Link()
            link.controller_path = '/poblesec/pictures/favorites/albums'
            link.controller_args = {}
            link.label = _('Albums')
            link.icon = 'elisa.plugins.poblesec.photo_album'

            self.model.append(link)

            link = Link()
            link.controller_path = '/poblesec/pictures/favorites/photos'
            link.controller_args = {}
            link.label = _('Photos')
            link.icon = 'elisa.plugins.poblesec.photo'

            self.model.append(link)

        return dfr


# This function is meant to be connected to the "focus" signal emitted from the
# main widget of a controller, in order to trigger a refresh of its model -
# except the first time the controller is put on the screen.
# It is a hack to deal with the dumb history navigation system we have in place
# at present.
def focus_refresh_cb(widget, focus, controller):
    if focus:
        if getattr(controller, '_first_initialize', True):
            controller._first_initialize = False
        else:
            controller.model[:] = []
            controller.initialize()


class FavoritesAlbumsController(GenericAlbumsDbController):

    def initialize(self, artist=None):
        self.artist = artist

        dfr = super(FavoritesAlbumsController, self).initialize()

        def get_albums(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'MusicAlbum')
            return application.store.find(MusicAlbum, MusicAlbum.name.is_in(subselect))

        def got_albums(albums):
            self.model.extend(albums)
            return self

        def sort_albums(result_set):
            result_set.order_by(MusicAlbum.name)
            return result_set.all()

        dfr.addCallback(get_albums)
        dfr.addCallback(sort_albums)
        dfr.addCallback(got_albums)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr


class FavoritesArtistsController(GenericArtistsController):

    def initialize(self):
        dfr = super(FavoritesArtistsController, self).initialize()

        def get_artists(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'Artist')
            return application.store.find(Artist, Artist.name.is_in(subselect))

        def got_artists(artists):
            self.model.extend(artists)
            return self

        def sort_artists(result_set):
            result_set.order_by(Artist.name)
            return result_set.all()

        dfr.addCallback(get_artists)
        dfr.addCallback(sort_artists)
        dfr.addCallback(got_artists)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr


class FavoritesTracksController(GenericTracksDbController):

    def initialize(self, album=None, track=None):
        self.album = album
        self.track = track

        dfr = super(FavoritesTracksController, self).initialize()

        def get_tracks(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'MusicTrack')
            return application.store.find(MusicTrack, MusicTrack.file_path.is_in(subselect))

        def got_tracks(tracks):
            self.model.extend(tracks)
            return self

        def sort_tracks_title(result_set):
            result_set.order_by(MusicTrack.title)
            return result_set.all()

        if self.track is None:
            dfr.addCallback(get_tracks)
            dfr.addCallback(sort_tracks_title)
            dfr.addCallback(got_tracks)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr

class FavoritesRadiosController(GenericShoutcastController):

    def initialize(self, uri=None, station=None):
        dfr = super(FavoritesRadiosController, self).initialize(uri=uri, station=station)

        if station is None:
            def get_radios(result):
                dfr = application.store.find(FavoritesItem,
                                             FavoritesItem.foreign_class == u'InternetRadio')
                return dfr

            def got_radios(radios):
                shoutcast_radios = []
                for radio in radios:
                    shoutcast_radio = ShoutcastRadioStationModel()
                    shoutcast_radio.name = radio.title
                    shoutcast_radio.get_playable = MediaUri(radio.uri)
                    shoutcast_radios.append(shoutcast_radio)

                self.model.extend(shoutcast_radios)
                return self

            def sort_radios(result_set):
                result_set.order_by(FavoritesItem.title)
                return result_set.all()

            dfr.addCallback(get_radios)
            dfr.addCallback(sort_radios)
            dfr.addCallback(got_radios)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr


class FavoritesAlbumsVerticalWithPreview(FavoritesAlbumsController, MenuItemPreviewListController):
    view_mode = AlbumsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class FavoritesAlbumsCoverflow(FavoritesAlbumsController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsDbViewMode

class FavoritesAlbumsGrid(FavoritesAlbumsController, GridItemGridController):
    view_mode = AlbumsDbViewMode

class FavoritesAlbumsListSwitcher(ListSwitcherController):
    modes = [FavoritesAlbumsVerticalWithPreview, FavoritesAlbumsCoverflow, FavoritesAlbumsGrid]
    default_mode = FavoritesAlbumsVerticalWithPreview


class FavoritesArtistsVerticalWithPreview(FavoritesArtistsController, MenuItemPreviewListController):
    view_mode = ArtistsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class FavoritesArtistsCoverflow(FavoritesArtistsController, ImageWithReflectionCoverflowController):
    view_mode = ArtistsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class FavoritesArtistsGrid(FavoritesArtistsController, GridItemGridController):
    view_mode = ArtistsDbViewMode

class FavoritesArtistsListSwitcher(ListSwitcherController):
    modes = [FavoritesArtistsVerticalWithPreview, FavoritesArtistsCoverflow, FavoritesArtistsGrid]
    default_mode = FavoritesArtistsVerticalWithPreview


class FavoritesTracksVerticalWithPreview(FavoritesTracksController, MenuItemPreviewListController):
    view_mode = DBTracksViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.title

class FavoritesTracksCoverflow(FavoritesTracksController, ImageWithReflectionCoverflowController):
    view_mode = DBTracksViewMode

class FavoritesTracksGrid(FavoritesTracksController, GridItemGridController):
    view_mode = DBTracksViewMode

class FavoritesTracksListSwitcher(ListSwitcherController):
    modes = [FavoritesTracksVerticalWithPreview, FavoritesTracksCoverflow, FavoritesTracksGrid]
    default_mode = FavoritesTracksVerticalWithPreview


class FavoritesRadiosVerticalWithPreview(FavoritesRadiosController, MenuItemPreviewListController):
    view_mode = ShoutcastViewMode

class FavoritesRadiosCoverflow(FavoritesRadiosController, ImageWithReflectionCoverflowController):
    view_mode = ShoutcastViewMode

class FavoritesRadiosGrid(FavoritesRadiosController, GridItemGridController):
    view_mode = ShoutcastViewMode

class FavoritesRadiosListSwitcher(ListSwitcherController):
    modes = [FavoritesRadiosVerticalWithPreview, FavoritesRadiosCoverflow, FavoritesRadiosGrid]
    default_mode = FavoritesRadiosVerticalWithPreview


class FavoritesPhotosController(GenericPhotosController):

    def initialize(self, album=None, photo=None, photos=None):
        dfr = super(FavoritesPhotosController, self).initialize(album=album, photo=photo, photos=photos)

        def get_photos(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'Image')
            return application.store.find(Image, Image.file_path.is_in(subselect))

        def got_photos(photos):
            self.model.extend(photos)
            return self

        def sort_photos(result_set):
            result_set.order_by(Image.shot_time)
            return result_set.all()

        if photo is None:
            dfr.addCallback(get_photos)
            dfr.addCallback(sort_photos)
            dfr.addCallback(got_photos)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr


class FavoritesPhotoAlbumsController(GenericPhotoAlbumsController):

    def initialize(self):
        dfr = super(FavoritesPhotoAlbumsController, self).initialize()

        def get_albums(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'PhotoAlbum')
            return application.store.find(PhotoAlbum, PhotoAlbum.name.is_in(subselect))

        def got_albums(albums):
            self.model.extend(albums)
            return self

        def sort_albums_title(result_set):
            result_set.order_by(PhotoAlbum.name)
            return result_set.all()

        dfr.addCallback(get_albums)
        dfr.addCallback(sort_albums_title)
        dfr.addCallback(got_albums)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr


class FavoritesPhotosVerticalWithPreview(FavoritesPhotosController, MenuItemPreviewListController):
    view_mode = PhotoViewMode

class FavoritesPhotosCoverflow(FavoritesPhotosController, ImageWithReflectionCoverflowController):
    view_mode = PhotoViewMode

class FavoritesPhotosGrid(FavoritesPhotosController, GridItemGridController):
    view_mode = PhotoViewMode

class FavoritesPhotosListSwitcher(ListSwitcherController):
    modes = [FavoritesPhotosVerticalWithPreview, FavoritesPhotosCoverflow, FavoritesPhotosGrid]
    default_mode = FavoritesPhotosVerticalWithPreview


class FavoritesPhotoAlbumsVerticalWithPreview(FavoritesPhotoAlbumsController, MenuItemPreviewListController):
    view_mode = PhotoAlbumViewMode

class FavoritesPhotoAlbumsCoverflow(FavoritesPhotoAlbumsController, ImageWithReflectionCoverflowController):
    view_mode = PhotoAlbumViewMode

class FavoritesPhotoAlbumsGrid(FavoritesPhotoAlbumsController, GridItemGridController):
    view_mode = PhotoAlbumViewMode

class FavoritesPhotoAlbumsListSwitcher(ListSwitcherController):
    modes = [FavoritesPhotoAlbumsVerticalWithPreview, FavoritesPhotoAlbumsCoverflow, FavoritesPhotoAlbumsGrid]
    default_mode = FavoritesPhotoAlbumsVerticalWithPreview


