# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.pigment_frontend import ControllerNotFound
from elisa.plugins.poblesec.transitions import Transition, FadeIn, FadeOut

from elisa.core.media_uri import MediaUri
from elisa.plugins.gstreamer.amp_master import GstMetadataAmpClient

from elisa.core.input_event import *
from elisa.core.utils import defer
from elisa.core.application import NewElisaVersionMessage, CONFIG_DIR
from elisa.core import common
from elisa.core.utils.i18n import install_translation

_ = install_translation('poblesec')

import platform
import os

class PartialFadeOut(Transition):
    def apply(self, controller):
        def cancel(dfr):
            self._stop(dfr, controller)
        dfr = defer.Deferred(canceller=cancel)

        def end_animation(timer):
            if not dfr.called:
                dfr.callback(controller)

        controller.widget_animated.update_animation_settings(duration=400,
                                                    end_callback=end_animation)
        controller.widget_animated.opacity = 50

        return dfr


class PoblesecController(PigmentController):

    def __init__(self):
        super(PoblesecController, self).__init__()
        self._overlay_in = FadeIn()
        self._overlay_out = FadeOut()
        self._background_in = FadeIn()
        self._background_out = PartialFadeOut()

        self.current_player = None

        self._popup_visible = False

    def initialize(self):
        common.application.bus.register(self._got_new_version, NewElisaVersionMessage)
        return super(PoblesecController, self).initialize()

    def _got_new_version(self, message, sender):
        if platform.system().lower() == 'windows' and message.installer_url:
            self._download_installer(message.installer_url, message.version)

    def _download_installer(self, url, version):
        uri = MediaUri(url)
        installer_path = os.path.join(CONFIG_DIR, uri.filename)

        # avoid downloading twice the same file
        if not os.path.exists(installer_path):

            def got_installer(model):
                fd = open(installer_path, 'wb')
                fd.write(model.data)
                fd.close()
                return self._show_installer_popup(installer_path, version)

            model, dfr = common.application.resource_manager.get(uri)
            dfr.addCallback(got_installer)
        else:
            self._show_installer_popup(installer_path, version)

    def _show_installer_popup(self, installer_path, version):
        icon = 'elisa.plugins.poblesec.warning'
        title = _('New Elisa version available')
        text = _('Version %s of Elisa is available. Would you like to install it now?')
        text = text % version

        def ok_clicked():
            # execute installer in background and stop Elisa
            from win32api import ShellExecute
            import pywintypes

            try:
                ShellExecute(0, 'open', installer_path, '', None, 1)
            except pywintypes.error, error:
                self.warning(error)
                self.hide_popup()
            else:
                common.application.stop()

        buttons = [(_('Yes'), ok_clicked), (_('Later'), self.hide_popup)]
        self.show_popup(icon, title, text, buttons)

    def set_frontend(self, frontend):
        super(PoblesecController, self).set_frontend(frontend)

        def browser_created(controller):
            self.browser = controller
            self.widget.add(self.browser.widget)
            self.browser.widget.width = 1.0
            self.browser.widget.height = 1.0
            self.browser.widget.x = 0.0
            self.browser.widget.y = 0.0
            self.focused_controller = controller

        dfr = self.frontend.create_controller('/poblesec/browser',
                                              home_path='/poblesec/sections_menu')
        dfr.addCallback(browser_created)
        # FIXME: losing a deferred here

        def video_player_created(controller):
            self.video_player = controller
            self.current_player = controller
            self.widget.add(self.video_player.widget)
            self.video_player.widget.width = 1.0
            self.video_player.widget.height = 1.0
            self.video_player.widget.x = 0.0
            self.video_player.widget.y = 0.0
            self.video_player.widget.visible = True

        dfr = self.frontend.create_controller('/poblesec/video_player')
        dfr.addCallback(video_player_created)
        # FIXME: losing a deferred here

        def dvd_player_created(controller):
            self.dvd_player = controller
            self.current_player = controller
            self.widget.add(self.dvd_player.widget)
            self.dvd_player.widget.width = 1.0
            self.dvd_player.widget.height = 1.0
            self.dvd_player.widget.x = 0.0
            self.dvd_player.widget.y = 0.0
            self.dvd_player.widget.visible = True


        # We don't want the dvd plugin to be a hard dependency for poblesec.
        try:
            dfr = self.frontend.create_controller('/poblesec/dvd_player')
            dfr.addCallback(dvd_player_created)
            # FIXME: losing a deferred here
        except ControllerNotFound:
            self.debug("Not loading /poblesec/dvd_player because its " +
                       "controller could not be found")
            self.dvd_player = None


        def music_player_created(controller):
            self.music_player = controller
            self.widget.add(self.music_player.widget)
            self.music_player.widget.width = 1.0
            self.music_player.widget.height = 1.0
            self.music_player.widget.x = 0.0
            self.music_player.widget.y = 0.0

        dfr = self.frontend.create_controller('/poblesec/music_player')
        dfr.addCallback(music_player_created)
        # FIXME: losing a deferred here

        def slideshow_player_created(controller):
            self.slideshow_player = controller
            self.widget.add(self.slideshow_player.widget)
            self.slideshow_player.widget.width = 1.0
            self.slideshow_player.widget.height = 1.0
            self.slideshow_player.widget.x = 0.0
            self.slideshow_player.widget.y = 0.0

        dfr = self.frontend.create_controller('/poblesec/slideshow_player')
        dfr.addCallback(slideshow_player_created)
        # FIXME: losing a deferred here

        self._create_gst_metadata()
        # FIXME: losing a deferred here

    def _create_gst_metadata(self):
        def gst_metadata_created(gst_metadata):
            self.gst_metadata = gst_metadata

        dfr = GstMetadataAmpClient.create()
        dfr.addCallback(gst_metadata_created)
        return dfr

    def show_video_player(self):
        # stop slideshow player
        self.slideshow_player.player.stop()
        FadeOut().apply(self.slideshow_player)

        # stop music player
        self.music_player.player.stop()
        FadeOut().apply(self.music_player)

        # stop dvd player
        if self.dvd_player:
            self.dvd_player.player.stop()
            FadeOut().apply(self.dvd_player)

        # show video player
        self.current_player = self.video_player
        self.show_current_player()

    def show_dvd_player(self):
        if not self.dvd_player:
            return

        # stop slideshow player
        self.slideshow_player.player.stop()
        FadeOut().apply(self.slideshow_player)

        # stop music player
        self.music_player.player.stop()
        FadeOut().apply(self.music_player)

        # stop video player
        self.video_player.player.stop()
        FadeOut().apply(self.video_player)

        # show dvd player
        self.current_player = self.dvd_player
        self.show_current_player()

    def show_music_player(self):
        # hide slideshow player
        FadeOut().apply(self.slideshow_player)

        # stop video player
        self.video_player.player.stop()
        FadeOut().apply(self.video_player)

        # stop dvd player
        if self.dvd_player:
            self.dvd_player.player.stop()
            FadeOut().apply(self.dvd_player)

        # show music player
        self.current_player = self.music_player
        self.show_current_player()

    def show_slideshow_player(self):
        # hide music player
        FadeOut().apply(self.music_player)

        # stop video player
        self.video_player.player.stop()
        FadeOut().apply(self.video_player)

        # stop dvd player
        if self.dvd_player:
            self.dvd_player.player.stop()
            FadeOut().apply(self.dvd_player)

        # show slideshow player
        self.current_player = self.slideshow_player
        self.show_current_player()

    def show_current_player(self):
        self._overlay_out.apply(self.browser)
        self._background_in.apply(self.current_player)

        self.focused_controller = self.current_player
        self.focused_controller.widget.focus = True

    def hide_current_player(self):
        self._overlay_in.apply(self.browser)
        self._background_out.apply(self.current_player)
        self.focused_controller = self.browser
        self.focused_controller.widget.focus = True

    def show_popup(self, icon, title, text, buttons):
        if self._popup_visible:
            # FIXME: should enqueue the popup request to display it later, when
            # the currently displayed popup is closed.
            return defer.fail(self.popup_controller)

        def popup_created(controller):
            self._popup_visible = True
            self.current_controller = self.focused_controller
            self.popup_controller = controller
            controller.widget.visible = False
            self.widget.add(controller.widget)
            controller.widget.x, controller.widget.y = (0.0, 0.0)
            controller.widget.width, controller.widget.height = (1.0, 1.0)
            controller.widget.regenerate()
            self._background_in.apply(controller)
            self.focused_controller = controller
            self.focused_controller.widget.focus = True

        popup_dfr = self.frontend.create_controller('/poblesec/popup',
                                                    icon=icon, title=title,
                                                    text=text, buttons=buttons)
        popup_dfr.addCallback(popup_created)
        return popup_dfr

    def hide_popup(self):
        if not self._popup_visible:
            # No popup to hide.
            return defer.fail(None)

        def faded(controller):
            self.focused_controller = self.current_controller
            self.focused_controller.widget.focus = True
            self.widget.remove(controller.widget)
            # FIXME: should display the next enqueued popup.
            self._popup_visible = False
            return controller

        return self._background_out.apply(self.popup_controller).addCallback(faded)
    
    def stop_all_players(self):
        self.slideshow_player.player.stop()
        self.music_player.player.stop()
        self.video_player.player.stop()
        self.dvd_player.player.stop()

    def handle_input(self, manager, input_event):

        if self.focused_controller.handle_input(manager, input_event):
            return True

        if input_event.value == EventValue.KEY_f:
            self.frontend.viewport.fullscreen = not self.frontend.viewport.fullscreen
            return True
        elif input_event.value == EventValue.KEY_MENU:
            if self.focused_controller == self.browser:
                if self.current_player.player.status != \
                                            self.current_player.player.STOPPED:
                    self.show_current_player()
            else:
                self.hide_current_player()
            return True
        elif input_event.value == EventValue.KEY_SPACE and self.current_player:
            if self.current_player.player.status != \
                                            self.current_player.player.STOPPED:
                    self.current_player.toggle_play_pause()
            return True

        return super(PoblesecController, self).handle_input(manager, input_event)
