/*
 * Elisa - Home multimedia server
 * Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
 * All rights reserved.
 *
 * This file is available under one of two license agreements.
 *
 * This file is licensed under the GPL version 3.
 * See "LICENSE.GPL" in the root of this distribution including a special
 * exception to use Elisa with Fluendo's plugins.
 *
 * The GPL part of Elisa is also available under a commercial licensing
 * agreement from Fluendo.
 * See "LICENSE.Elisa" in the root directory of this distribution package
 * for details on that license.
 */

#include <pygobject.h>
#include <gst/gst.h>

#if PY_VERSION_HEX < 0x02050000 
#define Py_ssize_t int
#endif

static PyTypeObject *uri_class;

static guint
_elisa_GstURIHandler__proxy_do_get_type ()
{
    PyGILState_STATE __py_state;
    PyTypeObject *py_class;
    PyObject *py_method;
    PyObject *py_retval;
    guint retval;
    
    __py_state = pyg_gil_state_ensure();
    if (uri_class == NULL) {
        pyg_gil_state_release (__py_state);
        return GST_URI_UNKNOWN;
    }

    Py_INCREF (uri_class);
    py_class = uri_class;

    py_method = PyObject_GetAttrString((PyObject *) py_class, "do_get_type");
    Py_DECREF (py_class);
    if (!py_method) {
        if (PyErr_Occurred())
            PyErr_Print();
        pyg_gil_state_release(__py_state);
        return GST_URI_UNKNOWN;
    }

    py_retval = PyObject_CallObject(py_method, NULL);
    Py_DECREF (py_method);
    if (!py_retval) {
        if (PyErr_Occurred())
            PyErr_Print();
        pyg_gil_state_release(__py_state);
        return GST_URI_UNKNOWN;
    }

    retval = PyLong_AsLong (py_retval);
    Py_DECREF(py_retval);
    pyg_gil_state_release(__py_state);
    
    return retval;
}

static gchar **
_elisa_GstURIHandler__proxy_do_get_protocols ()
{
    PyGILState_STATE __py_state;
    PyTypeObject *py_class = NULL;
    PyObject *py_method = NULL;
    PyObject *py_retval = NULL;
    Py_ssize_t ret_size, i;
    gchar **retval = NULL;

    __py_state = pyg_gil_state_ensure();
    if (uri_class == NULL)
        goto error;

    Py_INCREF (uri_class);
    py_class = uri_class;
    py_method = PyObject_GetAttrString((PyObject *) py_class, "do_get_protocols");
    Py_DECREF (py_class);
    py_class = NULL;
    if (!py_method) {
        if (PyErr_Occurred())
            PyErr_Print();
        goto error;
    }

    py_retval = PyObject_CallObject(py_method, NULL);
    Py_DECREF (py_method);
    py_method = NULL;
    if (!py_retval) {
        if (PyErr_Occurred())
            PyErr_Print();
        goto error;
    }

    if (!PySequence_Check (py_retval)) {
        PyErr_SetString (PyExc_TypeError, "GstURIHandler.do_get_protocols "
            "must return a sequence of strings");
        goto error;
    }
    
    ret_size = PySequence_Size (py_retval);
    if (ret_size == -1)
        goto error;

    retval = g_new (gchar *, ret_size + 1);
    retval [ret_size] = NULL;
    for (i = 0; i < ret_size; ++i) {
        PyObject *item = PySequence_GetItem (py_retval, i);
        if (!item) {
            if (PyErr_Occurred ())
                PyErr_Print ();
            goto error;
        }

        if (!PyString_Check (item)) {
            PyErr_SetString (PyExc_TypeError, "GstURIHandler.do_get_protocols "
                "must return a sequence of strings");
            Py_DECREF (item);
            goto error;
        }

        retval [i] = PyString_AsString (item);
        if (!retval [i]) {
            if (PyErr_Occurred ())
                PyErr_Print ();
            Py_DECREF (item);
            goto error;
        }

        Py_DECREF (item);
    }

    Py_DECREF(py_retval);
    pyg_gil_state_release(__py_state);
   
    /* NOTE: we leak retval because the current GstURIHandler API requires that
     * we return a const GStrv here but we can't. See
     * http://bugzilla.gnome.org/show_bug.cgi?id=339279 for more info. 
     * In our specific case we leak sizeof (gchar *) * 2 + strlen ("elisa") + 1,
     * which is acceptable for a one time leak (when we register the elisa
     * source).
     */
    return retval;

error:
    Py_XDECREF (py_class);
    Py_XDECREF (py_method);
    Py_XDECREF (py_retval);
    if (retval)
        g_strfreev (retval);
    pyg_gil_state_release(__py_state);
    return NULL;
}

static const gchar *
_elisa_GstURIHandler__proxy_do_get_uri(GstURIHandler *self)
{
    PyGILState_STATE __py_state;
    PyObject *py_self;
    const gchar *retval;
    PyObject *py_retval;
    PyObject *py_method;
    
    __py_state = pyg_gil_state_ensure();
    py_self = pygobject_new((GObject *) self);
    if (!py_self) {
        if (PyErr_Occurred())
            PyErr_Print();
        pyg_gil_state_release(__py_state);
        return NULL;
    }
    
    py_method = PyObject_GetAttrString(py_self, "do_get_uri");
    if (!py_method) {
        if (PyErr_Occurred())
            PyErr_Print();
        Py_DECREF(py_self);
        pyg_gil_state_release(__py_state);
        return NULL;
    }
    py_retval = PyObject_CallObject(py_method, NULL);
    if (!py_retval) {
        if (PyErr_Occurred())
            PyErr_Print();
        Py_DECREF(py_method);
        Py_DECREF(py_self);
        pyg_gil_state_release(__py_state);
        return NULL;
    }
    if (!PyString_Check(py_retval)) {
        if (PyErr_Occurred())
            PyErr_Print();
        PyErr_SetString(PyExc_TypeError, "retval should be a string");
        Py_DECREF(py_retval);
        Py_DECREF(py_method);
        Py_DECREF(py_self);
        pyg_gil_state_release(__py_state);
        return NULL;
    }
    retval = g_strdup(PyString_AsString(py_retval));
    
    
    Py_DECREF(py_retval);
    Py_DECREF(py_method);
    Py_DECREF(py_self);
    pyg_gil_state_release(__py_state);
    
    return retval;
}

static gboolean
_elisa_GstURIHandler__proxy_do_set_uri(GstURIHandler *self, const gchar*uri)
{
    PyGILState_STATE __py_state;
    PyObject *py_self;
    PyObject *py_uri;
    gboolean retval;
    PyObject *py_main_retval;
    PyObject *py_retval;
    PyObject *py_args;
    PyObject *py_method;
    
    __py_state = pyg_gil_state_ensure();
    py_self = pygobject_new((GObject *) self);
    if (!py_self) {
        if (PyErr_Occurred())
            PyErr_Print();
        pyg_gil_state_release(__py_state);
        return FALSE;
    }
    py_uri = PyString_FromString(uri);
    if (!py_uri) {
        if (PyErr_Occurred())
            PyErr_Print();
        Py_DECREF(py_self);
        pyg_gil_state_release(__py_state);
        return FALSE;
    }
    
    py_args = PyTuple_New(1);
    PyTuple_SET_ITEM(py_args, 0, py_uri);
    
    py_method = PyObject_GetAttrString(py_self, "do_set_uri");
    if (!py_method) {
        if (PyErr_Occurred())
            PyErr_Print();
        Py_DECREF(py_args);
        Py_DECREF(py_self);
        pyg_gil_state_release(__py_state);
        return FALSE;
    }
    py_retval = PyObject_CallObject(py_method, py_args);
    if (!py_retval) {
        if (PyErr_Occurred())
            PyErr_Print();
        Py_DECREF(py_method);
        Py_DECREF(py_args);
        Py_DECREF(py_self);
        pyg_gil_state_release(__py_state);
        return FALSE;
    }
    py_retval = Py_BuildValue("(N)", py_retval);
    if (!PyArg_ParseTuple(py_retval, "O", &py_main_retval)) {
        if (PyErr_Occurred())
            PyErr_Print();
        Py_DECREF(py_retval);
        Py_DECREF(py_method);
        Py_DECREF(py_args);
        Py_DECREF(py_self);
        pyg_gil_state_release(__py_state);
        return FALSE;
    }
    
    retval = PyObject_IsTrue(py_main_retval)? TRUE : FALSE;
    
    Py_DECREF(py_retval);
    Py_DECREF(py_method);
    Py_DECREF(py_args);
    Py_DECREF(py_self);
    pyg_gil_state_release(__py_state);
    
    return retval;
}

static void
__GstURIHandler__interface_init(GstURIHandlerInterface *iface, PyTypeObject *pytype)
{
    GstURIHandlerInterface *parent_iface = g_type_interface_peek_parent(iface);
    PyObject *py_method;

    py_method = pytype? PyObject_GetAttrString((PyObject *) pytype, "do_get_type") : NULL;
    if (py_method && !PyObject_TypeCheck(py_method, &PyCFunction_Type)) {
        iface->get_type = _elisa_GstURIHandler__proxy_do_get_type;
    } else {
        PyErr_Clear();
        if (parent_iface) {
            iface->get_type = parent_iface->get_type;
        }
    }
    Py_XDECREF(py_method);
    
    py_method = pytype? PyObject_GetAttrString((PyObject *) pytype, "do_get_protocols") : NULL;
    if (py_method && !PyObject_TypeCheck(py_method, &PyCFunction_Type)) {
        iface->get_protocols = _elisa_GstURIHandler__proxy_do_get_protocols;
    } else {
        PyErr_Clear();
        if (parent_iface) {
            iface->get_protocols = parent_iface->get_protocols;
        }
    }
    Py_XDECREF(py_method);

    py_method = pytype? PyObject_GetAttrString((PyObject *) pytype, "do_get_uri") : NULL;
    if (py_method && !PyObject_TypeCheck(py_method, &PyCFunction_Type)) {
        iface->get_uri = _elisa_GstURIHandler__proxy_do_get_uri;
    } else {
        PyErr_Clear();
        if (parent_iface) {
            iface->get_uri = parent_iface->get_uri;
        }
    }
    Py_XDECREF(py_method);
    
    py_method = pytype? PyObject_GetAttrString((PyObject *) pytype, "do_set_uri") : NULL;
    if (py_method && !PyObject_TypeCheck(py_method, &PyCFunction_Type)) {
        iface->set_uri = _elisa_GstURIHandler__proxy_do_set_uri;
    } else {
        PyErr_Clear();
        if (parent_iface) {
            iface->set_uri = parent_iface->set_uri;
        }
    }
    Py_XDECREF(py_method);
}

static GInterfaceInfo __GstURIHandler__iinfo = {
    (GInterfaceInitFunc) __GstURIHandler__interface_init,
    NULL,
    NULL
};

static PyObject *
elisa_uri_bind_to(PyObject *self, PyObject *args, PyObject *kw)
{
    GType type;
    char *keywords[] = {"klass", NULL};

    if (!PyArg_ParseTupleAndKeywords(args, kw, "O!:%(module_name)s.bind_to", keywords,
            &PyType_Type, &uri_class))
        return NULL;
    
    type = pyg_type_from_object ((PyObject *) uri_class);
    if (type == 0)
        /* not a gtype */
        return NULL;

    /* keep a ref in the module */
    Py_INCREF (uri_class);

    __GstURIHandler__iinfo.interface_data = uri_class;
    g_type_add_interface_static(type, GST_TYPE_URI_HANDLER,
        &__GstURIHandler__iinfo);

    Py_INCREF (Py_None);
    return Py_None;
}

static PyMethodDef elisa_uri_functions[] = {
    {"bind_to", elisa_uri_bind_to, METH_VARARGS | METH_KEYWORDS, ""},
    {NULL, NULL, 0, NULL}
};

DL_EXPORT(void)
init%(module_name)s(void)
{
    PyObject *module;

    init_pygobject ();
    module = Py_InitModule ("%(module_name)s", elisa_uri_functions);
}
