# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import copy

from twisted.trial import unittest

from elisa.core.utils import sorting

class TestSorting(unittest.TestCase):

    def test_natural_sort(self):
        # trivial
        original = ["a2", "a1"]
        expected = ["a1", "a2"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        # mixed case
        original = ["a2", "A1"]
        expected = ["A1", "a2"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        # spaces are equal
        original = ["a 2", "a 1"]
        expected = ["a 1", "a 2"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        # more complicated
        original = ["A13", "A1", "A2"]
        expected = ["A1", "A2", "A13"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = ["toto 12", "titi 42", "1 truc", "12 truc", "toto 1"]
        expected = ["1 truc", "12 truc", "titi 42", "toto 1", "toto 12"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

 
        # examples from ASPN recipe "Natural string sorting"
        # http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/285264
        original = ['ver-1.3.12', 'ver-1.3.3', 'ver-1.2.5', 'ver-1.2.15', \
                    'ver-1.2.3', 'ver-1.2.1']
        expected = ['ver-1.2.1', 'ver-1.2.3', 'ver-1.2.5', 'ver-1.2.15', \
                    'ver-1.3.3', 'ver-1.3.12']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = ['C1H2', 'C1H4', 'C2H2', 'C2H6', 'C2N', 'C3H6']
        expected = ['C1H2', 'C1H4', 'C2H2', 'C2H6', 'C2N', 'C3H6']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = ['Team 101', 'Team 58', 'Team 30', 'Team 1']
        expected = ['Team 1', 'Team 30', 'Team 58', 'Team 101']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = ['a5', 'A7', 'a15', 'a9', 'A8']
        expected = ['a5', 'A7', 'A8', 'a9', 'a15']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

    """
    # DEACTIVATED, because our sorting is not suposted to support it
    def test_mixed_encodings_sorting(self):

        original = [u'ä01', 'ä20', 'a13', 'b14']
        expected = ['a13', 'b14', u'\xe401', '\xc3\xa420']
        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = [u'©20', '©01', '#k13', 'üä14', 'öñ']
        expected = ['#k13', '\xc2\xa901', u'\xa920', '\xc3\xb6\xc3\xb1', '\xc3\xbc\xc3\xa414']
        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = [u'Ö20', 'ö0100', 'Ö10', 'ö02']
        expected = ['\xc3\xb602', '\xc3\x9610', u'\xd620', '\xc3\xb60100']
        sorting.natural_sort(original)
        self.assertEquals(original, expected)
    """

    def test_spaces(self):
        original = [u'  1Die1', '1Dae1']
        expected = ['1Dae1', u'  1Die1']
        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = [u'  Zwo', 'zao']
        expected = [u'zao', u'  Zwo']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

