# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Tests for the PatternMatcher
"""


from twisted.trial.unittest import TestCase
from elisa.core.pattern_matcher import UriPatternMatcher, \
        PatternNotFound, InvalidPattern, MatchNotFound

class TestPatternMatcher(TestCase):
    def setUp(self):
        self.resolver = UriPatternMatcher()

    def test_add(self):
        """
        Test whether adding works and fails as expected
        """
        self.resolver.add_pattern('elisa://.*', 'matched object')

        self.failUnlessRaises(InvalidPattern,
                self.resolver.add_pattern, '*', 'matched object')
    
    def test_remove(self):
        """
        test whether removing works and fails as expected
        """
        pattern = 'elisa://.*'
        obj = 'matched object'

        self.failUnlessRaises(PatternNotFound,
                self.resolver.remove_pattern, pattern)

        self.resolver.add_pattern(pattern, obj)
        self.resolver.remove_pattern(pattern)

    def test_no_match(self):
        pattern = 'elisa://.*'
        obj = 'matched object'

        self.resolver.add_pattern(pattern, obj)
        self.failUnlessRaises(MatchNotFound, self.resolver.match,
                'not_elisa://some_view')

    def test_no_wide_match(self):
        pattern = 'elisa://foo'
        obj = 'matched object'

        self.resolver.add_pattern(pattern, obj)
        self.failUnlessRaises(MatchNotFound, self.resolver.match,
                'elisa://foo/bar')

    def test_match_simple(self):
        """
        test whether simple matching works
        """
        pattern = 'elisa://.*'
        obj = 'matched object'

        self.resolver.add_pattern(pattern, obj)
        match = self.resolver.match('elisa://some_view')
        self.failUnlessEqual(match, obj)

    def test_match_order(self):
        """
        test a more complex matching and testing the ordering system
        """
        self.resolver.add_pattern('elisa://.*', 'catchall')
        self.resolver.add_pattern('elisa://first/.*', 'first')
        self.resolver.add_pattern('elisa://first/second/.*', 'first/second')

        match = self.resolver.match('elisa://first/second/abc')
        self.failUnlessEqual(match, 'first/second')

        match = self.resolver.match('elisa://first/abc')
        self.failUnlessEqual(match, 'first')
        
        match = self.resolver.match('elisa://abc')
        self.failUnlessEqual(match, 'catchall')

        self.resolver.add_pattern('elisa://.*', 'catchall 1')
        match = self.resolver.match('elisa://abc')
        self.failUnlessEqual(match, 'catchall 1')
        
        self.resolver.remove_pattern('elisa://.*')
        match = self.resolver.match('elisa://abc')
        self.failUnlessEqual(match, 'catchall')

