# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import os

from elisa.core import common
from elisa.core.plugin_registry import PluginRegistry, ComponentNotFound, \
        PluginAlreadyEnabled, PluginAlreadyDisabled, InvalidComponentPath, \
        PluginNotFound
from elisa.core.utils.misc import pkg_resources_copy_dir as copy_dir
from elisa.core.config import Config
from twisted.trial.unittest import TestCase
from twisted.python import reflect
from twisted.internet import defer

import pkg_resources

class FakeApplication(object):
    def __init__(self, config_file, plugin_registry, bus):
        self.config = Config(config_file)
        self.plugin_registry = plugin_registry
        self.bus = bus

class FakeBus(object):
    def __init__(self):
        self.messages = []

    def send_message(self, message):
        self.messages.append(message)

class TestPluginRegistryMixin(object):
    plugin_registry = None
    old_application = None

    def setUpClass(self): 
        # do everything under _trial_temp/test_plugin_registry
        self.test_dir = os.path.abspath('test_plugin_registry')
        pkg_resources.ensure_directory(self.test_dir)
        self.plugins_dir = self.test_dir

        # copy the test plugins
        self.copy_plugins()
        
        self.install_plugin('test_simple-0.1', '0.1/test_simple')

    def tearDownClass(self):
        self.uninstall_plugin('test_simple-0.1')

    def setUp(self):
        # setup common.application
        self.patch_application()
    
    def tearDown(self):
        self.unpatch_application()
    
    def patch_application(self):
        """
        Setup common.application, saving the old application object.

        Make common.application a generic object so we can set
        common.application.config and common.application.plugin_registry
        """
        assert self.old_application is None

        self.old_application = common.application
        config_file = os.path.join(self.test_dir,
                'test_plugin_registry.conf')

        self.bus = FakeBus()
        common.application = FakeApplication(config_file,
                self.plugin_registry, self.bus)

    def unpatch_application(self):
        """
        Restore the application object saved in patch_application().
        """
        common.application = self.old_application
        self.old_application = None

    def copy_plugins(self):
        dest_dir = os.path.join(self.test_dir, '0.1/test_simple')
        copy_dir('elisa.core', 'tests/data/plugins/test_simple-0.1', dest_dir) 
        dest_dir = os.path.join(self.test_dir, '0.2/test_simple')
        copy_dir('elisa.core', 'tests/data/plugins/test_simple-0.2', dest_dir) 

    def install_plugin(self, link_name, path):
        egg_link = os.path.join(self.test_dir, link_name + '.egg-link')
        link = file(egg_link, 'w')
        link.write(path + '\n')
        link.close()

    def uninstall_plugin(self, path):
        egg_link = os.path.join(self.test_dir, path + '.egg-link')
        assert os.path.exists(egg_link)
        os.unlink(egg_link)

class TestPluginRegistry(TestPluginRegistryMixin, TestCase):
    plugin_registry = None

    def setUpClass(self):
        TestPluginRegistryMixin.setUpClass(self)
         
        # create the PluginRegistry instance used by the tests
        self.plugin_registry = PluginRegistry([self.plugins_dir])
        self.plugin_registry.load_plugins()

    def test_import(self):
        """
        Import a component from a plugin.

        Check that the normal import statement works correctly.
        """
        from elisa.plugins.test_simple.test_module import TestComponent

    def test_create(self):
        """
        Create a component calling ComponentClass.create().

        Check that a component can be created given its class.
        """
        from elisa.plugins.test_simple.test_module import TestComponent

        path = 'elisa.plugins.test_simple.test_module:TestComponent'
        self.assertEqual(TestComponent.path, path)

        def clean(component):
            self.assertTrue(component.initialize_called)
            return component.clean()

        d = TestComponent.create()
        d.addCallback(clean)

        return d

    def test_create_component_factory(self):
        """
        Create a component with PluginRegistry.create_component.

        Check that a plugin can be created given its path.
        """
        def clean(component):
            klass = reflect.getClass(component)
            path = 'elisa.plugins.test_simple.test_module:TestComponent'
            self.assertEqual(klass.path, path)
            
            # this seems silly but was actually broken before
            from elisa.plugins.test_simple.test_module import TestComponent
            self.failUnless(isinstance(component, TestComponent))

            return component.clean()

        path = 'test_simple.test_module:TestComponent'
        d = self.plugin_registry.create_component(path)
        d.addCallback(clean)

        return d

    def test_create_component_factory_invalid_path(self):
        path = 'i_should_contain_a_colon'
        d = self.plugin_registry.create_component(path)
        self.failUnlessFailure(d, InvalidComponentPath)

    def test_get_plugin_names(self):
        """
        Get the list of available plugins.
        """
        plugins = list(self.plugin_registry.get_plugin_names())
        self.failUnlessIn('elisa-plugin-test-simple', plugins)

    def test_default_config(self):
        """
        Create a component and check that the default configuration is loaded.
        """
        from elisa.plugins.test_simple.test_module import TestComponent

        def check_config(component):
            # Component.create() should load the configuration
            self.assertEqual(component.config['sample_config_key'], 'default value')

            return component.clean()

        d = TestComponent.create()
        d.addCallback(check_config)

        return d

    def test_application_config(self):
        """
        Create a component and check that the configuration is loaded from the
        configuration file.
        """
        from elisa.plugins.test_simple.test_module import TestComponent

        config = common.application.config

        path = 'test_simple.test_module:TestComponent'
        user_value = 'user value'
        config.set_section(path, {})
        config.set_option('sample_config_key', user_value, section=path)

        def check_config(component):
            self.assertEqual(component.config['sample_config_key'], user_value)

            return component.clean()

        d = TestComponent.create()
        d.addCallback(check_config)

        return d

    def test_hot_upgrade(self):
        """
        Load a component from a plugin, then upgrade the plugin and check that
        the component is reloaded.
        """
        from elisa.plugins.test_simple.test_module import TestComponent
        from elisa.plugins.test_simple import test_module

        self.failUnlessEqual(test_module.version, '0.1')

        def upgrade_plugin(component):
            res = component.do_something()
            self.failUnlessEqual(res, 'done in 0.1')

            self.install_plugin('test_simple-0.2', '0.2/test_simple')
            self.plugin_registry.load_plugins()

            self.failUnlessEqual(test_module.version, '0.2')
            res = component.do_something()
            self.failUnlessEqual(res, 'done in 0.2')

            return component.clean()

        d = TestComponent.create()
        d.addCallback(upgrade_plugin)

        return d

class TestPluginActivation(TestPluginRegistryMixin, TestCase):
    def setUp(self):
        TestPluginRegistryMixin.setUp(self)
         
        # create the PluginRegistry instance used by the tests
        self.plugin_registry = PluginRegistry([self.plugins_dir])
        
        # don't enable any plugins
        self.plugin_registry.load_plugins([])

    def test_enable_plugin(self):
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 0)

        self.plugin_registry.enable_plugin('elisa-plugin-test-simple')
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 1)
        # we should get a plugin enabled message
        self.failUnlessEqual(len(self.bus.messages), 1)
        message = self.bus.messages[0]
        self.failUnlessEqual(message.plugin_name, 'elisa-plugin-test-simple')
        self.failUnlessEqual(message.action, message.ActionType.ENABLED)
        
        self.failUnlessRaises(PluginAlreadyEnabled,
                self.plugin_registry.enable_plugin,
                'elisa-plugin-test-simple')
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 1)
        # we shouldn't get a message here
        self.failUnlessEqual(len(self.bus.messages), 1)

    def test_enable_not_found(self):
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 0)
 
        self.failUnlessRaises(PluginNotFound,
                self.plugin_registry.enable_plugin,
                'a-plugin-named-like-this-cant-possibly-exist')
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 0)
        # we shouldn't get a message here
        self.failUnlessEqual(len(self.bus.messages), 0)        

    def test_disable_plugin(self):
        # start empty
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 0)
        
        # disable a plugin that is not enabled
        self.failUnlessRaises(PluginAlreadyDisabled,
                self.plugin_registry.disable_plugin,
                'elisa-plugin-test-simple')
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 0)
        # we shouldn't get a message here
        self.failUnlessEqual(len(self.bus.messages), 0)

        # enable a plugin
        self.plugin_registry.enable_plugin('elisa-plugin-test-simple')
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 1)
        # we should get a plugin enabled message
        self.failUnlessEqual(len(self.bus.messages), 1)
        message = self.bus.messages[0]
        self.failUnlessEqual(message.plugin_name, 'elisa-plugin-test-simple')
        self.failUnlessEqual(message.action, message.ActionType.ENABLED)

        # disable the plugin
        self.plugin_registry.disable_plugin('elisa-plugin-test-simple')
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 0)
        message = self.bus.messages[1]
        self.failUnlessEqual(message.plugin_name, 'elisa-plugin-test-simple')
        self.failUnlessEqual(message.action, message.ActionType.DISABLED)

    def test_disable_not_found(self):
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 0)
 
        self.failUnlessRaises(PluginNotFound,
                self.plugin_registry.disable_plugin,
                'a-plugin-named-like-this-cant-possibly-exist')
        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 0)
        # we shouldn't get a message here
        self.failUnlessEqual(len(self.bus.messages), 0)

class TestPluginLoadActivation(TestPluginRegistryMixin, TestCase):
    def test_load_plugins(self):
        self.plugin_registry = PluginRegistry([self.plugins_dir])
        
        # enable one plugin
        self.plugin_registry.load_plugins(['elisa-plugin-test-simple'])

        plugins = list(self.plugin_registry.get_enabled_plugins())
        self.failUnlessEqual(len(plugins), 1)

