#include <klocale.h>
#include <qvbox.h>
#include <qlabel.h>
#include <kmessagebox.h>
#include <kaction.h>
#include <kactionclasses.h>
#include <kapplication.h>
#include <kdebug.h>
#include <qwidgetstack.h>
#include <qsplitter.h>
#include <kparts/part.h>
#include <ktrader.h>
#include <klibloader.h>
#include <kstatusbar.h>

// bleeeh
#include <apt-pkg/init.h>

#include <apt-front/manager.h>
#include <apt-front/init.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/cache/component/state.h>
#include <apt-front/cache/component/history.h>
#include <apt-front/predicate/factory.h>

#include <libept/acqprogresswidget.h>
#include <libept/progress.h>

#include "app.h"

#include <libept/dpkgpm-gui.h> // EVIL

using namespace aptFront;
using namespace aptFront::cache;
using namespace ept;

TestApp::TestApp() {
    m_pkgSystem = new PkgSystem();
    aptFront::init();
    try {
        cache::Global::get().open( Cache::OpenDefault | Cache::OpenDebtags );
    } catch (...) {
        try {
            cache::Global::get().open( Cache::OpenDefault
                                       | Cache::OpenDebtags
                                       | Cache::OpenReadOnly );
            KMessageBox::information(
                this, "The APT Database will be opened"
                " in read-only mode, this means you"
                " cannot install/uninstall/upgrade"
                " anything. You have to run this program"
                " as root to be able to do that.", "Read-only mode" );
        } catch (...) {
            KMessageBox::sorry(
                this, "The APT Database could not be opened!"
                " This may be caused by incorrect APT configuration"
                " or something similar. Try running apt-setup and"
                " apt-get update in terminal and see if it helps"
                " to resolve the problem.", "Could not open cache" );
            exit( 1 );
        }
    }

    // xxx unhardcode the package name somehow?
    if (cache::Global::get().packages()
        .packageByName( "libqt-perl" ).isInstalled())
        putenv( "DEBIAN_FRONTEND=kde" );

    observeComponent< component::State >();

    cache::Global::get().addComponent(
        m_history = new component::History< component::State >() );

    m_stack = new QWidgetStack( this );

    m_stack->addWidget( m_list = new ept::View( m_stack ) );
    m_stack->addWidget( m_sources = new ept::SourcesEditor(
                            _config->FindFile(
                                "Dir::Etc::sourcelist").c_str(),
                            this ) );

    connect ( m_sources, SIGNAL( close() ),
              this, SLOT( closeSources() ) );

    // set up preview widget
    m_stack->addWidget( m_preview = new ept::View( m_stack ) );
    m_preview->filterList()->setHiddenPredicate(
        predicate::adapt< entity::Entity >(
            (not predicate::Package::member( &entity::Package::markedKeep ))
            or predicate::Package::member( &entity::Package::isBroken )
            or predicate::Package::member( &entity::Package::willBreak ) ) );

    QValueList< int > szl;
    szl.append( 0 );
    szl.append( 1 );
    m_preview->setSizes( szl );

    m_stack->addWidget( m_progress = new ept::AcqProgressWidget( m_stack ) );

    /* connect( m_list, SIGNAL( filterChanged( Lister::Predicate ) ),
       m_filters, SLOT( showPredicate( Lister::Predicate ) ) ); 
    connect( m_filters, SIGNAL( predicateChanged( Lister::Predicate ) ),
    m_list, SLOT( baseSet( Lister::Predicate ) ) ); */

    setupActions();
    setActionsEnabled( false );
    setupGUI();

    statusBar()->message( "Initializing..." );
    statusBar()->insertItem( "", 0 );
    statusBar()->insertItem( "", 1 );
    ept::Progress *pr = new ept::Progress();
    pr->setStatusBar( statusBar() );
    cache::Global::get().setProgress( pr );

    QTimer::singleShot(
        0, this,
        SLOT( delayed() ) );

    connect( m_list, SIGNAL( actionsChanged( ept::Lister * ) ),
             this, SLOT( updateListerActions( ept::Lister * ) ) );

    setCentralWidget( m_stack );
    loadKonsole();
}

void TestApp::delayed() {
    m_list->cleanRebuild();
    m_preview->cleanRebuild();
    setActionsEnabled( true );
    statusBar()->clear();
    notifyPostChange(0);
}

template< typename T >
void TestApp::plugAptActions() {
    utils::Range< actor::Actor< T > > r = actor::Global< T >::list();
    for (; r != r.last(); ++r) {
        m_actions.push_back(
            new KAction( QString( r->name() ), "",
                         0, this,
                         SLOT( listerAction() ), actionCollection(),
                         r->name().c_str() ) );
    }
}

void TestApp::setupActions()
{
    new KAction(
        i18n( "Fetch Updates" ), "adept_update",
        0, this, SLOT( update() ), actionCollection(),
        "update" );
    m_upgrade = new KAction(
        i18n( "Safe Upgrade" ), "adept_upgrade",
        0, this, SLOT( upgrade() ), actionCollection(),
        "upgrade" );
    m_distUpgrade = new KAction(
        i18n( "Full Upgrade" ), "adept_distupgrade",
        0, this, SLOT( distUpgrade() ), actionCollection(),
        "dist-upgrade" );
    m_commit = new KAction(
        i18n( "Commit Changes" ), "adept_commit",
        0, this, SLOT( commit() ), actionCollection(),
        "commit" );
    m_sourcesAction = new KToggleAction(
        i18n( "Manage Repositories" ), "adept_sourceseditor",
        0, this, SLOT( sources() ), actionCollection(),
        "sourceseditor" );
    m_previewAction = new KToggleAction(
        i18n( "Preview Changes" ), "adept_preview",
        0, this, SLOT( preview() ), actionCollection(),
        "preview" );
    m_undo = KStdAction::undo( this, SLOT( undo() ), actionCollection() );
    m_redo = KStdAction::redo( this, SLOT( redo() ), actionCollection() );

    m_previewAction->setChecked( false );
    m_sourcesAction->setChecked( false );
    m_commit->setEnabled( false );
    plugAptActions< entity::Package >();
    // plugAptActions< Action< cache::component::State > >();
    KStdAction::quit( this, SLOT( close() ), actionCollection() );
    setHistoryEnabled( false );
    createStandardStatusBarAction();
}

void TestApp::checkpoint() {
    setHistoryEnabled( false );
    m_history->checkpoint();
    setHistoryEnabled( true );
}

void TestApp::setHistoryEnabled( bool e ) {
    if (e) {
        m_undo->setEnabled( m_history->canUndo() );
        m_redo->setEnabled( m_history->canRedo() );
    } else {
        m_undo->setEnabled( false );
        m_redo->setEnabled( false );
    }
}

void TestApp::setActionsEnabled( bool e )
{
    KActionPtrList a = actionCollection()->actions();
    for (KActionPtrList::iterator i = a.begin(); i != a.end(); ++i)
        if ((*i)->name() == QString( "update" )
            || (*i)->name() == QString( "commit")
            || (*i)->name() == QString( "upgrade" )
            || (*i)->name() == QString( "dist-upgrade" )
            )
            (*i)->setEnabled( e && cache::Global::get().writeable() );
        else if ((*i)->name() == QString( "preview" ))
            (*i)->setEnabled( e );
    setHistoryEnabled( e );
}

void TestApp::listerAction() {
    /* Lister::Vector sel = m_list->selection();
    aptAction< Lister::Action >( sel.begin(), sel.end() );
    updateListerActions( m_list ); */
}

template< typename T, typename In >
void TestApp::aptAction( In b, In e ) {
    const KAction *a = dynamic_cast<const KAction *>( sender() ); // HACK
    typename T::Vector v = T::list();
    for (typename T::Vector::iterator i = v.begin(); i != v.end(); ++i) {
        if( a->name() == i->name() ) {
            (*i)( b, e );
            return;
        }
    }
}

void TestApp::undo() {
    setHistoryEnabled( false );
    m_history->undo();
    setHistoryEnabled( true );
}

void TestApp::redo() {
    setHistoryEnabled( false );
    m_history->redo();
    setHistoryEnabled( true );
}

void TestApp::notifyPostRebuild( component::Base *b )
{
    notifyPostChange( b );
}

void TestApp::notifyPreChange( component::Base * )
{
    checkpoint();
}

void TestApp::notifyPostChange( component::Base * )
{
    cache::component::State &s = cache::Global::get().state();
    kdDebug() << "TestApp::notifyPostChange()" << endl;
    statusBar()->changeItem(
        QString( " Changes: install %1, upgrade %2, remove %3 packages " )
        .arg( s.newInstallCount() ).arg( s.upgradeCount() )
        .arg( s.removeCount() ), 0 );
    statusBar()->changeItem(
        QString( " Currently %1 installed, %2 upgradable, %3 available packages " )
        .arg( s.installedCount() ).arg( s.upgradableCount() )
        .arg( s.availableCount() ), 1 );
    m_commit->setEnabled( s.changed() );
    m_upgrade->setEnabled( s.upgradableCount() );
    m_distUpgrade->setEnabled( s.upgradableCount() );
}

void TestApp::closeEvent( QCloseEvent *e ) {
    cache::component::State &s = cache::Global::get().state();
    if (s.changed()) {
        if (KMessageBox::warningYesNo(
                this, "You have done changes that were left uncommited. "
                "Are you sure you want to exit? ",
                "Uncommited changes, really quit?" ) == KMessageBox::Yes)
            e->accept();
    } else
        e->accept();
}

void TestApp::foregroundClosed()
{
    m_stack->raiseWidget( m_list );
}

void WaitForLister::waiting()
{
    kdDebug() << "WaitForLister::waiting()" << endl;
    if (app->m_list->lister()->busy() || app->m_preview->lister()->busy())
        QTimer::singleShot( 100, this, SLOT( waiting() ) );
    else {
        QTimer::singleShot( 0, app, slot );
        deleteLater();
    }
}

void TestApp::update() {
    closeModes();
    setActionsEnabled( false );
    if (m_list->lister()->busy() || m_preview->lister()->busy()) {
        new WaitForLister( this, SLOT( update() ) );
        return;
    }
    aptFront::Manager m;
    m.setProgressCallback( m_progress->callback() );
    m.setUpdateInterval( 100000 );
    try {
        m_stack->raiseWidget( m_progress );
        m.update();
    } catch (...) {
        KMessageBox::sorry(
            this, "There was an error downloading updates. ",
            "Could not fetch updates" );
    }
    kdDebug() << "closing progress widget" << endl;
    m_stack->raiseWidget( m_list );
    setActionsEnabled( true );
}

void TestApp::commit() {
    closeModes();
    setActionsEnabled( false );
    if (m_list->lister()->busy() || m_preview->lister()->busy()) {
        new WaitForLister( this, SLOT( commit() ) );
        return;
    }
    aptFront::Manager m;
    m.setProgressCallback( m_progress->callback() );
    m.setUpdateInterval( 100000 );
    try {
        m_stack->raiseWidget( m_progress );
        m.download();
        m_stack->raiseWidget( m_konsole->widget() );
        m.commit();
    } catch (...) {
        KMessageBox::sorry(
            this, "There was an error commiting changes. "
            "Possibly there was a problem downloading some "
            "packages or the commit would break packages. ",
            "Could not commit changes" );
        // FIXME: this should be handled by libapt-front
        cache::Global::get().reopen();
    }
    m_stack->raiseWidget( m_list );
    setActionsEnabled( true );
}

void TestApp::loadKonsole() {
    KLibFactory* factory = KLibLoader::self()->factory( "libsanekonsolepart" );
    if (!factory)
        factory = KLibLoader::self()->factory( "libkonsolepart" );
    assert( factory );
    m_konsole = static_cast<KParts::Part*>(
        factory->create( m_stack, "konsolepart", "QObject",
                         "KParts::ReadOnlyPart" ) );
    assert( terminal() );
    terminal()->setAutoDestroy( false );
    terminal()->setAutoStartShell( false );
    m_stack->addWidget( m_konsole->widget() );
    QStrList l; l.append( "echo" ); l.append( "-n" );
    terminal()->startProgram( "/bin/echo", l );
    m_pkgSystem->setTerminal( m_konsole );
}

ExtTerminalInterface *TestApp::terminal() {
    return static_cast<ExtTerminalInterface*>(
        m_konsole->qt_cast( "ExtTerminalInterface" ) );
}

void TestApp::upgrade() {
    closeModes();
    cache::Global::get().state().upgrade();
}

void TestApp::distUpgrade() {
    closeModes();
    cache::Global::get().state().distUpgrade();
}

void TestApp::preview() {
    if (!previewActive()) {
        m_stack->raiseWidget( m_list );
        m_list->lister()->scheduleRebuild();
    } else {
        closeSources();
        m_stack->raiseWidget( m_preview );
        m_preview->lister()->scheduleRebuild();
    }
}

void TestApp::sources() {
    if (!sourcesActive()) {
        m_stack->raiseWidget( m_list );
    } else {
        closePreview();
        m_stack->raiseWidget( m_sources );
        m_sources->reset();
    }
}

void TestApp::closePreview() {
    if (previewActive()) {
        m_previewAction->setChecked( false );
        preview();
    }
}

void TestApp::closeSources() {
    if (sourcesActive()) {
        m_sourcesAction->setChecked( false );
        sources();
    }
}

#include "app.moc"
