/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: resolver.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "segs.h"
#include "dm.h"
#include "resolver.h"


/*
 *  Functions:  free_seglist  &  add_to_seglist
 *
 *  Called to create a list of segment storage objects
 *  and to walk the list and free the list nodes.  Used by
 *  the resolver code to resolve differences between DM
 *  kernel objects and EVMS engine objects.
 */
typedef struct seglist_s {
        DISKSEG *seg;
        struct seglist_s *next;
        int minor;
} seglist_t;
static void free_seglist( seglist_t *slist)
{
        seglist_t *item=slist;
        seglist_t *next;

        while (item) {
                next = item->next;
                free(item);
                item = next;
        }

}
static int add_to_seglist( seglist_t **slist, DISKSEG *seg, int minor)
{
        int rc;
        seglist_t *new_item;
        seglist_t *item;

        LOG_ENTRY();

        new_item = malloc(sizeof(seglist_t));
        if (new_item) {

                new_item->seg=seg;
                new_item->next=NULL;
                new_item->minor=minor;

                if (*slist==NULL) {
                        *slist = new_item;
                }
                else {
                        for (item=*slist;item->next!=NULL;item=item->next);
                        item->next = new_item;
                }

                rc = 0;
        }
        else {
                rc = ENOMEM;
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function: matching metadata
 *
 *  Called to test if two segment storage objects have matching
 *  metadata ... starting LBA and SIZE.
 */
static inline boolean matching_metadata( DISKSEG *kernel_seg, DISKSEG *engine_seg )
{
        if ( kernel_seg->start == engine_seg->start &&
             kernel_seg->size  == engine_seg->size ) {
                return TRUE;
        }
        else {
                return FALSE;
        }
}

/*
 *  Function: schedule deactivate
 *
 *  Called to schedule the deactivation of a DM kernel segment
 *  at EVMS commit phase Setup (0). This occurs when we run
 *  discovery and find an active kernel DM object for which there
 *  is NO supporting metadata on disk. For example, the user went
 *  and ran FDISK to delete a partition.  Well, DM doesnt know
 *  about it and still has an active segment object even though
 *  the metadata on disk doesnt support it.  See ... defsegmgr.c
 *  commit code to see how this is handled.
 */
static int schedule_deactivate( LOGICALDISK *ld, DISKSEG *kseg )
{
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        DISKSEG *mbr=NULL;
        int rc=EINVAL;


        LOG_ENTRY();
        LOG_DEBUG("kernel segment %s needs to be deactivated\n", kseg->name );

        disk_pdata = get_disk_private_data(ld);
        mbr        = get_mbr_from_seglist(ld->parent_objects);

        if (disk_pdata && mbr) {

                if (EngFncs->insert_thing(disk_pdata->deactivate_object_list,kseg,INSERT_BEFORE,NULL)) {
                        rc = 0;
                }
                else {
                        disk_pdata->flags |= DISK_HAS_DEACTIVATE_OBJECTS;
                        mbr->flags |= SOFLAG_DIRTY;
                        rc = EPERM;
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function: dos_schedule_dm_rename
 *
 *  Called to schedule the renaming of an active kernel DM
 *  segment to a new name.  This is done when we discover that
 *  the name of the segment (e.g. hd1) doesn't match the
 *  name that is should have ... because names are based on
 *  metadata rules ... hd1 == name of partition found in the
 *  1st partition table entry of the MBR ptable.  If the name
 *  is not in agreement with the actual partition record ...
 *  but we have matching metadata (starting lba and size ) then
 *  the kernel has the wrong name for the DM object and we
 *  schedule a rename ... that will occur if the user commits
 *  changes ... see SEG_activate in dm.c
 */
void dos_schedule_dm_rename(DISKSEG *seg)
{
        SEG_PRIVATE_DATA *pdata = (SEG_PRIVATE_DATA *)seg->private_data;

        LOG_ENTRY();

        if (seg->flags & SOFLAG_ACTIVE) {

                LOG_DEBUG("scheduling rename of kernel segment %s to %s\n",
                          seg->dev_name, seg->name );

                pdata->flags  |= SEG_NEEDS_DM_RENAME;
                seg->flags    |= SOFLAG_NEEDS_ACTIVATE;

        }
        else {
                LOG_DEBUG("segment is not active so refusing to schedule DM rename\n");
        }

        LOG_EXIT_VOID();
}

/*
 *  Function: dos_update_dm_status
 *
 *  Called to update an EVMS segment storage object with DM
 *  device information.  This is done by conveying the
 *  information from the kernel DM segment storage object over
 *  to the EVMS storage object.
 */
static void dos_update_dm_status( DISKSEG *kernel_seg, DISKSEG *engine_seg)
{
        LOG_ENTRY();

        engine_seg->dev_major = kernel_seg->dev_major;
        engine_seg->dev_minor = kernel_seg->dev_minor;

        if (kernel_seg->flags & SOFLAG_ACTIVE) {
                engine_seg->flags |= SOFLAG_ACTIVE;
                engine_seg->flags &= ~SOFLAG_NEEDS_ACTIVATE;
        }
        else {
                engine_seg->flags &= ~SOFLAG_ACTIVE;
                engine_seg->flags |= SOFLAG_NEEDS_ACTIVATE;
        }

        if (kernel_seg->flags & SOFLAG_READ_ONLY) {
                engine_seg->flags |= SOFLAG_READ_ONLY;
        }
        else {
                engine_seg->flags &= ~SOFLAG_READ_ONLY;
        }

        strncpy(engine_seg->dev_name, kernel_seg->name, EVMS_NAME_SIZE);

        if (strncmp(kernel_seg->name, engine_seg->name, EVMS_NAME_SIZE)!=0) {
                dos_schedule_dm_rename(engine_seg);
        }

        LOG_EXIT_VOID();
}

/*
 *  Function: isa_existing_dm_segment
 *
 *  Called to test if the specified DM segment object actually
 *  exists in the kernel.  This is done, following a dm_update_status
 *  call, by simply checking for device major minor numbers which
 *  would indicate that DM has a device node for the objecct.
 */
static inline boolean isa_existing_dm_segment( DISKSEG *seg )
{
        LOG_DEBUG("test if seg %s exists\n", seg->name);
        if ( seg->dev_major==0 && seg->dev_minor==0) {
                LOG_DEBUG("   NO\n");
                return FALSE;
        }
        else {
                LOG_DEBUG("   YES\n");
                return TRUE;
        }
}

/*
 *  Function:  dos_free_segment
 *
 *  Called to free a storage_object_t that was previously
 *  allocated by dos_malloc_segment.
 */
static inline void dos_free_segment(DISKSEG *seg)
{
        if (seg) {
                if (seg->private_data) free(seg->private_data);
                free(seg);
        }
}


/*
 *  Function: get device name
 *
 *  On a devfs system ... we need to change the name
 *  from ide/..../disc   to ide/..../part before appending
 *  the partition number.
 */
static void get_device_name(LOGICALDISK *ld, char *device_name)
{
        int i;

        if (ld && device_name) {

                strcpy(device_name, ld->name);

                for (i=strlen(device_name)-1; i>=0; i--) {
                        if (device_name[i] == '/') {
                                if (strncmp(&device_name[i],"/disc",5)==0) {
                                        strcpy(&device_name[i+1],"");
                                        return;
                                }
                        }
                }


        }

}


/*
 *  Function:  dos_malloc_segment
 *
 *  Called to malloc and initialize a storage_object_t for the
 *  caller.  However, we cant call any engine services to do
 *  this because we are hiding these storage objects from the
 *  EVMS engine.
 */
static inline DISKSEG * dos_malloc_segment(LOGICALDISK *ld, int minor, storage_container_t * disk_group)
{
        DISKSEG *seg = calloc(1, sizeof(storage_object_t) );
        char name[EVMS_VOLUME_NAME_SIZE+1];

        if (seg) {

                strcpy(name,"");
                get_device_name(ld,name);
                if (strlen(name)==0) {
                        dos_free_segment(seg);
                        return NULL;
                }

                if (ld->object_type != DISK) {
			if (isa_partition_number(&name[strlen(name)-1])==TRUE) {
				strcat(name, ".");  // must be embedded dos partition
			}
                }
                else {
                        if (name[strlen(name)-1] == '/') {
                                strcat(name, "part");
                        }
                }
		sprintf(seg->name, "%s%d", name, minor);

                seg->disk_group = disk_group;

                seg->plugin      = Seg_My_PluginRecord_Ptr;
                seg->object_type = SEGMENT;
                seg->private_data = calloc(1, sizeof(SEG_PRIVATE_DATA));
                if (seg->private_data) {
                        ((SEG_PRIVATE_DATA *)seg->private_data)->signature    = DOS_SEG_MGR_PDATA_SIGNATURE;
                        ((SEG_PRIVATE_DATA *)seg->private_data)->logical_disk = ld;
                        ((SEG_PRIVATE_DATA *)seg->private_data)->part_number  = minor;
                        if (minor < 5) {
                                ((SEG_PRIVATE_DATA *)seg->private_data)->flags = SEG_IS_PRIMARY_PARTITION;
                        }
                        else {
                                ((SEG_PRIVATE_DATA *)seg->private_data)->flags = SEG_IS_LOGICAL_PARTITION;
                        }

                }
        }

        return seg;
}

/*
 *  Function:  get_kernel_seg_by_minor
 *
 *  Called to locate the DM object corresponding to the specified
 *  minor number and return an EVMS storage object representing the DM
 *  object if the object is active in the kernel.
 */
static DISKSEG * get_kernel_seg_by_minor( LOGICALDISK *ld, int minor, storage_container_t * disk_group )
{
        DISKSEG *seg=NULL;
        dm_target_t *targets=NULL;
        dm_device_t *dev = NULL;
        int rc=ENOMEM;

        LOG_ENTRY();
        LOG_DEBUG("ld= %s  minor= %d\n", ld->name, minor);

        seg = dos_malloc_segment(ld, minor, disk_group);
        if (seg) {

                LOG_DEBUG("Looking for %s\n", seg->name);
                rc = EngFncs->dm_update_status(seg);
                if (!rc) {
                        if (isa_existing_dm_segment(seg)==TRUE) {
                                rc = EngFncs->dm_get_targets(seg, &targets);
                                if ( !rc && targets != NULL) {
                                        dev = (dm_device_t *) targets->data.linear;
                                        seg->start = dev->start;
                                        seg->size  = targets->length;
                                }
                                if (targets) EngFncs->dm_deallocate_targets(targets);
                        }
                        else {
                                rc=ENODEV;
                        }
                }

        }

        if (rc) {
                if (seg) dos_free_segment( seg );
                seg = NULL;
        }

        LOG_DEBUG("returning seg= %p\n", seg);
        LOG_EXIT_PTR(seg);
        return seg;
}


/*
 *  Function:  get_engine_seg_by_minor
 *
 *  Called to locate the EVMS segment corresponding to the specified
 *  minor number.
 */
static DISKSEG * get_engine_seg_by_minor(LOGICALDISK *ld, int minor )
{
        DISKSEG *tseg;
        DISKSEG *seg=NULL;
        SEG_PRIVATE_DATA *pdata;
        list_element_t iter;

        LOG_ENTRY();
        LOG_DEBUG("ld= %s  minor= %d\n", ld->name, minor);

        LIST_FOR_EACH( ld->parent_objects, iter, tseg ) {

                if (tseg->data_type == DATA_TYPE) {
                        pdata = (SEG_PRIVATE_DATA *)tseg->private_data;
                        if (pdata->part_number == minor) {
                                LOG_DEBUG("found seg %s\n", tseg->name);
                                seg=tseg;
                                break;
                        }
                }

        }

        LOG_DEBUG("returning seg= %p\n", seg);
        LOG_EXIT_PTR(seg);
        return seg;
}


/*
 *  Function:  resolve_partitions_with_device_mapper
 *
 *  Called to resolve the on-disk metadata with active device mapper
 *  objects, deactivating and renaming as needed.
 */
int  resolve_partitions_with_device_mapper( LOGICALDISK *ld )
{
        seglist_t *k_seglist=NULL;
        seglist_t *e_seglist=NULL;
        seglist_t *klist=NULL;
        seglist_t *elist=NULL;
        int minor=0,last_engine_minor=0;
        DISKSEG *seg=NULL;
        boolean  found=FALSE;
        int rc=0;
        storage_container_t * disk_group = NULL;
        boolean  got_disk_group=FALSE;

        LOG_ENTRY();

        if (ld==NULL) {
                LOG_ERROR("entered with ld==NULL\n");
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }

        LOG_DEBUG("Logical Disk = %s\n", ld->name);


        // get engine segment objects -AND- init to NOT_ACTIVATE
        // -AND- get a nodeid to use with kernel objects.
        minor=1;
        do {
                seg = get_engine_seg_by_minor(ld,minor);
                if (seg) {
                        LOG_DEBUG("engine %s%d = %p\n", ld->name,minor, seg);

                        if (got_disk_group == FALSE) {
                                if (seg->disk_group != NULL) {
                                        disk_group = seg->disk_group;
                                        got_disk_group = TRUE;
                                }
                        }
                        seg->flags &= ~SOFLAG_ACTIVE;
                        seg->flags |= SOFLAG_NEEDS_ACTIVATE;
                        rc = add_to_seglist(&e_seglist,seg,minor);
                        if (rc) {
                                free_disk_segment( seg );
                                break;
                        }
                        last_engine_minor = minor;
                }
                ++minor;

        } while ( seg != NULL || minor < 6 );

        // always look for some logical partitions and if we have an
        // ebr chain ... always look for a couple of DM objects past
        // the end of the chain.
        if (last_engine_minor < 5) {
                last_engine_minor  = 7;
        }
        else {
                last_engine_minor += 2;
        }

        // get kernel DM segment objects
        minor=1;
        do {
                seg = get_kernel_seg_by_minor(ld,minor,disk_group);
                if (seg) {
                        LOG_DEBUG("kernel %s%d = %p\n", ld->name, minor, seg);
                        rc = add_to_seglist(&k_seglist,seg,minor);
                        if (rc) {
                                free_disk_segment( seg );
                                break;
                        }
                }
                ++minor;
        } while ( seg != NULL || minor <= last_engine_minor );

        // compare and process
        klist=k_seglist;
        while (klist) {

                found=FALSE;
                elist=e_seglist;
                while (elist && found==FALSE) {
                        if ( matching_metadata(klist->seg, elist->seg) == TRUE ) {
                                dos_update_dm_status(klist->seg, elist->seg);
                                found=TRUE;
                        }
                        elist=elist->next;
                }

                if (found==FALSE) {
                        if (schedule_deactivate(ld,klist->seg)==0) {
                                klist->seg = NULL;
                        }
                }

                klist=klist->next;
        }

        // toss kernel segment objects
        klist = k_seglist;
        while (klist) {
                if (klist->seg) dos_free_segment( klist->seg );
                klist=klist->next;
        }

        // free seg lists themselves
        if (k_seglist) free_seglist(k_seglist);
        if (e_seglist) free_seglist(e_seglist);

        LOG_EXIT_INT(0);
        return 0;
}



