/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libmac.so
 *
 *   File: helpers.c
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "mac_plugin.h"



storage_object_t *  allocate_mac_segment( storage_object_t *object )
{
        int   rc;
        DISKSEG  *seg=NULL;
        seg_private_data_t  *pdata=NULL;
        list_element_t  e;

        LOG_ENTRY();

        rc = EngFncs->allocate_segment( NULL, &seg );
        if (rc) {
                LOG_EXIT_PTR(NULL);
                return NULL;
        }

        e=EngFncs->insert_thing( seg->child_objects,
                                 object,
                                 INSERT_AFTER,
                                 NULL );
        if (e!=NULL) {
                seg->plugin      = mac_plugin;
                seg->object_type = SEGMENT;

                memcpy(&seg->geometry, &object->geometry, sizeof(geometry_t) );

                seg->private_data = calloc(1, sizeof(seg_private_data_t) );
                if (seg->private_data) {

                        pdata = (seg_private_data_t *)seg->private_data;

                        pdata->signature    = MAC_SEGMENT_PDATA_SIGNATURE ;
                        pdata->logical_disk = object;
                }
                else {
                        LOG_ERROR("call to malloc segment private storage area failed\n");
                        EngFncs->free_segment( seg );
                        seg = NULL;
                }
        }
        else {
                LOG_ERROR("call to insert DISK storage object in segment child_objects list failed, RC= %d\n", rc );
                EngFncs->free_segment( seg );
                seg = NULL;
        }

        LOG_EXIT_PTR(seg);
        return seg;
}


void free_mac_segment( DISKSEG *seg )
{
        LOG_ENTRY();
        LOG_DEBUG("segment name= %s\n", seg->name );

        if (seg->private_data) free(seg->private_data);

        EngFncs->free_segment( seg );

        LOG_EXIT_VOID();
}


DISKSEG * create_mac_metadata_segment( storage_object_t   *object,
                                       lba_t               start,
                                       sector_count_t      size,                                      
                                       u_int32_t           object_flags,
                                       char               *name )
{
        DISKSEG *metadata;
        seg_private_data_t *pdata=NULL;

        LOG_ENTRY();

        metadata = allocate_mac_segment( object );
        if (metadata) {

                pdata = (seg_private_data_t *) metadata->private_data;

                strcpy( pdata->p_record.type, "Apple disk label & pmap");
                strcpy( pdata->p_record.name, "Metadata");

                metadata->size        = size;
                metadata->start       = start;
                metadata->data_type   = META_DATA_TYPE;
                metadata->flags       = object_flags;

                if (object->object_type == DISK) {
                        sprintf( metadata->name, "%s_%s", object->name, name );
                }
                else {
                        sprintf( metadata->name, "%s.%s", object->name, name );
                }

        }

        LOG_EXIT_PTR(metadata);
        return metadata;
}


DISKSEG * create_mac_data_segment( LOGICALDISK        *ld,
                                   lba_t               start,
                                   sector_count_t      size, 
                                   mac_partition_t    *p,
                                   u_int32_t           minor,
                                   u_int32_t           pmap_index,
                                   u_int32_t           object_flags )
{
        DISKSEG *seg;
        seg_private_data_t *pdata=NULL;

        LOG_ENTRY();

        seg = allocate_mac_segment( ld );
        if (seg) {
                seg->size              = size;
                seg->start             = start;
                seg->data_type         = DATA_TYPE;
                seg->flags             = object_flags;

                pdata = (seg_private_data_t *) seg->private_data;

                memcpy((void *)&pdata->p_record, (void *)p, sizeof(mac_partition_t));

                pdata->minor           = minor;
                pdata->pmap_index      = pmap_index;

                if (ld->object_type == DISK) {
                        sprintf( seg->name, "%s%d", ld->name, minor );
                }
                else {
                        sprintf( seg->name, "%s.%d", ld->name, minor );
                }

        }

        LOG_EXIT_PTR(seg);
        return seg;
}


int  remove_mac_segment_from_list( list_anchor_t seglist, DISKSEG *seg )
{        
        LOG_ENTRY();
        LOG_DEBUG("segment name= %s\n", seg->name );

        EngFncs->remove_thing( seglist, seg );
        EngFncs->unregister_name( seg->name );

        LOG_EXIT_INT(0);
        return 0;
}


int insert_mac_segment_into_list( list_anchor_t seglist, DISKSEG *seg)
{
        int rc;

        LOG_ENTRY();

        rc = EngFncs->register_name( seg->name );

        if (rc==0) {
                rc = insert_mac_segment_into_ordered_list( seglist, seg );
                if (rc) {
                        EngFncs->unregister_name( seg->name );
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


int insert_mac_segment_into_ordered_list( list_anchor_t seglist, DISKSEG *seg )
{
        int         rc=-1;
        DISKSEG    *seg2;
        lba_t       seg2_end_lba;
        boolean     overlapping=FALSE;
        list_element_t iter, e;

        LOG_ENTRY();
        LOG_DEBUG("seg name= %s   seg start= %"PRId64"  ends= %"PRId64"  size= %"PRId64"\n", seg->name, seg->start, seg->start+seg->size-1, seg->size );


        LIST_FOR_EACH( seglist, iter, seg2 ) {

                seg2_end_lba = seg2->start + seg2->size - 1;

                // test and set ... overlapping segments flag
                if ( (  seg->start >= seg2->start )&&
                     (  seg->start <= seg2_end_lba)) {
                        overlapping = TRUE;
                }
                else if ( ( seg->start  <  seg2->start ) &&
                          ( seg2->start <= (seg->start + seg->size - 1)) ) {
                        overlapping = TRUE;
                }
                else {
                        overlapping = FALSE;
                }

                if ( overlapping == TRUE ) {

                        LOG_DEBUG("Error ... Overlapping Segments ...\n");
                        LOG_DEBUG("seg2:   name: %s\n", seg2->name );
                        LOG_DEBUG("       start: %"PRId64"\n", seg2->start );
                        LOG_DEBUG("        size: %"PRId64"\n", seg2->size );
                        LOG_DEBUG("         end: %"PRId64"\n", seg2_end_lba );
                        LOG_DEBUG(" overlap lba: %"PRId64"\n", seg->start );

                        rc = EINVAL;    // must be genuine partition overlap
                        break;          // break out of loop ... looking for insertion pt

                }

                // test for ... valid insertion point
                if (seg2->start > seg->start ) {
                        rc = 0;
                        break;
                }

        }


        switch (rc) {
        
        case 0:  /* Ok, found a segment we should insert in front of */
                e = EngFncs->insert_thing( seglist, 
                                           seg,
                                           INSERT_BEFORE|EXCLUSIVE_INSERT,
                                           EngFncs->find_in_list(seglist, seg2, NULL) );
                if (e != NULL) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }
                break;

        case -1:/* This new segment lays out on the disk at a higher */
                /* LBA than any other existing segment.  So, just    */
                /* insert at end of the segment list.                */
                e = EngFncs->insert_thing( seglist,
                                           seg,
                                           INSERT_AFTER|EXCLUSIVE_INSERT,
                                           NULL);
                if (e) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }
                break;

        default:     /* REAL ERROR ...  */
                LOG_ERROR("error, insertion failed ... RC= %d\n",  rc);
                break;
        }


        LOG_EXIT_INT(rc);
        return rc;
}

void prune_mac_seg_objects_from_list( list_anchor_t list )
{
        DISKSEG *seg;
        list_element_t iter,iter2;
        boolean prune;

        LIST_FOR_EACH_SAFE( list, iter, iter2, seg ) {

                prune = FALSE;

                if ( seg->plugin == mac_plugin ) {
                        free_mac_segment( seg );
                        prune = TRUE;            
                }

                if (prune==TRUE) EngFncs->delete_element(iter);
        }

}

