/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid0_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Raid0 MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#include "md.h"
#include "raid0_mgr.h"
#include "raid0_discover.h"

#define my_plugin_record raid0_plugin

/* Function: raid0_setup_evms_plugin
 *
 *  This function gets called shortly after the plugin is loaded by the
 *  Engine. It performs all tasks that are necessary before the initial
 *  discovery pass.
 */
static int raid0_setup_evms_plugin(engine_functions_t * functions) {
	int rc = 0;

	/* Parameter check */
	if (!functions) {
		return EINVAL;
	}

	EngFncs = functions;

	my_plugin = raid0_plugin;
	LOG_ENTRY();
	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Region Checking Functions ******/


/* All of the following md_can_* functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: raid0_can_delete
 *
 * Can we remove the specified MD logical volume?  Yes.
 */
static int raid0_can_delete( storage_object_t * region ) {

	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


static int raid0_can_replace_child(storage_object_t *region,
				   storage_object_t *child,
				   storage_object_t *new_child)
{
	int rc;
	my_plugin = raid0_plugin;
	LOG_ENTRY();
	rc = md_can_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_discover
 *
 *  Examine all disk segments and find MD PVs. Assemble volume groups
 *  and export all MD logical volumes as EVMS regions.
 *
 *  All newly created regions must be added to the output list, and all
 *  segments from the input list must either be claimed or moved to the
 *  output list.
 */
static int raid0_discover( list_anchor_t input_list,
			   list_anchor_t output_list,
			   boolean final_call ) {
	int count = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!input_list || !output_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (final_call) {
		md_discover_final_call(input_list, output_list, &count);
	} else {
		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("PV discovery complete.\n");

		// LV discovery and exporting
		raid0_discover_regions(output_list, &count, final_call);
		LOG_DETAILS("RAID0 volume discovery complete.\n");
	}

	LOG_EXIT_INT(count);
	return count;
}



/****** Region Functions ******/


static int get_create_options( option_array_t * options,
			       u_int32_t      * chunk_size) {
	int i;
	int rc = 0;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			
			case MD_OPTION_CHUNK_SIZE_INDEX:
				// Option is in 1K byte blocks
				*chunk_size = options->option[i].value.ui32;
				break;

			default:
				break;
			}

		} else {

			if (strcmp(options->option[i].name, MD_OPTION_CHUNK_SIZE_NAME) == 0) {
				// Option is in 1K byte blocks
				*chunk_size = options->option[i].value.ui32;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_create
 *
 *  Create a new MD RAID0 volume using the specified objects and options.
 */
static int raid0_create( list_anchor_t          objects,
			 option_array_t * options,
			 list_anchor_t          new_region_list ) {

	md_volume_t      * volume = NULL;
	storage_object_t * object;
	int nr_disks;
	unsigned long size = -1;
	int i, spare_disks = 0, index = 0;
	int rc = 0;
	mdp_disk_t disk;
	int chunk_size_blocks = MD_DEFAULT_CHUNK_SIZE;
	list_element_t iter1, iter2;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!objects || !options || !new_region_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	nr_disks = EngFncs->list_count(objects);

	if (nr_disks > MAX_MD_DEVICES) {
		LOG_ERROR("Too many objects (%d) given. Maximum is %d.\n", nr_disks, MAX_MD_DEVICES);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (!(volume = EngFncs->engine_alloc( sizeof(md_volume_t) ))) {
		LOG_CRITICAL("Memory error new volume structure.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		size = min(size, object->size);	 // track smallest object for super block
		volume->child_object[index] = object;
		index ++;
		EngFncs->remove_element(iter1);
	}

	get_create_options(options, &chunk_size_blocks);

	disk.number = 0;
	disk.raid_disk = 0;
	disk.state = (1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC);

	size = MD_CHUNK_ALIGN_NEW_SIZE_BLOCKS(chunk_size_blocks, VSECTORS_TO_BLOCKS(size));

	rc = md_create_first_superblock(volume, disk, pers_to_level(RAID0), chunk_size_blocks, size, nr_disks, spare_disks, (1 << MD_SB_CLEAN));

	if (rc) {
		EngFncs->engine_free(volume);
		LOG_EXIT_INT(rc);
		return rc;
	}

	for (i = 0; i < nr_disks; i++) {
		rc = md_clone_superblock(volume, i);
		if (rc) {
			for (i--; i>=0; i--) {
				EngFncs->engine_free(volume->super_array[i]);
			}
			EngFncs->engine_free(volume->super_block);
			EngFncs->engine_free(volume);
			LOG_EXIT_INT(rc);
			return rc;
		}
	}

	volume->personality = RAID0;
	volume->nr_disks = nr_disks;
	volume->next = volume_list_head;
	volume_list_head = volume;

	rc = raid0_create_region(volume, new_region_list, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: w_delete
 *
 * Worker function for raid0_delete and raid0_discard
 *
 */
static int w_delete(storage_object_t *region, list_anchor_t children, boolean tear_down)
{
	md_volume_t * volume;
	int           rc;

	LOG_ENTRY();

	// Check that this region can be removed.
	if ((rc = raid0_can_delete(region))) {
		LOG_EXIT_INT(rc);
		return rc;
	}
	volume = region->private_data;

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	// Delete the volume.
	md_delete_volume(volume, tear_down);
	region->private_data = NULL;
	EngFncs->free_region(region);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_delete
 *
 *  Remove the specified region.
 */
static int raid0_delete(storage_object_t * region, list_anchor_t children)
{
	int           rc;

	my_plugin = raid0_plugin;
	LOG_ENTRY();
	rc = w_delete(region, children, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: raid0_discard
 *
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the regions.
 */
static int raid0_discard(list_anchor_t regions)
{
	storage_object_t * region;
	list_element_t le;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	LIST_FOR_EACH(regions, le, region) {
		w_delete(region, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}

static int get_child_run( md_volume_t       * volume,
			  lsn_t               lsn,
			  sector_count_t      count,
			  storage_object_t ** child_object,
			  lsn_t             * child_lsn,
			  sector_count_t    * child_count) {

	unsigned int sect_in_chunk;
	unsigned int chunk_size_in_sectors;
	unsigned int chunksize_bits;
	raid0_conf_t *conf = mdvol_to_conf(volume);
	struct raid0_hash * hash;
	struct strip_zone * zone;
	unsigned long chunk;

	LOG_ENTRY();

	chunk_size_in_sectors = volume->super_block->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT;
	chunksize_bits = calc_log2(chunk_size_in_sectors);

	/* Sanity checks */
	if (!conf->hash_table || !conf->smallest_zone) {
		LOG_WARNING("Uninitialized raid0 configuration for %s\n",
			    volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	hash = conf->hash_table + (lsn / conf->smallest_zone->size);
	if (!hash) {
		LOG_WARNING("hash == NULL for lsn %"PRIu64"\n", lsn);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	if (!hash->zone0) {
		LOG_WARNING("hash->zone0 == NULL for lsn %"PRIu64"\n", lsn);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	if (lsn >= (hash->zone0->size + hash->zone0->zone_offset)) {
		if (!hash->zone1) {
			LOG_WARNING("hash->zone1 == NULL for lsn %"PRIu64"\n", lsn);
			LOG_EXIT_INT(EIO);
			return EIO;
		}
		zone = hash->zone1;
	} else {
		zone = hash->zone0;
	}

	sect_in_chunk = lsn & (chunk_size_in_sectors - 1);
	chunk = (lsn - zone->zone_offset) / (zone->nb_dev << chunksize_bits);
	*child_object = zone->dev[(lsn >> chunksize_bits) % zone->nb_dev];
	*child_lsn = ((chunk << chunksize_bits) + zone->dev_offset) + sect_in_chunk;
	*child_count = min(count, chunk_size_in_sectors - sect_in_chunk);

	LOG_EXIT_INT(0);
	return 0;
}


static int raid0_replace_child(storage_object_t *region,
			       storage_object_t *child,
			       storage_object_t *new_child)
{
	int rc;
	my_plugin = raid0_plugin;
	LOG_ENTRY();
	rc = md_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_add_sectors_to_kill_list
 *
 *  The kill_sectors list contains a list of sectors that need to be zeroed
 *  during the next commit. This function is very similar to read/write.
 */
static int raid0_add_sectors_to_kill_list( storage_object_t * region,
					   lsn_t              lsn,
					   sector_count_t     count ) {

	md_volume_t     *volume = (md_volume_t *)region->private_data;
	int             rc = 0;
	storage_object_t * child_object;
	lsn_t           child_lsn;
	sector_count_t  child_count;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	if (volume->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, data is suspect.\n",volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	while ((count != 0) && (rc == 0)) {
		rc = get_child_run(volume,lsn,count,&child_object,&child_lsn,&child_count);
		if (rc == 0) {
			rc = KILL_SECTORS(child_object, child_lsn, child_count);
			count -= child_count;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_commit_changes
 *
 *  Newly created regions may have the ZERO flag set. In this case, clear
 *  the first 1k of the LV.
 *
 *  All other commit operations are done in commit_container_changes.
 */
static int raid0_commit_changes( storage_object_t * region,
				 uint               phase ) {

	md_volume_t * volume;
	int         rc = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	if (!region ||  !(volume = (md_volume_t *)region->private_data)) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// Make sure this region belongs to MD, and is dirty
	if (region->plugin != raid0_plugin) {
		LOG_ERROR("Region %s does not belong to MD.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	if (!(region->flags & SOFLAG_DIRTY)) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	switch (phase) {
	case FIRST_METADATA_WRITE:
		volume->flags |= MD_DIRTY;
		volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
		rc = md_write_sbs_to_disk(volume);   // write super blocks
		region->flags &= ~SOFLAG_DIRTY;	     // mark clean
		break;
	default :
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_activate_region(storage_object_t * region)
{
	int rc = 0;
	md_volume_t   * volume = (md_volume_t *)region->private_data;
	dm_target_t *targets = NULL, *target=NULL;
	dm_target_stripe_t *stripe = NULL;
	int i, j;
	raid0_conf_t *conf = mdvol_to_conf(volume);
	
	my_plugin = raid0_plugin;
	LOG_ENTRY();
	
	for (i=0; (!rc) && (i<conf->nr_strip_zones); i++) {
		struct strip_zone *zone = conf->strip_zone + i;
		
		LOG_DEBUG("%s(zone[%d]): zone_offset=%"PRIu64", dev_offset=%"PRIu64", size=%"PRIu64" nb_dev=%d\n",
			region->name, i, zone->zone_offset, zone->dev_offset, zone->size, zone->nb_dev);
		
		target = EngFncs->dm_allocate_target(
					DM_TARGET_STRIPE,
					zone->zone_offset,
					zone->size,
					zone->nb_dev, 0);
		if (target) {
			stripe = target->data.stripe;
			stripe->num_stripes = zone->nb_dev;
			stripe->chunk_size = volume->super_block->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT;
			for (j=0; (!rc) && j<stripe->num_stripes; j++) {
				lsn_t lsn;
				storage_object_t *child_object;
				lsn_t child_lsn;
				sector_count_t child_count;
				
				lsn = zone->zone_offset + (j*stripe->chunk_size);
				
				rc = get_child_run(volume, lsn, 1, &child_object, &child_lsn, &child_count);
				if (!rc) {
					
					LOG_DEBUG("%s(zone[%d]): for lsn=%"PRIu64", %s was selected and I/O offset=%"PRIu64"\n",
						region->name, i, lsn, child_object->name, child_lsn);
				
					stripe->devices[j].major = child_object->dev_major;
					stripe->devices[j].minor = child_object->dev_minor;
					stripe->devices[j].start = zone->dev_offset;
				} else {
					LOG_ERROR("Could not find out which child to setup stripe target!\n");
				}
			}
		
		} else {
			rc = ENOMEM;
			break;
		}
		EngFncs->dm_add_target(target, &targets);
	}

	if (!rc) {
		rc = EngFncs->dm_activate(region, targets);
		if (!rc) {
			region->flags &= ~SOFLAG_NEEDS_ACTIVATE;
			LOG_DEBUG("Region %s has been activated, DM device(%d, %d)\n",
				  region->name, region->dev_major, region->dev_minor);
		}
	}

	if (targets) EngFncs->dm_deallocate_targets(targets); 	

	LOG_EXIT_INT(rc);
	return rc;

}

static int raid0_deactivate_region(storage_object_t * region)
{
	int rc;
	mdu_array_info_t info;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	rc = md_ioctl_get_array_info(region, &info);
	if (rc == 0) {
		/*
		 * This MD array was started by another tool.
		 * Stop the array via ioctl to the kernel MD driver.
		 */
		 rc = md_deactivate_region(region);
	} else {
		rc = EngFncs->dm_deactivate(region);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_get_option_count
 *
 *  Determine the type of Task that is being performed, and return
 *  the number of options that are available for that Task.
 */
static int raid0_get_option_count( task_context_t * task ) {
	int count = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	switch (task->action) {
	case EVMS_Task_Create:
		count = 1;
		break;

	default:
		count = 0;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}


/* Function: raid0_init_task
 *
 *  Determine the type of Task that is being performed, and set up the
 *  context structure with the appropriate initial values.
 */
static int raid0_init_task( task_context_t * context ) {

	int rc = 0;
	list_anchor_t tmp_list;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:

		context->option_descriptors->count = 1;

		// option 0 is chunk size
		context->option_descriptors->option[0].flags = 0;
		SET_POWER2_LIST(context->option_descriptors->option[0].constraint.list, MD_MIN_CHUNK_SIZE, MD_MAX_CHUNK_SIZE);
		context->option_descriptors->option[0].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[0].help = NULL;
		SET_STRING(context->option_descriptors->option[0].name,MD_OPTION_CHUNK_SIZE_NAME);
		SET_STRING(context->option_descriptors->option[0].tip, "Size of IO to each member of the array (also refered to as stripe size)." );
		SET_STRING(context->option_descriptors->option[0].title, "Chunk Size:" );
		context->option_descriptors->option[0].type = EVMS_Type_Unsigned_Int32;
		context->option_descriptors->option[0].unit = EVMS_Unit_Kilobytes;
		context->option_descriptors->option[0].value.ui32 = MD_DEFAULT_CHUNK_SIZE;

		// get a list of all valid input disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					 DATA_TYPE,
					 NULL,
					 NULL,
					 VALID_INPUT_OBJECT,
					 &tmp_list);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);

		context->min_selected_objects = 2;
		context->max_selected_objects = MAX_MD_DEVICES;
		break;
	case MD_RAID0_FUNCTION_FIX:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;
		break;

	case MD_RAID0_RESTORE_SUPERBLOCK:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_set_option
 *
 *  Determine the type of Task that is being performed. Then examine the
 *  desired option (using the index), and verify that the given value is
 *  appropriate. Reset the value if necessary and possible. Adjust other
 *  options as appropriate.
 */
static int raid0_set_option( task_context_t * context,
			     u_int32_t        index,
			     value_t        * value,
			     task_effect_t  * effect ) {
	int rc = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context || !value || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		switch (index) {
		
		case MD_OPTION_CHUNK_SIZE_INDEX:
			// Option is in kilobytes
			context->option_descriptors->option[index].value.ui32 = value->ui32;
			break;

		default:
			break;
                }

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_set_objects
 *
 *  Determine the type of task, and then validate that the objects on the
 *  "selected" list are valid for that task. If so, adjust the option
 *  descriptor as appropriate.
 */
static int raid0_set_objects( task_context_t * context,
			      list_anchor_t          declined_objects,
			      task_effect_t  * effect ) {
	int rc = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context || !declined_objects || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		// Assume front end has selected valid objects from the
		// acceptable_objects list.  Could add safety checking.
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_get_info
 *
 *  Return MD-specific information about the specified region. If the
 *  name field is set, only return the "extra" information pertaining
 *  to that name.
 */
static int raid0_get_info( storage_object_t       * region,
			   char                   * name,
			   extended_info_array_t ** info_array ) {

	md_volume_t * volume = NULL;
	int           rc= 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// Make sure this is an MD region
	if (region->plugin != raid0_plugin) {
		LOG_ERROR("Region %s is not owned by MD RAID1\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_get_plugin_info
 *
 *  Return information about the MD plugin. There is no "extra"
 *  information about MD, so "name" should always be NULL.
 */
static int raid0_get_plugin_info( char                   * name,
				  extended_info_array_t ** info_array ) {

	extended_info_array_t   * info = NULL;
	char                    buffer[50] = {0};
	int                     i = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (!name) {
		// Get memory for the info array
		if (!(info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*5))) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		// Short Name
		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid0_plugin->short_name);
		i++;

		// Long Name
		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid0_plugin->long_name);
		i++;

		// Plugin Type
		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Region Manager");
		i++;

		// Plugin Version
		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required Engine Services Version
		SET_STRING(info->info[i].name, "Required_Engine_Version");
		SET_STRING(info->info[i].title, "Required Engine Services Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine services that this plug-in requires.  It will not run on older versions of the Engine services.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid0_plugin->required_engine_api_version.major, raid0_plugin->required_engine_api_version.minor, raid0_plugin->required_engine_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required Plug-in API Version
		SET_STRING(info->info[i].name, "Required_Plugin_Version");
		SET_STRING(info->info[i].title, "Required Plug-in API Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine plug-in API that this plug-in requires.  It will not run on older versions of the Engine plug-in API.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid0_plugin->required_plugin_api_version.plugin.major, raid0_plugin->required_plugin_api_version.plugin.minor, raid0_plugin->required_plugin_api_version.plugin.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;
	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	info->count = i;
	*info_array = info;
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: raid0_read
 *
 *  Perform a logical-to-physical remapping, and send the read down to
 *  the next plugin.
 */
static int raid0_read( storage_object_t * region,
		       lsn_t              lsn,
		       sector_count_t     count,
		       void             * buffer ) {

	int                rc = 0;
	md_volume_t      * volume = (md_volume_t *)region->private_data;
	storage_object_t * child_object;
	lsn_t              child_lsn;
	sector_count_t     child_count;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (volume->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n",volume->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to read past end of region %s sector=%"PRIu64"\n",volume->name,lsn+count);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	rc = md_region_rw(region, lsn, count, buffer, 0);
	if (rc) {
		rc = 0;
		while ((count != 0) && (rc == 0)) {
			rc = get_child_run(volume, lsn, count, &child_object, &child_lsn, &child_count);
			if (rc == 0) {
				rc = READ(child_object, child_lsn, child_count, buffer);
				lsn += child_count;
				count -= child_count;
				buffer += child_count << EVMS_VSECTOR_SIZE_SHIFT;
			}
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_write
 *
 *  Perform a logical-to-physical remapping, and send the write down to
 *  the next plugin.
 */
static int raid0_write( storage_object_t * region,
			lsn_t              lsn,
			sector_count_t     count,
			void             * buffer ) {

	int                rc = 0;
	md_volume_t      * volume = (md_volume_t *)region->private_data;
	storage_object_t * child_object;
	lsn_t              child_lsn;
	sector_count_t     child_count;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (volume->flags & MD_CORRUPT) {
		MESSAGE("MD Object %s is corrupt, writing data is not allowed.\n",volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}
	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%"PRIu64"\n",volume->name,lsn+count);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	rc = md_region_rw(region, lsn, count, buffer, 1);
	if (rc) {
		rc = 0;
		while ((count != 0) && (rc == 0)) {
			rc = get_child_run(volume, lsn, count, &child_object, &child_lsn, &child_count);
			if (rc == 0) {
				rc = WRITE(child_object, child_lsn, child_count, buffer);
				lsn += child_count;
				count -= child_count;
				buffer += child_count << EVMS_VSECTOR_SIZE_SHIFT;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function:  raid0_get_plugin_functions
 */
static int raid0_get_plugin_functions(storage_object_t        *region,
				      function_info_array_t * * functions) {

	int rc = 0;
	function_info_array_t * fia;
	int function_count;
	md_volume_t * md_volume;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	/*
	 * If object is NULL, that means the user is asking for plug-in
	 * functions on the plug-in.  We don't have any plug-in functions that
	 * are global for the plug-in.
	 */
	if (region == NULL) {
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	md_volume = (md_volume_t *) region->private_data;

	if (md_volume->flags & MD_CORRUPT) {
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) + sizeof(function_info_t) * (MD_RAID0_FUNCTION_COUNT - 1));

	if (fia != NULL) {
		function_count = 0;

		/*
		 * Our functions are only allowed if the Engine is opened for
		 * writing.
		 */
		if (EngFncs->get_engine_mode() & ENGINE_WRITE) {

			/*
			 * If the array needs fixing, then that is the only function
			 * available.
			 */
			if (raid0_verify_and_fix_array(md_volume, 0, 0)) {

				fia->info[function_count].function = MD_RAID0_FUNCTION_FIX;
				SET_STRING(fia->info[function_count].name, "fix");
				SET_STRING(fia->info[function_count].title, "Fix");
				SET_STRING(fia->info[function_count].verb, "Fix");
				SET_STRING(fia->info[function_count].help, "The RAID array has inconsistent metadata.  Use this function to fix the metadata.");

				function_count++;

			} else {

				if (md_can_restore_saved_sb(region)) {
					fia->info[function_count].function = MD_RAID0_RESTORE_SUPERBLOCK;
					SET_STRING(fia->info[function_count].name, "ressuperblock");
					SET_STRING(fia->info[function_count].title, "Restore orginal major/minor");
					SET_STRING(fia->info[function_count].verb, "Restore");
					SET_STRING(fia->info[function_count].help, "Use this function to restore the original major and minor of all devices made up the MD array.");
					function_count++;
				}
			}
		}

		fia->count = function_count;
		*functions = fia;

	} else {
		LOG_CRITICAL("Error allocating memory for an action info array.\n");
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function:  raid0_plugin_function
 */
static int raid0_plugin_function(storage_object_t * object,
				 task_action_t      action,
				 list_anchor_t            objects,
				 option_array_t   * options) {

	int rc = 0;
	md_volume_t * volume = (md_volume_t *) object->private_data;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	if ((action < EVMS_Task_Plugin_Function) ||
	    (action >= EVMS_Task_Plugin_Function + MD_RAID0_FUNCTION_COUNT)) {
		LOG_ERROR("Action code 0x%x is out of range.\n", action);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	switch (action) {
		case MD_RAID0_FUNCTION_FIX:
			raid0_verify_and_fix_array(volume, 1, 0);
			break;
		case MD_RAID0_RESTORE_SUPERBLOCK:
			md_restore_saved_sb(volume);
			break;
		default:
			rc = EINVAL;
	}	

	if (rc == 0) {
		volume->region->flags |= SOFLAG_DIRTY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static void free_region (storage_object_t * region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;
	raid0_conf_t * conf = mdvol_to_conf(volume);

	if (conf != NULL) {
		EngFncs->engine_free(conf->strip_zone);
		EngFncs->engine_free(conf->hash_table);
	}

	EngFncs->engine_free(conf);

	md_free_volume(volume);
	LOG_EXIT_VOID();
}


static void raid0_plugin_cleanup() {

	int rc;
	list_anchor_t raid0_regions_list;
	list_element_t li;
	storage_object_t *region;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	rc = EngFncs->get_object_list(REGION, DATA_TYPE, raid0_plugin, NULL, 0, &raid0_regions_list);

	if (rc == 0) {
		LIST_FOR_EACH(raid0_regions_list, li, region) {
			free_region(region);
		}
		EngFncs->destroy_list(raid0_regions_list);
	}

        LOG_EXIT_VOID();
}


/* Function table for the MD RAID0 Region Manager */
static plugin_functions_t raid0_functions = {
	setup_evms_plugin               : raid0_setup_evms_plugin,
	cleanup_evms_plugin             : raid0_plugin_cleanup,
	can_delete                      : raid0_can_delete,
	can_replace_child		: raid0_can_replace_child,
	discover                        : raid0_discover,
	create                          : raid0_create,
	delete                          : raid0_delete,
	discard                         : raid0_discard,
	replace_child			: raid0_replace_child,
	add_sectors_to_kill_list        : raid0_add_sectors_to_kill_list,
	commit_changes                  : raid0_commit_changes,
	activate			: raid0_activate_region,
	deactivate			: raid0_deactivate_region,
	get_option_count                : raid0_get_option_count,
	init_task                       : raid0_init_task,
	set_option                      : raid0_set_option,
	set_objects                     : raid0_set_objects,
	get_info                        : raid0_get_info,
	get_plugin_info                 : raid0_get_plugin_info,
	read                            : raid0_read,
	write                           : raid0_write,
	get_plugin_functions		: raid0_get_plugin_functions,
	plugin_function			: raid0_plugin_function,
};



/*
 *  Initialize the local plugin record
 */

plugin_record_t raid0_plugin_record = {
	id:                     SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 6),

	version:                {major:      MAJOR_VERSION,
				 minor:      MINOR_VERSION,
				 patchlevel: PATCH_LEVEL},

	required_engine_api_version: {major:      13,
				      minor:      0,
				      patchlevel: 0},
	required_plugin_api_version: {plugin: {major:      12,
					       minor:      0,
					       patchlevel: 0} },

	short_name:             "MDRaid0RegMgr",
	long_name:              "MD Raid 0 Region Manager",
	oem_name:               "IBM",

	functions:              {plugin: &raid0_functions},

	container_functions:    NULL
};
