/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid1_discover.c
 *
 * Description: This file contains all functions related to the initial
 *              discovery of raid1 MD physical volumes and logical
 *              volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#include "md.h"
#include "raid1_mgr.h"
#include "raid1_discover.h"

#define my_plugin_record raid1_plugin

static void raid1_show_degraded(md_volume_t *volume)
{
	if (volume->flags & MD_DEGRADED) {
		int missing = volume->super_block->raid_disks - volume->active_disks;
		char * devices = "devices";
		char * device = "device";
		MESSAGE("Region %s is currently in degraded mode."
			"  To bring it back to normal state, add %d new spare %s to replace the faulty or missing %s.\n",
			volume->name, missing,
			(missing > 1) ? devices : device,
			(missing > 1) ? devices : device);
	}

}

static void raid1_show_stale_disks(md_volume_t *volume)
{
	int i;

	if (volume->stale_disks == 0) {
		return;
	}

	message_buffer[0] = '\0';

	for (i=0; i<MAX_MD_DEVICES; i++) {
		if (volume->stale_object[i]) {
			strcat(message_buffer, volume->stale_object[i]->name);
			strcat(message_buffer, " ");
		}
	}
	MESSAGE("Region %s : Invalid MD superblocks were found in object(s) : [%s]."
		"  [%s] will not be activated and should be removed from the region.\n",
		volume->name, message_buffer, message_buffer);

}

static int raid1_init_region(md_volume_t *volume, storage_object_t *region, int *new_minor)
{
	int i, rc = 0;
	storage_object_t *obj;

	LOG_ENTRY();

	md_analyze_volume(volume);
	
	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		obj = volume->child_object[i];
		if(!obj)
			obj = volume->stale_object[i];
		if (obj) {
			// if name registration failed and we changed the name, fix up all the minor numbers
			if (new_minor)
				volume->super_array[i]->md_minor = *new_minor;
			md_append_region_to_object(region, obj);
		}
	}
	
	region->size = volume->super_block->size *2; // get size from SB, convert from blocks to sectors
	region->data_type = DATA_TYPE;
	region->plugin = raid1_plugin;
	region->private_data = (void *)volume;
	region->dev_major = MD_MAJOR;
	region->dev_minor = volume->super_block->md_minor;
	if ((volume->flags & MD_DIRTY) && !(volume->flags & MD_CORRUPT)) {
		// mark region dirty
		region->flags |= SOFLAG_DIRTY;
	}
	volume->flags |= MD_DISCOVERED;
	volume->region = region;
	
	MD_CHECK_ACTIVE(region);
	if (region->flags & SOFLAG_ACTIVE) {
		mdu_array_info_t array_info;
		rc = md_ioctl_get_array_info(region, &array_info);
		if (!rc) {
			rc = md_sync_sbs((md_volume_t *)region->private_data, &array_info);
		}
		rc = 0;
	} else {
		region->flags |= SOFLAG_NEEDS_ACTIVATE;
		md_fix_dev_major_minor(volume, TRUE);
	}
	
	raid1_verify_and_fix_array(volume, 1);
	
	LOG_DETAILS("Region [%s] has been created (%s, %s, %s)\n",
		    region->name,
		    (volume->flags & MD_DISCOVERED) ? "discovered" : "BUG: not discovered",
		    (region->flags & SOFLAG_ACTIVE) ? "active" : "inactive",
		    (volume->flags & MD_DEGRADED) ? "degraded" : ((volume->flags & MD_CORRUPT) ? "corrupt" : "normal"));
	
	LOG_EXIT_INT(rc);
	return rc;
}

int raid1_create_region(md_volume_t * volume, list_anchor_t output_list, boolean final_call){
	int rc = 0;
	storage_object_t * region;
	int i, j = -1;

	LOG_ENTRY();

	if (!volume->super_block && !final_call) {
		LOG_WARNING("Volume %s does not have superblock, delaying discovery.\n",volume->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	if (!final_call && (volume->nr_disks != volume->super_block->nr_disks)) {
		/*
		 * It's not final discovery call and we have not found all the disks.
		 * If we can find 1 active disk, create the region in "degraded" mode.
		 */
		for (i = 0; i< MAX_MD_DEVICES; i++) {
			if (volume->child_object[i]) {
				if (disk_active(&volume->super_block->disks[i])) {
					LOG_WARNING("About to create %s region in degraded mode.\n",
						    volume->name);
					break;
				}
			}
		}

		if (i == MAX_MD_DEVICES) {
			LOG_WARNING("Volume %s currently does not have any active disk, delaying discovery.\n",
				    volume->name);
			LOG_EXIT_INT(0);
			return 0;
		}
	}

	LOG_DETAILS("Discovered region %s.\n",volume->name);
	if ((rc = EngFncs->allocate_region(volume->name, &region))){
		for (j = MAX_MD_MINORS -1;(rc != 0) && (j >=0) ; j--) {
			sprintf(volume->name, "md/md%d",j);
			rc = EngFncs->allocate_region(volume->name, &region);
		}
		if (j<0) {
			LOG_ERROR("No more names for MD ");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		} else {
			volume->super_block->md_minor = j+1;
		}
	}

	rc = raid1_init_region(volume, region,
			       (j >= 0) ? &volume->super_block->md_minor : NULL);

	if (volume->flags & MD_CORRUPT) {
		MESSAGE("Region %s is corrupt."
			"  Using the Fix... function, it may be possible to bring it back to normal state.\n",
			region->name);
	}

	if ((volume->flags & MD_DEGRADED) && !(volume->flags & MD_ARRAY_SYNCING)) {
		raid1_show_degraded(volume);
	}
	
	md_add_object_to_list(region, output_list);

	LOG_EXIT_INT(rc);
	return rc;
}




/* Function: discover_regions
 *
 *	run the global list of regions and pirce them together.
 */
int raid1_discover_regions( list_anchor_t output_list, int *count, boolean final_call)
{
	int rc = 0;
	md_volume_t * volume = volume_list_head;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	while (volume != NULL) {
		if ((!(volume->flags & MD_DISCOVERED)) && (volume->personality == RAID1)) {
			rc = raid1_create_region(volume, output_list, final_call);
			if (!rc && (volume->flags & MD_DISCOVERED)) {
				*count = *count + 1;
			}
		}
		volume = volume->next;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int raid1_rediscover_region( storage_object_t * region)
{
	md_volume_t   * volume;
	list_anchor_t children;
	list_anchor_t output_list;
	int rc = 0;
	mdu_array_info_t array_info;

	LOG_ENTRY();

	volume = region->private_data;
	
	if (md_is_recovery_running(region)) {
		/*
		 * If array is syncing, don't rediscover
		 * Set flag to rediscover the array when it's done.
		 */
		LOG_DEBUG("MD array %s is syncing, skipping rediscovery.\n", volume->name);
		volume->flags |= MD_ARRAY_SYNCING;
		LOG_EXIT_INT(0);
		return 0;
	}

	children = EngFncs->allocate_list();
	output_list = EngFncs->allocate_list();
	md_clear_child_list(region, children);

	rc = md_ioctl_get_array_info(region, &array_info);
	//if (!rc && (array_info.patch_version > 0)) {
	if (!rc) {
		/* for 2.6.x kernels, we will have to sync sbs via IOCTLs */
		volume->flags &= ~MD_DISCOVERED;
		rc = md_sync_sbs(volume, &array_info);
		if (rc) {
			LOG_EXIT_INT(rc);
			return rc;
		}
	} else {
		/* read all superblocks */
		md_free_volume(volume);
		md_discover_volumes(children, output_list);
	}


	region->private_data = NULL;

	volume = volume_list_head;
	while (volume != NULL) {
		if ((!(volume->flags & MD_DISCOVERED)) && (volume->personality == RAID1)) {
			region->flags &= ~(SOFLAG_DIRTY | SOFLAG_CORRUPT | SOFLAG_NEEDS_ACTIVATE | SOFLAG_NEEDS_DEACTIVATE | SOFLAG_ACTIVE);
			rc = raid1_init_region(volume, region, NULL);
		}
		volume = volume->next;
	}

	EngFncs->destroy_list(children);
	EngFncs->destroy_list(output_list);

	LOG_EXIT_INT(rc);
	return rc;
}

static void raid1_fix_master_sb(md_volume_t *vol)
{
	int i;
	mdp_super_t *sb = vol->super_block;
	boolean fix = FALSE;

	if (sb->active_disks != vol->active_disks) {
		sb->active_disks = vol->active_disks;
		fix = TRUE;
	}

	if (sb->working_disks != vol->working_disks) {
		sb->working_disks = vol->working_disks;
		fix = TRUE;
	}

	if (sb->failed_disks != vol->failed_disks) {
		sb->failed_disks = vol->failed_disks;
		fix = TRUE;
	}

	if (sb->spare_disks != vol->spare_disks) {
		sb->spare_disks = vol->spare_disks;
		fix = TRUE;
	}

	if (sb->nr_disks != vol->nr_disks) {
		sb->nr_disks = vol->nr_disks;
		fix = TRUE;
	}
	
	if (vol->stale_disks) {
		for (i=0; i<MAX_MD_DEVICES; i++) {
			if (vol->stale_object[i]) {
				md_remove_region_from_object(vol->region, vol->stale_object[i]);
				KILL_SECTORS(vol->stale_object[i], MD_NEW_SIZE_SECTORS(vol->stale_object[i]->size), MD_RESERVED_SECTORS);
				EngFncs->engine_free(vol->super_array[i]);

				vol->super_array[i] = NULL;
				vol->stale_object[i] = NULL;
				memset(&sb->disks[i], 0, sizeof(mdp_disk_t));
				fix = TRUE;
			}
		}
		vol->stale_disks = 0;
	}

	if (fix) {

		md_remove_missing_disk_entries(vol);
		
		raid1_verify_and_fix_array(vol, 0); //recursive call
		if (vol->flags & MD_CORRUPT) {
			MESSAGE("The last attempt to fix the MD region %s was not successful."
				"  There are still outstanding errors.\n",
				vol->name);
		} else {
			md_print_array_sb(message_buffer, MD_MESSAGE_BUF_SIZE, vol->super_block);
			MESSAGE("%s", message_buffer);
			MESSAGE("Region %s has been fixed.\n", vol->name);
			vol->flags |= MD_DIRTY;
			vol->region->flags |= (SOFLAG_DIRTY | SOFLAG_NEEDS_DEACTIVATE);
		}
	}

	LOG_EXIT_VOID();
}

/*
 * Function:	raid1_verify_and_fix_array
 *
 * Input:
 * 	action = 0, perform verification
 * 	action = 1, perform verification and notify the user
 * 	action = 2, perform verification and fix the array
 * Return:
 *	void.  Caller must check vol->flags
 */
void raid1_verify_and_fix_array(md_volume_t * vol, int action)
{
	mdp_super_t *sb = vol->super_block;
	LOG_ENTRY();

	if (!sb) {
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
		LOG_EXIT_VOID();
		return;
	}

	if (md_is_recovery_running(vol->region)) {
		/*
		 * Ignore verification if array is syncing.
		 * Set flag to rediscover the array when it's done.
		 */
		LOG_DEBUG("MD array %s is syncing, skipping array verification..\n", vol->name);
		vol->flags |= MD_ARRAY_SYNCING;
		LOG_EXIT_VOID();
		return;
	}
	

	if (vol->region_mgr_flags & MD_RAID1_IGNORE_VERIFY) {
		LOG_DEBUG("Skip verifying the superblock on %s\n", vol->name);
		LOG_EXIT_VOID();
		return;
	}

	vol->region->flags &= ~SOFLAG_CORRUPT;

	md_analyze_volume(vol);
	switch (action) {
	case 0:
		LOG_EXIT_VOID();
		return;
	case 1:
		if (vol->region && (md_is_region_active(vol->region) == TRUE))
			break;
		if (sb->active_disks != vol->active_disks) {
			MESSAGE("WARNING: Region %s : MD superblock has active_disks=%d, found %d.\n",
				vol->name, sb->active_disks, vol->active_disks);
		}

		if (sb->working_disks != vol->working_disks) {
			MESSAGE("WARNING: Region %s : MD superblock has working_disks=%d, found %d.\n",
				vol->name, sb->working_disks, vol->working_disks);
		}

		if (sb->failed_disks != vol->failed_disks) {
			MESSAGE("WARNING: Region %s : MD superblock has failed_disks=%d, found %d."
				"  If another disk had been added to replace the failed disk,"
				" it is safe to ignore this warning message.\n",
				vol->name, sb->failed_disks, vol->failed_disks);
		}

		if (sb->spare_disks != vol->spare_disks) {
			MESSAGE("WARNING: Region %s : MD superblock has spare_disks=%d, found %d.\n",
				vol->name, sb->spare_disks, vol->spare_disks);
		}

		if (sb->nr_disks != vol->nr_disks) {
			/* 
			 * Note: Kernel MD driver allows creating an MD array with missing disk(s).
			 * If this is the case, ignore the difference between sb->nr_disks and 
			 * volume->nr_disks.
			 */
			if ((sb->nr_disks > vol->nr_disks) && 
			    (sb->failed_disks == (sb->nr_disks - vol->nr_disks))) {
				LOG_WARNING("[%s] : MD superblock has nr_disks=%d, found %d.\n"
					"But we are ignoring the difference.\n",
					vol->name, sb->nr_disks, vol->nr_disks);
			} else {
				MESSAGE("WARNING: Region %s : MD superblock has nr_disks=%d, found %d.\n",
					vol->name, sb->nr_disks, vol->nr_disks);
			}
		}

		if (vol->stale_disks) {
			raid1_show_stale_disks(vol);
		}

		break;
	
	case 2:
		raid1_fix_master_sb(vol);
		break;
	}
	
	if (vol->flags & MD_CORRUPT) {
		vol->region->flags |= SOFLAG_CORRUPT;
	}

	LOG_EXIT_VOID();
}
