/*
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * File: replace.c
 *
 * Description: This file contains the code for the plug-in
 *              which handles replace.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include <plugin.h>
	
#include "replace.h"

static plugin_record_t   replace_plugin_record;							
static plugin_record_t * my_plugin_record = &replace_plugin_record;
			
typedef struct rep_private_data_s {
	storage_object_t * source;
	storage_object_t * target;
	copy_job_t         copy_job;
} rep_private_data_t;


static engine_functions_t * EngFncs;


/*
 * rep_setup_evms_plugin
 */
static int rep_setup_evms_plugin(engine_functions_t * functions) {

	EngFncs = functions;

	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return(0);
}


/*
 * rep_cleanup_evms_plugin
 */
static void rep_cleanup_evms_plugin() {

	LOG_ENTRY();

	LOG_EXIT_VOID();
	return;
}


/*
 * rep_can_delete
 *
 */
static int rep_can_delete(storage_object_t * obj) {

	int rc = 0;

	LOG_ENTRY();

	if (obj->plugin == my_plugin_record) {

		/*
		 * Only a new replace object, i.e., one that has not been committed yet,
		 * can be deleted.
		 */
		if (!(obj->flags & SOFLAG_NEW)) {
			LOG_DEBUG("Replace object %s cannot be deleted because it is in progress.\n", obj->name);
			rc = EINVAL;
		}

	} else {
		LOG_ERROR("%s is not a replace object.  I can't delete it.\n", obj->name);
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_can_replace_child
 *
 * Function not supported yet.
 */
static int rep_can_replace_child(storage_object_t * obj,
				 storage_object_t * child,
				 storage_object_t * new_child) {

	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return(ENOSYS);
}


static int make_replace_object(storage_object_t   * source,
			       storage_object_t   * target,
			       storage_object_t * * preplace_object) {
	int rc = 0;
	storage_object_t * replace_obj = NULL;
	char replace_obj_name[EVMS_NAME_SIZE+1];

	LOG_ENTRY();

	strcpy(replace_obj_name, "replace_");
	strncat(replace_obj_name, source->name, EVMS_NAME_SIZE - strlen(replace_obj_name));
	if (EVMS_NAME_SIZE - strlen(replace_obj_name) > 7) {
		strcat(replace_obj_name, "_with_");
		strncat(replace_obj_name, target->name, EVMS_NAME_SIZE - strlen(replace_obj_name));
	}

	switch (source->object_type) {
	case DISK:
		rc = EngFncs->allocate_logical_disk(replace_obj_name, &replace_obj);
		break;
	case SEGMENT:
		rc = EngFncs->allocate_segment(replace_obj_name, &replace_obj);
		break;
	case REGION:
		rc = EngFncs->allocate_region(replace_obj_name, &replace_obj);
		break;
	case EVMS_OBJECT:
		rc = EngFncs->allocate_evms_object(replace_obj_name, &replace_obj);
		break;
	default:
		LOG_ERROR("Object %s has an unsupported object type of %#x.\n", source->name, source->object_type);
		rc = EINVAL;
	}

	if (rc == 0) {
		rep_private_data_t * pd;

		pd = EngFncs->engine_alloc(sizeof(rep_private_data_t));

		if (pd != NULL) {
			list_element_t el;

			replace_obj->private_data = pd;

			pd->source = source;
			pd->target = target;

			/*
			 * Make the source and target objects children of the replace
			 * object.  Source is first in the list.
			 */
			el = EngFncs->insert_thing(replace_obj->child_objects,
						   source,
						   INSERT_BEFORE,
                                                   NULL);

			if (el != NULL) {
				el = EngFncs->insert_thing(replace_obj->child_objects,
							   target,
							   INSERT_AFTER,
                                                           NULL);

				if (el != NULL) {

					replace_obj->plugin = my_plugin_record;
					replace_obj->size = min(source->size, target->size);
					replace_obj->flags |= source->flags & (SOFLAG_READ_ONLY | SOFLAG_MUST_BE_TOP);
					replace_obj->flags |= SOFLAG_NEEDS_ACTIVATE;
					replace_obj->geometry = source->geometry;
					replace_obj->volume = source->volume;
					replace_obj->associated_object = source->associated_object;
					replace_obj->disk_group = source->disk_group;

					if (source->feature_header != NULL) {
						replace_obj->feature_header = EngFncs->engine_alloc(sizeof(evms_feature_header_t));
						if (replace_obj->feature_header != NULL) {
							memcpy(replace_obj->feature_header, source->feature_header, sizeof(evms_feature_header_t));

						} else {
							LOG_CRITICAL("Error getting memory for the replace object's feature header.\n");
							rc = ENOMEM;
						}
					}

				} else {
					LOG_CRITICAL("Error when inserting target object %s into the child_list of replace object %s.\n", target->name, replace_obj->name);
					EngFncs->free_evms_object(replace_obj);
					replace_obj = NULL;
					rc = ENOMEM;
				}

			} else {
				LOG_CRITICAL("Error when inserting source object %s into the child_list of replace object %s.\n", source->name, replace_obj->name);
				EngFncs->free_evms_object(replace_obj);
				replace_obj = NULL;
				rc = ENOMEM;
			}

		} else {
			LOG_CRITICAL("Error allocating memory for a private data structure for replace object %s.\n", replace_obj->name);
			rc = ENOMEM;
		}

	} else {
		LOG_CRITICAL("Error code %d when allocating a replace object.\n", rc);
	}

	if (rc == 0) {
		list_element_t el;
		/*
		 * Make the replace object the new parent of the source
		 * and target objects.
		 */
		/*
		 * The target pbject was guaranteed to be a top level object.
		 * By definition it has no parents.  So just put the
		 * replace_obj in the target's parent list.
		 */
		el = EngFncs->insert_thing(target->parent_objects,
					   replace_obj,
					   INSERT_AFTER,
                                           NULL);

		if (el != NULL) {
			el = EngFncs->insert_thing(source->parent_objects,
						   replace_obj,
						   INSERT_AFTER,
						   NULL);

			if (el == NULL) {
				LOG_CRITICAL("Error when inserting replace object %s into the parent_list of source object %s.\n", replace_obj->name, target->name);
				rc = ENOMEM;
			}

		} else {
			LOG_CRITICAL("Error when inserting replace object %s into the parent_list of target object %s.\n", replace_obj->name, source->name);
			rc = ENOMEM;
		}
	}

	*preplace_object = replace_obj;

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_discover
 *
 */
static int rep_discover(list_anchor_t input_list,
			list_anchor_t output_list,
			boolean       final_call) {
	int rc = 0;

	LOG_ENTRY();

	// Parameter check
	if (!input_list || !output_list) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	/*
	 * Can't discover replace objects since they leave no metadata on the
	 * disk.  Move the objects on the input_list to the output_list.
	 */
	rc = EngFncs->merge_lists(output_list, input_list, NULL);

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_create
 *
 * Create a new replace object using the specified objects.
 */
static int rep_create(list_anchor_t    objects,
		      option_array_t * options,
		      list_anchor_t    new_obj_list) {

	int rc = 0;
	storage_object_t * source;
	storage_object_t * target;
	storage_object_t * replace_object;
	uint nr_objs;

	LOG_ENTRY();

	/* Check parameters.  Don't care about options. */
	if (!objects || !new_obj_list) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	nr_objs = EngFncs->list_count(objects);

	if (nr_objs != 2) {
		LOG_ERROR("Must specify two objects, source and target, for the replace.\n");
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	source = EngFncs->first_thing(objects, NULL);
	target = EngFncs->last_thing(objects, NULL);

	if ((source == NULL) || (target == NULL)) {
		if (source == NULL) {
			LOG_SERIOUS("Error getting source object from input list.\n");
		}
		if (target == NULL) {
			LOG_SERIOUS("Error getting target object from input list.\n");
		}
		LOG_EXIT_INT(ENOENT);
		return(ENOENT);
	}

		rc = make_replace_object(source, target, &replace_object);

		if (rc == 0) {
			list_element_t el;

			el = EngFncs->insert_thing(new_obj_list,
						   replace_object,
						   INSERT_AFTER,
						   NULL);

			if (el == NULL) {
				// BUGBUG
				// How to clean up if InsertObject() fails?
				LOG_CRITICAL("Error inserting replace object %s into the new object list.\n", replace_object->name);
				rc = ENOMEM;
			}
		}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_discard
 *
 * We don't expect the replace plugin to ever be called for discard.
 */
static int rep_discard(list_anchor_t objects) {
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}


/*
 * rep_delete
 *
 * Rereplace the specified object.
 */
static int rep_delete(storage_object_t * obj,
		      list_anchor_t      children) {
	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) obj->private_data;

	LOG_ENTRY();

	rc = rep_can_delete(obj);

	if (rc == 0) {
		EngFncs->remove_thing(pd->source->parent_objects, obj);
		EngFncs->remove_thing(pd->target->parent_objects, obj);

		rc = EngFncs->concatenate_lists(children, obj->child_objects);

		if (rc == 0) {
			EngFncs->engine_free(pd->copy_job.title);

			EngFncs->copy_cleanup(&pd->copy_job);

			EngFncs->engine_free(pd);
			obj->private_data = NULL;

			rc = EngFncs->free_storage_object(obj);
		}
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_replace_child
 */
static int rep_replace_child(storage_object_t * obj,
			     storage_object_t * child,
			     storage_object_t * new_child) {
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return(ENOSYS);
}


/*
 * rep_add_sectors_to_kill_list
 *
 * Since kill sectors go straight to the disk and do not go through the call
 * stack, the kill sectors must be forwarded on to both the source and target.
 */
static int rep_add_sectors_to_kill_list(storage_object_t * obj,
					lsn_t              lsn,
					sector_count_t     count) {

	int rc = 0;
	rep_private_data_t * pd = obj->private_data;

	LOG_ENTRY();

	rc = pd->source->plugin->functions.plugin->add_sectors_to_kill_list(pd->source, lsn, count);

	if (rc == 0) {
		rc = pd->target->plugin->functions.plugin->add_sectors_to_kill_list(pd->target, lsn, count);

		if (rc != 0) {
			LOG_WARNING("Error code %d when writing kill sectors to target object %s.\n", rc, pd->target->name);
		}

	} else {
		LOG_WARNING("Error code %d when writing kill sectors to source object %s.\n", rc, pd->source->name);
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


#define COPY_TITLE_FMT "Copy %s to %s\n"

static void init_copy_job(rep_private_data_t * pd) {
	
	LOG_ENTRY();

	pd->copy_job.title = EngFncs->engine_alloc(sizeof(COPY_TITLE_FMT) +
						    strlen(pd->source->name) +
						    strlen(pd->target->name));

	if (pd->copy_job.title != NULL) {
		sprintf(pd->copy_job.title, COPY_TITLE_FMT, pd->source->name, pd->target->name);

	} else {
		LOG_CRITICAL("Unable to get memory for a copy progress title.\n");
	}
	pd->copy_job.description = NULL;

	pd->copy_job.src.obj = pd->source;
	pd->copy_job.src.start = 0;
	pd->copy_job.src.len = pd->source->size;
	pd->copy_job.trg.obj = pd->target;
	pd->copy_job.trg.start = 0;
	pd->copy_job.trg.len = pd->target->size;
	
	LOG_EXIT_VOID();
	return;
}


static int do_online_copy(storage_object_t * replace_obj) {

	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) replace_obj->private_data;
	dm_target_t * target;

	LOG_ENTRY();

	rc = EngFncs->copy_setup(&pd->copy_job);
	if (rc != 0) {
		LOG_SERIOUS("Error code %d when setting up a copy job: %s\n", rc, EngFncs->strerror(rc));
		LOG_EXIT_INT(rc);
		return rc;
	}

	/*
	 * Remap the replace object to point to the mirror rather than the
	 * source object.
	 */
	rc = EngFncs->dm_get_targets(replace_obj, &target);

	if (rc == 0) {
		target->data.linear->major = pd->copy_job.mirror->dev_major;
		target->data.linear->minor = pd->copy_job.mirror->dev_minor;
		target->data.linear->start = 0;

		rc = EngFncs->dm_load_targets(replace_obj, target);

		EngFncs->dm_deallocate_targets(target);

		if (rc == 0) {
			/*
			 * Tell the Engine we have a supended device.  The Engine will not
			 * write to the log while a device is suspended, in order to avoid
			 * deadlock.
			 */
			EngFncs->dm_set_suspended_flag(TRUE);

			rc = EngFncs->dm_suspend(replace_obj, TRUE);
			if (rc == 0) {
				rc = EngFncs->copy_start(&pd->copy_job);
				if (rc != 0) {
					LOG_SERIOUS("Error code %d when resuming object %s: %s\n", rc, replace_obj->name, EngFncs->strerror(rc));

					/*
					 * Mirror didn't start.  Don't
					 * load the new mapping for the
					 * replace object.
					 */
					EngFncs->dm_clear_targets(replace_obj);
				}
				rc = EngFncs->dm_suspend(replace_obj, FALSE);
			}

			EngFncs->dm_set_suspended_flag(FALSE);

		} else {
			LOG_SERIOUS("Error code %d when loading the new mirror target for the object %s: %s\n", rc, replace_obj->name, EngFncs->strerror(rc));
		}

	} else {
		LOG_SERIOUS("Error code %d when getting the target for the object %s: %s\n", rc, replace_obj->name, EngFncs->strerror(rc));
	}

	if (rc == 0) {
		rc = EngFncs->copy_wait(&pd->copy_job);

	} else {
		EngFncs->copy_cleanup(&pd->copy_job);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int do_offline_copy(storage_object_t * replace_obj) {

	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) replace_obj->private_data;

	LOG_ENTRY();

	rc = EngFncs->offline_copy(&pd->copy_job);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * rep_commit_changes
 */
static int rep_commit_changes(storage_object_t * obj,
			      uint               phase) {
	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) obj->private_data;
	logical_volume_t * vol;

	LOG_ENTRY();

	/* Sanity checks */
	if (obj->plugin->id != REPLACE_PLUGIN_ID) {
		LOG_ERROR("Object %s is not managed by the Replace plug-in.\n", obj->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	switch (phase) {
	case MOVE:
		
		init_copy_job(pd);

		if (EngFncs->can_online_copy()) {
			rc = do_online_copy(obj);

		} else {
			/* Make sure volume is not mounted for off-line replace. */
			if (!EngFncs->is_offline(obj, &vol)) {
				char * choices[] = {"Retry", "Cancel", NULL};
				int answer = 0;		// Default is "Retry".

				do {
					EngFncs->user_message(my_plugin_record, &answer, choices,
								"Object %s is part of volume %s which is currently mounted on %s.  "
								"The object cannot be replaced while the volume is mounted.  "
								"Either unmount the volume and press \"%s\" or press \"%s\" to cancel the replace.\n",
								obj->name, vol->name, vol->mount_point,
								choices[0], choices[1]);
				} while ((answer == 0) && !EngFncs->is_offline(obj, &vol));

				if (answer != 0) {
					LOG_EXIT_INT(E_CANCELED);
					return E_CANCELED;
				}
			}

			rc = do_offline_copy(obj);
		}

		if (rc == 0) {
			/*
			 * Turn off the dirty bit to signal that the
			 * copy succeeded.
			 */
			obj->flags &= ~SOFLAG_DIRTY;
		}
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_activate
 */
static int rep_activate(storage_object_t * obj) {
	
	int rc = 0;
	dm_target_t target;
	dm_device_t linear;
	rep_private_data_t * pd = obj->private_data;

	LOG_ENTRY();

	/* Sanity check */
	if (obj->plugin->id != REPLACE_PLUGIN_ID) {
		LOG_ERROR("Object %s is not managed by the Replace plug-in.\n", obj->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/*
	 * Activate with an initial mapping that is the same as the source.
	 */
	target.start = 0;
	target.length = obj->size;
	if (EngFncs->is_2_4_kernel()) {
		target.length &= ~(1);
	}
	target.type = DM_TARGET_LINEAR;
	target.data.linear = &linear;
	target.params = NULL;
	target.next = NULL;
	linear.major = pd->source->dev_major;
	linear.minor = pd->source->dev_minor;
	linear.start = obj->start;

	rc = EngFncs->dm_activate(obj, &target);

	if (rc == 0) {
		obj->flags &= ~SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_deactivate
 */
static int rep_deactivate(storage_object_t * obj) {
	
	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) obj->private_data;

	LOG_ENTRY();

	if (EngFncs->can_online_copy()) {
		rc = EngFncs->copy_cleanup(&pd->copy_job);

		if (rc == 0) {
			obj->flags &= ~SOFLAG_ACTIVE;
			obj->dev_major = 0;
			obj->dev_minor = 0;
		}
	}

	if (rc == 0) {
		rc = EngFncs->dm_deactivate(obj);
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_get_option_count
 *
 * Determine the type of Task that is being performed, and return
 * the number of options that are available for that Task.
 */
static int rep_get_option_count(task_context_t * task) {

	int count = 0;

	LOG_ENTRY();

	switch (task->action) {
	case EVMS_Task_Create:
		count = 0;
		break;

	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return(count);
}


/*
 * rep_init_task
 *
 * Determine the type of Task that is being performed, and set up the
 * context structure with the appropriate initial values.
 */
static int rep_init_task(task_context_t * context) {

	int rc = 0;
	list_anchor_t tmp_list;

	LOG_ENTRY();

	// Parameter check.
	if (!context) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		/* No options for create. */
		context->option_descriptors->count = 0;

		/* Must select one and only one target. */
		context->min_selected_objects = 1;
		context->max_selected_objects = 1;

		/* Get a list of all valid input objects. */
		EngFncs->get_object_list(DISK | SEGMENT | REGION | EVMS_OBJECT,
					   DATA_TYPE,
					   NULL,
					   NULL,
					   VALID_INPUT_OBJECT,
					   &tmp_list);

		/* Filter out objects that are too small. */

		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_set_option
 *
 * Determine the type of Task that is being performed. Then examine the
 * desired option (using the index), and verify that the given value is
 * appropriate. Reset the value if necessary and possible. Adjust other
 * options as appropriate.
 */
static int rep_set_option(task_context_t * context,
			  u_int32_t        index,
			  value_t        * value,
			  task_effect_t  * effect) {
	int rc = 0;

	LOG_ENTRY();

	// Parameter check.
	if (!context || !value || !effect) {
		LOG_EXIT_INT(EFAULT);
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_set_objects
 *
 * Determine the type of task, and then validate that the objects on the
 * "selected" list are valid for that task. If so, adjust the option
 * descriptor as appropriate.
 */
static int rep_set_objects(task_context_t * context,
			   list_anchor_t          declined_objects,
			   task_effect_t  * effect) {
	int rc = 0;

	LOG_ENTRY();

	// Parameter check.
	if (!context || !declined_objects || !effect) {
		LOG_EXIT_INT(EFAULT);
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		// Assume front end has selected valid objects from the
		// acceptable_objects list.  Could add safety checking.
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_get_info
 *
 * Return plug-in specific information about the specified object. If the
 * name field is set, only return the "extra" information pertaining
 * to that name.
 */
static int rep_get_info(storage_object_t        * obj,
			char                    * name,
			extended_info_array_t * * info_array) {

	int rc = 0;
	int i = 0;
	extended_info_array_t * info = NULL;
	rep_private_data_t * pd;
	storage_object_t * source;
	storage_object_t * target;

	LOG_ENTRY();

	/* Parameter check */
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	/* Make sure this is a replace object. */
	if (obj->plugin != my_plugin_record) {
		LOG_ERROR("Object %s is not owned by Replace.\n", obj->name);
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	if (name != NULL) {
		if (name[0] != '\0') {
			LOG_ERROR("There is no extra information for object %s.\n", obj->name);
			LOG_EXIT_INT(EINVAL);
			return(EINVAL);
		}
	}

	info = calloc(sizeof(extended_info_array_t) + sizeof(extended_info_t) * 3, 1);
	if (info == NULL) {
		LOG_ERROR("Error allocating memory for info array\n");
		LOG_EXIT_INT(ENOMEM);
		return(ENOMEM);
	}

	pd = (rep_private_data_t *) obj->private_data;
	source = EngFncs->first_thing(obj->child_objects, NULL);
	target = EngFncs->last_thing(obj->child_objects, NULL);
	if ((source == NULL) || (target == NULL)) {
		if (source == NULL) {
			LOG_SERIOUS("Error getting source object from replace object %s.\n", obj->name);
		}
		if (target == NULL) {
			LOG_SERIOUS("Error getting target object from replace object %s.\n", obj->name);
		}
		LOG_EXIT_INT(ENOENT);
		return(ENOENT);
	}

	SET_STRING(info->info[i].name, "source");
	SET_STRING(info->info[i].title, "Source object");
	SET_STRING(info->info[i].desc, "The source object for the replace");
	info->info[i].type = EVMS_Type_String;
	SET_STRING(info->info[i].value.s, source->name);
	i++;

	SET_STRING(info->info[i].name, "target");
	SET_STRING(info->info[i].title, "Target object");
	SET_STRING(info->info[i].desc, "The target object for the replace");
	info->info[i].type = EVMS_Type_String;
	SET_STRING(info->info[i].value.s, target->name);
	i++;

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_get_plugin_info
 *
 * Return information about the replace plug-in. There is no "extra"
 * information about MD, so "name" should always be NULL.
 */
static int rep_get_plugin_info(char                    * name,
			       extended_info_array_t * * info_array) {

	extended_info_array_t   * info = NULL;
	char                    buffer[50] = {0};
	int                     i = 0;

	LOG_ENTRY();

	/* Parameter check */
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	if (!name) {
		if (!(info = calloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*5, 1))) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return(ENOMEM);
		}

		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, my_plugin_record->short_name);
		i++;

		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, my_plugin_record->long_name);
		i++;

		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plug-ins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Region Manager");
		i++;

		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		SET_STRING(info->info[i].name, "Required_Engine_Version");
		SET_STRING(info->info[i].title, "Required Engine Services Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine services that this plug-in requires.  It will not run on older versions of the Engine services.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", my_plugin_record->required_engine_api_version.major, my_plugin_record->required_engine_api_version.minor, my_plugin_record->required_engine_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		SET_STRING(info->info[i].name, "Required_Plugin_Version");
		SET_STRING(info->info[i].title, "Required Plug-in API Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine plug-in API that this plug-in requires.  It will not run on older versions of the Engine plug-in API.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", my_plugin_record->required_plugin_api_version.plugin.major, my_plugin_record->required_plugin_api_version.plugin.minor, my_plugin_record->required_plugin_api_version.plugin.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return(0);
}


/*
 * rep_read
 *
 * Send the read to the source object.
 */
static int rep_read(storage_object_t * obj,
		    lsn_t              lsn,
		    sector_count_t     count,
		    void             * buffer) {

	int  rc = 0;
	rep_private_data_t * pd = obj->private_data;

	LOG_ENTRY();

	/* Parameter check. */
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
	}

	if ((lsn + count) > obj->size) {
		LOG_ERROR("Attempt to read past end of object %s at sector %"PRIu64"\n ", obj->name, lsn+count);
		LOG_EXIT_INT(EFAULT);
	}

	rc = pd->source->plugin->functions.plugin->read(pd->source, lsn, count, buffer);

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_write
 *
 * Send the write to the source and target objects.
 */
static int rep_write(storage_object_t * obj,
		     lsn_t              lsn,
		     sector_count_t     count,
		     void             * buffer) {

	int rc = 0;
	rep_private_data_t * pd = obj->private_data;

	LOG_ENTRY();

	/* Parameter check. */
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if ((lsn + count) > obj->size) {
		LOG_ERROR("Attempt to write past end of object %s at sector %"PRIu64"\n ", obj->name, lsn+count);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (obj->flags & SOFLAG_READ_ONLY) {
		LOG_ERROR("Object %s is read only.\n", obj->name);
		LOG_EXIT_INT(EACCES);
		return EACCES;
	}

	rc = pd->source->plugin->functions.plugin->write(pd->source, lsn, count, buffer);

	if (rc == 0) {
		/* Is this necessary? */
		pd->target->plugin->functions.plugin->write(pd->target, lsn, count, buffer);
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/* Function table for the Replace Feature */
static plugin_functions_t replace_functions = {
	setup_evms_plugin           : rep_setup_evms_plugin,
	cleanup_evms_plugin         : rep_cleanup_evms_plugin,
	can_delete                  : rep_can_delete,
	can_replace_child           : rep_can_replace_child,
	discover                    : rep_discover,
	create                      : rep_create,
	discard                     : rep_discard,
	delete                      : rep_delete,
	replace_child               : rep_replace_child,
	add_sectors_to_kill_list    : rep_add_sectors_to_kill_list,
	commit_changes              : rep_commit_changes,
	activate                    : rep_activate,
	deactivate                  : rep_deactivate,
	get_option_count            : rep_get_option_count,
	init_task                   : rep_init_task,
	set_option                  : rep_set_option,
	set_objects                 : rep_set_objects,
	get_info                    : rep_get_info,
	get_plugin_info             : rep_get_plugin_info,
	read                        : rep_read,
	write                       : rep_write,
};



/*
 *  Initialize the local plug-in record
 */

static plugin_record_t replace_plugin_record = {
	id:         REPLACE_PLUGIN_ID,

	version:    {major:      MAJOR_VERSION,
		     minor:      MINOR_VERSION,
		     patchlevel: PATCH_LEVEL},

	required_engine_api_version: {major:      13,
		                      minor:      0,
		                      patchlevel: 0},

	required_plugin_api_version: {plugin: {major:      12,
			                       minor:      0,
			                       patchlevel: 0}},

	short_name: "Replace",
	long_name:  "EVMS Replace",
	oem_name:   "IBM",

	functions:  {plugin: &replace_functions},

	container_functions:    NULL
};


plugin_record_t * evms_plugin_records[] = {
	&replace_plugin_record,
	NULL
};
