/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: swapfs.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <plugin.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <asm/page.h>

#include "swapfs.h"

static int fsim_rw_diskblocks( logical_volume_t*, int, int64_t, int32_t, void *, int );
static void set_mkfs_options( option_array_t *, char **, char * );

plugin_record_t *my_plugin_record = &swap_plugin_record;
engine_functions_t *EngFncs = NULL;


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                                   Common Routines                                    +
+                                                                                      +
+--------------------------------------------------------------------------------------*/



/*
 * Format the volume.
 */
static int fsim_mkfs(logical_volume_t * volume, option_array_t * options )
{
	int     rc = EINVAL;
	char   *argv[MKFS_OPTIONS_COUNT + 7];
	pid_t   pidm;
	int     fds2[2];  // pipe for stderr and stdout 0=read,1=write
	int     bytes_read;
	char    *buffer = NULL;
	int     status;


	LOG_ENTRY();
	rc = pipe(fds2);
	if (rc) {
		return(rc);
	}
	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		close(fds2[0]);
		close(fds2[1]);
		return(ENOMEM);
	}

	set_mkfs_options( options, argv, volume->dev_node );

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid( pidm, &status, WNOHANG ))) {
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				LOG_DEFAULT("mkswap output: \n%s",buffer); // on prompt if there is one
				memset(buffer,0,bytes_read); //clear out message
			}
			usleep(10000); // don't hog all the cpu
		}
		if ( WIFEXITED(status) && (WEXITSTATUS(status) != ENOENT)) {
			write(fds2[1],"\n",1);	// flush descriptor.
			do {
				bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
				if (bytes_read > 0) {
					LOG_DEFAULT("mkswap output: \n%s",buffer);
				}
			} while (bytes_read == MAX_USER_MESSAGE_LEN);
			LOG_DEFAULT("mkswap completed with rc = %d \n",status);
			rc = WEXITSTATUS(status);
		}
	} else {
		rc = EIO;
	}

	EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * NAME: set_mkfs_options
 *
 * FUNCTION: Build options array (argv) for mkfs.swap
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - mkfs options array
 *      vol_name  - volume name on which program will be executed
 *
 */
static void set_mkfs_options( option_array_t * options, char ** argv, char * vol_name )
{
	int i, opt_count = 1;
	LOG_ENTRY();

	argv[0] = "mkswap";

	for ( i=0; i<options->count; i++ ) {

		if ( options->option[i].is_number_based ) {

			switch (options->option[i].number) {
			
			case MKFS_BADBLOCK_INDEX:
				/* 'CHECK FOR BAD BLOCKS  option */
				if ( options->option[i].value.ui ) {
					argv[opt_count++] = "-c";
				}
				break;

			default:
				break;
			}

		} else {

			if ( !strcmp(options->option[i].name, "checkbad") ) {
				/* 'check for bad blocks' option */
				if ( options->option[i].value.s ) {
					argv[opt_count++] = "-c";
				}
			}

		}
	}

	argv[opt_count++] = vol_name;
	argv[opt_count] = NULL;

	LOG_EXIT_VOID();
	return;
}



/*
 * NAME: fsim_get_swap_swapfs_super_block
 *
 * FUNCTION: Get and validate a JFS swapfs_super_block
 *
 * PARAMETERS:
 *      vol_name   - volume name from which to get the swapfs_super_block
 *      sb_ptr     - pointer to swapfs_super_block
 *
 * RETURNS:
 *      (0) for success
 *      != 0 otherwise
 *
 */
static int fsim_get_swapfs_super_block( logical_volume_t *volume )
{
	int  fd;
	int  rc = 0;
	char magic[11];
	LOG_ENTRY();


	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) return EIO;

	/* get primary swapfs_super_block */
	rc = fsim_rw_diskblocks( volume, fd, SWAP_MAGIC_OFFSET_IN_BYTES, 10, magic, GET );

	if ( rc == 0 ) {
		/* see if primary swapfs_super_block is Swap */
		if (strncmp(magic, SWAPFS_MAGIC_STRING,
			    strlen(SWAPFS_MAGIC_STRING)) != 0 &&
		    strncmp(magic, SWAPFS_MAGIC_STRING2,
			    strlen(SWAPFS_MAGIC_STRING2)) != 0) {
			rc = EINVAL;
		}
	}
	EngFncs->close_volume(volume, fd);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * NAME: fsim_rw_diskblocks
 *
 * FUNCTION: Read or write specific number of bytes for an opened device.
 *
 * PARAMETERS:
 *      dev_ptr         - file handle of an opened device to read/write
 *      disk_offset     - byte offset from beginning of device for start of disk
 *                        block read/write
 *      disk_count      - number of bytes to read/write
 *      data_buffer     - On read this will be filled in with data read from
 *                        disk; on write this contains data to be written
 *      mode            - GET (read) or PUT (write)
 *
 * RETURNS:
 *      FSIM_SUCCESS (0) for success
 *      ERROR       (-1) can't lseek
 *      EINVAL
 *      EIO
 *
 */
static int fsim_rw_diskblocks( logical_volume_t * volume,
			       int      dev_ptr,
			       int64_t  disk_offset,
			       int32_t  disk_count,
			       void     *data_buffer,
			       int      mode )
{
	size_t  Bytes_Transferred;

	LOG_ENTRY();

	switch ( mode ) {
	case GET:
		Bytes_Transferred = EngFncs->read_volume(volume,dev_ptr,data_buffer,disk_count,disk_offset);
		break;
	case PUT:
		Bytes_Transferred = EngFncs->write_volume(volume,dev_ptr,data_buffer,disk_count,disk_offset);
		break;
	default:
		return EINVAL;
		break;
	}

	if ( Bytes_Transferred != disk_count ) {
		return EIO;
	}

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Test mkswap version.
 * all we care about is that it exists
 */
static int fsim_test_version( )
{
	int     rc = ENOSYS;
	char   *argv[3];
	pid_t   pidm;
	int     fds2[2];  // pipe for stderr and stdout 0=-read,1=write
	int     status;
	char    *buffer;
//	int 	bytes_read;
//	char    *ver,*end;

	LOG_ENTRY();
	rc = pipe(fds2);
	if (rc) {
		return(rc);
	}
	if (!(buffer = EngFncs->engine_alloc(1000))) {
		close(fds2[0]);
		close(fds2[1]);
		return(ENOMEM);
	}

	argv[0] = "mkswap";
	argv[1] = "-V";
	argv[2] = NULL;

	pidm = EngFncs->fork_and_execvp(NULL, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		waitpid( pidm, &status, 0 );
		if ( WIFEXITED(status) && (WEXITSTATUS(status) != ENOENT) ) {
			rc = 0;  // as long as not enoent, ok.
		} else {
			rc = ENOSYS;
		}
	} else {
		rc = EIO;
	}
	EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);

	LOG_EXIT_INT(rc);
	return rc;
}


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/


static int fs_setup( engine_functions_t *engine_function_table)
{
	int rc = 0;
	EngFncs = engine_function_table;

	LOG_ENTRY();

	/*
	 * as long as mkswap exists, ok for  this FSIM.
	 */
	rc = fsim_test_version();

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Free all of the private data item we have left on volumes.
 */
static void fs_cleanup()
{
	LOG_ENTRY();
	LOG_EXIT_VOID();
}


/*
 * Does this FSIM manage the file system on this volume?
 * Return 0 for "yes", else a reason code.
 */
static int fs_probe(logical_volume_t * volume)
{
	int  rc = 0;

	LOG_ENTRY();

	/* get and validate swapfs super block */
	rc = fsim_get_swapfs_super_block(volume);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Can mkfs this volume?
 */
static int fs_can_mkfs(logical_volume_t * volume)
{
	int  rc=0;

	LOG_ENTRY();

	/*****************************************************
	 *  FUTURE - ensure mkfs.swap exists                  *
	 *           match version with available functions  *
	 *****************************************************/

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't format. */
		rc = EBUSY;
	}
	if (volume->vol_size < SWAPFS_MIN_SIZE) {
		rc = EINVAL;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Can unmkfs this volume?
 */
static int fs_can_unmkfs(logical_volume_t * volume)
{
	int  rc=0;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	}
	LOG_EXIT_INT(rc);
	return rc;

}


/*
 * Can fsck this volume?
 */
static int fs_can_fsck(logical_volume_t * volume)
{
	int  rc=ENOSYS;  //NOPE, NO FSCK

	LOG_ENTRY();

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Get the current size of this volume
 */
static int fs_get_fs_size( logical_volume_t * volume,
			   sector_count_t   * size    )
{
	int  rc = 0;
//	struct swapfs_super_block *sb_ptr = (struct swapfs_super_block *)volume->private_data;

	LOG_ENTRY();

	*size = volume->vol_size;  // BUGBUG do min with max size

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Get the size limits for this volume.
 */
static int fs_get_fs_limits( logical_volume_t * volume,
			     sector_count_t   * min_size,
			     sector_count_t   * max_volume_size,
			     sector_count_t   * max_object_size)
{
	int rc = 0;

	LOG_ENTRY();
	*max_volume_size = (sector_count_t)PAGE_SIZE * (sector_count_t)PAGE_SIZE;  // ??
	*max_object_size = (sector_count_t)-1;  // no limit
	*min_size = (sector_count_t)PAGE_SIZE * (sector_count_t)10;  // 10 page minimum
	LOG_EXTRA("volume:%s, min:%"PRIu64", max:%"PRIu64"\n",volume->name, *min_size, *max_volume_size);
	LOG_EXTRA("fssize:%"PRIu64", vol_size:%"PRIu64"\n",volume->fs_size,volume->vol_size );

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Format the volume.
 */
static int fs_mkfs(logical_volume_t * volume, option_array_t * options )
{
	int  rc = 0;

	LOG_ENTRY();

	/* don't format if mounted */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		return EBUSY;
	}

	rc = fsim_mkfs(volume, options);

	/* probe to set up private data */
//	if ( !rc ) {
//		rc = fs_probe(volume);
//	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Expand the volume to new_size.  If the volume is not expanded exactly to
 * new_size, set new_size to the new_size of the volume.
 */
static int fs_expand( logical_volume_t * volume,
		      sector_count_t   * new_size )
{
	int     rc = 0;
	option_array_t options;

	LOG_ENTRY();
	options.count = 0;
	// since we know that the swap space is not in use,
	// we can do the expand by re-mkfsing the volume.  cheap, but easy.
	fs_mkfs(volume, &options);

	fs_get_fs_size(volume, new_size);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * "unmkfs" the volume
 */
static int fs_unmkfs(logical_volume_t * volume)
{
	int rc = EINVAL;
	int  fd;
	char magic[11];
	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	} else {

		fd = EngFncs->open_volume(volume, O_RDWR|O_EXCL, 0);
		if (fd < 0) return EIO;

		/* zero primary swapfs_super_block */
		memset(magic, 0, 11);
		rc = fsim_rw_diskblocks( volume, fd, SWAP_MAGIC_OFFSET_IN_BYTES, 10, magic, PUT );

		fd = EngFncs->close_volume(volume, fd);
		volume->private_data = NULL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Shrink the volume to new_size.  If the volume is not expanded exactly to
 * new_size, set new_size to the new_size of the volume.
 */
static int fs_shrink( logical_volume_t * volume,
		      sector_count_t     requested_size,
		      sector_count_t   * new_size )
{
	int     rc = 0;
	option_array_t options;

	LOG_ENTRY();
	options.count = 0;
	// since we know that the swap space is not in use,
	// we can do the shrink by re-mkfsing the volume.  cheap, but easy.
	fs_mkfs(volume, &options);

	fs_get_fs_size(volume, new_size);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Forget about this volume.  Don't remove the file system.  Just clean
 * up any data structures you may have associated with it.
 */
static int fs_discard(logical_volume_t * volume)
{
	LOG_ENTRY();
	/* Swap fsim has no private data. */
	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Run fsck on the volume.
 */
static int fs_fsck(logical_volume_t * volume, option_array_t * options )
{
	int rc = EINVAL;

	LOG_ENTRY();

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Return the total number of supported options for the specified task.
 */
static int fs_get_option_count(task_context_t * context)
{
	int count = 0;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_mkfs:
		count = MKFS_OPTIONS_COUNT;
		break;
	case EVMS_Task_fsck:
		count = -1;
		break;
	case EVMS_Task_Expand:
		count = 0;
		break;
	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}

/*
 * Initialize mkfs task acceptable objects by enumerating volumes, finding
 * those that have no FSIM claiming them and are of the proper size and
 * adding them to the acceptable objects list.
 */
static int fs_init_mkfs_acceptable_objects(task_context_t * context)
{
	int rc;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t * volume;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(NULL, NULL, 0, &global_volumes);
	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			/* only mkfs unformatted volumes */
			if (volume->file_system_manager == NULL &&
			    !(EngFncs->is_mounted(volume->dev_node, NULL)) &&
			    (volume->vol_size > SWAPFS_MIN_SIZE)) {
				EngFncs->insert_thing(context->acceptable_objects,
						      volume,
						      INSERT_BEFORE,
						      NULL);
			}
		}
		EngFncs->destroy_list(global_volumes);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Fill in the initial list of acceptable objects.  Fill in the minimum and
 * maximum nuber of objects that must/can be selected.  Set up all initial
 * values in the option_descriptors in the context record for the given
 * task.  Some fields in the option_descriptor may be dependent on a
 * selected object.  Leave such fields blank for now, and fill in during the
 * set_objects call.
 */
static int fs_init_task( task_context_t * context )
{
	int rc = 0;

	LOG_ENTRY();

	context->min_selected_objects = 0;
	context->max_selected_objects = 0;

	/* Parameter check */
	if (!context) {
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_mkfs:

		rc = fs_init_mkfs_acceptable_objects(context);
		if (rc) {
			break;
		}

		context->option_descriptors->count = MKFS_OPTIONS_COUNT;

		/* Log size option */
		SET_STRING(context->option_descriptors->option[MKFS_BADBLOCK_INDEX].name, "badblocks" );
		SET_STRING(context->option_descriptors->option[MKFS_BADBLOCK_INDEX].title, "Check for bad blocks" );
		SET_STRING(context->option_descriptors->option[MKFS_BADBLOCK_INDEX].tip, "Check for bad blocks when making swap space" );
		context->option_descriptors->option[MKFS_BADBLOCK_INDEX].help = NULL;
		context->option_descriptors->option[MKFS_BADBLOCK_INDEX].type = EVMS_Type_Boolean;
		context->option_descriptors->option[MKFS_BADBLOCK_INDEX].unit = EVMS_Unit_Kilobytes;
		context->option_descriptors->option[MKFS_BADBLOCK_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		context->option_descriptors->option[MKFS_BADBLOCK_INDEX].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[MKFS_BADBLOCK_INDEX].value.b = FALSE;

		context->min_selected_objects = 1;
		context->max_selected_objects = 1;
		break;

	case EVMS_Task_Expand:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		break;

	case EVMS_Task_fsck:

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);

	return rc;

}



/*
 * Examine the specified value, and determine if it is valid for the task
 * and option_descriptor index. If it is acceptable, set that value in the
 * appropriate entry in the option_descriptor. The value may be adjusted
 * if necessary/allowed. If so, set the effect return value accordingly.
 */
static int fs_set_option( task_context_t * context,
			  u_int32_t        index,
			  value_t        * value,
			  task_effect_t  * effect )
{
	int  rc= 0;

	LOG_ENTRY();

	/* Parameter check */
	if (!context || !value || !effect) {
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_mkfs:
		switch (index) {
		
		case MKFS_BADBLOCK_INDEX:
			/* 'set log size' option set? */
			context->option_descriptors->option[index].value.b = value->b;
			break;

		default:
			break;
		}
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Validate the volumes in the selected_objects list in the task context.
 * Remove from the selected objects lists any volumes which are not
 * acceptable.  For unacceptable volumes, create a declined_handle_t
 * structure with the reason why it is not acceptable, and add it to the
 * declined_volumes list.  Modify the accepatble_objects list in the task
 * context as necessary based on the selected objects and the current
 * settings of the options.  Modify any option settings as necessary based
 * on the selected objects.  Return the appropriate task_effect_t settings
 * if the object list(s), minimum or maximum objects selected, or option
 * settings have changed.
 */
static int fs_set_volumes( task_context_t * context,
			   list_anchor_t    declined_volumes,	 /* of type declined_handle_t */
			   task_effect_t  * effect )
{
	int  rc = 0;
	logical_volume_t * vol;

	LOG_ENTRY();

	if (context->action == EVMS_Task_mkfs) {

		/* get the selected volume */
		vol = EngFncs->first_thing(context->selected_objects, NULL);

		if (vol) {
			if (EngFncs->is_mounted(vol->dev_node, NULL)) {
				/* If mounted, can't mkfs.swap. */
				rc = EBUSY;
			} else {
				if ( vol->vol_size < SWAPFS_MIN_SIZE) {
					MESSAGE( "The size of volume %s is %"PRIu64" sectors.", vol->name, vol->vol_size );
					MESSAGE( "mkswap requires a minimum of %lu sectors to build a swap device.", SWAPFS_MIN_SIZE );
					rc = EPERM;
				}
			}
		} else {
			rc = ENODATA;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Return any additional information that you wish to provide about the
 * volume.  The Engine privides an external API to get the information
 * stored in the logical_volume_t.  This call is to get any other
 * information about the volume that is not specified in the
 * logical_volume_t.  Any piece of information you wish to provide must be
 * in an extended_info_t structure.  Use the Engine's engine_alloc() to
 * allocate the memory for the extended_info_t.  Also use engine_alloc() to
 * allocate any strings that may go into the extended_info_t.  Then use
 * engine_alloc() to allocate an extended_info_array_t with enough entries
 * for the number of exteneded_info_t structures you are returning.  Fill
 * in the array and return it in *info.
 * If you have extended_info_t descriptors that themselves may have more
 * extended information, set the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag
 * in the extended_info_t flags field.  If the caller wants more information
 * about a particular extended_info_t item, this API will be called with a
 * pointer to the sotrage_object_t and with a pointer to the name of the
 * extended_info_t item.  In that case, return an extended_info_array_t with
 * further information about the item.  Each of those items may have the
 * EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag set if you desire.  It is your
 * resposibility to give the items unique names so that you know which item
 * the caller is asking additional information for.  If info_name is NULL,
 * the caller just wants top level information about the object.
 */
static int fs_get_volume_info( logical_volume_t        * volume,
			       char                    * info_name,
			       extended_info_array_t * * info )
{
	int rc = EINVAL;

	LOG_ENTRY();

	/* There is no more information about any of the descriptors. */
	rc = ENOSYS;

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Returns Plugin specific information ...
 */
static int fs_get_plugin_info( char * descriptor_name, extended_info_array_t * * info )
{
	int                      rc = EINVAL;
	extended_info_array_t   *Info;
	char                     version_string[64];
	char                     required_engine_api_version_string[64];
	char                     required_fsim_api_version_string[64];

	LOG_ENTRY();

	if (info) {

		if (descriptor_name == NULL) {
			*info = NULL;	  // init to no info returned

			Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (6*sizeof(extended_info_t))  );
			if (Info) {

				Info->count = 6;

				sprintf(version_string, "%d.%d.%d",
					MAJOR_VERSION,
					MINOR_VERSION,
					PATCH_LEVEL );

				sprintf(required_engine_api_version_string, "%d.%d.%d",
					my_plugin_record->required_engine_api_version.major,
					my_plugin_record->required_engine_api_version.minor,
					my_plugin_record->required_engine_api_version.patchlevel );

				sprintf(required_fsim_api_version_string, "%d.%d.%d",
					my_plugin_record->required_plugin_api_version.fsim.major,
					my_plugin_record->required_plugin_api_version.fsim.minor,
					my_plugin_record->required_plugin_api_version.fsim.patchlevel );

				SET_STRING( Info->info[0].name, "Short Name" );
				SET_STRING( Info->info[0].title, "Short Name" );
				SET_STRING( Info->info[0].desc, "A short name given to this plugin.");
				Info->info[0].type               = EVMS_Type_String;
				Info->info[0].unit               = EVMS_Unit_None;
				SET_STRING( Info->info[0].value.s, my_plugin_record->short_name );
				Info->info[0].collection_type    = EVMS_Collection_None;
				memset( &Info->info[0].group, 0, sizeof(group_info_t));

				SET_STRING( Info->info[1].name, "Long Name" );
				SET_STRING( Info->info[1].title, "Long Name" );
				SET_STRING( Info->info[1].desc, "A long name given to this plugin.");
				Info->info[1].type               = EVMS_Type_String;
				Info->info[1].unit               = EVMS_Unit_None;
				SET_STRING( Info->info[1].value.s, my_plugin_record->long_name );
				Info->info[1].collection_type    = EVMS_Collection_None;
				memset( &Info->info[1].group, 0, sizeof(group_info_t));

				SET_STRING( Info->info[2].name, "Type" );
				SET_STRING( Info->info[2].title, "Plugin Type" );
				SET_STRING( Info->info[2].desc, "There are various types of plugins; each responsible for some kind of storage object.");
				Info->info[2].type               = EVMS_Type_String;
				Info->info[2].unit               = EVMS_Unit_None;
				SET_STRING( Info->info[2].value.s, "File System Interface Module" );
				Info->info[2].collection_type    = EVMS_Collection_None;
				memset( &Info->info[2].group, 0, sizeof(group_info_t));

				SET_STRING( Info->info[3].name, "Version" );
				SET_STRING( Info->info[3].title, "Plugin Version" );
				SET_STRING( Info->info[3].desc, "This is the version number of the plugin.");
				Info->info[3].type               = EVMS_Type_String;
				Info->info[3].unit               = EVMS_Unit_None;
				SET_STRING( Info->info[3].value.s, version_string );
				Info->info[3].collection_type    = EVMS_Collection_None;
				memset( &Info->info[3].group, 0, sizeof(group_info_t));

				SET_STRING( Info->info[4].name, "Required Engine Services Version" );
				SET_STRING( Info->info[4].title, "Required Engine Services Version" );
				SET_STRING( Info->info[4].desc, "This is the version of the Engine services that this plug-in requires. It will not run on older versions of the Engine services.");
				Info->info[4].type               = EVMS_Type_String;
				Info->info[4].unit               = EVMS_Unit_None;
				SET_STRING( Info->info[4].value.s, required_engine_api_version_string );
				Info->info[4].collection_type    = EVMS_Collection_None;
				memset( &Info->info[4].group, 0, sizeof(group_info_t));

				SET_STRING( Info->info[5].name, "Required Engine FSIM API Version" );
				SET_STRING( Info->info[5].title, "Required Engine FSIM API Version" );
				SET_STRING( Info->info[5].desc, "This is the version of the Engine FSIM API that this plug-in requires. It will not run on older versions of the Engine FSIM API.");
				Info->info[5].type               = EVMS_Type_String;
				Info->info[5].unit               = EVMS_Unit_None;
				SET_STRING( Info->info[5].value.s, required_fsim_api_version_string );
				Info->info[5].collection_type    = EVMS_Collection_None;
				memset( &Info->info[5].group, 0, sizeof(group_info_t));

				*info = Info;

				rc = 0;
			} else {
				rc = ENOMEM;
			}

		} else {
			/* There is no more information on any of the descriptors. */
			rc = EINVAL;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * How much can file system expand?
 */
static int fs_can_expand_by(logical_volume_t * volume,
			    sector_count_t   * delta)
{
	int  rc = 0;

	LOG_ENTRY();
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		rc = EBUSY; // If mounted, can't expand
	} else {
		fs_get_fs_limits( volume,		   // reset limits.
				  &volume->min_fs_size,
				  &volume->max_vol_size,
				  &volume->max_fs_size);
		if (volume->fs_size + *delta > volume->max_fs_size) {
			*delta = volume->max_fs_size - volume->fs_size;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;

}


/*
 * How much can file system shrink?
 */
static int fs_can_shrink_by(logical_volume_t * volume,
			    sector_count_t * delta)
{
	int  rc = 0;

	LOG_ENTRY();
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		rc = EBUSY; // If mounted, can't shrink
	} else {
		fs_get_fs_limits( volume,		   // reset limits.
				  &volume->min_fs_size,
				  &volume->max_vol_size,
				  &volume->max_fs_size);
		if (volume->fs_size - *delta < volume->min_fs_size) {
			*delta = volume->fs_size - volume->min_fs_size;
		}
		if (volume->min_fs_size >= volume->vol_size) {
			rc = ENOSPC;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}

static fsim_functions_t fsim_ops = {
	.setup_evms_plugin	= fs_setup,
	.cleanup_evms_plugin	= fs_cleanup,
	.probe			= fs_probe,
	.can_mkfs		= fs_can_mkfs,
	.can_unmkfs		= fs_can_unmkfs,
	.can_fsck		= fs_can_fsck,
	.get_fs_size		= fs_get_fs_size,
	.get_fs_limits		= fs_get_fs_limits,
	.can_expand_by		= fs_can_expand_by,
	.can_shrink_by		= fs_can_shrink_by,
	.expand			= fs_expand,
	.shrink			= fs_shrink,
	.mkfs			= fs_mkfs,
	.discard		= fs_discard,
	.fsck			= fs_fsck,
	.unmkfs			= fs_unmkfs,
	.get_option_count	= fs_get_option_count,
	.init_task		= fs_init_task,
	.set_option		= fs_set_option,
	.set_volumes		= fs_set_volumes,
	.get_volume_info	= fs_get_volume_info,
	.get_plugin_info	= fs_get_plugin_info
};

plugin_record_t swap_plugin_record = {
	.id = SetPluginID(EVMS_OEM_IBM,
			  EVMS_FILESYSTEM_INTERFACE_MODULE,
			  FS_TYPE_SWAPFS),
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 13,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.fsim = {
			.major		= 10,
			.minor		= 0,
			.patchlevel	= 0
		}
	},
	.short_name = "SWAPFS",
	.long_name = "Swap File System Interface Module",
	.oem_name = "IBM",
	.functions = {
		.fsim = &fsim_ops
	},
	.container_functions = NULL
};

plugin_record_t *evms_plugin_records[] = {
	&swap_plugin_record,
	NULL
};

