/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_main.c
 *
 * Description: Global and core for MD plugins
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <plugin.h>
#include <pthread.h>
#include <signal.h>
#include <time.h>
#include <wait.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/time.h>

#include "md.h"

#define my_plugin_record my_plugin

plugin_record_t	    		* linear_plugin = &linear_plugin_record;
plugin_record_t	    		* raid1_plugin = &raid1_plugin_record;
plugin_record_t	    		* raid0_plugin = &raid0_plugin_record;
plugin_record_t                 * raid5_plugin = &raid5_plugin_record;
plugin_record_t			* mp_plugin = &multipath_plugin_record;
md_volume_t			* volume_list_head = NULL;	// List of real volume groups, indexed by vg_number.
engine_functions_t		* EngFncs;		// The Engine's internal API set.
plugin_record_t			* my_plugin = NULL;
char message_buffer[MD_MESSAGE_BUF_SIZE];
plugin_record_t * evms_plugin_records[] = {&linear_plugin_record,
	                                   &raid1_plugin_record,
	                                   &raid0_plugin_record,
	                                   &raid5_plugin_record,
					   &multipath_plugin_record,
	                                   NULL};

static boolean md_driver_loaded = FALSE;

static int kernel_major = 0;
static int kernel_minor = 0;
static int kernel_patch = 0;

static void check_kernel(void)
{
	FILE *fp;
	char needle[] = "version";
	char haystack[256];
	char *ver;

	LOG_ENTRY();

	fp = fopen("/proc/version", "r");
	if (fp >= 0) {
		fread(haystack, 1, 255, fp);
		fclose(fp);
		
		ver = strstr(haystack, needle);
		if (ver) {
			sscanf(ver, "%*s %d.%d.%d", &kernel_major, &kernel_minor, &kernel_patch);

			LOG_DETAILS("Kernel version is: %d.%d.%d\n", kernel_major, kernel_minor, kernel_patch);

		} else {
			LOG_WARNING("Could not find \"version\" in the version string in /proc/version.");
		}
	} else {
		LOG_WARNING("Open of /proc/version failed with error code %d: %s\n", errno, strerror(errno));
	}

	LOG_EXIT_VOID();
}

boolean md_can_create_sb_1(void)
{
	boolean sb1 = FALSE;

	LOG_ENTRY();
	if (kernel_major == 0 && kernel_minor == 0 && kernel_patch == 0) {
		check_kernel();
	}
	if (kernel_major >= 2 && kernel_minor >= 6 && kernel_patch >= 10) {
		sb1 = TRUE;
	}
	LOG_EXIT_BOOL(sb1);
	return sb1;
}

static FILE *mdstat_open()
{
	return fopen("/proc/mdstat", "r");
}

static int load_md_module(void) {

	int rc = 0;
	pid_t pid;
	int   status;
	char * argv[3] = {"modprobe", "md", NULL};

	LOG_ENTRY();

        pid = fork();

	switch (pid) {
	case -1:
		rc = errno;
		LOG_WARNING("fork() to run \"%s %s\" returned error %d: %s\n", argv[0], argv[1], rc, strerror(rc));
		break;

	case 0:
		/* Child process */

		execvp(argv[0], argv);

		/* Should not get here unless execvp() fails. */
		rc = errno;
		LOG_WARNING("execvp() to run \"%s %s\" returned error %d: %s\n", argv[0], argv[1], rc, strerror(rc));

		/* exit() kills the GUI.  Use _exit() instead. */
		_exit(rc);
		break;

	default:
		/* Parent process */
		waitpid(pid, &status, 0);
	}

	if (rc == 0) {
		if (WIFSIGNALED(status)) {
			LOG_WARNING("\"%s %s\" was terminated by signal %s\n", argv[0], argv[1], sys_siglist[WTERMSIG(status)]);
			rc = EINTR;

		} else {
			rc = WEXITSTATUS(status);
			LOG_DEBUG("\"%s %s\" exited with error code %d: %s\n", argv[0], argv[1], rc, strerror(rc));
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

boolean load_kernel_md_driver(void)
{
	FILE *fp;
	
	LOG_ENTRY();

	if (md_driver_loaded == FALSE) {
		/* check for /proc/mdstat */

		fp = mdstat_open();
		if (fp) {
			md_driver_loaded = TRUE;
			fclose(fp);
		} else {
			/* load MD kernel module */
			load_md_module();
			fp = mdstat_open();
			if (fp) {
				md_driver_loaded = TRUE;
				fclose(fp);
			}
		}
	}

	LOG_EXIT_BOOL(md_driver_loaded);
	return md_driver_loaded;
}

static storage_object_t * md_get_region_for_object(storage_object_t *object)
{
	md_volume_t * vol = volume_list_head;
	storage_object_t *region;

	LOG_ENTRY();
	
	while (vol) {
		region = vol->region;
		if (region  && (vol->flags & MD_DISCOVERED)) {
			if (md_volume_find_object(vol, object)) {
				LOG_DEBUG("[%s] belongs to %s\n", object->name, region->name);
				LOG_EXIT_PTR(region);
				return region;
			}
		}
		vol = vol->next;
	}
	LOG_DEFAULT("MD does not own this object [%s]\n", object->name);
	LOG_EXIT_PTR(NULL);
	return NULL;
}

/* Function: md_can_replace_child
 *
 *  Can we replace a child?
 */
int md_can_replace_child(storage_object_t *region,
			 storage_object_t *child,
			 storage_object_t *new_child)
{
	int rc = 0;
	logical_volume_t *evms_volume;	
	u_int64_t child_size, new_child_size;
	md_volume_t *vol;

	LOG_ENTRY();

	/*
	 * If the volume is mounted, we can't replace a child object.
	 */
	if (EngFncs->is_offline(region, &evms_volume) == FALSE) {
		rc = EBUSY;
		goto out;
	}

	vol = (md_volume_t *)region->private_data;
	if (!vol) {
		rc = EFAULT;
		goto out;
	}

	if (region != md_get_region_for_object(child)) {
		LOG_ERROR("[%s] does not belong to %s\n", child->name, region->name);
		rc = EINVAL;
		goto out;
	}
	
	LOG_DEBUG("region: %s, child:%s, new child:%s\n",
		  region->name, child->name, new_child ? new_child->name : "<unknown>");

	
	if (new_child) {
		/*
		 * The Engine passed in new child object,
		 * make sure it is a data object and
		 * make sure that the sizes are the same
		 */
		if (new_child->data_type != DATA_TYPE) {
			rc = EINVAL;
			goto out;
		}

		child_size = md_object_usable_size(child, &vol->sb_ver, vol->chunksize);
		new_child_size = md_object_usable_size(new_child, &vol->sb_ver, vol->chunksize);
		if (child_size > new_child_size) {
			rc = EINVAL;
			goto out;
		}
	}
out:
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * md_replace_child
 *
 * Replace the current child with new child object.
 */
int md_replace_child(storage_object_t *region,
		     storage_object_t *child,
		     storage_object_t *new_child)
{
	int rc = 0;
	md_volume_t *vol;
	u_int64_t child_size, new_child_size;
	md_member_t *member;

	LOG_ENTRY();

	vol = (md_volume_t *)region->private_data;
	if (!vol) {
		rc = EFAULT;
		goto out;
	}
	member = md_volume_find_object(vol, child);
	if (!member) {
		LOG_ERROR("%s is not in MD %s region.\n", child->name, vol->name);
		rc = EINVAL;
		goto out;
	}
	
	child_size = md_object_usable_size(child, &vol->sb_ver, vol->chunksize);
	new_child_size = md_object_usable_size(new_child, &vol->sb_ver, vol->chunksize);

	LOG_DEBUG("region: %s, CHILD: %s, NEW CHILD: %s\n", 
		  region->name, child->name, new_child->name);
	LOG_DEBUG(" CHILD object size=%"PRIu64", MD size=%"PRIu64""
		  " and NEW CHILD: object size=%"PRIu64", MD size=%"PRIu64"\n",
		  child->size, child_size, new_child->size, new_child_size);

	if (child_size > new_child_size) {
		LOG_ERROR("Child size=%"PRIu64" (MD size=%"PRIu64")"
			  " is not equal to new child size=%"PRIu64" (MD size=%"PRIu64")\n",
			  child->size, child_size, new_child->size, new_child_size);
		rc = EINVAL;
		goto out;
	}

	md_remove_region_from_object(region, child);
	md_append_region_to_object(region, new_child);
	vol->sb_func->zero_superblock(member, FALSE);
	vol->sb_func->replace_disk(member, new_child);

	if (md_is_region_active(region)) {
		region->flags |= SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE;
	}
	region->flags |= SOFLAG_DIRTY;
	vol->flags |= MD_DIRTY;
out:
	LOG_EXIT_INT(rc);
	return rc;
}

int md_get_kernel_info(storage_object_t * region, mdu_array_info_t* md_info)
{
	int rc=0;
	u_int64_t size = 0;
	md_volume_t *vol = (md_volume_t *)region->private_data;

	LOG_ENTRY();

	rc = md_ioctl_get_array_info(region, md_info);
	if (rc) {
		region->flags &= ~SOFLAG_ACTIVE;
	} else {
		rc = md_ioctl_get_blk_size(region, &size);
		if (!rc && size != 0) {
			if (vol) {
				vol->flags |= MD_ACTIVE;
			}
			region->flags |= SOFLAG_ACTIVE;
		} else {
			if (vol) {
				vol->flags &= ~MD_ACTIVE;
			}
			region->flags &= ~SOFLAG_ACTIVE;
			rc = EINVAL;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}


int md_can_activate_region(storage_object_t * region)
{
	LOG_ENTRY();

	if (region->flags & SOFLAG_CORRUPT) {
		LOG_WARNING("Region %s is corrupt.  It can not be activated.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}


int md_activate_region(storage_object_t * region)
{
	int rc=0;
	md_volume_t  *vol = (md_volume_t *)region->private_data;
	md_member_t *member;
	list_element_t iter;
	mdu_array_info_t md_info;

	LOG_ENTRY();

	if (region->flags & SOFLAG_CORRUPT) {
		LOG_WARNING("Region %s is corrupt.  It can not be activated.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (region->flags & SOFLAG_ACTIVE) {
		LOG_DEFAULT("%s is currently active, deactivating...\n", region->name);
		rc = md_deactivate_region(region);
		if (rc) {
			LOG_EXIT_INT(rc);
			return rc;
		}
	}

	if (load_kernel_md_driver() == FALSE) {
		LOG_WARNING("KERNEL MD driver failed to load.\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	if (vol->sb_ver.major_version == MD_SB_VER_0) {
		rc = md_ioctl_set_array_info(region, NULL);
	} else if (vol->sb_ver.major_version == MD_SB_VER_1) {
		memset(&md_info, 0, sizeof(mdu_array_info_t));
		md_info.major_version = 1;
		rc = md_ioctl_set_array_info(region, &md_info);
	} else {
		LOG_MD_BUG();
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (!rc) {
		LIST_FOR_EACH(vol->members, iter, member) {
			if (member->obj &&
			    !(member->flags & MD_MEMBER_NEW) &&
			    !(member->flags & MD_MEMBER_DISK_PENDING) &&
			    !(member->flags & MD_MEMBER_STALE) &&
			    !(member->flags & MD_MEMBER_DISK_FAULTY)) {
				mdu_disk_info_t disk_info;
				memset(&disk_info, 0, sizeof(disk_info));
				disk_info.major = member->obj->dev_major;
				disk_info.minor = member->obj->dev_minor;
				rc = md_ioctl_add_new_disk(region, &disk_info);
				if (rc) {
					break;
				}
			}
		}
		if (!rc) {
			rc = md_ioctl_run_array(region);
		}
	}

	if (!rc) {
		// activate succeeded, turn on active flag by calling md_get_kernel_info
		rc = md_get_kernel_info(region, &md_info);
	}

	if (!rc) {
		region->flags &= ~SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;

}

int md_can_deactivate_region(storage_object_t * region)
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

int md_deactivate_region(storage_object_t * region)
{
	int rc=0;
	md_volume_t *volume;

	LOG_ENTRY();
	if (!region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	/* If the region is not active, just clear flags and return success */
	if (!md_is_region_active(region)) {
		region->flags &= ~(SOFLAG_ACTIVE | SOFLAG_NEEDS_DEACTIVATE);
		LOG_EXIT_INT(rc);
		return rc;
	}

	rc = md_ioctl_stop_array(region);
	if (!rc) {
		if (md_is_region_active(region)) {
			rc = EBUSY;
			LOG_MD_BUG();
		} else {
			/*
			 * Reload all superblocks ???
			 * Just log some information
			 */
			volume = (md_volume_t *)region->private_data;
			if (volume) {
				LOG_DETAILS("Region %s has been deactivated."
					    "  MD volume is still present.\n",
					    region->name);
			} else {
				LOG_DETAILS("Region %s has been deactivated."
					    "  MD volume was deleted.\n",
					    region->name);
			}
		}
	}
	
	if (!rc || (rc == ENODEV)) {
		region->flags &= ~(SOFLAG_ACTIVE | SOFLAG_NEEDS_DEACTIVATE);
		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;

}

boolean md_is_region_active(storage_object_t * region)
{
	int rc = 1;
	mdu_array_info_t info;

	if ((region->plugin == linear_plugin) ||
	    (region->plugin == raid0_plugin)) {
		rc = EngFncs->dm_update_status(region);
	}

	if (rc) {
		md_get_kernel_info(region, &info);
	}

	return (region->flags & SOFLAG_ACTIVE) ? TRUE : FALSE;
}


static boolean mdstat_check_recovery(int mddev_minor)
{
	FILE *fp;
	char string[256];
	char mddev_str[10];
	boolean recovery_running = FALSE;

	LOG_ENTRY();

	sprintf(mddev_str, "md%d :", mddev_minor);

	fp = mdstat_open();
	if (fp) {
		while (fgets(string, 256, fp) != NULL) {
			if (!strncmp(string, mddev_str, strlen(mddev_str))) {

				if (fgets(string, 256, fp) == NULL)
					break;
				if (fgets(string, 256, fp) == NULL)
					break;
				if (strstr(string, "recovery"))
					recovery_running = TRUE;
				else if (strstr(string, "resync"))
						recovery_running = TRUE;
				
				/* found mdx, break out of while() loop */
				break;
			}
		}
		fclose(fp);
	} else {
		LOG_WARNING("Could not open /proc/mdstat for reading.\n");
	}
	LOG_EXIT_INT(recovery_running);
	return recovery_running;
}

boolean md_is_recovery_running(storage_object_t * region)
{
	int rc;
	boolean recovery_running = FALSE;
	mdu_array_info_t md_info;
	md_volume_t *vol;

	LOG_ENTRY();

	rc = md_ioctl_get_array_info(region, &md_info);
	if (rc) {
		LOG_EXIT_INT(FALSE);
		return FALSE;
	}

	recovery_running = (md_info.state & (1<<MD_ARRAY_RECOVERY_RUNNING)) ? TRUE : FALSE;
	if (recovery_running == FALSE) {
		/*
		 * Just in case the kernel does not provide the information
		 * about resync/recovery, try scanning /proc/mdstat
		 */
		vol = (md_volume_t *)region->private_data;
		recovery_running = mdstat_check_recovery(vol->md_minor);
	}

	LOG_EXIT_INT(recovery_running);
	return recovery_running;
}

int md_fix_dev_major_minor(md_volume_t *vol, boolean do_msg)
{
	storage_object_t *obj;
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();

	if (!vol->sb) {
		LOG_MD_BUG();
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	if (!vol->region) {
		LOG_MD_BUG();
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (vol->flags & MD_CORRUPT) {
		LOG_WARNING("MD region %s is corrupt.\n", vol->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	if (vol->sb_ver.major_version == MD_SB_VER_1) {
		LOG_DEBUG("Version %d superblock does not store device major/minor.\n",
			  vol->sb_ver.major_version);
		LOG_EXIT_INT(0);
		return 0;
	}
	
	if (md_is_region_active(vol->region)) {
		LOG_WARNING("MD region %s is active, can not change device major/minor.\n",
			    vol->name);
		LOG_EXIT_INT(0);
		return 0;		
	}

	LIST_FOR_EACH(vol->members, iter, member) {
		int major, minor;
		obj = member->obj;
		if (obj->flags & SOFLAG_ACTIVE) {
			major = md_member_get_disk_major(member);
			minor = md_member_get_disk_minor(member);
			if (( major != obj->dev_major) ||
			    ( minor != obj->dev_minor)) {
				MESSAGE(_("Region %s: The MD superblock has old device "
					  "major/minor number for object (%s). "
					  "When this MD region is activated,"
					  "  the old device [%d:%d] will be replaced with new device [%d:%d].\n"),
					vol->region->name, obj->name,
					major, minor, obj->dev_major, obj->dev_minor );
				vol->flags |= MD_DIRTY;
				vol->region->flags |= SOFLAG_DIRTY;
			}
		}
	}

	LOG_EXIT_INT(0);
	return 0;
}

boolean follow_up_mark_faulty(md_volume_t *volume, storage_object_t *faulty)
{
	int    answer = 0;
	char * choice_text[3] = { _("Don't remove faulty"), _("Remove faulty"), NULL };
	QUESTION(&answer, choice_text,
		_("The object [%s] has been marked faulty.  "
		  "Would you also like to remove [%s] from %s region?\n\n"
		  "If you elect not to remove [%s] at this time, "
		  "you should consider removing it later via the region's option menu.\n"),
		faulty->name, faulty->name, volume->name, faulty->name);
	
	return answer ? TRUE : FALSE;
}

/*
 * Function:  md_can_stop_array
 *
 * Note : MD driver will fail MD_STOP ioctl if the "open" count is not 0.
 */
boolean md_can_stop_array( storage_object_t *region )
{
	boolean rc = TRUE;
	md_volume_t *vol = region->private_data;

	LOG_ENTRY();

	if ((vol->personality == LINEAR) || (vol->personality == RAID0)) {
		LOG_EXIT_BOOL(TRUE);
		return TRUE;
	}
	
	if (md_is_region_active(region)) {
		if (region->volume) {
			if (!(region->volume->flags & VOLFLAG_COMPATIBILITY)) {
				LOG_DETAILS("Region %s is part of EVMS volume %s.\n",
					    region->name, region->volume->name);
				rc = FALSE;
			}
			if (EngFncs->is_mounted(region->volume->name, NULL)) {
				LOG_DETAILS("Region %s is part of volume %s which is mounted on %s.\n",
					    region->name, region->volume->name, region->volume->mount_point);
				rc = FALSE;
			}
		}
	
		if (EngFncs->list_count(region->parent_objects)) {
			LOG_DETAILS("Region %s has parent(s).\n", region->name);
			rc = FALSE;
		}
	}

	LOG_EXIT_BOOL(rc);
	return rc;
}

/*
 * find_disk_in_active_region
 *
 * Query the kernel MD driver for a disk which has the specified major and minor.
 * Return -1 if not found.
 */
int find_disk_in_active_region(storage_object_t *region, int major, int minor)
{
	int index;
	int rc;
	int limit;
	md_volume_t *vol;
	mdu_disk_info_t info;

	vol = (md_volume_t *)region->private_data;
	if (!vol || !vol->sb_func) {
		LOG_MD_BUG();
		return -1;
	}

	limit = MAX_DISKS(vol);
	for (index=0; index < limit; index++) {
		info.number = index;
		rc = md_ioctl_get_disk_info(region, &info);
		if (!rc && (info.major == major) && (info.minor == minor))
			break;
	}
	if (index ==limit) {
		LOG_WARNING("Could not find disk[%d:%d] in MD region %s.\n",
			major, minor, region->name);
		index = -1;
	}
	return index;
}

/*
 * md_find_member
 *
 * This function performs an extensive search for a member whose object
 * major an minor matche the input parameters.
 */
md_member_t *md_find_member(int major, int minor)
{
	md_volume_t *vol;
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();
	for (vol = volume_list_head; vol; vol = vol->next) {
		LIST_FOR_EACH(vol->members, iter, member) {
			if ((member->obj) &&
			    (member->obj->dev_major = major) && 
			    (member->obj->dev_minor == minor)) {
				LOG_EXIT_PTR(member);
				return member;
			}
		}
	}
	LOG_EXIT_PTR(NULL);
	return NULL;
}

/*
 * md_volume_find_object
 *
 * If the object is found, return the corresponding member
 * Otherwise, return NULL.
 */
md_member_t * md_volume_find_object(md_volume_t *vol, storage_object_t *obj)
{
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj == obj) {
			LOG_EXIT_PTR(member);
			return member;
		}
	}

	LOG_EXIT_PTR(NULL);
	return NULL;
}

md_member_t * md_volume_find_member(md_volume_t *vol, int dev_number)
{
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->dev_number == dev_number) {
			LOG_EXIT_PTR(member);
			return member;
		}
	}

	LOG_EXIT_PTR(NULL);
	return NULL;
}

/*
 * md_volume_is_member
 *
 * If the member is found, return TRUE
 * Otherwise, return FALSE
 */
boolean md_volume_is_member(md_volume_t *volume, md_member_t *member)
{
	LOG_ENTRY();
	if (EngFncs->find_in_list(volume->members, member, NULL, NULL)) {
		LOG_EXIT_BOOL(TRUE);
		return TRUE;
	}

	LOG_EXIT_BOOL(FALSE);
	return FALSE;
}

int md_volume_count_children(md_volume_t *vol)
{
	int count=0;
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (!(member->flags & MD_MEMBER_STALE)) {
			count++;
		}
	}

	LOG_EXIT_INT(count);
	return count;
}

int md_volume_count_active_disks(md_volume_t *vol)
{
	int count=0;
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (!(member->flags & MD_MEMBER_STALE) &&
		    !(member->flags & MD_MEMBER_DISK_FAULTY) &&
		    (member->flags & MD_MEMBER_DISK_ACTIVE)) {
			count++;
		}
	}

	LOG_EXIT_INT(count);
	return count;
}

int md_volume_count_faulty_disks(md_volume_t *vol)
{
	int count=0;
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (!(member->flags & MD_MEMBER_STALE) &&
		    (member->flags & MD_MEMBER_DISK_FAULTY)) {
			count++;
		}
	}

	LOG_EXIT_INT(count);
	return count;
}

int md_volume_count_spare_disks(md_volume_t *vol)
{
	int count=0;
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (!(member->flags & MD_MEMBER_STALE) &&
		    (member->flags & MD_MEMBER_DISK_SPARE)) {
			count++;
		}
	}

	LOG_EXIT_INT(count);
	return count;
}

int md_volume_count_stale_disks(md_volume_t *vol)
{
	int count=0;
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->flags & MD_MEMBER_STALE) {
			count++;
		}
	}
	LOG_EXIT_INT(count);
	return count;
}

u_int64_t md_volume_smallest_data_size(md_volume_t *vol)
{
	u_int64_t smallest = -1;
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		smallest = min(smallest, member->data_size);
	}
	LOG_EXIT_U64(smallest);
	return smallest;
}

md_member_t *md_allocate_member(storage_object_t *obj)
{
	md_member_t *new_member;

	LOG_ENTRY();
	new_member = EngFncs->engine_alloc(sizeof(md_member_t));
	if (new_member) {
		new_member->obj = obj;
		new_member->raid_disk = -1;
		new_member->dev_number = -1;
		new_member->data_size = -1;
		new_member->data_offset = -1;
		new_member->super_offset = -1;
	}
	LOG_EXIT_PTR(new_member);
	return new_member;
}

void md_free_member(md_member_t *member)
{
	LOG_ENTRY();
	if (!member) {
		return;
	}

	if (member->sb) {
		EngFncs->engine_free(member->sb);
		member->sb = NULL;
	}

	if (member->saved_info) {
		EngFncs->engine_free(member->saved_info);
		member->saved_info = NULL;
	}
	member->obj = NULL;
	member->vol = NULL;
	EngFncs->engine_free(member);
	LOG_EXIT_VOID();
}


md_volume_t * md_allocate_volume(void)
{
	md_volume_t *new_vol = NULL;

	LOG_ENTRY();

	new_vol = EngFncs->engine_alloc(sizeof(md_volume_t));

	if (!new_vol) {
		LOG_EXIT_PTR(NULL);
		return NULL;
	}

	new_vol->md_minor = -1;
	new_vol->personality = INVALID_LEVEL;
	new_vol->members = EngFncs->allocate_list();
	new_vol->setup_funcs = EngFncs->allocate_list();
	new_vol->ioctl_pkgs = EngFncs->allocate_list();
	new_vol->ioctl_cleanup = EngFncs->allocate_list();

	md_add_volume_to_list(new_vol);

	LOG_EXIT_PTR(new_vol);
	return new_vol;
}

void md_free_volume(md_volume_t *vol)
{
	list_element_t iter;
	md_member_t *member;

	LOG_ENTRY();

	if (!vol) {
		LOG_MD_BUG();
		LOG_EXIT_VOID();
		return;
	}
	
	LIST_FOR_EACH(vol->members, iter, member) {
		md_free_member(member);
	}
	EngFncs->destroy_list(vol->members);
	vol->members = NULL;

	if (vol->setup_funcs != NULL) {
		empty_setup_funcs_queue(vol);
		EngFncs->destroy_list(vol->setup_funcs);
		vol->setup_funcs = NULL;
	}

	if (vol->ioctl_pkgs != NULL) {
		empty_ioctl_queue(vol);
		EngFncs->destroy_list(vol->ioctl_pkgs);
		vol->ioctl_pkgs = NULL;
	}
	
	if (vol->ioctl_cleanup != NULL) {
		free_ioctl_pkgs(vol);
		EngFncs->destroy_list(vol->ioctl_cleanup);
		vol->ioctl_cleanup = NULL;
	}

	if (vol->sb) {
		EngFncs->engine_free(vol->sb);
		vol->sb = NULL;
		vol->sb_func = NULL;
	}

	md_remove_volume_from_list(vol);
	EngFncs->engine_free(vol);
	LOG_EXIT_VOID();
}

int md_delete_volume(md_volume_t * vol, boolean tear_down)
{
	int rc = 0;
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();
	if (tear_down == TRUE) {
		LIST_FOR_EACH(vol->members, iter, member) {
			md_zero_superblock(member, FALSE);
			md_zero_saved_info(member, FALSE);
		}
	}

	if (vol->personality != MULTIPATH) {
		md_free_volume(vol);
	} else {
		/* For multipath regions, only delete the volume if the object
		 * is not active. If it is active, we will need this
		 * private-data in order to correctly deactivate the region.
		 */
		if (!(vol->region->flags & SOFLAG_ACTIVE)) {
			md_free_volume(vol);
		} else {
			vol->flags |= MD_MP_DELETE_VOLUME;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


md_member_t * md_clone_member( md_member_t *org)
{
	md_member_t *new_member = NULL;
	int rc=0;

	LOG_ENTRY();
	new_member = md_allocate_member(org->obj);
	if (new_member) {
		*new_member = *org;
		new_member->sb = NULL;
		new_member->saved_info = NULL;
		if (org->vol && org->vol->sb_func) {
			org->vol->sb_func->duplicate_sb(&new_member->sb, org->sb);
			if (!new_member->sb) {
				rc = ENOMEM;
				goto out;
			}
			if (org->saved_info) {
				new_member->saved_info = EngFncs->engine_alloc(sizeof(*(org->saved_info)));
				if (new_member->saved_info) {
					*(new_member->saved_info) = *(org->saved_info);
				} else {
					rc = ENOMEM;
					goto out;
				}
			}
		}
	}
out:
	if (rc) {
		md_free_member(new_member);
		new_member = NULL;
	}
	LOG_EXIT_PTR(new_member);
	return new_member;
}

/*
 * md_clone_volume
 *
 * Duplicate most of the fields in the original volume.
 * Allocate and copy the superblock.
 * Create a new list and copy the members.
 */
md_volume_t * md_clone_volume( md_volume_t *org)
{
	md_volume_t *new_vol;
	md_member_t *member;
	md_member_t *new_member;
	list_element_t iter;
	int rc=0;

	LOG_ENTRY();
	new_vol = md_allocate_volume();
	if (new_vol) {
		*new_vol = *org;
		new_vol->nr_disks = 0;
		new_vol->members = EngFncs->allocate_list();
		new_vol->setup_funcs = EngFncs->allocate_list();
		new_vol->ioctl_pkgs = EngFncs->allocate_list();
		new_vol->ioctl_cleanup = EngFncs->allocate_list();
		new_vol->sb = NULL;
		/* duplicate superblock */
		if (org->sb) {
			rc = new_vol->sb_func->duplicate_sb(&new_vol->sb, org->sb);
			if (rc) {
				goto out;
			}
		}
		LIST_FOR_EACH(org->members, iter, member) {
			new_member = md_clone_member(member);
			if (new_member) {
				md_volume_add_member(new_vol, new_member);
			} else {
				rc = ENOMEM;
				goto out;
			}
		}
		if (rc) {
			goto out;
		}
	}
out:
	if (rc) {
		md_free_volume(new_vol);
		new_vol = NULL;
	}
	LOG_EXIT_PTR(new_vol);
	return new_vol;
}

int md_volume_sort_members(void *thing1, void *thing2, void *user_data)
{
	md_member_t *m1 = (md_member_t *)thing1;
	md_member_t *m2 = (md_member_t *)thing2;
	
	if (m1->dev_number < m2->dev_number) return -1;
	if (m1->dev_number > m2->dev_number) return 1;
	return 0;
}

/*
 * md_volume_add_member
 *
 * This function should be called during discovery of existing MD arrays.
 * For newly created MD region, use md_volume_add_new_raid_member() or
 * md_volume_add_new_spare_member() instead.
 */
void md_volume_add_member(md_volume_t *vol, md_member_t *member)
{
	list_element_t e;

	LOG_ENTRY();
	e = EngFncs->insert_thing(vol->members, member, INSERT_AFTER, NULL);
	if (e) {
		EngFncs->sort_list(vol->members, md_volume_sort_members, NULL);
		vol->nr_disks++;
		member->vol = vol;
	} else {
		LOG_CRITICAL("Can't insert %s into MD volume %s.\n",
			     member->obj->name, vol->name);
	}
	LOG_EXIT_VOID();
}

boolean md_volume_is_minor_taken(int md_minor)
{
	md_volume_t *vol = volume_list_head;

	while (vol) {
		if (vol->md_minor == md_minor) {
			return TRUE;
		}
		vol = vol->next;
	}
	return FALSE;
}


storage_object_t * md_find_valid_input_object( char * name )
{
	int rc = 0;
	storage_object_t * object = NULL;
	storage_object_t *obj;
	list_anchor_t objects;
	list_element_t li;

	LOG_ENTRY();
	if (!name) {
		LOG_EXIT_PTR(NULL);
		return NULL;
	}

	// get the list of objects
	rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,
				 DATA_TYPE,
				 NULL,
				 NULL,
				 VALID_INPUT_OBJECT,
				 &objects);
	if (!rc) {

		LIST_FOR_EACH(objects, li, obj) {
			if (!strncmp(obj->name, name, EVMS_NAME_SIZE)) {
				object = obj;
				break;
			}
		}
		EngFncs->destroy_list(objects);
	} else {
		LOG_ERROR("Error getting object list = %d....\n",rc);
	}
	LOG_EXIT_PTR(object);
	return object;
}

/*
 * md_volume_get_available_name
 *
 * Try to set volume name and minor.  The limit is determined by version of
 * MD superblock.  The limit is 256 for old superblock format.
 */
int md_volume_get_available_name(md_volume_t *vol, int limit)
{
	int md_minor;
	storage_object_t *region;
	char region_name[EVMS_NAME_SIZE+1];
	mdu_array_info_t md_info;
	int rc = 0;

	LOG_ENTRY();
	for (md_minor=0; md_minor < limit; md_minor++) {
		if (md_volume_is_minor_taken(md_minor) == FALSE) {
			sprintf(region_name, "md/md%d", md_minor);
			rc = EngFncs->allocate_region(region_name, &region);
			if (rc == 0) {
				/*
				 * The name is available on the user space side.
				 * Query the kernel MD driver to make sure.
				 * If the md minor is not in use, we should get an error.
				 */
				region->dev_major = MD_MAJOR;
				region->dev_minor = md_minor;
				rc = md_ioctl_get_array_info(region, &md_info);
				EngFncs->free_region(region);
				if (rc) {
					/* Good, the name is available */
					break;
				}
			}
		}
	}
	if (md_minor < limit) {
		rc = 0;
		strcpy(vol->name, region_name);
		vol->md_minor = md_minor;
	} else {
		rc = ENODEV;
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * md_volume_get_alternative_name
 *
 * Try to set volume name and minor.  The limit is determined by version of
 * MD superblock.  The limit is 256 for old superblock format.
 */
int md_volume_get_alternative_name(md_volume_t *vol, int limit)
{
	int md_minor;
	storage_object_t *region;
	char region_name[EVMS_NAME_SIZE+1];
	mdu_array_info_t md_info;
	int rc = 0;
	boolean found = FALSE;
	list_element_t iter;
	md_member_t *member;
	char *disk_group_name = NULL;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj->disk_group) {
			disk_group_name = member->obj->disk_group->name;
			break;
		}
	}
	

	for (md_minor=limit; md_minor >= 0; md_minor--) {
		if (md_volume_is_minor_taken(md_minor) == FALSE) {
			if (disk_group_name) {
				sprintf(region_name, "%s/md/md%d", disk_group_name, md_minor);
			} else {
				sprintf(region_name, "md/md%d", md_minor);
			}
			rc = EngFncs->allocate_region(region_name, &region);
			if (rc == 0) {
				/*
				 * The name is available on the user space side.
				 * Query the kernel MD driver to make sure.
				 * If the md minor is not in use, we should get an error.
				 */
				region->dev_major = MD_MAJOR;
				region->dev_minor = md_minor;
				rc = md_ioctl_get_array_info(region, &md_info);
				EngFncs->free_region(region);
				if (rc) {
					/* Good, the name is available */
					found = TRUE;
					break;
				}
			}
		}
	}
	if (found == TRUE) {
		rc = 0;
		strcpy(vol->name, region_name);
		vol->md_minor = md_minor;
	} else {
		rc = ENODEV;
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * md_volume_remove_member
 *
 * Remove the member from the MD volume.  This function affectively
 * changes the MD array configuration.
 *
 * Note:  This does not remove the storage object from the MD region.
 */
int md_volume_remove_member(md_member_t *member, boolean resize)
{
	md_super_info_t info;
	md_volume_t *vol;
	int rc = 0;

	LOG_ENTRY();
	if (!member->vol) {
		LOG_MD_BUG();
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = member->vol;

	LOG_DEBUG("Removing %s from MD volume %s.\n",
		  member->obj->name, vol->name);

	rc = vol->sb_func->remove_disk(member, resize);
	if (!rc) {
		EngFncs->remove_thing(vol->members, member);
		md_volume_get_super_info(vol, &info);
		vol->raid_disks = info.raid_disks;
		vol->active_disks = info.active_disks;
		vol->spare_disks = info.spare_disks;
		vol->working_disks = info.working_disks;
		vol->failed_disks = info.failed_disks;
		vol->nr_disks--;
		member->vol = NULL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_volume_mark_faulty(md_member_t *member, boolean mark_removed)
{
	md_super_info_t info;
	md_volume_t *vol;
	int rc = 0;

	LOG_ENTRY();
	if (!member->vol) {
		LOG_MD_BUG();
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = member->vol;

	LOG_DEBUG("MD region %s: marking faulty (%s).\n",
		  vol->name, member->obj->name);

	rc = vol->sb_func->mark_disk_faulty(member, mark_removed);
	if (!rc) {
		md_volume_get_super_info(vol, &info);
		vol->raid_disks = info.raid_disks;
		vol->active_disks = info.active_disks;
		vol->spare_disks = info.spare_disks;
		vol->working_disks = info.working_disks;
		vol->failed_disks = info.failed_disks;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_volume_activate_spare(md_member_t *member)
{
	md_super_info_t info;
	md_volume_t *vol;
	int rc = 0;

	LOG_ENTRY();

	if (!member->vol) {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}

	vol = member->vol;
	if (!(member->flags & MD_MEMBER_DISK_SPARE)) {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("MD region %s: activating spare (%s).\n",
		  vol->name, member->obj->name);

	rc = vol->sb_func->activate_spare(member);
	if (!rc) {
		md_volume_get_super_info(vol, &info);
		vol->raid_disks = info.raid_disks;
		vol->active_disks = info.active_disks;
		vol->spare_disks = info.spare_disks;
		vol->working_disks = info.working_disks;
		vol->failed_disks = info.failed_disks;
	}
out:
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * This common callback handler will free the memory allocated for disk_info
 */
static int free_disk_info_post_ioctl(md_volume_t *vol, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	if (pkg->parm.disk_info) {
		EngFncs->engine_free(pkg->parm.disk_info);
		pkg->parm.disk_info = NULL;
	}
	LOG_EXIT_INT(0);
	return 0;
}


/*
 * md_sb1_add_spare_setup
 *
 * This function should be called during SETUP phase of commit.
 * - Retrieve the original spare disk
 * - Check for cancellation of the operation
 * - Mark MD volume DIRTY to update superblocks.
 * - Schedule the IOCTL (EVMS_MD_ADD)
 */
static int md_sb1_add_spare_setup(md_volume_t *vol, md_setup_func_t *setup)
{
	int rc = 0;
	evms_md_ioctl_parm_t parm;
	evms_md_disk_info_t *disk_info;
	md_member_t *member;

	LOG_ENTRY();

	disk_info = setup->disk_info;

	if (setup->proceed == FALSE) {
		LOG_WARNING("Add spare operation has been cancelled for object %s.\n",
			    disk_info->object->name);
		rc = 0;
		goto out;
	}

	member = md_volume_find_object(vol, disk_info->object);
	if (!member) {
		LOG_ERROR("Could not find the original spare object %s.\n",
			  (disk_info->object) ? disk_info->object->name : "???");
		rc = EINVAL;
		goto out;
	}
	
	vol->flags |= MD_DIRTY;
	vol->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;

	parm.disk_info = disk_info;
	rc = schedule_md_ioctl_pkg(vol, EVMS_MD_ADD, &parm, free_disk_info_post_ioctl);
	if (rc) {
		vol->flags &= ~MD_DIRTY;
		EngFncs->engine_free(setup->disk_info);
	}

out:
	
	LOG_EXIT_INT(rc);
	return rc;
}

int md_volume_add_spare_to_active_region(md_volume_t * vol, storage_object_t * spare)
{
	int rc=0;
	evms_md_ioctl_parm_t parm;
	evms_md_disk_info_t *disk_info = NULL;
	md_member_t *member = NULL;

	LOG_ENTRY();

	member = md_allocate_member(spare);
	if (!member) {
		rc = ENOMEM;
		goto out;
	}

	if (vol->flags & MD_DEGRADED) {
		member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_SPARE | MD_MEMBER_DISK_PENDING);
	} else {
		member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_SPARE);
	}

	rc = md_volume_add_new_member(vol, member);
	if (rc) {
		goto out;
	}

	disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
	if ( !disk_info ) {
		rc = ENOMEM;
		goto out;
	}

	disk_info->number = member->dev_number;
	disk_info->object = spare;
	parm.disk_info = disk_info;

	switch (vol->sb_ver.major_version) {
	case MD_SB_VER_0:
		rc = schedule_md_ioctl_pkg(vol, EVMS_MD_HOT_ADD, &parm, free_disk_info_post_ioctl);
		break;
	case MD_SB_VER_1:
		rc = schedule_setup_func(vol, disk_info, md_sb1_add_spare_setup);
		break;
	default:
		LOG_MD_BUG();
		rc = EINVAL;
		break;
	}
	if (rc) {
		goto out;
	}

	md_append_region_to_object(vol->region, spare);
	
	LOG_EXIT_INT(0);
	return 0;

out:
	if (rc) {
		if (disk_info) {
			EngFncs->engine_free(disk_info);
		}
		if (member) {
			md_volume_remove_member(member, FALSE);
			md_free_member(member);
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int md_volume_add_spare_to_inactive_region(md_volume_t * vol, storage_object_t *spare)
{
	int rc=0;
	md_member_t *member = NULL;

	LOG_ENTRY();

	if (vol->flags & MD_CORRUPT) {
		rc = EINVAL;
		goto out;
	}

	if (vol->flags & MD_DEGRADED) {
		/*
		 * We are about to add a spare to a degraded array.
		 * We first schedule an activation, then use
		 * add_spare_to_active_region() to hot add the spare later.
		 */
		vol->region->flags |= SOFLAG_NEEDS_ACTIVATE;
		rc = md_volume_add_spare_to_active_region(vol, spare);
	} else {
		member = md_allocate_member(spare);
		if (!member) {
			rc = ENOMEM;
			goto out;
		}

		member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_SPARE | MD_MEMBER_DISK_PENDING);
		rc = md_volume_add_new_member(vol, member);
		if (rc) {
			goto out;
		}

		vol->flags |= MD_DIRTY;
		md_append_region_to_object(vol->region, spare);

		LOG_EXIT_INT(0);
		return 0;
	}


out:
	if (rc && member) {
		md_volume_remove_member(member, FALSE);
		md_free_member(member);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int md_volume_remove_spare_from_active_region(md_volume_t *vol, storage_object_t * spare)
{
	int rc=0;
	evms_md_disk_info_t *disk_info;
	evms_md_ioctl_parm_t parm;
	mdu_disk_info_t info;
	md_member_t *member;
	
	LOG_ENTRY();

	member = md_volume_find_object(vol, spare);
	if (!member) {
		LOG_WARNING("%s is not found in region %s.\n",
			    spare->name, vol->name);
		rc = EINVAL;
		goto out;
	}
	if (!(member->flags & MD_MEMBER_DISK_SPARE)) {
		LOG_WARNING("%s is not a spare.\n",
			    spare->name);
		rc = EINVAL;
		goto out;
	}

	/*
	 * IOCTL the kernel
	 * do not update superblocks (ie. do not set DIRTY flag)
	 */
	info.number = member->dev_number;
	rc = md_ioctl_get_disk_info(vol->region, &info);
	if (rc) {
		LOG_ERROR("(%s) does not exist\n", spare->name);
		rc = ENODEV;
		goto out;
	} else {
		if ((info.major != spare->dev_major) || (info.minor != spare->dev_minor)) {

			MESSAGE(_("WARNING: Region:%s, Device:%s: Index:%d\n"
				" There is a mismatch major/minor, Kernel MD driver has (%d:%d), EVMS has (%d:%d)."
				"  However, if %s was created by another MD tool such as mdadm or raidtools,"
				" the operation will succeed.\n"),
			  vol->name, spare->name, member->dev_number,
			  info.major, info.minor,
			  spare->dev_major, spare->dev_minor,
			  vol->name);
		}
	}
	
	disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
	if ( !disk_info ) {
		rc = ENOMEM;
		goto out;
	}

	disk_info->number = member->dev_number;
	disk_info->major = info.major;
	disk_info->minor = info.minor;
	disk_info->object = spare;
	parm.disk_info = disk_info;
	schedule_md_ioctl_pkg(vol, EVMS_MD_HOT_REMOVE, &parm, free_disk_info_post_ioctl);

	rc = md_volume_remove_member(member, FALSE);
	if (!rc) {
		md_remove_region_from_object(vol->region, spare);
		vol->sb_func->zero_superblock(member, FALSE);

		md_free_member(member);
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

int md_volume_remove_spare_from_inactive_region(md_volume_t *vol, storage_object_t *spare)
{
	int rc=0;
	md_member_t *member;
	
	LOG_ENTRY();
	if (!vol || !spare) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	member = md_volume_find_object(vol, spare);
	if (!member) {
		LOG_WARNING("%s is not found in region %s.\n",
			    spare->name, vol->name);
		rc = EINVAL;
		goto out;
	}
	if (!(member->flags & MD_MEMBER_DISK_SPARE)) {
		LOG_WARNING("%s is not a spare.\n",
			    spare->name);
		rc = EINVAL;
		goto out;
	}

	rc = md_volume_remove_member(member, FALSE);
	if (!rc) {
		md_remove_region_from_object(vol->region, spare);
		vol->sb_func->zero_superblock(member, FALSE);
		md_free_member(member);
		vol->flags |= MD_DIRTY;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * md_volume_remove_faulty_from_active_region
 *
 *	- schedule an HOT_REMOVE_DISK ioctl (will be issued during commit)
 *	- remove faulty disk from region's children list
 *	- set kill sectors to wipe out the MD superblock on the faulty disk
 *	- mark the faulty disk removed
 */
int md_volume_remove_faulty_from_active_region(md_volume_t * vol, storage_object_t * faulty)
{
	int rc = 0;
	evms_md_disk_info_t *disk_info;
	evms_md_ioctl_parm_t parm;
	mdu_disk_info_t info;
	md_member_t *member;
	
	LOG_ENTRY();
	
	member = md_volume_find_object(vol, faulty);
	if (!member) {
		LOG_WARNING("%s is not in region %s.\n", faulty->name, vol->name);
		rc = EINVAL;
		goto out;
	}
	if (!(member->flags & MD_MEMBER_DISK_FAULTY)) {
		LOG_WARNING("%s is not faulty.\n", faulty->name);
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Faulty disk (%s) is found at index:%d.\n",
		  faulty->name, member->dev_number);
	/*
	 * Removing a faulty disk which is still in the array
	 */
	int kernel_idx;
	kernel_idx = find_disk_in_active_region(vol->region,
						faulty->dev_major,
						faulty->dev_minor);
	if (kernel_idx != -1) {
		/* Faulty disk is still in the active region */
		if (kernel_idx != member->dev_number) {
			LOG_WARNING("The superblock has %s at index=%d,"
				" whereas the kernel said index=%d.\n",
				faulty->name, member->dev_number, kernel_idx);
		}
		info.number = kernel_idx;
		rc = md_ioctl_get_disk_info(vol->region, &info);
		if (rc) {
			goto out;
		}

		if ((info.major != faulty->dev_major) || (info.minor != faulty->dev_minor)) {
			MESSAGE(_("WARNING: Region:%s, Device:%s: Index:%d\n"
				" There is a mismatch major/minor, Kernel MD driver has (%d:%d),"
				" EVMS has (%d:%d).  However, if %s was created by another MD tool"
				" such as mdadm or raidtools,"
				" the operation will succeed.\n"),
				vol->name, faulty->name, kernel_idx,
				info.major, info.minor,
				faulty->dev_major, faulty->dev_minor,
				vol->name);
		}

		disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));

		if (!disk_info ) {
			rc = ENOMEM;
			goto out;
		}

		disk_info->number = kernel_idx;
		disk_info->major = info.major;
		disk_info->minor = info.minor;
		disk_info->object = faulty;
		parm.disk_info = disk_info;
		schedule_md_ioctl_pkg(vol, EVMS_MD_HOT_REMOVE, &parm, free_disk_info_post_ioctl);
	} else {
		goto out_remove_faulty;
	}
		
out_remove_faulty:
	rc = md_volume_remove_member(member, FALSE);
	if (!rc) {
		md_remove_region_from_object(vol->region, faulty);
		vol->sb_func->zero_superblock(member, FALSE);
		md_free_member(member);
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

int md_volume_remove_faulty_from_inactive_region(md_volume_t *vol, storage_object_t *faulty)
{
	int rc=0;
	md_member_t *member;

	member = md_volume_find_object(vol, faulty);
	if (!member) {
		LOG_WARNING("%s is not in region %s.\n", faulty->name, vol->name);
		rc = EINVAL;
		goto out;
	}
	if (!(member->flags & MD_MEMBER_DISK_FAULTY)) {
		LOG_WARNING("%s is not faulty.\n", faulty->name);
		rc = EINVAL;
		goto out;
	}
	rc = md_volume_remove_member(member, FALSE);
	if (!rc) {
		md_remove_region_from_object(vol->region, faulty);
		vol->sb_func->zero_superblock(member, FALSE);
		md_free_member(member);
		vol->flags |= MD_DIRTY;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;

}

int md_volume_remove_stale_from_active_region(md_volume_t *vol, storage_object_t * stale_disk)
{
	int rc=0;
	evms_md_disk_info_t *disk_info;
	evms_md_ioctl_parm_t parm;
	mdu_disk_info_t info;
	md_member_t *member;
	int kernel_idx;

	LOG_ENTRY();

	member = md_volume_find_object(vol, stale_disk);
	if (!member) {
		LOG_WARNING("%s is not in region %s.\n", stale_disk->name, vol->name);
		rc = EINVAL;
		goto out;
	}
	if (!(member->flags & MD_MEMBER_STALE)) {
		LOG_WARNING("%s is not stale.\n", stale_disk->name);
		rc = EINVAL;
		goto out;
	}


	/*
	 * Removing a stale disk which is still in the array.
	 * Make sure to get the index from the kernel MD driver.
	 */
	kernel_idx = find_disk_in_active_region(vol->region,
						stale_disk->dev_major,
						stale_disk->dev_minor);
	if (kernel_idx != -1) {
		/* Stale disk is still in the active region */
		if (kernel_idx != member->dev_number) {
			LOG_WARNING("The superblock has %s at index=%d,"
				" whereas the kernel said index=%d.\n",
				stale_disk->name, member->dev_number, kernel_idx);
		}
		info.number = kernel_idx;
		rc = md_ioctl_get_disk_info(vol->region, &info);
		if (!rc && (info.major == stale_disk->dev_major) &&
		    (info.minor == stale_disk->dev_minor)) {
			if (!(info.state & (1<<MD_DISK_REMOVED))) {
				disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
				if (!disk_info ) {
					rc = ENOMEM;
					goto out;
				}
				disk_info->number = kernel_idx;
				disk_info->major = stale_disk->dev_major;
				disk_info->minor = stale_disk->dev_minor;
				disk_info->object = stale_disk;
				parm.disk_info = disk_info;
				schedule_md_ioctl_pkg(vol,
						      EVMS_MD_HOT_REMOVE,
						      &parm,
						      free_disk_info_post_ioctl);
			} else {
				LOG_WARNING("%s has already been removed.\n",
					    stale_disk->name);
				goto out_remove_stale;
			}
		}
	} else {
		LOG_WARNING("%s is not in the active region %s.\n",
			    stale_disk->name, vol->name);
		goto out_remove_stale;
	}

out_remove_stale:
	rc = md_volume_remove_member(member, FALSE);
	if (!rc) {
		md_remove_region_from_object(vol->region, stale_disk);
		vol->sb_func->zero_superblock(member, FALSE);
		md_free_member(member);
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

int md_volume_remove_stale_from_inactive_region(md_volume_t *vol, storage_object_t * stale_disk)
{
	int rc=0;
	md_member_t *member;

	LOG_ENTRY();

	member = md_volume_find_object(vol, stale_disk);
	if (!member) {
		LOG_WARNING("%s is not in region %s.\n", stale_disk->name, vol->name);
		rc = EINVAL;
		goto out;
	}
	if (!(member->flags & MD_MEMBER_STALE)) {
		LOG_WARNING("%s is not stale.\n", stale_disk->name);
		rc = EINVAL;
		goto out;
	}

	rc = md_volume_remove_member(member, FALSE);
	if (!rc) {
		md_remove_region_from_object(vol->region, stale_disk);
		vol->sb_func->zero_superblock(member, FALSE);
		md_free_member(member);
		vol->flags |= MD_DIRTY;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

