# (C) Copyright IBM Corp. 2004
#
# This program is free software;  you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY;  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program;  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Common subroutines for calling the EVMS CLI and other command-line tools.

package Evms::Common;

use strict;
use warnings;

use Evms::Log;

BEGIN {
	use Exporter();
	our (@ISA, @EXPORT);
        @ISA = qw(Exporter);
        @EXPORT = qw(&run_evms_command
		     &run_evms_commands
		     &evms_activate

		     &get_extended_info_value
		     &get_blkdev_sectors
		     &get_blkdev_major_minor

		     &delete_thing
		     &clean_object
		     &clean_objects

		     &size_to_sectors
		     &sectors_to_size
		     &compare_sectors
		     &compare_sizes
		     &add_sizes

		     &write_sequence
		     &validate_sequence
		    );
}

# run_evms_command
# Use the EVMS CLI to run a command. Optionally return the screen output.
#
# Arguments:
#    command: String containing a single command to run.
#    output: Reference of array to pass back command output (optional).
sub run_evms_command($; $)
{
	my $command = $_[0];
	my $rc_output = $_[1];  # Optional.
	my @output;
	my $rc;

	log_debug("Running command: $command\n");

	@output = `echo "$command" | evms -s 2>&1`;
	$rc = $? >> 8;

	if (ref($rc_output)) {
		@{$rc_output} = @output;
	}

	return $rc;
}

# run_evms_commands
# Use the EVMS CLI to run an array of commands. Optionally return the screen output.
#
# Arguments:
#    commands: Reference to an array of single command strings.
#    output: Reference of array to pass back command output (optional).
sub run_evms_commands($; $)
{
	my @commands = @{$_[0]};
	my $rc_output = $_[1];  # Optional.
	my $command = "";
	my @output;
	my $rc;
	my $i;

	for ($i = 0; $i < $#commands; $i++) {
		log_debug("Command: $commands[$i]\n");
		$command .= "$commands[$i]:";
	}
	log_debug("Command: $commands[$#commands]\n");
	$command .= $commands[$#commands];

	log_debug("Command string is:\n$command\n");
	@output = `echo "$command" | evms -c -s 2>&1`;
	$rc = $? >> 8;

	if (ref($rc_output)) {
		@{$rc_output} = @output;
	}

	return $rc;
}

sub evms_activate()
{
	`evms_activate 2>&1`;
	return ($? >> 8);
}

# get_extended_info_value
# Return the value associated with a specified field from the output from
# an extended-info query.
#
# Arguments:
#    field: Name of field to get value for.
#    extended_info: Reference of array containing extended-info output.
sub get_extended_info_value($ $)
{
	my $field = $_[0];
	my $extended_info = $_[1];
	my $value;
	my ($i, $j);

	for ($i = 0; $i < @{$extended_info}; $i++) {
		if (${$extended_info}[$i] =~ /Field Name:\s+$field/) {
			for ($j = $i+1; $j < @{$extended_info}; $j++) {
				if (${$extended_info}[$j] =~
				    /The value of this field is:\s+(.+)$/) {
					return $1;
				}
			}
		}
	}

	return;
}

# get_blkdev_sectors
# Get the number of sectors in the specified device.
#
# Arguments:
#    name: Name of object or volume to query. For objects, this routine will
#          prepend "/dev/evms/.nodes/" to form a full device name.
# Returns:
#    Number of sectors (numerical scalar)
sub get_blkdev_sectors($)
{
	my $name = $_[0];
	my $device;
	my $sectors;

	if ($name !~ /^\//) {
		$device = "/dev/evms/.nodes/" . $name;
	} else {
		$device = $name;
	}

	$sectors = `blockdev --getsize $device 2>&1`;
	if ($? >> 8) {
		return 0;
	}

	chomp($sectors);
	return $sectors;
}

# get_blkdev_major_minor
# Return an array of the (major,minor) for the specified device. Values
# returned are in decimal, not hexidecimal.
sub get_blkdev_major_minor($)
{
	my $name = $_[0];
	my $device;
	my $major;
	my $minor;

	if ($name !~ /^\//) {
		$device = "/dev/evms/.nodes/" . $name;
	} else {
		$device = $name;
	}

	$major = `stat --format=%t $device`;
	$minor = `stat --format=%T $device`;
	chomp($major);
	chomp($minor);
	$major = hex($major);
	$minor = hex($minor);

	return ($major, $minor);
}

# delete_thing
#    Delete an object, volume, or container.
#
# Arguments:
#    name: Name of thing to delete.
sub delete_thing($)
{
	my $name = $_[0];
	my $command = "delete:$name";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# clean_object
# Remove all parent volumes, objects and containers from the specified object.
sub clean_object($)
{
	my $object = $_[0];
	`evms_clean $object 2>&1`;
	return ($? >> 8);
}

# clean_objects
# Remove all parent volumes, objects and containers from the specified
# list of objects.
sub clean_objects(@)
{
	my @objects = @_;
	my $object;
	my $object_list;

	foreach $object (@objects) {
		$object_list .= "$object ";
	}

	`evms_clean $object_list 2>&1`;
	return ($? >> 8);
}

# size_to_sectors
# Convert a "size" to corresponding number of sectors. "Size" should include
# a suffix (e.g. KB, MB, GB).
sub size_to_sectors($)
{
	my $size = $_[0];
	my ($value, $suffix);
	my $sectors = 0;

	if ($size =~ /(\d+\.*\d*)\s*([kKmMgG])[bB]/) {
		$value = $1;
		$suffix = $2;

		if ($suffix eq "G" || $suffix eq "g") {
			$sectors = $value * 1024 * 1024 * 2;
		} elsif ($suffix eq "M" || $suffix eq "m") {
			$sectors = $value * 1024 * 2;
		} else {
			$sectors = $value * 2;
		}
	}

	return $sectors;
}

# sectors_to_size
# Convert number of sectors to a "size" with the appropriate suffix (e.g. KB,
# MB, GB).
sub sectors_to_size($)
{
	my $sectors = $_[0];
	my ($gb, $mb, $kb, $bytes);

	$gb = $sectors / (2 * 1024 * 1024);
	if ($gb > 1) {
		return $gb . "GB";
	}

	$mb = $sectors / (2 * 1024);
	if ($mb > 1) {
		return $mb . "MB";
	}

	$kb = $sectors / 2;
	if ($kb > 1) {
		return $kb . "KB";
	}

	$bytes = $sectors * 512;
	return $bytes . "B";
}

# compare_sectors
# Is sectors2 within the specified percentage of sectors1?
# Returns:
#    >0 if sectors1 is larger than sectors2
#    <0 if sectors1 is smaller than sectors2
#    0  if sectors1 equals sectors2
sub compare_sectors($ $ $)
{
	my $sectors1 = $_[0];
	my $sectors2 = $_[1];
	my $tolerance = $_[2];
	my ($sectors1_high, $sectors1_low);
	my $rc = 1;

	if ($tolerance < 0 || $tolerance > 100) {
		return $rc;
	}

	$sectors1_high = $sectors1 * ((100 + $tolerance) / 100);
	$sectors1_low = $sectors1 * ((100 - $tolerance) / 100);

	if ($sectors2 < $sectors1_low) {
		$rc = 1;
	} elsif ($sectors2 > $sectors1_high) {
		$rc = -1;
	} else {
		$rc = 0;
	}

	return $rc;
}

# compare_sizes
# Is size2 within the specified percentage of size1? Same return value
# as compare_sectors.
sub compare_sizes($ $ $)
{
	my $size1 = $_[0];
	my $size2 = $_[1];
	my $tolerance = $_[2];
	my $sectors1 = size_to_sectors($size1);
	my $sectors2 = size_to_sectors($size2);
	my $rc;

	$rc = compare_sectors($sectors1, $sectors2, $tolerance);

	return $rc;
}

# add_sizes
# Add two "size" values together. Each value must have an appropriate size
# suffix, and the returned value will have the appropriate size suffix.
sub add_sizes($ $)
{
	my $size1 = $_[0];
	my $size2 = $_[1];
	my $total_sectors;
	my $total_size;

	$total_sectors = size_to_sectors($size1) + size_to_sectors($size2);
	$total_size = sectors_to_size($total_sectors);

	return $total_size;
}

# write_sequence
# Use "seq_dd" to write a sequence of numbers to the specified volume.
#
# Arguments
#    volume: Name of volume to write to (full path required).
#    count: Number of sectors to write to.
#    start: Sector to start writing at (optional, default = 0)
#    index: Value to start the sequence at (optional, default = 0)
sub write_sequence($ $; $ $)
{
	my $volume = $_[0];
	my $count = $_[1];
	my $start = $_[2];
	my $index = $_[3];
	my $rc;

	if (!defined($count)) {
		return 1;
	}

	$start = defined($start) ? "start=$start" : "";
	$index = defined($index) ? "index=$index" : "";

	log_debug("seq_dd of=$volume count=$count $start $index 2>&1");
	`seq_dd of=$volume count=$count $start $index 2>&1`;
	$rc = $? >> 8;

	`sync`;

	return $rc;
}

# validate_sequence
# Use "seq_dd" to verify a sequence of numbers on the specified volume.
#
# Arguments
#    volume: Name of volume to read from (full path required).
#    count: Number of sectors to read.
#    start: Sector to start reading at (optional, default = 0)
#    index: Value to start the sequence at (optional, default = 0)
sub validate_sequence($ $; $ $)
{
	my $volume = $_[0];
	my $count = $_[1];
	my $start = $_[2];
	my $index = $_[3];
	my $rc;

	if (!defined($count)) {
		return 1;
	}

	$start = defined($start) ? "start=$start" : "";
	$index = defined($index) ? "index=$index" : "";

	`seq_dd if=$volume count=$count $start $index 2>&1`;
	$rc = $? >> 8;

	return $rc;
}

1;

