/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: Multipath Plugin
 * File: evms2/engine/plugins/multipath/mp_daemon.c
 *
 * Helper routines for starting, stopping, and getting information about the
 * evms_mpathd daemon.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <glob.h>
#include <sys/wait.h>
#include <plugin.h>
#include "multipath.h"

#define LOCK_FILE_PREFIX	"/var/lock/evms-mpathd-"
#define SLASH_REPLACEMENT	'|'

#define DEBUG_DAEMON		1

static void remove_slashes(char *string)
{
	for (; *string; string++)
		if (*string == '/') *string = SLASH_REPLACEMENT;
}

static void get_lock_file_name(storage_object_t *object,
			       char *lock_file_name)
{
	char segment_name[EVMS_NAME_SIZE];

	LOG_ENTRY();

	strncpy(segment_name, object->name, EVMS_NAME_SIZE);
	remove_slashes(segment_name);
	snprintf(lock_file_name, 256, "%s%s", LOCK_FILE_PREFIX, segment_name);

	LOG_EXIT_VOID();
}

/**
 * stop_daemon
 *
 * Stop the evms_mpathd instance that's monitoring this segment.
 **/
int stop_daemon(storage_object_t *object)
{
	multipath_t *mp = object->private_data;
	char lock_file_name[256];
	int rc = 0;

	LOG_ENTRY();

	if (mp->daemon_pid > 0) {
		LOG_DEBUG("Sending SIGTERM to process %d\n", mp->daemon_pid);
		rc = kill(mp->daemon_pid, SIGTERM);
		if (!rc) {
			LOG_DEBUG("Waiting for process %d to terminate.\n",
				  mp->daemon_pid);
			mp->daemon_pid = 0;

			get_lock_file_name(object, lock_file_name);
			unlink(lock_file_name);
		} else {
			rc = errno;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * start_daemon
 *
 * Start an instance of evms_mpathd to monitor this segment.
 **/
int start_daemon(storage_object_t *object)
{
	multipath_t *mp = object->private_data;
	storage_object_t *child;
	list_element_t itr;
	char **argv = NULL;
	char object_size[20];
	char command[256];
	pid_t pid;
	int status;
	int argc = 0, sz = 0;
	int i = 0, rc = 0;

	LOG_ENTRY();

	if (mp->daemon_pid > 0) {
		goto out;
	}

	argc = 5 + EngFncs->list_count(object->child_objects);
	argv = EngFncs->engine_alloc(argc * sizeof(char *));
	if (!argv) {
		rc = ENOMEM;
		goto out;
	}

	snprintf(object_size, 20, "%"PRIu64, object->size);
	argv[i++] = "evms_mpathd";
#if DEBUG_DAEMON
	argv[i++] = "-d";
#endif
	argv[i++] = object->name;
	argv[i++] = object_size;
	
	LIST_FOR_EACH(object->child_objects, itr, child) {
		argv[i++] = child->name;
	}

	for (i = 0; i < argc; i++)
		if (argv[i])
			sz += snprintf(command + sz, 256 - sz, "%s ", argv[i]);
	LOG_DEBUG("Starting daemon process: %s\n", command);

	pid = EngFncs->fork_and_execvp(NULL, argv, NULL, NULL, NULL);
	if (pid < 0) {
		rc = errno;
		goto out;
	}

	LOG_DEBUG("Started daemon as process %d\n", pid);
	waitpid(pid, &status, WNOHANG);
	mp->daemon_pid = pid;

out:
	EngFncs->engine_free(argv);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * check_daemon
 *
 * Check to see if an instance of evms_mpathd is monitoring this segment. We
 * do this by trying to lock the appropriate lock-file. If we get the lock,
 * the daemon is not running. If we can't get the lock, then we can get the
 * pid of the daemon that has the lock. We can then use this pid to stop the
 * daemon if necessary.
 **/
int check_daemon(storage_object_t *object)
{
	multipath_t *mp = object->private_data;
	struct flock lockinfo;
	char lock_file_name[256];
	int lock_fd, rc;

	LOG_ENTRY();

	get_lock_file_name(object, lock_file_name);

	lock_fd = open(lock_file_name, O_RDWR | O_CREAT, 0660);
	if (lock_fd < 0) {
		rc = errno;
		goto out;
	}

	lockinfo.l_whence = SEEK_SET;
	lockinfo.l_start = 0;
	lockinfo.l_len = 0;
	lockinfo.l_type = F_WRLCK;

	LOG_DEBUG("Attempting to lock file %s\n", lock_file_name);
	rc = fcntl(lock_fd, F_SETLK, &lockinfo);
	if (rc) {
		/* File is locked. Get the lock-holder's pid. */
		rc = fcntl(lock_fd, F_GETLK, &lockinfo);
		if (rc) {
			rc = errno;
			goto out;
		}

		if (lockinfo.l_type != F_UNLCK) {
			LOG_DEBUG("File %s is locked by process %d\n",
				  lock_file_name, lockinfo.l_pid);
			mp->daemon_pid = lockinfo.l_pid;
		}
	} else {
		/* File was not locked. Unlock it so the daemon can start. */
		LOG_DEBUG("File %s is not locked. Need to start daemon.\n",
			  lock_file_name);
		lockinfo.l_type = F_UNLCK;
		rc = fcntl(lock_fd, F_SETLK, &lockinfo);
		if (rc) {
			rc = errno;
			goto out;
		}

		object->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

out:
	if (lock_fd > 0) {
		close(lock_fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

static int globerror(const char *path, int error)
{
	return 0;
}

static int glob_lock_files(glob_t *result)
{
	char pattern[128];
	int i, rc, flags = 0;

	LOG_ENTRY();

	for (i = 0; i < MULTIPATH_TYPE_MAX; i++) {
		snprintf(pattern, 128, "%smp|%s*",
			 LOCK_FILE_PREFIX, mp_modules[i].name);

		rc = glob(pattern, flags, globerror, result);
		if (rc) {
			if (!flags)
				globfree(result);
			break;
		}

		flags |= GLOB_APPEND;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * filter_discovered_segments
 *
 * Search the list of evms_mpathd lock files, and filter out any files for
 * which we discovered a multipath segment. All remaining files represent
 * stale daemons.
 */
static void filter_discovered_segments(glob_t *result)
{
	storage_object_t *object;
	list_anchor_t objects;
	list_element_t itr;
	char lock_file_name[256];
	int i, rc;

	LOG_ENTRY();

	rc = EngFncs->get_object_list(SEGMENT, DATA_TYPE,
				      &multipath_plugin, NULL, 0, &objects);
	if (rc) {
		goto out;
	}

	LIST_FOR_EACH(objects, itr, object) {
		get_lock_file_name(object, lock_file_name);

		for (i = 0; i < result->gl_pathc; i++) {
			rc = strncmp(lock_file_name, result->gl_pathv[i], 256);
			if (!rc) {
				*(result->gl_pathv[i]) = '\0';
				break;
			}
		}
	}

out:
	LOG_EXIT_VOID();
}

/*
 * cleanup_stale_daemon
 *
 * Open and attempt to lock the specified lock file. If the file is locked,
 * get the PID of that daemon instance and kill the daemon.
 */
static void cleanup_stale_daemon(char *lock_file_name)
{
	struct flock lockinfo;
	int lock_fd, rc;

	LOG_ENTRY();

	lock_fd = open(lock_file_name, O_RDWR | O_CREAT, 0660);
	if (lock_fd < 0) {
		goto out;
	}

	lockinfo.l_whence = SEEK_SET;
	lockinfo.l_start = 0;
	lockinfo.l_len = 0;
	lockinfo.l_type = F_WRLCK;

	LOG_DEBUG("Attempting to lock file %s\n", lock_file_name);
	rc = fcntl(lock_fd, F_SETLK, &lockinfo);
	if (rc) {
		/* File is locked. Get the lock-holder's pid. */
		rc = fcntl(lock_fd, F_GETLK, &lockinfo);
		if (rc) {
			close(lock_fd);
			goto out;
		}

		if (lockinfo.l_type != F_UNLCK) {
			LOG_DEBUG("File %s is locked by process %d\n",
				  lock_file_name, lockinfo.l_pid);

			kill(lockinfo.l_pid, SIGTERM);
		}
	} else {
		/* File was not locked. */
		LOG_DEBUG("File %s is not locked.\n", lock_file_name);
		lockinfo.l_type = F_UNLCK;
		fcntl(lock_fd, F_SETLK, &lockinfo);
	}

	close(lock_fd);
	unlink(lock_file_name);

out:
	LOG_EXIT_VOID();
}

/**
 * cleanup_stale_daemons
 **/
void cleanup_stale_daemons(void)
{
	glob_t result;
	int i;

	LOG_ENTRY();

	glob_lock_files(&result);

	filter_discovered_segments(&result);

	/* Cleanup all daemons that don't have a corresponding segment. */
	for (i = 0; i < result.gl_pathc; i++) {
		if (*(result.gl_pathv[i]) != '\0') {
			cleanup_stale_daemon(result.gl_pathv[i]);
		}
	}

	globfree(&result);

	LOG_EXIT_VOID();
}

