/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvmregmgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the LVM region manager. This is also where all LVM global
 *              data is declared.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"

/* Global variables. */
engine_functions_t		* EngFncs;		/* The Engine's internal API set. */
plugin_record_t			LVM_Plugin;		/* Pre-declared here. Initialized at the end. */
plugin_record_t			* my_plugin_record;	/* Used to identify containers and regions as owned by LVM. */
list_anchor_t			lvm_group_list;		/* List of volume groups. */

/**
 * lvm_setup_evms_plugin
 *
 * This function gets called shortly after the plugin is loaded by the
 * Engine. It performs all tasks that are necessary before the initial
 * discovery pass.
 **/
static int lvm_setup_evms_plugin(engine_functions_t * functions)
{
	int rc = 0;

	/* Initialize global variables. */
	EngFncs			= functions;
	my_plugin_record	= &LVM_Plugin;

	LOG_ENTRY();

	/* Allocate the global group list. */
	lvm_group_list = EngFncs->allocate_list();
	if (!lvm_group_list) {
		LOG_CRITICAL("Error creating LVM container list\n");
		rc = ENOMEM;
	} else {
		/* Register the LVM namespace so no one tries to use it. */
		EngFncs->register_name("lvm");
		EngFncs->register_name(EVMS_DEV_NODE_PATH "lvm");
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_cleanup_evms_plugin
 *
 * Clean up all volume groups from memory before closing the engine.
 **/
static void lvm_cleanup_evms_plugin(void)
{
	lvm_volume_group_t * group;
	list_element_t itr1, itr2;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(lvm_group_list, itr1, itr2, group) {
		//lvm_deallocate_volume_group(group);
	}

	EngFncs->destroy_list(lvm_group_list);

	LOG_EXIT_VOID();
}


/****** Region Checking Functions ******/


/* All of the following lvm_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/**
 * lvm_can_delete
 *
 * Can we remove the specified LVM logical volume, and consolidate the
 * space with the freespace volume?
 **/
static int lvm_can_delete(storage_object_t * region)
{
	int rc = 0;

	LOG_ENTRY();

	/* The region must be a regular (non-freespace) volume. */
	if (region->data_type != DATA_TYPE) {
		LOG_DEFAULT("Cannot delete freespace region %s.\n",
			    region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_expand
 *
 * Can this region be expanded, using freespace from its group? If so,
 * add the region to the expansion_points output list.
 **/
static int lvm_can_expand(storage_object_t * region,
			  u_int64_t expand_limit,
			  list_anchor_t expansion_points)
{
	lvm_logical_volume_t * volume = region->private_data;
	lvm_volume_group_t * group = volume->group;
	expand_object_info_t * expand_point;
	int rc = 0;

	LOG_ENTRY();

	rc = lvm_can_expand_volume(volume);
	if (rc) {
		goto out;
	}

	/* Make sure the expand_limit is at least as big as the PE-size. */
	if (expand_limit < group->vg->pe_size) {
		rc = ENOSPC;
		goto out;
	}

	/* Create a new expansion info object. */
	expand_point = EngFncs->engine_alloc(sizeof(expand_object_info_t));
	if (!expand_point) {
		LOG_CRITICAL("Memory error creating expansion info object.\n");
		rc = ENOMEM;
		goto out;
	}

	expand_point->object = region;
	expand_point->max_expand_size = group->freespace->region->size;
	if (volume->lv->lv_stripes > 1) {
		expand_point->max_expand_size -= expand_point->max_expand_size %
						 volume->lv->lv_stripes;
	}

	if (expand_limit < expand_point->max_expand_size) {
		expand_point->max_expand_size = round_down(expand_limit,
							   group->vg->pe_size *
							   volume->lv->lv_stripes);
	}

	/* Add to the expansion points list. */
	EngFncs->insert_thing(expansion_points, expand_point,
			      INSERT_AFTER, NULL);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_shrink
 *
 * Can this region be shrunk, and consolidate the space with this group's
 * freespace? If so, add this region to the shrink_points list.
 **/
static int lvm_can_shrink(storage_object_t * region,
			  u_int64_t shrink_limit,
			  list_anchor_t shrink_points)
{
	lvm_logical_volume_t * volume = region->private_data;
	lvm_volume_group_t * group = volume->group;
	shrink_object_info_t * shrink_point;
	int rc;

	LOG_ENTRY();

	rc = lvm_can_shrink_volume(volume);
	if (rc) {
		goto out;
	}

	/* Make sure the shrink_limit is at least as big as the PE-size. */
	if (shrink_limit < group->vg->pe_size) {
		rc = ENOSPC;
		goto out;
	}

	/* Create a new shrink info object. */
	shrink_point = EngFncs->engine_alloc(sizeof(shrink_object_info_t));
	if (!shrink_point) {
		LOG_CRITICAL("Memory error creating shrink info object.\n");
		rc = ENOMEM;
		goto out;
	}

	shrink_point->object = region;
	shrink_point->max_shrink_size = (volume->lv->lv_allocated_le - 1) *
					group->vg->pe_size;
	if (volume->lv->lv_stripes > 1) {
		shrink_point->max_shrink_size -= shrink_point->max_shrink_size %
						 volume->lv->lv_stripes;

	}

	if (shrink_limit < shrink_point->max_shrink_size) {
		shrink_point->max_shrink_size = round_down(shrink_limit,
							   group->vg->pe_size *
							   volume->lv->lv_stripes);
	}

	/* Add to the shrink points list. */
	EngFncs->insert_thing(shrink_points, shrink_point, INSERT_AFTER, NULL);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_set_volume
 *
 * Any non-freespace LVM region can be converted into an EVMS
 * or a compatibility volume.
 **/
static int lvm_can_set_volume(storage_object_t * region,
			      boolean flag)
{
	int rc = 0;

	LOG_ENTRY();

	/* Can't make freespace into an EVMS volume. */
	if (region->data_type != DATA_TYPE) {
		LOG_DEFAULT("Cannot make a volume from freespace region %s\n",
			    region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_discover
 *
 * @segments:	Input list of storage objects
 * @regions:	Output list of storage objects
 * @final_call:	Is this the last discovery iteration?
 *
 * Examine all disk segments and find LVM PVs. Assemble volume groups
 * and export all LVM logical volumes as EVMS regions.
 *
 * All newly created regions must be added to the output list, and all
 * segments from the input list must either be claimed or moved to the
 * output list.
 **/
static int lvm_discover(list_anchor_t segments,
			list_anchor_t regions,
			boolean	final_call)
{
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Beginning LVM Discovery\n");

	lvm_discover_volume_groups(segments, regions);

	lvm_check_volume_groups(final_call);

	lvm_discover_logical_volumes(final_call);

	rc = lvm_export_logical_volumes(regions, final_call);

	/* No cleanup in the engine. All metadata structures are
	 * kept around for the life of the engine session.
	 */

	LOG_DEBUG("LVM Discovery complete.\n");
	LOG_EXIT_INT(rc);
	return rc;
}


/****** Region Functions ******/


/**
 * lvm_create
 *
 * Create a new LVM region using the specified freespace.
 **/
static int lvm_create(list_anchor_t freespace_region_list,
		      option_array_t * options,
		      list_anchor_t new_region_list)
{
	lvm_logical_volume_t * freespace_volume, * new_volume;
	lvm_volume_group_t * group;
	lv_disk_t * lv;
	lvm_lv_create_options_t lv_opts;
	int rc;

	LOG_ENTRY();

	/* Get the input freespace volume and group. */
	rc = lvm_get_freespace_volume(freespace_region_list, &freespace_volume);
	if (rc) {
		LOG_ERROR("Error getting freespace region from list\n");
		goto out;
	}

	group = freespace_volume->group;

	/* Get the input options and verify. */
	rc = lvm_create_region_parse_option_array(options, group, &lv_opts);
	if (rc) {
		LOG_ERROR("Error parsing and verifying creation options\n");
		goto out;
	}

	/* Make sure there is enough space for the desired allocation scheme. */
	rc = lvm_check_available_extents(group, &lv_opts);
	if (rc) {
		MESSAGE(_("Can not create new region in container %s using specified options\n"),
			group->container->name);
		goto out;
	}

	/* Initialize the appropriate LV entry. */
	rc = lvm_initialize_new_lv(&lv_opts, group, &lv);
	if (rc) {
		LOG_ERROR("Error initializing the LV metadata\n");
		goto out;
	}

	/* Allocate a new logical volume. */
	new_volume = lvm_allocate_logical_volume(lv, group);
	if (!new_volume) {
		LOG_ERROR("Error creating new region %s\n", lv_opts.lv_name);
		lvm_clear_lv(lv);
		rc = ENOMEM;
		goto out;
	}

	/* Set up the LE-to-PE mappings for the new volume. */
	rc = lvm_allocate_extents_to_volume(new_volume, &lv_opts);
	if (rc) {
		LOG_ERROR("Unable to allocate PEs to the new region %s\n",
			  lv_opts.lv_name);
		lvm_clear_lv(lv);
		lvm_deallocate_logical_volume(new_volume);
		goto out;
	}

	/* Update the group information. */
	group->volume_list[new_volume->number] = new_volume;
	group->volume_count++;
	group->vg->lv_cur++;
	group->vg->pe_allocated += new_volume->lv->lv_allocated_le;

	/* Update the freespace and rebuild the LE maps. */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n",
			  group->container->name);
		goto out;
	}

	group->container->flags |= SCFLAG_DIRTY;
	EngFncs->insert_thing(new_region_list, new_volume->region,
			      INSERT_AFTER, NULL);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_discard
 *
 * Forget about these regions. Similar to delete, but we're not changing any
 * metadata. Simply deallocate the internal data structures associated with
 * the regions. The Engine will call to deactivate these regions during commit.
 **/
static int lvm_discard(list_anchor_t regions)
{
	storage_object_t * region;
	lvm_logical_volume_t * volume;
	list_element_t itr;

	LOG_ENTRY();

	LIST_FOR_EACH(regions, itr, region) {
		if (region->data_type != DATA_TYPE) {
			LOG_ERROR("Cannot discard freespace region %s.\n",
				  region->name);
			continue;
		}

		volume = region->private_data;

		lvm_discard_extents_from_volume(volume);
		lvm_deallocate_logical_volume(volume);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_delete
 *
 * Remove the specified region and consolidate all of its space into
 * the appropriate freespace region.
 **/
static int lvm_delete(storage_object_t * region,
		      list_anchor_t children)
{
	lvm_logical_volume_t * volume;
	lvm_volume_group_t * group;
	int rc = 0;

	LOG_ENTRY();

	/* Check that this region can be removed. */
	rc = lvm_can_delete(region);
	if (rc) {
		LOG_ERROR("Not allowed to delete region %s.\n", region->name);
		goto out;
	}

	volume = region->private_data;
	group = volume->group;

	/* Remove all the extents from the volume. */
	lvm_deallocate_extents_from_volume(volume);

	/* Remove the parent/child associations with the PVs. */
	lvm_clear_child_list(region);

	/* Update the group information. */
	group->vg->lv_cur--;
	group->vg->pe_allocated -= volume->lv->lv_allocated_le;

	/* Delete the volume. */
	lvm_clear_lv(volume->lv);
	lvm_deallocate_logical_volume(volume);

	/* Update the freespace and rebuild the LE maps. */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n",
			  group->container->name);
	}

	group->container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_expand
 *
 * Expand an LVM region using the freespace in that region's container.
 **/
static int lvm_expand(storage_object_t * region,
		      storage_object_t * expand_object,
		      list_anchor_t input_objects,
		      option_array_t * options)
{
	lvm_volume_group_t * group;
	lvm_logical_volume_t * volume;
	lvm_lv_expand_options_t lv_opts;
	sector_count_t expand_delta;
	int rc = 0;

	LOG_ENTRY();

	/* If region and expand_object are not the same, someone is
	 * trying to expand an object below LVM. Not allowed.
	 */
	if (region != expand_object) {
		LOG_ERROR("Cannot expand object %s below LVM region %s.\n",
			  expand_object->name, region->name);
		rc = EINVAL;
		goto out;
	}

	volume = region->private_data;
	group = volume->group;

	rc = lvm_can_expand_volume(volume);
	if (rc) {
		LOG_ERROR("Region %s cannot be expanded at this time.\n",
			  region->name);
		goto out;
	}

	rc = lvm_expand_region_parse_option_array(options, group,
						  volume, &lv_opts);
	if (rc) {
		LOG_ERROR("Error parsing and verifying expand options\n");
		goto out;
	}

	/* Call the engine's can_expand_by() to make sure all
	 * plugins and FSIMS above LVM approve of the expand.
	 */
	expand_delta = lv_opts.add_size;
	rc = EngFncs->can_expand_by(region, &expand_delta);
	if (rc) {
		LOG_ERROR("A parent of LVM has disallowed the expand of region %s\n",
			  region->name);
		goto out;
	}

	/* Make sure there is enough space for the expand
	 * based on the volume's allocation scheme.
	 */
	rc = lvm_check_available_expand_extents(volume, &lv_opts);
	if (rc) {
		MESSAGE(_("Can not expand region %s using specified options\n"),
			region->name);
		goto out;
	}

	/* Set up the expanded LE-to-PE mappings for the volume. */
	rc = lvm_allocate_expand_extents_to_volume(volume, &lv_opts);
	if (rc) {
		LOG_ERROR("Unable to allocate enough PEs to expand region %s\n",
			  region->name);
		goto out;
	}

	/* Update the volume information. */
	rc = lvm_update_expanded_volume(volume, &lv_opts);
	if (rc) {
		LOG_ERROR("Error updating expanded region %s\n", region->name);
		goto out;
	}

	/* Update the group information. */
	group->vg->pe_allocated += lv_opts.add_extents;

	/* Update the freespace and rebuild the LE maps. */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n",
			  group->container->name);
	}

	/* Mark container dirty. */
	group->container->flags |= SCFLAG_DIRTY;
	if (region->flags & SOFLAG_ACTIVE) {
		region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_shrink
 *
 * A lot like lvm_expand, but making regions smaller instead of bigger. :)
 **/
static int lvm_shrink(storage_object_t * region,
		      storage_object_t * shrink_object,
		      list_anchor_t input_objects,
		      option_array_t * options)
{
	lvm_volume_group_t * group;
	lvm_logical_volume_t * volume;
	u_int32_t remove_extents;
	sector_count_t shrink_delta;
	int rc = 0;

	LOG_ENTRY();

	/* If region and shrink_object are not the same, someone
	 * is trying to shrink an object below LVM. Not allowed.
	 */
	if (region != shrink_object) {
		LOG_ERROR("Cannot shrink object %s below LVM region %s.\n",
			  shrink_object->name, region->name);
		rc = EINVAL;
		goto out;
	}

	volume = region->private_data;
	group = volume->group;

	rc = lvm_can_shrink_volume(volume);
	if (rc) {
		LOG_ERROR("Region %s cannot be shrunk at this time.\n",
			  region->name);
		goto out;
	}

	/* Parse the options for the shrink command. */
	rc = lvm_shrink_region_parse_option_array(options, group,
						  volume, &remove_extents);
	if (rc) {
		LOG_ERROR("Error parsing and verifying shrink options\n");
		goto out;
	}

	/* Call the engine's can_shrink_by() to make sure all plugins
	 * and FSIMS above LVM approve of the shrink.
	 */
	shrink_delta = remove_extents * group->vg->pe_size;
	rc = EngFncs->can_shrink_by(region, &shrink_delta);
	if (rc) {
		LOG_ERROR("A parent of LVM has disallowed the shrink of region %s\n",
			  region->name);
		goto out;
	}

	/* Deallocate the necessary extents from the volume. */
	lvm_deallocate_shrink_extents_from_volume(volume, remove_extents);

	/* Update the volume information. */
	rc = lvm_update_shrunk_volume(volume, remove_extents);
	if (rc) {
		LOG_ERROR("Error updating shrunk region %s\n", region->name);
		goto out;
	}

	/* Update the group information. */
	group->vg->pe_allocated -= remove_extents;

	/* Update the freespace and rebuild the LE maps. */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n",
			  group->container->name);
	}

	/* Mark container dirty. */
	group->container->flags |= SCFLAG_DIRTY;
	if (region->flags & SOFLAG_ACTIVE) {
		region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_set_volume
 **/
static void lvm_set_volume(storage_object_t * region,
			   boolean flag)
{
	LOG_ENTRY();
	LOG_DEBUG("That's nice. :)\n");
	LOG_EXIT_VOID();
}

/**
 * lvm_add_sectors_to_kill_list
 *
 * The kill_sectors list contains a list of sectors that need to be zeroed
 * during the next commit. This function is very similar to read/write.
 **/
static int lvm_add_sectors_to_kill_list(storage_object_t * region,
					lsn_t lsn,
					sector_count_t count)
{
	lvm_logical_volume_t * volume = region->private_data;
	lvm_physical_volume_t * pv_entry = NULL;
	lsn_t new_lsn = 0;
	sector_count_t new_count;
	int rc = 0;

	LOG_ENTRY();

	/* Region cannot be freespace. Don't kill
	 * past the end of the region either.
	 */
	if (region->data_type != DATA_TYPE ||
	    lsn + count > region->size) {
		rc = EIO;
		goto out;
	}

	/* I/Os needs to deal with the possibility that a request can
	 * come in that spans PEs or stripes. This is possible because
	 * there is no limit on count. To fix this, we loop through
	 * lvm_remap_sector and KILL_SECTORS until count reaches zero.
	 * Also make sure to check for missing LE entries.
	 */
	while (count) {
		lvm_remap_sector(volume, lsn, count,
				 &new_lsn, &new_count, &pv_entry);
		if (pv_entry && new_lsn) {
			rc = KILL_SECTORS(pv_entry->segment, new_lsn, new_count);
		}
		count -= new_count;
		lsn += new_count;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_commit_changes
 *
 * All commit operations are done in commit_container_changes.
 **/
static int lvm_commit_changes(storage_object_t * region,
			      unsigned int phase)
{
	LOG_ENTRY();
	LOG_DEBUG("That's nice. :)\n");
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_can_activate
 **/
static int lvm_can_activate(storage_object_t * region)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_activate
 **/
static int lvm_activate(storage_object_t * region)
{
	lvm_logical_volume_t *volume = region->private_data;
	dm_target_t *target_list;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Activating region %s.\n", region->name);

	rc = rename_region(region);
	if (!rc) {
		target_list = lvm_build_volume_targets(volume);
		if (target_list) {
			rc = EngFncs->dm_activate(region, target_list);
			EngFncs->dm_deallocate_targets(target_list);
		} else {
			rc = ENOMEM;
		}
	}

	if (rc) {
		LOG_ERROR("Error activating region %s\n", region->name);
	} else {
		region->flags &= ~SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_deactivate
 **/
static int lvm_can_deactivate(storage_object_t * region)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_deactivate
 **/
static int lvm_deactivate(storage_object_t * region)
{
	int rc;

	LOG_ENTRY();

	switch_region_name(region);
	rc = EngFncs->dm_deactivate(region);
	switch_region_name(region);

	if (rc) {
		LOG_ERROR("Error deactivating region %s\n", region->name);
	} else {
		region->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
		remove_old_region_name(region);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_get_option_count
 *
 * Determine the type of Task that is being performed, and return
 * the number of options that are available for that Task.
 **/
static int lvm_get_option_count(task_context_t * context)
{
	int count = 0;

	LOG_ENTRY();

	switch(context->action) {
	case EVMS_Task_Create:
		count = LVM_OPTION_PV_NAMES_INDEX + 1;
		break;
	case EVMS_Task_Create_Container:
		count = LVM_OPTION_PE_SIZE_INDEX + 1;
		break;
	case EVMS_Task_Expand:
		count = LVM_OPTION_EXPAND_PV_NAMES_INDEX + 1;
		break;
	case EVMS_Task_Shrink:
		count = LVM_OPTION_SHRINK_SIZE_INDEX + 1;
		break;
	case EVMS_Task_Set_Info:
		count = LVM_OPTION_LV_NAME_INDEX + 1;
		break;
	case EVMS_Task_Set_Container_Info:
		count = LVM_OPTION_VG_NAME_INDEX + 1;
		break;
	case LVM_FUNCTION_MOVE_EXTENT:
		count = LVM_OPTION_MOVE_EXTENT_PE_INDEX + 1;
		break;
	case LVM_FUNCTION_MOVE_PV:
		count = LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_IDX + 1;
		break;
	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}

/**
 * lvm_init_task
 *
 * Determine the type of Task that is being performed, and set up the
 * context structure with the appropriate initial values.
 **/
static int lvm_init_task(task_context_t * context)
{
	lvm_volume_group_t * group;
	int rc = 0;

	LOG_ENTRY();

	switch(context->action) {

	case EVMS_Task_Create:
		rc = lvm_create_region_allocate_option_descriptor(context->option_descriptors);
		if (!rc) {
			lvm_create_region_get_acceptable(context->acceptable_objects);
			context->min_selected_objects = 1;
			context->max_selected_objects = 1;
		}
		break;

	case EVMS_Task_Create_Container:
		rc = lvm_create_container_allocate_option_descriptor(context->option_descriptors);
		if (!rc) {
			rc = lvm_create_container_get_acceptable(context->acceptable_objects);
			context->min_selected_objects = 1;
			context->max_selected_objects = MAX_PV;
		}
		break;

	case EVMS_Task_Expand:
		if (context->object) {
			/* Don't allow expanding the freespace region. */
			if (context->object->data_type == FREE_SPACE_TYPE) {
				rc = EINVAL;
				break;
			}

			rc = lvm_expand_region_allocate_option_descriptor(context->option_descriptors);
			if (!rc) {
				context->min_selected_objects = 0;
				context->max_selected_objects = 0;
				rc = lvm_expand_region_init_options(context);
			}
		} else if (context->container) {
			/* If the container already has the max number
			 * of PVs, then no segments are acceptable.
			 */
			group = context->container->private_data;
			if (group->vg->pv_cur >= MAX_PV) {
				LOG_WARNING("Container %s has maximum number "
					    "of PVs (%d). Cannot expand.\n",
					    context->container->name,
					    group->vg->pv_cur);
				rc = EINVAL;
				break;
			}
	
			lvm_expand_container_allocate_option_descriptor(context->option_descriptors);
			rc = lvm_expand_container_get_acceptable(context->container,
								 context->acceptable_objects);
			context->min_selected_objects = 1;
			context->max_selected_objects = MAX_PV - group->vg->pv_cur;
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Shrink:
		if (context->object) {
			/* Don't allow shrinking the freespace region. */
			if (context->object->data_type == FREE_SPACE_TYPE) {
				rc = EINVAL;
				break;
			}

			rc = lvm_shrink_region_allocate_option_descriptor(context->option_descriptors);
			if (!rc) {
				context->min_selected_objects = 0;
				context->max_selected_objects = 0;
				rc = lvm_shrink_region_init_options(context);
			}
		} else if (context->container) {
			/* If the container has only one
			 * PV, then it cannot be shrunk.
			 */
			group = context->container->private_data;
			if (group->vg->pv_cur == 1) {
				LOG_WARNING("Container %s has a single PV. "
					    "Cannot shrink.\n",
					    context->container->name);
				rc = EINVAL;
				break;
			}

			lvm_shrink_container_allocate_option_descriptor(context->option_descriptors);
			lvm_shrink_container_get_acceptable(context->container,
							    context->acceptable_objects);
			context->min_selected_objects = 1;
			context->max_selected_objects = group->vg->pv_cur - 1;
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Set_Info:
		rc = lvm_set_volume_info_init_task(context);
		break;

	case EVMS_Task_Set_Container_Info:
		rc = lvm_set_group_info_init_task(context);
		break;

	case LVM_FUNCTION_MOVE_EXTENT:
		rc = lvm_move_extent_init_task(context);
		break;

	case LVM_FUNCTION_MOVE_PV:
		rc = lvm_move_pv_init_task(context);
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_set_option
 *
 * Determine the type of Task that is being performed. Then examine the
 * desired option (using the index), and verify that the given value is
 * appropriate. Reset the value if necessary and possible. Adjust other
 * options as appropriate.
 **/
static int lvm_set_option(task_context_t * context,
			  u_int32_t index,
			  value_t * value,
			  task_effect_t * effect)
{
	int rc = 0;

	LOG_ENTRY();

	switch (context->action) {

	case EVMS_Task_Create:
		rc = lvm_create_region_set_option(context, index,
						  value, effect);
		break;

	case EVMS_Task_Create_Container:
		rc = lvm_create_container_set_option(context, index,
						     value, effect);
		break;

	case EVMS_Task_Expand:
		if (context->object) {
			rc = lvm_expand_region_set_option(context, index,
							  value, effect);
		} else if (context->container) {
			lvm_expand_container_set_option(context, index,
							value, effect);
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Shrink:
		if (context->object) {
			rc = lvm_shrink_region_set_option(context, index,
							  value, effect);
		} else if (context->container) {
			lvm_shrink_container_set_option(context, index,
							value, effect);
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Set_Info:
		rc = lvm_set_volume_info_set_option(context, index,
						    value, effect);
		break;

	case EVMS_Task_Set_Container_Info:
		rc = lvm_set_group_info_set_option(context, index,
						   value, effect);
		break;

	case LVM_FUNCTION_MOVE_EXTENT:
		rc = lvm_move_extent_set_option(context, index,
						value, effect);
		break;

	case LVM_FUNCTION_MOVE_PV:
		rc = lvm_move_pv_set_option(context, index,
					    value, effect);
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_set_objects
 *
 * Determine the type of task, and then validate that the objects on the
 * "selected" list are valid for that task. If so, adjust the option
 * descriptor as appropriate.
 **/
static int lvm_set_objects(task_context_t * context,
			   list_anchor_t declined_objects,
			   task_effect_t * effect)
{
	int rc;

	LOG_ENTRY();

	switch(context->action) {

	case EVMS_Task_Create:
		rc = lvm_create_region_set_objects(context,
						   declined_objects, effect);
		break;

	case EVMS_Task_Create_Container:
		rc = lvm_create_container_set_objects(context,
						      declined_objects, effect);
		break;

	case EVMS_Task_Expand:
		if (context->object) {
			rc = 0;
		} else if (context->container) {
			rc = lvm_expand_container_set_objects(context,
							      declined_objects,
							      effect);
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Shrink:
		if (context->object) {
			rc = 0;
		} else if (context->container) {
			rc = lvm_shrink_container_set_objects(context,
							      declined_objects,
							      effect);
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Set_Info:
	case EVMS_Task_Set_Container_Info:
		rc = 0;
		break;

	case LVM_FUNCTION_MOVE_PV:
		rc = lvm_move_pv_set_objects(context, declined_objects, effect);
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_get_info
 *
 * Return LVM-specific information about the specified region. If the
 * name field is set, only return the "extra" information pertaining
 * to that name.
 **/
static int lvm_get_info(storage_object_t * region,
			char * name,
			extended_info_array_t ** info_array)
{
	lvm_logical_volume_t * volume = region->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (!name) {
		/* Default case. Return all basic info about the region. */
		rc = lvm_get_volume_info(volume, info_array);
	} else if (! strcmp(name, "Extents")) {
		/* "Extra" information about LE-to-PE mapping. */
		rc = lvm_get_volume_extent_info(volume, info_array);
	} else {
		LOG_ERROR("No support for extra region information about \"%s\"\n",
			  name);
		rc = EINVAL;
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_set_info
 *
 * Apply the settings of the options to the given region. This API currently
 * supports renaming the region.
 **/
static int lvm_set_info(storage_object_t * region,
			option_array_t * options)
{
	lvm_logical_volume_t *volume = region->private_data;
	lvm_volume_group_t *group = volume->group;
	char *lv_name;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Setting info for region %s.\n", region->name);

	/* No info to set for freespace regions. */
	if (region->data_type != DATA_TYPE) {
		LOG_ERROR("No information to set for freespace region %s.\n",
			  region->name);
		rc = EINVAL;
		goto out;
	}

	lvm_set_volume_info_parse_options(volume, options, &lv_name);
	rc = lvm_set_volume_info_validate_options(volume, lv_name);
	if (rc) {
		goto out;
	}

	lvm_set_new_volume_name(volume, lv_name);
	group->container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_get_plugin_info
 *
 * Return information about the LVM plugin. There is no "extra"
 * information about LVM, so "name" should always be NULL.
 **/
static int lvm_get_plugin_info(char * name,
			       extended_info_array_t ** info_array)
{
	extended_info_array_t * info = NULL;
	char buffer[50] = {0};
	int i = 0, rc = 0;
	
	LOG_ENTRY();

	if (!name) {
		/* Get memory for the info array. */
		info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
					     sizeof(extended_info_t)*7);
		if (!info) {
			LOG_CRITICAL("Memory error creating info array\n");
			rc = ENOMEM;
			goto out;
		}

		/* Short Name. */
		info->info[i].name = EngFncs->engine_strdup("ShortName");
		info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->short_name);
		i++;

		/* Long Name. */
		info->info[i].name = EngFncs->engine_strdup("LongName");
		info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->long_name);
		i++;

		/* Plugin Type. */
		info->info[i].name = EngFncs->engine_strdup("Type");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Type"));
		info->info[i].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("Region Manager"));
		i++;

		/* Plugin Version. */
		info->info[i].name = EngFncs->engine_strdup("Version");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d",
			 MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		/* Required Engine Services Version. */
		info->info[i].name = EngFncs->engine_strdup("Required_Engine_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
							      "It will not run on older versions of the Engine services."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d",
			 my_plugin_record->required_engine_api_version.major,
			 my_plugin_record->required_engine_api_version.minor,
			 my_plugin_record->required_engine_api_version.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		/* Required Plug-in API Version. */
		info->info[i].name = EngFncs->engine_strdup("Required_Plugin_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Plug-in API Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine plug-in API that this plug-in requires.  "
							      "It will not run on older versions of the Engine plug-in API."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d",
			 my_plugin_record->required_plugin_api_version.plugin.major,
			 my_plugin_record->required_plugin_api_version.plugin.minor,
			 my_plugin_record->required_plugin_api_version.plugin.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		/* Required Container API Version. */
		info->info[i].name = EngFncs->engine_strdup("Required_Container_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Container API Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine container API that this plug-in requires.  It will not run on older versions of the Engine container API."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d",
			 my_plugin_record->required_container_api_version.major,
			 my_plugin_record->required_container_api_version.minor,
			 my_plugin_record->required_container_api_version.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;
		info->count = i;

		*info_array = info;
	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n",
			  name);
		rc = EINVAL;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_read
 *
 * Perform a logical-to-physical remapping, and send the read down to
 * the next plugin.
 **/
static int lvm_read(storage_object_t * region,
		    lsn_t lsn,
		    sector_count_t count,
		    void * buffer)
{
	lvm_logical_volume_t * volume = region->private_data;
	lvm_physical_volume_t * pv_entry = NULL;
	lsn_t new_lsn = 0;
	sector_count_t new_count;
	int rc = 0;

	LOG_ENTRY();

	/* Boundary check. Also, don't read from freespace regions. */
	if (lsn + count > region->size ||
	    region->data_type != DATA_TYPE) {
		rc = EIO;
		goto out;
	}

	/* I/Os needs to deal with requests that can span PEs
	 * or stripes. This is possible because there is no
	 * limit on count. To fix this, we loop through
	 * lvm_remap_sector and READ until count reaches zero.
	 * Also make sure to check for missing LE entries.
	 */
	while (count) {
		lvm_remap_sector(volume, lsn, count,
				 &new_lsn, &new_count, &pv_entry);
		if (pv_entry && new_lsn) {
			rc = READ(pv_entry->segment, new_lsn, new_count, buffer);
		} else {
			memset(buffer, 0, sectors_to_bytes(new_count));
			rc = EIO;
		}
		count -= new_count;
		lsn += new_count;
		buffer = (void*)(((unsigned long)buffer) +
				 (unsigned long)(new_count << SECTOR_SIZE_BITS));
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_write
 *
 * Perform a logical-to-physical remapping, and send the write down to
 * the next plugin.
 **/
static int lvm_write(storage_object_t * region,
		     lsn_t lsn,
		     sector_count_t count,
		     void * buffer)
{
	lvm_logical_volume_t * volume = region->private_data;
	lvm_physical_volume_t * pv_entry;
	lsn_t new_lsn;
	sector_count_t new_count;
	int rc = 0;

	LOG_ENTRY();

	/* Boundary check. Also, don't write to freespace regions. */
	if (lsn + count > region->size ||
	    region->data_type != DATA_TYPE) {
		rc = EIO;
		goto out;
	}

	/* I/Os needs to deal with requests that span PEs or
	 * stripes. This is possible because there is no limit
	 * on count. To fix this, we loop through
	 * lvm_remap_sector and WRITE until count reaches zero.
	 * Also check for missing LE entries.
	 */
	while (count) {
		lvm_remap_sector(volume, lsn, count,
				 &new_lsn, &new_count, &pv_entry);
		if (pv_entry && new_lsn) {
			rc = WRITE(pv_entry->segment, new_lsn, new_count, buffer);
		} else {
			rc = EIO;
		}
		count -= new_count;
		lsn += new_count;
		buffer = (void*)(((unsigned long)buffer) +
				 (unsigned long)(new_count << SECTOR_SIZE_BITS));
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_get_region_functions
 *
 * Set up an array to describe the LVM-private-functions supported for the
 * specified region.
 **/
static int lvm_get_region_functions(storage_object_t * region,
				    function_info_array_t ** functions)
{
	lvm_logical_volume_t * volume;
	function_info_array_t * fia;
	int rc = 0;

	LOG_ENTRY();

	/* No LVM-global functions, and no functions for freespace regions. */
	if (!region ||
	    region->data_type == FREE_SPACE_TYPE) {
		rc = EINVAL;
		goto out;
	}

	/* Allocate the function info array. */
	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) +
				    sizeof(function_info_t) *
				    LVM_REGION_FUNCTION_COUNT);
	if (!fia) {
		LOG_CRITICAL("Memory error creating function-info array for region %s\n",
			     region->name);
		rc = ENOMEM;
		goto out;
	}

	*functions = fia;
	fia->count = LVM_REGION_FUNCTION_COUNT;
	volume = region->private_data;

	/* Set up the function-info array, with
	 * all functions initially inactive.
	 */
	fia->info[0].function = LVM_FUNCTION_MOVE_EXTENT;
	fia->info[0].flags = EVMS_FUNCTION_FLAGS_INACTIVE;
	fia->info[0].name = EngFncs->engine_strdup("move_extent");
	fia->info[0].title = EngFncs->engine_strdup(_("Move Extent"));
	fia->info[0].verb = EngFncs->engine_strdup(_("Move Extent"));
	fia->info[0].help = EngFncs->engine_strdup(_("Move a logical extent from its current physical extent to a different physical extent."));

	/* Determine if a "Move Extent" function is available. */

	/* The container must have free extents in order to do a move. */
	if (volume->group->freespace->lv->lv_allocated_le != 0) {

		/* Online move must be enabled or the region must be offline. */
		if (EngFncs->can_online_copy() ||
		    EngFncs->is_offline(region, NULL)) {
			fia->info[0].flags &= ~EVMS_FUNCTION_FLAGS_INACTIVE;
		}
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_region_function
 **/
static int lvm_region_function(storage_object_t * region,
			       task_action_t action,
			       list_anchor_t objects,
			       option_array_t * options)
{
	lvm_logical_volume_t * volume = region->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (region->data_type != DATA_TYPE) {
		rc = EINVAL;
		goto out;
	}

	switch (action) {

	case LVM_FUNCTION_MOVE_EXTENT:
		rc = lvm_move_extent(volume, options);
		break;

	default:
		LOG_ERROR("Action %d is not allowed for region %s\n",
			  action, region->name);
		rc = ENOSYS;
		break;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_backup_metadata
 *
 * To backup the metadata for a region, we just need to record the region's
 * dependency on it's producing container, since all the metadata is backed
 * up relative to the container.
 **/
static int lvm_backup_metadata(storage_object_t *region)
{
	int rc = 0;

	LOG_ENTRY();

	if (region->data_type == DATA_TYPE) {
		rc = EngFncs->save_metadata(region->name,
					    region->producing_container->name,
					    0, 0, NULL);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Container Functions ******/


/**
 * lvm_can_delete_container
 *
 * Can we remove the specified volume group?
 **/
static int lvm_can_delete_container(storage_container_t * container)
{
	lvm_volume_group_t * group = container->private_data;
	int rc = 0;

	LOG_ENTRY();

	/* The container must not contain any regions. */
	if (group->volume_count) {
		LOG_DEFAULT("Container %s is still exporting %d regions.\n",
			    container->name, group->volume_count);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_expand_container
 *
 * Get the list of available objects and see which ones can be added to this
 * container. If any object can be added, create an expand-point to add to
 * this list. Also check if any PVs can be expanded.
 **/
static int lvm_can_expand_container(storage_container_t *container,
				    list_anchor_t expand_points)	/* of type expand_object_info_t */
{
	lvm_volume_group_t *group = container->private_data;
	expand_object_info_t *expand_point;
	list_anchor_t objects = NULL;
	storage_object_t *segment;
	sector_count_t limit = 0;
	list_element_t iter;
	int rc, i;

	LOG_ENTRY();
	LOG_DEBUG("Checking if container %s can expand.\n", container->name);

	/* Get the list of available objects from the engine. */
	lvm_get_available_objects(container, &objects);
	if (group->vg->pv_cur < MAX_PV) {
		/* See which objects can be added to this container. */
		LIST_FOR_EACH(objects, iter, segment) {
			rc = lvm_can_add_object(segment, container);
			if (!rc) {
				limit += segment->size;
			}
		}

		if (limit) {
			expand_point = EngFncs->engine_alloc(sizeof(*expand_point));
			if (expand_point) {
				LOG_DEBUG("Adding container %s to the expand-"
					  "points list.\n", container->name);
				expand_point->container = container;
				expand_point->max_expand_size = limit;
				EngFncs->insert_thing(expand_points, expand_point,
						      INSERT_AFTER, NULL);
			}
		}
	} else {
		LOG_DEBUG("Container %s has maximum number of PVs (%d).\n",
			  container->name, group->vg->pv_cur);
	}
	EngFncs->destroy_list(objects);

	/* Check each existing PV to see if it can expand. */
	for (i = 0; i <= MAX_PV; i++) {
		if (group->pv_list[i]) {
			group->pv_list[i]->segment->plugin->
				functions.plugin->can_expand(group->pv_list[i]->segment,
							     -1, expand_points);
		}
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_can_expand_container_by
 *
 * PVs can expand by any amount.
 **/
static int lvm_can_expand_container_by(storage_container_t *container,
				       storage_object_t *consumed_object,
				       sector_count_t *delta_size)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_can_shrink_container
 *
 * If any one of the consumed objects can be removed, then the container can
 * shrink. Create a shrink_object_info_t and add it to the shrink_points list.
 * Also check if any consumed objects can be shrunk.
 **/
static int lvm_can_shrink_container(storage_container_t *container,
				    list_anchor_t shrink_points)	/* of type shrink_object_info_t */
{
	lvm_volume_group_t *group = container->private_data;
	shrink_object_info_t *shrink_point;
	sector_count_t limit = 0;
	int rc, i;

	LOG_ENTRY();
	LOG_DEBUG("Checking if container %s can shrink.\n", container->name);

	for (i = 0; i <= MAX_PV; i++) {
		if (group->pv_list[i]) {
			rc = lvm_can_remove_object(group->pv_list[i]->segment);
			if (!rc) {
				limit += group->pv_list[i]->pv->pe_total *
					 group->pv_list[i]->pv->pe_size;
			}
		}
	}

	if (limit) {
		shrink_point = EngFncs->engine_alloc(sizeof(*shrink_point));
		if (shrink_point) {
			LOG_DEBUG("Adding container %s to the shrink-points "
				  "list.\n", container->name);
			shrink_point->container = container;
			shrink_point->max_shrink_size = limit;
			EngFncs->insert_thing(shrink_points, shrink_point,
					      INSERT_AFTER, NULL);
		}
	}

	/* Determine which (if any) consumed objects can be shrunk. If they
	 * can, ask the object's plugin if they will allow a shrink. If they
	 * can, they will add themselves to the shrink_points list.
	 */
	for (i = 0; i <= MAX_PV; i++) {
		if (group->pv_list[i]) {
			rc = lvm_can_shrink_pv(group->pv_list[i], &limit);
			if (!rc) {
				group->pv_list[i]->segment->plugin->
					functions.plugin->can_shrink(group->pv_list[i]->segment,
								     limit,
								     shrink_points);
			}
		}
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_can_shrink_container_by
 *
 * Verify that we can allow this object to be shrunk by the specified amount.
 * If we can shrink, but not by the given delta-size, reduce the delta-size
 * appropriately.
 **/
static int lvm_can_shrink_container_by(storage_container_t *container,
				       storage_object_t *consumed_object,
				       sector_count_t *delta_size)
{
	lvm_physical_volume_t *pv_entry;
	u_int64_t shrink_limit;
	int rc;

	LOG_ENTRY();

	/* Make sure the consumed-object is really a PV. */
	if (consumed_object->consuming_container != container) {
		LOG_ERROR("Attempt to shrink object %s which isn't a PV in "
			  "container %s.\n", consumed_object->name,
			  container->name);
		rc = EINVAL;
		goto out;
	}

	pv_entry = lvm_get_pv_for_segment(consumed_object);
	if (!pv_entry) {
		rc = EINVAL;
		goto out;
	}

	rc = lvm_can_shrink_pv(pv_entry, &shrink_limit);
	if (rc) {
		LOG_ERROR("Cannot shrink object %s.\n", consumed_object->name);
		goto out;
	}

	if (shrink_limit < *delta_size) {
		LOG_WARNING("Object %s requested to be shrunk by %"PRIu64
			    " sectors. LVM can only allow shrinking by %"PRIu64
			    " sectors.\n", consumed_object->name,
			    *delta_size, shrink_limit);
		*delta_size = shrink_limit;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_container
 *
 * Create a new LVM volume group, using the specified segments.
 **/
static int lvm_create_container(list_anchor_t objects,
				option_array_t * options,
				storage_container_t ** container)
{
	storage_object_t * object;
	vg_disk_t * vg;
	lvm_volume_group_t * new_group;
	storage_container_t * disk_group = NULL;
	unsigned char vg_name[NAME_LEN] = {0};
	list_element_t itr;
	u_int32_t pe_size;
	int rc;

	LOG_ENTRY();

	/* There needs to be at least one segment.
	 * Don't create an empty group.
	 */
	if (EngFncs->list_empty(objects)) {
		LOG_ERROR("No objects specified.\n");
		LOG_ERROR("Cannot create an empty container\n");
		rc = EINVAL;
		goto out;
	}

	/* Get the appropriate options for group creation. */
	rc = lvm_create_container_parse_option_array(options, vg_name, &pe_size);
	if (rc) {
		LOG_ERROR("Error parsing and verifying options\n");
		goto out;
	}

	/* Make sure every segment is valid for use in a new
	 * group, and large enough for the specified pe_size.
	 */
	LIST_FOR_EACH(objects, itr, object) {
		rc = lvm_check_segment_for_group_inclusion(object, NULL);
		if (rc) {
			LOG_ERROR("Object %s cannot be added to an LVM container.\n",
				  object->name);
			goto out;
		}

		rc = lvm_check_segment_for_pe_size(object, &pe_size);
		if (rc) {
			LOG_ERROR("Object %s cannot be added to an LVM container.\n",
				  object->name);
			goto out;
		}

		/* The engine will ensure that all segments come from the same
		 * CSM disk-group (or no disk-group), so get the disk-group from
		 * any one of the objects.
		 */
		if (!disk_group && object->disk_group) {
			disk_group = object->disk_group;
		}
	}

	/* Allocate and initialize a vg_disk_t
	 * structure with the user options.
	 */
	vg = lvm_initialize_new_vg(pe_size);
	if (!vg) {
		LOG_ERROR("Error initializing new VG metadata\n");
		rc = EINVAL;
		goto out;
	}

	/* Create the volume group. */
	new_group = lvm_allocate_volume_group(vg, vg_name, disk_group);
	if (!new_group) {
		LOG_CRITICAL("Memory error creating new container %s\n",
			     vg_name);
		rc = ENOMEM;
		goto out;
	}

	/* The group has no UUID list or LV array to read yet, so
	 * set the flags to prevent spurious error messages.
	 */
	new_group->flags |= (LVM_VG_FLAG_UUID_LIST_PRESENT |
			     LVM_VG_FLAG_LV_LIST_PRESENT);

	/* Create the freespace region for the new group. */
	rc = lvm_create_freespace_volume(new_group);
	if (rc) {
		lvm_deallocate_volume_group(new_group);
		goto out;
	}

	/* Add each segment to the new group. */
	LIST_FOR_EACH(objects, itr, object) {
		rc = lvm_transfer_segment_to_group(object, new_group);
		if (rc) {
			lvm_deallocate_volume_group(new_group);
			goto out;
		}
	}

	/* Update the freespace region. */
	rc = lvm_update_freespace_volume(new_group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace in container %s\n",
			    new_group->container->name);
		goto out;
	}

	/* Mark the container dirty. */
	new_group->container->flags |= SCFLAG_DIRTY;

	/* Place the new group in the global list. */
	EngFncs->insert_thing(lvm_group_list, new_group, INSERT_AFTER, NULL);
	*container = new_group->container;

	LOG_DEFAULT("Successfully created new container %s\n",
		    new_group->container->name);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_add_object
 *
 * Add the specified object to the specified (existing) container. The
 * object must have already been removed from its previous location
 * before this function is called. This function is not used to move a
 * segment from one LVM container to another LVM container. The function
 * lvm_transfer_object will be used for that purpose. Externally, this
 * function will be used to emulate LVM's pvcreate command. No options
 * are available for this function, since all requirements have been set
 * by the existing container.
 **/
static int lvm_add_object(storage_object_t * object,
			  storage_container_t * container,
			  option_array_t * options)
{
	lvm_volume_group_t * group = container->private_data;
	lvm_physical_volume_t * pv_entry;
	u_int32_t pe_size;
	int rc;

	LOG_ENTRY();

	/* Check that the specified segment is not owned by LVM. */
	rc = lvm_check_segment_for_group(object, NULL);
	if (rc) {
		goto out;
	}

	/* Check that the segment can be added to a group. */
	rc = lvm_check_segment_for_group_inclusion(object, group);
	if (rc) {
		goto out;
	}

	/* Check that the segment is big enough to add to this group. */
	pe_size = group->vg->pe_size;
	rc = lvm_check_segment_for_pe_size(object, &pe_size);
	if (rc) {
		goto out;
	}
	
	/* Create a new physical volume for the segment. */
	pv_entry = lvm_create_pv_from_segment(object);
	if (!pv_entry) {
		rc = ENOMEM;
		goto out;
	}

	/* Add the new physical volume to the group. */
	rc = lvm_add_new_pv_to_group(pv_entry, group);
	if (rc) {
		LOG_SERIOUS("Error adding object %s to container %s\n",
			    object->name, group->container->name);
		lvm_deallocate_physical_volume(pv_entry);
		goto out;
	}

	/* The freespace volume needs to be adjusted
	 * to include the space on this new PV.
	 */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace for container %s\n",
			    group->container->name);
		goto out;
	}

	/* Mark the container dirty. */
	group->container->flags |= SCFLAG_DIRTY;

	LOG_DEFAULT("Successfully added object %s to container %s\n",
		    object->name, container->name);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_remove_object
 *
 * Remove the specified segment from its current container. Make sure the
 * segment does not contain any allocated regions. In addition to removing
 * it from the container, we want to kill any metadata that might have been
 * on the segment.
 **/
static int lvm_remove_object(storage_object_t * object)
{
	lvm_volume_group_t * group;
	lvm_physical_volume_t * pv_entry;
	int rc;

	LOG_ENTRY();

	/* Check that the segment is part of an LVM container. */
	rc = lvm_check_segment_for_group(object, NULL);
	if (!rc) {
		rc = EINVAL;
		goto out;
	}
	group = object->consuming_container->private_data;

	/* Make sure this PV isn't the last one in the group. To take the last
	 * PV out, use the lvm_delete_container function. We don't want empty
	 * groups hanging around.
	 */
	if (group->pv_count == 1) {
		LOG_ERROR("Cannot remove the last object from container %s\n",
			  group->container->name);
		rc = EINVAL;
		goto out;
	}

	/* Get the PV for this segment. */
	pv_entry = lvm_get_pv_for_segment(object);
	if (!pv_entry) {
		LOG_ERROR("Could not find PV entry for object %s\n",
			  object->name);
		rc = EINVAL;
		goto out;
	}

	/* Make sure the PV is not in use by any volumes. */
	if (pv_entry->pv->lv_cur) {
		LOG_ERROR("Object %s is in use by %d regions\n",
			  object->name, pv_entry->pv->lv_cur);
		rc = EINVAL;
		goto out;
	}

	/* Make sure no extents are being moved to this PV. */
	if (pv_entry->move_extents > 0) {
		LOG_WARNING("Object %s is scheduled to have %d extents moved to it\n",
			    object->name, pv_entry->move_extents);
		rc = EINVAL;
		goto out;
	}

	/* Take the segment out of the source group. */
	rc = lvm_remove_pv_from_group(pv_entry);
	if (rc) {
		LOG_SERIOUS("Error removing object %s from container %s\n",
			    object->name, group->container->name);
		goto out;
	}

	/* The freespace volume needs to be adjusted. */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace for container %s\n",
			    group->container->name);
		goto out;
	}

	/* Erase the PV metadata and deallocate. */
	lvm_erase_pv(object);
	lvm_deallocate_physical_volume(pv_entry);

	/* Mark the container dirty. */
	group->container->flags |= SCFLAG_DIRTY;

	LOG_DEFAULT("Successfully removed object %s from container %s\n",
		    object->name, group->container->name);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_expand_container
 *
 * Expand the specified container. This can take place in one of two ways.
 * 1. If expand_object is set, then one of the container's consumed objects
 *    will be expanded. Send the expand command down to the specified
 *    consumed_object. If the command succeeds, adjust the PV data for that
 *    object as necessary to expand the container.
 * 2. If expand_object is not set, then the container will expand by adding
 *    one or more objects, which are specified in the input_objects list.
 *    Add each object using the specified options.
 **/
static int lvm_expand_container(storage_container_t *container,
				storage_object_t *consumed_object,
				storage_object_t *expand_object,
				list_anchor_t input_objects,
				option_array_t *options)
{
	storage_object_t *object;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Expanding container %s.\n", container->name);

	if (expand_object) {
		/* One of the consumed objects will be expanding. */
		rc = lvm_expand_pv_in_container(container, consumed_object,
						expand_object,
						input_objects, options);
	} else {
		/* Add this list of new objects to the container. */
		LIST_FOR_EACH(input_objects, iter, object) {
			rc = lvm_add_object(object, container, options);
			if (rc) {
				LOG_ERROR("Error adding object %s to container "
					  "%s. Aborting remaining expands.\n",
					  object->name, container->name);
				break;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_shrink_container
 *
 * Shrink the specified container. This can take place in one of two ways.
 * 1. If shrink_object is set, then one of the container's consumed objects
 *    will be shrunk. Send the shrink command down to the specified
 *    consumed_object. If the command succeeds, adjust the PV data for that
 *    object as necessary to shrink the container.
 * 2. If shrink_object is not set, then the container will shrink by removing
 *    one or more objects, which are specified in the input_objects list.
 *    Remove each object using the specified options.
 **/
static int lvm_shrink_container(storage_container_t *container,
				storage_object_t *consumed_object,
				storage_object_t *shrink_object,
				list_anchor_t input_objects,
				option_array_t *options)
{
	storage_object_t *object;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Shrinking container %s.\n", container->name);

	if (shrink_object) {
		/* One of the consumed objects will be shrinking. */
		rc = lvm_shrink_pv_in_container(container, consumed_object,
						shrink_object,
						input_objects, options);
	} else {
		/* Remove this list of objects from the container. */
		LIST_FOR_EACH(input_objects, iter, object) {
			rc = lvm_remove_object(object);
			if (rc) {
				LOG_ERROR("Error removing object %s from "
					  "container %s. Aborting remaining "
					  "shrinks.\n", object->name,
					  container->name);
				break;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_discard_container
 *
 * Forget about this container. Similar to delete, but we're not updating
 * any metadata. Simply deallocate any private data structures associated
 * with this container. Assume that all produced regions have already been
 * discarded.
 **/
static int lvm_discard_container(storage_container_t * container)
{
	lvm_volume_group_t * group = container->private_data;

	LOG_ENTRY();

	lvm_deallocate_volume_group(group);

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_delete_container
 *
 * Remove the specified container. Take all segments from the container
 * and put back in the unassigned group. Free all memory used by the
 * specified volume group.
 **/
static int lvm_delete_container(storage_container_t * container,
				list_anchor_t objects)
{
	lvm_volume_group_t * group = container->private_data;
	lvm_physical_volume_t * pv_entry;
	storage_object_t * object;
	int i, rc = 0;

	LOG_ENTRY();

	/* Does this group contain any volumes? */
	if (group->volume_count) {
		LOG_ERROR("Container %s contains %d regions\n",
			  container->name, group->volume_count);
		rc = EINVAL;
		goto out;
	}

	/* Move each PV in the group to the unassigned group. This array
	 * traversal goes backwards, because when PVs are removed from the
	 * group, all remaining PVs are shifted down to fill in the slots.
	 */
	for (i = MAX_PV; i > 0; i--) {
		if (!group->pv_list[i]) {
			continue;
		}
		pv_entry = group->pv_list[i];

		LOG_DETAILS("Removing object %s\n", pv_entry->segment->name);

		/* Take the PV out of the group. */
		rc = lvm_remove_pv_from_group(pv_entry);
		if (rc) {
			LOG_SERIOUS("Error removing object %s from container %s\n",
				    pv_entry->segment->name, container->name);
		}

		/* Erase the PV metadata and deallocate. */
		object = pv_entry->segment;
		lvm_erase_pv(object);
		lvm_deallocate_physical_volume(pv_entry);

		/* Put the object on the output list. */
		EngFncs->insert_thing(objects, object, INSERT_AFTER, NULL);
	}

	/* Update the freespace region. This is done to free up the
	 * parent/child relationships between the freespace and the PVs.
	 */
	lvm_update_freespace_volume(group);

	/* Release memory for the group. */
	lvm_deallocate_volume_group(group);

	LOG_DEFAULT("Successfully removed container\n");

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_commit_metadata
 *
 * Write all of the metadata for this container to the PVs.
 **/
static int lvm_commit_metadata(lvm_volume_group_t * group, boolean backup)
{
	lvm_physical_volume_t * pv_entry;
	int i;

	LOG_ENTRY();

	/* Do the LV metadata endian-conversion here, so it only
	 * needs to be done once, instead of once for every PV.
	 */
	lvm_endian_convert_lvs(group);

	/* For each PV in the group, we need to write the PV structure, the
	 * VG structure, the UUID list, the LV array, and the PE map.
	 */
	for (i = 1; i <= MAX_PV; i++) {
		pv_entry = group->pv_list[i];
		if (pv_entry) {
			lvm_write_pv(pv_entry, backup);
			lvm_write_vg(pv_entry, backup);
			lvm_write_uuid_list(pv_entry, backup);
			lvm_write_lv_array(pv_entry, backup);
			lvm_write_pe_map(pv_entry, backup);
		}
	}

	/* Have to convert the LVs back. */
	lvm_endian_convert_lvs(group);

	LOG_DETAILS("Wrote container %s to disk\n", group->container->name);
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_commit_container_changes
 *
 * On phase 1, write all container metadata.
 * On phase 4, move all pending extents.
 **/
static int lvm_commit_container_changes(storage_container_t * container,
					commit_phase_t phase)
{
	lvm_volume_group_t * group = container->private_data;

	LOG_ENTRY();

	switch (phase) {
	case FIRST_METADATA_WRITE:
		lvm_commit_metadata(group, FALSE);
		break;
	case MOVE:
		lvm_commit_move_extents(group);
		break;
	default:
		goto out;
	}

	if (!group->move_extents) {
		container->flags &= ~SCFLAG_DIRTY;
	}

out:
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_get_container_info
 *
 * Return an information array for the specified container. If the name
 * field is set, only return the "extra" information pertaining to that
 * name.
 **/
static int lvm_get_container_info(storage_container_t * container,
				  char * name,
				  extended_info_array_t ** info_array)
{
	lvm_volume_group_t * group = container->private_data;
	char * buf = NULL;
	int pv_number, rc;

	LOG_ENTRY();

	if (!name) {
		/* Default case. Return all basic info about the container. */
		rc = lvm_get_group_info(group, info_array);
	} else if (! strcmp(name, "Current_PVs")) {
		/* Return a list of names of all PVs. */
		rc = lvm_get_group_pv_list_info(group, info_array);
	} else if (! strcmp(name, "Current_LVs")) {
		/* Return a list of names of all LVs. */
		rc = lvm_get_group_lv_list_info(group, info_array);
	} else if ((buf = strstr(name, "PV")) == name) {
		/* PV-specific information. This is
		 * what we will use for pvdisplay.
		 */
		pv_number = atoi(&buf[2]);
		rc = lvm_get_pv_info(group->pv_list[pv_number], info_array);
	} else if ((buf = strstr(name, "PEMapPV")) == name) {
		/* PV-specific extent information. This
		 * will be used for pvdisplay -v.
		 */
		pv_number = atoi(&buf[7]);
		rc = lvm_get_pv_extent_info(group->pv_list[pv_number],
					    info_array);
	} else {
		LOG_ERROR("No support for extra information about \"%s\"\n",
			  name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_set_container_info
 **/
static int lvm_set_container_info(storage_container_t * container,
				  option_array_t * options)
{
	lvm_volume_group_t *group = container->private_data;
	char new_container_name[EVMS_NAME_SIZE+1];
	char old_container_name[EVMS_NAME_SIZE+1];
	char lv_name[EVMS_NAME_SIZE+1];
	char *vg_name, *region_base_name;
	lvm_physical_volume_t *pv_entry;
	lvm_logical_volume_t *volume;
	int rc, i;

	LOG_ENTRY();
	LOG_DEBUG("Setting info for container %s.\n", container->name);

	lvm_set_group_info_parse_options(group, options, &vg_name);
	rc = lvm_set_group_info_validate_options(group, vg_name);
	if (rc) {
		goto out;
	}

	/* Save the old container name to use when renaming the regions. */
	strncpy(old_container_name, container->name, EVMS_NAME_SIZE);

	/* Copy the new name to the container. */
	lvm_translate_vg_name_to_container_name(vg_name, new_container_name,
						container->disk_group);
	EngFncs->register_name(new_container_name);
	EngFncs->unregister_name(old_container_name);
	memset(container->name, 0, EVMS_NAME_SIZE+1);
	strncpy(container->name, new_container_name, EVMS_NAME_SIZE);

	/* Copy the new VG-name into all the PV metadata. */
	for (i = 0; i <= MAX_PV; i++) {
		if ((pv_entry = group->pv_list[i])) {
			memset(pv_entry->pv->vg_name, 0, NAME_LEN);
			strncpy(pv_entry->pv->vg_name, vg_name, NAME_LEN-1);
		}
	}

	/* Copy the new VG-name into all the LV metadata, and
	 * set the new names for each of the produced volumes.
	 */
	for (i = 0; i <= MAX_LV; i++) {
		if ((volume = group->volume_list[i])) {
			memset(volume->lv->vg_name, 0, NAME_LEN);
			strncpy(volume->lv->vg_name, vg_name, NAME_LEN-1);

			region_base_name = volume->region->name +
					   strlen(old_container_name) + 1;
			strncpy(lv_name, region_base_name, EVMS_NAME_SIZE);
			lvm_set_new_volume_name(volume, lv_name);
		}
	}

	/* Set the new name for the freespace volume. */
	volume = group->freespace;
	region_base_name = volume->region->name + strlen(old_container_name) + 1;
	strncpy(lv_name, region_base_name, EVMS_NAME_SIZE);
	lvm_set_new_volume_name(volume, lv_name);

	container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_get_container_functions
 *
 * Determine if any LVM-container-specific functions are available to run
 * on the specified container.
 *
 * Possible functions:
 *   Move-PV
 **/
static int lvm_get_container_functions(storage_container_t * container,
				       function_info_array_t ** functions)
{
	function_info_array_t * fia;
	lvm_volume_group_t * group;
	lvm_physical_volume_t * dummy_pv = NULL;
	int i, rc;

	LOG_ENTRY();

	/* No LVM-global functions. */
	if (!container) {
		rc = EINVAL;
		goto out;
	}

	/* Allocate the function info array. */
	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) +
				    sizeof(function_info_t) *
				    LVM_CONTAINER_FUNCTION_COUNT);
	if (!fia) {
		LOG_CRITICAL("Memory error creating function-info array for "
			     "container %s\n", container->name);
		rc = ENOMEM;
		goto out;
	}

	*functions = fia;
	fia->count = LVM_CONTAINER_FUNCTION_COUNT;
	group = container->private_data;

	/* Set up the function-info array, with
	 * all functions initially inactive.
	 */
	fia->info[0].function = LVM_FUNCTION_MOVE_PV;
	fia->info[0].flags = EVMS_FUNCTION_FLAGS_INACTIVE;
	fia->info[0].name = EngFncs->engine_strdup("move_pv");
	fia->info[0].title = EngFncs->engine_strdup(_("Move PV"));
	fia->info[0].verb = EngFncs->engine_strdup(_("Move PV"));
	fia->info[0].help =
		EngFncs->engine_strdup(_("Move all allocated extents from one "
					 "PV to other PVs in the container."));

	/* Determine if a Move-PV function is available. At least one PV in the
	 * group must be movable. Don't bother with more complicated checks if
	 * there is no freespace. Also, assume striping does not have to be
	 * maintained, to get the widest range of possible options for the user.
	 */
	if (group->freespace->lv->lv_allocated_le > 0) {
		for (i = 0; i <= MAX_PV; i++) {
			if (!group->pv_list[i]) {
				continue;
			}

			rc = lvm_can_move_pv(group->pv_list[i], &dummy_pv,
					     MAINTAIN_STRIPES_OFF);
			if (!rc) {
				/* Found at least one moveable PV. */
				fia->info[0].flags &= ~EVMS_FUNCTION_FLAGS_INACTIVE;
				break;
			}
		}
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_container_function
 *
 * Determine which container function is being performed, and call the
 * appropriate subroutine. Available container functions are:
 *  MOVE_PV
 **/
static int lvm_container_function(storage_container_t * container,
				  task_action_t action,
				  list_anchor_t objects,
				  option_array_t * options)
{
	lvm_volume_group_t * group = container->private_data;
	int rc = 0;

	LOG_ENTRY();

	switch (action) {

	case LVM_FUNCTION_MOVE_PV:
		rc = lvm_move_pv(group, objects, options);
		break;

	default:
		LOG_ERROR("Action %d is not allowed for container %s\n",
			  action, container->name);
		rc = ENOSYS;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_backup_container_metadata
 *
 * Write the metadata for this container to the Engine's metadata-backup
 * database.
 **/
static int lvm_backup_container_metadata(storage_container_t * container)
{
	lvm_volume_group_t * group = container->private_data;
	int rc;

	LOG_ENTRY();

	rc = lvm_commit_metadata(group, TRUE);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function tables for the LVM Region Manager */
static plugin_functions_t lvm_functions = {
	.setup_evms_plugin		= lvm_setup_evms_plugin,
	.cleanup_evms_plugin		= lvm_cleanup_evms_plugin,
	.can_delete			= lvm_can_delete,
	.can_expand			= lvm_can_expand,
	.can_shrink			= lvm_can_shrink,
	.can_set_volume			= lvm_can_set_volume,
	.discover			= lvm_discover,
	.create				= lvm_create,
	.discard			= lvm_discard,
	.delete				= lvm_delete,
	.expand				= lvm_expand,
	.shrink				= lvm_shrink,
	.set_volume			= lvm_set_volume,
	.add_sectors_to_kill_list	= lvm_add_sectors_to_kill_list,
	.commit_changes			= lvm_commit_changes,
	.activate			= lvm_activate,
	.can_activate			= lvm_can_activate,
	.can_deactivate			= lvm_can_deactivate,
	.deactivate			= lvm_deactivate,
	.get_option_count		= lvm_get_option_count,
	.init_task			= lvm_init_task,
	.set_option			= lvm_set_option,
	.set_objects			= lvm_set_objects,
	.get_info			= lvm_get_info,
	.set_info			= lvm_set_info,
	.get_plugin_info		= lvm_get_plugin_info,
	.read				= lvm_read,
	.write				= lvm_write,
	.get_plugin_functions		= lvm_get_region_functions,
	.plugin_function		= lvm_region_function,
	.backup_metadata		= lvm_backup_metadata,
};

static container_functions_t lvm_container_functions = {
	.can_delete_container		= lvm_can_delete_container,
	.can_expand_container		= lvm_can_expand_container,
	.can_expand_container_by	= lvm_can_expand_container_by,
	.can_shrink_container		= lvm_can_shrink_container,
	.can_shrink_container_by	= lvm_can_shrink_container_by,
	.create_container		= lvm_create_container,
	.expand_container		= lvm_expand_container,
	.shrink_container		= lvm_shrink_container,
	.discard_container		= lvm_discard_container,
	.delete_container		= lvm_delete_container,
	.commit_container_changes	= lvm_commit_container_changes,
	.get_container_info		= lvm_get_container_info,
	.set_container_info		= lvm_set_container_info,
	.get_plugin_functions		= lvm_get_container_functions,
	.plugin_function		= lvm_container_function,
	.backup_container_metadata	= lvm_backup_container_metadata,
};

plugin_record_t LVM_Plugin = {
	.id = EVMS_LVM_PLUGIN_ID,
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.plugin	= {
			.major		= 13,
			.minor		= 1,
			.patchlevel	= 0
		}
	},
	.required_container_api_version = {
		.major		= 10,
		.minor		= 1,
		.patchlevel	= 0
	},
	.short_name = EVMS_LVM_PLUGIN_SHORT_NAME,
	.long_name = EVMS_LVM_PLUGIN_LONG_NAME,
	.oem_name = EVMS_IBM_OEM_NAME,
	.functions = {
		.plugin = &lvm_functions
	},
	.container_functions = &lvm_container_functions
};

plugin_record_t * evms_plugin_records[] = {
	&LVM_Plugin,
	NULL
};

