/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_discover.c
 *
 * Description: This file contains all functions related to the initial
 *              discovery of MD physical volumes, volume groups, and logical
 *              volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <plugin.h>

#include "md.h"

#define my_plugin_record my_plugin

/*
 * md_check_for_saved_info
 *
 *	This function is used during discovery.  It's called after we know that
 * the object is an MD device.  If a saved SB info block exists, then return 0
 * and saved the allocate saved_info.
 */
static void md_check_for_saved_info(md_member_t *member)
{
	int rc = 0;

	LOG_ENTRY();
	rc = md_read_saved_info(member);
	if (!rc && member->saved_info) {
		LOG_DEFAULT("Found saved info on %s.\n", member->obj->name);
	}
	LOG_EXIT_VOID();
}

static boolean md_find_slot_for_multipath_object(md_volume_t *vol, md_member_t *member)
{
	md_array_info_t *array;
	boolean success = FALSE;
	int i;

	LOG_ENTRY();

	array = vol->sb_func->get_disk_array_info(vol->sb);
	if (!array) {
		goto out;
	}

	if (array->personality != MULTIPATH) {
		LOG_ERROR("Region %s is not a multipath array.\n", vol->name);
		goto out;
	}

	i = EngFncs->list_count(vol->members) - 1;
	member->dev_number = i;
	member->raid_disk = i;
	member->flags = MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC;
	vol->sb_func->set_this_device_info(member);

	LOG_DEBUG("Assigned multipath device %s to slot %d\n",
		  member->obj->name, i);
	success = TRUE;

out:
	if (array) {
		EngFncs->engine_free(array);
	}
	LOG_EXIT_BOOL(success);
	return success;
}


/* Function: md_find_volume_for_object
 *
 * The input object has been validated as MD member object.  Find the MD volume
 * set that this object belongs to.  If this is the first object, create a new
 * MD volume structure.
 *
 * Note that this function will duplicate the input superblock, thus the
 * caller should free it.
 */
static int md_find_volume_for_object(storage_object_t * obj, md_sb_ver_t *sb_ver, void *sb)
{
	md_volume_t * vol = volume_list_head;
	md_member_t *member;
	int rc = 0;
	
	LOG_ENTRY();

	member = md_allocate_member(obj);
	if (!member) {
		rc = ENOMEM;
		goto out;
	}
	
	while (vol) {
		if (!vol->sb_func) {
			LOG_MD_BUG();
			continue;
		}
		if ( !(memcmp(sb_ver, &vol->sb_ver, sizeof(*sb_ver)))
		    && (md_same_uuid(vol->sb_func, vol->sb, sb)) ) {
			md_volume_add_member(vol, member);
			md_member_set_sb(member, sb);
			
			if (vol->personality == MULTIPATH) {
				if (!md_find_slot_for_multipath_object(vol, member)) {
					LOG_MD_BUG();
					md_free_member(member);
					rc = ENODEV;
					goto out;
				}
			}


			if ((vol->flags & MD_DISCOVERED) && (vol->region != NULL)) {
				// The volume has been discovered.  The
				// volume may have been exported in degrade
				// mode and this object is late for the show,
				// or this may be a stale object that has
				// been resurrected.
				LOG_WARNING("Volume %s has been discovered already."
					    "  Object %s comes in late.\n",
					    vol->name, obj->name);
				md_append_region_to_object(vol->region, member->obj);
				vol->flags |= MD_NEEDS_REDISCOVER;
			}
			break;
		}
		vol = vol->next;
	}

	if (!vol) {
		// Create a new volume entry
		vol = md_allocate_volume();
		if ( !vol ) {
			md_free_member(member);
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		md_volume_set_master_sb(vol, sb_ver, sb);
		md_volume_add_member(vol, member);
		md_member_set_sb(member, sb);
		md_volume_set_name(vol, NULL);

		if (vol->personality == MULTIPATH) {
			if (!md_find_slot_for_multipath_object(vol, member)) {
				LOG_MD_BUG();
				md_free_volume(vol);
				rc = ENODEV;
				goto out;
			}
		}
	}
	
	// Cluster naming support
	if (obj->disk_group) {
		char new_name[EVMS_NAME_SIZE + 1];
		sprintf(new_name, "%s/%s", obj->disk_group->name, vol->name);
		md_volume_set_name(vol, new_name);
	}
	
	md_check_for_saved_info(member);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: md_check_for_sb
 *
 *	This function is used during discovery.
 *	
 *	First, the old MD superblock format is assumed and MD_SB_BYTES
 *	will be read from the known superblock location.  If successful,
 *	the function will return 0 along with the allocated superblock
 *	and version of this MD superblock (MD_SB_VER_0)
 *
 *	This object may have MD version 1 superblock.  MD_SB_1_BYTES
 *	will be read from various locations.  If successful, the function will
 *	return 0 along with the allocated superblock and version MD_SB_VER_1.
 *	
 *	Otherwise, this function will return a non zero error code.
 *
 *	Note:  The caller should use the Engine service to free the buffer.
 */
static int md_check_for_sb(storage_object_t *obj, md_sb_ver_t *sb_ver, void **sb)
{
	int rc = 0;
	LOG_ENTRY();

	/* Try superblock version 0.90 first */

	rc = md_read_sb0(obj, sb);
	if (!rc) {
		sb_ver->major_version = MD_SB_VER_0;
		sb_ver->minor_version = 90;
		sb_ver->patchlevel = 0;
		LOG_EXIT_INT(rc);
		return rc;
	} else {
		rc = md_read_sb1(obj, sb, sb_ver);
		if (!rc) {
			LOG_EXIT_INT(rc);
			return rc;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}



/*
 * md_check_for_expand_shrink_in_progress
 *
 * This function checks for MD_SAVED_INFO_EXPAND_IN_PROGRESS and
 * MD_SAVED_INFO_SHRINK_IN_PROGRESS in the "saved" area on MD child objects.
 *
 * If one of these flags is set, this function will return TRUE and set
 * the corresponding member.
 */
boolean md_check_for_expand_shrink_in_progress(md_volume_t *vol, md_member_t **member)
{
	md_member_t *my_member;
	list_element_t iter;
	md_saved_info_t *saved_info;

	LOG_ENTRY();

	LIST_FOR_EACH(vol->members, iter, my_member) {
		saved_info = my_member->saved_info;
		if (saved_info) {
			if (saved_info->flags & (MD_SAVED_INFO_EXPAND_IN_PROGRESS | MD_SAVED_INFO_SHRINK_IN_PROGRESS)) {
				LOG_DEFAULT("Saved area on %s indicates that the %s"
					" process of [%s] is (was) in progress."
					"  The sector mark is %"PRIu64".\n",
					my_member->obj->name,
					(saved_info->flags & MD_SAVED_INFO_EXPAND_IN_PROGRESS) ? "EXPANSION" : "SHRINKING",
					vol->name,
					saved_info->sector_mark);
				*member = my_member;
				LOG_EXIT_BOOL(TRUE);
				return TRUE;
			}
		}
	}
	LOG_EXIT_BOOL(FALSE);
	return FALSE;
}


/* md_discover_volumes
 *
 * This function is the entry point into the first half of discovery,
 * which examines all objects to find MD superblock.  All MD objects
 * will be grouped into MD volumes (arrays).
 */
void md_discover_volumes( list_anchor_t input_list, list_anchor_t output_list)
{
	storage_object_t * obj;
	void * sb;
	md_sb_ver_t sb_ver;
	list_element_t iter = NULL;
	md_volume_t * vol;
	md_member_t *member;

	LOG_ENTRY();
	LOG_DETAILS("Searching for MD Super Blocks.\n");

	LIST_FOR_EACH(input_list, iter, obj) {
		if (obj->data_type == DATA_TYPE) {
			if (md_check_for_sb(obj, &sb_ver, &sb)) {
				// wasn't ours, put it on the output list now.
				EngFncs->insert_thing(output_list, obj, INSERT_AFTER, NULL);
			} else {
				if ( md_find_volume_for_object(obj, &sb_ver, sb)) {	
					LOG_WARNING("Error finding volume for object %s\n",
						    obj->name);
					// Object was not claimed for a volume.
					// Put it on the output list.
					EngFncs->insert_thing(output_list, obj, INSERT_AFTER, NULL);
				}
				EngFncs->engine_free(sb);
			}
		} else {
			LOG_DETAILS("Skipping object %s because not DATA_TYPE.\n", obj->name);
		}
	}

	for (vol = volume_list_head; vol; vol = vol->next) {

		if ((vol->flags & MD_DISCOVERED) && !(vol->flags & MD_NEEDS_REDISCOVER)) {
			continue;
		}

		LOG_DEBUG("Found %d disks for %s region.\n", vol->nr_disks, vol->name);
		vol->active_disks = 0;
		vol->working_disks = 0;
		vol->spare_disks = 0;
		vol->failed_disks = 0;

		//LIST_FOR_EACH(vol->members, iter, member) {
		//	vol->sb_func->load_this_device_info(member);
		//}

		EngFncs->sort_list(vol->members, md_volume_sort_members, NULL);

		LIST_FOR_EACH(vol->members, iter, member) {
			if (md_member_is_raid_disk(member)) {
				LOG_DEBUG("  %s is a raid disk.\n", member->obj->name);
				vol->active_disks++;
				vol->working_disks++;
			} else {
				if (member->flags & MD_MEMBER_DISK_SPARE) {
					LOG_DEBUG("  %s is a spare disk.\n", member->obj->name);
					vol->spare_disks++;
					vol->working_disks++;
				} else {
					LOG_WARNING("%s is faulty.\n", member->obj->name);
					vol->failed_disks++;
				}
			}
		}
	}

	LOG_EXIT_VOID();
}



static boolean  md_namespace_registered = FALSE;

int md_register_name_space(void) {

	int rc = 0;

	LOG_ENTRY();

	if (!md_namespace_registered) {
		rc = EngFncs->register_name(MD_NAME_SPACE);

		if (rc == 0) {
			md_namespace_registered = TRUE;
		} else {
			LOG_SERIOUS("Error registering the MD name space \"%s\".\n", MD_NAME_SPACE);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static boolean md_final_call = FALSE;

void md_discover_final_call( list_anchor_t input_list, list_anchor_t output_list, int *count ) {

	LOG_ENTRY();

	if (!md_final_call) {
		md_discover_volumes(input_list, output_list);

		raid5_discover_regions(output_list, count, TRUE);
		LOG_DETAILS("RAID4/5 volume discovery complete.\n");

		EngFncs->delete_all_elements(input_list);
		EngFncs->merge_lists(input_list, output_list, NULL, NULL);

		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("PV discovery complete.\n");

		raid1_discover_regions(output_list, count, TRUE);
		LOG_DETAILS("RAID1 volume discovery complete.\n");

		EngFncs->delete_all_elements(input_list);
		EngFncs->merge_lists(input_list, output_list, NULL, NULL);

		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("PV discovery complete.\n");

		raid0_discover_regions(output_list, count, TRUE);
		LOG_DETAILS("RAID0 volume discovery complete.\n");

		EngFncs->delete_all_elements(input_list);
		EngFncs->merge_lists(input_list, output_list, NULL, NULL);

		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("PV discovery complete.\n");

		linear_discover_regions(output_list, count, TRUE);
		LOG_DETAILS("LINEAR volume discovery complete.\n");

		md_destroy_list_corrupt_messages();
		md_final_call = TRUE;

	} else {
		/*
		 * Final discovery has been done.  Just move the
		 * objects from the input list to the output list.
		 */
		EngFncs->concatenate_lists(output_list, input_list);
	}

	LOG_EXIT_VOID();
}

/*
 * md_analyze_volume
 *
 * Analyze an inactive MD region.
 */
void md_analyze_volume(md_volume_t *vol)
{
	md_member_t *member, *freshest;
	list_element_t iter;
	u_int64_t e1, e2;

	LOG_ENTRY();

	if (!vol->region) {
		LOG_ERROR("No region for %s.\n", vol->name);
		LOG_EXIT_VOID();
		return;
	}

	if (vol->flags & MD_NEW_REGION) {
		LOG_DEBUG("%s is a new region, skip analyzing superblock.\n", vol->name);
		goto out;
	}

	freshest = NULL;
	LIST_FOR_EACH(vol->members, iter, member) {
		if (!freshest) {
			freshest = member;
			continue;
		}
		e1 = vol->sb_func->get_event(freshest->sb);
		e2 = vol->sb_func->get_event(member->sb);
		if (e2 > e1) {
			freshest = member;
		}
	}

	vol->stale_disks = 0;
	e1 = vol->sb_func->get_event(freshest->sb);
	LIST_FOR_EACH(vol->members, iter, member) {
		e2 = vol->sb_func->get_event(member->sb);
		if (e2 < e1) {
			vol->stale_disks++;
			LOG_WARNING("Object %s is out of date.\n", member->obj->name);
			if (member->flags & MD_MEMBER_DISK_ACTIVE) {
				vol->active_disks--;
				vol->working_disks--;
			} else if (member->flags & MD_MEMBER_DISK_SPARE) {
				vol->spare_disks--;
				vol->working_disks--;
			} else {
				LOG_WARNING("Faulty disk %s that was not removed.\n",
					    member->obj->name);
			}
			member->flags = MD_MEMBER_STALE;
			member->raid_disk = -1;
		}
	}

	if (vol->stale_disks > 0) {
		LOG_WARNING("Found %d stale objects in region %s.\n",
			    vol->stale_disks, vol->name);

		//set master superblock
		EngFncs->engine_free(vol->sb);
		if (vol->sb_func->duplicate_sb(&vol->sb, freshest->sb)) {
			LOG_MD_BUG();
			vol->flags = MD_CORRUPT;
			goto out;
		}

		LIST_FOR_EACH(vol->members, iter, member) {
			if (!(member->flags & MD_MEMBER_STALE)) {
				if (member->sb) {
					EngFncs->engine_free(member->sb);
				}
				
				if (vol->sb_func->duplicate_sb(&member->sb, vol->sb)) {
					LOG_MD_BUG();
					vol->flags = MD_CORRUPT;
					goto out;
				}
				vol->sb_func->set_this_device_info(member);
			}
		}
	}

	vol->sb_func->analyze_sb(vol);	

out:
	if (vol->flags & MD_CORRUPT) {
		vol->region->flags |= SOFLAG_CORRUPT;
	}
	LOG_EXIT_VOID();
}

/*
 * md_analyze_active_region
 *
 * Analyze an active MD region.
 */
int md_analyze_active_region(md_volume_t *vol)
{
	int rc = 0;
	mdu_array_info_t md_info;
	md_super_info_t sb_info;
	md_member_t *member;
	mdu_disk_info_t d;
	storage_object_t *region;
	int i;
	boolean change;
	int change_count = 0;
	int length = 0;

	LOG_ENTRY();

	if (!vol) {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}

	region = vol->region;
	if (!region) {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}

	if (!(region->flags & SOFLAG_ACTIVE)) {
		goto out;
	}

	switch (vol->personality) {
	case RAID1:
	case RAID5:
		break;
	default:
		goto out;
	}

	if (md_is_recovery_running(region)) {
		LOG_DEFAULT("Recovery is running on %s.\n", region->name);
		goto out;
	}

	rc = md_ioctl_get_array_info(region, &md_info);
	if (rc) {
		LOG_MD_BUG();
		goto out;
	}

	for (i=0; i<md_info.nr_disks && !rc; i++) {
		d.number = i;
		rc = md_ioctl_get_disk_info(region, &d);
		if (rc) {
			LOG_WARNING("Can't get info for disk[%d].\n", i);
			rc = 0;
			continue;
		}
		member = md_volume_find_member_from_major_minor(vol, d.major, d.minor);
		if (member && (member->dev_number != d.number)) {
			LOG_WARNING("The kernel said disk(%d:%d) is at index[%d].\n",
				    d.major, d.minor, d.number);
			member->dev_number = d.number;
			EngFncs->sort_list(vol->members, md_volume_sort_members, NULL);
		}
	}

	md_volume_get_super_info(vol, &sb_info);
	if (md_info.raid_disks != sb_info.raid_disks) {
		if (vol->flags & MD_ARRAY_RESIZE_PENDING) {
			LOG_DEBUG("Resize operation is pending on region %s."
				  "raid_disks   : KERNEL(%03d) SuperBlock(%03d).\n",
				  vol->name, md_info.nr_disks, sb_info.nr_disks);
		} else {
			LOG_MD_BUG();
			LOG_WARNING("raid_disks   : KERNEL(%03d) SuperBlock(%03d).\n",
				    md_info.nr_disks, sb_info.nr_disks);
			rc = EINVAL;
			goto out;
		}
	}
	if (md_info.nr_disks != sb_info.nr_disks) {
		LOG_WARNING("nr_disks     : KERNEL(%03d) SuperBlock(%03d).\n",
			    md_info.nr_disks, sb_info.nr_disks);
		sb_info.nr_disks = md_info.nr_disks;
		change_count++;
	}
	if (md_info.active_disks != sb_info.active_disks) {
		LOG_WARNING("active_disks : KERNEL(%03d) SuperBlock(%03d).\n",
			    md_info.active_disks, sb_info.active_disks);
		sb_info.active_disks = md_info.active_disks;
		change_count++;
	}
	if (md_info.spare_disks != sb_info.spare_disks) {
		LOG_WARNING("spare_disks  : KERNEL(%03d) SuperBlock(%03d).\n",
			    md_info.spare_disks, sb_info.spare_disks);
		sb_info.spare_disks = md_info.spare_disks;
		change_count++;
	}
	if (md_info.failed_disks != sb_info.failed_disks) {
		LOG_WARNING("failed_disks : KERNEL(%03d) SuperBlock(%03d).\n",
			    md_info.failed_disks, sb_info.failed_disks);
		sb_info.failed_disks = md_info.failed_disks;
		change_count++;
	}

	if (md_info.active_disks < md_info.raid_disks) {
		switch (md_info.level) {
			case MD_LEVEL_MULTIPATH:
			case MD_LEVEL_RAID1:
				if (md_info.active_disks >= 1) {
					vol->flags |= MD_DEGRADED;
				} else {
					length = sprintf(message_buffer,
							 _("%s region %s is corrupt."
							   "  None of the disks are active to start the array."),
							 level_to_string(md_info.level), vol->name);
					vol->flags |= MD_CORRUPT;
					goto queue_corrupt_message;
				}
				break;
			case MD_LEVEL_RAID5:
				if ( (md_info.raid_disks - md_info.active_disks) == 1 ) {
					vol->flags |= MD_DEGRADED;
				} else {
					length = sprintf(message_buffer,
							 _("%s region %s is corrupt."
							   "  The number of raid disks for a full functional array is %d."
							   "  The number of active disks is %d."),
							 level_to_string(md_info.level), vol->name,
							 md_info.raid_disks, md_info.active_disks);
					vol->flags |= MD_CORRUPT;
					goto queue_corrupt_message;
				}
				break;
			default:
				length = sprintf(message_buffer,
						 _("%s region %s is corrupt."
						   "  The number of raid disks for a full functional array is %d."
						   "  The number of active disks is %d.\n"),
						 level_to_string(md_info.level), vol->name,
						 md_info.raid_disks, md_info.active_disks);
				vol->flags |= MD_CORRUPT;
				goto queue_corrupt_message;
		}
	}

	for (i=0; i<md_info.nr_disks && !rc; i++) {
		d.number = i;
		rc = md_ioctl_get_disk_info(region, &d);
		if (rc) {
			LOG_WARNING("Can't get info for disk[%d].\n", i);
			rc = 0;
			continue;
		}

		member = md_volume_find_member(vol, i);
		if (member) {

			change = FALSE;

			if ((d.state & (1<<MD_DISK_ACTIVE)) && (d.state & (1<<MD_DISK_SYNC))) {
				/* raid disk */
				if (!md_member_is_raid_disk(member)) {
					LOG_WARNING("Kernel said disk[%d] is active (raid_disk=%d)\n",
						    i, d.raid_disk);
					change = TRUE;
					member->raid_disk = d.raid_disk;
					member->flags = (MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC);
				} else {
					//Make sure raid_disk is correct
					if (d.raid_disk != member->raid_disk) {
						LOG_DEBUG("raid disk does not match,"
							  "KERNEL(%d) EVMS(%d).\n",
							  d.raid_disk, member->raid_disk);
						/* This is ok if resize operartion is pending */
						if (!(vol->flags & MD_ARRAY_RESIZE_PENDING)) {
							LOG_MD_BUG();
							rc = EINVAL;
						}

					}
				}
			} else if (d.state & (1<<MD_DISK_FAULTY)) {
				/* faulty disk */
				if (!(member->flags & MD_MEMBER_DISK_FAULTY)) {
					LOG_WARNING("Kernel said disk[%d] is faulty.\n", i);
					change = TRUE;
					member->raid_disk = -1;
					member->flags = MD_MEMBER_DISK_FAULTY;
				}
			} else if (d.state == 0) {
				/* spare disk */
				if (!(member->flags & MD_MEMBER_DISK_SPARE)) {
					LOG_WARNING("Kernel said disk[%d] is spare.\n", i);
					change = TRUE;
					member->raid_disk = -1;
					member->flags = MD_MEMBER_DISK_SPARE;
				}
			} else if (d.state & (1<<MD_DISK_REMOVED)) {
				if (!(member->flags & MD_MEMBER_STALE)) {
					LOG_WARNING("Kernel said disk[%d] was removed.\n", i);
					change = TRUE;
					member->raid_disk = -1;
					member->flags = MD_MEMBER_STALE;
				}
			}

			if (change == TRUE) {
				LOG_WARNING("Region %s, member %s: "
					    " Updating superblock to the info obtained from Kernel driver.\n",
					    vol->name, member->obj->name);
				vol->sb_func->set_this_device_info(member);  //update member's superblock
				vol->sb_func->set_this_device_state(member); //update master superblock
				change_count++;
			}

		} else {
			/* This disk is not in the MD volume. */
			if (d.state & (1<<MD_DISK_REMOVED)) {
				LOG_DEBUG("Disk[%d] was removed.\n", d.number);
			} else if (d.state & (1<<MD_DISK_FAULTY)) {
			       length = sprintf(message_buffer,
						_("%s region %s contains a failed disk with major %d and minor %d."
						  "  The Kernel MD driver still has a reference to this disk."
						  "  However, EVMS did not find MD superblock on this disk.\n\n"
						  "  Would you like to request the kernel to release the reference to this disk?\n"),
						level_to_string(md_info.level), vol->name,
						d.major, d.minor);
			
				if (md_get_yes_no_response(message_buffer) == TRUE) {
					md_ioctl_hot_remove_disk(region, makedev(d.major, d.minor));
				}

			} else {
				LOG_WARNING("Missing disk[%d] info:"
					    " raid_disk(%d), major(%d), minor(%d) state(0x%08X).\n",
					    d.number, d.raid_disk, d.major, d.minor, d.state);
				if (EngFncs->is_2_4_kernel() == FALSE) {
					LOG_MD_BUG();
					rc = EINVAL;
				}
			}
		}
	}

	vol->stale_disks = md_volume_count_stale_disks(vol);

queue_corrupt_message:
	if (vol->flags & MD_CORRUPT) {
		if (length > 0) {
			md_queue_corrupt_message(vol, message_buffer, length);
		}
		rc = EINVAL;
	}

	if (rc) {
		vol->flags |= MD_CORRUPT;
		vol->region->flags |= SOFLAG_CORRUPT;
	} else {
		if (change_count) {
			LOG_WARNING("Region %s: Updating superblock to the info obtained from Kernel driver.\n",
				    vol->name);
			vol->sb_func->set_sb_info(vol->sb, &sb_info);
		}
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}


