/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: mdregmgr.h
 */

#ifndef _MD_REGION_MANAGER_H_
#define _MD_REGION_MANAGER_H_ 1

#define MD_SAVED_INFO_BYTES		1024
#define MD_SAVED_INFO_WORDS		(MD_SAVED_INFO_BYTES / 4)
#define MD_SAVED_INFO_SECTS		(MD_SAVED_INFO_BYTES / 512)

#define MD_SB_VER_0	0
#define MD_SB_VER_1	1


typedef struct md_sb_ver_s {
	int major_version;
	int minor_version;
	int patchlevel;
} md_sb_ver_t;

struct super_func;

/* flags */
#define MD_SAVED_INFO_SIGNATURE                 0x5f6d645f /* "_md_" */
#define MD_SAVED_INFO_EXPAND_IN_PROGRESS	(1<<0)
#define MD_SAVED_INFO_SHRINK_IN_PROGRESS	(1<<1)

typedef struct md_saved_info_s {
	u_int32_t signature;		/*  MD_SAVED_INFO_SIGNATURE */
	u_int32_t flags;		/*  flags */
	u_int64_t sector_mark;		/*  to continue the SHRINK or unwind EXPAND */
	u_int32_t csum;			/*  checksum upto expand_shrink_devs[cnt] */
	u_int32_t expand_shrink_cnt;	/*  number of entries for expand_shrink_devs[] */
	u_int16_t expand_shrink_devs[0]; /* list of indexes of added/removed members */
}md_saved_info_t;


#define MD_LEVEL_MULTIPATH    -4
#define MD_LEVEL_LINEAR       -1
#define MD_LEVEL_RAID0         0
#define MD_LEVEL_RAID1         1
#define MD_LEVEL_RAID4         4
#define MD_LEVEL_RAID5         5

#define MD_RESERVED       0UL
#define LINEAR            1UL
#define RAID0             2UL
#define RAID1             3UL
#define RAID5             4UL
#define TRANSLUCENT       5UL
#define HSM               6UL
#define MULTIPATH         7UL
#define MAX_PERSONALITY   8UL

#define INVALID_LEVEL    99UL

static inline int pers_to_level (int pers) {
	switch (pers) {
	case MULTIPATH:     return MD_LEVEL_MULTIPATH;
	case HSM:           return -3;
	case TRANSLUCENT:   return -2;
	case LINEAR:        return MD_LEVEL_LINEAR;
	case RAID0:         return MD_LEVEL_RAID0;
	case RAID1:         return MD_LEVEL_RAID1;
	case RAID5:         return MD_LEVEL_RAID5;
	}
	return INVALID_LEVEL;
}

static inline int level_to_pers (int level) {
	switch (level) {
	case MD_LEVEL_MULTIPATH: return MULTIPATH;
	case -3: return HSM;
	case -2: return TRANSLUCENT;
	case MD_LEVEL_LINEAR: return LINEAR;
	case MD_LEVEL_RAID0:  return RAID0;
	case MD_LEVEL_RAID1:  return RAID1;
	case MD_LEVEL_RAID4:
	case MD_LEVEL_RAID5:  return RAID5;
	}
	return MD_RESERVED;
}

static inline char * level_to_string (int level) {
	switch (level) {
	case MD_LEVEL_MULTIPATH: return "MULTIPATH";
	case -3: return "HSM";
	case -2: return "TRANSLUCENT";
	case MD_LEVEL_LINEAR: return "LINEAR";
	case MD_LEVEL_RAID0:  return "RAID0";
	case MD_LEVEL_RAID1:  return "RAID1";
	case MD_LEVEL_RAID4:
	case MD_LEVEL_RAID5:  return "RAID5";
	}
	return "INVALID LEVEL";
}

#define MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, size_in_sectors) \
	((chunk_size_in_sectors) ? (MD_NEW_SIZE_SECTORS(size_in_sectors) & ~((chunk_size_in_sectors) - 1)) : MD_NEW_SIZE_SECTORS(size_in_sectors))
#define MD_CHUNK_ALIGN_NEW_SIZE_BLOCKS(chunk_size_in_blocks, size_in_blocks) \
	((chunk_size_in_blocks) ? (MD_NEW_SIZE_BLOCKS(size_in_blocks) & ~((chunk_size_in_blocks) - 1)) : MD_NEW_SIZE_BLOCKS(size_in_blocks))

#define VSECTORS_TO_BLOCKS(sectors) ((sector_count_t) sectors / (BLOCK_SIZE >> EVMS_VSECTOR_SIZE_SHIFT))
#define BLOCKS_TO_VSECTORS(blocks)  ((sector_count_t) blocks  * (BLOCK_SIZE >> EVMS_VSECTOR_SIZE_SHIFT))

/* ioctls */
#define MD_MAJOR 9

/* status */
#define RAID_VERSION		_IOR (MD_MAJOR, 0x10, mdu_version_t)
#define GET_ARRAY_INFO		_IOR (MD_MAJOR, 0x11, mdu_array_info_t)
#define GET_DISK_INFO		_IOR (MD_MAJOR, 0x12, mdu_disk_info_t)
#define PRINT_RAID_DEBUG	_IO (MD_MAJOR, 0x13)
#define RAID_AUTORUN		_IO (MD_MAJOR, 0x14)

/* configuration */
#define CLEAR_ARRAY		_IO (MD_MAJOR, 0x20)
#define ADD_NEW_DISK		_IOW (MD_MAJOR, 0x21, mdu_disk_info_t)
#define HOT_REMOVE_DISK		_IO (MD_MAJOR, 0x22)
#define SET_ARRAY_INFO		_IOW (MD_MAJOR, 0x23, mdu_array_info_t)
#define SET_DISK_INFO		_IO (MD_MAJOR, 0x24)
#define WRITE_RAID_INFO		_IO (MD_MAJOR, 0x25)
#define UNPROTECT_ARRAY		_IO (MD_MAJOR, 0x26)
#define PROTECT_ARRAY		_IO (MD_MAJOR, 0x27)
#define HOT_ADD_DISK		_IO (MD_MAJOR, 0x28)
#define SET_DISK_FAULTY		_IO (MD_MAJOR, 0x29)
#define HOT_GENERATE_ERROR	_IO (MD_MAJOR, 0x2a)

/* usage */
#define RUN_ARRAY		_IOW (MD_MAJOR, 0x30, mdu_param_t)
#define START_ARRAY		_IO (MD_MAJOR, 0x31)
#define STOP_ARRAY		_IO (MD_MAJOR, 0x32)
#define STOP_ARRAY_RO		_IO (MD_MAJOR, 0x33)
#define RESTART_ARRAY_RW	_IO (MD_MAJOR, 0x34)

typedef struct mdu_version_s {
	int major;
	int minor;
	int patchlevel;
} mdu_version_t;


#define MD_ARRAY_CLEAN			0
#define MD_ARRAY_ERRORS			1

typedef struct mdu_array_info_s {
	/*
	 * Generic constant information
	 */
	int major_version;
	int minor_version;
	int patch_version;
	int ctime;
	int level;
	int size;
	int nr_disks;
	int raid_disks;
	int md_minor;
	int not_persistent;

	/*
	 * Generic state information
	 */
	int utime;		/*  0 Superblock update time		      */
	int state;		/*  1 State bits (clean, ...)		      */
	int active_disks;	/*  2 Number of currently active disks	      */
	int working_disks;	/*  3 Number of working disks		      */
	int failed_disks;	/*  4 Number of failed disks		      */
	int spare_disks;	/*  5 Number of spare disks		      */

	/*
	 * Personality information
	 */
	int layout;		/*  0 the array's physical layout	      */
	int chunk_size;	/*  1 chunk size in bytes		      */

} mdu_array_info_t;

typedef struct mdu_disk_info_s {
	/*
	 * configuration/status of one particular disk
	 */
	int number;
	int major;
	int minor;
	int raid_disk;
	int state;

} mdu_disk_info_t;

typedef struct mdu_start_info_s {
	/*
	 * configuration/status of one particular disk
	 */
	int major;
	int minor;
	int raid_disk;
	int state;

} mdu_start_info_t;

typedef struct mdu_param_s
{
	int			personality;	/* 1,2,3,4 */
	int			chunk_size;	/* in bytes */
	int			max_fault;	/* unused for now */
} mdu_param_t;


// md_volume_t flags
#define MD_NEW_REGION		(1 << 0)
#define MD_DISCOVERED		(1 << 1)
#define MD_DEGRADED		(1 << 2)
#define MD_CORRUPT		(1 << 3)
#define MD_DIRTY		(1 << 4)
#define MD_ACTIVE		(1 << 5)  // This md volume is active in the kernel
#define MD_ARRAY_SYNCING	(1 << 6)  // Resync/recovery is running
#define MD_MP_ACTIVATE_REGION	(1 << 7)  // Multipath region needs to be activated.
#define MD_MP_DELETE_VOLUME	(1 << 8)  // Delete the private-data after the region is deactivated.
#define MD_NEEDS_REDISCOVER	(1 << 9)  // Set this flag to signal a re-discovery
#define MD_NEEDS_VALIDATE	(1 << 10) // Validation of md array is required
#define MD_NEEDS_UPDATE_SIZE	(1 << 11) // Set this flag to signal a size recalculation of md array
#define MD_ARRAY_RESIZE_PENDING	(1 << 12) // Array has been resized

// commit_flags
#define MD_COMMIT_DONT_CHECK_ACTIVE (1 << 0)	// Don't check if the region is active before writing
						// the superblock.
#define MD_COMMIT_BACKUP_METADATA   (1 << 1)

typedef struct md_volume_s {
	storage_object_t * region;	
	list_anchor_t      members;               // List of all members (ie. disks)
	md_sb_ver_t        sb_ver;                // Version of MD superblock
	void               *sb;                   // Master superblock
	struct super_func  *sb_func;              // Version specific superblock handler
	u_int32_t          flags;                 // MD volume flags
	char               name[EVMS_NAME_SIZE+1]; // name in this form "md/mdX"
	u_int32_t          md_minor;              // The minor number is X part of the volume name "md/mdX"
	u_int32_t          commit_flag;           // if set, use SB state from disk, else use from Master SB
	u_int32_t          uuid;
	u_int32_t          nr_disks;
	u_int32_t          raid_disks;
	u_int32_t          personality;           // uses the positive defines, not level
	u_int32_t          chunksize;             // in sectors
	/* analyze code uses the following fields */
	u_int32_t          active_disks;
	u_int32_t          working_disks;
	u_int32_t          spare_disks;
	u_int32_t          failed_disks;
	u_int32_t          stale_disks;
	/* commands to perform during SETUP phase of commit process */
	list_anchor_t	   setup_funcs;
	/* ioctl packages to perform during POST_ACTIVATE phase of commit process */
	list_anchor_t      ioctl_pkgs;
	list_anchor_t      ioctl_cleanup;
	u_int32_t          region_mgr_flags;      // Personality specific flag
	u_int32_t          daemon_pid;            // Multipath uses this field
	void             * private_data;          // Personality private data
	struct md_volume_s * next;
} md_volume_t;


#define MD_MEMBER_NEW            (1 << 0)
#define MD_MEMBER_STALE          (1 << 1)
#define MD_MEMBER_DISK_ACTIVE    (1 << 2)
#define MD_MEMBER_DISK_SYNC      (1 << 3)
#define MD_MEMBER_DISK_FAULTY    (1 << 4)
#define MD_MEMBER_DISK_REMOVED   (1 << 5)
#define MD_MEMBER_DISK_SPARE     (1 << 6)
#define MD_MEMBER_DISK_PENDING   (1 << 7)

typedef struct md_member_s {
	storage_object_t       *obj;             // EVMS object
	md_volume_t            *vol;             // MD volume which this object is a member of
	void                   *sb;              // MD superblock
	md_saved_info_t        *saved_info;      // EVMS saved area in MD superblock reserved space
	u_int32_t               flags;           // Flags (see above)
	int                     raid_disk;       // disk role or (-1 for not a raid disk)
	u_int64_t               data_offset;	 // sector start of data, often 0
	u_int64_t               data_size;	 // sectors in this device that can be used for data
	u_int64_t               super_offset;	 // sector start of this superblock
	u_int64_t               recovery_offset; // sectors before this offset (from data_offset) have been recovered
	u_int32_t               dev_number;	 // permanent identifier of this  device - not role in raid
	u_int32_t               cnt_corrected_read; // number of read errors that were corrected by re-writing
	u_int8_t                device_uuid[16]; // user-space setable, ignored by kernel

} md_member_t;

typedef struct md_array_info_s {
	u_int32_t       raid_disks;
	u_int32_t       personality;           // uses the positive defines, not level
	u_int32_t       chunksize;             // in sectors
	u_int32_t       active_disks;
	u_int32_t       working_disks;
	u_int32_t       spare_disks;
	u_int32_t       failed_disks;
	u_int32_t       nr_disks;              // size of disk[] array
	mdu_disk_info_t disk[0];               // variable size disk array
} md_array_info_t;

// Global data used by mdregmgr
extern engine_functions_t   * EngFncs;
extern plugin_record_t      * linear_plugin;
extern plugin_record_t        linear_plugin_record;
extern plugin_record_t      * raid1_plugin;
extern plugin_record_t        raid1_plugin_record;
extern plugin_record_t      * raid0_plugin;
extern plugin_record_t        raid0_plugin_record;
extern plugin_record_t      * raid5_plugin;
extern plugin_record_t        raid5_plugin_record;
extern plugin_record_t      * mp_plugin;
extern plugin_record_t        multipath_plugin_record;
extern plugin_record_t      * my_plugin;

extern md_volume_t          * volume_list_head;	// List of real volume groups, indexed by vg_number.


#define MD_MESSAGE_BUF_SIZE 4096*4
extern char message_buffer[MD_MESSAGE_BUF_SIZE];

int md_can_replace_child(storage_object_t *region,
			 storage_object_t *child,
			 storage_object_t *new_child);
int md_replace_child(storage_object_t *region,
		     storage_object_t *child,
		     storage_object_t *new_child);
int md_can_deactivate_region(storage_object_t * region);
int md_deactivate_region(storage_object_t * region);
int md_can_activate_region(storage_object_t * region);
int md_activate_region(storage_object_t * region);
int md_get_kernel_info(storage_object_t * region, mdu_array_info_t * info);
boolean md_is_region_active(storage_object_t *region);
boolean mdstat_check_recovery(int mddev_minor, char *status);
boolean md_is_recovery_running(storage_object_t *region);

boolean md_can_stop_array(storage_object_t *region);

md_member_t * md_volume_find_object(md_volume_t *vol, storage_object_t *obj);
int find_disk_in_active_region(storage_object_t *region, int major, int minor);
md_member_t *md_find_member(int major, int minor);
md_member_t * md_volume_find_member(md_volume_t *vol, int dev_number);
md_member_t * md_volume_find_member_from_major_minor(md_volume_t *vol, int major, int minor);

//int find_empty_slot(mdp_super_t *sb);
int md_fix_dev_major_minor(md_volume_t * vol, boolean do_msg);
boolean follow_up_mark_faulty(md_volume_t *volume, storage_object_t *faulty);
boolean md_get_yes_no_response(char *question);

md_volume_t * md_allocate_volume(void);
void md_free_volume(md_volume_t *vol);
int md_delete_volume(md_volume_t * volume, boolean tear_down);
md_volume_t * md_clone_volume(md_volume_t *vol);
boolean md_volume_is_minor_taken(int md_minor);


storage_object_t * md_find_valid_input_object( char * name );
int md_volume_get_available_name(md_volume_t *vol, int limit);
int md_volume_get_alternative_name(md_volume_t *vol, int limit);
int md_volume_sort_members(void *thing1, void *thing2, void *user_data);

int md_volume_count_children(md_volume_t *vol);
int md_volume_count_active_disks(md_volume_t *vol);
int md_volume_count_faulty_disks(md_volume_t *vol);
int md_volume_count_spare_disks(md_volume_t *vol);
int md_volume_count_stale_disks(md_volume_t *vol);
u_int64_t md_volume_smallest_data_size(md_volume_t *vol);

void md_volume_add_member(md_volume_t *vol, md_member_t *member);
int  md_volume_remove_member(md_member_t *member, boolean resize);

md_member_t *md_allocate_member(storage_object_t *obj);
void md_free_member(md_member_t *member);
static inline void md_member_set_data_offset(md_member_t *member, u_int64_t offset) {
	member->data_offset = offset;
}
static inline void md_member_set_data_size(md_member_t *member, u_int64_t size) {
	member->data_size = size;
}
static inline void md_member_set_super_offset(md_member_t *member, u_int64_t offset) {
	member->super_offset = offset;
}


#define LOG_MD_BUG() LOG_CRITICAL(" MD INTERNAL ERROR from %s, in %s function, at line %d\n", \
	__FILE__, __FUNCTION__, __LINE__)

static inline boolean md_member_is_raid_disk(md_member_t *member) {
	return (member->raid_disk == -1) ? FALSE : TRUE;
}

static inline boolean md_volume_is_spare_disk(md_member_t *member) {
	return (member->flags & MD_MEMBER_DISK_SPARE) ? TRUE : FALSE;
}

int md_volume_add_spare_to_active_region(md_volume_t * vol, storage_object_t * spare);
int md_volume_add_spare_to_inactive_region(md_volume_t * vol, storage_object_t * spare);
int md_volume_remove_spare_from_active_region(md_volume_t *vol, storage_object_t * spare);
int md_volume_remove_spare_from_inactive_region(md_volume_t *vol, storage_object_t *spare);
int md_volume_remove_faulty_from_active_region(md_volume_t *vol, storage_object_t *faulty);
int md_volume_remove_faulty_from_inactive_region(md_volume_t *vol, storage_object_t *faulty);
int md_volume_remove_stale_from_active_region(md_volume_t *vol, storage_object_t *stale);
int md_volume_remove_stale_from_inactive_region(md_volume_t *vol, storage_object_t *stale);
int md_volume_activate_spare(md_member_t *member);
int md_volume_mark_faulty(md_member_t *member, boolean mark_removed);

boolean load_kernel_md_driver(void);
boolean md_can_create_sb_1(void);


int linear_discover_regions(list_anchor_t output_list, int * count, boolean final_call);
int raid0_discover_regions(list_anchor_t output_list, int * count, boolean final_call);
int raid1_discover_regions(list_anchor_t output_list, int * count, boolean final_call);
int raid5_discover_regions(list_anchor_t output_list, int * count, boolean final_call);

#include "md_info.h"
#include "md_super.h"
#include "md_discover.h"
#include "md_dlist.h"
#include "md_io.h"

#endif

