/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: widget.c
 */

#include <frontend.h>
#include <gtk/gtk.h>
#include <string.h>
#include <math.h>

#include "support.h"
#include "value.h"
#include "widget.h"
#include "task.h"
#include "thing.h"
#include "readable.h"
#include "logging.h"

/* Forward function declaration */
void set_option_radio_button_active_from_boolean(GtkToggleButton * yes_button,
						 GtkToggleButton * no_button, gboolean is_yes,
						 option_widget_info_t * info);

/*
 *
 *   void display_set_option_value_error (gint, gchar *, GtkWidget *)
 *
 *   Description:
 *      This routine displays a results window identifying problems setting
 *      an option value.
 *
 *   Entry:
 *      rc           - the return code from evms_option_set_value()
 *      option_title - the field title for the option that could not be set
 *      widget       - the option widget
 *
 *   Exit:
 *      A results window is displayed communicating the problem to the user.
 *
 */
void display_set_option_value_error(gint rc, gchar * option_title, GtkWidget * widget)
{
	gchar *error_msg;

	error_msg =
	    g_strdup_printf(_
			    ("An error occurred communicating the value for the \"%s\" field to the plugin."),
			    option_title);

	display_results_window(rc, NULL, error_msg, NULL, FALSE, NULL);

	g_free(error_msg);
}

/*
 *
 *   void set_widget_option_value (option_widget_info_t *, value_t)
 *
 *   Description:
 *      This routine gets called by most of the option callback routines
 *      after they have retrieved and converted the input data into a
 *      value_t. We essentially retrieve the task handle from the top
 *      level widget and attempt to set the new value. If the set is
 *      accepted we clear the old value in the widget_info and copy
 *      in the new value and clear the given value.
 *
 *   Entry:
 *      info               - everything we need to know about widget and option
 *      value              - address of new value (in proper form and cleared
 *                           on exit)
 *      was_value_modified - address to write TRUE if value was fudged
 *
 *   Exit:
 *      The option value is updated and callout routine invoked.
 *      We return TRUE if given value was modified by backend so
 *      caller can update widget in their respective way. This
 *      function returns the error code of the set API so that
 *      the signal emission can be stopped if an error occurred.
 *
 */
gint set_widget_option_value(option_widget_info_t * info, value_t * value,
			     gboolean * was_value_modified)
{
	gint rc;
	task_handle_t task;
	task_effect_t effect = 0;
	options_callback_callout callout;

	*was_value_modified = FALSE;

	task = retrieve_task_handle_from_toplevel_widget(info->widget);
	callout = retrieve_options_callout_from_toplevel_widget(info->widget);

	rc = evms_set_option_value_by_name(task, info->option->name, value, &effect);

	log_debug("%s: Called evms_set_option_value_by_name() for option %s. rc == %d.\n",
		  __FUNCTION__, info->option->name, rc);

	if (rc != SUCCESS) {
		display_set_option_value_error(rc, info->option->title, info->widget);

		/*
		 * If this is an entry field then indicate we have no value in order to
		 * have the user enter a different one. Setting the has_value flag to
		 * FALSE and then calling the options callout will also de-sensitize the
		 * operation completion button so we can't complete the operation if this
		 * had been the last required field that got input.
		 */

		if (GTK_IS_ENTRY(info->widget))
			info->has_value = FALSE;

		if (callout != NULL)
			(*callout) (info, *value, 0);
	} else {
		guint list_count = 0;
		gboolean is_list;

		info->has_value = TRUE;

		log_debug("%s: evms_set_option_value_by_name() side-effect was %d\n", __FUNCTION__,
			  effect);

		if (callout != NULL)
			(*callout) (info, *value, effect);

		*was_value_modified = effect & EVMS_Effect_Inexact;

		is_list = EVMS_OPTION_VALUE_IS_LIST(info->option->flags);

		if (is_list)
			list_count = info->option->constraint.list->count;

		/*
		 * Keep info->value up-to-date on a successful set operation.
		 */
		clear_value(&(info->value), info->option->type, is_list, list_count);
		duplicate_value(*value, info->option->type, is_list, list_count,
				info->option->max_len, &(info->value));
	}

	return rc;
}

/*
 *
 *   inline void convert_string_to_option_value (gchar *, option_widget_info_t *, value_t *)
 *
 *   Description:
 *      This routine takes an input string and converts it to a value_t. It
 *      makes any unit conversion necessary before coming up with the final
 *      value.
 *
 *   Entry:
 *      text  - the value in string form
 *      info  - address of the option widget info associated with widget
 *      value - the value in its natural form
 *
 *   Exit:
 *      Returns the string as an option value in the the correct unit of measurement
 *
 */
inline void convert_string_to_option_value(gchar * text, option_widget_info_t * info,
					   value_t * value)
{
	if ((info->option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION) == 0 &&
	    info->option->unit == EVMS_Unit_Sectors)
		convert_size_string_to_sector_value(text, info->unit_conversion, info->option->type,
						    value);
	else
		convert_string_to_value(text, info->option->type, info->option->max_len, value);

	if (info->option->constraint_type == EVMS_Collection_Range)
		*value = clamp_value(*value, info->option->constraint.range, info->option->type);
}

/*
 *
 *   void on_option_entry_focus_out (GtkEditable *, GdkEventFocus *event, option_widget_info_t *)
 *
 *   Description:
 *      This routine gets called when an option entry has lost its
 *      focus. If the widget had any updates then we generate the
 *      "activate" signal so that it can provide the current text
 *      value to the plugin, 
 *
 *   Entry:
 *      editable - the id of the GtkEntry widget that lost focus
 *      event    - focus event information
 *      info     - address of the option widget info associated with widget
 *
 *   Exit:
 *      We compare entry value and current value to see if they match.
 *      If not, we issue an "activate" signal in order to have the
 *      value updated in the plugin's eyes.
 *
 */
void on_option_entry_focus_out(GtkEditable * editable, GdkEventFocus * event,
			       option_widget_info_t * info)
{
	gchar *text;
	value_t value;
	gboolean value_changed;

	text = g_strstrip(gtk_editable_get_chars(editable, 0, -1));

	if (!(info->option->type == EVMS_Type_String && (strlen(text) < info->option->min_len))) {
		convert_string_to_option_value(text, info, &value);

		value_changed = !values_are_equal(value, info->value, info->option->type);

		if (value_changed)
			gtk_signal_emit_by_name(GTK_OBJECT(editable), "activate", editable, info);

		clear_value(&value, info->option->type, FALSE, 0);
	}

	g_free(text);
}

/*
 *
 *   void on_option_check_button_toggled (GtkToggleButton *, option_widget_info_t *)
 *
 *   Description:
 *      This routine gets called when an option checkbutton is toggled.
 *      We basically check the state of the button, convert the state
 *      to the boolean value, retrieve the task handle and call 
 *      evms_set_option_value() to update the current option value for
 *      the task.
 *
 *   Entry:
 *      button - the id of the toggle button that was pressed
 *      info   - address of the option widget info associated with widget
 *
 *   Exit:
 *      The option value is updated and if necessary we notify our top
 *      level window of sideaffects of the value update.
 *
 */
void on_option_check_button_toggled(GtkToggleButton * button, option_widget_info_t * info)
{
	gint rc;
	value_t value;
	gboolean value_changed;

	log_entry;

	value.b = gtk_toggle_button_get_active(button);

	rc = set_widget_option_value(info, &value, &value_changed);

	if ((rc == SUCCESS && value_changed) || (rc != SUCCESS && info->has_value)) {
		gtk_signal_handler_block_by_func(GTK_OBJECT(button), on_option_check_button_toggled,
						 info);
		gtk_toggle_button_set_active(button, info->value.b);
		gtk_signal_handler_unblock_by_func(GTK_OBJECT(button),
						   on_option_check_button_toggled, info);
	}

	log_exit;
}

/*
 *
 *   void on_option_radio_button_toggled (GtkToggleButton *, option_widget_info_t *)
 *
 *   Description:
 *      This routine gets called when an option radio button is toggled.
 *      This routine deals with boolean radio buttons only. We test to
 *      see which button we are (the Yes or the No button) and then
 *      determine whether we are active and if the boolean value should
 *      be changed.
 *
 *   Entry:
 *      button - the id of the toggle button that was pressed
 *      info   - address of the option widget info associated with widget
 *
 *   Exit:
 *      The option value is updated if necessary and we notify our top
 *      level window of sideaffects of the value update.
 *
 */
void on_option_radio_button_toggled(GtkToggleButton * button, option_widget_info_t * info)
{
	log_entry;

	if (gtk_toggle_button_get_active(button)) {
		gint rc;
		value_t value;
		gboolean value_changed;
		GtkToggleButton *no_button;
		GtkToggleButton *yes_button;

		no_button = gtk_object_get_data(GTK_OBJECT(button), "no_button");
		yes_button = gtk_object_get_data(GTK_OBJECT(button), "yes_button");

		value.b = button == yes_button;

		rc = set_widget_option_value(info, &value, &value_changed);

		if ((rc == SUCCESS && value_changed) || (rc != SUCCESS && info->has_value))
			set_option_radio_button_active_from_boolean(yes_button, no_button,
								    info->value.b, info);
	}

	log_exit;
}

/*
 *
 *   void set_option_radio_button_active_from_boolean (GtkToggleButton *, GtkToggleButton *,
 *                                                     gboolean, option_widget_info_t *)
 *
 *   Description:
 *      This routine sets either the yes or no radio button active depending
 *      on the value for the is_yes boolean.
 *
 *   Entry:
 *      yes_button - the id of the Yes radio button
 *      no_button  - the id of the No radio button
 *      is_yes     - TRUE if the Yes button should active, FALSE for No button
 *      info       - address of the option widget info associated with widget
 *
 *   Exit:
 *      Either the Yes or No button is made the active button.
 *
 */
void set_option_radio_button_active_from_boolean(GtkToggleButton * yes_button,
						 GtkToggleButton * no_button, gboolean is_yes,
						 option_widget_info_t * info)
{
	GtkToggleButton *active_button;

	if (is_yes)
		active_button = yes_button;
	else
		active_button = no_button;

	gtk_signal_handler_block_by_func(GTK_OBJECT(active_button), on_option_radio_button_toggled,
					 info);
	gtk_toggle_button_set_active(active_button, TRUE);
	gtk_signal_handler_unblock_by_func(GTK_OBJECT(active_button),
					   on_option_radio_button_toggled, info);
}

/*
 *
 *   void set_option_menu_history (GtkMenuShell *, option_widget_info_t *, value_t)
 *
 *   Description:
 *      This routine sets the selected item in an GtkOptionMenu to the
 *      menu item corresponding to the value provided.
 *
 *   Entry:
 *      menu  - the GtkMenu containing the menu items (labels)
 *      info  - address of the option widget info associated with widget
 *      value - the value that corresponds to the menu item to select
 *
 *   Exit:
 *      The menu item corresponding to the value provided is made the
 *      selected item.
 *
 */
void set_option_menu_history(GtkMenuShell * menu, option_widget_info_t * info, value_t value)
{
	gint count;
	GList *children;

	children = gtk_container_children(GTK_CONTAINER(menu));

	count = g_list_length(children);

	if (count > 0) {
		gint i;
		gint index;
		gchar *text;
		value_t menu_value;
		GtkMenuItem *menuitem;

		for (i = 0, index = 0; i < count; i++) {
			menuitem = g_list_nth_data(children, i);

			if (GTK_IS_MENU_ITEM(menuitem)) {
				if (GTK_IS_LABEL(GTK_BIN(menuitem)->child)) {
					gtk_label_get(GTK_LABEL(GTK_BIN(menuitem)->child), &text);

					convert_string_to_option_value(text, info, &menu_value);

					if (values_are_equal(value, menu_value, info->option->type)) {
						/*
						 * Save the index of the menu item that matches and
						 * set the loop index to count so we can terminate
						 * the loop.
						 */
						index = i;
						i = count;
					}

					clear_value(&menu_value, info->option->type, FALSE, 0);
				}
			}
		}

		gtk_option_menu_set_history(gtk_object_get_data(GTK_OBJECT(menu), "option_menu"),
					    index);
	}
}

/*
 *
 *   void on_option_menu_item_selected (GtkMenuShell *, option_widget_info_t *)
 *
 *   Description:
 *      This routine gets called when an option menu item has been activated
 *      and the selection is complete. We convert the label text of the item
 *      selected to the corresponding value and update it.
 *
 *   Entry:
 *      menu - the menu on which the item was selected
 *      info - address of the option widget info associated with widget
 *
 *   Exit:
 *      The option value is updated and if necessary we notify our top
 *      level window of sideaffects of the value update.
 *
 */
void on_option_menu_item_selected(GtkMenuShell * menu, option_widget_info_t * info)
{
	log_entry;

	if (GTK_BIN(info->widget)->child) {
		GtkWidget *child = GTK_BIN(info->widget)->child;

		if (GTK_IS_LABEL(child)) {
			gchar *text;
			value_t value;

			gtk_label_get(GTK_LABEL(child), &text);

			if (strcasecmp(text, OPTION_MENU_NO_SELECTION_TEXT) == 0) {
				options_callback_callout callout;

				/*
				 * If the special menu item is selected then we take
				 * this to mean that nothing is selected so we turn
				 * the has_value flag off and invoke the callout 
				 * routine to hopefully check to see if this was
				 * required in order to make the "Next" button insensitive.
				 */

				info->has_value = FALSE;

				clear_value(&(info->value), info->option->type, FALSE, 0);

				callout =
				    retrieve_options_callout_from_toplevel_widget(info->widget);

				if (callout != NULL)
					(*callout) (info, info->value, 0);
			} else {
				gint rc;
				gboolean value_changed;

				convert_string_to_option_value(text, info, &value);

				rc = set_widget_option_value(info, &value, &value_changed);

				if (rc == SUCCESS) {
					GtkWidget *no_selection_menuitem;

					/*
					 * Check to see if we need to walk through the menu
					 * items and set the history to the one the plugin chose.
					 */

					if (value_changed) {
						gtk_signal_handler_block_by_func(GTK_OBJECT(menu),
										 on_option_menu_item_selected,
										 info);

						set_option_menu_history(menu, info, value);

						gtk_signal_handler_unblock_by_func(GTK_OBJECT(menu),
										   on_option_menu_item_selected,
										   info);
					}

					/*
					 * If we have made a successful selection and we had the special no-selection
					 * menu item on the menu then this meant that the option had been a required
					 * option with no initial value. Now we have a value so we should remove our
					 * visual cue menu item.
					 */

					no_selection_menuitem =
					    gtk_object_get_data(GTK_OBJECT(menu),
								"no_selection_menuitem");

					if (no_selection_menuitem != NULL) {
						gtk_container_remove(GTK_CONTAINER(menu),
								     no_selection_menuitem);
						gtk_object_set_data(GTK_OBJECT(menu),
								    "no_selection_menuitem", NULL);
					}
				} else if (info->has_value) {
					/*
					 * Revert value to original value on error.
					 */
					gtk_signal_handler_block_by_func(GTK_OBJECT(menu),
									 on_option_menu_item_selected,
									 info);

					set_option_menu_history(menu, info, info->value);

					gtk_signal_handler_unblock_by_func(GTK_OBJECT(menu),
									   on_option_menu_item_selected,
									   info);
				}

				clear_value(&value, info->option->type, FALSE, 0);
			}
		}
	}

	log_exit;
}

/*
 *
 *   void static inline update_option_entry_text (GtkEntry *, value_t, option_widget_info_t *)
 *
 *   Description:
 *      This routine basically takes a value_t and converts it to a string
 *      and updates the entry box text.
 *
 *   Entry:
 *      editable - the entry box to contain the next text
 *      info     - address of the option widget info associated with widget
 *
 *   Exit:
 *      The entry text is updated.
 *
 */
void static inline update_option_entry_text(GtkEntry * entry, value_t value,
					    option_widget_info_t * info)
{
	gchar *text;

	if ((info->option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION) == 0 &&
	    info->option->unit == EVMS_Unit_Sectors) {
		if (info->option->constraint_type == EVMS_Collection_None)
			text =
			    make_sectors_readable_string(convert_value_to_ui64
							 (value, info->option->type));
		else
			text =
			    convert_sector_value_to_new_unit_string(value, info->option->type,
								    info->unit_conversion);
	} else
		text = convert_value_to_string(value, info->option->type, EVMS_Format_Normal);

	gtk_entry_set_text(entry, text == NULL ? "" : text);

	g_free(text);
}

/*
 *
 *   void on_option_entry_text_entered (GtkEditable *, option_widget_info_t *)
 *
 *   Description:
 *      This routine gets called when the Enter keystroke is pressed 
 *      in an option entry box.
 *
 *   Entry:
 *      editable - the entry box containing the text
 *      info     - address of the option widget info associated with widget
 *
 *   Exit:
 *      The option value is updated and if necessary we notify our toplevel
 *      window of any update sideaffects.
 *
 */
void on_option_entry_text_entered(GtkEditable * editable, option_widget_info_t * info)
{
	gint rc;
	gchar *entry_text;
	value_t value;
	gboolean value_changed;

	log_entry;

	entry_text = g_strstrip(gtk_editable_get_chars(editable, 0, -1));

	if (!
	    (info->option->type == EVMS_Type_String
	     && (strlen(entry_text) < info->option->min_len))) {
		convert_string_to_option_value(entry_text, info, &value);

		rc = set_widget_option_value(info, &value, &value_changed);

		/*
		 * Always update the entry field to allow for reverting values on error or for 
		 * showing any adjusted or converted value.
		 */

		gtk_signal_handler_block_by_func(GTK_OBJECT(editable), on_option_entry_text_entered,
						 info);

		if (info->has_value)
			update_option_entry_text(GTK_ENTRY(editable), info->value, info);
		else
			gtk_entry_set_text(GTK_ENTRY(editable), "");

		gtk_signal_handler_unblock_by_func(GTK_OBJECT(editable),
						   on_option_entry_text_entered, info);

		clear_value(&value, info->option->type, FALSE, 0);
	}

	g_free(entry_text);

	log_exit;
}

/*
 *
 *   void on_option_entry_text_inserted (GtkEditable *, const gchar *, gint, gint *,option_descriptor_t *)
 *
 *   Description:
 *      This routine gets called when text is inserted in an option 
 *      entry box. This callback routine does not call 
 *      evms_set_option_value() like the other callbacks. Instead, 
 *      it does simple validation on the current value based upon 
 *      known constraints such as maximum string size for a string
 *      and maximum value for a numeric type.
 *
 *   Entry:
 *      editable        - the entry box that had characters inserted
 *      new_text        - string with new text to be inserted
 *      new_text_length - length of new text to insert
 *      position        - the position at which to insert the new text
 *      info            - address of the option widget info associated with widget
 *
 *   Exit:
 *      Current value is validated, if the value is not proper we disallow
 *      the characters inserted and display a popup giving acceptable
 *      value information, e.g. "Only text shorter than x characters
 *      are accepted" or "Only integers from 0 to 65535 are accepted.".
 *
 */
void on_option_entry_text_inserted(GtkEditable * editable, const gchar * new_text,
				   gint new_text_length, gint * position,
				   option_widget_info_t * info)
{
	gchar *string;
	gchar *inserted_text;
	gchar *pre_insertion_text;
	gchar *post_insertion_text;
	gchar *acceptable_values_msg = NULL;

	/*
	 * Build the string as it would look if the new text got inserted.
	 * Use the generated string for the validation.
	 */

	inserted_text = g_malloc0(new_text_length + 1);
	pre_insertion_text = gtk_editable_get_chars(editable, 0, *position);
	post_insertion_text = gtk_editable_get_chars(editable, *position, -1);

	g_memmove(inserted_text, new_text, new_text_length);
	string =
	    g_strstrip(g_strconcat(pre_insertion_text, inserted_text, post_insertion_text, NULL));

	if (info->option->type == EVMS_Type_String) {
		if (info->option->max_len < strlen(string))
			acceptable_values_msg =
			    g_strdup_printf(_
					    ("Only text shorter than %d characters is acceptable."),
					    info->option->max_len + 1);
	} else
		acceptable_values_msg =
		    validate_string_as_numeric_value(string, info->option->type);

	if (acceptable_values_msg != NULL) {
		gchar *title = g_strdup_printf(_("Invalid input for %s"), info->option->title);

		/*
		 * Validation error has occurred. Create a popup with the name
		 * of the field and the acceptable values text. Stop the 
		 * "insert-text" from being emitted as the new text inserted
		 * was unacceptable.
		 */

		display_popup_window(title, acceptable_values_msg);
		gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");

		g_free(title);
		g_free(acceptable_values_msg);
	} else {
		/*
		 * Just in case this is the last entry widget, we indicate it
		 * now has a value. This should allow the "Next" button to get
		 * sensitized. This should lead the user to click on it which
		 * causes the entry widget to lose focus and have its value
		 * set. We however ignore string types values that have not
		 * met their minimum length.
		 */

		if (!
		    (info->option->type == EVMS_Type_String
		     && (strlen(string) < info->option->min_len))) {
			value_t value;
			options_callback_callout callout;

			convert_string_to_option_value(string, info, &value);

			info->has_value = TRUE;

			callout = retrieve_options_callout_from_toplevel_widget(info->widget);

			if (callout != NULL)
				(*callout) (info, value, 0);
		}
	}

	g_free(string);
	g_free(pre_insertion_text);
	g_free(post_insertion_text);
}

/*
 *
 *   void on_option_entry_text_deleted (GtkEditable *, gint, gint, option_descriptor_t *)
 *
 *   Description:
 *      This routine gets called when text is deleted in an option
 *      entry box. This callback routine does not call 
 *      evms_set_option_value() like the other callbacks. Instead, 
 *      it checks to see if the option is for a EVMS_Type_String
 *      and do executes the options callout if the remaining text
 *      falls below the plug-in's minimum length.
 *
 *   Entry:
 *      editable  - the entry box that had characters deleted
 *      start_pos - the starting position of characters removed
 *      end_pos   - the end position of characters removed
 *      info      - address of the option widget info associated with widget
 *
 *   Exit:
 *      The regsitered options callout is invoked if the length of
 *      string type has fallen below the minimum length.
 *
 */
void on_option_entry_text_deleted(GtkEditable * editable, gint start_pos, gint end_pos,
				  option_widget_info_t * info)
{
	if (info->option->type == EVMS_Type_String) {
		gint remaining_len;
		gchar *current_text;
		gchar *deleted_text;

		current_text = g_strstrip(gtk_editable_get_chars(editable, 0, -1));
		deleted_text = gtk_editable_get_chars(editable, start_pos, end_pos);

		/*
		 * If deleted all the way to the end then trim off trailing
		 * whitespace since we don't count that when checking against
		 * min_len.
		 */

		if (end_pos < 0)
			g_strchomp(deleted_text);

		remaining_len = strlen(current_text) - strlen(deleted_text);

		if (remaining_len < info->option->min_len) {
			value_t value;
			options_callback_callout callout;

			value.s = current_text;

			/*
			 * We're below the minimum length. We must behave as if the
			 * option no longer has a value.
			 */

			info->has_value = FALSE;

			callout = retrieve_options_callout_from_toplevel_widget(info->widget);

			if (callout != NULL)
				(*callout) (info, value, 0);
		}

		g_free(current_text);
		g_free(deleted_text);
	}
}

inline void convert_float_to_option_value(gfloat number, option_widget_info_t * info,
					  value_t * value)
{
	if (info->option->unit == EVMS_Unit_Sectors && info->unit_conversion != info->option->unit)
		convert_float_to_sector_value(number, info->option->type, info->unit_conversion,
					      value);
	else
		convert_float_to_value(number, info->option->type, value);
}

inline void convert_option_value_to_float(value_t value, option_widget_info_t * info,
					  gfloat * number)
{
	if (info->option->unit == EVMS_Unit_Sectors && info->unit_conversion != info->option->unit)
		*number = convert_sector_value_to_float_with_unit(value, info->option->type,
								  info->unit_conversion);
	else
		*number = convert_value_to_float(value, info->option->type);
}

/*
 *
 *   inline void snap_adjustment_value_to_constraint_increment (GtkAdjustment *, option_widget_info_t *, gfloat *)
 *
 *   Description:
 *      This routine ensures that adjustment value given is an increment multiple as
 *      stipulated by the range constraint for the adjustment. If not, it is snapped
 *      to the closest proper value.
 *
 *   Entry:
 *      adjustement - the adjustment object which holds the current value and increment
 *      value       - address to store the (possibly) modified adjustment value
 *
 *   Exit:
 *      The adjustment value is snapped to a range increment if necessary
 *
 */
inline void snap_adjustment_value_to_increment(GtkAdjustment * adjustment, gfloat * value)
{
	gfloat inc;
	gfloat tmp;
	gfloat val;

	val = *value;
	inc = adjustment->step_increment;
	tmp = (val - adjustment->lower) / inc;

	if (tmp - floor(tmp) < ceil(tmp) - tmp)
		val = adjustment->lower + floor(tmp) * inc;
	else
		val = adjustment->lower + ceil(tmp) * inc;

	if (fabs(val - adjustment->value) > 1e-5)
		*value = val;
}

/*
 *
 *   void on_option_adjustment_changed (GtkAdjustment *, option_widget_info_t *)
 *
 *   Description:
 *      This routine gets called when an option widget with an adjustment
 *      such as a GtkHScale or GtkSpinButton has its value changed.
 *
 *   Entry:
 *      adjustment - GtkAdjustment that has been changed with a new value
 *      info       - address of the option widget info associated with widget
 *
 *   Exit:
 *      The option value is updated and if necessary we notify our top
 *      level window of sideaffects of the value update.
 *
 */
void on_option_adjustment_changed(GtkAdjustment * adjustment, option_widget_info_t * info)
{
	gint rc;
	gfloat adj_value;
	value_t value;
	gboolean value_changed;

	log_entry;

	adj_value = adjustment->value;

	snap_adjustment_value_to_increment(adjustment, &adj_value);

	convert_float_to_option_value(adj_value, info, &value);

	rc = set_widget_option_value(info, &value, &value_changed);

	if (info->has_value) {
		gfloat float_value;

		convert_option_value_to_float(info->value, info, &float_value);

		gtk_signal_handler_block_by_func(GTK_OBJECT(adjustment),
						 on_option_adjustment_changed, info);
		gtk_adjustment_set_value(adjustment, float_value);
		gtk_signal_handler_unblock_by_func(GTK_OBJECT(adjustment),
						   on_option_adjustment_changed, info);
	}

	log_exit;
}

/*
 *
 *   void set_listbox_selections_from_value_list (GtkCList *, option_widget_info_t *, value_list_t *)
 *
 *   Description:
 *      This routine selects only the rows in a option listbox that
 *      match entries in the value list supplied.
 *
 *   Entry:
 *      clist      - the option listbox widget
 *      info       - address of the option widget info associated with widget
 *      value_list - the list of values corresponding to rows that should be selected
 *
 *   Exit:
 *      The listbox has the rows corresponding to values in value list
 *      selected.
 *
 */
void set_listbox_selections_from_value_list(GtkCList * clist, option_widget_info_t * info,
					    value_list_t * value_list)
{
	gint i;
	gchar *text;
	value_t value;

	for (i = 0; i < clist->rows; i++) {
		if (gtk_clist_get_text(clist, i, 0, &text)) {
			convert_string_to_value(text, info->option->type, info->option->max_len,
						&value);

			if (value_in_value_list(value, value_list, info->option->type))
				gtk_clist_select_row(clist, i, 0);
			else
				gtk_clist_unselect_row(clist, i, 0);

			clear_value(&value, info->option->type, FALSE, 0);
		}
	}
}

/*
 *
 *   void on_option_listbox_selection_change (GtkCList *, gint, gint, GdkEventButton *, option_widget_info_t *)
 *
 *   Description:
 *      This routine gets called when a row is selected or deselected 
 *      in an options listbox.
 *
 *   Entry:
 *      clist  - the option listbox widget
 *      row    - the row that was (de)selected
 *      column - the column that was (de)selected (ignored)
 *      event  - additional context information (ignored)
 *      info   - address of the option widget info associated with widget
 *
 *   Exit:
 *      The option value is updated and if necessary we notify our top
 *      level window of sideaffects of the value update.
 *
 */
void on_option_listbox_selection_change(GtkCList * clist, gint row, gint column,
					GdkEventButton * event, option_widget_info_t * info)
{
	gint rc;
	guint i;
	guint j;
	gint selection_row;
	gchar *selection_string;
	guint count;
	value_t value;
	gboolean value_changed;

	log_entry;

	/*
	 * Since it is possible that the backend may modify the selection
	 * list to add items up to the number of constraint list items, we
	 * must be prepared and always allocate enough space to match the
	 * constraint list count of items. If the value type is string then
	 * the strings need to be allocated with the max size as well.
	 */

	value.list = allocate_value_list(info->option->constraint.list->count,
					 info->option->type, info->option->max_len);

	/*
	 * Walk through the currently selected rows and convert the strings
	 * representing the value to value_t to copy to the value list.
	 */

	count = g_list_length(clist->selection);

	for (i = 0, j = 0; i < count; i++) {
		selection_row = GPOINTER_TO_INT(g_list_nth_data(clist->selection, i));

		if (gtk_clist_get_text(clist, selection_row, 0, &selection_string)) {
			if (info->option->type == EVMS_Type_String)
				strcpy(value.list->value[j].s, selection_string);
			else
				convert_string_to_value(selection_string, info->option->type,
							info->option->max_len,
							&(value.list->value[j]));
			j++;
		}
	}

	value.list->count = j;

	rc = set_widget_option_value(info, &value, &value_changed);

	if ((rc == SUCCESS && value_changed) || (rc != SUCCESS && info->has_value)) {
		/*
		 * Rebuild the listbox selections here.
		 */
		gtk_signal_handler_block_by_func(GTK_OBJECT(clist),
						 on_option_listbox_selection_change, info);
		set_listbox_selections_from_value_list(clist, info, info->value.list);
		gtk_signal_handler_unblock_by_func(GTK_OBJECT(clist),
						   on_option_listbox_selection_change, info);
	}

	clear_value(&value, info->option->type, TRUE, info->option->constraint.list->count);

	log_exit;
}

/*
 *
 *   GtkWidget *create_check_button_widget (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine is called when there is a boolean value to display.
 *
 *   Entry:
 *      descriptor - address of structure that contains the boolean value
 *
 *   Exit:
 *      A GtkCheckButton is created with the initial value set.
 *
 */
GtkWidget *create_check_button_widget(evms_value_descriptor_t * descriptor)
{
	GtkWidget *checkbutton;

	checkbutton = gtk_check_button_new();

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton), descriptor->value.b);

	return checkbutton;
}

/*
 *
 *   GtkWidget *create_boolean_radio_button_widget_group (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine is called when there is a boolean value to display using
 *      a couple of radio buttons rather than one check button.
 *
 *   Entry:
 *      descriptor - address of structure that contains the boolean value
 *
 *   Exit:
 *      A GtkHBox is created containing two radio buttons with the proper labels
 *      and the initial button activated.
 *
 */
GtkWidget *create_boolean_radio_button_widget_group(evms_value_descriptor_t * descriptor)
{
	GtkWidget *hbox;
	GtkWidget *radiobutton;
	GtkWidget *yes_button;
	GtkWidget *no_button;

	hbox = gtk_hbox_new(FALSE, 0);

	yes_button = gtk_radio_button_new_with_label(NULL, _("Yes"));
	no_button =
	    gtk_radio_button_new_with_label(gtk_radio_button_group(GTK_RADIO_BUTTON(yes_button)),
					    _("No"));

	gtk_widget_show(yes_button);
	gtk_widget_show(no_button);

	gtk_box_pack_start(GTK_BOX(hbox), yes_button, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), no_button, FALSE, FALSE, 8);

	/*
	 * Due to the fact we share a signal handler and the weirdness with
	 * how radio buttons work when one goes active, we save the address
	 * of the "Yes/No" radio buttons in case we need to reference them
	 * in the signal handler.
	 */

	gtk_object_set_data(GTK_OBJECT(yes_button), "yes_button", yes_button);
	gtk_object_set_data(GTK_OBJECT(yes_button), "no_button", no_button);
	gtk_object_set_data(GTK_OBJECT(no_button), "yes_button", yes_button);
	gtk_object_set_data(GTK_OBJECT(no_button), "no_button", no_button);

	if (descriptor->value.b)
		radiobutton = yes_button;
	else
		radiobutton = no_button;

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobutton), TRUE);

	return hbox;
}

/*
 *
 *   GtkWidget *create_entry_widget (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine is called when a text entry box is needed.
 *
 *   Entry:
 *      descriptor - address of structure that contains the initial
 *                   value
 *
 *   Exit:
 *      A GtkEntry is created with the initial value in the descriptor
 *      converted to text and placed in the entry box.
 *
 */
GtkWidget *create_entry_widget(evms_value_descriptor_t * descriptor)
{
	GtkWidget *entry;

	entry = gtk_entry_new();

	if (descriptor->has_initial_value) {
		gchar *string;

		/* 
		 * When entry is being used for display purposes only, it is
		 * favorable to convert certain unit types, e.g. EVMS_Unit_Sectors,
		 * to a friendlier readable format and also append the unit measurement.
		 */

		if (descriptor->is_info_only)
			string =
			    make_value_readable_string(descriptor->value, descriptor->type,
						       descriptor->unit, descriptor->format,
						       descriptor->ok_to_convert);
		else
			string =
			    convert_value_to_string(descriptor->value, descriptor->type,
						    descriptor->format);

		gtk_entry_set_text(GTK_ENTRY(entry), string == NULL ? "" : string);
		g_free(string);
	}

	return entry;
}

/*
 *
 *   GtkWidget *create_adjustment_object (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine is called when there is a range of numeric
 *      values to display and an adjustment object is needed to
 *      represent the range.
 *
 *   Entry:
 *      descriptor - address of structure that contains the range of
 *                   numeric values
 *
 *   Exit:
 *      A GtkAdjustment is created with the numeric range provided.
 *
 */
GtkAdjustment *create_adjustment_object(evms_value_descriptor_t * descriptor)
{
	gfloat value;
	gfloat lower;
	gfloat upper;
	gfloat increment;

	if (descriptor->ok_to_convert && descriptor->unit == EVMS_Unit_Sectors) {
		descriptor->unit_conversion =
		    select_readable_unit_for_sector_value(descriptor->collection.range->min,
							  descriptor->type);
		value =
		    convert_sector_value_to_float_with_unit(descriptor->value, descriptor->type,
							    descriptor->unit_conversion);
		lower =
		    convert_sector_value_to_float_with_unit(descriptor->collection.range->min,
							    descriptor->type,
							    descriptor->unit_conversion);
		upper =
		    convert_sector_value_to_float_with_unit(descriptor->collection.range->max,
							    descriptor->type,
							    descriptor->unit_conversion);
		increment =
		    convert_sector_value_to_float_with_unit(descriptor->collection.range->increment,
							    descriptor->type,
							    descriptor->unit_conversion);
	} else {
		value = convert_value_to_float(descriptor->value, descriptor->type);
		lower = convert_value_to_float(descriptor->collection.range->min, descriptor->type);
		upper = convert_value_to_float(descriptor->collection.range->max, descriptor->type);
		increment =
		    convert_value_to_float(descriptor->collection.range->increment,
					   descriptor->type);
	}

	return GTK_ADJUSTMENT(gtk_adjustment_new(value, lower, upper, increment, increment, 0.0));
}

/*
 *
 *   GtkWidget *create_spin_button_widget (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine is called when there is a range of numeric
 *      values to display.
 *
 *   Entry:
 *      descriptor - address of structure that contains the range of
 *                   numeric values
 *
 *   Exit:
 *      A GtkSpinButton is created with the numeric range provided 
 *      and the entry field contains the initial value.
 *
 */
GtkWidget *create_spin_button_widget(evms_value_descriptor_t * descriptor)
{
	guint digits;
	gfloat increment;
	GtkWidget *spinbutton;
	GtkAdjustment *adjustment;

	adjustment = create_adjustment_object(descriptor);

	if (descriptor->ok_to_convert && descriptor->unit == EVMS_Unit_Sectors &&
	    descriptor->unit_conversion != EVMS_Unit_Sectors) {
		increment =
		    convert_sector_value_to_float_with_unit(descriptor->collection.range->increment,
							    descriptor->type,
							    descriptor->unit_conversion);
		digits = 2;
	} else {
		increment =
		    convert_value_to_float(descriptor->collection.range->increment,
					   descriptor->type);

		if (descriptor->type == EVMS_Type_Real32 || descriptor->type == EVMS_Type_Real64)
			digits = 2;
		else
			digits = 0;
	}

	spinbutton = gtk_spin_button_new(adjustment, increment, digits);

	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinbutton), TRUE);
	gtk_spin_button_set_snap_to_ticks(GTK_SPIN_BUTTON(spinbutton), TRUE);

	return spinbutton;
}

/*
 *
 *   GtkWidget *create_hscale_widget (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine is called when there is a range of numeric
 *      values to display and the range is small (0 - 100).
 *
 *   Entry:
 *      descriptor - address of structure that contains the range of
 *                   numeric values
 *
 *   Exit:
 *      A GtkHScale is created with the numeric range provided.
 *
 */
GtkWidget *create_hscale_widget(evms_value_descriptor_t * descriptor)
{
	GtkWidget *hscale;
	GtkAdjustment *adjustment;

	adjustment = create_adjustment_object(descriptor);

	hscale = gtk_hscale_new(adjustment);

	if (descriptor->type == EVMS_Type_Real32 || descriptor->type == EVMS_Type_Real64 ||
	    (descriptor->unit == EVMS_Unit_Sectors
	     && descriptor->unit_conversion != EVMS_Unit_Sectors))
		gtk_scale_set_digits(GTK_SCALE(hscale), 2);
	else
		gtk_scale_set_digits(GTK_SCALE(hscale), 0);

	gtk_range_set_update_policy(GTK_RANGE(hscale), GTK_UPDATE_DISCONTINUOUS);

	return hscale;
}

/*
 *
 *   GtkWidget *create_list_box_widget (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine is called when the backend requires multiple
 *      selections from a constraint list as input rather than
 *      just one value.
 *
 *   Entry:
 *      descriptor - address of structure that contains the constraint
 *                   list and the current selections.
 *
 *   Exit:
 *      A one column GtkCList is created with the selection list
 *      provided.
 *
 */
GtkWidget *create_list_box_widget(evms_value_descriptor_t * descriptor)
{
	guint i;
	gint row;
	gchar *item;
	guint count;
	value_list_t *vlist;
	GtkWidget *listbox;

	listbox = gtk_clist_new(1);

	gtk_clist_set_selection_mode(GTK_CLIST(listbox), GTK_SELECTION_MULTIPLE);

	vlist = descriptor->collection.list;
	count = descriptor->collection.list->count;

	for (i = 0; i < count; i++) {
		item =
		    convert_value_to_string(vlist->value[i], descriptor->type, descriptor->format);

		if (item != NULL) {
			row = gtk_clist_append(GTK_CLIST(listbox), &item);

			/*
			 * If we have an initial value list then mark the value
			 * just appended as selected if it is in the initial
			 * value list.
			 */

			if (descriptor->has_initial_value &&
			    value_in_value_list(vlist->value[i], descriptor->value.list,
						descriptor->type))
				gtk_clist_select_row(GTK_CLIST(listbox), row, 0);

			g_free(item);
		}
	}

	return listbox;
}

/*
 *
 *   GtkWidget *create_combo_box_widget (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine is called when there is a collection of valid
 *      values to display. It is typically created for long lists
 *      so that a scollbar can used.
 *
 *   Entry:
 *      descriptor - address of structure that contains the list of
 *                   valid values
 *
 *   Exit:
 *      A GtkComboBox is created with the value list provided and the
 *      entry field contains the initial value.
 *
 */
GtkWidget *create_combo_box_widget(evms_value_descriptor_t * descriptor)
{
	guint i;
	gchar *item;
	guint count;
	GList *glist;
	GtkCombo *combo;
	value_list_t *vlist;

	g_return_val_if_fail(descriptor->collection_type == EVMS_Collection_List, NULL);

	glist = NULL;
	vlist = descriptor->collection.list;
	count = descriptor->collection.list->count;

	for (i = 0; i < count; i++) {
		if (descriptor->ok_to_convert && descriptor->unit == EVMS_Unit_Sectors)
			item =
			    make_sectors_readable_string(convert_value_to_ui64
							 (vlist->value[i], descriptor->type));
		else
			item =
			    convert_value_to_string(vlist->value[i], descriptor->type,
						    descriptor->format);

		if (item != NULL)
			glist = g_list_append(glist, item);
	}

	combo = GTK_COMBO(gtk_combo_new());

	/* 
	 * Set the available selections. Set the properties for
	 * the list indicating that values entered must be in
	 * the list and that an empty value is not valid. Lastly,
	 * display the initial value in the entry.
	 *
	 * We may have converted some sector values to readable forms, 
	 * e.g. KB or MB and in different units so that the label does
	 * not indicate a unit, turn off the original unit.
	 */

	if (descriptor->ok_to_convert && descriptor->unit == EVMS_Unit_Sectors)
		descriptor->unit_conversion = EVMS_Unit_None;

	gtk_combo_set_popdown_strings(combo, glist);

	if (descriptor->has_initial_value) {
		gchar *choice;

		if (descriptor->ok_to_convert && descriptor->unit == EVMS_Unit_Sectors)
			choice =
			    make_sectors_readable_string(convert_value_to_ui64
							 (descriptor->value, descriptor->type));
		else
			choice =
			    convert_value_to_string(descriptor->value, descriptor->type,
						    descriptor->format);

		gtk_entry_set_text(GTK_ENTRY(combo->entry), choice);
		g_free(choice);
	} else if (!descriptor->is_info_only)
		gtk_editable_delete_text(GTK_EDITABLE(combo->entry), 0, -1);

	gtk_combo_set_value_in_list(combo, TRUE, TRUE);
	gtk_editable_set_editable(GTK_EDITABLE(combo->entry), FALSE);

	return GTK_WIDGET(combo);
}

/*
 *
 *   GtkWidget *create_option_menu_widget (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine is called when there is a collection of valid
 *      values to display. It is typically created for short lists
 *      that allows the list to be viewable entirely on the screen
 *      without the aid of a scollbar.
 *
 *   Entry:
 *      descriptor - address of structure that contains the list of
 *                   valid values
 *
 *   Exit:
 *      A GtkOptionMenu is created with the value list provided and the
 *      initial value it displays is set
 *
 */
GtkWidget *create_option_menu_widget(evms_value_descriptor_t * descriptor)
{
	guint i;
	guint count;
	guint init_value_index = 0;
	gchar *item;
	value_list_t *vlist;
	GtkWidget *menu;
	GtkWidget *option_menu;

	g_return_val_if_fail(descriptor->collection_type == EVMS_Collection_List, NULL);

	vlist = descriptor->collection.list;
	count = descriptor->collection.list->count;

	/*
	 * Create the option menu and associated submenu and populate
	 * the menu items from the value list provided. Check to see
	 * if initial value matches one in the list and if so set the
	 * menu item index as the selected value.
	 */

	option_menu = gtk_option_menu_new();
	menu = gtk_menu_new();

	for (i = 0; i < count; i++) {
		if (descriptor->ok_to_convert && descriptor->unit == EVMS_Unit_Sectors)
			item =
			    make_sectors_readable_string(convert_value_to_ui64
							 (vlist->value[i], descriptor->type));
		else
			item =
			    convert_value_to_string(vlist->value[i], descriptor->type,
						    descriptor->format);

		if (item != NULL) {
			GtkWidget *menuitem;

			menuitem = gtk_menu_item_new_with_label(item);

			if (descriptor->has_initial_value &&
			    values_are_equal(vlist->value[i], descriptor->value, descriptor->type))
				init_value_index = i;

			gtk_menu_append(GTK_MENU(menu), menuitem);
			gtk_widget_show(menuitem);

			g_free(item);
		}
	}

	/*
	 * We may have converted some sector values to readable forms, e.g. KB or MB
	 * and in different units so that the label does not indicate a unit, turn
	 * off the original unit.
	 */

	if (descriptor->ok_to_convert && descriptor->unit == EVMS_Unit_Sectors)
		descriptor->unit_conversion = EVMS_Unit_None;

	/*
	 * If this widget will be used for required option input and 
	 * no initial value was provided then insert a message that 
	 * provides a cue to the user that something needs to be selected.
	 * The special menu item will be removed the first time the user
	 * selects some other menu item.
	 */

	if (descriptor->is_info_only == FALSE && descriptor->has_initial_value == FALSE) {
		if (descriptor->is_required) {
			GtkWidget *menuitem;

			menuitem = gtk_menu_item_new_with_label(OPTION_MENU_NO_SELECTION_TEXT);

			gtk_menu_prepend(GTK_MENU(menu), menuitem);
			gtk_object_set_data(GTK_OBJECT(menu), "no_selection_menuitem", menuitem);

			gtk_widget_show(menuitem);
		}
	}

	gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(option_menu), init_value_index);

	return option_menu;
}

/*
 *
 *   void remove_widget_properties (GtkWidget *, widget_properties_t *)
 *
 *   Description:
 *      This routine takes a widget and the address of its associated
 *      widget properties structure and has the structure memory freed.
 *      The object data in the widget corresponding to the "widget_properties"
 *      key is also cleared.
 *
 *   Entry:
 *      widget - the widget to have the widget properties structure removed
 *
 *   Exit:
 *      The widget properties are freed and removed from the widget.
 *
 */
void remove_widget_properties(GtkWidget * widget, widget_properties_t * properties)
{
	log_entry;

	if (gtk_object_get_data(GTK_OBJECT(widget), "widget_properties")) {
		if (properties->name != NULL)
			g_free(properties->name);

		if (properties->tip != NULL)
			g_free(properties->tip);

		if (properties->title != NULL)
			g_free(properties->title);

		if (properties->group.group_name != NULL)
			g_free(properties->group.group_name);

		g_free(properties);

		gtk_object_set_data(GTK_OBJECT(widget), "widget_properties", NULL);
	}

	log_exit;
}

/*
 *
 *   void bind_widget_properties_to_widget (evms_value_descriptor_t *, GtkWidget *)
 *
 *   Description:
 *      This routine creates a widget properties structure that contain information
 *      used when creating additional widget notebook information such as the field
 *      label and the tooltip that contains a description of the field.
 *
 *   Entry:
 *      descriptor - address of structure that fully describes the value the
 *                   widget represents
 *      widget     - widget that will have additional data bound to it
 *
 *   Exit:
 *      A widget_properties_t structure is dynamically allocated and initialized and
 *      bound to the widget.
 *
 */
void bind_widget_properties_to_widget(evms_value_descriptor_t * descriptor, GtkWidget * widget)
{
	widget_properties_t *properties;

	properties = g_new0(widget_properties_t, 1);

	properties->name = g_strdup(descriptor->name);
	properties->unit = descriptor->unit_conversion;
	properties->is_info_only = descriptor->is_info_only;
	properties->more_data_avail = descriptor->more_data_avail;
	properties->is_required = descriptor->is_required;
	properties->group.group_level = descriptor->group.group_level;
	properties->group.group_number = descriptor->group.group_number;

	if (descriptor->group.group_number != 0 && descriptor->group.group_name != NULL)
		properties->group.group_name = g_strdup(descriptor->group.group_name);

	if (descriptor->desc != NULL)
		properties->tip = g_strdup(descriptor->desc);

	if (descriptor->title != NULL)
		properties->title = g_strdup(descriptor->title);

	gtk_object_set_data(GTK_OBJECT(widget), "widget_properties", properties);

	gtk_signal_connect(GTK_OBJECT(widget), "destroy",
			   GTK_SIGNAL_FUNC(remove_widget_properties), properties);

}

/*
 *
 *   GtkWidget* create_widget_from_value_descriptor (evms_value_descriptor_t *)
 *
 *   Description:
 *      This routine creates a widget that will suitably represent the 
 *      value described. 
 *
 *   Entry:
 *      descriptor    - address of structure that fully describes the value the
 *                      widget represents
 *
 *   Exit:
 *      A suitable widget is created that can represent the value described.
 *
 */
GtkWidget *create_widget_from_value_descriptor(evms_value_descriptor_t * descriptor)
{
	GtkWidget *widget = NULL;

	if (descriptor->collection_type == EVMS_Collection_None) {
		if (descriptor->type == EVMS_Type_Boolean) {
			if (descriptor->is_info_only)
				widget = create_entry_widget(descriptor);
			else {
				widget = create_boolean_radio_button_widget_group(descriptor);
			}
		} else
			widget = create_entry_widget(descriptor);
	} else {
		/*
		 * Select the proper widget that allows multiple
		 * selections for ranges and lists.
		 */

		if (descriptor->collection_type == EVMS_Collection_Range) {
			if (descriptor->type >= EVMS_Type_Real32
			    && descriptor->type <= EVMS_Type_Unsigned_Int64) {
				if (value_less_than_integer
				    (descriptor->collection.range->max, descriptor->type, 101))
					widget = create_hscale_widget(descriptor);
				else
					widget = create_spin_button_widget(descriptor);
			} else {
				log_error
				    ("%s: A non-numeric range option descriptor was supplied!\n",
				     __FUNCTION__);
			}
		} else if (descriptor->collection_type == EVMS_Collection_List) {
			if (descriptor->type != EVMS_Type_Boolean) {
				if (descriptor->multi_input_list)
					widget = create_list_box_widget(descriptor);
				else if (descriptor->collection.list->count > 25)
					widget = create_combo_box_widget(descriptor);
				else
					widget = create_option_menu_widget(descriptor);
			} else
				log_error
				    ("%s: An unsupported option descriptor list was supplied!\n",
				     __FUNCTION__);
		} else {
			log_error("%s: Unknown collection type supplied.\n", __FUNCTION__);
		}
	}

	if (widget != NULL)
		bind_widget_properties_to_widget(descriptor, widget);

	return widget;
}

/*
 *
 *   void connect_option_widget_callback (GtkWidget *, option_widget_info_t *)
 *
 *   Description:
 *      This routine makes the connection(s) with the proper callback(s) for
 *      the corresponding widget input signals.
 *
 *   Entry:
 *      widget - widget that needs callback connection(s) established
 *      info   - address of the option widget info associated with the widget
 *
 *   Exit:
 *      The corresponding callback function(s) is/are connected to allow 
 *      handling input signals.
 *
 */
void connect_option_adjustment_callback(GtkWidget * widget, option_widget_info_t * info)
{
	GtkAdjustment *adjustment = NULL;

	/*
	 * In order to find out when the value for a widget that
	 * uses an adjustment has been changed, we need to connect
	 * the "value_changed" signal with the GtkAdjustment object
	 * owned by the widget.
	 */

	if (GTK_IS_SPIN_BUTTON(widget))
		adjustment = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(widget));
	else if (GTK_IS_RANGE(widget))
		adjustment = gtk_range_get_adjustment(GTK_RANGE(widget));
	else
		log_error
		    ("%s: Not a range or spinbutton so don't know if it has an adjustment widget.\n",
		     __FUNCTION__);

	if (adjustment != NULL)
		gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed",
				   GTK_SIGNAL_FUNC(on_option_adjustment_changed), info);
}

/*
 *
 *   void on_option_widget_destroy (GtkWidget *, option_widget_info_t *)
 *
 *   Description:
 *      This routine frees the option descriptor, current value information
 *      and widget info structure as the widget gets destroyed.
 *
 *   Entry:
 *      widget - the option widget
 *      info   - address of the option widget info associated with the widget
 *
 *   Exit:
 *      The option descriptor, value and widget info are freed.
 *
 */
void on_option_widget_destroy(GtkWidget * widget, option_widget_info_t * info)
{
	option_descriptor_t *option = info->option;

	log_debug("%s: Dude, I'm going to totally destroy this option widget info: %p!\n",
		  __FUNCTION__, info);

	if (info->has_value) {
		guint list_count = 0;

		if (EVMS_OPTION_VALUE_IS_LIST(option->flags))
			list_count = option->constraint.list->count;

		clear_value(&(info->value), option->type, EVMS_OPTION_VALUE_IS_LIST(option->flags),
			    list_count);
	}

	evms_free(option);
	g_free(info);
}

/*
 *
 *   void connect_box_children_callbacks (GtkBox *, option_widget_info_t *)
 *
 *   Description:
 *      This routine connects the callbacks for the widgets located in
 *      box container.
 *
 *   Entry:
 *      box  - the GtkBox container type
 *      info - address of the option widget info associated with the widget
 *
 *   Exit:
 *      The signal handlers for the appropriate signals and widgets are connected.
 *
 */
void connect_box_children_callbacks(GtkBox * box, option_widget_info_t * info)
{
	if (g_list_length(box->children) > 0) {
		GList *child;

		child = box->children;

		while (child) {
			if (child->data)
				connect_option_widget_callback(((GtkBoxChild *) (child->data))->
							       widget, info, FALSE);

			child = g_list_next(child);
		}
	} else {
		log_warning("%s: Box has no children.\n", __FUNCTION__);
	}
}

/*
 *
 *   void connect_option_widget_callback (GtkWidget *, option_widget_info_t *, gboolean)
 *
 *   Description:
 *      This routine makes the connection(s) with the proper callback(s) for
 *      the corresponding widget input signals.
 *       
 *   Entry:
 *      widget          - widget that needs callback connection(s) established
 *      info            - address of the option widget info associated with the widget
 *      connect_destroy - TRUE to register signal handler to cleanup info during widget
 *                        "destroy" event
 *
 *   Exit:
 *      The corresponding callback function(s) is/are connected to allow 
 *      handling input signals.
 *
 */
void connect_option_widget_callback(GtkWidget * widget, option_widget_info_t * info,
				    gboolean connect_destroy)
{
	/*
	 * When doing widget class checks for connecting callbacks,
	 * start with subclasses and work your way up to something
	 * that "fits". Sorry to use if/else but this is the way
	 * it is done when checking what kind of widget we got
	 * due to GTK+'s inheritance model.
	 */

	if (GTK_IS_COMBO(widget)) {
		gtk_signal_connect(GTK_OBJECT(GTK_COMBO(widget)->entry), "changed",
				   GTK_SIGNAL_FUNC(on_option_entry_text_entered), info);
		gtk_combo_disable_activate(GTK_COMBO(widget));
	} else if (GTK_IS_BOX(widget)) {
		/*
		 * We have a container type, enumerate the children and connect
		 * callbacks appropriately.
		 */
		connect_box_children_callbacks(GTK_BOX(widget), info);
	} else if (GTK_IS_CLIST(widget)) {
		gtk_signal_connect(GTK_OBJECT(widget), "select_row",
				   GTK_SIGNAL_FUNC(on_option_listbox_selection_change), info);

		gtk_signal_connect(GTK_OBJECT(widget), "unselect_row",
				   GTK_SIGNAL_FUNC(on_option_listbox_selection_change), info);
	} else if (GTK_IS_HSCALE(widget) || GTK_IS_SPIN_BUTTON(widget)) {
		if (GTK_IS_SPIN_BUTTON(widget)) {
			gtk_signal_connect(GTK_OBJECT(widget), "activate",
					   GTK_SIGNAL_FUNC(on_option_entry_text_entered), info);

			gtk_signal_connect(GTK_OBJECT(widget), "focus-out-event",
					   GTK_SIGNAL_FUNC(on_option_entry_focus_out), info);
		}

		connect_option_adjustment_callback(widget, info);
	} else if (GTK_IS_OPTION_MENU(widget)) {
		GtkWidget *menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(widget));

		gtk_object_set_data(GTK_OBJECT(menu), "option_menu", widget);

		gtk_signal_connect(GTK_OBJECT(menu), "selection_done",
				   GTK_SIGNAL_FUNC(on_option_menu_item_selected), info);
	} else if (GTK_IS_RADIO_BUTTON(widget)) {
		gtk_signal_connect(GTK_OBJECT(widget), "toggled",
				   GTK_SIGNAL_FUNC(on_option_radio_button_toggled), info);
	} else if (GTK_IS_CHECK_BUTTON(widget)) {
		gtk_signal_connect(GTK_OBJECT(widget), "toggled",
				   GTK_SIGNAL_FUNC(on_option_check_button_toggled), info);
	} else if (GTK_IS_ENTRY(widget)) {
		/*
		 * Text entry widgets are a funny thing. We can't really call
		 * evms_set_option_value() on it whenever text in the box changes
		 * since we can't be sure when the user is done. We set a callback
		 * for the "insert-text" event simply to do simple validation such 
		 * as ensure that the value is numeric and does not exceed the range
		 * for the numeric type according to the descriptor. Calling the
		 * evms_set_option_value() call for entry widgets will be delayed
		 * until the user presses Enter or the entry widget loses focus.
		 */

		gtk_signal_connect(GTK_OBJECT(widget), "insert-text",
				   GTK_SIGNAL_FUNC(on_option_entry_text_inserted), info);

		gtk_signal_connect(GTK_OBJECT(widget), "delete-text",
				   GTK_SIGNAL_FUNC(on_option_entry_text_deleted), info);

		gtk_signal_connect(GTK_OBJECT(widget), "activate",
				   GTK_SIGNAL_FUNC(on_option_entry_text_entered), info);

		gtk_signal_connect(GTK_OBJECT(widget), "focus-out-event",
				   GTK_SIGNAL_FUNC(on_option_entry_focus_out), info);
	} else if (GTK_IS_LABEL(widget)) {
		/*
		 * Bogus.
		 */
	} else {
		log_error("%s: Unexpected widget type was provided!\n", __FUNCTION__);
	}

	/*
	 *  Bind the option widget info to the object user data to allow
	 *  anyone passed this widget to extract it.
	 */

	gtk_object_set_user_data(GTK_OBJECT(widget), info);

	/*
	 * Connect the destroy signal to each widget to allow cleaning up the
	 * option widget info, option descriptor and value at that time.
	 */

	if (connect_destroy)
		gtk_signal_connect(GTK_OBJECT(widget), "destroy",
				   GTK_SIGNAL_FUNC(on_option_widget_destroy), info);
}

/*
 *
 *   option_widget_info_t* create_widget_from_option_descriptor (option_descriptor_t *)
 *
 *   Description:
 *      This routine creates a suitable input widget based on an option
 *      descriptor. The widget has the option_descriptor_t * information
 *      passed to the callback so that the callback function can call
 *      evms_set_option_value() when invoked.
 *
 *      Once we get the widget, we make the connections with the proper
 *      wrapper callback for the corresponding widget input signals. 
 *       
 *   Entry:
 *      option - address of the option descriptor to create an input widget for
 *
 *   Exit:
 *      The suitable widget is created and a corresponding callback function
 *      is connected to allow handling input signals. A dynamically allocated
 *      option_widget_info_t is returned containing all necessary information
 *      about the option widget.
 *
 */
option_widget_info_t *create_widget_from_option_descriptor(option_descriptor_t * option)
{
	GtkWidget *widget;
	option_widget_info_t *option_widget_info;
	evms_value_descriptor_t descriptor;

	/*
	 * Copy fields from option descriptor to the common structure used
	 * by the routines used to create a widget.
	 */

	descriptor.desc = option->tip;
	descriptor.group = option->group;
	descriptor.name = option->name;
	descriptor.title = option->title;
	descriptor.type = option->type;
	descriptor.unit = option->unit;
	descriptor.value = option->value;
	descriptor.collection = option->constraint;
	descriptor.collection_type = option->constraint_type;
	descriptor.format = option->format;
	descriptor.is_info_only = FALSE;
	descriptor.has_initial_value = EVMS_OPTION_HAS_VALUE(option->flags);
	descriptor.ok_to_convert = (option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION) == 0;
	descriptor.more_data_avail = FALSE;
	descriptor.multi_input_list = EVMS_OPTION_VALUE_IS_LIST(option->flags);
	descriptor.is_required = EVMS_OPTION_IS_REQUIRED(option->flags);
	descriptor.unit_conversion = option->unit;

	widget = create_widget_from_value_descriptor(&descriptor);

	option_widget_info = (option_widget_info_t *) g_malloc0(sizeof(option_widget_info_t));

	option_widget_info->widget = widget;
	option_widget_info->option = option;
	option_widget_info->unit_conversion = descriptor.unit_conversion;
	option_widget_info->has_value = descriptor.has_initial_value;

	if (option_widget_info->has_value) {
		guint list_count = 0;

		if (EVMS_OPTION_VALUE_IS_LIST(option->flags))
			list_count = option->constraint.list->count;

		duplicate_value(option->value, option->type,
				EVMS_OPTION_VALUE_IS_LIST(option->flags),
				list_count, option->max_len, &(option_widget_info->value));
	}

	/*
	 * Check to see if the option is active. If not, don't worry
	 * about setting up callbacks but do make the widget
	 * insensitive. 
	 */

	if (EVMS_OPTION_IS_ACTIVE(option->flags)) {
		connect_option_widget_callback(widget, option_widget_info, TRUE);
	} else {
		gtk_widget_set_sensitive(widget, FALSE);
	}

	return option_widget_info;
}

/*
 *
 *   GtkWidget* create_widget_from_extended_info (extended_info_t *)
 *
 *   Description:
 *      This routine creates a widget to be used to represent an
 *      extended info value for display purposes only (no input).
 *      This means text entry widgets are non-editable, no signal
 *      handlers are registered and some widgets are insensitive.
 * 
 *   Entry:
 *      info - address of an extended info description
 *
 *   Exit:
 *      A widget is created representing the extended info field.
 *
 */
GtkWidget *create_widget_from_extended_info(extended_info_t * info)
{
	GtkWidget *widget;
	evms_value_descriptor_t descriptor;

	descriptor.desc = info->desc;
	descriptor.group = info->group;
	descriptor.name = info->name;
	descriptor.title = info->title;
	descriptor.type = info->type;
	descriptor.unit = info->unit;
	descriptor.value = info->value;
	descriptor.collection = info->collection;
	descriptor.collection_type = info->collection_type;
	descriptor.format = info->format;
	descriptor.is_info_only = TRUE;
	descriptor.has_initial_value = info->collection_type == EVMS_Collection_None;
	descriptor.ok_to_convert = (info->flags & EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION) == 0;
	descriptor.more_data_avail = info->flags & EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	descriptor.multi_input_list = FALSE;
	descriptor.is_required = FALSE;
	descriptor.unit_conversion = info->unit;

	widget = create_widget_from_value_descriptor(&descriptor);

	if (GTK_IS_ENTRY(widget))
		gtk_editable_set_editable(GTK_EDITABLE(widget), FALSE);
	else if (GTK_IS_COMBO(widget))
		gtk_editable_set_editable(GTK_EDITABLE(GTK_COMBO(widget)->entry), FALSE);
	else if (GTK_IS_CHECK_BUTTON(widget))
		gtk_widget_set_sensitive(widget, FALSE);

	return widget;
}

/*
 *
 *   void inline add_label_to_notebook_table (GtkTable *, widget_properties_t *, GtkTooltips *, 
 *                                            guint, guint, guint)
 *
 *   Description:
 *      This routine inserts a label widget with descriptive text at the
 *      specified row between the starting and ending column. We also 
 *      set the widget tooltip on this label.
 * 
 *   Entry:
 *      table           - the table to contain the widgets
 *      properties      - the widget properties 
 *      tooltips        - id of tooltips group
 *      row             - the row in the table to place the widget at
 *      starting_column - the starting column in the table for the label
 *      ending_column   - the ending column in the table for the label
 *
 *   Exit:
 *      A label with the associated text is inserted at the given row and column
 *      and the field tooltip is set on it as well.
 *
 */
void inline add_label_to_notebook_table(GtkTable * table, widget_properties_t * properties,
					GtkTooltips * tooltips, guint row, guint starting_column,
					guint ending_column)
{
	GtkWidget *label;
	GtkWidget *widget_to_attach;

	if (properties->unit != EVMS_Unit_None && !properties->is_info_only) {
		gchar *unit;
		gchar *text;

		unit = make_unit_readable_string(properties->unit, TRUE);
		text = g_strconcat(properties->title, " (", unit, ")", NULL);

		label = gtk_label_new(text);

		g_free(unit);
		g_free(text);
	} else {
		label = gtk_label_new(properties->title);
	}

	gtk_widget_show(label);

	/* 
	 * If the widget has a tip, place it on the label to avoid
	 * extra events going to the widget plus allow tips to
	 * show in case the widget is insensitive. It also simplifies
	 * the code if we always do it on the field label.
	 */

	if (properties->tip != NULL) {
		GtkWidget *eventbox = gtk_event_box_new();

		gtk_widget_show(eventbox);
		gtk_container_add(GTK_CONTAINER(eventbox), label);
		gtk_tooltips_set_tip(tooltips, eventbox, properties->tip, NULL);

		widget_to_attach = eventbox;
	} else {
		widget_to_attach = label;
	}

	gtk_table_attach(table, widget_to_attach, starting_column, ending_column, row, row + 1,
			 (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
}

/*
 *
 *   void inline mark_field_as_required (GtkTable *, widget_properties_t *, GtkTooltips *, 
 *                                       guint, guint, guint)
 *
 *   Description:
 *      This routine inserts a label widget that contains an asterik
 *      that indicates that this field is required input. The tooltip
 *      for the label also indicates this fact. 
 * 
 *   Entry:
 *      table           - the table to contain the widgets
 *      properties      - the widget properties 
 *      tooltips        - id of tooltips group
 *      row             - the row in the table to place the widget at
 *      starting_column - the starting column in the table for the label
 *      ending_column   - the ending column in the table for the label
 *
 *   Exit:
 *      A label with the associated text is inserted at the given row and column
 *      and the field tooltip is set on it as well.
 *
 */
void inline mark_field_as_required(GtkTable * table, widget_properties_t * properties,
				   GtkTooltips * tooltips, guint row, guint starting_column,
				   guint ending_column)
{
	GtkWidget *label;
	GtkWidget *eventbox;

	label = gtk_label_new("*");
	eventbox = gtk_event_box_new();

	set_widget_foreground_color(label, "red");
	gtk_widget_show(label);
	gtk_widget_show(eventbox);

	gtk_container_add(GTK_CONTAINER(eventbox), label);
	gtk_tooltips_set_tip(tooltips, eventbox, _("This is a required field"), NULL);

	gtk_table_attach(table, eventbox, starting_column, ending_column, row, row + 1,
			 (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
}

/*
 *
 *   void on_more_button_destroy (GtkWidget *, field_details_t *)
 *
 *   Description:
 *      This routine frees the information used in obtaining
 *      additional extended info on a specific object field.
 *
 *   Entry:
 *      widget - the option widget
 *      info   - address of the field details info associated with widget
 *
 *   Exit:
 *      The field_details_t structure and its contents are freed.
 *
 */
void on_more_button_destroy(GtkWidget * widget, field_details_t * field)
{
	g_free(field->name);
	g_free(field->title);
	g_free(field);
}

/*
 *
 *   void inline add_more_button_to_notebook_table (GtkTable *, GtkTooltips *, 
 *                                                  widget_properties_t *, 
 *                                                  guint, guint, guint)
 *
 *   Description:
 *      This routine inserts a button widget with the given text at the
 *      specified row between the starting and ending column.
 * 
 *   Entry:
 *      table           - the table to contain the widgets
 *      tooltips        - id of tooltips group
 *      properties      - the widget properties
 *      row             - the row in the table to place the widget at
 *      starting_column - the starting column in the table for the button
 *      ending_column   - the ending column in the table for the button
 *
 *   Exit:
 *      A button to allow displaying more info is inserted at the given row and column.
 *
 */
void inline add_more_button_to_notebook_table(GtkTable * table, GtkTooltips * tooltips,
					      widget_properties_t * properties, guint row,
					      guint starting_column, guint ending_column)
{
	GtkWidget *button = gtk_button_new_with_label(_("More.."));
	field_details_t *field = g_new0(field_details_t, 1);

	field->name = g_strdup(properties->name);

	if (properties->title != NULL)
		field->title = g_strdup(properties->title);

	gtk_widget_show(button);
	gtk_tooltips_set_tip(tooltips, button, _("Display more detailed information..."), NULL);
	gtk_table_attach(table, button, starting_column, ending_column, row, row + 1,
			 (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);

	/*
	 * Setup signal handler that displays more detail on a field when
	 * the "More" button is clicked. The second signal handler handles
	 * freeing the field_details_t structure and strings.
	 */

	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(on_display_more_thing_details_button_clicked), field);

	gtk_signal_connect(GTK_OBJECT(button), "destroy",
			   GTK_SIGNAL_FUNC(on_more_button_destroy), field);
}

/*
 *
 *   void add_widget_to_notebook_table (GtkTable *, GtkWidget *, GtkTooltips * guint, guint)
 *
 *   Description:
 *      This routine takes a widget and places the widget at the given
 *      row in a table widget. The first column will contain a label for
 *      the field. The widget will begin at the column adjusted for indention
 *      level. If the widget has a full field description it will be set
 *      as the widget tooltip.
 * 
 *   Entry:
 *      table    - the table to contain the widget
 *      widget   - the widget to be added to the table
 *      tooltips - the tooltips object that to hold the field description
 *      row      - the row in the table to place the widget at
 *      columns  - the total number of columns in the table
 *
 *   Exit:
 *      Notebook is created with one or more pages and the widgets
 *      organized within.
 *
 */
void add_widget_to_notebook_table(GtkTable * table, GtkWidget * widget, GtkTooltips * tooltips,
				  guint * row, guint columns)
{
	guint end_row;
	guint starting_column;
	GtkWidget *widget_to_attach;
	widget_properties_t *properties;

	properties = gtk_object_get_data(GTK_OBJECT(widget), "widget_properties");

	/*
	 * If widget properties were not retrieved then create an empty 
	 * properties structure to allow the widget to be displayed anyway.
	 */

	if (properties == NULL)
		properties = g_new0(widget_properties_t, 1);

	if (GTK_WIDGET_SENSITIVE(widget) && properties->is_required)
		mark_field_as_required(table, properties, tooltips, *row, VALUE_REQUIRED_COLUMN,
				       WIDGET_LABEL_COLUMN);

	if (properties->title != NULL)
		add_label_to_notebook_table(table, properties, tooltips, *row, WIDGET_LABEL_COLUMN,
					    WIDGET_COLUMN);

	if (properties->group.group_number != 0)
		starting_column = WIDGET_COLUMN + properties->group.group_level;
	else
		starting_column = WIDGET_COLUMN;

	/*
	 * Do special handling for a list widget. It takes
	 * up additional rows in the table and it needs to
	 * be inside of a scrollbar widget.
	 */

	if (GTK_IS_CLIST(widget)) {
		widget_to_attach = gtk_scrolled_window_new(NULL, NULL);

		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(widget_to_attach),
					       GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_container_add(GTK_CONTAINER(widget_to_attach), widget);

		gtk_widget_show(widget);
		end_row = *row + CLIST_MAX_TABLE_ROWS;
	} else {
		end_row = *row + 1;
		widget_to_attach = widget;
	}

	gtk_widget_show(widget_to_attach);

	gtk_table_attach(table, widget_to_attach, starting_column, columns - 1, *row, end_row,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);

	/*
	 * Check to see if this extended info widget indicates more data
	 * can be queried for this field. If so, place a "More" button
	 * next to the field.
	 */

	if (properties->more_data_avail)
		add_more_button_to_notebook_table(table, tooltips, properties, *row, columns - 1,
						  columns);

	*row = end_row;
}

/*
 *
 *   guint get_max_indentation_level (GSList *, guint, guint)
 *
 *   Description:
 *      This routine takes a list of widgets and determines the maximum
 *      indentation level for a widget that is part of a logical group.
 *      The indentation level help assists in the number of columns
 *      contained in the widget table and the starting column the widget
 *      will be placed.
 * 
 *   Entry:
 *      widgets      - singly linked list containing widgets to be displayed
 *      first_widget - the index of the first widget to inspect
 *      last_widget  - the index of the last widget to inspect
 *
 *   Exit:
 *      The specified widget group structures are examined and the highest number
 *      indention level is returned.
 *
 */
guint get_max_indentation_level(GSList * widgets, guint first_widget, guint last_widget)
{
	guint i;
	guint max_indentation_level = 0;
	GtkWidget *widget;
	widget_properties_t *properties;

	for (i = first_widget; i <= last_widget; i++) {
		widget = g_slist_nth_data(widgets, i);

		if (widget != NULL) {
			properties = gtk_object_get_data(GTK_OBJECT(widget), "widget_properties");

			if (properties != NULL && properties->group.group_number != 0 &&
			    properties->group.group_level > max_indentation_level)
				max_indentation_level = properties->group.group_level;
		}
	}

	return max_indentation_level;
}

/*
 *
 *   void add_widget_page_to_notebook (guint, GSList *, guint, guint, GtkNotebook *, GtkTooltips *, gchar *)
 *
 *   Description:
 *      This routine takes a set of widgets with a list and creates a table
 *      to hold them within a new notebook page.
 * 
 *   Entry:
 *      page         - our page number
 *      widgets      - singly linked list containing widgets to be displayed
 *      first_widget - index in list of first widget to work with
 *      last_widget  - index in list of last widget to work with
 *      rows         - count of rows for the widget table
 *      notebook     - address of notebook widget
 *      tooltips     - address of tooltips data
 *      label        - optional string for tab label
 *
 *   Exit:
 *      A table of widgets is created and added to the notebook within
 *      a new page.
 *
 */
void add_widget_page_to_notebook(gint page, GSList * widgets, guint first_widget, guint last_widget,
				 guint rows, GtkNotebook * notebook, GtkTooltips * tooltips,
				 gchar * label)
{
	guint i;
	guint row;
	guint columns;
	GtkWidget *widget;
	GtkWidget *widget_table;

	/*
	 * The number of columns should be determined after making a pass
	 * through the descriptors and determining the indentation level. 
	 * We also have at least four columns (one for required field 
	 * marker (*), one for the field label, one for the widget, and
	 * one for any potential "More" button).
	 */

	columns = 4 + get_max_indentation_level(widgets, first_widget, last_widget);

	widget_table = gtk_table_new(rows, columns, FALSE);

	/*
	 * Add each widget with a text label in the first column. Start the
	 * widget at the indentation level and always end them at the last
	 * column.
	 */

	for (i = first_widget, row = 0; i <= last_widget; i++) {
		widget = g_slist_nth_data(widgets, i);

		if (widget != NULL)
			add_widget_to_notebook_table(GTK_TABLE(widget_table), widget, tooltips,
						     &row, columns);
	}

	gtk_widget_show(widget_table);

	gtk_container_add(GTK_CONTAINER(notebook), widget_table);
	gtk_container_set_border_width(GTK_CONTAINER(widget_table), 15);
	gtk_table_set_row_spacings(GTK_TABLE(widget_table), 10);
	gtk_table_set_col_spacings(GTK_TABLE(widget_table), 5);
	gtk_table_set_col_spacing(GTK_TABLE(widget_table), columns - 1, 0);

	if (label != NULL) {
		GtkWidget *tab_label;

		tab_label = gtk_label_new(label);
		gtk_widget_show(tab_label);

		gtk_notebook_set_tab_label(notebook, gtk_notebook_get_nth_page(notebook, page),
					   tab_label);
	}
}

/*
 *
 *   guint get_page_limits (GSList *, guint, guint, guint, guint *)
 *
 *   Description:
 *      This routine determines the number of rows for the table in
 *      a notebook page to hold a certain number of widgets.
 * 
 *   Entry:
 *      widgets      - singly linked list containing widgets to be displayed
 *      first_widget - index in list of first widget to work with
 *      actual_last  - index in list of last widget in the list
 *      rows         - address to write count of rows for the widget table
 *
 *   Exit:
 *      Returns the index of the last widget to fit and the number of
 *      table rows required.
 *
 */
guint get_page_limits(GSList * widgets, guint first_widget, guint actual_last,
		      guint max_widgets_per_page, guint * row_count)
{
	guint i;
	guint last_widget;
	guint max_widgets_this_page;
	GtkWidget *widget;

	max_widgets_this_page = MIN(max_widgets_per_page, (actual_last - first_widget + 1));

	for (i = first_widget, last_widget = first_widget, *row_count = 0;
	     *row_count < max_widgets_this_page; i++) {
		widget = g_slist_nth_data(widgets, i);

		if (GTK_IS_CLIST(widget))
			*row_count += CLIST_MAX_TABLE_ROWS;
		else
			*row_count += 1;

		if (*row_count <= max_widgets_per_page)
			last_widget = i;
	}

	return last_widget;
}

/*
 *
 *   GtkWidget* create_widget_notebook (GSList *, gchar *, guint)
 *
 *   Description:
 *      This routine takes a list of widgets and creates a notebook
 *      with the widgets organized by group and laid out within
 *      a table container widget. Additional pages and tables are 
 *      added to the notebook if necessary.
 * 
 *   Entry:
 *      widgets              - singly linked list containing widgets to be 
 *                             displayed
 *      label                - optional string for tab labels
 *      max_widgets_per_page - maximum number of widgets per page
 *      currpage             - page to be made current page
 *
 *   Exit:
 *      Notebook is created with one or more pages and the widgets
 *      organized within.
 *
 */
GtkWidget *create_widget_notebook(GSList * widgets, gchar * label, guint max_widgets_per_page,
				  gint currpage)
{
	guint page = 0;
	guint first = 0;
	guint last;
	guint row_count = 0;
	guint actual_last;
	GtkWidget *notebook;
	GtkTooltips *tooltips;

	notebook = gtk_notebook_new();
	tooltips = gtk_tooltips_new();

	gtk_notebook_set_tab_hborder(GTK_NOTEBOOK(notebook), 14);

	actual_last = g_slist_length(widgets) - 1;

	do {
		last =
		    get_page_limits(widgets, first, actual_last, max_widgets_per_page, &row_count);

		add_widget_page_to_notebook(page, widgets, first, last, row_count,
					    GTK_NOTEBOOK(notebook), tooltips, label);

		/* 
		 * If all the widgets have not been sunken into a page,
		 * setup to add the next set of widgets on another page.
		 */

		if (last < actual_last) {
			page += 1;
			first = last + 1;
		}

	} while (last < actual_last);

	gtk_notebook_set_page(GTK_NOTEBOOK(notebook), currpage);

	if (page == 0) {
		/*
		 * There is only one page. Don't show the tabs
		 * or the notebook border.
		 */
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), FALSE);
		gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);
	}

	return notebook;
}

/*
 *
 *   GSList *get_widget_notebook_page_children (GtkNotebook *, gint)
 *
 *   Description:
 *      This routine returns a list of widgets located on a given
 *      page of a widget notebook.
 * 
 *   Entry:
 *      notebook - the address of the widget notebook
 *      page     - the number of the page we want for the widget list
 *
 *   Exit:
 *      A singly linked-list is returned containing the widgets created
 *      within the requested page. The list should be freed with
 *      g_slist_free() when no longer needed by the caller.
 *
 */
GSList *get_widget_notebook_page_children(GtkNotebook * notebook, gint page)
{
	GSList *children = NULL;
	GtkWidget *table;

	table = gtk_notebook_get_nth_page(notebook, page);

	if (GTK_IS_TABLE(table)) {
		GList *list;
		GtkTableChild *table_child;

		list = GTK_TABLE(table)->children;

		while (list) {
			table_child = list->data;

			if (table_child)
				children = g_slist_append(children, table_child->widget);

			list = list->next;
		}
	}

	return children;
}

/*
 * The following routines are basically getter/setter functions
 * that are used by most of the option widget callbacks.
 */

inline void bind_options_callout_to_toplevel_widget(GtkWidget * widget,
						    options_callback_callout callout)
{
	GtkWidget *toplevel = gtk_widget_get_toplevel(widget);

	gtk_object_set_data(GTK_OBJECT(toplevel), "callback_callout", callout);
}

inline options_callback_callout retrieve_options_callout_from_toplevel_widget(GtkWidget * widget)
{
	GtkWidget *toplevel = gtk_widget_get_toplevel(widget);

	return (options_callback_callout) gtk_object_get_data(GTK_OBJECT(toplevel),
							      "callback_callout");
}
