/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: segoptions.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "segs.h"
#include "checks.h"
#include "discovery.h"
#include "commit.h"
#include "display.h"
#include "os2dlat.h"
#include "sn.h"
#include "segoptions.h"
#include "move.h"

#define  NEED_SOLARISX86_NAMES
#define  NEED_BSD_NAMES
#define  NEED_UNIXWARE_NAMES
#include "bsd.h"
#include "solarisX86.h"
#include "unixware.h"

static const struct seg_partition_type_info_s partition_types[] =
{
        {"FAT16", 6},
        {"HPFS",  7},
        {"NTFS",  7},
        {"Linux", 0x83},
        {"Linux Swap", 0x82},
        {"Linux LVM", 0x8e},
        {OTHER_TYPENAME, 0},
        {NULL, 0}                    /* end-of-list */
};


static const struct disk_type_info_s disk_types[] =
{
        {"Linux"},
        {"OS/2"},
        {NULL}                    /* end-of-list */
};



/*
 *  Returns TRUE if the specified Drive Letter is valid letter C-Z
 */
static boolean  isa_valid_drive_letter( char  *drive_letter )
{

        LOG_ENTRY();

        if ( drive_letter ) {

                if ( (*(drive_letter) >= 'C') && ( *(drive_letter) <= 'Z') ) {
                        LOG_EXIT_BOOL(TRUE);
                        return TRUE;
                }
        }

        LOG_EXIT_BOOL(FALSE);
        return FALSE;
}

/*
 *  Returns the 1st object found in a LIST.
 */
static  storage_object_t *  get_first_object_in_list( list_anchor_t  list )
{
        storage_object_t *object_from_list = NULL;

        LOG_ENTRY();

        if (EngFncs->list_count(list) > 0) {
                object_from_list = EngFncs->first_thing(list,NULL);
        }

        LOG_EXIT_PTR(object_from_list);
        return  object_from_list;
}


/*
 *  Called by init_task() to allocate the create option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_move_option_descriptors( task_context_t * context )
{
        int rc;

        LOG_ENTRY();
        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to initialize the MOVE options in the task context
 *  for the specified LOGICALDISK.
 */
static int initialize_move_option_descriptors( DISKSEG *seg, task_context_t * context)
{
        int rc;

        LOG_ENTRY();
        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to initialize the ASSIGN options in the task context
 *  for the specified LOGICALDISK.
 */
static int initialize_assign_option_descriptors( LOGICALDISK *ld, task_context_t * context)
{
        int rc=EINVAL;


        LOG_ENTRY();

        if ( ld && context ) {

                // there should be exactly SEG_ASSIGN_OPTION_COUNT options
                if ( context->option_descriptors->count == SEG_ASSIGN_OPTION_COUNT) {

                        // Disk Type by name
                        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].flags = 0;
                        strcpy (context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].value.s, "Linux");

                        // Disk Name
                        memset( context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].value.s, 0, DISK_NAME_SIZE );
                        context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;

                        rc = 0;

                }
                else {
                        LOG_ERROR("error, wrong number of assign options.\n");
                        rc = EINVAL;
                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  Called to initialize the CREATE segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_create_option_descriptors( LOGICALDISK *ld, DISKSEG *freespace, task_context_t * context)
{
        int                 rc=EINVAL;
        lba_t               Start;
        lba_t               End;
        sector_count_t      MaxSize;
        sector_count_t      CylinderSize=0;
        DISK_PRIVATE_DATA  *disk_pdata;
        boolean             OS2_Disk = FALSE;
        DISKSEG            *mbr;

        LOG_ENTRY();

        disk_pdata = get_disk_private_data( ld );
        if (disk_pdata) {
                if (disk_pdata->signature == DOS_SEG_MGR_PDATA_SIGNATURE) {

                        // we should be able to get the cylinder size if there are no problems with disk geometry
                        CylinderSize = get_cylinder_size(ld);
                        if (CylinderSize ) {

                                // there should be exactly SEG_CREATE_OPTION_COUNT options
                                if ( context->option_descriptors->count == SEG_CREATE_OPTION_COUNT) {
                                        rc = 0;
                                }
                                else {
                                        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                                }

                        }
                        else {
                                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
                        }

                }
        }

        if (rc == 0) {

                // check if disk has dlats
                if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {
                        OS2_Disk = TRUE;
                }

                // determine defaults based on geometry
                Start    = freespace->start;
                End      = rounddown_to_cylinder_boundary( ld, freespace->start + freespace->size ) - 1;
                MaxSize  = End - Start + 1;

                // if MAX_USEABLE_SIZE is at least 1 cylinder in size ... well minus a track ...
                // then things are Ok ... init the default options.  this lets me pickup the
                // first cylinder on the disk that has an mbr in the 1st track. it also lets
                // me pickup freespace at the front of an extd partition when the 1st sector
                // is being used to anchor the ebr chain.
                if ( MaxSize >= CylinderSize-disk_pdata->geometry.sectors_per_track) {

                        if (rc == 0) {

                                // Segment Size
                                if ( MaxSize >= CylinderSize ) {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = 0;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = CylinderSize;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = CylinderSize;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = MaxSize;
                                }
                                else {  // cuz we track allign start of partition, the freespace can be less than a cylinder
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = 0;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = MaxSize;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 0;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = MaxSize;
                                }

                                // Sector Offset to start of Partition
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].flags = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->min.ui64       = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64       = MaxSize-CylinderSize;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->increment.ui64 = CylinderSize;


                                // Partition Type by name
                                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
                                if (OS2_Disk == TRUE) {
                                        strcpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s, "FAT16");
                                }
                                else {
                                        strcpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s, "Linux");
                                }

                                // Partition Type
                                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].flags = EVMS_OPTION_FLAGS_ADVANCED;
                                if (OS2_Disk == TRUE) {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.ui8 = 0x06;
                                }
                                else {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.ui8 = 0x83;
                                }

                                // Make Partition Bootable ?
                                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].flags = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.b = FALSE;

                                // Make Primary Partition ?
                                mbr = get_mbr_from_seglist(ld->parent_objects);

                                if ( disk_has_extended_partition( ld ) == TRUE ) {

                                        if ( freespace->start < disk_pdata->extd_partition_lba ) {
                                                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.b = TRUE;
                                        }
                                        else if ( seg_is_within_or_adjacant_to_extended_partition( ld, freespace ) == TRUE ) {
                                                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.b = FALSE;
                                        }
                                        else {
                                                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.b = TRUE;
                                        }

                                }
                                else {  // no extd partition yet. check if mbr ptable has room for
                                        // another primary partition record. if not ... then we will
                                        // create the extd partition.
                                        if (get_first_unused_ptable_entry(ld->parent_objects, mbr)==-1) {
                                                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.b = FALSE;
                                        }
                                        else {
                                                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.b = TRUE;
                                        }
                                }

                                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].flags = 0;


                                // Partition Name
                                if (OS2_Disk == TRUE) {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
                                        memset( context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].value.s, 0, PARTITION_NAME_SIZE );
                                }
                                else {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
                                }

                                // Volume Name
                                if (OS2_Disk == TRUE) {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE | EVMS_OPTION_FLAGS_NOT_REQUIRED;
                                        memset(context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].value.s, 0, VOLUME_NAME_SIZE);
                                }
                                else {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
                                }

                                // OS/2 Drive Letter
                                if (OS2_Disk == TRUE) {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE | EVMS_OPTION_FLAGS_NOT_REQUIRED;
                                        memset( context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s, 0, DRIVE_LETTER_SIZE);
                                }
                                else {
                                        strcpy(context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s, " ");
                                        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                                }

                                rc = 0;
                        }

                }
                else {
                        rc = EINVAL;
                        LOG_ERROR("error trying to create a segment out of something smaller than a cylinder.\n");
                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to initialize the EXPAND segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_expand_option_descriptors( LOGICALDISK *ld, DISKSEG *freespace, task_context_t * context)
{
        int                rc=EINVAL;
        sector_count_t     MaxSize=0;
        sector_count_t     MinSize=0;
        sector_count_t     CylinderSize=0;
        DISK_PRIVATE_DATA *disk_pdata;


        LOG_ENTRY();

        disk_pdata = get_disk_private_data( ld );
        if (disk_pdata) {

                if (disk_pdata->signature == DOS_SEG_MGR_PDATA_SIGNATURE) {

                        // we should be able to get the cylinder size if there are no problems with disk geometry
                        CylinderSize = get_cylinder_size(ld);
                        if (CylinderSize ) {

                                // there should be exactly SEG_EXPAND_OPTION_COUNT options
                                if ( context->option_descriptors->count == SEG_EXPAND_OPTION_COUNT) {

                                        rc = 0;

                                }
                                else {
                                        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                                }

                        }
                        else {
                                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
                        }

                }
        }

        if (rc == 0) {

                // make sure we can actually expand the DATA segment by at least 1 cylinder
                if ( freespace->size >= CylinderSize ) {

                        LOG_DEBUG("freespace= %s   size= %"PRIu64"\n", freespace->name, freespace->size );

                        // get minimum and maximum sizes for expanding this DATA segment
                        MaxSize  = freespace->size - (freespace->size % CylinderSize);
                        MinSize  = CylinderSize;

                        LOG_DEBUG("Max= %"PRIu64"   Min= %"PRIu64"\n", MaxSize, MinSize );

                        if ( MinSize >= CylinderSize ) {

                                // ask engine how much we can expand by
				rc = EngFncs->can_expand_by(context->object, &MaxSize);

				while (rc == EAGAIN) {

					// The Engine said, "Try again." and
					// updated the MaxSize to an accptable
					// size.  Round down to a cylinder
					// boundary and try again.

					MaxSize = rounddown_to_cylinder_boundary(ld, MaxSize);
					rc = EngFncs->can_expand_by(context->object, &MaxSize);
				}

                                if ((rc == 0) && (MaxSize > 0)) {

                                        context->option_descriptors->count = SEG_EXPAND_OPTION_COUNT;
                                        LOG_DEBUG("Setting constraints ... Max= %"PRId64"   Min= %"PRId64"\n", MaxSize, MinSize );
                                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->min.ui64       = MinSize;
                                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = CylinderSize;
                                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].value.ui64 = MaxSize;
                                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].flags = 0;

                                        rc = 0;
                                }

                        }
                }
                else {
                        rc = EINVAL;
                        LOG_ERROR("error trying to expand segment into freespace < 1 cylinder.\n");
                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  Called to initialize the SHRINK segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_shrink_option_descriptors( LOGICALDISK *ld, DISKSEG *seg, task_context_t * context)
{
        int                rc=EINVAL;
        sector_count_t     MaxSize=0;
        lba_t              MaxEnd=0;
        lba_t              MinEnd=0;
        sector_count_t     MinSize=0;
        sector_count_t     CylinderSize=0;
        DISK_PRIVATE_DATA *disk_pdata;
        sector_count_t     BiggestDelta=0;
        sector_count_t     SmallestDelta=0;

        LOG_ENTRY();

        disk_pdata = get_disk_private_data( ld );
        if (disk_pdata) {

                if (disk_pdata->signature == DOS_SEG_MGR_PDATA_SIGNATURE) {

                        // we should be able to get the cylinder size if there are no problems with disk geometry
                        CylinderSize = get_cylinder_size(ld);
                        if (CylinderSize ) {

                                // there should be exactly SEG_SHRINK_OPTION_COUNT options
                                if ( context->option_descriptors->count == SEG_SHRINK_OPTION_COUNT) {
                                        rc = 0;
                                }
                                else {
                                        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                                }

                        }
                        else {
                                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
                        }

                }
        }

        if (rc == 0) {

                // we shrink partition sizes ... keeping cylinder allignment
                CylinderSize = get_cylinder_size( ld );

                // get minimum and maximum sizes for shrinking this DATA segment
                if ( seg->size > CylinderSize ) {

                        MaxEnd   = roundup_to_cylinder_boundary( ld, seg->start + (seg->size - CylinderSize) - 63 );
                        MaxSize  = MaxEnd - seg->start + 1;

                        MinEnd   = roundup_to_cylinder_boundary(ld, seg->start + 63 );
                        MinSize  = MinEnd - seg->start + 1;

                        BiggestDelta  = seg->size - MinSize;
                        SmallestDelta = seg->size - MaxSize;

                        // ask engine how much we can shrink by

                        rc = EngFncs->can_shrink_by( seg, &BiggestDelta );

			while (rc == EAGAIN) {

				// The Engine said, "Try again." and
				// updated the BiggestDelta to an accptable
				// size.  Round down to a cylinder
				// boundary and try again.

				BiggestDelta = rounddown_to_cylinder_boundary( ld, BiggestDelta );
				rc = EngFncs->can_shrink_by( seg, &BiggestDelta );
			}

                }
                else {  // cant shrink this segment !
                        rc = EINVAL;
                }


                // make sure we can actually shrink the DATA segment
                if ( (rc == 0) && (BiggestDelta >  SmallestDelta) ) {

                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->min.ui64       = SmallestDelta;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->max.ui64       = BiggestDelta;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = CylinderSize;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].flags = 0;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].value.ui64 = SmallestDelta;

                        rc = 0;

                }


        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called by init_task() to allocate the assign option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_assign_option_descriptors( task_context_t * context )
{
        int rc = EINVAL;


        LOG_ENTRY();

        if (context) {

                context->option_descriptors->count = SEG_ASSIGN_OPTION_COUNT ;      // must set count


                // Disk Type
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint.list =
                EngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t) * (sizeof(disk_types)/sizeof(struct disk_type_info_s)));

                if (context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint.list) {
                        int i=0;

                        while (disk_types[i].name) {
                                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint.list->value[i].s = EngFncs->engine_strdup(disk_types[i].name);
                                i++;
                        }

                        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint.list->count = i;
                }
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint_type = EVMS_Collection_List;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].help = NULL;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_TYPENAME_NAME );
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].min_len = 1;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].max_len = MAX_TYPENAME_SIZE;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].tip = EngFncs->engine_strdup( _("Choose between OS/2 and Linux disk types. An OS/2 disk will maintain extra metadata that is used by OS/2."));
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].title = EngFncs->engine_strdup( _("Disk Type"));
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].type = EVMS_Type_String;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].value.s = EngFncs->engine_alloc (MAX_TYPENAME_SIZE+1);

                if (context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].value.s)
                        strcpy (context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].value.s, "Linux");
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }


                // Disk Name
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].constraint_type = EVMS_Collection_None;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].help = NULL;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].name = EngFncs->engine_strdup( SEG_ASSIGN_OPTION_DISKNAME_NAME );
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].min_len = 1;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].max_len = DISK_NAME_SIZE;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].tip = EngFncs->engine_strdup( _("OS/2 requires that you choose a name for the disk that is no more than 20 characters."));
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].title = EngFncs->engine_strdup( _("Disk Name") );
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].type = EVMS_Type_String;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].value.s = EngFncs->engine_alloc(DISK_NAME_SIZE+1);
                if (context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].value.s) {
                        memset(context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].value.s, 0, DISK_NAME_SIZE );
                        rc = 0;
                }
                else {
                        rc = ENOMEM;
                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called by init_task() to allocate the create option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_create_option_descriptors( task_context_t * context )
{
        int rc = EINVAL;


        LOG_ENTRY();

        if (context) {

                context->option_descriptors->count = SEG_CREATE_OPTION_COUNT ;      // must set count


                // Segment Size
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                if (context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range==NULL) {
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint_type                  = EVMS_Collection_Range;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = 0;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = 0xffffffff;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 1;

                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_SIZE_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].tip = EngFncs->engine_strdup( _("This option allows you to set the size of the segment that you are creating.") );
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].title = EngFncs->engine_strdup( _("Size") );
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = 0;

                // Sector Offset to start of Partition
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                if (context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range==NULL) {
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint_type                  = EVMS_Collection_Range;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->min.ui64       = 0;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64       = 1;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->increment.ui64 = 1;


                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_OFFSET_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].tip =
			EngFncs->engine_strdup( _("Use this option if you don't want the segment you are creating to start at the beginning of freespace. "
						  "It allows you to specifiy the number of sectors to skip before creating the segment.") );
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].title = EngFncs->engine_strdup( _("Offset"));
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].type = EVMS_Type_Unsigned_Int64;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].unit = EVMS_Unit_Sectors;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = 0;

                // Partition Type
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint.list =
                EngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t) * (sizeof(partition_types)/sizeof(struct seg_partition_type_info_s)));

                if (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint.list) {
                        int i=0;

                        while (partition_types[i].name) {
                                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint.list->value[i].s = EngFncs->engine_strdup(partition_types[i].name);
                                i++;
                        }

                        context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint.list->count = i;
                }
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint_type = EVMS_Collection_List;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_TYPENAME_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].min_len = 1;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].max_len = MAX_TYPENAME_SIZE;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].tip = EngFncs->engine_strdup( _("Select from a list of common partition types"));
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].title = EngFncs->engine_strdup( _("Partition Type"));
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].type = EVMS_Type_String;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s = EngFncs->engine_alloc (MAX_TYPENAME_SIZE+1);

                if (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s)
                        strcpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s, "Linux");
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                // Partition Type
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].constraint_type = EVMS_Collection_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_TYPE_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].tip = EngFncs->engine_strdup( _("The type of partition you are creating, e.g. 6 = FAT16") );
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].title = EngFncs->engine_strdup( _("Partition Type Id") );
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].type = EVMS_Type_Unsigned_Int8;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].format = EVMS_Format_Hex;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.ui8 = 0x83;

                // Make Partition Bootable ?
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].constraint_type = EVMS_Collection_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_BOOTABLE_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].tip =
			EngFncs->engine_strdup( _("Choose Yes if the primary partition should be marked active so that you can boot from it.") );
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].title = EngFncs->engine_strdup( _("Bootable") );
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].type = EVMS_Type_Boolean;
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.b = FALSE;

                // Make Primary Partition ?
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].constraint_type = EVMS_Collection_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_PRIMARY_PARTITION_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].tip =
			EngFncs->engine_strdup( _("This option allows you to choose between creating a logical or primary partition. "
						  "Choose Yes if you would like a primary partition created.") );
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].title = EngFncs->engine_strdup( _("Primary Partition") );
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].type = EVMS_Type_Boolean;
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.b = FALSE;

                // Partition Name
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].constraint_type = EVMS_Collection_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_PARTITIONNAME_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].min_len = 1;
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].max_len = PARTITION_NAME_SIZE;
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].tip = EngFncs->engine_strdup( _("OS/2 partition name") );
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].title = EngFncs->engine_strdup( _("Partition Name") );
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].type = EVMS_Type_String;
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].value.s = EngFncs->engine_alloc(PARTITION_NAME_SIZE+1);
                if (context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].value.s) {
                        memset( context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].value.s, 0, PARTITION_NAME_SIZE );
                }
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                // Volume Name
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].constraint_type = EVMS_Collection_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_VOLUMENAME_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].min_len = 1;
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].max_len = VOLUME_NAME_SIZE;
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].tip = EngFncs->engine_strdup( _("Name of OS/2 volume this segment will reside in") );
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].title = EngFncs->engine_strdup( _("Volume Name") );
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].type = EVMS_Type_String;
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].value.s = EngFncs->engine_alloc(VOLUME_NAME_SIZE+1);
                if (context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].value.s) {
                        memset( context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].value.s, 0, VOLUME_NAME_SIZE);
                }
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }


                // OS/2 Drive Letter
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].constraint_type = EVMS_Collection_None;

                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].constraint_type = EVMS_Collection_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_DRIVELETTER_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].min_len = DRIVE_LETTER_SIZE;
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].max_len = DRIVE_LETTER_SIZE;
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].tip = EngFncs->engine_strdup( _("OS/2 drive letter for this new segment") );
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].title = EngFncs->engine_strdup( _("Drive Letter") );
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].type = EVMS_Type_String;
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s = EngFncs->engine_alloc(DRIVE_LETTER_SIZE+1);
                if (context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s) {
                        memset( context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s, 0, DRIVE_LETTER_SIZE);
                }
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Called by init_task() to allocate the expand option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int  allocate_expand_option_descriptors( task_context_t * context )
{
        int rc = EINVAL;
        storage_object_t *obj;
        LOGICALDISK      *ld=NULL;
        DISKSEG          *freespace=NULL;
        sector_count_t    CylinderSize;


        LOG_ENTRY();

        context->option_descriptors->count = 0;

        obj = context->object;
        if ( obj ) {

                if (obj->object_type == SEGMENT) {

                        if (obj->data_type == DATA_TYPE) {

                                ld = get_logical_disk( (DISKSEG *) obj );
                                if (ld) {

                                        // we increment partition sizes ... 1 cylinder at a time ... due to forced cyl allignment
                                        CylinderSize = get_cylinder_size(ld);

                                        // DATA segment better have freespace following it.
                                        freespace =  get_freespace_following_seg( obj );
                                        if (freespace == NULL) {
                                                rc = EINVAL;
                                                LOG_EXIT_INT(rc);
                                                return rc;
                                        }

                                        // make sure we can actually expand the DATA segment by at least 1 cylinder
                                        if ( freespace->size >= CylinderSize ) {

                                                context->option_descriptors->count = SEG_EXPAND_OPTION_COUNT;

                                                // Expanded segment size delta
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.list = NULL;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                                                if (context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range==NULL) {
                                                        LOG_EXIT_INT(ENOMEM);
                                                        return ENOMEM;
                                                }
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint_type                  = EVMS_Collection_Range;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].flags = 0;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].help = NULL;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].name = EngFncs->engine_strdup( SEG_EXPAND_OPTION_SIZE_NAME );
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].tip =
							EngFncs->engine_strdup( _("Use this option to specify how much space to add to the segment. "
										  "Remember that segments grow by cylinder size amounts.") );
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].title = EngFncs->engine_strdup( _("Additional Size") );
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;

                                                rc = 0;
                                        }

                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;

}


/*
 *  Called by init_task() to allocate the shrink option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_shrink_option_descriptors( task_context_t * context )
{
        int                rc = EINVAL;
        storage_object_t  *obj;
        LOGICALDISK       *ld;
        sector_count_t     CylinderSize;
        DISK_PRIVATE_DATA *disk_pdata;

        LOG_ENTRY();

        context->option_descriptors->count = 0;

        obj = context->object;

        if ( obj ) {

                if (obj->object_type == SEGMENT) {

                        if (obj->data_type == DATA_TYPE) {

                                ld = get_logical_disk( (DISKSEG *) obj );

                                disk_pdata = get_disk_private_data(ld);

                                if (disk_pdata) {

                                        // we shrink partition sizes ... 1 cylinder at a time ... due to forced cyl allignment
                                        CylinderSize = disk_pdata->geometry.sectors_per_track * disk_pdata->geometry.heads;

                                        // make sure we can actually shrink the DATA segment while falling
                                        // on a cylinder boundary.
                                        if ( obj->size > CylinderSize ) {

                                                context->option_descriptors->count = SEG_SHRINK_OPTION_COUNT ;

                                                // Shrunken segment size
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.list = NULL;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                                                if (context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range==NULL) {
                                                        LOG_EXIT_INT(ENOMEM);
                                                        return ENOMEM;
                                                }
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].flags = 0;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].help = NULL;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].name = EngFncs->engine_strdup( SEG_SHRINK_OPTION_SIZE_NAME );
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].tip =
							EngFncs->engine_strdup( _("Use this option to specify how much space to remove from the segment. "
										  "Remember that segments shrink by cylinder size amounts.") );
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].title = EngFncs->engine_strdup( _("Shrink by Size") );
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;

                                                rc = 0;
                                        }

                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to set and validate an option for a CREATE task.
 */
static int set_assign_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int rc = EINVAL;


        LOG_ENTRY();

        switch (index) {

        case SEG_ASSIGN_OPTION_TYPENAME_INDEX:

                if ( strlen(value->s) > 0 ) {

                        strcpy( context->option_descriptors->option[index].value.s, value->s);

                        if ( strncmp(value->s, "OS/2",4)==0) {
                                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].flags = 0;
                        }
                        else {
                                context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].flags=EVMS_OPTION_FLAGS_INACTIVE;
                        }
                        *effect |= EVMS_Effect_Reload_Options;
                        rc = 0;

                }
                else {
                        rc = EINVAL;
                }
                break;


        case SEG_ASSIGN_OPTION_DISKNAME_INDEX:

                if ( (strlen(value->s) > 0) &&
                     (strlen(value->s) <= DISK_NAME_SIZE) ) {

                        rc = EngFncs->validate_name( value->s);

                        if (rc == 0) {
                                strcpy( context->option_descriptors->option[index].value.s, value->s);
                        }
                }
                else if ( strlen(value->s) > DISK_NAME_SIZE ) {
                        char temp_name[DISK_NAME_SIZE+1];

                        // get shorter disk name
                        strncpy(temp_name, value->s, DISK_NAME_SIZE );

                        // try and validate a shortened name
                        if ( EngFncs->validate_name( temp_name) == 0) {
                                strcpy( context->option_descriptors->option[index].value.s, temp_name );
                                *effect |= EVMS_Effect_Inexact;
                                rc = 0;
                        }
                        else {
                                rc = EINVAL;
                        }
                }
                else {
                        rc = EINVAL;
                }
                break;


        default:
                LOG_ERROR("index is unknown or unsupported\n");
                rc = EINVAL;
                break;
        }


        LOG_EXIT_INT(rc);
        return rc;
}

/*
 * Scans partition_types array for a name matching given
 * name in order to return corresponding partition type
 * id number.
 */
static u_int8_t get_partition_type_number_from_name (char *name)
{
        int      i=0;
        u_int8_t type=0;

        while (partition_types[i].name) {
                if (strcasecmp (partition_types[i].name, name) == 0) {
                        type = partition_types[i].number;
                        break;
                }

                i++;
        }

        return type;
}

/*
 * Scans partition_types array for a number matching given
 * number in order to return corresponding partition type
 * name.
 */
static char* get_partition_type_name_from_number (u_int8_t number)
{
        int   i=0;
        char *name=OTHER_TYPENAME;

        while (partition_types[i].name) {
                if (partition_types[i].number == number) {
                        name = partition_types[i].name;
                        break;
                }

                i++;
        }

        return name;
}

/*
 *  Called to set and validate an option for a CREATE task.
 */
static int set_create_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int                rc = EINVAL;
        storage_object_t  *obj=NULL;
        LOGICALDISK       *ld=NULL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        sector_count_t     MaxSize = 0;
        sector_count_t     size = 0;
        lba_t              start_lba = 0;
        lba_t              end_lba = 0;
        lba_t              rounded_end_lba = 0;
        sector_count_t     CylinderSize=0;


        LOG_ENTRY();

        obj = get_first_object_in_list( context->selected_objects );
        if (obj) {

                if (obj->data_type == META_DATA_TYPE) {
                        rc = ENODEV;
                        LOG_ERROR("error, i do not create segs from metadata type segments\n");
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                // get disk info
                ld = get_logical_disk(obj);
                if (ld == NULL) {
                        rc = ENODEV;
                        LOG_ERROR("error, no disk info available\n");
                        LOG_EXIT_INT(rc);
                        return rc;
                }
                else {
                        disk_pdata = get_disk_private_data(ld);
                        if (disk_pdata == NULL) {
                                rc = ENODATA;
                                LOG_ERROR("error, no disk private data available\n");
                                LOG_EXIT_INT(rc);
                                return rc;
                        }
                }


                CylinderSize = get_cylinder_size(ld);

                start_lba    = obj->start;
                end_lba      = obj->start + obj->size - 1;
                size         = obj->size;

                rounded_end_lba = rounddown_to_cylinder_boundary( ld, end_lba+1 ) - 1;

                MaxSize = rounded_end_lba - start_lba + 1;

        }
        else {
                rc = ENODEV;
                LOG_ERROR("error, no disk segment found in the selected_objects list\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        switch (index) {

        case SEG_CREATE_OPTION_SIZE_INDEX:
                {

                        sector_count_t segoffset = context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64;

                        // SIZE == 0 ... ERROR
                        if ( value->ui64 == 0) {
                                *effect |= EVMS_Effect_Inexact ;
                                rc = EINVAL;
                        }
                        // if OFFSET + SIZE < MAX_USEABLE_FREESPACE_SIZE ... then SIZE is OK
                        else if ( value->ui64+segoffset <= MaxSize ) {

                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                                *effect |= EVMS_Effect_Reload_Options;
                                context->option_descriptors->option[index].value.ui64 = value->ui64;
                                rc = 0;

                        }
                        // user is trying to create something that runs off end
                        // of the useable area of the freespace segment.
                        else {

                                if (segoffset > 0) { // OFFSET and SIZE both need to be examined
                                                     // to determine how to fix things up.

                                        if (segoffset > obj->size) {     // OFFSET is toooo big ... ERROR

                                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = MaxSize - value->ui64;
                                                context->option_descriptors->option[index].value.ui64 = value->ui64;

                                                *effect |= EVMS_Effect_Reload_Options ;
                                                rc = 0;

                                        }
                                        else {   // OFFSET is OK ... perhaps big ... so reduce SIZE
                                                 // and let the user figure things out.

                                                sector_count_t newsize = ((obj->size - segoffset)/CylinderSize)*CylinderSize;

                                                context->option_descriptors->option[index].constraint.range->max.ui64 = newsize;
                                                context->option_descriptors->option[index].value.ui64 = newsize;
                                                *effect |= EVMS_Effect_Reload_Options ;
                                                rc = 0;
                                        }

                                }
                                else {  // OFFSET == 0 ... so SIZE is the only option we need
                                        // to adjust downward for user.
                                        context->option_descriptors->option[index].value.ui64 = MaxSize;
                                        *effect |= EVMS_Effect_Inexact;
                                        rc = 0;
                                }

                        }


                }

                break;

        case SEG_CREATE_OPTION_OFFSET_INDEX:

                {
                        sector_count_t segsize = context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64;

                        // if OFFSET + SIZE < MAX_USEABLE_FREESPACE_SIZE ... then OFFSET is OK
                        if ( value->ui64+segsize <= MaxSize ) {

                                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                                context->option_descriptors->option[index].value.ui64 = value->ui64;
                                *effect |= EVMS_Effect_Reload_Options ;
                                rc = 0;
                        }
                        else {

                                context->option_descriptors->option[index].constraint.range->max.ui64 = MaxSize - segsize;
                                context->option_descriptors->option[index].value.ui64 = MaxSize - segsize;
                                *effect |= EVMS_Effect_Reload_Options  ;
                                rc = 0;
                        }

                }

                break;

        case SEG_CREATE_OPTION_TYPENAME_INDEX:
                if (strcasecmp (value->s, OTHER_TYPENAME) == 0)
                        rc = 0;
                else {
                        u_int8_t type;

                        type = get_partition_type_number_from_name (value->s);

                        if (type) {
                                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.ui8 = type;
                                *effect |= EVMS_Effect_Reload_Options;
                                rc = 0;
                        }
                }

                if (rc == 0)
                        strncpy (context->option_descriptors->option[index].value.s, value->s, MAX_TYPENAME_SIZE);

                break;

        case SEG_CREATE_OPTION_TYPE_INDEX:
                if (value->uc != 0) {
                        context->option_descriptors->option[index].value.ui8 = value->ui8;
                        rc = 0;
                }
                else {
                        if (disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {
                                context->option_descriptors->option[index].value.ui8 = 6;
                        }
                        else {
                                context->option_descriptors->option[index].value.ui8 = 0x83;
                        }
                        *effect |= EVMS_Effect_Inexact ;
                        rc = 0;
                }

                if (rc == 0) {
                        char *name;

                        name = get_partition_type_name_from_number (context->option_descriptors->option[index].value.ui8);
                        strncpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s, name, MAX_TYPENAME_SIZE);
                        *effect |= EVMS_Effect_Reload_Options;
                }
                break;

        case SEG_CREATE_OPTION_BOOTABLE_INDEX:
                if ( ((value->uc == TRUE)||(value->uc==FALSE)) &&
                     ( context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.b == TRUE )) {
                        context->option_descriptors->option[index].value.b = value->b;
                        rc = 0;
                }
                else {
                        context->option_descriptors->option[index].value.b = FALSE;
                        *effect |= EVMS_Effect_Inexact;
                        rc = 0;
                }
                break;

        case SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX:

                if ((value->b == TRUE)||(value->b == FALSE)) {

                        if ( ( value->b == FALSE ) &&
                             ( disk_has_extended_partition( ld ) == TRUE ) &&
                             ( seg_is_within_or_adjacant_to_extended_partition( ld, obj ) == FALSE )) {

                                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.b = FALSE;
                                context->option_descriptors->option[index].value.b = FALSE;
                                *effect |= EVMS_Effect_Reload_Options ;
                                rc = 0;

                        }
                        else {

                                // dont let them try to create primary if mbr has no unused records
                                // and dont let them try to create a primary if freespace falls
                                // within the extd partition.
                                if (value->b==TRUE) {

                                        DISKSEG *mbr = get_mbr_from_seglist(ld->parent_objects);

                                        if (mbr) {

                                                if ( get_first_unused_ptable_entry(ld->parent_objects, mbr) == -1) {

                                                        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.b = FALSE;
                                                        context->option_descriptors->option[index].value.b = FALSE;
                                                        *effect |= EVMS_Effect_Reload_Options ;
                                                        rc = 0;

                                                }
                                                else {  // ok have room for primary so check if it lays
                                                        // within the extd partition.
                                                        if (disk_has_extended_partition( ld ) == TRUE) {

                                                                if ( ( obj->start >= disk_pdata->extd_partition_lba ) &&
                                                                     ( obj->start <= disk_pdata->extd_partition_end_lba ) ) {
                                                                        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.b = FALSE;
                                                                        context->option_descriptors->option[index].value.b = FALSE;
                                                                        *effect |= EVMS_Effect_Reload_Options ;
                                                                        rc = 0;
                                                                }
                                                                else {
                                                                        rc = 0;
                                                                }
                                                        }
                                                        else {
                                                                rc = 0;
                                                        }

                                                }
                                        }
                                        else {
                                                context->option_descriptors->option[index].value.b = FALSE;
                                                *effect |= EVMS_Effect_Inexact;
                                                rc = 0;
                                        }

                                }
                                else {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.b = FALSE;
                                        *effect |= EVMS_Effect_Reload_Options;
                                        rc = 0;
                                }

                        }

                        if (rc==0) {
                                context->option_descriptors->option[index].value.b = value->b;
                        }

                }
                else {  // invalid option value ... not TRUE or FALSE ... just do
                        // some fixups for recovery attempt.

                        DISKSEG *mbr = get_mbr_from_seglist(ld->parent_objects);

                        if (mbr) {

                                // not bootable
                                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.b = FALSE;

                                // if it can fit in mbr ptable then make_primary == true
                                if ( get_first_unused_ptable_entry(ld->parent_objects, mbr) == -1) {
                                        context->option_descriptors->option[index].value.b = FALSE;
                                }
                                else {
                                        context->option_descriptors->option[index].value.b = TRUE;
                                }

                                // reload options and good rc
                                *effect |= EVMS_Effect_Reload_Options ;
                                rc = 0;

                        }
                        else {
                                rc = EINVAL;  // no mbr for disk ... ERROR ... fail
                        }

                }
                break;

        case SEG_CREATE_OPTION_PARTITIONNAME_INDEX:

                if ( (strlen(value->s) > 0) &&
                     (strlen(value->s) <= PARTITION_NAME_SIZE) ) {

                        char pname[PARTITION_NAME_SIZE+12];

                        strcpy(pname, "os2_seg_");
                        strncat(pname, value->s, PARTITION_NAME_SIZE );

                        rc = EngFncs->validate_name( pname );

                        if (rc == 0) {
                                strcpy( context->option_descriptors->option[index].value.s, value->s);
                        }

                }
                else if ( strlen(value->s) > PARTITION_NAME_SIZE ) {
                        char  temp_name[PARTITION_NAME_SIZE+1];

                        // shorten name to allowed length
                        strncpy(temp_name, context->option_descriptors->option[index].value.s, PARTITION_NAME_SIZE );

                        // try and validate a shortened name
                        if ( EngFncs->validate_name( temp_name) == 0) {
                                strcpy( context->option_descriptors->option[index].value.s, temp_name );
                                *effect |= EVMS_Effect_Inexact;
                                rc = 0;
                        }
                        else {
                                rc = EINVAL;
                        }

                }
                else {
                        memset( context->option_descriptors->option[index].value.s, 0, PARTITION_NAME_SIZE );
                        rc = EINVAL;
                }
                break;

        case SEG_CREATE_OPTION_VOLUMENAME_INDEX:
                if ( (strlen(value->s) > 0) &&
                     (strlen(value->s) <= VOLUME_NAME_SIZE) ) {

                        rc = EngFncs->validate_name( value->s);

                        if (rc == 0) {
                                strcpy( context->option_descriptors->option[index].value.s, value->s);
                        }
                        else {
                                memset( context->option_descriptors->option[index].value.s, 0, VOLUME_NAME_SIZE );
                        }
                }
                else if ( strlen(value->s) > VOLUME_NAME_SIZE ) {
                        char  temp_name[VOLUME_NAME_SIZE+1];

                        // shorten name to allowed length
                        strncpy(temp_name, context->option_descriptors->option[index].value.s, VOLUME_NAME_SIZE );

                        // try and validate a shortened name
                        if ( EngFncs->validate_name( temp_name) == 0) {
                                strcpy( context->option_descriptors->option[index].value.s, temp_name );
                                *effect |= EVMS_Effect_Inexact;
                                rc = 0;
                        }
                        else {
                                memset( context->option_descriptors->option[index].value.s, 0, VOLUME_NAME_SIZE);
                                rc = EINVAL;
                        }
                }
                else {
                        memset( context->option_descriptors->option[index].value.s, 0, VOLUME_NAME_SIZE);
                        rc = 0;
                }
                break;

        case SEG_CREATE_OPTION_DRIVELETTER_INDEX:

                if ( strlen(value->s) == 0 ) {
                        memset( context->option_descriptors->option[index].value.s, 0, DRIVE_LETTER_SIZE);
                        rc = 0;
                }
                else if ( strlen(value->s) > DRIVE_LETTER_SIZE ) {
                        memset( context->option_descriptors->option[index].value.s, 0, DRIVE_LETTER_SIZE);
                        rc = EINVAL;
                }
                else {

                        if ( isa_valid_drive_letter(value->s)==TRUE) {
                                strncpy( context->option_descriptors->option[index].value.s, value->s, 1);
                                rc = 0;
                        }
                        else {
                                memset( context->option_descriptors->option[index].value.s, 0, DRIVE_LETTER_SIZE);
                                rc = EINVAL;
                        }

                }
                break;

        default:
                LOG_ERROR("error, index is unknown or unsupported\n");
                rc = EINVAL;
                break;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to set and validate an option for a EXPAND task.
 */
static int set_expand_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int               rc=EINVAL;
        storage_object_t *obj=NULL;
        LOGICALDISK      *ld=NULL;
        DISKSEG          *freespace=NULL;
        DISKSEG          *seg=NULL;
        sector_count_t    SectorsPerCylinder;
        lba_t             end_lba = 0;
        sector_count_t    expand_sectors=0;

        LOG_ENTRY();
        if (context && context->object && context->object->name) LOG_DEBUG("  segment= %s\n", context->object->name);

        if (index == SEG_EXPAND_OPTION_SIZE_INDEX ) {

                obj = get_first_object_in_list( context->selected_objects );
                if ( obj ) {

                        if (obj->object_type == SEGMENT) {

                                if (obj->data_type == FREE_SPACE_TYPE) {

                                        seg = context->object;

                                        ld = get_logical_disk( (DISKSEG *) seg );

                                        if (ld && seg) {

                                                freespace = get_freespace_following_seg( seg );

                                                if ( freespace == obj ) {

                                                        // calc the minimum number of sectors we need for a partition
                                                        SectorsPerCylinder = get_cylinder_size(ld);

                                                        // get expand sector count
                                                        expand_sectors = value->ui64;

                                                        LOG_DEBUG("Expand Option Value= %"PRIu64" (sectors)\n", expand_sectors );

                                                        // we can only expand if in cylinder size amounts so the freespace
                                                        // following the expanding segment must be at least 1 cylinder.
                                                        // and caller should be asking for at least a cylinder
                                                        if ( ( freespace->size >= SectorsPerCylinder ) &&
                                                             ( expand_sectors >= SectorsPerCylinder ) &&
                                                             ( expand_sectors <= freespace->size ) ) {

                                                                // expand in cylinder size amounts
                                                                expand_sectors = (expand_sectors/SectorsPerCylinder)*SectorsPerCylinder;

                                                                // do cylinder allignment
                                                                end_lba = seg->start + seg->size + expand_sectors - 1;
                                                                if (ends_on_cylinder_boundary(ld, end_lba) == FALSE) {
                                                                        end_lba = roundup_to_cylinder_boundary( ld, end_lba );
                                                                }

                                                                // adjust downwards if too big ...
                                                                if ( end_lba > (freespace->start + freespace->size - 1) ) {
                                                                        end_lba = roundup_to_cylinder_boundary(ld, end_lba - SectorsPerCylinder );
                                                                }

                                                                // NOW ... test if we can expand the data seg into the freespace area
                                                                if ( ( end_lba > freespace->start ) &&
                                                                     ( end_lba <= freespace->start+freespace->size - 1) ) {

                                                                        // calc actual expand sector count
                                                                        expand_sectors = end_lba - freespace->start + 1;

                                                                        if (value->ui64 != expand_sectors) {
                                                                                *effect |= EVMS_Effect_Inexact;
                                                                        }

                                                                        context->option_descriptors->option[index].value.ui64 = expand_sectors;
                                                                        rc = 0;
                                                                }
                                                                else {
                                                                        LOG_ERROR("error, cannot expand segment ... still passes end of freespace\n");
                                                                }
                                                        }
                                                        else {
                                                                LOG_ERROR("error, invalid expand sector count\n");
                                                        }
                                                }
                                                else {
                                                        LOG_ERROR("error, selected freespace does not immediately follow data segment\n");
                                                }
                                        }
                                        else {
                                                LOG_ERROR("error, no logical drive found for segment\n");
                                        }
                                }
                                else {
                                        LOG_ERROR("error, selected freespace is NOT a freespace segment\n");
                                }
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to set and validate an option for a SHRINK task.
 */
static int set_shrink_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int               rc=EINVAL;
        storage_object_t *obj=NULL;
        LOGICALDISK      *ld=NULL;
        sector_count_t    SectorsPerCylinder;
        sector_count_t    shrink_sector_count=0;
        lba_t             end_lba=0;


        LOG_ENTRY();

        if (index == SEG_SHRINK_OPTION_SIZE_INDEX ) {

                obj = get_first_object_in_list( context->selected_objects );
                if ( obj ) {

                        if (obj->object_type == SEGMENT) {

                                if (obj->data_type == DATA_TYPE) {

                                        ld = get_logical_disk( (DISKSEG *) obj );

                                        if ( ld ) {

                                                // get option value
                                                shrink_sector_count = value->ui64;

                                                // calc the minimum number of sectors we need for a partition
                                                SectorsPerCylinder = get_cylinder_size(ld);

                                                // we can only shrink in cylinder size amounts so the shrink
                                                // sectors must be >= 1 cylinder.
                                                if ( ( shrink_sector_count < obj->size ) &&
                                                     ( shrink_sector_count >= SectorsPerCylinder ) ) {

                                                        // shrink in cylinder size amounts
                                                        shrink_sector_count = (shrink_sector_count/SectorsPerCylinder)*SectorsPerCylinder;

                                                        // do cylinder allignment
                                                        end_lba = obj->start + obj->size - shrink_sector_count - 1;
                                                        if (ends_on_cylinder_boundary(ld, end_lba) == FALSE) {
                                                                end_lba = rounddown_to_cylinder_boundary( ld, end_lba-1 ) - 1;
                                                        }

                                                        // adjust upwards if we shrunk too small ...
                                                        if ( end_lba <= obj->start ) {
                                                                end_lba = roundup_to_cylinder_boundary(ld, end_lba + 1 );
                                                        }

                                                        // final test if we can shrink the segment
                                                        if (  ( end_lba > obj->start ) &&
                                                              ( end_lba < (obj->start+obj->size-1)) ) {

                                                                shrink_sector_count = obj->size - (end_lba - obj->start + 1);

                                                                if ( value->ui64 != shrink_sector_count ) {
                                                                        value->ui64 = shrink_sector_count;
                                                                        *effect |= EVMS_Effect_Inexact;
                                                                }

                                                                context->option_descriptors->option[index].value.ui64 = shrink_sector_count;
                                                                rc = 0;
                                                        }
                                                        else {
                                                                LOG_ERROR("error, invalid shrink sector count\n");
                                                        }
                                                }
                                                else {
                                                        LOG_ERROR("error, invalid shrink sector count\n");
                                                }
                                        }
                                        else {
                                                LOG_ERROR("error, logical drive is missing for this segment\n");
                                        }
                                }
                                else {
                                        LOG_ERROR("error, segment is NOT a data segment\n");
                                }

                        }
                        else {
                                LOG_ERROR("error, not a SEGMENT object type\n");
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to set and validate an option for a MOVE task.
 */
static int set_move_option( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            task_effect_t  * effect )
{
        int               rc=EINVAL;
        DISKSEG          *seg=NULL;
        LOGICALDISK      *ld=NULL;


        LOG_ENTRY();

        if (index == SEG_MOVE_OPTION_INDEX ) {

                seg = get_first_object_in_list( context->selected_objects );
                if ( seg ) {

                        ld = get_logical_disk( seg );

                        if ( ld != NULL &&
                             seg->object_type == SEGMENT &&
                             seg->data_type   == DATA_TYPE ) {

                                if ( strlen(value->s) > 0 ||
                                     strlen(value->s) <= EVMS_NAME_SIZE ) {

                                        strncpy( context->option_descriptors->option[index].value.s,
                                                 value->s,
                                                 EVMS_NAME_SIZE );
                                        rc = 0;

                                }

                        }
                        else {
                                LOG_ERROR("error, segment cannot be moved.\n");
                        }

                }
                else {
                        LOG_ERROR("error, no object in selected object list.\n");
                }
        }
        else {
                LOG_ERROR("error, invalid move option index.\n");
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Get the user selected object from the assign context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the boolean flag found_good_object.
 */
static int set_assign_object( task_context_t * context,
                              list_anchor_t    declined_objects,
                              task_effect_t  * effect )
{
        int                 rc=0;
        storage_object_t   *obj;
        uint                segment_count;
        int                 found_good_object = FALSE;
        declined_object_t  *declined_object=NULL;
        list_element_t      iter;

        LOG_ENTRY();

        LIST_FOR_EACH( context->selected_objects, iter, obj ) {

                segment_count = EngFncs->list_count(obj->parent_objects);

                if ( ( found_good_object == FALSE) &&
                     ( obj->object_type  == DISK || obj->object_type == SEGMENT ) &&
                     ( obj->volume       == NULL ) &&
                     ( segment_count     == 0 ) ) {

                        found_good_object = TRUE;

                        rc = initialize_assign_option_descriptors(obj, context);

                        if (rc == 0) {
                                *effect |=  EVMS_Effect_Reload_Options;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                if (EngFncs->insert_thing(declined_objects,declined_object,INSERT_BEFORE,NULL)) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("error, unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

                if (rc) break;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Get the user selected object from the create context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the boolean flag found_good_object.
 */
static int set_create_object( task_context_t * context,
                              list_anchor_t    declined_objects,
                              task_effect_t  * effect )
{
        int                 rc=0;
        storage_object_t   *obj;
        LOGICALDISK        *ld;
        int                 found_good_object = FALSE;
        declined_object_t  *declined_object=NULL;
        list_element_t      iter;

        LOG_ENTRY();

        LIST_FOR_EACH( context->selected_objects, iter, obj ) {

                if ( ( obj->data_type == FREE_SPACE_TYPE ) &&
                     ( obj->plugin    == Seg_My_PluginRecord_Ptr) &&
                     ( found_good_object == FALSE ) ) {

                        ld = get_logical_disk(obj);

                        if (ld) {

                                found_good_object = TRUE;
                                rc = initialize_create_option_descriptors(ld, obj, context);

                                if (rc == 0) {
                                        *effect |=  EVMS_Effect_Reload_Options;
                                }

                        }
                        else {
                                rc = ENODEV;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                if (EngFncs->insert_thing(declined_objects,declined_object,INSERT_BEFORE,NULL)) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 *
 * Context->object == expansion point
 * Context->selected_objects->object == freespace segment
 * Context->acceptable_objects->object == freespace segment chosen during inittask time
 *
 */
static int set_expand_object( task_context_t * context,
                              list_anchor_t    declined_objects,
                              task_effect_t  * effect )
{
        int  rc=0;
        storage_object_t  *obj;
        storage_object_t  *freespace;
        list_element_t     iter;
        LOGICALDISK       *ld;
        boolean            found_good_object = FALSE;
        declined_object_t *declined_object=NULL;


        LOG_ENTRY();

        LIST_FOR_EACH( context->selected_objects, iter, obj ) {

                if ( ( obj->data_type == FREE_SPACE_TYPE ) &&
                     ( obj->plugin    == Seg_My_PluginRecord_Ptr) &&
                     ( found_good_object == FALSE ) ) {

                        freespace = get_freespace_following_seg( context->object );

                        ld = get_logical_disk(obj);

                        if (ld && freespace) {

                                found_good_object = TRUE;

                                // is selected object correct?
                                if (freespace == obj) {

                                        rc = initialize_expand_option_descriptors(ld, obj, context);

                                        if (rc == 0) {
                                                *effect |= EVMS_Effect_Reload_Options;
                                        }

                                }
                                else {
                                        LOG_ERROR("error, selected object is not freespace that follows the data segment\n");
                                        rc = EINVAL;
                                }

                        }
                        else {
                                rc = ENODEV;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                if (EngFncs->insert_thing(declined_objects,declined_object,INSERT_BEFORE,NULL)) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("error, unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int set_shrink_object( task_context_t * context,
                              list_anchor_t    declined_objects,
                              task_effect_t  * effect )
{
        int  rc=0;
        storage_object_t  *obj;
        list_element_t     iter;
        LOGICALDISK       *ld;
        boolean            found_good_object = FALSE;
        declined_object_t *declined_object=NULL;


        LOG_ENTRY();

        LIST_FOR_EACH( context->selected_objects, iter, obj ) {

                if ( ( obj->data_type == DATA_TYPE ) &&
                     ( obj->plugin    == Seg_My_PluginRecord_Ptr) &&
                     ( found_good_object == FALSE ) ) {

                        ld = get_logical_disk(obj);

                        if (ld) {

                                found_good_object = TRUE;

                                rc = initialize_shrink_option_descriptors(ld, obj, context);

                                if (rc == 0) {
                                        *effect |=  EVMS_Effect_Reload_Options;
                                }

                        }
                        else {
                                rc = ENODEV;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                if (EngFncs->insert_thing(declined_objects,declined_object,INSERT_BEFORE,NULL)) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Function: set move object
 *
 *  Called by set_object() to validate the task selected
 *  object.
 */
static int set_move_object( task_context_t * context,
                            list_anchor_t    declined_objects,
                            task_effect_t  * effect )
{
        int  rc=0;
        DISKSEG           *seg;
        list_element_t     iter;
        boolean            found_good_object = FALSE;
        declined_object_t *declined_object=NULL;


        LOG_ENTRY();

        LIST_FOR_EACH( context->selected_objects, iter, seg ) {

                if ( found_good_object == FALSE ) {

                        found_good_object = TRUE;

                        rc = initialize_move_option_descriptors(seg, context);

                        if (rc == 0) {
                                *effect |=  EVMS_Effect_Reload_Options;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = seg;
                                declined_object->reason = EINVAL;

                                if (EngFncs->insert_thing(declined_objects,declined_object,INSERT_BEFORE,NULL)) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the selected_objects list in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects list.
 * Modify the accepatble_objects list in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int SEG_SetObjects( task_context_t * context,
                    list_anchor_t    declined_objects,
                    task_effect_t  * effect )
{

        int rc = EINVAL;

        LOG_ENTRY();

        if (context) {

                switch (context->action) {

                case EVMS_Task_Create:

                        rc = set_create_object( context, declined_objects, effect );
                        break;

                case  EVMS_Task_Assign_Plugin:

                        rc = set_assign_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_Expand:

                        rc = set_expand_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_Shrink:

                        rc = set_shrink_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_Dos_Move_Segment:

                        rc = set_move_object( context, declined_objects, effect );
                        break;

                default:

                        LOG_ERROR("context->action is unknown or unsupported\n");
                        break;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called from get_acceptable_assign_objects, once for each disk
 *  storage object we got back from the kernel. The purpose is to weed out
 *  any logical disks that have already been claimed by segment managers.
 *  This is so we only lay down our partition scheme on disks that
 *  have unrecognized partition schemes and the user has deliberately
 *  requested the defsegmgr to assign itself to the DISK.
 *
 *
 */
static void prune_acceptable_disks( list_anchor_t list )
{
        storage_object_t  *obj;
        list_element_t iter, iter2;
        SEG_PRIVATE_DATA  *pdata=NULL;
        boolean prune;

        LIST_FOR_EACH_SAFE( list, iter, iter2, obj ) {

                prune = TRUE;

                // actually, any object we got from the filter is Ok at this point
                // as long as we dont own it, meaning it is either a DISK object
                // or else a DATA SEGMENT object produced by another seg mgr.
                if ( obj->plugin != Seg_My_PluginRecord_Ptr ) {

                        if (obj->object_type == DISK) {
                                prune = FALSE;
                        }
                        else if (obj->object_type == SEGMENT) {

                                pdata = (SEG_PRIVATE_DATA *)obj->private_data;

                                if ( !(pdata->cflags & SEG_CFLAG_TOP_SEGMENT) ) {
                                        prune = FALSE;
                                }

                        }

                }

                if (prune==TRUE) EngFncs->delete_element(iter);
        }

}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_assign_objects( task_context_t * context )
{
        int   rc = EINVAL;

        LOG_ENTRY();

        if ( EngFncs->list_count(context->acceptable_objects) == 0) {

                        rc = EngFncs->get_object_list( DISK | SEGMENT,
                                                       DATA_TYPE,
                                                       NULL,
                                                       NULL,
                                                       TOPMOST,
                                                       &context->acceptable_objects );

                        if (rc == 0) {

                                if ( EngFncs->list_count( context->acceptable_objects ) > 0) {
                                        prune_acceptable_disks( context->acceptable_objects );
                                }
                                else {
                                        rc = EINVAL;
                                        LOG_DEBUG("no storage objects returned by get_object_list call\n");
                                }

                        }

        }
        else {
                LOG_ERROR("context already has acceptable objects\n");
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called from get_acceptable_create_objects, once for each freespace
 *  segment we got back from the kernel. The purpose is to weed out
 *  any FREESPACE segment that we cannot create a DATA segment from.
 *  This is so we only present FREESPACE segments that are truly acceptable
 *  for CREATE.
 */
static void prune_small_segs( list_anchor_t list )
{
        LOGICALDISK       *ld;
        DISK_PRIVATE_DATA *disk_pdata;
        DISKSEG           *seg;
        sector_count_t     CylinderSize;
        lba_t              Start;
        lba_t              End;
        sector_count_t     MaxSize;
        DISKSEG           *mbr;
        list_element_t     iter,iter2;
        boolean            prune;

        LIST_FOR_EACH_SAFE( list, iter, iter2, seg ) {

                prune=TRUE;

                ld = get_logical_disk(seg);
                if (ld) {

                        disk_pdata = get_disk_private_data(ld);
                        mbr        = get_mbr_from_seglist(ld->parent_objects);

                        CylinderSize = get_cylinder_size(ld);

                        if ( (seg->start == disk_pdata->extd_partition_lba+1) ||
                             (seg->start <= disk_pdata->geometry.sectors_per_track) ) {
                                Start = seg->start;
                        }
                        else {
                                Start = roundup_to_cylinder_boundary( ld, seg->start ) + 1;
                        }

                        if ( ends_on_cylinder_boundary( ld, seg->start + seg->size - 1 ) == TRUE ) {
                                End = seg->start + seg->size - 1;
                        }
                        else if (seg->size > CylinderSize) {
                                End = rounddown_to_cylinder_boundary( ld, seg->start + seg->size - 1 ) - 1;
                        }
                        else {
                                End = Start; // force failure ... cuz we cant round down this seg to cyl bdy
                        }

                        if (Start < End) {

                                MaxSize  = End - Start + 1;

                                // if this seg is at least 1 cylinder (minus a track) in size then it is Ok
                                // because they can start on track boundaries.
                                if ( MaxSize >= CylinderSize - disk_pdata->geometry.sectors_per_track ) {

                                        // we may have free space but are unable to create a segment on the
                                        // disk due to not being able to create a partition record in the mbr.
                                        if ( ( get_first_unused_ptable_entry(ld->parent_objects, mbr)==-1) &&
                                             ( disk_has_extended_partition( ld ) == FALSE ) ) {
                                                prune = TRUE;
                                        }
                                        else if ( ( get_first_unused_ptable_entry(ld->parent_objects, mbr)==-1) &&
                                                  ( disk_has_extended_partition( ld ) == TRUE ) &&
                                                  ( seg_is_within_or_adjacant_to_extended_partition( ld, seg ) == FALSE )) {
                                                prune = TRUE;
                                        }
                                        else {
                                                prune = FALSE;
                                        }
                                }
                                else {
                                        LOG_DEBUG("max size < cyl size\n");
                                }

                        }

                }

                if (prune==TRUE) EngFncs->delete_element(iter);
        }

}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_create_objects( task_context_t * context )
{
        int   rc = EINVAL;

        LOG_ENTRY();

        if (EngFncs->list_count(context->acceptable_objects) == 0) {

                rc = EngFncs->get_object_list( SEGMENT,
                                               FREE_SPACE_TYPE,
                                               Seg_My_PluginRecord_Ptr,
                                               NULL,
                                               VALID_INPUT_OBJECT,
                                               &context->acceptable_objects );

                if (rc == 0) {
                        prune_small_segs(context->acceptable_objects);
                }

        }
        else {
                LOG_ERROR("context already has acceptable objects\n");
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_expand_objects( task_context_t * context )
{
        int   rc = EINVAL;
        uint  count=0;
        storage_object_t *freespace;
        LOGICALDISK      *ld;
        sector_count_t    CylinderSize;
        lba_t             end_lba;
        SEG_PRIVATE_DATA *pdata;


        LOG_ENTRY();

        if ( ( context ) &&
             ( context->acceptable_objects ) &&
             ( context->selected_objects ) &&
             ( context->object ) ) {

                count = EngFncs->list_count(context->acceptable_objects);

                pdata = (SEG_PRIVATE_DATA *)context->object->private_data;

                if ( ( count == 0 ) &&                            // cant start with any
                     ( (pdata->flags & SEG_IS_EMBEDDED) == 0 )) {   // cant use embedded partitions

                        freespace = get_freespace_following_seg( context->object );
                        if ( freespace ) {

                                ld = get_logical_disk(context->object);
                                if (ld) {

                                        CylinderSize = get_cylinder_size(ld);

                                        if (freespace->size >= CylinderSize) {

                                                end_lba = roundup_to_cylinder_boundary(ld, freespace->start + 1);

                                                if ( end_lba <= freespace->start + freespace->size - 1) {

                                                        if (EngFncs->insert_thing(context->acceptable_objects,freespace,INSERT_BEFORE,NULL)) {
                                                                rc = 0;
                                                        }
                                                        else {
                                                                rc = EPERM;
                                                        }

                                                }

                                        }

                                }

                        }
                }
        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_shrink_objects( task_context_t * context )
{
        int   rc = EINVAL;


        LOG_ENTRY();

        if ( (context) &&
             (context->object) &&
             (context->acceptable_objects) ) {

                if ( EngFncs->list_count(context->acceptable_objects) == 0) {

                        // only acceptable object is the focus object for the shrink
                        // task we are initializing.
                        if (EngFncs->insert_thing(context->acceptable_objects,context->object,INSERT_BEFORE,NULL)) {
                                rc = 0;
                        }
                        else {
                                rc = EPERM;
                        }

                }
                else {
                        LOG_ERROR("error, context already has acceptable objects\n");
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Function: prune unmoveable segs
 *
 *  Called by get_acceptable_move_objects() to prune
 *  any segment that cannot be moved from a list of
 *  segments.
 */
static void prune_invalid_move_targets( list_anchor_t list, DISKSEG *src )
{
        LOGICALDISK *ld1=NULL;
        LOGICALDISK *ld2=NULL;
        list_element_t iter, iter2;
        int rc=0;
        boolean prune;
        DISKSEG *trg;

        ld1 = get_logical_disk(src);

        LIST_FOR_EACH_SAFE( list, iter, iter2, trg ) {

                prune = TRUE;

                ld2 = get_logical_disk(trg);

                if ( ld1 != NULL && ld1 == ld2 ) {
                        rc = dos_validate_move_target(src,trg);
                        if (rc == 0) {
                                prune=FALSE;
                        }
                }

                if (prune==TRUE) EngFncs->delete_element(iter);
        }

}


/*
 *  Function: get acceptable move objects
 *
 *  Called when setting up a move task to go out and find all the
 *  data segments that we could move on their disks.
 */
static int get_acceptable_move_objects( task_context_t * context )
{
        int   rc = EINVAL;


        LOG_ENTRY();

        if ( (context) &&
             (context->object) &&
             (context->acceptable_objects) ) {

                if (EngFncs->list_count(context->acceptable_objects) == 0) {

                        rc = EngFncs->get_object_list( SEGMENT,
                                                       FREE_SPACE_TYPE,
                                                       NULL,
                                                       NULL,
                                                       TOPMOST,
                                                       &context->acceptable_objects );

                        if (rc == 0) {
                                prune_invalid_move_targets( context->acceptable_objects, context->object );
                        }

                }
                else {
                        LOG_ERROR("error, context already has acceptable objects\n");
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Initialize a new task context by allocating the option descriptor
 *  array and by getting acceptable objects, from the engine, for the
 *  task context, i.e. create, shrink, expand, assign ...
 */
int SEG_InitTask(task_context_t * context)
{
        int rc = EINVAL;

        LOG_ENTRY();

        if (context) {

                switch (context->action) {

                case EVMS_Task_Create:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_create_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_create_objects( context );
                        }
                        break;

                case EVMS_Task_Assign_Plugin:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_assign_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_assign_objects( context );
                        }
                        break;


                case EVMS_Task_Expand:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_expand_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_expand_objects( context );
                        }
                        break;

                case EVMS_Task_Shrink:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_shrink_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_shrink_objects( context );
                        }
                        break;

                case EVMS_Task_Dos_Move_Segment:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_move_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_move_objects( context );
                        }
                        break;

                default:
                        LOG_ERROR("error, context->action is unknown or unsupported\n");
                        break;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Returns count of options for specified task
 */
int SEG_GetOptionCount(task_context_t * task)
{
        int count;

        LOG_ENTRY();

        switch (task->action) {

        case EVMS_Task_Create:
                count = SEG_CREATE_OPTION_COUNT;
                break;

        case EVMS_Task_Assign_Plugin:
                count = SEG_ASSIGN_OPTION_COUNT;
                break;

        case EVMS_Task_Expand:
                count = SEG_EXPAND_OPTION_COUNT;
                break;

        case EVMS_Task_Shrink:
                count = SEG_SHRINK_OPTION_COUNT;
                break;

        case EVMS_Task_Dos_Move_Segment:
                count = SEG_MOVE_OPTION_COUNT;
                break;

        default:
                count = 0;
                break;
        }

        LOG_EXIT_INT(count);
        return count;
}



/*
 *
 */
int SEG_SetOption( task_context_t * context,
                   u_int32_t        index,
                   value_t        * value,
                   task_effect_t  * effect )
{
        int rc=EINVAL;

        LOG_ENTRY();


        if (context ) {

                switch (context->action) {

                case EVMS_Task_Create:
                        rc = set_create_option( context, index, value, effect );
                        break;

                case EVMS_Task_Assign_Plugin:
                        rc = set_assign_option( context, index, value, effect );
                        break;

                case EVMS_Task_Expand:
                        rc = set_expand_option( context, index, value, effect );
                        break;

                case EVMS_Task_Shrink:
                        rc = set_shrink_option( context, index, value, effect );
                        break;

                case EVMS_Task_Dos_Move_Segment:

                        LOG_ERROR("error move has no options\n");

                        rc = set_move_option(context, index, value, effect );
                        break;

                default:
                        LOG_ERROR("error, context->action is unknown or unsupported\n");
                        break;
                }
        }

        LOG_EXIT_INT(rc);

        return rc;
}





/*
 * Returns segment specific information
 */
int SEG_GetInfo( storage_object_t  * object, char * name, extended_info_array_t * * info)
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        char                     drive_letter[8];
        char                     disk_name[EVMS_NAME_SIZE];
        char                     volume_name[EVMS_NAME_SIZE];
        char                     partition_name[EVMS_NAME_SIZE];
        LOGICALDISK             *ld;
        DISK_PRIVATE_DATA       *disk_pdata;

        LOG_ENTRY();

        // a measure of protection ...
        if (info == NULL) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        if ( object->object_type == SEGMENT ) {

                ld         = get_logical_disk( object );
                disk_pdata = get_disk_private_data(ld);

                Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( SEG_INFO_COUNT * sizeof(extended_info_t) ) );
                if (Info) {

                        Info->count = 3;

                        Info->info[SEG_INFO_NAME_INDEX].name = EngFncs->engine_strdup( "Name" );
                        Info->info[SEG_INFO_NAME_INDEX].title = EngFncs->engine_strdup( _("Name") );
                        Info->info[SEG_INFO_NAME_INDEX].desc = EngFncs->engine_strdup( _("This is the partition name. It must be unique on the system.") );
                        Info->info[SEG_INFO_NAME_INDEX].type               = EVMS_Type_String;
                        Info->info[SEG_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
                        Info->info[SEG_INFO_NAME_INDEX].value.s = EngFncs->engine_strdup( object->name );
                        Info->info[SEG_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[SEG_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

                        Info->info[SEG_INFO_SIZE_INDEX].name = EngFncs->engine_strdup( "Size" );
                        Info->info[SEG_INFO_SIZE_INDEX].title = EngFncs->engine_strdup( _("Size") );
                        Info->info[SEG_INFO_SIZE_INDEX].desc = EngFncs->engine_strdup( _("This is the size of the partition in sectors.") );
                        Info->info[SEG_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                        Info->info[SEG_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
                        Info->info[SEG_INFO_SIZE_INDEX].value.ui64         = object->size;
                        Info->info[SEG_INFO_SIZE_INDEX].format             = EVMS_Format_Normal;
                        Info->info[SEG_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[SEG_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));
                        Info->info[SEG_INFO_SIZE_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

                        Info->info[SEG_INFO_START_INDEX].name = EngFncs->engine_strdup( "Start" );
                        Info->info[SEG_INFO_START_INDEX].title = EngFncs->engine_strdup( _("Start Logical Block Address") );
                        Info->info[SEG_INFO_START_INDEX].desc = EngFncs->engine_strdup( _("This is the sector offset of the partition on the disk, i.e., the logical block address of the first sector of the partition.") );
                        Info->info[SEG_INFO_START_INDEX].type               = EVMS_Type_Unsigned_Int64;
                        Info->info[SEG_INFO_START_INDEX].unit               = EVMS_Unit_None;
                        Info->info[SEG_INFO_START_INDEX].value.ui64         = object->start;
                        Info->info[SEG_INFO_START_INDEX].format             = EVMS_Format_Normal;
                        Info->info[SEG_INFO_START_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[SEG_INFO_START_INDEX].group, 0, sizeof(group_info_t));


                        // dont show the following info for METADATA and FREESPACE segments
                        if ( object->data_type == DATA_TYPE ) {

                                // embedded partition info
                                if ( ((SEG_PRIVATE_DATA *)object->private_data)->flags & SEG_IS_EMBEDDED ) {

                                        char  tag_string[EVMS_VOLUME_NAME_SIZE];
                                        SEG_PRIVATE_DATA *pdata = (SEG_PRIVATE_DATA *)object->private_data;

                                        Info->info[SEG_INFO_FLAGS_INDEX].name = EngFncs->engine_strdup( "Flags" );
                                        Info->info[SEG_INFO_FLAGS_INDEX].title = EngFncs->engine_strdup( _("Flag") );
                                        Info->info[SEG_INFO_FLAGS_INDEX].desc = EngFncs->engine_strdup( _("This is the permission flags field from the partition.") );
                                        Info->info[SEG_INFO_FLAGS_INDEX].type               = EVMS_Type_Unsigned_Int32;
                                        Info->info[SEG_INFO_FLAGS_INDEX].unit               = EVMS_Unit_None;
                                        Info->info[SEG_INFO_FLAGS_INDEX].format             = EVMS_Format_Hex;
                                        Info->info[SEG_INFO_FLAGS_INDEX].value.ui32         = pdata->permissions;
                                        Info->info[SEG_INFO_FLAGS_INDEX].collection_type    = EVMS_Collection_None;
                                        memset( &Info->info[SEG_INFO_FLAGS_INDEX].group, 0, sizeof(group_info_t));

                                        Info->count += 1;

                                        Info->info[SEG_INFO_TYPE_INDEX].name = EngFncs->engine_strdup( "Type" );
                                        Info->info[SEG_INFO_TYPE_INDEX].title = EngFncs->engine_strdup( _("Type") );
                                        Info->info[SEG_INFO_TYPE_INDEX].desc = EngFncs->engine_strdup( _("Type of embedded partition.") );
                                        Info->info[SEG_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                                        Info->info[SEG_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                                        Info->info[SEG_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                                        memset( &Info->info[SEG_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));
                                        if ( pdata->flags & SEG_IS_SOLARIS_X86_PARTITION ) {
                                                Info->info[SEG_INFO_TYPE_INDEX].value.s = EngFncs->engine_strdup(_("Solaris x86 partition"));
                                        }
                                        else if ( pdata->flags & SEG_IS_UNIXWARE_PARTITION ) {
                                                Info->info[SEG_INFO_TYPE_INDEX].value.s = EngFncs->engine_strdup(_("Unixware partition"));
                                        }
                                        else if ( pdata->flags & SEG_IS_BSD_PARTITION ) {
                                                Info->info[SEG_INFO_TYPE_INDEX].value.s = EngFncs->engine_strdup(_("BSD partition"));
                                        }

                                        Info->count += 1;

                                        Info->info[SEG_INFO_SLICE_INDEX].name = EngFncs->engine_strdup( "Slice" );
                                        Info->info[SEG_INFO_SLICE_INDEX].title = EngFncs->engine_strdup( _("Slice") );
                                        Info->info[SEG_INFO_SLICE_INDEX].desc = EngFncs->engine_strdup( _("This is the index of the slice in its partition table.") );
                                        Info->info[SEG_INFO_SLICE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                                        Info->info[SEG_INFO_SLICE_INDEX].unit               = EVMS_Unit_None;
                                        Info->info[SEG_INFO_SLICE_INDEX].format             = EVMS_Format_Normal;
                                        Info->info[SEG_INFO_SLICE_INDEX].value.ui32         = pdata->ptable_index;
                                        Info->info[SEG_INFO_SLICE_INDEX].collection_type    = EVMS_Collection_None;
                                        memset( &Info->info[SEG_INFO_SLICE_INDEX].group, 0, sizeof(group_info_t));

                                        Info->count += 1;

                                        Info->info[SEG_INFO_TAG_INDEX].name = EngFncs->engine_strdup( "Tag" );
                                        Info->info[SEG_INFO_TAG_INDEX].title = EngFncs->engine_strdup( _("Tag") );
                                        Info->info[SEG_INFO_TAG_INDEX].desc = EngFncs->engine_strdup( _("This is the tag or file system type field from the partition.") );
                                        Info->info[SEG_INFO_TAG_INDEX].type               = EVMS_Type_String;
                                        Info->info[SEG_INFO_TAG_INDEX].unit               = EVMS_Unit_None;
                                        Info->info[SEG_INFO_TAG_INDEX].collection_type    = EVMS_Collection_None;
                                        memset( &Info->info[SEG_INFO_TAG_INDEX].group, 0, sizeof(group_info_t));
                                        if ( pdata->flags & SEG_IS_SOLARIS_X86_PARTITION ) {

                                                if (pdata->tag < SOLARISX86_TAG_NAME_COUNT) {
                                                        sprintf(tag_string, "%d - %s", pdata->tag, solarisx86_names[pdata->tag] );
                                                }
                                                else {
                                                        sprintf(tag_string, _("%d - unknown"), pdata->tag );
                                                }

                                        }
                                        else if ( pdata->flags & SEG_IS_UNIXWARE_PARTITION ) {

                                                if (pdata->tag < UW_TAG_NAME_COUNT) {
                                                        sprintf(tag_string, "%d - %s", pdata->tag, uw_names[pdata->tag] );
                                                }
                                                else {
                                                        sprintf(tag_string, _("%d - unknown"), pdata->tag );
                                                }

                                        }
                                        else if ( pdata->flags & SEG_IS_BSD_PARTITION ) {

                                                if (pdata->tag < BSD_TAG_NAME_COUNT) {
                                                        sprintf(tag_string, "%d - %s", pdata->tag, bsd_names[pdata->tag] );
                                                }
                                                else {
                                                        sprintf(tag_string, _("%d - unknown"), pdata->tag );
                                                }

                                        }

                                        Info->info[SEG_INFO_TAG_INDEX].value.s = EngFncs->engine_strdup(tag_string );

                                        Info->count += 1;


                                }
                                else {  // msdos partition info

                                        if ( disk_pdata->disk_name ) {
                                                strcpy(disk_name, disk_pdata->disk_name );
                                        }
                                        else {
                                                strcpy(disk_name, "n/a");
                                        }

                                        Info->info[SEG_INFO_FLAGS_INDEX].name = EngFncs->engine_strdup( "Flag" );
                                        Info->info[SEG_INFO_FLAGS_INDEX].title = EngFncs->engine_strdup( _("Primary") );
                                        Info->info[SEG_INFO_FLAGS_INDEX].desc = EngFncs->engine_strdup( _("Logical or primary partition.") );
                                        Info->info[SEG_INFO_FLAGS_INDEX].type               = EVMS_Type_String;
                                        Info->info[SEG_INFO_FLAGS_INDEX].unit               = EVMS_Unit_None;
                                        Info->info[SEG_INFO_FLAGS_INDEX].collection_type    = EVMS_Collection_None;
                                        memset( &Info->info[SEG_INFO_FLAGS_INDEX].group, 0, sizeof(group_info_t));
                                        if ( ((SEG_PRIVATE_DATA *)object->private_data)->flags & SEG_IS_PRIMARY_PARTITION ) {
                                                Info->info[SEG_INFO_FLAGS_INDEX].value.s = EngFncs->engine_strdup(_("Yes"));
                                        }
                                        else {
                                                Info->info[SEG_INFO_FLAGS_INDEX].value.s = EngFncs->engine_strdup(_("No"));
                                        }

                                        Info->count += 1;

                                        Info->info[SEG_INFO_TYPE_INDEX].name = EngFncs->engine_strdup( "Type" );
                                        Info->info[SEG_INFO_TYPE_INDEX].title = EngFncs->engine_strdup( _("Type") );
                                        Info->info[SEG_INFO_TYPE_INDEX].desc = EngFncs->engine_strdup( _("Type of partition, e.g. 0x83 = Linux EXT2 Partition") );
                                        Info->info[SEG_INFO_TYPE_INDEX].type               = EVMS_Type_Unsigned_Int8;
                                        Info->info[SEG_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                                        Info->info[SEG_INFO_TYPE_INDEX].format             = EVMS_Format_Hex;
                                        Info->info[SEG_INFO_TYPE_INDEX].value.ui8          = ((SEG_PRIVATE_DATA *)object->private_data)->sys_id;
                                        Info->info[SEG_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                                        memset( &Info->info[SEG_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                                        Info->count += 1;

                                        Info->info[SEG_INFO_BOOT_INDEX].name = EngFncs->engine_strdup( "BootInd" );
                                        Info->info[SEG_INFO_BOOT_INDEX].title = EngFncs->engine_strdup( _("Flags") );
                                        Info->info[SEG_INFO_BOOT_INDEX].desc = EngFncs->engine_strdup( _("This is the flag field from the partition record.") );
                                        Info->info[SEG_INFO_BOOT_INDEX].type               = EVMS_Type_Unsigned_Int8;
                                        Info->info[SEG_INFO_BOOT_INDEX].unit               = EVMS_Unit_None;
                                        Info->info[SEG_INFO_BOOT_INDEX].format             = EVMS_Format_Hex;
                                        Info->info[SEG_INFO_BOOT_INDEX].value.ui8          = ((SEG_PRIVATE_DATA *)object->private_data)->boot_ind;
                                        Info->info[SEG_INFO_BOOT_INDEX].collection_type    = EVMS_Collection_None;
                                        memset( &Info->info[SEG_INFO_BOOT_INDEX].group, 0, sizeof(group_info_t));

                                        Info->count += 1;

                                        if (  disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

                                                Info->info[SEG_INFO_DSKNAME_INDEX].name = EngFncs->engine_strdup( "DiskName" );
                                                Info->info[SEG_INFO_DSKNAME_INDEX].title = EngFncs->engine_strdup( _("Disk Name") );
                                                Info->info[SEG_INFO_DSKNAME_INDEX].desc = EngFncs->engine_strdup( _("OS/2 disks are assigned names that are stored on the disk for persistance") );
                                                Info->info[SEG_INFO_DSKNAME_INDEX].type               = EVMS_Type_String;
                                                Info->info[SEG_INFO_DSKNAME_INDEX].unit               = EVMS_Unit_None;
                                                Info->info[SEG_INFO_DSKNAME_INDEX].value.s = EngFncs->engine_strdup( disk_name );
                                                Info->info[SEG_INFO_DSKNAME_INDEX].collection_type    = EVMS_Collection_None;
                                                memset( &Info->info[SEG_INFO_DSKNAME_INDEX].group, 0, sizeof(group_info_t));

                                                Info->count += 1;


                                                if (((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Drive_Letter != 0x00) {
                                                        drive_letter[0] = ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Drive_Letter;
                                                        drive_letter[1] = '\0';
                                                }
                                                else {
                                                        strcpy(drive_letter, "n/a" );
                                                }

                                                if (strlen(((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Volume_Name)) {
                                                        strcpy(volume_name, ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Volume_Name );
                                                }
                                                else {
                                                        strcpy(volume_name, "n/a");
                                                }

                                                if (strlen(((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Partition_Name)) {
                                                        strcpy(partition_name, ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Partition_Name );
                                                }
                                                else {
                                                        strcpy(partition_name, "n/a");
                                                }


                                                Info->info[SEG_INFO_PNAME_INDEX].name = EngFncs->engine_strdup( "Name" );
                                                Info->info[SEG_INFO_PNAME_INDEX].title = EngFncs->engine_strdup( _("OS/2 Name") );
                                                Info->info[SEG_INFO_PNAME_INDEX].desc = EngFncs->engine_strdup( _("OS/2 partition name.") );
                                                Info->info[SEG_INFO_PNAME_INDEX].type               = EVMS_Type_String;
                                                Info->info[SEG_INFO_PNAME_INDEX].unit               = EVMS_Unit_None;
                                                Info->info[SEG_INFO_PNAME_INDEX].value.s = EngFncs->engine_strdup( partition_name );
                                                Info->info[SEG_INFO_PNAME_INDEX].collection_type    = EVMS_Collection_None;
                                                memset( &Info->info[SEG_INFO_PNAME_INDEX].group, 0, sizeof(group_info_t));

                                                Info->count += 1;

                                                Info->info[SEG_INFO_VNAME_INDEX].name = EngFncs->engine_strdup( "Volume Name" );
                                                Info->info[SEG_INFO_VNAME_INDEX].title = EngFncs->engine_strdup( _("Volume Name") );
                                                Info->info[SEG_INFO_VNAME_INDEX].desc = EngFncs->engine_strdup( _("OS/2 assigns volume names to Compatibility volumes and LVM volumes.") );
                                                Info->info[SEG_INFO_VNAME_INDEX].type               = EVMS_Type_String;
                                                Info->info[SEG_INFO_VNAME_INDEX].unit               = EVMS_Unit_None;
                                                Info->info[SEG_INFO_VNAME_INDEX].value.s = EngFncs->engine_strdup( volume_name );
                                                Info->info[SEG_INFO_VNAME_INDEX].collection_type    = EVMS_Collection_None;
                                                memset( &Info->info[SEG_INFO_VNAME_INDEX].group, 0, sizeof(group_info_t));

                                                Info->count += 1;

                                                Info->info[SEG_INFO_DRVLTR_INDEX].name = EngFncs->engine_strdup( "Drive Letter" );
                                                Info->info[SEG_INFO_DRVLTR_INDEX].title = EngFncs->engine_strdup( _("Drive Letter") );
                                                Info->info[SEG_INFO_DRVLTR_INDEX].desc = EngFncs->engine_strdup( _("OS/2 volumes are given a drive letter assignment, e.g. C:") );
                                                Info->info[SEG_INFO_DRVLTR_INDEX].type               = EVMS_Type_String;
                                                Info->info[SEG_INFO_DRVLTR_INDEX].unit               = EVMS_Unit_None;
                                                Info->info[SEG_INFO_DRVLTR_INDEX].value.s = EngFncs->engine_strdup( drive_letter );
                                                Info->info[SEG_INFO_DRVLTR_INDEX].collection_type    = EVMS_Collection_None;
                                                memset( &Info->info[SEG_INFO_DRVLTR_INDEX].group, 0, sizeof(group_info_t));

                                                Info->count += 1;

                                                Info->info[SEG_INFO_VSN_INDEX].name = EngFncs->engine_strdup( "Volume SN" );
                                                Info->info[SEG_INFO_VSN_INDEX].title = EngFncs->engine_strdup( _("Volume Serial Number") );
                                                Info->info[SEG_INFO_VSN_INDEX].desc = EngFncs->engine_strdup( _("OS/2 volume serial number") );
                                                Info->info[SEG_INFO_VSN_INDEX].type               = EVMS_Type_Unsigned_Int32;
                                                Info->info[SEG_INFO_VSN_INDEX].unit               = EVMS_Unit_None;
                                                Info->info[SEG_INFO_VSN_INDEX].format             = EVMS_Format_Hex;
                                                Info->info[SEG_INFO_VSN_INDEX].value.ui32         = ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Volume_Serial_Number;
                                                Info->info[SEG_INFO_VSN_INDEX].collection_type    = EVMS_Collection_None;
                                                memset( &Info->info[SEG_INFO_VSN_INDEX].group, 0, sizeof(group_info_t));

                                                Info->count += 1;

                                                Info->info[SEG_INFO_PSN_INDEX].name = EngFncs->engine_strdup( "Partition SN" );
                                                Info->info[SEG_INFO_PSN_INDEX].title = EngFncs->engine_strdup( _("Partition Serial Number") );
                                                Info->info[SEG_INFO_PSN_INDEX].desc = EngFncs->engine_strdup( _("OS/2 partition serial number"));
                                                Info->info[SEG_INFO_PSN_INDEX].type               = EVMS_Type_Unsigned_Int32;
                                                Info->info[SEG_INFO_PSN_INDEX].unit               = EVMS_Unit_None;
                                                Info->info[SEG_INFO_PSN_INDEX].format             = EVMS_Format_Hex;
                                                Info->info[SEG_INFO_PSN_INDEX].value.ui32         = ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Partition_Serial_Number;
                                                Info->info[SEG_INFO_PSN_INDEX].collection_type    = EVMS_Collection_None;
                                                memset( &Info->info[SEG_INFO_PSN_INDEX].group, 0, sizeof(group_info_t));

                                                Info->count += 1;

                                        } //end of OS2 disk
                                } //end of msdos info
                        } //end of data partition

                        *info = Info;

                        rc = 0;
                }
                else {
                        LOG_ERROR("unable to malloc memory for extended info array\n");
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *
 */
int SEG_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        char                     version_string[64];
        char                     required_engine_api_version_string[64];
        char                     required_plugin_api_version_string[64];


        LOG_ENTRY();

        // a measure of protection ...
        if (info == NULL) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (SEG_PLUGIN_INFO_COUNT*sizeof(extended_info_t))  );
        if (Info) {

                Info->count = SEG_PLUGIN_INFO_COUNT;

                sprintf(version_string, "%d.%d.%d",
                        MAJOR_VERSION,
                        MINOR_VERSION,
                        PATCH_LEVEL );

                sprintf(required_engine_api_version_string, "%d.%d.%d",
                        Seg_My_PluginRecord_Ptr->required_engine_api_version.major,
                        Seg_My_PluginRecord_Ptr->required_engine_api_version.minor,
                        Seg_My_PluginRecord_Ptr->required_engine_api_version.patchlevel );

                sprintf(required_plugin_api_version_string, "%d.%d.%d",
                        Seg_My_PluginRecord_Ptr->required_plugin_api_version.plugin.major,
                        Seg_My_PluginRecord_Ptr->required_plugin_api_version.plugin.minor,
                        Seg_My_PluginRecord_Ptr->required_plugin_api_version.plugin.patchlevel );

                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].name = EngFncs->engine_strdup( "ShortName" );
                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].title = EngFncs->engine_strdup( _("Short Name") );
                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].desc = EngFncs->engine_strdup( _("A short name given to this plug-in") );
                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].value.s = EngFncs->engine_strdup( Seg_My_PluginRecord_Ptr->short_name );
                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].name = EngFncs->engine_strdup( "LongName" );
                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].title = EngFncs->engine_strdup( _("Long Name") );
                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].desc = EngFncs->engine_strdup( _("A longer, more descriptive name for this plug-in") );
                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].value.s = EngFncs->engine_strdup( Seg_My_PluginRecord_Ptr->long_name );
                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].name = EngFncs->engine_strdup( "Type" );
                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].title = EngFncs->engine_strdup( _("Plug-in Type") );
                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].desc = EngFncs->engine_strdup( _("There are various types of plug-ins, each responsible for some kind of storage object or logical volume.") );
                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("Segment Manager") );
                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].name = EngFncs->engine_strdup( "Version" );
                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].title = EngFncs->engine_strdup( _("Plugin Version") );
                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].desc = EngFncs->engine_strdup( _("This is the version number of the plugi-n.") );
                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].value.s = EngFncs->engine_strdup( version_string );
                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].name = EngFncs->engine_strdup( "Required Engine Services Version" );
                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].title = EngFncs->engine_strdup( _("Required Engine Services Version") );
                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].desc = EngFncs->engine_strdup( _("This is the version of the Engine services that this plug-in requires.  "
												    "It will not run on older versions of the Engine services.") );
                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].value.s = EngFncs->engine_strdup( required_engine_api_version_string );
                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].group, 0, sizeof(group_info_t));

                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].name = EngFncs->engine_strdup( "Required Plug-in API Version" );
                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].title = EngFncs->engine_strdup( _("Required Plug-in API Version") );
                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].desc = EngFncs->engine_strdup( _("This is the version of the Engine plug-in API that this plug-in requires.  "
												     "It will not run on older versions of the Engine plug-in API.") );
                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].value.s = EngFncs->engine_strdup( required_plugin_api_version_string );
                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].group, 0, sizeof(group_info_t));
                *info = Info;

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}
