/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gconf/gconf.h>

#include <canberra.h>
#include <libnotify/notify.h>
#include <libindicate/indicator-message.h>

#include <time.h>

#include <e-util/e-config.h>
#include <mail/em-utils.h>
#include <mail/em-event.h>
#include <mail/em-folder-tree-model.h>
#include <shell/evolution-config-control.h>
#include <shell/e-component-view.h>
#include <shell/es-event.h>

#include "e-shell.h"
#include "mail-server.h"
#include "xutils.h"

#define CONF_DIR          "/apps/evolution/eplugin/evolution_indicator"
#define ONLY_INBOX        CONF_DIR"/only_inbox"
#define PLAY_SOUND        CONF_DIR"/play_sound"
#define SHOW_BUBBLE       CONF_DIR"/show_bubble"
#define SHOW_NEW_IN_PANEL CONF_DIR"/show_new_messages_in_panel"

static EShell       *evo_shell   = NULL;
static MailServer   *mail_server = NULL;
static GStaticMutex  mlock       = G_STATIC_MUTEX_INIT;
static GConfClient  *client      = NULL;

static NotifyNotification *notification   = NULL;
static ca_context         *canberra_cxt   = NULL;
static ca_proplist        *canberra_props = NULL;

static gboolean      is_active   = TRUE;
static gboolean      only_inbox  = TRUE;
static gboolean      play_sound  = TRUE;
static gboolean      show_bubble = TRUE;
static gboolean      show_count  = FALSE;

static guint         only_inbox_id  = 0;
static guint         play_sound_id  = 0;
static guint         show_bubble_id = 0;
static guint         show_count_id  = 0;

static gint message_count = 0;

void org_gnome_mail_new_notify    (EPlugin *ep, EMEventTargetFolder *t);
void org_gnome_mail_read_notify   (EPlugin *ep, EMEventTargetMessage *t);
void org_gnome_shell_started_done (EPlugin *ep, ESEventTargetShell *t);

int         e_plugin_lib_enable               (EPluginLib *ep, int enable);
GtkWidget * e_plugin_lib_get_configure_widget (EPlugin *epl);

static GtkWidget *
get_cfg_widget (void)
{
	GtkWidget *vbox;

	vbox = gtk_vbox_new (FALSE, 6);

	gtk_widget_show (vbox);

	return vbox;
}

static gboolean
evolution_is_focused (void)
{
#define MAIL_ICON "evolution-mail"
  static GdkScreen *screen = NULL;
  static GdkWindow *root = NULL;
  gboolean          res = FALSE;
  Window            xwindow;
  
  /* Try and do a match through X, by grabbing the current active window and
     checking to see if it's an evolution window */
  if (screen == NULL || root == NULL)
  {
    screen = gdk_screen_get_default ();
    root = gdk_screen_get_root_window (screen);
  }

  xwindow = None;
  res = _wnck_get_window (GDK_WINDOW_XWINDOW (root), 
                    gdk_x11_get_xatom_by_name ("_NET_ACTIVE_WINDOW"),
                    &xwindow);
  if (res)
  {
    gchar *res_class = NULL;
    gchar *res_name = NULL;

    _wnck_get_wmclass (xwindow, &res_class, &res_name);

    if (g_str_equal (res_name, "evolution"))
    {
      g_free (res_class);
      g_free (res_name);
      return TRUE;
    }
    g_free (res_class);
    g_free (res_name);
  }

  if (evo_shell)
  {
    GList *w;
    
    for (w = evo_shell->priv->windows; w; w = w->next)
    {
      GtkWindow *window = w->data;

      if (GTK_IS_WINDOW (window))
      {
        if (gtk_window_is_active (GTK_WINDOW (window)))
        {
          return TRUE;
        }
      }
    }
  }
  return FALSE;
}

void
org_gnome_mail_new_notify (EPlugin *ep, EMEventTargetFolder *t)
{
	g_return_if_fail (t != NULL);
	
  if (!t->new)
		return;

  if (only_inbox && !t->is_inbox)
  {
    g_debug ("Not inbox");
    return;
  }

  if (evolution_is_focused ())
  {
    g_debug ("Evolution is focused");
    return;
  }

	g_static_mutex_lock (&mlock);

  g_debug ("EI:mail_new_notify");

  message_count += t->new;
 
  if (show_count)
  {
    /* Update libindicate */
    mail_server_set_message_count (mail_server, message_count);
  }

  /* Show bubble */
  if (show_bubble)
  {
    GError *error = NULL;
    const gchar *trans;
    gchar *title;

    if (notification == NULL)
    {
      notification = notify_notification_new (" ", " ", "mail-unread", NULL);
    }

    trans = g_dngettext (PACKAGE, 
                         _("%d New Message"), 
                         _("%d New Messages"),
                         message_count);
    
    title = g_strdup_printf (trans, message_count);
    
    notify_notification_update (notification,
                                title, NULL, "notification-message-email");

    /* This plays a libcanberra sound */
    if (play_sound)
      notify_notification_set_hint_string (notification, 
                                           "sound-themed",
                                           "message-new-email");

    notify_notification_show (notification, &error);

    if (error)
    {
      g_warning ("Could not update: %s", error->message);
      g_error_free (error);
    }
  }

  /* Check if we need to play the sound differently */
  if (!show_bubble && play_sound)
  {
    gint ret;
    
    g_debug ("EI: No bubbles enabled so playing sound ourselves");

    ret = ca_context_play (canberra_cxt, 0,
                           CA_PROP_EVENT_ID, "message-new-email",
                           CA_PROP_MEDIA_LANGUAGE, "en_EN",
                           CA_PROP_CANBERRA_CACHE_CONTROL, "permanent",
                           NULL);

    g_warning ("Unable to play sound: %s\n", ca_strerror (ret));
  }

  g_static_mutex_unlock (&mlock);
}

void
org_gnome_mail_read_notify (EPlugin *ep, EMEventTargetMessage *t)
{
	g_return_if_fail (t != NULL);

	g_static_mutex_lock (&mlock);

  g_debug ("EI: mail_read_notify");

  message_count = 0;

  if (show_count)
  {
    mail_server_set_message_count (mail_server, message_count);
  }
  
  g_static_mutex_unlock (&mlock);
}

/*
 * GCONF CLIENTS
 */
static void
only_inbox_changed (GConfClient *gclient, 
                    guint        id,
                    GConfEntry  *entry,
                    gpointer     data)
{
  GConfValue *value;

  value = entry->value;

  only_inbox = gconf_value_get_bool (value);

  g_debug ("EI: Only show Inbox %s", only_inbox ? "true" : "false");
}

static void
play_sound_changed (GConfClient *gclient, 
                    guint        id,
                    GConfEntry  *entry,
                    gpointer     data)
{
  GConfValue *value;

  value = entry->value;

  play_sound = gconf_value_get_bool (value);

  g_debug ("EI: Play Sounds %s", play_sound ? "true" : "false");
}

static void
show_new_in_panel_changed (GConfClient *gclient, 
                           guint        id,
                           GConfEntry  *entry,
                           gpointer     data)
{
  GConfValue *value;

  value = entry->value;

  show_count = gconf_value_get_bool (value);

  show_count ? indicate_server_show (INDICATE_SERVER (mail_server))
             : indicate_server_hide (INDICATE_SERVER (mail_server));

  g_debug ("EI: Messages in panel %s", 
           show_count ? "true" : "false");
}

static void
show_bubble_changed (GConfClient *gclient, 
                     guint        id,
                     GConfEntry  *entry,
                     gpointer     data)
{
  GConfValue *value;

  value = entry->value;

  show_bubble = gconf_value_get_bool (value);

  g_debug ("EI: Show Bubbles %s", show_bubble ? "true" : "false");
}

int
e_plugin_lib_enable (EPluginLib *ep, int enable)
{
  is_active = enable;

  if (is_active)
  {
    if (notification == NULL)
      notify_init ("evolution-indicator");

    if (canberra_cxt == NULL)
    {
      gint ret;

      ret = ca_context_create (&canberra_cxt);
      if (ret)
      {
        g_warning ("Canberra Init Error: %s", ca_strerror (ret));
      }
      else
      {
        ret = ca_context_change_props (canberra_cxt, 
                                       CA_PROP_APPLICATION_NAME, "evolution-indicator", 
                                       CA_PROP_APPLICATION_ID, "org.freedesktop.evolution-indicator",
                                       CA_PROP_WINDOW_X11_XID, ":0",
                                       NULL);
        if (ret)
        {
          g_warning ("Unable to set props: %s\n", ca_strerror (ret));
        }
      }
    }

    mail_server = mail_server_get_default ();

    client = gconf_client_get_default ();
    gconf_client_add_dir (client, CONF_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);

    only_inbox = gconf_client_get_bool (client, ONLY_INBOX, NULL);
    only_inbox_id = gconf_client_notify_add (client, ONLY_INBOX, 
                             only_inbox_changed, NULL, NULL, NULL);

    play_sound = gconf_client_get_bool (client, PLAY_SOUND, NULL);
    play_sound_id = gconf_client_notify_add (client, PLAY_SOUND, 
                             play_sound_changed, NULL, NULL, NULL);

    play_sound = gconf_client_get_bool (client, PLAY_SOUND, NULL);
    play_sound_id = gconf_client_notify_add (client, PLAY_SOUND, 
                             play_sound_changed, NULL, NULL, NULL);

    show_bubble = gconf_client_get_bool (client, SHOW_BUBBLE, NULL);
    show_bubble_id = gconf_client_notify_add (client, SHOW_BUBBLE, 
                             show_bubble_changed, NULL, NULL, NULL);


    show_count = gconf_client_get_bool (client, 
                                        SHOW_NEW_IN_PANEL, 
                                        NULL);
    show_count_id = gconf_client_notify_add (client, SHOW_NEW_IN_PANEL, 
                             show_new_in_panel_changed, NULL, NULL, NULL);

    if (show_count)
    {
      indicate_server_show (INDICATE_SERVER (mail_server));
    }
  }
  else
  {
    gconf_client_notify_remove (client, only_inbox_id);
    gconf_client_notify_remove (client, play_sound_id);
    gconf_client_notify_remove (client, show_bubble_id);
    gconf_client_notify_remove (client, show_count_id);

    g_object_unref (client);      client = NULL;
    g_object_unref (mail_server); mail_server = NULL;
    
    g_debug ("EI: Disabled");
  }

	return 0;
}

GtkWidget *
e_plugin_lib_get_configure_widget (EPlugin *epl)
{
  g_debug ("EI: Get Configure Widget");
	return get_cfg_widget ();
}

/*
 *
 * THIS IS A MAJOR HACK -- DO NOT EMAIL ME ABOUT HOW MAJOR OF A HACK IT REALLY
 * IS. I'M ALREADY LOSING SLEEP OVER IT.
 *
 * IF YOU HAVE ISSUES WITH THIS CODE, FEEL FREE TO ADD _PROPER_ PLUGIN SUPPORT
 * TO EVOLUTION, OR AT LEAST A DBUS INTERFACE.
 *
 * -- 
 * this message was brought to you by Holiday Inn coffee and a cold
 * Berlin morning 
 *
 */
void 
org_gnome_shell_started_done (EPlugin *ep, ESEventTargetShell *t)
{
  g_debug ("EI: SHELL STARTUP");
  
  evo_shell = t->shell;
  mail_server_set_e_shell (mail_server_get_default (), t->shell);
}

static void
on_combo_changed (GtkComboBox *combo, gpointer null)
{
  gconf_client_set_bool (client,
                         ONLY_INBOX,
                         gtk_combo_box_get_active (combo) ? FALSE : TRUE,
                         NULL);
}

static void
on_sound_toggled (GtkToggleButton *button, gpointer null)
{
  gconf_client_set_bool (client, PLAY_SOUND, 
                         gtk_toggle_button_get_active (button), NULL);
}

static void
on_bubble_toggled (GtkToggleButton *button, gpointer null)
{
  gconf_client_set_bool (client, SHOW_BUBBLE,
                         gtk_toggle_button_get_active (button), NULL);
}

static void
on_show_panel_toggled (GtkToggleButton *button, gpointer null)
{
  gconf_client_set_bool (client, SHOW_NEW_IN_PANEL,
                         gtk_toggle_button_get_active (button), NULL);
}

GtkWidget *
org_gnome_get_prefs (EPlugin *epl, EConfigHookItemFactoryData *data)
{
  GtkWidget *vbox, *frame=NULL, *check;
  
  g_print ("EI: MAIL PREFS");

	if (data->old)
    return data->old;

  frame = data->parent;
  while (!GTK_IS_FRAME (frame))
  {
    frame = frame->parent;

    if (GTK_IS_WINDOW (frame) || !GTK_IS_WIDGET (frame))
      break;
  }

  if (GTK_IS_FRAME (frame))
  {
    GtkWidget *frame, *box, *label1, *label2, *label3, *combo;
    const gchar *markup = "<b>%s</b>";
    gchar *str1;
    gchar *str2;

    frame = (GtkWidget*)data->parent->parent->parent;

    box = gtk_hbox_new (FALSE, 0);
    gtk_frame_set_label_widget (GTK_FRAME (frame), box);
    gtk_widget_show (frame);

    label1 = gtk_label_new (" ");
    str1 = g_strdup_printf (markup, _("When New Mail Arri_ves In"));
    gtk_label_set_markup_with_mnemonic (GTK_LABEL (label1), str1);
    g_free (str1);

    label2 = gtk_label_new (" ");

    combo = gtk_combo_box_new_text ();
    gtk_combo_box_append_text (GTK_COMBO_BOX (combo), _("Inbox"));
    gtk_combo_box_append_text (GTK_COMBO_BOX (combo), _("Any Folder"));
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 
                              only_inbox ? 0 : 1);
    g_signal_connect (combo, "changed", G_CALLBACK (on_combo_changed), NULL);
     
    label3 = gtk_label_new (":");
    /* i18n: ':' is used in the end of the above line (When New Mail Arrives:)*/
    str2 = g_strdup_printf (markup, _(":"));
    gtk_label_set_markup (GTK_LABEL (label3), str2);
    g_free (str2);

    if (gtk_widget_get_default_direction () == GTK_TEXT_DIR_RTL)
    {
      gtk_box_pack_end (GTK_BOX (box), label3, FALSE, FALSE, 0);
      gtk_box_pack_end (GTK_BOX (box), combo, FALSE, FALSE, 0);
      gtk_box_pack_end (GTK_BOX (box), label2, FALSE, FALSE, 0);
      gtk_box_pack_end (GTK_BOX (box), label1, FALSE, FALSE, 0);
    }
    else
    {
      gtk_box_pack_start (GTK_BOX (box), label1, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (box), label2, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (box), combo, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (box), label3, FALSE, FALSE, 0);
    }

    gtk_widget_show_all (box);
  }
  else
  {
    g_debug ("EI: Woops, couldn't find the GtkFrame in the widget hierarchy");  
  }

  vbox = gtk_vbox_new (FALSE, 8);
  gtk_table_attach((GtkTable *)data->parent, vbox, 0, 1, 0, 1, 0, 0, 0, 0);

	check = gtk_check_button_new_with_mnemonic (_("Pla_y a sound"));
  g_object_set (check, "active", play_sound, NULL);
  gtk_box_pack_start (GTK_BOX (vbox), check, FALSE, FALSE, 0);
  g_signal_connect (check, "toggled", G_CALLBACK (on_sound_toggled), NULL);

	check = gtk_check_button_new_with_mnemonic (_("_Display a notification"));
  g_object_set (check, "active", show_bubble, NULL);
  gtk_box_pack_start (GTK_BOX (vbox), check, FALSE, FALSE, 0);
  g_signal_connect (check, "toggled", G_CALLBACK (on_bubble_toggled), NULL);

	check = gtk_check_button_new_with_mnemonic (_("_Indicate new messages in the panel"));
  g_object_set (check, "active", show_count, NULL);
  gtk_box_pack_start (GTK_BOX (vbox), check, FALSE, FALSE, 0);
  g_signal_connect (check, "toggled", G_CALLBACK (on_show_panel_toggled), NULL);

	gtk_widget_show_all (vbox);
	
  return check;
}
