/*****************************************************************************
 *
 * frame - Touch Frame Library
 *
 * Copyright (C) 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/
#include <oif/frame-xi2.h>
#include <xorg/xserver-properties.h>
#include <linux/input.h>
#include "frame-impl.h"
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <math.h>
#include <string.h>
#include <limits.h>
#include <X11/Xlib.h>

static Atom transform_atom;

#define set_field(fh, name, number, field)			\
	if (!strcmp(name, AXIS_LABEL_PROP_##field) &&		\
	    (fh->evmap[number] = field))

int oif_frame_is_supported_xi2(Display *dpy, const XIDeviceInfo *dev)
{

    int i;

    for (i = 0; i < dev->num_classes; i++)
	    if (dev->classes[i]->type == XITouchClass)
		    return 1;

    return 0;
}

static void init_properties(oif_frame_handle fh, Display *dpy,
			    const XIDeviceInfo *dev)
{
	struct oif_surface *s = fh->surface;
	XITouchClassInfo *v;
	int i;

	for (i = 0; i < dev->num_classes; i++) {
		v = (void *)dev->classes[i];
		if (v->type != XITouchClass)
			continue;
		switch(v->mode) {
		case XIDependentTouch:
			s->needs_pointer = 1;
			break;
		case XIDirectTouch:
			s->is_direct = 1;
			break;
		default:
			fprintf(stderr, "unknown mode\n");
			break;
		}
	}

	if (transform_atom == None)
		transform_atom = XInternAtom(dpy, XI_PROP_TRANSFORM, True);

	fh->map[0] = fh->map[4] = fh->map[8] = 1;
}

static void update_transform(oif_frame_handle fh, Display *dpy, int devid)
{
	unsigned long nitem, nbyte;
	unsigned char *data;
	int format;
	Atom type;

	if (transform_atom == None)
		return;

	XIGetProperty(dpy, devid, transform_atom, 0, 9, False,
		      AnyPropertyType, &type, &format, &nitem, &nbyte, &data);

	if (format == 32 && nitem == 9 && nbyte == 0)
		memcpy(fh->map, data, sizeof(fh->map));
	else
		fprintf(stderr, "Invalid transform matrix\n");

	XFree(data);
}

static void update_screen(oif_frame_handle fh, Display *dpy)
{
	struct oif_surface *s = fh->surface;
	XWindowAttributes attrs;
	float x1, y1, x2, y2;

	XGetWindowAttributes(dpy, XDefaultRootWindow(dpy), &attrs);

	x1 = 0;
	y1 = 0;
	x2 = attrs.width;
	y2 = attrs.height;

	s->mapped_min_x = fh->map[0] * x1 + fh->map[1] * y1 + fh->map[2];
	s->mapped_min_y = fh->map[3] * x1 + fh->map[4] * y1 + fh->map[5];
	s->mapped_max_x = fh->map[0] * x2 + fh->map[1] * y2 + fh->map[2];
	s->mapped_max_y = fh->map[3] * x2 + fh->map[4] * y2 + fh->map[5];
	s->mapped_max_pressure = 1;

	fprintf(stderr, "frame map: %f %f %f %f %f\n",
		s->mapped_min_x, s->mapped_min_y,
		s->mapped_max_x, s->mapped_max_y, s->mapped_max_pressure);
}

static void init_valuators(oif_frame_handle fh, Display *dpy,
			   const XIDeviceInfo *dev)
{
	struct oif_surface *s = fh->surface;
	XITouchValuatorClassInfo *v;
	char *name;
	int i;

	s->min_id = 0;
	s->max_id = UINT_MAX;
	s->phys_width = 100;
	s->phys_height = 65;
	s->phys_pressure = 10;
	s->max_x = 1024;
	s->max_y = 768;
	s->max_pressure = 256;
	s->max_orient = 1;

	for (i = 0; i < dev->num_classes; i++) {
		v = (void *)dev->classes[i];
		if (v->type != XITouchValuatorClass)
			continue;
		name = XGetAtomName(dpy, v->label);
		set_field(fh, name, v->number, ABS_MT_POSITION_X) {
			s->min_x = v->min;
			s->max_x = v->max;
			if (v->resolution > 0)
				s->phys_width = (v->max - v->min) /
					v->resolution;
		}
		set_field(fh, name, v->number, ABS_MT_POSITION_Y) {
			s->min_y = v->min;
			s->max_y = v->max;
			if (v->resolution > 0)
				s->phys_height = (v->max - v->min) /
					v->resolution;
		}
		set_field(fh, name, v->number, ABS_MT_TOUCH_MAJOR) {
			s->use_touch_major = 1;
		}
		set_field(fh, name, v->number, ABS_MT_TOUCH_MINOR) {
			s->use_touch_minor = 1;
		}
		set_field(fh, name, v->number, ABS_MT_WIDTH_MAJOR) {
			s->use_width_major = 1;
		}
		set_field(fh, name, v->number, ABS_MT_WIDTH_MINOR) {
			s->use_width_minor = 1;
		}
		set_field(fh, name, v->number, ABS_MT_ORIENTATION) {
			s->use_orientation = 1;
			s->max_orient = v->max > 0 ? v->max : 1;
		}
		set_field(fh, name, v->number, ABS_MT_PRESSURE) {
			s->use_pressure = 1;
			s->max_pressure = v->max > 0 ? v->max : 256;
			if (v->resolution > 0)
				s->phys_pressure = v->max / v->resolution;
		}
#ifdef ABS_MT_DISTANCE
#ifdef AXIS_LABEL_PROP_ABS_MT_DISTANCE
		set_field(fh, name, v->number, ABS_MT_DISTANCE) {
			s->use_distance = 1;
		}
#endif
#endif
		XFree(name);
	}
}

int oif_frame_init_xi2(oif_frame_handle fh, Display *dpy,
		       const XIDeviceInfo *dev)
{
	free(fh->evmap);
	fh->evmap =calloc(dev->num_classes, sizeof(int));
	if (!fh->evmap)
		return -ENOMEM;

	if (!oif_frame_is_supported_xi2(dpy, dev))
		return -ENODEV;

	init_properties(fh, dpy, dev);
	init_valuators(fh, dpy, dev);
	update_transform(fh, dpy, dev->deviceid);
	update_screen(fh, dpy);

	return 0;
}

static int handle_event(oif_frame_handle fh, struct oif_contact *slot, int ix,
			float value)
{
	switch (fh->evmap[ix]) {
	case ABS_MT_POSITION_X:
		slot->x = value;
		slot->vx = 0;
		return 1;
	case ABS_MT_POSITION_Y:
		slot->y = value;
		slot->vy = 0;
		return 1;
	case ABS_MT_TOUCH_MAJOR:
		slot->touch_major = value;
		return 1;
	case ABS_MT_TOUCH_MINOR:
		slot->touch_minor = value;
		return 1;
	case ABS_MT_WIDTH_MAJOR:
		slot->width_major = value;
		return 1;
	case ABS_MT_WIDTH_MINOR:
		slot->width_minor = value;
		return 1;
	case ABS_MT_ORIENTATION:
		slot->orientation = value;
		return 1;
	case ABS_MT_PRESSURE:
		slot->pressure = value;
		return 1;
#ifdef ABS_MT_DISTANCE
	case ABS_MT_DISTANCE:
		slot->distance = value;
		return 1;
#endif
	case ABS_MT_TOOL_TYPE:
		slot->tool_type = value;
		return 1;
	default:
		fprintf(stderr, "did not get that one: %d\n", ix);
		return 0;
	}
}

int oif_frame_configure_xi2(oif_frame_handle fh, const XConfigureEvent *ev)
{
	if (ev->display)
		update_screen(fh, ev->display);
	return 1;
}

const struct oif_frame *sync_xi2(oif_frame_handle fh, oif_frame_time_t time)
{
	struct oif_frame *next = fh->next;
	int i;

	/* no velocity adjustments without device time stamps in place */
	for (i = 0; i < fh->num_slots; i++) {
		struct oif_contact *q = next->slots[i];
		q->vx = 0;
		q->vy = 0;
	}

	return oif_frame_sync(fh, time);
}

const struct oif_frame *
oif_frame_pump_xi2(oif_frame_handle fh, const XIDeviceEvent *ev)
{
	const XIPropertyEvent *prop_ev;
	const double *value;
	const unsigned char *mask;
	struct oif_contact *slot;
	int touchid, nbyte, nbit, num, i;

	switch (ev->evtype) {
	case XI_TouchBegin:
	case XI_TouchUpdate:
	case XI_TouchEnd:
		value = ev->valuators.values;
		mask = ev->valuators.mask;
		touchid = ev->detail;
		nbyte = ev->valuators.mask_len;
		nbit = nbyte << 3;

		oif_frame_set_current_id(fh, touchid);
		slot = oif_frame_get_current_slot(fh);
		num = 0;
		for (i = 0; i < nbit; i++)
			if (XIMaskIsSet(mask, i))
				handle_event(fh, slot, i, value[num++]);
		if (ev->evtype == XI_TouchBegin)
			oif_frame_get_current_slot(fh)->active = 1;
		else if (ev->evtype == XI_TouchEnd)
			oif_frame_get_current_slot(fh)->active = 0;
		return sync_xi2(fh, ev->time);
	case XI_PropertyEvent:
		prop_ev = (XIPropertyEvent *)ev;
		if (prop_ev->property == transform_atom && ev->display) {
			update_transform(fh, ev->display, ev->deviceid);
			update_screen(fh, ev->display);
		}
		break;
	}

	return 0;
}
