/**
 * GMyth Library
 *
 * @file gmyth/gmyth_tvchain.c
 * 
 * @brief <p> This component contains functions for creating and accessing
 * the tvchain functions for live tv playback.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Hallyson Luiz de Morais Melo <hallyson.melo@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_tvchain.h"

#include <glib.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "gmyth_epg.h"
#include "gmyth_util.h"
#include "gmyth_query.h"
#include "gmyth_scheduler.h"
#include "gmyth_debug.h"

static void     gmyth_tvchain_class_init(GMythTVChainClass * klass);
static void     gmyth_tvchain_init(GMythTVChain * object);

static void     gmyth_tvchain_dispose(GObject * object);
static void     gmyth_tvchain_finalize(GObject * object);

static void     free_tvchain_entry(gpointer data, gpointer user_data);

G_DEFINE_TYPE(GMythTVChain, gmyth_tvchain, G_TYPE_OBJECT)
    static void     gmyth_tvchain_class_init(GMythTVChainClass * klass)
{
    GObjectClass   *gobject_class;

    gobject_class = (GObjectClass *) klass;

    gobject_class->dispose = gmyth_tvchain_dispose;
    gobject_class->finalize = gmyth_tvchain_finalize;
}

static void
gmyth_tvchain_init(GMythTVChain * tvchain)
{
    tvchain->tvchain_id = NULL;

    tvchain->cur_chanid = g_string_new("");
    tvchain->cur_startts = NULL;

    tvchain->mutex = g_mutex_new();
}

GMythTVChain   *
gmyth_tvchain_new()
{
    GMythTVChain   *tvchain =
        GMYTH_TVCHAIN(g_object_new(GMYTH_TVCHAIN_TYPE, NULL));

    return tvchain;
}

static void
gmyth_tvchain_dispose(GObject * object)
{
    GMythTVChain   *tvchain = GMYTH_TVCHAIN(object);

    if (tvchain->tvchain_id != NULL) {
        g_string_free(tvchain->tvchain_id, TRUE);
        tvchain->tvchain_id = NULL;
    }

    if (tvchain->mutex != NULL) {
        g_mutex_free(tvchain->mutex);
        tvchain->mutex = NULL;
    }

    if (tvchain->tvchain_list != NULL) {
        g_list_foreach(tvchain->tvchain_list, free_tvchain_entry, NULL);
        g_list_free(tvchain->tvchain_list);
    }

    if (tvchain->cur_chanid != NULL) {
        g_string_free(tvchain->cur_chanid, TRUE);
        tvchain->cur_chanid = NULL;
    }

    if (tvchain->backend_info) {
        g_object_unref(tvchain->backend_info);
        tvchain->backend_info = NULL;
    }


    G_OBJECT_CLASS(gmyth_tvchain_parent_class)->dispose(object);
}

static void
gmyth_tvchain_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_tvchain_parent_class)->finalize(object);
}

/** Initializes the tvchain and generates the tvchain id.
 * 
 * @param tvchain The GMythTVChain instance.
 * @param hostname The local hostname used to generate the tvchain id.
 */
gboolean
gmyth_tvchain_initialize(GMythTVChain * tvchain,
                         GMythBackendInfo * backend_info)
{
    const char     *hostname;

    assert(tvchain);
    g_return_val_if_fail(backend_info != NULL, FALSE);

    g_object_ref(backend_info);
    tvchain->backend_info = backend_info;

    hostname = gmyth_backend_info_get_hostname(backend_info);

    if (tvchain->tvchain_id == NULL) {
        gchar          *isodate = NULL;
        GTimeVal       *cur_time = g_new0(GTimeVal, 1);

        // struct tm* gmyth_util_time_val_to_date ( const GTimeVal* time )

        g_get_current_time(cur_time);
        isodate =
            gmyth_util_time_to_isoformat_from_time_val_fmt
            ("%Y-%m-%dT%H:%M:%S", cur_time);

        tvchain->tvchain_id =
            g_string_sized_new(7 + strlen(hostname) + strlen(isodate));
        g_string_printf(tvchain->tvchain_id, "live-%s-%s", hostname,
                        isodate);

        gmyth_debug("[%s] tv_chain_id: %s", __FUNCTION__,
                    tvchain->tvchain_id->str);

        g_free(isodate);
        g_free(cur_time);
    } else {
        gmyth_debug("[%s] TVchain already initialized", __FUNCTION__);
    }

    return TRUE;
}

/** Gets the tvchain id.
 * 
 * @param tvchain The GMythTVChain instance.
 * @return The tvchain id.
 */
GString        *
gmyth_tvchain_get_id(GMythTVChain * tvchain)
{
    g_return_val_if_fail(tvchain != NULL
                         && tvchain->tvchain_id != NULL, NULL);

    return tvchain->tvchain_id;
}

/** Reloads all tvchain entries in the database.
 * 
 * @param tvchain The GMythTVChain instance.
 * @return  TRUE if success, or FALSE if error.
 */
gboolean
gmyth_tvchain_reload_all(GMythTVChain * tvchain)
{
    MYSQL_ROW       msql_row;
    MYSQL_RES      *msql_res = NULL;
    GMythQuery     *gmyth_query = NULL;
    gboolean        ret = TRUE;
    GString        *stmt_str = NULL;

    g_mutex_lock(tvchain->mutex);

    /*
     * gets the initial size of the TVChain entries list 
     */
    guint           prev_size = g_list_length(tvchain->tvchain_list);

    gmyth_debug("[%s] chainid: %s", __FUNCTION__,
                tvchain->tvchain_id->str);

    if (tvchain != NULL && tvchain->tvchain_list != NULL) {
        g_list_free(tvchain->tvchain_list);
        tvchain->tvchain_list = NULL;
    }

    /*
     * TODO: Reuse gmyth_query already connected from context 
     */
    gmyth_query = gmyth_query_new();
    if (!gmyth_query_connect(gmyth_query, tvchain->backend_info)) {
        gmyth_debug("[%s] Could not connect to db", __FUNCTION__);
        g_mutex_unlock(tvchain->mutex);
        ret = FALSE;
        goto done;
    }

    stmt_str = g_string_new("");
    g_string_printf(stmt_str,
                    "SELECT chanid, starttime, endtime, discontinuity, "
                    "chainpos, hostprefix, cardtype, channame, input "
                    "FROM tvchain "
                    "WHERE chainid = \"%s\" ORDER BY chainpos;",
                    tvchain->tvchain_id->str);

    msql_res = gmyth_query_process_statement(gmyth_query, stmt_str->str);
    if (msql_res != NULL) {

        while ((msql_row = mysql_fetch_row(msql_res)) != NULL) {
            struct LiveTVChainEntry *entry =
                g_new0(struct LiveTVChainEntry, 1);
            entry->chanid = g_string_new(msql_row[0]);
            entry->starttime =
                gmyth_util_string_to_time_val((const gchar *) msql_row[1]);
            entry->endtime =
                gmyth_util_string_to_time_val((const gchar *) msql_row[2]);
            entry->discontinuity =
                g_ascii_strtoull(msql_row[3], NULL, 10) != 0;
            entry->hostprefix = g_string_new(msql_row[5]);
            entry->cardtype = g_string_new(msql_row[6]);
            entry->channum = g_string_new(msql_row[7]);
            entry->inputname = g_string_new(msql_row[8]);

            // m_maxpos = query.value(4).toInt() + 1;
            gmyth_debug
                ("[%s] Reading TV chain entry (channel %s): [%s, %s, %s]\n",
                 __FUNCTION__, entry->channum->str, entry->chanid->str,
                 (gchar *) msql_row[1], (gchar *) msql_row[2]);

            /*
             * add this to get the actual start timestamp of the last
             * recording 
             */
            if (tvchain->cur_startts < entry->starttime)
                tvchain->cur_startts = entry->starttime;

            tvchain->tvchain_list =
                g_list_append(tvchain->tvchain_list, entry);
        }
    } else {
        gmyth_debug("gmyth_tvchain_reload_all query error!\n");
        g_mutex_unlock(tvchain->mutex);

        ret = FALSE;
        goto done;
    }

    g_mutex_unlock(tvchain->mutex);

    tvchain->cur_pos =
        gmyth_tvchain_program_is_at(tvchain, tvchain->cur_chanid,
                                    tvchain->cur_startts);
    gmyth_debug("[%s] TVChain current position = %d.\n", __FUNCTION__,
                tvchain->cur_pos);

    if (tvchain->cur_pos < 0)
        tvchain->cur_pos = 0;

    // if (m_switchid >= 0)
    // m_switchid =
    // ProgramIsAt(m_switchentry.chanid,m_switchentry.starttime);

    if (prev_size != g_list_length(tvchain->tvchain_list)) {
        gmyth_debug("[%s] Added new recording", __FUNCTION__);
    }

  done:
    if (stmt_str != NULL)
        g_string_free(stmt_str, TRUE);

    if (msql_res != NULL)
        mysql_free_result(msql_res);

    if (gmyth_query != NULL)
        g_object_unref(gmyth_query);

    return ret;
}

/** 
 * Get all the program info entries in the database.
 * 
 * @param tvchain The GMythTVChain instance.
 * 
 * @return  A program info listage.
 */
GList          *
gmyth_tvchain_get_program_info_list(GMythTVChain * tvchain)
{
    GList          *prog_list = NULL;
    MYSQL_ROW       msql_row;
    MYSQL_RES      *msql_res = NULL;
    GMythQuery     *gmyth_query = NULL;
    GString        *stmt_str = NULL;

    g_mutex_lock(tvchain->mutex);

    gmyth_query = gmyth_query_new();
    if (!gmyth_query_connect(gmyth_query, tvchain->backend_info)) {
        gmyth_debug("Could not connect to db.");
        goto done;
    }

    stmt_str = g_string_new("");
    g_string_printf(stmt_str,
                    "SELECT channum, icon " "FROM channel "
                    "ORDER BY channum;");

    msql_res = gmyth_query_process_statement(gmyth_query, stmt_str->str);
    if (msql_res != NULL) {

        while ((msql_row = mysql_fetch_row(msql_res)) != NULL) {
            GMythProgramInfo *entry = gmyth_program_info_new();

            entry->channame = g_string_new(msql_row[0]);
            entry->chansign = g_string_new(msql_row[1]);

            gmyth_debug
                ("Reading TV program info entry (channel %s): [%s - {%s, %s}]\n",
                 entry->channame->str, entry->chansign->str,
                 (gchar *) msql_row[0], (gchar *) msql_row[1]);

            prog_list = g_list_append(prog_list, entry);
        }
    } else {
        gmyth_debug
            ("Query error when trying to get the channel list from database!\n");
        goto done;
    }

  done:
    g_mutex_unlock(tvchain->mutex);

    if (stmt_str != NULL)
        g_string_free(stmt_str, TRUE);

    if (msql_res != NULL)
        mysql_free_result(msql_res);

    if (gmyth_query != NULL)
        g_object_unref(gmyth_query);

    return prog_list;
}

/** 
 * Get all the program info entries in the database, given a channel name.
 * 
 * @param tvchain The GMythTVChain instance.
 * @param channel The channel name.
 * 
 * @return  A program info listage, based on a given channel name.
 */
GList          *
gmyth_tvchain_get_program_info_from_channel(GMythTVChain * tvchain,
                                            const gchar * channel)
{
    GList          *prog_list = NULL;
    MYSQL_ROW       msql_row;
    MYSQL_RES      *msql_res = NULL;
    GMythQuery     *gmyth_query = NULL;
    GString        *stmt_str = NULL;

    g_mutex_lock(tvchain->mutex);

    gmyth_query = gmyth_query_new();
    if (!gmyth_query_connect(gmyth_query, tvchain->backend_info)) {
        gmyth_debug("Could not connect to db.");
        goto done;
    }

    stmt_str = g_string_new("");
    g_string_printf(stmt_str,
                    "SELECT channum, icon "
                    "FROM channel "
                    "WHERE channum = \"%s\" ORDER BY channum;", channel);

    msql_res = gmyth_query_process_statement(gmyth_query, stmt_str->str);
    if (msql_res != NULL) {

        while ((msql_row = mysql_fetch_row(msql_res)) != NULL) {
            GMythProgramInfo *entry = gmyth_program_info_new();

            entry->channame = g_string_new(msql_row[0]);
            entry->chansign = g_string_new(msql_row[1]);

            gmyth_debug
                ("Reading TV program info entry (channel %s): [%s - {%s, %s}]\n",
                 entry->channame->str, entry->chansign->str,
                 (gchar *) msql_row[0], (gchar *) msql_row[1]);

            prog_list = g_list_append(prog_list, entry);
        }
    } else {
        gmyth_debug
            ("Query error when trying to get the channel list from database!\n");
        goto done;
    }

  done:
    g_mutex_unlock(tvchain->mutex);

    if (stmt_str != NULL)
        g_string_free(stmt_str, TRUE);

    if (msql_res != NULL)
        mysql_free_result(msql_res);

    if (gmyth_query != NULL)
        g_object_unref(gmyth_query);

    return prog_list;
}

/** 
 * Returns the internal index for the TV chain related to the given
 * channel and start time.
 * 
 * @param tvchain The GMythTVChain instance.
 * @param chanid The channel id.
 * @param startts The program start time.
 * 
 * @return The position of the related program info in the TV chain.
 */
gint
gmyth_tvchain_program_is_at(GMythTVChain * tvchain, GString * chanid,
                            GTimeVal * startts)
{
    gint            count = 0;
    struct LiveTVChainEntry *entry;
    GList          *tmp_list = tvchain->tvchain_list;
    guint           list_size = g_list_length(tvchain->tvchain_list);

    g_mutex_lock(tvchain->mutex);

    for (; tmp_list && (count < list_size);
         tmp_list = tvchain->tvchain_list->next, count++) {
        entry = (struct LiveTVChainEntry *) tmp_list->data;
        if (!g_strncasecmp(entry->chanid->str, chanid->str, chanid->len)
            && entry->starttime == startts) {
            g_mutex_unlock(tvchain->mutex);
            return count;
        }
    }
    g_mutex_unlock(tvchain->mutex);

    return -1;
}

/** Get the program info associated to the tvchain.
 * 
 * @param tvchain The GMythTVChain instance.
 * @param index The tvchain index.
 * 
 * @return The program info structure.
 */
GMythProgramInfo *
gmyth_tvchain_get_program_at(GMythTVChain * tvchain, gint index)
{
    struct LiveTVChainEntry *entry;

    entry = gmyth_tvchain_get_entry_at(tvchain, index);

    if (entry)
        return gmyth_tvchain_entry_to_program(tvchain, entry);

    return NULL;
}

/** Gets a LiveTVChainEntry associated to the tvchain by its index.
 * 
 * @param tvchain The GMythTVChain instance.
 * @param index The tvchain entry index
 * @return The LiveTVchainEntry structure.
 */
struct LiveTVChainEntry *
gmyth_tvchain_get_entry_at(GMythTVChain * tvchain, gint index)
{
    struct LiveTVChainEntry *chain_entry = NULL;

    g_return_val_if_fail(tvchain != NULL
                         && tvchain->tvchain_list != NULL, NULL);

    g_mutex_lock(tvchain->mutex);

    gint            size = g_list_length(tvchain->tvchain_list);
    gint            new_index = (index < 0
                                 || index >= size) ? size - 1 : index;

    if (new_index >= 0)
        chain_entry =
            (struct LiveTVChainEntry *) g_list_nth_data(tvchain->
                                                        tvchain_list,
                                                        new_index);

    g_mutex_unlock(tvchain->mutex);

    if (chain_entry != NULL) {
        gmyth_debug("[%s] Got TV Chain entry at %d.\n", __FUNCTION__,
                    new_index);

    } else {
        gmyth_debug("[%s] failed to get entry at index %d", __FUNCTION__,
                    index);
    }

    return chain_entry;
}

/** 
 * Gets the program info from backend database associated to the tv chain entry.
 * 
 * @param tvchain The GMythTVChain instance.
 * @param entry the LiveTVChainEntry to be converted.
 * 
 * @return The program info.
 */
GMythProgramInfo *
gmyth_tvchain_entry_to_program(GMythTVChain * tvchain,
                               struct LiveTVChainEntry * entry)
{
    GMythProgramInfo *proginfo = NULL;

    g_return_val_if_fail(tvchain != NULL, NULL);

    if (!entry || !tvchain) {
        gmyth_debug
            ("gmyth_tvchain_entry_to_program() received NULL argument");
        return NULL;
    }

    GMythScheduler *scheduler = gmyth_scheduler_new();

    gmyth_scheduler_connect(scheduler, tvchain->backend_info);
    proginfo = gmyth_scheduler_get_recorded(scheduler,
                                            entry->chanid,
                                            entry->starttime);
    gmyth_scheduler_disconnect(scheduler);

    if (proginfo) {
        proginfo->pathname =
            g_string_prepend(proginfo->pathname, entry->hostprefix->str);
    } else {
        gmyth_debug
            ("tvchain_entry_to_program( chan id = %s, starttime = %ld) failed!",
             entry->chanid->str, entry->starttime->tv_sec);
    }

    return proginfo;
}

static void
free_tvchain_entry(gpointer data, gpointer user_data)
{
    struct LiveTVChainEntry *entry;

    g_return_if_fail(data != NULL);

    entry = (struct LiveTVChainEntry *) data;

    if (entry->chanid != NULL) {
        g_string_free(entry->chanid, TRUE);
    }

    if (entry->starttime != NULL) {
        g_free(entry->starttime);
    }

    if (entry->endtime != NULL) {
        g_free(entry->endtime);
    }

    if (entry->hostprefix) {
        g_string_free(entry->hostprefix, TRUE);
    }

    if (entry->cardtype) {
        g_string_free(entry->cardtype, TRUE);
    }

    if (entry->channum) {
        g_string_free(entry->channum, TRUE);
    }

    if (entry->inputname) {
        g_string_free(entry->inputname, TRUE);
    }

    g_free(entry);
}
