/*
 * Copyright (C) 2016-2019 Tim Kosse
 * Copyright (C) 2019 Nikos Mavrogiannopoulos
 *
 * Author: Tim Kosse, Nikos Mavrogiannopoulos
 *
 * This file is part of GnuTLS.
 *
 * GnuTLS is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * GnuTLS is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#if defined(_WIN32)

int main(void)
{
	exit(77);
}

#else

#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <time.h>
#include <gnutls/gnutls.h>
#include <gnutls/dtls.h>
#include <signal.h>

#include "utils.h"
#include "cert-common.h"

/* This program tests that handshakes succeed if the server includes the
 * requested certificate status with the server certificate having
 * TLS feature 5 (status request).
 *
 * See RFC 7633
 */

static time_t mytime(time_t *t)
{
	time_t then = 1559941819;
	if (t)
		*t = then;

	return then;
}

static void server_log_func(int level, const char *str)
{
	fprintf(stderr, "server|<%d>| %s", level, str);
}

static void client_log_func(int level, const char *str)
{
	fprintf(stderr, "client|<%d>| %s", level, str);
}

const unsigned char ocsp_resp[] = {
	0x30, 0x82, 0x02, 0x3f, 0x0a, 0x01, 0x00, 0xa0, 0x82, 0x02, 0x38, 0x30,
	0x82, 0x02, 0x34, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30,
	0x01, 0x01, 0x04, 0x82, 0x02, 0x25, 0x30, 0x82, 0x02, 0x21, 0x30, 0x81,
	0x8a, 0xa1, 0x11, 0x30, 0x0f, 0x31, 0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55,
	0x04, 0x03, 0x13, 0x04, 0x43, 0x41, 0x2d, 0x33, 0x18, 0x0f, 0x32, 0x30,
	0x31, 0x39, 0x30, 0x36, 0x30, 0x37, 0x32, 0x31, 0x31, 0x35, 0x32, 0x32,
	0x5a, 0x30, 0x64, 0x30, 0x62, 0x30, 0x4d, 0x30, 0x09, 0x06, 0x05, 0x2b,
	0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0xb7, 0xca, 0x0f, 0xab,
	0xdc, 0x6f, 0xb8, 0xb0, 0x96, 0x7a, 0x15, 0xac, 0x98, 0x0a, 0x0f, 0x19,
	0xfe, 0xa4, 0x12, 0xde, 0x04, 0x14, 0x1e, 0x85, 0xed, 0x7f, 0x9e, 0x71,
	0xfa, 0x08, 0x9d, 0x37, 0x48, 0x43, 0xa0, 0x12, 0xef, 0xe5, 0xaa, 0xe1,
	0xe3, 0x8a, 0x02, 0x14, 0x60, 0x14, 0x5f, 0x01, 0xcb, 0xe0, 0x05, 0x45,
	0x38, 0x8c, 0x26, 0xfc, 0x5b, 0xcf, 0x6c, 0x41, 0xc3, 0xcb, 0xaa, 0xcc,
	0x80, 0x00, 0x18, 0x0f, 0x32, 0x30, 0x31, 0x39, 0x30, 0x36, 0x30, 0x37,
	0x32, 0x31, 0x31, 0x35, 0x32, 0x32, 0x5a, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82,
	0x01, 0x81, 0x00, 0x44, 0xc4, 0x59, 0xab, 0x7b, 0x6e, 0x35, 0x4e, 0x18,
	0x83, 0x02, 0xbd, 0x94, 0x26, 0x50, 0x01, 0xe2, 0xb1, 0x50, 0xdd, 0xca,
	0x61, 0x30, 0xb0, 0x93, 0x18, 0x56, 0xfe, 0x8d, 0x4f, 0xcc, 0x33, 0xc8,
	0x01, 0x1e, 0xac, 0xa1, 0x8e, 0xb0, 0x76, 0x0f, 0x41, 0x38, 0x7d, 0x06,
	0x9b, 0xfe, 0x09, 0x50, 0x6d, 0x86, 0x07, 0x2a, 0x48, 0x6e, 0x6a, 0xb1,
	0x13, 0xf4, 0xc0, 0x0f, 0x7c, 0x7d, 0x89, 0xb9, 0x69, 0xe7, 0x04, 0x2e,
	0xa4, 0x3d, 0xf6, 0xbd, 0x51, 0xbf, 0x52, 0x7d, 0xfb, 0x38, 0x7a, 0xbf,
	0xe6, 0xd7, 0x32, 0x57, 0x36, 0x87, 0xec, 0x91, 0x07, 0x0c, 0xac, 0xb9,
	0x79, 0xe7, 0x79, 0x4e, 0x49, 0x72, 0x1d, 0x16, 0xb6, 0x94, 0xbf, 0xc4,
	0x9f, 0x4e, 0x8b, 0x51, 0x54, 0x73, 0xb4, 0x4d, 0xe7, 0x01, 0x91, 0xcd,
	0x7c, 0xb2, 0x91, 0x4a, 0xc3, 0x4d, 0xc4, 0x4f, 0xa3, 0x42, 0xf1, 0x89,
	0xc7, 0xab, 0x36, 0x11, 0xf0, 0x7c, 0xc6, 0x8f, 0x03, 0x53, 0x85, 0x0c,
	0xfb, 0x30, 0x6b, 0xdd, 0x9e, 0x72, 0xd7, 0x77, 0xe5, 0xea, 0xd3, 0x39,
	0xb5, 0xb8, 0xdd, 0x61, 0xb9, 0xe7, 0x24, 0x9c, 0x85, 0x42, 0xd7, 0x2b,
	0x2e, 0x99, 0xdf, 0xe5, 0x8b, 0x79, 0xe3, 0x6e, 0x56, 0x6e, 0xd6, 0xed,
	0x5f, 0x9b, 0x5f, 0x40, 0x89, 0x17, 0x1a, 0x76, 0xbb, 0x3c, 0x9f, 0x33,
	0x71, 0xc1, 0xc5, 0x2f, 0xf4, 0x69, 0xe5, 0x5f, 0x83, 0xd4, 0x3a, 0x3d,
	0xd7, 0x44, 0xaa, 0xc0, 0x9d, 0xd9, 0xd9, 0x99, 0xec, 0x80, 0x4c, 0x46,
	0x5f, 0x91, 0xf4, 0x09, 0x06, 0xef, 0x37, 0x7c, 0x32, 0x64, 0x67, 0x85,
	0x99, 0xde, 0x9c, 0xce, 0x3e, 0x58, 0x1a, 0x6c, 0x59, 0xc9, 0x60, 0x26,
	0x02, 0xeb, 0x95, 0x52, 0x3e, 0x4f, 0xdd, 0x5f, 0x6c, 0x2d, 0x37, 0xc2,
	0x3b, 0x72, 0x70, 0xab, 0x1d, 0xf5, 0x2a, 0xbe, 0x8c, 0x70, 0x8e, 0xf0,
	0x25, 0x18, 0x68, 0xe5, 0xe9, 0xd1, 0xcf, 0xd8, 0x1f, 0x6c, 0x8e, 0xcf,
	0x18, 0x46, 0x51, 0xb4, 0x69, 0xbb, 0x6f, 0x4f, 0x1e, 0x2a, 0x61, 0x3f,
	0x64, 0x8b, 0x07, 0x7f, 0xc5, 0x80, 0xb9, 0x06, 0xd6, 0xb1, 0x8d, 0x47,
	0x4a, 0x61, 0xd2, 0x3e, 0xb4, 0xa6, 0xab, 0x12, 0xc6, 0x5c, 0x90, 0x9e,
	0x2e, 0x16, 0x2e, 0xd4, 0xfc, 0x4b, 0x08, 0x41, 0x94, 0xaf, 0x1d, 0x6e,
	0x6c, 0x11, 0x5c, 0x88, 0x3d, 0xd9, 0x30, 0x9d, 0x69, 0xf7, 0x45, 0xbe,
	0x5d, 0x1e, 0xd5, 0xe2, 0xf6, 0x38, 0xfa, 0xe1, 0xbf, 0xae, 0x9f, 0x2f,
	0xc6, 0x7b, 0x7b, 0x98, 0x89, 0x05, 0x8d, 0x4c, 0x01, 0xad, 0x61, 0x14,
	0x00, 0xca, 0xa3, 0xed, 0xd0, 0x2c, 0xfe, 0x1b, 0x7e, 0x1d, 0x70, 0x5b,
	0x2e, 0xc2, 0x54, 0xcf, 0x4c, 0x0a, 0xb3, 0x21, 0x58, 0xed, 0x51, 0xe7,
	0xeb, 0x8d, 0xb7
};

static int received = 0;

static int handshake_callback(gnutls_session_t session, unsigned int htype,
			      unsigned post, unsigned int incoming,
			      const gnutls_datum_t *msg)
{
	received = 1;
	return 0;
}

#define MAX_BUF 1024

static void client(int fd, const char *prio)
{
	int ret;
	unsigned int status;
	gnutls_certificate_credentials_t x509_cred;
	gnutls_session_t session;

	gnutls_global_set_time_function(mytime);
	global_init();

	if (debug) {
		gnutls_global_set_log_function(client_log_func);
		gnutls_global_set_log_level(7);
	}

	assert(gnutls_certificate_allocate_credentials(&x509_cred) >= 0);
	assert(gnutls_certificate_set_x509_trust_mem(x509_cred, &ca3_cert,
						     GNUTLS_X509_FMT_PEM) >= 0);

	assert(gnutls_init(&session, GNUTLS_CLIENT) >= 0);

	assert(gnutls_priority_set_direct(session, prio, NULL) >= 0);

	gnutls_handshake_set_hook_function(session,
					   GNUTLS_HANDSHAKE_CERTIFICATE_STATUS,
					   GNUTLS_HOOK_POST,
					   handshake_callback);

	gnutls_credentials_set(session, GNUTLS_CRD_CERTIFICATE, x509_cred);

	gnutls_transport_set_int(session, fd);

	do {
		ret = gnutls_handshake(session);
	} while (ret < 0 && gnutls_error_is_fatal(ret) == 0);
	if (ret < 0) {
		fail("client: Handshake failed: %s\n", gnutls_strerror(ret));
	} else {
		if (debug)
			success("client: Handshake was completed\n");
	}

	if (debug)
		success("client: TLS version is: %s\n",
			gnutls_protocol_get_name(
				gnutls_protocol_get_version(session)));

	if (received == 0 &&
	    gnutls_protocol_get_version(session) == GNUTLS_TLS1_2) {
		fail("client: did not receive certificate status when we should.\n");
	}

	ret = gnutls_certificate_verify_peers2(session, &status);
	if (ret != GNUTLS_E_SUCCESS) {
		fail("client: Peer certificate validation failed: %s\n",
		     gnutls_strerror(ret));
	} else {
		if (status) {
			gnutls_datum_t tmp;
			assert(gnutls_certificate_verification_status_print(
				       status, GNUTLS_CRT_X509, &tmp, 0) >= 0);
			fail("client: Validation status is not success (%x: %s)\n",
			     status, (char *)tmp.data);
		}
	}

	gnutls_bye(session, GNUTLS_SHUT_WR);

	close(fd);

	gnutls_deinit(session);

	gnutls_certificate_free_credentials(x509_cred);

	gnutls_global_deinit();
}

static int status_func(gnutls_session_t session, void *ptr,
		       gnutls_datum_t *resp)
{
	resp->data = gnutls_malloc(sizeof(ocsp_resp));
	if (resp->data == NULL)
		return -1;

	memcpy(resp->data, ocsp_resp, sizeof(ocsp_resp));
	resp->size = sizeof(ocsp_resp);
	return 0;
}

static void server(int fd, const char *prio)
{
	int ret;
	char buffer[MAX_BUF + 1];
	gnutls_session_t session;
	gnutls_certificate_credentials_t x509_cred;

	/* this must be called once in the program
	 */
	global_init();
	memset(buffer, 0, sizeof(buffer));

	if (debug) {
		gnutls_global_set_log_function(server_log_func);
		gnutls_global_set_log_level(4711);
	}

	assert(gnutls_certificate_allocate_credentials(&x509_cred) >= 0);
	assert(gnutls_certificate_set_x509_key_mem(
		       x509_cred, &server_ca3_tlsfeat_cert, &server_ca3_key,
		       GNUTLS_X509_FMT_PEM) >= 0);

	assert(gnutls_init(&session, GNUTLS_SERVER) >= 0);

	assert(gnutls_priority_set_direct(session, prio, NULL) >= 0);

	gnutls_certificate_set_ocsp_status_request_function(x509_cred,
							    status_func, NULL);

	gnutls_credentials_set(session, GNUTLS_CRD_CERTIFICATE, x509_cred);

	gnutls_transport_set_int(session, fd);

	do {
		ret = gnutls_handshake(session);
	} while (ret < 0 && gnutls_error_is_fatal(ret) == 0);

	if (ret < 0) {
		fail("server: Handshake failed: %s\n", gnutls_strerror(ret));
	}

	if (debug) {
		success("server: Handshake was completed\n");
	}

	if (debug)
		success("server: TLS version is: %s\n",
			gnutls_protocol_get_name(
				gnutls_protocol_get_version(session)));

	/* do not wait for the peer to close the connection.
	 */
	gnutls_bye(session, GNUTLS_SHUT_WR);

	close(fd);
	gnutls_deinit(session);

	gnutls_certificate_free_credentials(x509_cred);

	gnutls_global_deinit();

	if (debug)
		success("server: finished\n");
}

static void ch_handler(int sig)
{
	return;
}

static void start(const char *name, const char *prio)
{
	pid_t child;
	int fd[2];
	int ret, status = 0;

	signal(SIGCHLD, ch_handler);
	signal(SIGPIPE, SIG_IGN);

	received = 0;
	success("running: %s\n", name);

	ret = socketpair(AF_UNIX, SOCK_STREAM, 0, fd);
	if (ret < 0) {
		perror("socketpair");
		exit(1);
	}

	child = fork();
	if (child < 0) {
		perror("fork");
		fail("fork");
		exit(1);
	}

	if (child) {
		/* parent */
		close(fd[1]);
		client(fd[0], prio);
		waitpid(child, &status, 0);
		check_wait_status(status);
	} else {
		close(fd[0]);
		server(fd[1], prio);
		exit(0);
	}

	return;
}

void doit(void)
{
	start("tls1.2", "NORMAL:-VERS-TLS-ALL:+VERS-TLS1.2");
	start("tls1.3", "NORMAL:-VERS-TLS-ALL:+VERS-TLS1.3");
	start("default", "NORMAL");
}

#endif /* _WIN32 */
