/* grepmap
 * arch-tag: 6471b74d-3641-457e-b0d6-d59d7ee3f74b
 *
 * Copyright © 2004 Canonical Ltd.
 * Author: Scott James Remnant <scott@canonical.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#if HAVE_GETOPT_H
# define _GNU_SOURCE
# include <getopt.h>
#else
# include <unistd.h>
# define getopt_long(argc, argv, optstring, longopts, longindex) \
                    getopt ((argc), (argv), (optstring))
#endif /* HAVE_GETOPT_H */

#include <sys/utsname.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <errno.h>

#include "grepmap.h"


/* Path to kernel map files */
#define MAP_PATH "/lib/modules/%s/modules.%s"


/* Types of map file we can open */
typedef enum {
	NO_MAP,
	PCIMAP,
	USBMAP,
	INPUTMAP,
	IEEE1394MAP,
	LAST_MAP
} MapFileType;

/* Program name */
const char *program_name = NULL;


/* Forward prototypes */
static void  _print_usage   (void);
static void  _print_version (void);
static char *_map_filename  (const char *kernel, MapFileType map_type);


/* Command-line options */
static const char opts[] = "k:f:puiw";
static const struct option longopts[] = {
	{ "kernel",      required_argument, NULL, 'k' },
	{ "file",        required_argument, NULL, 'f' },
	{ "help",        no_argument,       NULL, 0400 + 'h' },
	{ "version",     no_argument,       NULL, 0400 + 'v' },
	{ "pcimap",      no_argument,       NULL, 'p' },
	{ "usbmap",      no_argument,       NULL, 'u' },
	{ "inputmap",    no_argument,       NULL, 'i' },
	{ "ieee1394map", no_argument,       NULL, 'w' },
	{ NULL,          no_argument,       NULL, 0 }
};

/* Information about each MapFileType */
typedef struct {
	int          opt, nargs;
	const char  *filename;

	int (*function)(FILE *, const char *, char *[]);
} MapFileTypeInfo;
static const MapFileTypeInfo map_info[] = {
	{ 0,    0, NULL,          NULL },
	{ 'p',  5, "pcimap",      pcimap_modules },
	{ 'u',  9, "usbmap",      usbmap_modules },
	{ 'i', 12, "inputmap",    inputmap_modules },
	{ 'w',  3, "ieee1394map", ieee1394map_modules }
};


/**
 * _print_usage:
 *
 * Print usage instructions to stdout.
 **/
static void
_print_usage (void)
{
	printf (_("Usage: %s [OPTION]... MAPTYPE ARG...\n"), program_name);
	printf (_("Parses module map files produced by module-init-tools and outputs the modules\n"
		  "you should load for a particular device.\n"));
	printf ("\n");
	printf (_("Mandatory arguments to long options are mandatory for short options too.\n"));
	printf ("\n");
	printf (_("Options:\n"
		  "  -k, --kernel=VERSION       read maps from alternative kernel VERSION\n"
		  "  -f, --file=FILE            read maps from FILE.\n"
		  "      --help                 display this help and exit.\n"
		  "      --version              output version information and exit.\n"));
	printf ("\n");
	printf (_("Map types:\n"
		  "  -p, --pcimap               pci device mappings.\n"
		  "  -u, --usbmap               usb device mappings.\n"
		  "  -i, --inputmap             generic input device mappings.\n"
		  "  -w, --ieee1394map          ieee1394 (firewire) device mappings.\n"));
	printf ("\n");
	printf (_("If neither FILE or VERSION is specified it reads the file for the currently\n"
		  "running kernel.  When FILE is `-', standard input is read.\n"));
	printf ("\n");
	printf (_("Each MAPTYPE has different requirements for ARG.  Unless otherwise specified\n"
		  "these should be the hexadecimal identifiers obtained from the kernel's sysfs\n"
		  "filesystem.  An optional `0x' prefix is permitted.\n"));
	printf ("\n");
	printf (_("  --pcimap      VENDOR DEVICE SUBSYSTEM-VENDOR SUBSYSTEM-DEVICE CLASS\n"));
	printf ("\n");
	printf (_("  --usbmap      VENDOR PRODUCT VERSION DEVICE-CLASS DEVICE-SUBCLASS\n"
		  "                DEVICE-PROTOCOL INTERFACE-CLASS INTERFACE-SUBCLASS\n"
		  "                INTERFACE-PROTOCOL\n"
		  "        VERSION must be specified without any period between the high and low\n"
		  "        parts.  e.g. `0x0123' not `1.23'.\n"));
	printf ("\n");
	printf (_("  --inputmap    BUS VENDOR PRODUCT VERSION EVBIT KEYBIT RELBIT ABSBIT MSCBIT\n"
		  "                LEDBIT SNDBIT FFBIT\n"
		  "        The BIT arguments should be specified as a colon-delimited array of\n"
		  "        hexademical values.\n"));
	printf ("\n");
	printf (_("  --ieee1394map VENDOR SPECIFIER VERSION\n"));
	printf ("\n");
	printf (_("Report bugs to <ubuntu-devel@lists.ubuntu.com>\n"));
}

/**
 * suggest_help:
 *
 * Print a message suggesting --help to stderr.
 **/
void
suggest_help (void)
{
	fprintf (stderr, _("Try `%s --help' for more information.\n"),
		 program_name);
}

/**
 * _print_version:
 *
 * Print version to stdout.
 **/
static void
_print_version (void)
{
	printf ("%s %s\n", PACKAGE_NAME, PACKAGE_VERSION);
	printf (_("Written by Scott James Remnant.\n"));
	printf ("\n");
	printf ("Copyright (C) 2004 Canonical Ltd.\n");
	printf (_("This is free software; see the source for copying conditions.  There is NO\n"
		  "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"));
}


int
main (int   argc,
      char *argv[])
{
	MapFileType  file_type = NO_MAP;
	char        *kernel = NULL, *file = NULL;
	FILE        *mapf;
	int          opt, ret = 0;

	setlocale (LC_ALL, "");
	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);

	program_name = argv[0];

	while ((opt = getopt_long (argc, argv, opts, longopts, NULL)) != -1) {
		switch (opt) {
		case 'k':
			kernel = optarg;
			break;
		case 'f':
			file = strdup (optarg);
			break;
		case 'p':
		case 'u':
		case 'i':
		case 'w':
			if (file_type != NO_MAP) {
				fprintf (stderr, "%s: %s\n", program_name,
					 _("only one map type may be specified"));
				suggest_help ();
				return 1;
			}

			for (; file_type < LAST_MAP; file_type++) {
				if (map_info[file_type].opt == opt)
					break;
			}

			break;
		case 0400 + 'h':
			_print_usage ();
			return 0;
		case 0400 + 'v':
			_print_version ();
			return 0;
		case '?':
			suggest_help ();
			return 1;
		default:
			/* Not reached */
			abort();
		}
	}

	if (file_type == NO_MAP) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("you must specify a map type"));
		suggest_help ();
		return 1;
	}

	if ((argc - optind) < map_info[file_type].nargs) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("insufficient arguments"));
		suggest_help ();
		return 1;
	} else if ((argc - optind) > map_info[file_type].nargs) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("some arguments ignored"));
	}

	if (! file) {
		file = _map_filename (kernel, file_type);
		if (! file)
			return 1;
	}

	if (! strcmp (file, "-")) {
		mapf = stdin;
	} else {
		mapf = fopen (file, "r");
		if (! mapf) {
			fprintf (stderr, "%s: %s: %s\n", program_name, file,
				 strerror (errno));
			return 1;
		}
	}

	ret = map_info[file_type].function (mapf, file, argv + optind);

	if (fclose (mapf)) {
		fprintf (stderr, "%s: %s: %s\n", program_name, file,
			 strerror (errno));
		return 1;
	}
	free (file);

	return ret;
}

/**
 * _map_filename:
 * @kernel: Alternative kernel version, or NULL.
 *
 * Looks up the current kernel version if required and constructs the
 * path to the map file.
 *
 * Returns: newly allocated string, or NULL on error.
 **/
static char *
_map_filename (const char  *kernel,
	       MapFileType  file_type)
{
	char *file;

	if (! kernel) {
		struct utsname uname_buf;

		if (uname (&uname_buf)) {
			fprintf (stderr, "%s: %s: %s\n", program_name,
				 _("unable to identify kernel version"),
				 strerror (errno));
			return NULL;
		}

		kernel = uname_buf.release;
	}

	file = malloc (sizeof (MAP_PATH) + strlen (kernel)
		       + strlen (map_info[file_type].filename) - 3);
	if (! file)
		abort();
	sprintf (file, MAP_PATH, kernel, map_info[file_type].filename);

	return file;
}
