/* grepmap
 * arch-tag: bf93e85c-dea8-409d-90da-741c7eccf448
 *
 * pcimap.c - parse modules.pcimap
 *
 * Copyright © 2004 Canonical Ltd.
 * Author: Scott James Remnant <scott@canonical.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "grepmap.h"


/* The "any device" id */
#define PCI_ANY 0xffffffff


/* PCI Device structure */
typedef struct {
	unsigned int id_vendor, id_device;
	unsigned int subid_vendor, subid_device;
	unsigned int class, class_mask;
} PciDevice;


/**
 * pcimap_modules:
 * @mapf: open map file to read from.
 * @file: filename or map file, or '-' for standard input.
 * @args: arguments (vendor, device, subvendor, subdevice, class)
 *
 * Handles the modules.pcimap file looking for a matching entry for
 * the device given, printing the module name to stdout if found.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
pcimap_modules (FILE       *mapf,
		const char *file,
		char       *argv[])
{
	PciDevice  dev;
	char      *line;
	int        lineno = 0, ret = 0;

	ret |= parse_hex (argv[0], &dev.id_vendor);
	ret |= parse_hex (argv[1], &dev.id_device);
	ret |= parse_hex (argv[2], &dev.subid_vendor);
	ret |= parse_hex (argv[3], &dev.subid_device);
	ret |= parse_hex (argv[4], &dev.class);
	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("arguments must be in hexadecimal format"));
		suggest_help ();
		return 1;
	}

	ret = 1;
	while ((line = fgets_alloc (mapf)) != NULL) {
		PciDevice map;
		int       s;

		++lineno;
		if ((line[0] == '#') || (line[0] == '\0'))
			continue;

		s = sscanf (line, "%*s 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x",
			    &map.id_vendor, &map.id_device,
			    &map.subid_vendor, &map.subid_device,
			    &map.class, &map.class_mask);
		if (s < 6) {
			fprintf (stderr, "%s:%s:%d: %s\n", program_name, file,
				 lineno, _("unrecognised line format"));
			continue;
		}

		if ((map.id_vendor != PCI_ANY)
		    && (map.id_vendor != dev.id_vendor))
			continue;

		if ((map.id_device != PCI_ANY)
		    && (map.id_device != dev.id_device))
			continue;

		if ((map.subid_vendor != PCI_ANY)
		    && (map.subid_vendor != dev.subid_vendor))
			continue;

		if ((map.subid_device != PCI_ANY)
		    && (map.subid_device != dev.subid_device))
			continue;

		if ((map.class & map.class_mask)
		    != (dev.class & map.class_mask))
			continue;

		line[strcspn (line, " \t\r\n")] = '\0';
		printf ("%s\n", line);
		ret = 0;
	}

	return ret;
}
